C ================================================================
      Subroutine FEM2Dtri(
C ================================================================
     &      XY1, XY2, XY3, 
     &      operatorA, FEMtypeA, operatorB, FEMtypeB, 
     &      label, D, DATA, order, 
     &      LDA, A, nRow, nCol)
C ================================================================
      include 'fem2Dtri.fd'
C ================================================================
C  *** ITRODUCTION ***
C  The routine computes elemental matrix for a bilinear form 
C
C  (1)            <D OpA(u), OpB(v)>               
C
C  where D is a tensor, OpA and OpB are linear operators, and
C  u and v are finite element functions, u in "A", and v in "B".
C  Note that solution of a non-linear problem may involve a
C  Newton-like iterative loop. In this case D may depend on 
C  a discrete function (e.g. approximation from the previous 
C  iterative step). If this is the case, evaluation of D maybe 
C  quite complex and require additional data. At the moment only 
C  one plug for additional data has been reserved for the user 
C  (see formula (3)). 
C 
C  In order to compute the right hand side, we can use the following
C  trick:
C
C (2)             f(v) = < D v, FEM_P0 > 
C
C  where action of D is given by function f, read also comments 
C  about formula (3) below.
C
C   
C  *** BILINER FORMS ***
C  The finite element space "A" is assumed to be richer that the
C  finite element space "B". The possible choices for these spaces 
C  are:
C    FEM_P0        - piecewise constant
C    FEM_P1        - piecewise linear 
C    FEM_P2        - picewise quadratic
C    FEM_P1vector  - vector piecewise linear. The unknowns
C                    are ordered first by vertices and then by
C                    the space directions (x and y)
C    FEM_4P1vector - piecewise linear on the uniform partition  
C                    onto 4 triangles
C    FEM_RT0       - lower order Raviart-Thomas finite elements
C
C  The available operators are:
C    IDEN  - identity operator 
C    GRAD  - gradient operator
C    DIV   - divergence operator
C    CURL  - rotor operator
C    DUDX  - partial derivative d/dx
C
C  A quadrature formula is chosen as follows:
C    order = 1     - quadrature formula with one center point
C    order = 2     - quadrature formula with 3 points inside triangle
C
C  The matrix A is defined by bilinear form (1). The following rules 
C  are applied for numbering basis functions:
C                    A) First, basis function associated with vertices are 
C                       numerated in the same order as the vertices XYi,
C                       i = 1,2 and 3
C
C                    B) Second, basis function associated with edges are 
C                       numerated in the same order as egdes 12,13 and 23. 
C
C                    C) The vector basis functions with 2 degrees of freedom 
C                       per a mesh object (vertex, edge) are enumerated 
C                       first by the corresponding mesh objects and then by
C                       the space coordinates, x and y.
C  
C     LDA          - leading dimention of matrix A(LDA, LDA)
C     nRow         - the number of rows of A
C     nCol         - the number of columns of A
C
C
C *** DESCRIPTION OF THE TENSOR ***  
C  The external function D has the following STANDARD format
C
C  (3)     INTEGER FUNCTION D(x, y, label, DATA, iSYS, Diff)
C
C    The function returns type of the tendor Diff which is defined
C    in fem2Dtri.fd.
C
C    (x, y)       - Real*8 Cartesian coordinates of a 2D point where
C                   tensor Diff should be evaluated
C
C    label        - identificator of a mesh element
C
C    DATA         - Real*8 user given data (a number or an array)
C
C    iSYS         - system buffer for information exchange:
C                   iD  = iSYS(1) -> number of rows in tensor Diff 
C                   jD  = iSYS(2) -> number of columns in tensor Diff
C
C    Diff(iD, jD) - Real*8 matrix. In the case of vector finite elements
C                   U = (u, v) the following ordering should be used:
C                   u_x, u_y, v_x, v_y.
C                  
C                   Examples. 
C                   A) isotropic diffusion problem:  
C                      iD = jD = 1
C                      Diff = diffusion value at the point (x,y)
C                      the user returns TENSOR_SCALAR
C
C                   B) anisotropic diffusion problem:
C                      iD = jD = 3
C                      Diff(i, j) = diffusion value at the point (x,y)
C                      the user returns TENSOR_SYMMETRIC
C
C                   C) convection problem:
C                      iD = 1, jD = 3
C                      Diff(1, j) = velocity value at the point (x,y)
C                      the user returns TENSOR_GENERAL
C
C ================================================================
      Real*8   XY1(*), XY2(*), XY3(*)

      Integer  FEMtypeA, FEMtypeB, operatorA, operatorB
      Integer  label, order, LDA, D

      Real*8   DATA(*)
      EXTERNAL D

      Real*8   A(LDA, *)
C ================================================================
c Local variables
      Real*8  XYP(2, 3)
      Real*8  det, vol, s
   
      Real*8  PSI(2, 2)
      Real*8  U(4, 6, MaxPointGauss),    V(4, 6, MaxPointGauss)
      Real*8  Diff(4, 4, MaxPointGauss), DU(4, 6, MaxPointGauss)

      Real*8  w(MaxPointGauss), XYG(2, MaxPointGauss)
      Real*8  XYL(3, AllPointGauss)

      Integer iSYS(2), tensor
      Logical FEMtype, operator
C ================================================================
      DATA XYL/3 * T1A, 
c ... 3 points (order 2)
     &         T2A,T2B,T2B,  T2B,T2A,T2B,  T2B,T2B,T2A/
C ================================================================
      If(order.LE.0 .OR. order.GT.2) 
     &  Call errMes(2001, 'fem2Dtri', 'input data are wrong')

      FEMtype = FEMtypeA.EQ.FEMtypeB
      operator = operatorA.EQ.operatorB

c ... transformation of variables y = PSI * (x - x_0)
      Do i = 1, 2
         XYP(i, 1) = 0D0
         XYP(i, 2) = XY2(i) - XY1(i)
         XYP(i, 3) = XY3(i) - XY1(i)
      End do

      Call solve2x2(XYP(1, 2), XYP(2, 2), PSI(1, 1),
     &              XYP(1, 3), XYP(2, 3), PSI(1, 2), det)

      Call solve2x2(XYP(1, 3), XYP(2, 3), PSI(2, 1),
     &              XYP(1, 2), XYP(2, 2), PSI(2, 2), det)

c ... weights and points
      vol = dabs(det) / 2
      Call WeightsPoints(XY1, XY2, XY3, vol, order, 
     &                   XYG, w, iGauss, iL)

c ... compute operatorA * FEMtypeA
      If(operatorA.EQ.GRAD) Then
         Call applyGRAD(iGauss, XYL(1, iL), PSI, FEMtypeA, nfa, idim, U)
      Else If(operatorA.EQ.DIV) Then
         Call applyDIV( iGauss, XYL(1, iL), PSI, FEMtypeA, 
     &                  nfa, idim, U, XYP, det)
      Else If(operatorA.EQ.IDEN) Then
         Call applyIDEN(iGauss, XYL(1, iL), PSI, FEMtypeA, 
     &                  nfa, idim, U, XYP, det)
      Else If(operatorA.EQ.CURL) Then
         Call applyCURL(iGauss, XYL(1, iL), PSI, FEMtypeA, 
     &                  nfa, idim, U, XYP, det)
      Else If(operatorA.EQ.DUDX) Then
         Call applyDUDX(iGauss, XYL(1, iL), PSI, FEMtypeA, nfa, idim, U)
      Else
         Call errMes(2001, 'fem2Dtri', 'the operatorA is not supported')
      End if
      If(nfa.GT.LDA) Call errMes(2001, 'fem2Dtri',
     &     'the local matrix leading dimension, LDA, is too small')


c ... compute operatorB * FEMtypeB
      nfb = nfa
      jdim = idim
      if(operator .AND. FEMtype) Goto 100

      If(operatorB.EQ.GRAD) Then
         Call applyGRAD(iGauss, XYL(1, iL), PSI, FEMtypeB, nfb, jdim, V)
      Else If(operatorB.EQ.DIV) Then
         Call applyDIV( iGauss, XYL(1, iL), PSI, FEMtypeB, 
     &                  nfb, jdim, V, XYP, det)
      Else If(operatorB.EQ.IDEN) Then
         Call applyIDEN(iGauss, XYL(1, iL), PSI, FEMtypeB, 
     &                  nfb, jdim, V, XYP, det)
      Else If(operatorB.EQ.CURL) Then
         Call applyCURL(iGauss, XYL(1, iL), PSI, FEMtypeB, 
     &                  nfb, jdim, V, XYP, det)
      Else If(operatorB.EQ.DUDX) Then
         Call applyDUDX(iGauss, XYL(1, iL), PSI, FEMtypeB, nfb, jdim, V)
      Else
         Call errMes(2001, 'fem2Dtri', 'the operatorB is not supported')
      End if
      If(nfb.GT.LDA) Call errMes(2001, 'fem2Dtri',
     &     'the local matrix second dimension, LDA, is too small')

 

c ... compute D * U
 100  iD = jdim
      jD = idim

      Do n = 1, iGauss
         tensor = D(XYG(1,n), XYG(2,n), label, 
     &                        DATA, iSYS, Diff(1, 1, n))
         If(tensor.EQ.TENSOR_NULL) Goto 200
      End do
      iD = iSYS(1)
      jD = iSYS(2)


 200  Continue
      If(tensor.EQ.TENSOR_NULL) Then
         If(idim.NE.jdim) Call errMes(2001, 
     &        'fem2Dtri', 'the operators A and B are not compatible')

         Do n = 1, iGauss
            Do i = 1, idim
               Do k = 1, nfa
                  DU(i, k, n) = U(i, k, n) * w(n)
               End do
            End do
         End do
      Else If(tensor.EQ.TENSOR_SCALAR) Then
         If(idim.NE.jdim) Call errMes(2001, 'fem2Dtri', 
     &        'the operators A and B are not compatible')

         Do n = 1, iGauss
            s = Diff(1, 1, n) * w(n) 
            Do i = 1, idim
               Do k = 1, nfa
                  DU(i, k, n) = U(i, k, n) * s
               End do
            End do
         End do
      Else If(tensor.EQ.TENSOR_SYMMETRIC .OR. 
     &        tensor.EQ.TENSOR_GENERAL) Then
         If(jD.NE.idim .OR. iD.NE.jdim) Call errMes(2001, 
     &        'fem2Dtri', 'the operators A and B are not compatible')

         Do n = 1, iGauss
            Do i = 1, iD
               Do k = 1, nfa
                  s = 0D0
                  Do j = 1, jD
                     s = s + Diff(i, j, n) * U(j, k, n)
                  End do
                  DU(i, k, n) = s * w(n)
               End do
            End do
         End do
      Else
         Call errMes(2001, 'fem2Dtri', 
     &        'the given tensor is not supported') 
      End if


c ... compute <D U, V>
      If(operator .AND. FEMtype .AND. tensor.NE.TENSOR_GENERAL) Then
         Do i = 1, nfa
            Do j = 1, i - 1
               A(i, j) = A(j, i)
            End do

            Do j = i, nfa
               s = 0D0
               Do k = 1, iD
                  Do n = 1, iGauss
                     s = s + DU(k, i, n) * U(k, j, n)
                  End do
                  A(i, j) = s
               End do
            End do
         End do
      Else If(operator .AND. FEMtype) Then
         Do i = 1, nfa
            Do j = 1, nfa
               s = 0D0
               Do k = 1, iD
                  Do n = 1, iGauss
                     s = s + DU(k, i, n) * U(k, j, n)
                  End do
                  A(i, j) = s
               End do
            End do
         End do
      Else 
         Do i = 1, nfa
            Do j = 1, nfb
               s = 0D0
               Do k = 1, iD
                  Do n = 1, iGauss
                     s = s + DU(k, i, n) * V(k, j, n)
                  End do
                  A(i, j) = s
               End do
            End do
         End do
      End if

      nRow = nfa
      nCol = nfb

      Return
      End


c ==============================================================
      Subroutine applyGRAD(iGauss, XYL, PSI, FEMtype, nfa, dim, U)
c ==============================================================
      include 'fem2Dtri.fd'
c ==============================================================
      Integer iGauss, FEMtype, nfa, dim
      Real*8  PSI(2, 2), U(4, 6, *)
c ==============================================================
c Data for the reference triangle
      Real*8  XYL(3, *)
      Real*8  GRAD_P1(2, 3), GRAD_P2(2, 6, MaxPointGauss)
      Real*8  x, y, Lfun

      DATA    GRAD_P1/-1,-1, 1,0, 0,1/
c ==============================================================
      If(FEMtype.EQ.FEM_P0) Then
         nfa = 1
         dim = 1
         Do n = 1, iGauss
            U(1, 1, n) = 0D0
         End do
      Else If(FEMtype.EQ.FEM_P1) Then
         nfa = 3
         dim = 2 
         Do i = 1, nfa
            Do k = 1, dim
               U(k, i, 1) = 0D0
               Do j = 1, 2
                  U(k, i, 1) = U(k, i, 1) + PSI(j, k) * GRAD_P1(j, i)
               End do
            End do
         End do
         Call copyGauss(iGauss, nfa, dim, U)
      Else If(FEMtype.EQ.FEM_P2) Then
         nfa = 6
         dim = 2
         Do n = 1, iGauss
            x = XYL(2, n)
            y = XYL(3, n)
            Do i = 1, 4
               Do k = 1, dim
                  GRAD_P2(k, i, n) = GRAD_P1(k, i) *
     &                               (4 * Lfun(i, x, y) - 1D0) 
               End do
            End do

            mfa = 3
            Do i = 1, 3
               Do j = i + 1, 3
                  mfa = mfa + 1
                  Do k = 1, dim
                     GRAD_P2(k, mfa, n) = 
     &                    4 * (Lfun(i, x, y) * GRAD_P1(k, j) +
     &                         Lfun(j, x, y) * GRAD_P1(k, i))
                  End do
               End do
            End do
         End do

         Do n = 1, iGauss
            Do i = 1, nfa
               Do k = 1, dim
                  U(k, i, n) = 0D0
                  Do j = 1, 2
                     U(k, i, n) = U(k, i, n) 
     &                          + PSI(j, k) * GRAD_P2(j, i, n)
                  End do
               End do
            End do
         End do
      Else If(FEMtype.EQ.FEM_P1vector) Then
         nfa = 6
         dim = 4 
         Call clearU(1, nfa, dim, U)

         Do i = 1, 3
            Do k = 1, 2
               Do j = 1, 2
                  U(k, i, 1) = U(k, i, 1) + PSI(j, k) * GRAD_P1(j, i)
               End do
               U(k + 2, i + 3, 1) = U(k, i, 1)
            End do
         End do
         Call copyGauss(iGauss, nfa, dim, U)
      Else
         nfa = 0
         dim = -FEMtype
         Call errMes(2001, 'fem2Dtri', 
     &        'unsupported operation for the given element type')
      End if
      Return
      End


c ==============================================================
      Subroutine applyDIV(iGauss, XYL, PSI, FEMtype, 
     &                    nfa, dim, U, XYP, det)
c ==============================================================
      include 'fem2Dtri.fd'
c ==============================================================
      Integer iGauss, FEMtype, nfa, dim
      Real*8  PSI(2, 2), XYL(3, *), U(4, 6, *)
      Real*8  XYP(2, 3), det
c ==============================================================
c Local variables 
      Real*8  V(4, 6, MaxPointGauss)
      Real*8  vol, edge, calEdge, xyn(2), xym(2)
c ==============================================================
      Integer IPF(3, 3)

      DATA    IPF/1,2,3, 1,3,2, 2,3,1/ 
c ==============================================================
      If(FEMtype.EQ.FEM_P0) Then
         nfa = 1
         dim = 1
         Do n = 1, iGauss
            U(1, 1, n) = 0D0
         End do
      Else If(FEMtype.EQ.FEM_RT0) Then
         Call applyGRAD(1, XYL, PSI, FEM_P1, nfc, dim, V)

         nfa = 3
         dim = 1
         vol = dabs(det)

         Do i = 1, nfa
            iP1 = IPF(1, i)
            iP2 = IPF(2, i)
            iP3 = IPF(3, i)
            
            edge = calEdge(XYP(1, iP1), XYP(1, iP2)) 
           
            Do n = 1, iGauss
               U(1, i, n) = 2 * edge / vol 
            End do
         End do
      Else If(FEMtype.EQ.FEM_P1vector) Then
         Call applyGRAD(1, XYL, PSI, FEM_P1, nfc, dim, V)

         nfa = 6
         dim = 1 
         Call clearU(1, nfa, dim, U)

         Do i = 1, 3
            U(1, i,     1) = V(1, i, 1)
            U(1, i + 3, 1) = V(2, i, 1)
         End do
         Call copyGauss(iGauss, nfa, dim, U)
      Else
         nfa = 0
         dim = -FEMtype
         Call errMes(2001, 'fem2Dtri', 
     &        'unsupported operation for the given element type')
      End if
      Return
      End


c ==============================================================
      Subroutine applyCURL(iGauss, XYL, PSI, FEMtype, 
     &                     nfa, dim, U, XYP, det)
c ==============================================================
      include 'fem2Dtri.fd'
c ==============================================================
      Integer iGauss, FEMtype, nfa, dim
      Real*8  PSI(2, 2), XYL(3, *), U(4, 6, *)
      Real*8  XYP(2, *), det
c ==============================================================
c Local variables 
      Real*8  V(4, 6, MaxPointGauss)
c ==============================================================
      If(FEMtype.EQ.FEM_P0) Then
         nfa = 1
         dim = 1
         Do n = 1, iGauss
            U(1, 1, n) = 0D0
         End do
      Else
         nfa = 0
         dim = -FEMtype
         Call errMes(2001, 'fem2Dtri', 
     &        'unsupported operation for the given element type')
      End if
      Return
      End



c ==============================================================
      Subroutine applyIDEN(iGauss, XYL, PSI, FEMtype, 
     &                     nfa, dim, U, XYP, det)
c ==============================================================
      include 'fem2Dtri.fd'
c ==============================================================
      Integer iGauss, FEMtype, nfa, dim
      Real*8  PSI(2, 2), XYL(3, *), U(4, 6, *)
      Real*8  XYP(2, *), det
c ==============================================================
c Local variables 
      Real*8  vol, edge, calEdge, xyn(2), xym(2)
      Real*8  x, y, Lfun, s
c ==============================================================
c Data for the reference triangle
      Integer IPF(3, 3)

      DATA    IPF/1,2,3, 1,3,2, 2,3,1/ 
c ==============================================================
      If(FEMtype.EQ.FEM_P0) Then
         nfa = 1
         dim = 1
         Do n = 1, iGauss
            U(1, 1, n) = 1D0
         End do
      Else If(FEMtype.EQ.FEM_P1) Then
         nfa = 3
         dim = 1
         Do n = 1, iGauss
            Do i = 1, nfa
               U(1, i, n) = XYL(i, n)
            End do
         End do
      Else If(FEMtype.EQ.FEM_P2) Then
         nfa = 6
         dim = 1
         Do n = 1, iGauss
            x = XYL(2, n)
            y = XYL(3, n)
            Do i = 1, 3
               s = Lfun(i, x, y)
               U(1, i, n) = (4 * s - 1D0) * s
            End do

            mfa = 3
            Do i = 1, 3
               Do j = i + 1, 3
                  mfa = mfa + 1
                  U(1, mfa, n) = 4 * Lfun(i, x, y) * Lfun(j, x, y)
               End do
            End do
         End do
      Else If(FEMtype.EQ.FEM_P1vector) Then
         nfa = 6
         dim = 2

         Call clearU(iGauss, nfa, dim, U)
         Do n = 1, iGauss
            Do i = 1, 3
               U(1, i,     n) = XYL(i, n)
               U(2, i + 3, n) = XYL(i, n)
            End do
         End do
      Else If(FEMtype.EQ.FEM_RT0) Then
         nfa = 3
         dim = 2
         vol = dabs(det)

         Do i = 1, nfa
            iP1 = IPF(1, i)
            iP2 = IPF(2, i)
            iP3 = IPF(3, i)
            
            edge = calEdge(XYP(1, iP1), XYP(1, iP2)) 
            edge = edge / vol
            Do n = 1, iGauss
               Do k = 1, dim
                  U(k, i, n) = edge *
     &                 (XYL(iP1, n) * (XYP(k, iP1) - XYP(k, iP3))
     &                + XYL(iP2, n) * (XYP(k, iP2) - XYP(k, iP3)))
               End do
            End do
         End do
      Else
         nfa = 0
         dim = -FEMtype
         Call errMes(2001, 'fem2Dtri', 
     &        'unsupported operation for the given element type')
      End if
      Return
      End



c ==============================================================
      Subroutine applyDUDX(iGauss, XYL, PSI, FEMtype, nfa, dim, U)
c ==============================================================
      include 'fem2Dtri.fd'
c ==============================================================
      Integer iGauss, FEMtype, nfa, dim
      Real*8  PSI(2, 2), U(4, 6, *)
c ==============================================================
c Data for the reference triangle
      Real*8  XYL(3, *)
      Real*8  GRAD_P1(2, 3), GRAD_P2(2, 6, MaxPointGauss)
      Real*8  x, y, Lfun

      DATA    GRAD_P1/-1,-1, 1,0, 0,1/
c ==============================================================
      If(FEMtype.EQ.FEM_P0) Then
         nfa = 1
         dim = 1
         Do n = 1, iGauss
            U(1, 1, n) = 0D0
         End do
      Else If(FEMtype.EQ.FEM_P1) Then
         nfa = 3
         dim = 1 

c  ...   choose ixy = 1 for DuDx and ixy = 2 for DuDy
         ixy = 1
         Do i = 1, nfa
            U(1, i, 1) = 0D0
            Do j = 1, 2
               U(1, i, 1) = U(1, i, 1) + PSI(j, ixy) * GRAD_P1(j, i)
            End do
         End do
         Call copyGauss(iGauss, nfa, dim, U)
      Else
         nfa = 0
         dim = -FEMtype
         Call errMes(2001, 'fem2Dtri', 
     &        'unsupported operation for the given element type')
      End if
      Return
      End



c ==============================================================
      Subroutine WeightsPoints(XY1, XY2, XY3, vol, order, 
     &                         XYG, w, iGauss, iLref)
c ==============================================================
      include 'fem2Dtri.fd'
c ==============================================================
C The procedure is used for effective computing points for 
C numerical integration, b/c of a symmetry.
C
C Remark: A 1-to-1 coresspondance between XYL and XYG should be
C         hold.
c ==============================================================
      Real*8  XY1(2), XY2(2), XY3(2), vol
      Integer order

      Real*8  XYG(2, *), w(*)
      Integer iGauss, iLref
c ==============================================================
      If(order.EQ.1) Then
         iGauss = LDG1
         iLref = 1
         w(1) = W1A * vol

         Do i = 1, 2
            XYG(i, 1) = T1A * (XY1(i) + XY2(i) + XY3(i))
         End do
      Else If(order.EQ.2) Then
         iGauss = LDG2
         iLref = LDG1 + 1
         Do i = 1, iGauss
            w(i) = W2A * vol
         End do

         Do i = 1, 2
            XYG(i, 1) = T2B * (XY2(i) + XY3(i)) 
            XYG(i, 2) = T2B * (XY1(i) + XY3(i)) 
            XYG(i, 3) = T2B * (XY1(i) + XY2(i)) 
         End do
      End if
      Return
      End


c ==============================================================
      Real*8 Function Lfun(i, x, y)
c ==============================================================
      Real*8 x, y

      If(i.EQ.1) Then
         Lfun = 1D0 - x - y
      Else If(i.EQ.2) Then
         Lfun = x
      Else If(i.EQ.3) Then
         Lfun = y
      End if
      Return
      End


c ==============================================================
      Subroutine solve2x2(
c ==============================================================
     &     a11, a12, a,
     &     a21, a22, b, det)
c ==============================================================
      Real*8  a11, a12, a
      Real*8  a21, a22, b, det
c ==============================================================
c Local variables
      Real*8  s
c ==============================================================
      det = a11 * a22 - a21 * a12
      s = 1 / det

      a = a22 * s
      b =-a21 * s
      Return
      End


C ================================================================
      Subroutine calNormalVec(xy1, xy2, xyn)
C ================================================================
C Routine computes a normal vector to the oriented edge xy1 -> xy2
C ================================================================
      Real*8 xy1(2), xy2(2), xyn(2)
      Real*8 ax, ay

      ax = xy2(1) - xy1(1)
      ay = xy2(2) - xy1(2)

      xyn(1) =  ay
      xyn(2) = -ax
     
      Return
      End


C ================================================================
      Subroutine copyGauss(iGauss, nfa, idim, U)
C ================================================================
      Real*8  U(4, 6, *)

      Do n = 2, iGauss
         Do i = 1, nfa
            Do k = 1, idim
               U(k, i, n) = U(k, i, 1)
            End do
         End do
      End do

      Return
      End


C ================================================================
      Subroutine clearU(iGauss, nfa, idim, U)
C ================================================================
      Real*8  U(9, 6, *)

      Do n = 1, iGauss
         Do i = 1, nfa
            Do k = 1, idim
               U(k, i, n) = 0D0
            End do
         End do
      End do

      Return
      End

