C ================================================================
      Subroutine BilinearFormVolume(
C ================================================================
     &           nP, nE, XYP, IPE, lbE,
     &           operatorA, FEMtypeA, operatorB, FEMtypeB,
     &           D, DATA, order,
     &           assembleStatus, MaxIA, MaxA,
     &                           IA, JA, DA, A, nRow, nCol,
     &           MaxWi, iW)
C ================================================================
      Include 'fem2Dtri.fd'
      Include 'assemble.fd'
C ================================================================
C  The routine computes a stiffness matrix for bilinear form (2).
C ================================================================
C *** GENERAL ***
C  Here is the decription of input parameters for all routines in
C  this file are:
C  (1.1)   BilinearFormVolume( ... )
C          BilinearFormSurface( ... )
C
C  (1.2)   LinearFormVolume( ... )
C          LinearFormSurface( ... )
C
C  (1.3)   BoundaryConditions( ... )
C
C  The order of calls is important to save on arithmetical
C  operations. BilinearFormVolume( ... ) should be called first,
C  and BoundaryConditions( ... ) should be called last.
C
C
C *** INTRODUCTION ***
C  The routines assemble the local matrices for the bilinear form
C
C  (2)            <D OpA(u), OpB(v)>
C
C  where D is a tensor, OpA and OpB are linear operators.
C
C  In order to compute the right hand side, we can use the following
C  trick:
C
C  (3)             f(v) = < D v, FEM_P0 >
C
C  where action of D is given by function f, read also comments
C  about formula (4) below.
C
C  A general variational problem may consists of a few bilinear
C  forms. We supply a library for operating with sparce matrices
C  given in a few formats (see files  algebra_*.f).
C
C
C *** GRID ***
C   The grid is the union of elements (triangles, quads, etc.)
C   At the moment only triangles are allowed.
C
C     nP  - the number of points (P)
C     nF  - the number of edges (F)
C     nE  - the number of elements (E)
C
C     XYP(2, nP) - The Cartesian coordinates of mesh points
C
C     IPF(4, nF) - connectivity list of boundary faces
C
C     IPE(3, nE) - connectivity list of element. On output,
C                  each column of it is ordered by increasing.
C
C     lbE(nE)    - element indentificator (a positive number)
C
C
C *** BILINEAR & LINEAR FORMS ***
C   The bilinear form is defined as follows: <D OpA(u), OpB(v)>
C   where D is a tensor, OpA and OpB are linear operators, and
C   u and v are finite element functions.
C
C     operatorA - the available operators are:
C                 IDEN  - identity operator
C                 GRAD  - gradient operator
C                 DIV   - divergence operator
C                 CURL  - rotor operator
C                 DUDX  - partial derivative d/dx 
C
C                 REMARK: It is assumed that operatorA is richer than
C                          operatorB.
C
C     FEMtypeA - The possible choices of finite elements are available:
C
C                FEM_P0        - piecewise constant
C                FEM_P1        - piecewise linear
C                FEM_P2        - picewise quadratic
C                FEM_P1vector  - vector piecewise linear. The
C                                unknowns are ordered first by
C                                vertices and then by the space
C                                directions (x, y and z)
C                FEM_RT0       - lower order Raviart-Thomas finite
C                                elements
C
C     operatorB - see operatorA
C
C     FEMtypeB  - see FEMtypeA. 
C
C
C *** DESCRIPTION OF TENSORS ***
C     D         - the external function D has the following format
C
C     (4)    INTEGER FUNCTION D(x, y, label, DATA, iSYS, Diff)
C
C     The function returns type of the tensor Diff (defined in fem2Dtri.fd)
C     or type of the boundary condition (defined in assemble.fd)
C
C     Parameters:
C              (x, y)   - Real*8 Cartesian coordinates of a 3D point where
C                         tensor Diff should be evaluated
C
C              label    - identificator of a mesh face
C
C              DATA     - Real*8 user given data (a number or an array)
C
C              iSYS(2)  - system buffer for information exchange:
C                         iD  = iSYS(1) -> number of rows in tensor Diff
C                         jD  = iSYS(2) -> number of columns in Diff
C
C              Diff(4, jD) - Real*8 matrix. In the case of vector finite
C                            elements U = (u, v) the following ordering
C                            should be used:
C                            u_x, u_y, v_x, v_y.
C
C                            REMARK 1: the leading dimension of Diff is 4.
C
C              There are three type of external functions:
C              1. Bilinear form tensors:
C                 The user retuns on the following values: TENSOR_NULL,
C                 TENSOR_SCALAR, TENSOR_SYMMETRIC or TENSOR_GENERAL
C                 (see description below).
C
C              2. Boundary conditions:
C                 Parameter Diff means the value of the essential boundary
C                 condition at point (x,y) which is usually the middle
C                 point of a mesh object (point, face or edge).
C                 The user returns one of the following values: BC_NULL,
C                 BC_DIRICHLET, BC_NEUMANN, BC_ROBIN.
C
C              3. Right hand sides:
C                 The user retuns on the following values: TENSOR_NULL,
C                 TENSOR_SCALAR, TENSOR_SYMMETRIC or TENSOR_GENERAL.
C
C              Examples of finction D may be found in file user.f.
C
C
C     DATA    - Real*8 user given data (a number or an array)
C
C     order - the order of a quadrature formula:
C             order = 1 - quadrature formula with one center point
C             order = 2 - quadrature formula with 3 points inside triangle
C
C
C *** SPARSE MATRIX ***
C   By default, our sparse matrix is assembled in one of the row formats
C   (IA, JA, DA, A) specified by assembleStatus. Other formats are supported
C   thorough the library of format transformations.
C
C     assembleStatus - some a priory information about the matrix A:
C
C                      MATRIX_SYMMETRIC - symmetric matrix
C                      MATRIX_GENERAL   - general matrix
C
C                      FORMAT_AMG       - format used in AMG
C                      FORMAT_CSR       - compressed row format 
C                                         (diagonal is not saved)
C
C                      REMARK: Any logical AND can be used to defined
C                              the variable, possible contradictions will
C                              be checked by the code.
C
C     MaxA      - the maximal size of array A - the maximal
C                 number of nonzero entries
C
C     IA, JA, DA, A - sparcity structure of matrix A:
C
C                     IA(k + 1) - IA(k) equals to the number of
C                                    nonzero entries in the k-th row
C
C                     JA(M)        - list of column indexes of non-zero
C                                    entries ordered by rows;
C                                    M = IA(nRow + 1) - 1
C
C                     A(M)         - non-zero entries ordered as in JA
C
C                     DA(nRow)     - main diagonal of A
C
C     nRow      - the number of rows in A
C     nCol      - the number of columns in A, nCol = nRow for
C                 symmetric bilinear forms.
C
C
C *** WORKING MEMORY ***
C     MaxWi  - the size of the working integer array
C
C     iW(MaxWi)  - the integer working array. On output it contains:
C                  L          = iW(1) - the leading dimension of IAE,
C                  IAE(L, nE) = iW(2) - connectivity list for triangles
C                                       and degrees of freedom (points,
C                                       faces, or edges).
C
C                  REMARK: L * nE memory cells are reserved for other
C                          routines assuming that they are called
C                          after computing the stiffness matrix.
C
C ================================================================
C *** A short description of available tensors:
C
C               TENSOR_NULL      - identity tensor.
C                                  Example: isotropic diffusion problem in
C                                  homogeneous domains.
C
C               TENSOR_SCALAR    - piecewise constant scalar tensor
C                                  Example: isotropic diffusion problem in
C                                  heterogeneous domains.
C
C               TENSOR_SYMMETRIC - symmetric tensor
C                                  Example: anisotropic diffusion problem in
C                                  heterogeneous/homegeneous domains.
C
C               TENSOR_GENERAL   - general tensor
C
C ================================================================
C *** Note:
C       Input parameters:  nP, nE, XYP,
C                          operatorA, FEMtypeA, operatorB, FEMtypeB,
C                          D, DATA, order,
C                          assembleStatus, MaxIA, MaxA,
C                          MaxWi, iW
C
C       Input / output:    IPE, iW
C
C       Output parameters: IA, JA, A, nRow, nCol
C
C ================================================================
C
C *** Authors: K. Lipnikov (lipnikov@hotmail.com)
C              Yu. Vassilevski (vasilevs@dodo.inm.ras.ru)
C *** Date:   2000 - 2002
C *** Update: 19.01.02 - version 1.0 (sequential code is expected)
C *** Complains & Comments: lipnikov@hotmail.com
C *** External routines: AMG1R5, D
C
C ================================================================
C  A short version (BilinearFormVolumeShort) will be available
C  in the future.
C
C ==========================================================
      Real*8   XYP(2, *)
      Integer  IPE(3, *), lbE(*)

      Integer  FEMtypeA, FEMtypeB, operatorA, operatorB
      Integer  order, D

      Real*8   DATA(*)
      EXTERNAL D

      Integer  IA(*), JA(*), assembleStatus
      Real*8   A(*), DA(*)

      Integer  iW(*)

C ================================================================
C Local variables
      Real*8  Aloc(MaxSize, MaxSize)

      Integer label
      Logical FEMtype, flagMS, flagMG, flagFA, flagFR
 
      Character*80 message

C ================================================================
      If(FEMtypeA.EQ.FEM_P2 .OR. FEMtypeB.EQ.FEM_P2)
     &   Call errMes(2001, 'BilinearFormVolume',
     &               'FEM_P2 is not supported')

c ... default values for flags
      flagMG = .TRUE.

      flagMS = IAND(assembleStatus, MATRIX_SYMMETRIC).NE.0
      flagMG = IAND(assembleStatus, MATRIX_GENERAL).NE.0

      flagFA = IAND(assembleStatus, FORMAT_AMG).NE.0
      flagFR = IAND(assembleStatus, FORMAT_CSR).NE.0

      If(flagMS .AND. flagMG .OR.
     &   flagFA .AND. flagFR)
     &     Call errMes(2001, 'BilinearFormVolume',
     &                 'assembleStatus is wrong')

      Call order2D(3, nE, IPE)

      FEMtype = FEMtypeA .EQ. FEMtypeB


c ... memory distribution
      iL = 1
      iIAE = iL + 1
      inEP = iIAE + 3 * nE
      iIEP = inEP + MaxIA + 1
      iIP1 = iIEP + 3 * nE
      iEnd = iIP1 + MaxIA + 1
      If(iEnd.GT.MaxWi) Then

         Write(message,'(A,I10)')
     &        'The approximate size of iW is ', iEnd
         Call errMes(1001, 'BilinearFormVolume', message)
      End if


      If(FEMtypeA.EQ.FEM_P1) Then
         L = 3
         Call copy2D(L, nE, IPE, iW(iIAE))
         nRow = nP
      Else If(FEMtypeA.EQ.FEM_RT0) Then
         L = 3
         Call makMf(nP, nF, nE, IPE, iW(iIAE), iW(inEP), iW(iIEP))
         nRow = nF
      End if

      If(nRow.GT.MaxIA) Call errMes(1013, 'BilinearFormVolume',
     &                             'local parameter MaxIA is small')

      Call backReferences(nRow, nE, L, L, iW(iIAE), iW(inEP), iW(iIEP))


      If(FEMtype) Then
         nCol = nRow
         iIBE = iIAE
         M = L
      Else
         Call errMes(6001, 'BilinearFormVolume',
     &              'rectangular matrices are not supported')
      End if


c ... shifting some of the memory pointers
      iIAE = iIAE - 1
      iIBE = iIBE - 1
      inEP = inEP - 1
      iIEP = iIEP - 1


c ... the symbolic assembling of the matrix (nRow = nCol)
      Do n = 1, nRow
         iW(iIP1 + n) = 0
      End do

      IA(1) = 1

      i2 = 0
      m2 = 0
      Do n = 1, nRow
         m1 = m2

         i1 = i2 + 1
         i2 = iW(inEP + n)
         If(m2 + L * (i2 - i1) + 1.GT.MaxA) Call errMes(1014,
     &        'BilinearFormVolume', 'local parameter MaxA is small')

         If(flagFA .AND. n.LE.nCol) Then
            m2 = m2 + 1
            JA(m2) = n
            iW(iIP1 + n) = 1
         End if

         Do i = i1, i2
            iE = iW(iIEP + i)
            Do 100 j = 1, L
               iP = iW(iIAE + (iE - 1) * L + j)

               If(flagMS .AND. iP.LT.n) Goto 100

               If(iW(iIP1 + iP).EQ.0) Then
                  iW(iIP1 + iP) = 1

                  m2 = m2 + 1
                  JA(m2) = iP
               End if
 100        Continue
         End do
         IA(n + 1) = m2 + 1

         Do i = m1 + 1, m2
            iW(iIP1 + JA(i)) = 0
         End do
      End do


C ... fill in the sparcity structure
      Do n = 1, IA(nRow + 1)
         A(n) = 0D0
      End do

      Do n = 1, nRow
         DA(n) = 0D0
      End do

      Do n = 1, nE
         iP1 = IPE(1, n)
         iP2 = IPE(2, n)
         iP3 = IPE(3, n)

         label = lbE(n)
         Call FEM2Dtri(XYP(1,iP1), XYP(1,iP2), XYP(1,iP3), 
     &                 operatorA, FEMtypeA, operatorB, FEMtypeB,
     &                 label, D, DATA, order,
     &                 MaxSize, Aloc, ir, ic)

         Do i = 1, ir
            iP = iW(iIAE + (n - 1) * L + i)
            DA(iP) = DA(iP) + Aloc(i, i)

            js = 1
            If(flagFR .AND. flagMS) js = i + 1

            Do 200 j = js, ic
               iQ = iW(iIBE + (n - 1) * M + j)
               Do k = IA(iP), IA(iP + 1) - 1
                  If(JA(k).EQ.iQ) Then
                     A(k) = A(k) + Aloc(i, j)
                     Goto 200
                  End if
               End do
 200        Continue
         End do
      End do


c ... output of the connectivity list (only for symmetric problems)
      iW(1) = L

      Return
      End



C ================================================================
      Subroutine addBilinearFormVolume(
C ================================================================
     &           nP, nE, XYP, IPE, lbE,
     &           operatorA, FEMtypeA, operatorB, FEMtypeB,
     &           D, DATA, order,
     &           assembleStatus, MaxIA, MaxA,
     &                           IA, JA, DA, A, nRow, nCol,
     &           MaxWi, iW)
C ================================================================
      Include 'fem2Dtri.fd'
      Include 'assemble.fd'
C ================================================================
      Return
      End



C ================================================================
      Subroutine LinearFormVolume(
C ================================================================
     &           nP, nE, XYP, IPE, lbE,
     &           FEMtypeA,
     &           D, DATA, order,
     &           F, nRow,
     &           MaxWi, iW)
C ================================================================
      Include 'fem2Dtri.fd'
      Include 'assemble.fd'
C ================================================================
C  The routine computes the right hand side for linear form (3).
C ================================================================
      Real*8   XYP(2, *)
      Integer  IPE(3, *), lbE(*)

      Integer  FEMtypeA, order, D

      Real*8   DATA(*)
      EXTERNAL D

      Real*8   F(*)

      Integer  iW(*)

C ================================================================
C Local variables
      Real*8  Floc(MaxSize, MaxSize)

C ================================================================
c ... memory distribution 
      iL = 1
      L  = iW(1)

      iIAE = iL + 1
      iEnd = iIAE + L * nE


      Do n = 1, nRow
         F(n) = 0D0
      End do


      Do n = 1, nE
         iP1 = IPE(1, n)
         iP2 = IPE(2, n)
         iP3 = IPE(3, n)

         label = lbE(n)
         Call FEM2Dtri(XYP(1,iP1), XYP(1,iP2), XYP(1,iP3), 
     &                 IDEN, FEMtypeA, IDEN, FEM_P0,
     &                 label, D, DATA, order,
     &                 MaxSize, Floc, ir, ic)

         Do i = 1, ir
            m = iW(iIAE + L * (n - 1) + i - 1)
            F(m) = F(m) + Floc(i, 1)
         End do
      End do

      Return
      End



C ================================================================
      Subroutine BoundaryConditions(
C ================================================================
     &           nP, nF, nE, XYP, IPF, IPE, 
     &           FEMtypeA,
     &           D, DATA,
     &           IA, JA, DA, A, F, nRow,
     &           MaxWi, iW)
C ================================================================
      Include 'assemble.fd'
      Include 'fem2Dtri.fd'
C ================================================================
C  The routine imposes essential boundary conditions.
C ================================================================
      Real*8   XYP(2, *)
      Integer  IPF(4, *), IPE(3, *)

      Integer  FEMtypeA, D

      Real*8   DATA(*)
      EXTERNAL D

      Integer  IA(*), JA(*)
      Real*8   A(*), DA(*), F(*)

      Integer  iW(*)

C ================================================================
C Local variables
      Integer dof, ipr(2, 3), IAA(3), iSYS(2), p
      Real*8  XYA(2,2), XYE(2), eBC(2)
      Real*8  Aloc(MaxSize, MaxSize)
      Character*80 message

      DATA ipr /1,2, 1,3, 2,3/

C ================================================================
c ... memory distribution
      iL = 1
      L  = iW(1)

      iIAE = iL + 1
      iIFE = iIAE + L * nE
      inEP = iIFE + 3 * nE
      iIEP = inEP + nRow
      iEnd = iIEP + L * nE

      If(iEnd.GT.MaxWi) Then
         Write(message,'(A,I10)')
     &        'The approximate size of iW is ', iEnd
         Call errMes(1001, 'BoundaryCondition', message)
      End if

      Call makMb(nP, nF, nE, IPF, IPE, iW(iIFE), iW(inEP), iW(iIEP))

      Do n = 1, nE
         Do 100 i1 = 1, 3
            iF = iW(iIFE + 3 * (n - 1) + i1 - 1)
            If(iF.EQ.0) Goto 100

            iP1 = IPF(1, iF)
            iP2 = IPF(2, iF)
            Do i = 1, 2
               XYE(i) = (XYP(i, iP1) + XYP(i, iP2)) / 2
            End do

            If(FEMtypeA.EQ.FEM_P1) Then
               dof = 2
               Do k = 1, dof
                  Do i = 1, 2
                     XYA(i, k) = XYP(i, IPF(k, iF))
                  End do

                  IAA(k) = IPF(k, iF)
               End do

            Else If(FEMtypeA.EQ.FEM_RT0) Then
               dof = 1
               Do i = 1, 2
                  XYA(i, 1) = XYE(i)
               End do

               IAA(1) = iW(iIAE + L * (n - 1) + i1 - 1)
            End if


c  ...  change matrix and right-hand side
            label = IPF(4, iF)
            iBC = D(XYE(1), XYE(2), label, DATA, iSYS, Aloc)

            If(iBC.EQ.BC_DIRICHLET) Then
               Do k = 1, dof
                  iBC = D(XYA(1,k), XYA(2,k), label, DATA, iSYS, eBC)

                  m = IAA(k)
                  F(m) = DA(m) * eBC(1)

                  Do 50 i = IA(m), IA(m + 1) - 1
                     p = JA(i)

                     If(p.NE.m) Then
                        A(i) = 0D0

                        Do j = IA(p), IA(p + 1) - 1
                           If(JA(j).EQ.m) Then
                              F(p) = F(p) - A(j) * eBC(1)
                              A(j) = 0D0

                              Goto 50
                           End if
                        End do
                     End if
 50               Continue
               End do

            Else If(iBC.EQ.BC_NEUMANN) Then
               Call FEM1Dedge(
     &              XYA(1, 1), XYA(1, 2),
     &              IDEN, FEMtypeA, IDEN, FEM_P0,
     &              label, D, DATA, 1, 
     &              MaxSize, Aloc, nR, nC)

               Do k = 1, dof
                  m = IAA(k)
                  F(m) = F(m) + Aloc(k, 1)
               End do
            End if
 100     Continue
      End do

      Return
      End



C ================================================================
      Subroutine order2D(L, nE, IPE)
C ================================================================
C  The routine orders each column of 2D array
C ================================================================
      Integer IPE(L, *)

      Do n = 1, nE
         Do i = 1, L
            im = i
            mv = IPE(i, n)
            Do j = i + 1, L
               If(mv.GT.IPE(j, n)) Then
                  im = j
                  mv = IPE(j, n)
               End if
            End do
            IPE(im, n) = IPE(i,  n)
            IPE(i,  n) = mv
         End do
      End do

      Return
      End




C ================================================================
      Subroutine copy2D(L, nE, IPE1, IPE2)
C ================================================================
C  The routine copies IPE1 into IPE2
C ================================================================
      Integer IPE1(L, *), IPE2(L, *)

      Do n = 1, nE
         Do i = 1, L
            IPE2(i, n) = IPE1(i, n)
         End do
      End do

      Return
      End
