************************************************************************
*
* PCG: Preconditioned Conjugate Gradient Package
* Version: f77
*
************************************************************************
************************************************************************
*# 
*#                      COPYRIGHT/DISCLAIMER NOTICE
*#
*# This program was prepared by the Regents of the University of
*# California at Los Alamos National Laboratory under Contract No.
*# W-7405-ENG-36 with the U.S. Department of Energy (DOE), and by
*# the University of Texas at Austin under ARPA Contract No.
*# DABT63-92-C-0024.
*# 
*# The University of California and the University of Texas at
*# Austin have certain rights in the program pursuant to these
*# contracts.
*# 
*# Permission is hereby granted to use the program for the user's
*# own internal use.  The user is not granted the right to reproduce,
*# prepare derivative works, or redistribute the program without
*# prior permission of the University of California or the University
*# of Texas at Austin.
*# 
*# All rights in the program are reserved by the University of
*# California and the University of Texas at Austin.
*# 
*# Portions of this material resulted from work developed under a
*# U.S. Government Contract and are subject to the following
*# license: the Government is granted for itself and others acting
*# on its behalf a paid-up, nonexclusive, irrevocable worldwide
*# license in this computer software to reproduce, prepare derivative
*# works, and perform publicly and display publicly.
*# 
*# Neither the U.S. Government, the University of California nor
*# the University of Texas at Austin, nor any of their employees,
*# makes any warranty, express or implied, or assumes any liability
*# or responsibility for the use of this software.
*# 
*# Copyright (c) 1992-1995 the University of California and the
*# University of Texas at Austin.
*# 
************************************************************************
************************************************************************
************************************************************************
**  ^ROUTINE: WGMCHR - Reverse communication level routine for
**                     restarted Chebyshev-basis GMRES acceleration. (^)
**
************************************************************************
**
      subroutine zgmchr ( ijob , ireq ,
     &       u , uexact , b ,
     &       iva , ivql , ivqr ,
     &       iwk , fwk , iparm , fparm , ier )
**
************************************************************************
**
**  ^DESCRIPTION:
**    Restarted Chebyshev-basis GMRES iterative method.  This routine
**    implements GMRES acceleration, which minimizes Q_L(b-Au^{(n)})
**    over an appropriate Krylov space.  This implementation uses a
**    Chebyshev basis for the Krylov space, which is faster and more
**    parallel but might be less well-conditioned.
**
**  ^AUTHOR:   wdj@beta.lanl.gov
**
**  ^MODIFIED: wdj@lithos.c3.lanl.gov Thu May  6 13:12:38 MDT 1993
**
**  ^ARGUMENTS: see *Subroutine Arguments* below.
**
**  ^REQUIREMENTS:
**    Common Blocks: none
**    Subroutines:   see below.
**
**  ^SIDE_EFFECTS:
**    See the definition of METHRDCLS to see which parameters
**    are modified
**
**  ^ALGORITHM:
**
**  ^REFERENCES:
**
**  Wayne D. Joubert and Graham F. Carey, `` Parallelizable Restarted
**  Iterative Methods for Nonsymmetric Linear Systems.  Part I:
**  Theory,'' {\sl International Journal of Computer Mathematics},
**  vol. 44, pp. 243--267 (1992).
**
**  Wayne D. Joubert and Graham F. Carey, `` Parallelizable Restarted
**  Iterative Methods for Nonsymmetric Linear Systems.  Part II:
**  Parallel Implementation,'' {\sl International Journal of Computer
**  Mathematics}, vol. 44, pp. 269--290 (1992).
**
**  ^DOCUMENTATION:
**
**  ^MACROS:
**
************************************************************************
*
*     implicit character*1 (a-z)
*
*#    METHRAL        - (^)
      integer ijob
      integer ireq
      integer ier
      integer iwk(*)
      integer iparm(*)
      double complex       fwk(*)
      double complex       fparm(*)
      double complex u(*)
      double complex uexact(*)
      double complex b(*)
      integer ivql
      integer iva
      integer ivqr
*
        integer itimer
        double precision tc1, tr1, tc2, tr2
      integer    iitab
      integer    iftab
      integer    iwkstg
      integer    iclstg
      integer    in
      integer    is
      integer    iretlb
      integer    ivitmp, ivotmp
      integer    iv1tst, iv2tst
      integer    istab1, istab2, istab
      integer    ib1,ib2, iu1,iu2
      integer    iud
      parameter (iud  =0)
      integer    ibd
      parameter (ibd  =1)
      integer    ilbd
      parameter (ilbd =2)
      integer    irlbd
      parameter (irlbd=3)
      integer    ied
      parameter (ied  =4)
      integer    ird
      parameter (ird  =5)
      integer    ilrd
      parameter (ilrd =6)
      integer    irlrd
      parameter (irlrd=7)
      double complex      urndf
      double complex      top
      double complex      bot
      parameter (urndf=.1)
      double complex      mydot1, mydot2
      external   zargck
      external   zckstg
      external   zwrnd
      external   zifppr
      external   zitput
      external   xtimer
        integer ipme
        integer nproc
      external ximini
      external ximal
      external xifre
      external zfmini
      external zfmal
      external zffre
*     integer imv1, imv2
      integer    ir
      integer    iqlr
      integer    iqrqlr
      integer    ip
      integer    iqlp
      integer    iqrqlp
      integer    iv1
      integer    iv2
      integer    ipf, icos, isin, ih, iu, izc, iy, iyold, ipz, ipzold
      integer    ns
      integer    i, j
      integer    ipp, ihpp, ihpph, ihpr
      integer    ivtmat, ismat, iwkla, infola, itmpv, idiag, nrank
      integer    iww, iwr, iwi
      integer    nm
      double complex      rdot
      double complex      rnorm
      double complex      r0norm
      double complex      coe1, coe2, coe3
      double complex      d, c2, gam
      double complex      h10
      double complex      vrmin, vrmax, vimin, vimax, al, be
      double precision      epsm
      double complex      uudot, uexdt
      double complex      vnorm, pap, vdot, ut, h, v1, v2, denom, sum, v
     &al
*
      double precision    zmch
      external zmch
*
*
*
****^^******************************************************************
*     $Modified: wdj@lanl.gov Mon Aug 28 19:20:55 MDT 1995
*     $Id: gmchr.fm4,v 1.14 1994/11/22 05:20:25 joubert Exp $
*     $Revision: 1.0 $
*     $Log: gmchr.fm4,v $
************************************************************************
*
      iitab = (iparm(6))
      iftab = (iparm(6))
      ir = (iparm(6))
      iqlr = (iparm(6))
      iqrqlr = (iparm(6))
      ip = (iparm(6))
      iqlp = (iparm(6))
      iqrqlp = (iparm(6))
      iv1 = (iparm(6))
      iv2 = (iparm(6))
      ipf = (iparm(6))
      icos = (iparm(6))
      isin = (iparm(6))
      ih = (iparm(6))
      iu = (iparm(6))
      izc = (iparm(6))
      iy = (iparm(6))
      iyold = (iparm(6))
      ipz = (iparm(6))
      ipzold = (iparm(6))
      ipp = (iparm(6))
      ihpp = (iparm(6))
      ihpph = (iparm(6))
      ihpr = (iparm(6))
      ivtmat = (iparm(6))
      ismat = (iparm(6))
      iwkla = (iparm(6))
      itmpv = (iparm(6))
      idiag = (iparm(6))
*
*     *---go to the brancher unless this is the init call---
      if (ijob .ne. 1) go to 1000
*
*     *---inits----------
      iclstg = 1
      itimer = 0
      in = 0
      iparm(5) = in
      is = 0
*
*     *---initialize pointers----------
      iu1 = (iparm(6))
      iu2 = (iparm(6))
      ib1 = (iparm(6))
      ib2 = (iparm(6))
      istab1 = (iparm(6))
      istab2 = (iparm(6))
*
*     *---initialize----------
      ipme  = 0
      nproc = 1
      if (iparm(27) .eq. 0) iparm(27) = 0
      if (iparm(27) .eq. 0) then
        ier = 0
        if (ijob.eq.1 .or. ijob.eq.2 .or. ijob.eq.0) then
*       *---init integer memory----------
        call ximini ( iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
*       *---init floating point memory----------
        call zfmini ( iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        endif
*       *---check arguments----------
        call zargck ( iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
*       *---initialize timer----------
        call xtimer (tc1,tr1,iparm(22),1)
      endif
*
*     *---start timer------
      call xtimer (tc1,tr1,iparm(22),0)
      itimer = 1
*
*     *---print i/fparm if requested----------
      if (iparm(27) .eq. 0) then
        call zifppr ( 1 , iparm, fparm, ier)
           if (ier .lt. 0) go to 900
      endif
*
*     *---reserve place to save integer and float variables----------
*     *---across revcom calls----------
      call ximal (iitab,55, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (iftab,19, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
*
      if (iparm(2) .ge. 3) then
        write (iparm(1),'(/''     Restarted Chebyshev-basis GMRES Method
     &.''/)') 
      endif
*
      ns = min(iparm(17),iparm(4))
*
      call zfmal (iqlr,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (iqlp,iparm(3)*(ns+1), iparm , fparm , iwk , fwk , ier 
     &)
           if (ier .lt. 0) go to 900
      if (((iparm(13).eq.1).or.(iparm(13).eq.3))) then
        if (((mod(iparm(15)/2,2).eq.1)     .or.
     &    iparm(12).eq.2     .or. iparm(12).eq.6)) then
          call zfmal (ir,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          call zfmal (ip,iparm(3)*(ns+1), iparm , fparm , iwk , fwk , ie
     &r )
           if (ier .lt. 0) go to 900
        endif
      else
        ir = iqlr
        ip = iqlp
      endif
      if (((iparm(13).eq.2).or.(iparm(13).eq.3))) then
        call zfmal (iqrqlr,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        call zfmal (iqrqlp,iparm(3)*(ns+1), iparm , fparm , iwk , fwk , 
     &ier )
           if (ier .lt. 0) go to 900
      else
        iqrqlr = iqlr
        iqrqlp = iqlp
      endif
      call zfmal (iv1,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (iv2,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (ipf   ,ns+1, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (icos  ,ns+1, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (isin  ,ns+1, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (ih    ,ns*(ns+1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (iu    ,ns*(ns+1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (izc   ,ns+1, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (iy    ,ns, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (iyold ,ns, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (ipz   ,ns+1, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (ipzold,ns+1, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (ipp   ,(ns+1)*(ns+1), iparm , fparm , iwk , fwk , ier 
     &)
           if (ier .lt. 0) go to 900
      call zfmal (ihpp  ,(ns  )*(ns+1), iparm , fparm , iwk , fwk , ier 
     &)
           if (ier .lt. 0) go to 900
      call zfmal (ihpph ,(ns  )*(ns  ), iparm , fparm , iwk , fwk , ier 
     &)
           if (ier .lt. 0) go to 900
      call zfmal (ihpr  ,(ns  ), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (ivtmat,(ns  )*(ns  ), iparm , fparm , iwk , fwk , ier 
     &)
           if (ier .lt. 0) go to 900
      call zfmal (ismat ,(ns  ), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (iwkla ,((5*ns)+5*ns ), iparm , fparm , iwk , fwk , ier
     & )
           if (ier .lt. 0) go to 900
      call zfmal (itmpv ,(ns  ), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (idiag ,(ns  ), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      iww = ismat
      iwr = itmpv
      iwi = idiag
*
*
*     *---set u to zero----------
      if (iparm(14) .eq. -2
     &    .or. iparm(14) .eq. -1) then
          call zwfil (u,(0d0,0.0d0),
     &     iparm , fparm , iwk , fwk , ier  )
      endif
*
*     *---random u----------
      if (iparm(14) .eq. 2
     &    ) go to 5500
                 go to 5501
 5500  continue
          call zwrdot (mydot1,b,
     &    b,  iparm , fparm , iwk , fwk , ier  )
        call zwrnd (u,  iparm , fparm , iwk , fwk , ier  )
*        *---begin revcom matvec----------
          call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call zwset (fwk(ivqr),u,
     &     iparm , fparm , iwk , fwk , ier  )
          iva = iv1 
         ireq = 3
        iretlb = 2000
        go to 1500
 2000   continue
          call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , i
     &er )
           if (ier .lt. 0) go to 900
         if (ijob .eq. -1) go to 900
*        *---end revcom matvec----------
          call zwrdot (mydot2,fwk(iv1),
     &    fwk(iv1),  iparm , fparm , iwk , fwk , ier  )
          call zwscl ( u, u, urndf*sqrt(mydot1/mydot2),  iparm , fparm ,
     & iwk , fwk , ier  )
                go to 5502
 5501 continue
 5502 continue
*
*     *---random u perturbation----------
      if (iparm(14) .eq. 3) go to 5503
                 go to 5504
 5503  continue
*        *---begin revcom matvec----------
          call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call zwset (fwk(ivqr),u,
     &     iparm , fparm , iwk , fwk , ier  )
          iva = iv1 
         ireq = 3
        iretlb = 2001
        go to 1500
 2001   continue
          call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , i
     &er )
           if (ier .lt. 0) go to 900
         if (ijob .eq. -1) go to 900
*        *---end revcom matvec----------
          call zwsub ( fwk(iv1),
     &    b, fwk(iv1),  iparm , fparm , iwk , fwk , ier  )
          call zwrdot (mydot1,fwk(iv1),
     &    fwk(iv1),  iparm , fparm , iwk , fwk , ier  )
          call zwset (fwk(iv1),u,
     &     iparm , fparm , iwk , fwk , ier  )
        call zwrnd (u,  iparm , fparm , iwk , fwk , ier  )
*        *---begin revcom matvec----------
          call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call zwset (fwk(ivqr),u,
     &     iparm , fparm , iwk , fwk , ier  )
          iva = iv2 
         ireq = 3
        iretlb = 2002
        go to 1500
 2002   continue
          call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , i
     &er )
           if (ier .lt. 0) go to 900
         if (ijob .eq. -1) go to 900
*        *---end revcom matvec----------
          call zwrdot (mydot2,fwk(iv2),
     &    fwk(iv2),  iparm , fparm , iwk , fwk , ier  )
          call zwsax (u,fwk(iv1),
     &    u, -urndf*sqrt(mydot1/mydot2),  iparm , fparm , iwk , fwk , ie
     &r  )
                go to 5505
 5504 continue
 5505 continue
*
*     *---begin iteration statistics calculation----------
      if (iparm(21) .eq. 1) go to 5506
                 go to 5507
 5506  continue
        call zfmal (istab1,8, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        call zfmal (istab2,8, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        call zfmal (ib1,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        call zfmal (ib2,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        call zfmal (iu1,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        call zfmal (iu2,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        istab = istab1
        if (iparm(19) .eq. 1) then
          call zwdot (fwk(istab+iud), uexact,
     &    uexact,  iparm , fparm , iwk , fwk , ier  )
        endif
          call zwdot (fwk(istab+ibd ), b,
     &    b,  iparm , fparm , iwk , fwk , ier  )
*        *---begin revcom precon----------
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5509
                 go to 5510
 5509  continue
          call zwset (fwk(ib2),b,
     &     iparm , fparm , iwk , fwk , ier  )
                go to 5511
 5510 continue
          call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          iva = ivitmp
          call zwset (fwk(iva),b,
     &     iparm , fparm , iwk , fwk , ier  )
          ivql = ib2 
         ireq = 5
        iretlb = 2003
        go to 1500
 2003   continue
          call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , i
     &er )
           if (ier .lt. 0) go to 900
         if (ijob .eq. -1) go to 900
 5511 continue
*        *---end revcom precon----------
          call zwdot (fwk(istab+ilbd), fwk(ib2),
     &    fwk(ib2),  iparm , fparm , iwk , fwk , ier  )
          call zwset (fwk(iu1),fwk(ib2),
     &     iparm , fparm , iwk , fwk , ier  )
*        *---begin revcom precon----------
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5512
                 go to 5513
 5512  continue
          call zwset (fwk(iu2),fwk(iu1),
     &     iparm , fparm , iwk , fwk , ier  )
                go to 5514
 5513 continue
          ivql = iu1 
          ivqr = iu2 
         ireq = 7
        iretlb = 2004
        go to 1500
 2004   continue
         if (ijob .eq. -1) go to 900
 5514 continue
*        *---end revcom precon----------
          call zwdot (fwk(istab+irlbd), fwk(iu2),
     &    fwk(iu2),  iparm , fparm , iwk , fwk , ier  )
        if (iparm(19) .eq. 1) then
          call zwsub ( fwk(iu1),
     &    u, uexact,  iparm , fparm , iwk , fwk , ier  )
          call zwdot (fwk(istab+ied), fwk(iu1),
     &    fwk(iu1),  iparm , fparm , iwk , fwk , ier  )
        endif
*        *---begin revcom matvec----------
          call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call zwset (fwk(ivqr),u,
     &     iparm , fparm , iwk , fwk , ier  )
          iva = ib1 
         ireq = 3
        iretlb = 2005
        go to 1500
 2005   continue
          call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , i
     &er )
           if (ier .lt. 0) go to 900
         if (ijob .eq. -1) go to 900
*        *---end revcom matvec----------
          call zwsub ( fwk(ib1),
     &    b, fwk(ib1),  iparm , fparm , iwk , fwk , ier  )
          call zwdot (fwk(istab+ird), fwk(ib1),
     &    fwk(ib1),  iparm , fparm , iwk , fwk , ier  )
*        *---begin revcom precon----------
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5515
                 go to 5516
 5515  continue
          call zwset (fwk(ib2),fwk(ib1),
     &     iparm , fparm , iwk , fwk , ier  )
                go to 5517
 5516 continue
          iva = ib1 
          ivql = ib2 
         ireq = 5
        iretlb = 2006
        go to 1500
 2006   continue
         if (ijob .eq. -1) go to 900
 5517 continue
*        *---end revcom precon----------
          call zwdot (fwk(istab+ilrd ), fwk(ib2),
     &    fwk(ib2),  iparm , fparm , iwk , fwk , ier  )
          call zwset (fwk(iu1),fwk(ib2),
     &     iparm , fparm , iwk , fwk , ier  )
*        *---begin revcom precon----------
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5518
                 go to 5519
 5518  continue
          call zwset (fwk(iu2),fwk(iu1),
     &     iparm , fparm , iwk , fwk , ier  )
                go to 5520
 5519 continue
          ivql = iu1 
          ivqr = iu2 
         ireq = 7
        iretlb = 2007
        go to 1500
 2007   continue
         if (ijob .eq. -1) go to 900
 5520 continue
*        *---end revcom precon----------
          call zwdot (fwk(istab+irlrd), fwk(iu2),
     &    fwk(iu2),  iparm , fparm , iwk , fwk , ier  )
        if (iparm(2) .ge. 4) then
        write (iparm(1),'()') 
        write (iparm(1),'(1x, 60(''*''))') 
        write (iparm(1),'()') 
        write (iparm(1),'(5x,''Initial iteration statistics'')') 
        write (iparm(1),'()') 
        if (iparm(19) .eq. 1) then
        write (iparm(1),'(5x,''2-norm of uexact            '',1pg20.10,2
     &x,1pg20.10)') sqrt(fwk(istab+iud))
        endif
        write (iparm(1),'(5x,''2-norm of b                 '',1pg20.10,2
     &x,1pg20.10)') sqrt(fwk(istab+ibd))
        write (iparm(1),'(5x,''2-norm of Ql*b              '',1pg20.10,2
     &x,1pg20.10)') sqrt(fwk(istab+ilbd))
        write (iparm(1),'(5x,''2-norm of Qr*Ql*b           '',1pg20.10,2
     &x,1pg20.10)') sqrt(fwk(istab+irlbd))
        if (iparm(19) .eq. 1) then
        write (iparm(1),'(5x,''2-norm of error             '',1pg20.10,2
     &x,1pg20.10)') sqrt(fwk(istab+ied))
        endif
        write (iparm(1),'(5x,''2-norm of residual          '',1pg20.10,2
     &x,1pg20.10)') sqrt(fwk(istab+ird))
        write (iparm(1),'(5x,''2-norm of Ql*r              '',1pg20.10,2
     &x,1pg20.10)') sqrt(fwk(istab+ilrd))
        write (iparm(1),'(5x,''2-norm of Qr*Ql*r           '',1pg20.10,2
     &x,1pg20.10)') sqrt(fwk(istab+irlrd))
        write (iparm(1),'()') 
        write (iparm(1),'(1x, 60(''*''))') 
        write (iparm(1),'()') 
        endif
        call zffre (iu2,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        call zffre (iu1,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        call zffre (ib2,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        call zffre (ib1,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
                go to 5508
 5507 continue
 5508 continue
*     *---end iteration statistics calculation----------
*
      if (((iparm(14).eq.0).or.(iparm(14).eq.-2)) .or. iparm(14).eq.-1) 
     &go to 5521
                 go to 5522
 5521  continue
        if (((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5524
                 go to 5525
 5524  continue
          if (((mod(iparm(15)/2,2).eq.1)     .or.
     &    iparm(12).eq.2     .or. iparm(12).eq.6)) go to 5527
                 go to 5528
 5527  continue
          call zwset (fwk(ir),b,
     &     iparm , fparm , iwk , fwk , ier  )
*        *---begin revcom precon----------
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5530
                 go to 5531
 5530  continue
          call zwset (fwk(iqlr),fwk(ir),
     &     iparm , fparm , iwk , fwk , ier  )
                go to 5532
 5531 continue
          iva = ir 
          ivql = iqlr 
         ireq = 5
        iretlb = 2008
        go to 1500
 2008   continue
         if (ijob .eq. -1) go to 900
 5532 continue
*        *---end revcom precon----------
                go to 5529
 5528 continue
*        *---begin revcom precon----------
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5533
                 go to 5534
 5533  continue
          call zwset (fwk(iqlr),b,
     &     iparm , fparm , iwk , fwk , ier  )
                go to 5535
 5534 continue
          call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          iva = ivitmp
          call zwset (fwk(iva),b,
     &     iparm , fparm , iwk , fwk , ier  )
          ivql = iqlr 
         ireq = 5
        iretlb = 2009
        go to 1500
 2009   continue
          call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , i
     &er )
           if (ier .lt. 0) go to 900
         if (ijob .eq. -1) go to 900
 5535 continue
*        *---end revcom precon----------
 5529 continue
                go to 5526
 5525 continue
          call zwset (fwk(iqlr),b,
     &     iparm , fparm , iwk , fwk , ier  )
 5526 continue
                go to 5523
 5522 continue
        if (((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5536
                 go to 5537
 5536  continue
          if (((mod(iparm(15)/2,2).eq.1)     .or.
     &    iparm(12).eq.2     .or. iparm(12).eq.6)) go to 5539
                 go to 5540
 5539  continue
*        *---begin revcom matvec----------
          call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call zwset (fwk(ivqr),u,
     &     iparm , fparm , iwk , fwk , ier  )
          iva = ir 
         ireq = 3
        iretlb = 2010
        go to 1500
 2010   continue
          call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , i
     &er )
           if (ier .lt. 0) go to 900
         if (ijob .eq. -1) go to 900
*        *---end revcom matvec----------
          call zwsub ( fwk(ir),
     &    b, fwk(ir),  iparm , fparm , iwk , fwk , ier  )
*        *---begin revcom precon----------
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5542
                 go to 5543
 5542  continue
          call zwset (fwk(iqlr),fwk(ir),
     &     iparm , fparm , iwk , fwk , ier  )
                go to 5544
 5543 continue
          iva = ir 
          ivql = iqlr 
         ireq = 5
        iretlb = 2011
        go to 1500
 2011   continue
         if (ijob .eq. -1) go to 900
 5544 continue
*        *---end revcom precon----------
                go to 5541
 5540 continue
*        *---begin revcom matvec----------
          call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call zwset (fwk(ivqr),u,
     &     iparm , fparm , iwk , fwk , ier  )
          iva = iv1 
         ireq = 3
        iretlb = 2012
        go to 1500
 2012   continue
          call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , i
     &er )
           if (ier .lt. 0) go to 900
         if (ijob .eq. -1) go to 900
*        *---end revcom matvec----------
          call zwsub ( fwk(iv1),
     &    b, fwk(iv1),  iparm , fparm , iwk , fwk , ier  )
*        *---begin revcom precon----------
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5545
                 go to 5546
 5545  continue
          call zwset (fwk(iqlr),fwk(iv1),
     &     iparm , fparm , iwk , fwk , ier  )
                go to 5547
 5546 continue
          iva = iv1 
          ivql = iqlr 
         ireq = 5
        iretlb = 2013
        go to 1500
 2013   continue
         if (ijob .eq. -1) go to 900
 5547 continue
*        *---end revcom precon----------
 5541 continue
                go to 5538
 5537 continue
*        *---begin revcom matvec----------
          call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call zwset (fwk(ivqr),u,
     &     iparm , fparm , iwk , fwk , ier  )
          iva = iqlr 
         ireq = 3
        iretlb = 2014
        go to 1500
 2014   continue
          call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , i
     &er )
           if (ier .lt. 0) go to 900
         if (ijob .eq. -1) go to 900
*        *---end revcom matvec----------
          call zwsub ( fwk(iqlr),
     &    b, fwk(ir),  iparm , fparm , iwk , fwk , ier  )
 5538 continue
 5523 continue
      if (((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5548
                 go to 5549
 5548  continue
*        *---begin revcom precon----------
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5551
                 go to 5552
 5551  continue
          call zwset (fwk(iqrqlr),fwk(iqlr),
     &     iparm , fparm , iwk , fwk , ier  )
                go to 5553
 5552 continue
          ivql = iqlr 
          ivqr = iqrqlr 
         ireq = 7
        iretlb = 2015
        go to 1500
 2015   continue
         if (ijob .eq. -1) go to 900
 5553 continue
*        *---end revcom precon----------
                go to 5550
 5549 continue
 5550 continue
*
          call zwrdot (rdot,fwk(iqlr),
     &    fwk(iqlr),  iparm , fparm , iwk , fwk , ier  )
      rnorm  = sqrt(abs(rdot))
      r0norm = rnorm
*
      epsm = zmch('E',  iparm , fparm , iwk , fwk , ier  )
*
*---------------------------iteration loop------------------------------
*
 100  continue
*
        if (iparm(2) .ge. 5) then
        write (iparm(1),500) 'QL r norm           ', rnorm
        endif
*
*     *---begin user stopping test----------
        if (iparm(12).eq.-3) go to 5554
                 go to 5555
 5554  continue
          if ((mod(iparm(15)/1,2).eq.1)) then
          endif
          if ((mod(iparm(15)/2,2).eq.1)) then
          iva = ir 
          endif
          if ((mod(iparm(15)/4,2).eq.1)) then
          ivql = iqlr 
          endif
          if ((mod(iparm(15)/8,2).eq.1)) then
          ivqr = iqrqlr 
          endif
          ireq   = 9
        iretlb = 2016
        go to 1500
 2016   continue
          if ((mod(iparm(15)/8,2).eq.1)) then
          endif
          if ((mod(iparm(15)/4,2).eq.1)) then
          endif
          if ((mod(iparm(15)/2,2).eq.1)) then
          endif
          if ((mod(iparm(15)/1,2).eq.1)) then
          endif
          if (ier .ge. 0) then
            call zitput ( in , is ,  iparm , fparm , iwk , fwk , ier )
          endif
          if (ijob .eq. -1) then
            if (iparm(2).ge.3 .and. iparm(27).eq.0) then
        write (iparm(1),'(/a/)') ' User-requested termination of iterati
     &ve method.'
            endif
            go to 900
          endif
                go to 5556
 5555 continue
 5556 continue
        if (iparm(12) .gt. 1) go to 5557
                 go to 5558
 5557  continue
          if (iparm(12) .eq. 1 ) then
            if (iparm(5) .eq. 0) then
              if (iparm(19) .eq. 0) then
                ier = -5
                call xersho ( ier, 'zgmchr' , iparm ,
     &                        '19 and 12 incompatible' )
                go to 900
              endif
          call zwdot (bot, uexact,
     &    uexact,  iparm , fparm , iwk , fwk , ier  )
            endif
            call zfmal (iv1tst,iparm(3), iparm , fparm , iwk , fwk , ier
     & )
           if (ier .lt. 0) go to 900
          call zwsub ( fwk(iv1tst),
     &    u, uexact,  iparm , fparm , iwk , fwk , ier  )
          call zwdot (top, fwk(iv1tst),
     &    fwk(iv1tst),  iparm , fparm , iwk , fwk , ier  )
            call zffre (iv1tst,iparm(3)*(1), iparm , fparm , iwk , fwk ,
     & ier )
           if (ier .lt. 0) go to 900
          endif
          if (iparm(12) .eq. 2 ) then
            if (iparm(5) .eq. 0) then
          call zwdot (bot, b,
     &    b,  iparm , fparm , iwk , fwk , ier  )
            endif
          call zwdot (top, fwk(ir),
     &    fwk(ir),  iparm , fparm , iwk , fwk , ier  )
          endif
          if (iparm(12) .eq. 3) go to 5560
                 go to 5561
 5560  continue
            if (iparm(5) .eq. 0) go to 5563
                 go to 5564
 5563  continue
              if (((iparm(14).eq.0).or.(iparm(14).eq.-2))) go to 5566
                 go to 5567
 5566  continue
          call zwdot (bot, fwk(iqlr),
     &    fwk(iqlr),  iparm , fparm , iwk , fwk , ier  )
                go to 5568
 5567 continue
                call zfmal (iv1tst,iparm(3), iparm , fparm , iwk , fwk ,
     & ier )
           if (ier .lt. 0) go to 900
*        *---begin revcom precon----------
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5569
                 go to 5570
 5569  continue
          call zwset (fwk(iv1tst),b,
     &     iparm , fparm , iwk , fwk , ier  )
                go to 5571
 5570 continue
          call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          iva = ivitmp
          call zwset (fwk(iva),b,
     &     iparm , fparm , iwk , fwk , ier  )
          ivql = iv1tst 
         ireq = 5
        iretlb = 2017
        go to 1500
 2017   continue
          call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , i
     &er )
           if (ier .lt. 0) go to 900
         if (ijob .eq. -1) go to 900
 5571 continue
*        *---end revcom precon----------
          call zwdot (bot, fwk(iv1tst),
     &    fwk(iv1tst),  iparm , fparm , iwk , fwk , ier  )
                call zffre (iv1tst,iparm(3)*(1), iparm , fparm , iwk , f
     &wk , ier )
           if (ier .lt. 0) go to 900
 5568 continue
                go to 5565
 5564 continue
 5565 continue
          call zwdot (top, fwk(iqlr),
     &    fwk(iqlr),  iparm , fparm , iwk , fwk , ier  )
                go to 5562
 5561 continue
 5562 continue
          if (iparm(12) .eq. 4) go to 5572
                 go to 5573
 5572  continue
            if (iparm(5) .eq. 0) go to 5575
                 go to 5576
 5575  continue
              if (((iparm(14).eq.0).or.(iparm(14).eq.-2))) go to 5578
                 go to 5579
 5578  continue
          call zwdot (bot, fwk(iqrqlr),
     &    fwk(iqrqlr),  iparm , fparm , iwk , fwk , ier  )
                go to 5580
 5579 continue
                call zfmal (iv1tst,iparm(3), iparm , fparm , iwk , fwk ,
     & ier )
           if (ier .lt. 0) go to 900
                call zfmal (iv2tst,iparm(3), iparm , fparm , iwk , fwk ,
     & ier )
           if (ier .lt. 0) go to 900
*        *---begin revcom precon----------
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5581
                 go to 5582
 5581  continue
          call zwset (fwk(iv1tst),b,
     &     iparm , fparm , iwk , fwk , ier  )
                go to 5583
 5582 continue
          call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          iva = ivitmp
          call zwset (fwk(iva),b,
     &     iparm , fparm , iwk , fwk , ier  )
          ivql = iv1tst 
         ireq = 5
        iretlb = 2018
        go to 1500
 2018   continue
          call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , i
     &er )
           if (ier .lt. 0) go to 900
         if (ijob .eq. -1) go to 900
 5583 continue
*        *---end revcom precon----------
*        *---begin revcom precon----------
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5584
                 go to 5585
 5584  continue
          call zwset (fwk(iv2tst),fwk(iv1tst),
     &     iparm , fparm , iwk , fwk , ier  )
                go to 5586
 5585 continue
          ivql = iv1tst 
          ivqr = iv2tst 
         ireq = 7
        iretlb = 2019
        go to 1500
 2019   continue
         if (ijob .eq. -1) go to 900
 5586 continue
*        *---end revcom precon----------
          call zwdot (bot, fwk(iv2tst),
     &    fwk(iv2tst),  iparm , fparm , iwk , fwk , ier  )
                call zffre (iv2tst,iparm(3)*(1), iparm , fparm , iwk , f
     &wk , ier )
           if (ier .lt. 0) go to 900
                call zffre (iv1tst,iparm(3)*(1), iparm , fparm , iwk , f
     &wk , ier )
           if (ier .lt. 0) go to 900
 5580 continue
                go to 5577
 5576 continue
 5577 continue
          call zwdot (top, fwk(iqrqlr),
     &    fwk(iqrqlr),  iparm , fparm , iwk , fwk , ier  )
                go to 5574
 5573 continue
 5574 continue
          if (iparm(12) .eq. 1 ) then
            if (iparm(19) .eq. 0) then
                ier = -5
                call xersho ( ier, 'zgmchr' , iparm ,
     &                        '19 and 12 incompatible' )
                go to 900
            endif
            if (((iparm(14).eq.0).or.(iparm(14).eq.-2))) then
          call zwdot (top, uexact,
     &    uexact,  iparm , fparm , iwk , fwk , ier  )
            else
              call zfmal (iv1tst,iparm(3), iparm , fparm , iwk , fwk , i
     &er )
           if (ier .lt. 0) go to 900
          call zwsub ( fwk(iv1tst),
     &    u, uexact,  iparm , fparm , iwk , fwk , ier  )
          call zwdot (top, fwk(iv1tst),
     &    fwk(iv1tst),  iparm , fparm , iwk , fwk , ier  )
              call zffre (iv1tst,iparm(3)*(1), iparm , fparm , iwk , fwk
     & , ier )
           if (ier .lt. 0) go to 900
            endif
            if (iparm(5) .eq. 0) bot = top
          endif
          if (iparm(12) .eq. 6) then
          call zwdot (top, fwk(ir),
     &    fwk(ir),  iparm , fparm , iwk , fwk , ier  )
            if (iparm(5) .eq. 0) bot = top
          endif
          if (iparm(12) .eq. 7) then
          call zwdot (top, fwk(iqlr),
     &    fwk(iqlr),  iparm , fparm , iwk , fwk , ier  )
            if (iparm(5) .eq. 0) bot = top
          endif
          if (iparm(12) .eq. 8) then
          call zwdot (top, fwk(iqrqlr),
     &    fwk(iqrqlr),  iparm , fparm , iwk , fwk , ier  )
            if (iparm(5) .eq. 0) bot = top
          endif
          if (dble(top) .lt. dble((0d0,0.0d0))) then
                ier = -10
                call xersho ( ier, 'zgmchr' , iparm ,
     &                        'Attempt to take sqrt of negative number' 
     &)
                go to 900
          endif
          if (dble(bot) .le. dble((0d0,0.0d0))) then
                ier = -10
                call xersho ( ier, 'zgmchr' , iparm ,
     &                        'Attempt to divide by zero' )
                go to 900
          endif
          fparm(5) = sqrt(dble(top)/dble(bot))
          call zitput ( in , is ,  iparm , fparm , iwk , fwk , ier )
          if (dble(fparm(5)) .le. dble(fparm(4))) go to 2500
                go to 5559
 5558 continue
 5559 continue
*     *---end user stopping test----------
*
        if (iparm(12) .eq. -1) then
          if (r0norm .eq. (0d0,0.0d0)) then
                ier = -10
                call xersho ( ier, 'zgmchr' , iparm ,
     &                        'Attempt to divide by zero' )
                go to 900
          endif
          fparm(5) = rnorm/r0norm
          call zitput ( in , is ,  iparm , fparm , iwk , fwk , ier )
          if (dble(fparm(5)) .le. dble(fparm(4))) go to 2500
        endif
*
*       *---begin exact stopping test----------
        if (iparm(12) .eq. -2) then
          call zwsub ( fwk(iv1),
     &    u, uexact,  iparm , fparm , iwk , fwk , ier  )
          call zwrdot (uudot,fwk(iv1),
     &    fwk(iv1),  iparm , fparm , iwk , fwk , ier  )
          call zwrdot (uexdt,uexact,
     &    uexact,  iparm , fparm , iwk , fwk , ier  )
          if (dble(uudot)  .lt. dble((0d0,0.0d0))) then
                ier = -10
                call xersho ( ier, 'zgmchr' , iparm ,
     &                        'Attempt to take sqrt of negative number' 
     &)
                go to 900
          endif
          if (dble(uexdt) .le. dble((0d0,0.0d0))) then
                ier = -10
                call xersho ( ier, 'zgmchr' , iparm ,
     &                        'Attempt to divide by zero' )
                go to 900
          endif
          fparm(5) = sqrt(dble(uudot)/dble(uexdt))
          call zitput ( in , is ,  iparm , fparm , iwk , fwk , ier )
          if (dble(fparm(5)) .le. dble(fparm(4))) go to 2500
        endif
*       *---end exact stopping test----------
*
        if (in .ge. iparm(4)) then
                ier = 2
                call xersho ( ier, 'zgmchr' , iparm , ' ' )
          go to 900
        endif
*
        if (.not.(is .ne. in) .or. is.eq.0) then
          call zckstg ( iclstg, iwkstg,  iparm , fparm , iwk , fwk , ier
     &  )
           if (ier .lt. 0) go to 900
        endif
*
*-----------------------proceed with iteration--------------------------
*
        if (is .eq. 0) then
          if (((iparm(13).eq.1).or.(iparm(13).eq.3)) .and. ((mod(iparm(1
     &5)/2,2).eq.1)     .or.
     &    iparm(12).eq.2     .or. iparm(12).eq.6)) then
          call zwset (fwk(ip+iparm(3)*((mod((is),ns+1)))),fwk(ir),
     &     iparm , fparm , iwk , fwk , ier  )
          endif
          call zwset (fwk(iqlp+iparm(3)*((mod((is),ns+1)))),fwk(iqlr),
     &     iparm , fparm , iwk , fwk , ier  )
          if (((iparm(13).eq.2).or.(iparm(13).eq.3))) then
          call zwset (fwk(iqrqlp+iparm(3)*((mod((is),ns+1)))),fwk(iqrqlr
     &),
     &     iparm , fparm , iwk , fwk , ier  )
          endif
          if ((is .ne. in)) then
            fwk(ipf + mod((is),ns+1))        = (1d0,0.0d0)
            fwk(ipp    + ((is+1)-1) + (ns+1)*((is+1)-1)) = rdot
            fwk(ipz   -1+(1))    = (0d0,0.0d0)
            h10           = (1d0,0.0d0)
          else
            if (rnorm .eq. (0d0,0.0d0)) then
                ier = -7
                call xersho ( ier, 'zgmchr' , iparm , ' ' )
                go to 900
            endif
            fwk(ipf + mod((is),ns+1))   = (1d0,0.0d0)/rnorm
            fwk(izc + mod((is+1)-1,ns+1)) = rnorm
            fwk(ipz   -1+(1))    = (0d0,0.0d0)
            h10      = rnorm
          endif
          if (iparm(2) .ge. 5) then
        write (iparm(1),500) 'h(1,0)              ', h10
          endif
        endif
*
        go to (501,502,503,504), iparm(13)+1
 501    continue
*        *---begin revcom matvec----------
          ivqr = iqrqlp + iparm(3)*((mod((is),ns+1)))
          iva = ip + iparm(3)*((mod((is+1),ns+1)))
         ireq = 3
        iretlb = 2020
        go to 1500
 2020   continue
         if (ijob .eq. -1) go to 900
*        *---end revcom matvec----------
        go to 510
 502    continue
          if (((mod(iparm(15)/2,2).eq.1)     .or.
     &    iparm(12).eq.2     .or. iparm(12).eq.6)) go to 5587
                 go to 5588
 5587  continue
*        *---begin revcom matvec----------
          ivqr = iqrqlp + iparm(3)*((mod((is),ns+1)))
          iva = ip + iparm(3)*((mod((is+1),ns+1)))
         ireq = 3
        iretlb = 2021
        go to 1500
 2021   continue
         if (ijob .eq. -1) go to 900
*        *---end revcom matvec----------
*        *---begin revcom precon----------
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5590
                 go to 5591
 5590  continue
          call zwset (fwk(iqlp+iparm(3)*((mod((is+1),ns+1)))),fwk(ip+ipa
     &rm(3)*((mod((is+1),ns+1)))),
     &     iparm , fparm , iwk , fwk , ier  )
                go to 5592
 5591 continue
          iva = ip + iparm(3)*((mod((is+1),ns+1)))
          ivql = iqlp + iparm(3)*((mod((is+1),ns+1)))
         ireq = 5
        iretlb = 2022
        go to 1500
 2022   continue
         if (ijob .eq. -1) go to 900
 5592 continue
*        *---end revcom precon----------
                go to 5589
 5588 continue
*        *---begin revcom matvec----------
          ivqr = iqrqlp + iparm(3)*((mod((is),ns+1)))
          iva = iv1 
         ireq = 3
        iretlb = 2023
        go to 1500
 2023   continue
         if (ijob .eq. -1) go to 900
*        *---end revcom matvec----------
*        *---begin revcom precon----------
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5593
                 go to 5594
 5593  continue
          call zwset (fwk(iqlp+iparm(3)*((mod((is+1),ns+1)))),fwk(iv1),
     &     iparm , fparm , iwk , fwk , ier  )
                go to 5595
 5594 continue
          iva = iv1 
          ivql = iqlp + iparm(3)*((mod((is+1),ns+1)))
         ireq = 5
        iretlb = 2024
        go to 1500
 2024   continue
         if (ijob .eq. -1) go to 900
 5595 continue
*        *---end revcom precon----------
 5589 continue
        go to 510
 503    continue
*        *---begin revcom matvec----------
          ivqr = iqrqlp + iparm(3)*((mod((is),ns+1)))
          iva = ip + iparm(3)*((mod((is+1),ns+1)))
         ireq = 3
        iretlb = 2025
        go to 1500
 2025   continue
         if (ijob .eq. -1) go to 900
*        *---end revcom matvec----------
*        *---begin revcom precon----------
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5596
                 go to 5597
 5596  continue
          call zwset (fwk(iqrqlp+iparm(3)*((mod((is+1),ns+1)))),fwk(iqlp
     &+iparm(3)*((mod((is+1),ns+1)))),
     &     iparm , fparm , iwk , fwk , ier  )
                go to 5598
 5597 continue
          ivql = iqlp + iparm(3)*((mod((is+1),ns+1)))
          ivqr = iqrqlp + iparm(3)*((mod((is+1),ns+1)))
         ireq = 7
        iretlb = 2026
        go to 1500
 2026   continue
         if (ijob .eq. -1) go to 900
 5598 continue
*        *---end revcom precon----------
        go to 510
 504    continue
          if (((mod(iparm(15)/2,2).eq.1)     .or.
     &    iparm(12).eq.2     .or. iparm(12).eq.6)) go to 5599
                 go to 5600
 5599  continue
*        *---begin revcom matvec----------
          ivqr = iqrqlp + iparm(3)*((mod((is),ns+1)))
          iva = ip + iparm(3)*((mod((is+1),ns+1)))
         ireq = 3
        iretlb = 2027
        go to 1500
 2027   continue
         if (ijob .eq. -1) go to 900
*        *---end revcom matvec----------
*        *---begin revcom precon----------
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5602
                 go to 5603
 5602  continue
          call zwset (fwk(iqlp+iparm(3)*((mod((is+1),ns+1)))),fwk(ip+ipa
     &rm(3)*((mod((is+1),ns+1)))),
     &     iparm , fparm , iwk , fwk , ier  )
                go to 5604
 5603 continue
          iva = ip + iparm(3)*((mod((is+1),ns+1)))
          ivql = iqlp + iparm(3)*((mod((is+1),ns+1)))
         ireq = 5
        iretlb = 2028
        go to 1500
 2028   continue
         if (ijob .eq. -1) go to 900
 5604 continue
*        *---end revcom precon----------
                go to 5601
 5600 continue
*        *---begin revcom matvec----------
          ivqr = iqrqlp + iparm(3)*((mod((is),ns+1)))
          iva = iv1 
         ireq = 3
        iretlb = 2029
        go to 1500
 2029   continue
         if (ijob .eq. -1) go to 900
*        *---end revcom matvec----------
*        *---begin revcom precon----------
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5605
                 go to 5606
 5605  continue
          call zwset (fwk(iqlp+iparm(3)*((mod((is+1),ns+1)))),fwk(iv1),
     &     iparm , fparm , iwk , fwk , ier  )
                go to 5607
 5606 continue
          iva = iv1 
          ivql = iqlp + iparm(3)*((mod((is+1),ns+1)))
         ireq = 5
        iretlb = 2030
        go to 1500
 2030   continue
         if (ijob .eq. -1) go to 900
 5607 continue
*        *---end revcom precon----------
 5601 continue
*        *---begin revcom precon----------
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5608
                 go to 5609
 5608  continue
          call zwset (fwk(iqrqlp+iparm(3)*((mod((is+1),ns+1)))),fwk(iqlp
     &+iparm(3)*((mod((is+1),ns+1)))),
     &     iparm , fparm , iwk , fwk , ier  )
                go to 5610
 5609 continue
          ivql = iqlp + iparm(3)*((mod((is+1),ns+1)))
          ivqr = iqrqlp + iparm(3)*((mod((is+1),ns+1)))
         ireq = 7
        iretlb = 2031
        go to 1500
 2031   continue
         if (ijob .eq. -1) go to 900
 5610 continue
*        *---end revcom precon----------
 510    continue
        fwk(ipf + mod((is+1),ns+1)) = fwk(ipf + mod((is),ns+1))
*
        if ((is .ne. in)) then
          coe2 = d
          if (is .eq. 0) then
            coe1 = (0d0,0.0d0)
            coe3 = 2*gam
          else
            coe1 = c2/(4*gam)
            fwk(ih   + ((is  )-1) + (ns+1)*((is+1)-1)) = coe1
            coe3 = gam
          endif
          fwk(ih   + ((is+1)-1) + (ns+1)*((is+1)-1)) = coe2
          fwk(ih   + ((is+2)-1) + (ns+1)*((is+1)-1)) = coe3
            val = - coe2 * fwk(ipf + mod((is  ),ns+1)) / fwk(ipf + mod((
     &is+1),ns+1))
            if (((iparm(13).eq.1).or.(iparm(13).eq.3)) .and. ((mod(iparm
     &(15)/2,2).eq.1)     .or.
     &    iparm(12).eq.2     .or. iparm(12).eq.6)) then
          call zwsax (fwk(ip+iparm(3)*((mod((is+1),ns+1)))),fwk(ip+iparm
     &(3)*((mod((is+1),ns+1)))),
     &    fwk(ip+iparm(3)*((mod((is  ),ns+1)))), val,  iparm , fparm , i
     &wk , fwk , ier  )
            endif
          call zwsax (fwk(iqlp+iparm(3)*((mod((is+1),ns+1)))),fwk(iqlp+i
     &parm(3)*((mod((is+1),ns+1)))),
     &    fwk(iqlp+iparm(3)*((mod((is  ),ns+1)))), val,  iparm , fparm ,
     & iwk , fwk , ier  )
            if (((iparm(13).eq.2).or.(iparm(13).eq.3))) then
          call zwsax (fwk(iqrqlp+iparm(3)*((mod((is+1),ns+1)))),fwk(iqrq
     &lp+iparm(3)*((mod((is+1),ns+1)))),
     &    fwk(iqrqlp+iparm(3)*((mod((is  ),ns+1)))), val,  iparm , fparm
     & , iwk , fwk , ier  )
            endif
          if (is .ne. 0) then
            val = - coe1 * fwk(ipf + mod((is-1),ns+1)) / fwk(ipf + mod((
     &is+1),ns+1))
            if (((iparm(13).eq.1).or.(iparm(13).eq.3)) .and. ((mod(iparm
     &(15)/2,2).eq.1)     .or.
     &    iparm(12).eq.2     .or. iparm(12).eq.6)) then
          call zwsax (fwk(ip+iparm(3)*((mod((is+1),ns+1)))),fwk(ip+iparm
     &(3)*((mod((is+1),ns+1)))),
     &    fwk(ip+iparm(3)*((mod((is-1),ns+1)))), val,  iparm , fparm , i
     &wk , fwk , ier  )
            endif
          call zwsax (fwk(iqlp+iparm(3)*((mod((is+1),ns+1)))),fwk(iqlp+i
     &parm(3)*((mod((is+1),ns+1)))),
     &    fwk(iqlp+iparm(3)*((mod((is-1),ns+1)))), val,  iparm , fparm ,
     & iwk , fwk , ier  )
            if (((iparm(13).eq.2).or.(iparm(13).eq.3))) then
          call zwsax (fwk(iqrqlp+iparm(3)*((mod((is+1),ns+1)))),fwk(iqrq
     &lp+iparm(3)*((mod((is+1),ns+1)))),
     &    fwk(iqrqlp+iparm(3)*((mod((is-1),ns+1)))), val,  iparm , fparm
     & , iwk , fwk , ier  )
            endif
          endif
          fwk(ipf + mod((is+1),ns+1)) = fwk(ipf + mod((is+1),ns+1))/coe3
        else
          do 5611 i = 0, is
          call zwdot (pap, fwk(iqlp+iparm(3)*((mod((i),ns+1)))),
     &    fwk(iqlp+iparm(3)*((mod((is+1),ns+1)))),  iparm , fparm , iwk 
     &, fwk , ier  )
            pap = pap * conjg(fwk(ipf + mod((i),ns+1))) * fwk(ipf + mod(
     &(is+1),ns+1))
            fwk(ih   + ((i+1)-1) + (ns+1)*((is+1)-1)) = pap
            if (iparm(2) .ge. 5) then
        write (iparm(1),'(1x,a,i5,a,i5,a,1pg20.10,2x,1pg20.10)') 'h(',i+
     &1,',',is+1,')      ', pap
            endif
            val = - pap * fwk(ipf + mod((i),ns+1)) / fwk(ipf + mod((is+1
     &),ns+1))
            if (((iparm(13).eq.1).or.(iparm(13).eq.3)) .and. ((mod(iparm
     &(15)/2,2).eq.1)     .or.
     &    iparm(12).eq.2     .or. iparm(12).eq.6)) then
          call zwsax (fwk(ip+iparm(3)*((mod((is+1),ns+1)))),fwk(ip+iparm
     &(3)*((mod((is+1),ns+1)))),
     &    fwk(ip+iparm(3)*((mod((i),ns+1)))), val,  iparm , fparm , iwk 
     &, fwk , ier  )
            endif
          call zwsax (fwk(iqlp+iparm(3)*((mod((is+1),ns+1)))),fwk(iqlp+i
     &parm(3)*((mod((is+1),ns+1)))),
     &    fwk(iqlp+iparm(3)*((mod((i),ns+1)))), val,  iparm , fparm , iw
     &k , fwk , ier  )
            if (((iparm(13).eq.2).or.(iparm(13).eq.3))) then
          call zwsax (fwk(iqrqlp+iparm(3)*((mod((is+1),ns+1)))),fwk(iqrq
     &lp+iparm(3)*((mod((is+1),ns+1)))),
     &    fwk(iqrqlp+iparm(3)*((mod((i),ns+1)))), val,  iparm , fparm , 
     &iwk , fwk , ier  )
            endif
5611  continue
          call zwrdot (vdot,fwk(iqlp+iparm(3)*((mod((is+1),ns+1)))),
     &    fwk(iqlp+iparm(3)*((mod((is+1),ns+1)))),  iparm , fparm , iwk 
     &, fwk , ier  )
          vdot = vdot * conjg(fwk(ipf + mod((is+1),ns+1))) * fwk(ipf + m
     &od((is+1),ns+1))
          vnorm = sqrt(vdot)
          fwk(ih   + ((is+2)-1) + (ns+1)*((is+1)-1)) = vnorm
          if (iparm(2) .ge. 5) then
        write (iparm(1),'(1x,a,i5,a,i5,a,1pg20.10,2x,1pg20.10)') 'h(',is
     &+2,',',is+1,')      ', vnorm
          endif
          if (vnorm .eq. (0d0,0.0d0)) then
            fwk(ipf + mod((is+1),ns+1)) = (1d0,0.0d0)
          else
            fwk(ipf + mod((is+1),ns+1)) = fwk(ipf + mod((is+1),ns+1))/vn
     &orm
          endif
        endif
        if (.true.) then
          if (((iparm(13).eq.1).or.(iparm(13).eq.3)) .and. ((mod(iparm(1
     &5)/2,2).eq.1)     .or.
     &    iparm(12).eq.2     .or. iparm(12).eq.6)) then
          call zwscl ( fwk(ip+iparm(3)*((mod((is+1),ns+1)))), fwk(ip+ipa
     &rm(3)*((mod((is+1),ns+1)))), fwk(ipf + mod((is+1),ns+1)),  iparm ,
     & fparm , iwk , fwk , ier  )
          endif
          call zwscl ( fwk(iqlp+iparm(3)*((mod((is+1),ns+1)))), fwk(iqlp
     &+iparm(3)*((mod((is+1),ns+1)))), fwk(ipf + mod((is+1),ns+1)),  ipa
     &rm , fparm , iwk , fwk , ier  )
          if (((iparm(13).eq.2).or.(iparm(13).eq.3))) then
          call zwscl ( fwk(iqrqlp+iparm(3)*((mod((is+1),ns+1)))), fwk(iq
     &rqlp+iparm(3)*((mod((is+1),ns+1)))), fwk(ipf + mod((is+1),ns+1)), 
     & iparm , fparm , iwk , fwk , ier  )
          endif
          fwk(ipf + mod((is+1),ns+1)) = (1d0,0.0d0)
        endif
*
        if ((is .ne. in)) then
          do 5612 i = 0, is+1
            if (i .eq. is+1) then
          call zwrdot (vdot,fwk(iqlp+iparm(3)*((mod((i),ns+1)))),
     &    fwk(iqlp+iparm(3)*((mod((is+1),ns+1)))),  iparm , fparm , iwk 
     &, fwk , ier  )
            else
          call zwdot (vdot, fwk(iqlp+iparm(3)*((mod((i),ns+1)))),
     &    fwk(iqlp+iparm(3)*((mod((is+1),ns+1)))),  iparm , fparm , iwk 
     &, fwk , ier  )
            endif
            fwk(ipp    + ((i +1)-1) + (ns+1)*((is+2)-1)) = vdot * conjg(
     &fwk(ipf + mod((i),ns+1))) * fwk(ipf + mod((is+1),ns+1))
            fwk(ipp    + ((is+2)-1) + (ns+1)*((i +1)-1)) = conjg(fwk(ipp
     &    + ((i+1)-1) + (ns+1)*((is+2)-1)))
5612  continue
        else
       do 5613 i = 1, is+2
       fwk(iu   + ((i)-1) + (ns+1)*((is+1)-1)) = fwk(ih   + ((i)-1) + (n
     &s+1)*((is+1)-1))
 5613  continue
          do 5614 i = 1, is
            ut          = fwk(iu   + ((i  )-1) + (ns+1)*((is+1)-1))
            h           = fwk(iu   + ((i+1)-1) + (ns+1)*((is+1)-1))
            fwk(iu   + ((i  )-1) + (ns+1)*((is+1)-1)) = conjg(fwk(icos +
     & mod((i),ns)))*ut + conjg(fwk(isin + mod((i),ns)))*h
            fwk(iu   + ((i+1)-1) + (ns+1)*((is+1)-1)) =      -fwk(isin +
     & mod((i),ns)) *ut +       fwk(icos + mod((i),ns)) *h
5614  continue
          v1 = fwk(iu   + ((is+1)-1) + (ns+1)*((is+1)-1))
          v2 = fwk(iu   + ((is+2)-1) + (ns+1)*((is+1)-1))
          denom = sqrt (conjg(v1)*v1 + conjg(v2)*v2)
          if (denom .eq. (0d0,0.0d0)) then
                ier = -6
                call xersho ( ier, 'zgmchr' , iparm ,
     &                        'Iterative method has apparently converged
     &' )
                go to 900
          endif
          fwk(icos + mod((is+1),ns)) = v1/denom
          fwk(isin + mod((is+1),ns)) = v2/denom
          fwk(iu   + ((is+1)-1) + (ns+1)*((is+1)-1)) = denom
          fwk(iu   + ((is+2)-1) + (ns+1)*((is+1)-1)) = (0d0,0.0d0)
          if (iparm(2) .ge. 5) then
            do 5615 i = 1, is+1
        write (iparm(1),'(1x,a,i5,a,i5,a,1pg20.10,2x,1pg20.10)') 'u(',i,
     &',',is+1,')      ', fwk(iu   + ((i)-1) + (ns+1)*((is+1)-1))
5615  continue
          endif
          fwk(izc + mod((is+2)-1,ns+1)) =       -fwk(isin + mod((is+1),n
     &s)) *fwk(izc + mod((is+1)-1,ns+1))
          fwk(izc + mod((is+1)-1,ns+1)) =  conjg(fwk(icos + mod((is+1),n
     &s)))*fwk(izc + mod((is+1)-1,ns+1))
        endif
*
        nm = is + 1
        if (((mod(iparm(15)/1,2).eq.1)     .or.
     &    iparm(12).eq.-2    .or.
     &    iparm(12).eq.1     .or. iparm(12).eq.5) .or.
     &      ((mod(iparm(15)/2,2).eq.1)     .or.
     &    iparm(12).eq.2     .or. iparm(12).eq.6) .or. ((mod(iparm(15)/4
     &,2).eq.1)   .or.
     &    iparm(12).eq.3    .or. iparm(12).eq.7) .or. ((mod(iparm(15)/8,
     &2).eq.1) .or.
     &    iparm(12).eq.4    .or. iparm(12).eq.8) .or. (is.eq.ns-1)) then
       do 5616 i = 1, nm-1
       fwk(iyold -1+(i)) = fwk(iy    -1+(i))
 5616  continue
          fwk(iyold -1+(nm)) = (0d0,0.0d0)
          if ((is .ne. in)) then
            do 5617 i = 1, is+1
            do 5618 j = 1, is+2
              fwk(ihpp   + ((i)-1) + (ns  )*((j)-1)) =            conjg(
     &fwk(ih   + ((i  )-1) + (ns+1)*((i)-1)))*fwk(ipp    + ((i  )-1) + (
     &ns+1)*((j)-1))
     &                            + conjg(fwk(ih   + ((i+1)-1) + (ns+1)*
     &((i)-1)))*fwk(ipp    + ((i+1)-1) + (ns+1)*((j)-1))
              if (i .gt. 1)
     &        fwk(ihpp   + ((i)-1) + (ns  )*((j)-1)) = fwk(ihpp   + ((i)
     &-1) + (ns  )*((j)-1)) + conjg(fwk(ih   + ((i-1)-1) + (ns+1)*((i)-1
     &)))*fwk(ipp    + ((i-1)-1) + (ns+1)*((j)-1))
5618  continue
5617  continue
            do 5619 i = 1, is+1
              fwk(ihpr   + ((i)-1)) = fwk(ihpp   + ((i)-1) + (ns  )*((1)
     &-1))*h10
5619  continue
            do 5620 i = 1, is+1
            do 5621 j = 1, is+1
              fwk(ihpph  + ((i)-1) + (ns  )*((j)-1)) =             fwk(i
     &hpp   + ((i)-1) + (ns  )*((j  )-1))*fwk(ih   + ((j  )-1) + (ns+1)*
     &((j)-1))
     &                              + fwk(ihpp   + ((i)-1) + (ns  )*((j+
     &1)-1))*fwk(ih   + ((j+1)-1) + (ns+1)*((j)-1))
              if (j .gt. 1)
     &        fwk(ihpph  + ((i)-1) + (ns  )*((j)-1)) = fwk(ihpph  + ((i)
     &-1) + (ns  )*((j)-1)) + fwk(ihpp   + ((i)-1) + (ns  )*((j-1)-1))*f
     &wk(ih   + ((j-1)-1) + (ns+1)*((j)-1))
5621  continue
5620  continue
            do 5622 i = 1, nm
              fwk(idiag  + ((i)-1)) = fwk(ihpph  + ((i)-1) + (ns  )*((i)
     &-1))
              if (dble(fwk(idiag  + ((i)-1))) .le. dble((0d0,0.0d0))) th
     &en
                fwk(idiag  + ((i)-1)) = (1d0,0.0d0)
              else
                fwk(idiag  + ((i)-1)) = (1d0,0.0d0)/sqrt(dble(fwk(idiag 
     & + ((i)-1))))
              endif
5622  continue
       do 5623 i = 1, nm
       do 5624 j = 1, nm
       fwk(ihpph  + ((i)-1) + (ns  )*((j)-1)) = fwk(ihpph  + ((i)-1) + (
     &ns  )*((j)-1))*fwk(idiag  + ((i)-1))*fwk(idiag  + ((j)-1))
 5624  continue
 5623  continue
            call zgesvd ('O','A',nm,nm,fwk(ihpph),ns,
     &        fwk(ismat),fwk(ihpph),ns,fwk(ivtmat),ns,
     &        fwk(iwkla),(5*ns),
     &        fwk(iwkla+(5*ns)),
     &        infola)
            if (infola .ne. 0) then
                ier = -6
                call xersho ( ier, 'zgmchr' , iparm ,
     &                        'Error in Lapack gesvd while computing pse
     &udoinverse' )
                go to 900
            endif
               do 5625 i = nm, 1, -1
                 if (mod(i,2) .eq. 1) then
                   fwk(itmpv  + ((i)-1)) = dble(fwk(ismat  + (((i+1)/2)-
     &1)))
                 else
                   fwk(itmpv  + ((i)-1)) = dimag(fwk(ismat  + ((i   /2)-
     &1)))
                 endif
5625  continue
       do 5626 i = 1, nm
       fwk(ismat  + ((i)-1)) = fwk(itmpv  + ((i)-1))
 5626  continue
            if (iparm(2) .ge. 5) then
        write (iparm(1),'(1x,a,/,1x,1pg20.10,2x,1pg20.10,a,1pg20.10,2x,1
     &pg20.10)') 'Extremal singular values of small matrix:', fwk(ismat 
     & + ((nm)-1)), ', ', fwk(ismat  + ((1)-1))
            endif
            nrank = nm
            do 5627 i = 1, nm
              if (dble(fwk(ismat  + ((i)-1))) .le. dble(fwk(ismat  + ((1
     &)-1)))*epsm) then
                fwk(ismat  + ((i)-1)) = (0d0,0.0d0)
                nrank = nrank - 1
              else
                fwk(ismat  + ((i)-1)) = (1d0,0.0d0)/fwk(ismat  + ((i)-1)
     &)
              endif
5627  continue
            if (iparm(2) .ge. 5) then
        write (iparm(1),'(1x,a,i6)') 'Rank of small matrix: ', nrank
            endif
       do 5628 i = 1, nm
       fwk(iy    -1+(i)) = fwk(ihpr   + ((i)-1))*fwk(idiag  + ((i)-1))
 5628  continue
       do 5629 i = 1, nm
       fwk(itmpv  + ((i)-1)) = (0d0,0.0d0)
       do 5631 j = 1, nm
       fwk(itmpv  + ((i)-1)) = fwk(itmpv  + ((i)-1)) + conjg(fwk(ihpph  
     &+ ((j)-1) + (ns  )*((i)-1)))*fwk(iy    -1+(j))
 5631  continue
 5629  continue
       do 5632 i = 1, nm
       fwk(itmpv  + ((i)-1)) = fwk(itmpv  + ((i)-1))*fwk(ismat  + ((i)-1
     &))
 5632  continue
       do 5633 i = 1, nm
       fwk(iy    -1+(i)) = (0d0,0.0d0)
       do 5635 j = 1, nm
       fwk(iy    -1+(i)) = fwk(iy    -1+(i)) + conjg(fwk(ivtmat + ((j)-1
     &) + (ns  )*((i)-1)))*fwk(itmpv  + ((j)-1))
 5635  continue
 5633  continue
       do 5636 i = 1, nm
       fwk(iy    -1+(i)) = fwk(iy    -1+(i))*fwk(idiag  + ((i)-1))
 5636  continue
          else
            do 5637 i = nm, 1, -1
              sum = fwk(izc + mod((i)-1,ns+1))
              do 5638 j = i+1, nm
                sum = sum - fwk(iy    -1+(j))*fwk(iu   + ((i)-1) + (ns+1
     &)*((j)-1))
5638  continue
              fwk(iy    -1+(i)) = sum/fwk(iu   + ((i)-1) + (ns+1)*((i)-1
     &))
5637  continue
          endif
          if (iparm(2) .ge. 5) then
            do 5639 i = 1, nm
        write (iparm(1),'(1x,a,i5,a,1pg20.10,2x,1pg20.10)') 'u-from-p(',
     &i,')     ', fwk(iy    -1+(i))
5639  continue
          endif
        endif
*
        if (((mod(iparm(15)/2,2).eq.1)     .or.
     &    iparm(12).eq.2     .or. iparm(12).eq.6) .or. ((mod(iparm(15)/4
     &,2).eq.1)   .or.
     &    iparm(12).eq.3    .or. iparm(12).eq.7) .or. ((mod(iparm(15)/8,
     &2).eq.1) .or.
     &    iparm(12).eq.4    .or. iparm(12).eq.8) .or. (is.eq.ns-1)) then
       do 5640 i = 1, nm
       fwk(ipzold-1+(i)) = fwk(ipz   -1+(i))
 5640  continue
          fwk(ipzold-1+(nm+1)) = (0d0,0.0d0)
          if ((is .ne. in)) then
            do 5641 i = 1, is+2
              fwk(ipz   -1+(i)) = (0d0,0.0d0)
              if (i .gt. 1)
     &        fwk(ipz   -1+(i)) = fwk(ipz   -1+(i)) + fwk(ih   + ((i)-1)
     & + (ns+1)*((i-1)-1))*fwk(iy    -1+(i-1))
              if (i .lt. is+2)
     &        fwk(ipz   -1+(i)) = fwk(ipz   -1+(i)) + fwk(ih   + ((i)-1)
     & + (ns+1)*((i)-1))  *fwk(iy    -1+(i))
              if (i .lt. is+1)
     &        fwk(ipz   -1+(i)) = fwk(ipz   -1+(i)) + fwk(ih   + ((i)-1)
     & + (ns+1)*((i+1)-1))*fwk(iy    -1+(i+1))
5641  continue
          else
       do 5642 i = 1, is+1
       fwk(ipz   -1+(i)) = fwk(izc + mod((i)-1,ns+1))
 5642  continue
            fwk(ipz   -1+(nm+1)) = (0d0,0.0d0)
            do 5643 i = nm, 1, -1
              v1 = fwk(icos + mod((i),ns))*fwk(ipz   -1+(i)) - conjg(fwk
     &(isin + mod((i),ns)))*fwk(ipz   -1+(i+1))
              v2 = fwk(isin + mod((i),ns))*fwk(ipz   -1+(i)) + conjg(fwk
     &(icos + mod((i),ns)))*fwk(ipz   -1+(i+1))
              fwk(ipz   -1+(i  )) = v1
              fwk(ipz   -1+(i+1)) = v2
5643  continue
          endif
          if (iparm(2) .ge. 5) then
            do 5644 i = 1, nm+1
        write (iparm(1),'(1x,a,i5,a,1pg20.10,2x,1pg20.10)') 'r-from-p(',
     &i,')     ', fwk(ipz   -1+(i))
5644  continue
          endif
        endif
*
        if (((mod(iparm(15)/1,2).eq.1)     .or.
     &    iparm(12).eq.-2    .or.
     &    iparm(12).eq.1     .or. iparm(12).eq.5)) then
          do 5645 i = 0, nm-1
            val = (fwk(iy    -1+(i+1))-fwk(iyold -1+(i+1))) * fwk(ipf + 
     &mod((i),ns+1))
          call zwsax (u,u,
     &    fwk(iqrqlp+iparm(3)*((mod((i),ns+1)))), val,  iparm , fparm , 
     &iwk , fwk , ier  )
5645  continue
        else if ((is.eq.ns-1)) then
          do 5646 i = 0, nm-1
            val = fwk(iy    -1+(i+1)) * fwk(ipf + mod((i),ns+1))
          call zwsax (u,u,
     &    fwk(iqrqlp+iparm(3)*((mod((i),ns+1)))), val,  iparm , fparm , 
     &iwk , fwk , ier  )
5646  continue
        endif
*
        if (((iparm(13).eq.1).or.(iparm(13).eq.3)) .and. ((mod(iparm(15)
     &/2,2).eq.1)     .or.
     &    iparm(12).eq.2     .or. iparm(12).eq.6)) then
          do 5647 i = 0, is+1
            val = - (fwk(ipz   -1+(i+1))-fwk(ipzold-1+(i+1))) * fwk(ipf 
     &+ mod((i),ns+1))
          call zwsax (fwk(ir),fwk(ir),
     &    fwk(ip+iparm(3)*((mod((i),ns+1)))), val,  iparm , fparm , iwk 
     &, fwk , ier  )
5647  continue
        endif
*
        if (((mod(iparm(15)/4,2).eq.1)   .or.
     &    iparm(12).eq.3    .or. iparm(12).eq.7) .or. ((.not.((iparm(13)
     &.eq.1).or.(iparm(13).eq.3))).and.((mod(iparm(15)/2,2).eq.1)     .o
     &r.
     &    iparm(12).eq.2     .or. iparm(12).eq.6))
     &              .or. ((.not.((iparm(13).eq.2).or.(iparm(13).eq.3))).
     &and.((mod(iparm(15)/8,2).eq.1) .or.
     &    iparm(12).eq.4    .or. iparm(12).eq.8))) then
          do 5648 i = 0, is+1
            val = - (fwk(ipz   -1+(i+1))-fwk(ipzold-1+(i+1))) * fwk(ipf 
     &+ mod((i),ns+1))
          call zwsax (fwk(iqlr),fwk(iqlr),
     &    fwk(iqlp+iparm(3)*((mod((i),ns+1)))), val,  iparm , fparm , iw
     &k , fwk , ier  )
5648  continue
        else if ((is.eq.ns-1)) then
          do 5649 i = 0, is+1
            val = - fwk(ipz   -1+(i+1)) * fwk(ipf + mod((i),ns+1))
          call zwsax (fwk(iqlr),fwk(iqlr),
     &    fwk(iqlp+iparm(3)*((mod((i),ns+1)))), val,  iparm , fparm , iw
     &k , fwk , ier  )
5649  continue
        endif
*
        if (((iparm(13).eq.2).or.(iparm(13).eq.3)) .and. ((mod(iparm(15)
     &/8,2).eq.1) .or.
     &    iparm(12).eq.4    .or. iparm(12).eq.8)) then
          do 5650 i = 0, is+1
            val = - (fwk(ipz   -1+(i+1))-fwk(ipzold-1+(i+1))) * fwk(ipf 
     &+ mod((i),ns+1))
          call zwsax (fwk(iqrqlr),fwk(iqrqlr),
     &    fwk(iqrqlp+iparm(3)*((mod((i),ns+1)))), val,  iparm , fparm , 
     &iwk , fwk , ier  )
5650  continue
        else if (((iparm(13).eq.2).or.(iparm(13).eq.3)) .and. (is.eq.ns-
     &1)) then
          do 5651 i = 0, is+1
            val = - fwk(ipz   -1+(i+1)) * fwk(ipf + mod((i),ns+1))
          call zwsax (fwk(iqrqlr),fwk(iqrqlr),
     &    fwk(iqrqlp+iparm(3)*((mod((i),ns+1)))), val,  iparm , fparm , 
     &iwk , fwk , ier  )
5651  continue
        endif
*
        if ((is .ne. in)) then
          if (is .eq. ns-1) then
       do 5652 i = 1, is+1
       rdot = rdot - conjg(fwk(ihpr   + ((i)-1)))*fwk(iy    -1+(i))
 5652  continue
          endif
        else
          rdot  = conjg(fwk(izc + mod((is+2)-1,ns+1)))*fwk(izc + mod((is
     &+2)-1,ns+1))
        endif
        rnorm = sqrt(rdot)
*
        if (is .eq. ns-1) then
          call zwrdot (rdot,fwk(iqlr),
     &    fwk(iqlr),  iparm , fparm , iwk , fwk , ier  )
          rnorm  = sqrt(abs(rdot))
        endif
*
        if (in .eq. ns-1) then
          call zgeev ('N','N',nm,fwk(ih),ns+1,
     &      fwk(iww),
     &      fwk(ih),ns+1,fwk(ih),ns+1,fwk(iwkla),(5*ns),
     &      fwk(iwkla+(5*ns)),
     &      infola)
          if (infola .ne. 0) then
                ier = -6
                call xersho ( ier, 'zgmchr' , iparm ,
     &                        'Error in Lapack geev while estimating spe
     &ctrum' )
                go to 900
          endif
             do 5653 i = 1, nm
               fwk(iwr + ((i)-1)) = dble(fwk(iww + ((i)-1)))
               fwk(iwi + ((i)-1)) = dimag(fwk(iww + ((i)-1)))
5653  continue
          vrmin = fwk(iwr + ((1)-1))
          vrmax = fwk(iwr + ((1)-1))
          vimin = fwk(iwi + ((1)-1))
          vimax = fwk(iwi + ((1)-1))
          do 5654 i = 2, nm
            vrmin = min(dble(vrmin),dble(fwk(iwr + ((i)-1))))
            vrmax = max(dble(vrmax),dble(fwk(iwr + ((i)-1))))
            vimin = min(dble(vimin),dble(fwk(iwi + ((i)-1))))
            vimax = max(dble(vimax),dble(fwk(iwi + ((i)-1))))
5654  continue
          if (iparm(2) .ge. 5) then
        write (iparm(1),500) 'Min eigval realpart ', vrmin
        write (iparm(1),500) 'Max eigval realpart ', vrmax
        write (iparm(1),500) 'Min eigval imagpart ', vimin
        write (iparm(1),500) 'Max eigval imagpart ', vimax
          endif
          d  = cmplx(dble((vrmin+vrmax)/(2d0,0.0d0)),
     &               dble((vimin+vimax)/(2d0,0.0d0)))
          al = (vrmax-vrmin)/(2d0,0.0d0)
          be = (vimax-vimin)/(2d0,0.0d0)
          c2 = (al)**2 - (be)**2
          gam = max(dble(al),dble(be))
          if (gam .eq. (0d0,0.0d0)) gam = (1d0,0.0d0)
          if (iparm(2) .ge. 5) then
        write (iparm(1),500) 'Ellipse center d    ', d
        write (iparm(1),500) 'c**2                ', c2
          endif
        endif
*
        in         = in + 1
        iparm(5) = in
        is         = is + 1
        if (is .eq. ns) is = 0
*
      go to 100
*
 2500 continue
      if (iparm(2) .ge. 3) then
        write (iparm(1),'(/a,i7,a/)') '     Restarted Chebyshev-basis GM
     &RES Method has converged in ', iparm(5), ' iterations.'
      endif
*
*-------------------------------terminate-------------------------------
*
 900  continue
*
      nm = is
      if (is.ne.0 .and. .not.((mod(iparm(15)/1,2).eq.1)     .or.
     &    iparm(12).eq.-2    .or.
     &    iparm(12).eq.1     .or. iparm(12).eq.5)) then
        if ((is .ne. in)) then
          do 5655 i = 1, is+1
          do 5656 j = 1, is+2
            fwk(ihpp   + ((i)-1) + (ns  )*((j)-1)) =            conjg(fw
     &k(ih   + ((i  )-1) + (ns+1)*((i)-1)))*fwk(ipp    + ((i  )-1) + (ns
     &+1)*((j)-1))
     &                          + conjg(fwk(ih   + ((i+1)-1) + (ns+1)*((
     &i)-1)))*fwk(ipp    + ((i+1)-1) + (ns+1)*((j)-1))
            if (i .gt. 1)
     &      fwk(ihpp   + ((i)-1) + (ns  )*((j)-1)) = fwk(ihpp   + ((i)-1
     &) + (ns  )*((j)-1)) + conjg(fwk(ih   + ((i-1)-1) + (ns+1)*((i)-1))
     &)*fwk(ipp    + ((i-1)-1) + (ns+1)*((j)-1))
5656  continue
5655  continue
          do 5657 i = 1, is+1
            fwk(ihpr   + ((i)-1)) = fwk(ihpp   + ((i)-1) + (ns  )*((1)-1
     &))*h10
5657  continue
          do 5658 i = 1, is+1
          do 5659 j = 1, is+1
            fwk(ihpph  + ((i)-1) + (ns  )*((j)-1)) =             fwk(ihp
     &p   + ((i)-1) + (ns  )*((j  )-1))*fwk(ih   + ((j  )-1) + (ns+1)*((
     &j)-1))
     &                            + fwk(ihpp   + ((i)-1) + (ns  )*((j+1)
     &-1))*fwk(ih   + ((j+1)-1) + (ns+1)*((j)-1))
            if (j .gt. 1)
     &      fwk(ihpph  + ((i)-1) + (ns  )*((j)-1)) = fwk(ihpph  + ((i)-1
     &) + (ns  )*((j)-1)) + fwk(ihpp   + ((i)-1) + (ns  )*((j-1)-1))*fwk
     &(ih   + ((j-1)-1) + (ns+1)*((j)-1))
5659  continue
5658  continue
          do 5660 i = 1, nm
            fwk(idiag  + ((i)-1)) = fwk(ihpph  + ((i)-1) + (ns  )*((i)-1
     &))
            if (dble(fwk(idiag  + ((i)-1))) .le. dble((0d0,0.0d0))) then
              fwk(idiag  + ((i)-1)) = (1d0,0.0d0)
            else
              fwk(idiag  + ((i)-1)) = (1d0,0.0d0)/sqrt(dble(fwk(idiag  +
     & ((i)-1))))
            endif
5660  continue
       do 5661 i = 1, nm
       do 5662 j = 1, nm
       fwk(ihpph  + ((i)-1) + (ns  )*((j)-1)) = fwk(ihpph  + ((i)-1) + (
     &ns  )*((j)-1))*fwk(idiag  + ((i)-1))*fwk(idiag  + ((j)-1))
 5662  continue
 5661  continue
          call zgesvd ('O','A',nm,nm,fwk(ihpph),ns,
     &      fwk(ismat),fwk(ihpph),ns,fwk(ivtmat),ns,
     &      fwk(iwkla),(5*ns),
     &      fwk(iwkla+(5*ns)),
     &      infola)
          if (infola .ne. 0) then
                ier = -6
                call xersho ( ier, 'zgmchr' , iparm ,
     &                        'Error in Lapack gesvd while computing pse
     &udoinverse' )
          endif
             do 5663 i = nm, 1, -1
               if (mod(i,2) .eq. 1) then
                 fwk(itmpv  + ((i)-1)) = dble(fwk(ismat  + (((i+1)/2)-1)
     &))
               else
                 fwk(itmpv  + ((i)-1)) = dimag(fwk(ismat  + ((i   /2)-1)
     &))
               endif
5663  continue
       do 5664 i = 1, nm
       fwk(ismat  + ((i)-1)) = fwk(itmpv  + ((i)-1))
 5664  continue
          if (iparm(2) .ge. 5) then
        write (iparm(1),'(1x,a,/,1x,1pg20.10,2x,1pg20.10,a,1pg20.10,2x,1
     &pg20.10)') 'Extremal singular values of small matrix:', fwk(ismat 
     & + ((nm)-1)), ', ', fwk(ismat  + ((1)-1))
          endif
          nrank = nm
          do 5665 i = 1, nm
            if (dble(fwk(ismat  + ((i)-1))) .le. dble(fwk(ismat  + ((1)-
     &1)))*epsm) then
              fwk(ismat  + ((i)-1)) = (0d0,0.0d0)
              nrank = nrank - 1
            else
              fwk(ismat  + ((i)-1)) = (1d0,0.0d0)/fwk(ismat  + ((i)-1))
            endif
5665  continue
          if (iparm(2) .ge. 5) then
        write (iparm(1),'(1x,a,i6)') 'Rank of small matrix: ', nrank
          endif
       do 5666 i = 1, nm
       fwk(iy    -1+(i)) = fwk(ihpr   + ((i)-1))*fwk(idiag  + ((i)-1))
 5666  continue
       do 5667 i = 1, nm
       fwk(itmpv  + ((i)-1)) = (0d0,0.0d0)
       do 5669 j = 1, nm
       fwk(itmpv  + ((i)-1)) = fwk(itmpv  + ((i)-1)) + conjg(fwk(ihpph  
     &+ ((j)-1) + (ns  )*((i)-1)))*fwk(iy    -1+(j))
 5669  continue
 5667  continue
       do 5670 i = 1, nm
       fwk(itmpv  + ((i)-1)) = fwk(itmpv  + ((i)-1))*fwk(ismat  + ((i)-1
     &))
 5670  continue
       do 5671 i = 1, nm
       fwk(iy    -1+(i)) = (0d0,0.0d0)
       do 5673 j = 1, nm
       fwk(iy    -1+(i)) = fwk(iy    -1+(i)) + conjg(fwk(ivtmat + ((j)-1
     &) + (ns  )*((i)-1)))*fwk(itmpv  + ((j)-1))
 5673  continue
 5671  continue
       do 5674 i = 1, nm
       fwk(iy    -1+(i)) = fwk(iy    -1+(i))*fwk(idiag  + ((i)-1))
 5674  continue
        else
          do 5675 i = nm, 1, -1
            sum = fwk(izc + mod((i)-1,ns+1))
            do 5676 j = i+1, nm
              sum = sum - fwk(iy    -1+(j))*fwk(iu   + ((i)-1) + (ns+1)*
     &((j)-1))
5676  continue
            fwk(iy    -1+(i)) = sum/fwk(iu   + ((i)-1) + (ns+1)*((i)-1))
5675  continue
        endif
        if (iparm(2) .ge. 5) then
          do 5677 i = 1, nm
        write (iparm(1),'(1x,a,i5,a,1pg20.10,2x,1pg20.10)') 'u-from-p(',
     &i,')     ', fwk(iy    -1+(i))
5677  continue
        endif
        do 5678 i = 0, nm-1
          val = fwk(iy    -1+(i+1)) * fwk(ipf + mod((i),ns+1))
          call zwsax (u,u,
     &    fwk(iqrqlp+iparm(3)*((mod((i),ns+1)))), val,  iparm , fparm , 
     &iwk , fwk , ier  )
5678  continue
      endif
*
      if (iclstg .ne. 1) then
        call zckstg (-1, iwkstg,  iparm , fparm , iwk , fwk , ier )
      endif
*
*     *---begin iteration statistics calculation----------
      if (iparm(21).eq.1 .and. ier.ge.0) go to 5679
                 go to 5680
 5679  continue
        call zfmal (ib1,iparm(3), iparm , fparm , iwk , fwk , ier )
        call zfmal (ib2,iparm(3), iparm , fparm , iwk , fwk , ier )
        call zfmal (iu1,iparm(3), iparm , fparm , iwk , fwk , ier )
        call zfmal (iu2,iparm(3), iparm , fparm , iwk , fwk , ier )
        istab = istab2
        if (iparm(19) .eq. 1) then
          call zwdot (fwk(istab+iud), uexact,
     &    uexact,  iparm , fparm , iwk , fwk , ier  )
        endif
          call zwdot (fwk(istab+ibd ), b,
     &    b,  iparm , fparm , iwk , fwk , ier  )
*        *---begin revcom precon----------
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5682
                 go to 5683
 5682  continue
          call zwset (fwk(ib2),b,
     &     iparm , fparm , iwk , fwk , ier  )
                go to 5684
 5683 continue
          call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
          iva = ivitmp
          call zwset (fwk(iva),b,
     &     iparm , fparm , iwk , fwk , ier  )
          ivql = ib2 
         ireq = 5
        iretlb = 2032
        go to 1500
 2032   continue
          call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , i
     &er )
         if (ijob .eq. -1) go to 900
 5684 continue
*        *---end revcom precon----------
          call zwdot (fwk(istab+ilbd), fwk(ib2),
     &    fwk(ib2),  iparm , fparm , iwk , fwk , ier  )
          call zwset (fwk(iu1),fwk(ib2),
     &     iparm , fparm , iwk , fwk , ier  )
*        *---begin revcom precon----------
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5685
                 go to 5686
 5685  continue
          call zwset (fwk(iu2),fwk(iu1),
     &     iparm , fparm , iwk , fwk , ier  )
                go to 5687
 5686 continue
          ivql = iu1 
          ivqr = iu2 
         ireq = 7
        iretlb = 2033
        go to 1500
 2033   continue
         if (ijob .eq. -1) go to 900
 5687 continue
*        *---end revcom precon----------
          call zwdot (fwk(istab+irlbd), fwk(iu2),
     &    fwk(iu2),  iparm , fparm , iwk , fwk , ier  )
        if (iparm(19) .eq. 1) then
          call zwsub ( fwk(iu1),
     &    u, uexact,  iparm , fparm , iwk , fwk , ier  )
          call zwdot (fwk(istab+ied), fwk(iu1),
     &    fwk(iu1),  iparm , fparm , iwk , fwk , ier  )
        endif
*        *---begin revcom matvec----------
          call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
          ivqr = ivitmp
          call zwset (fwk(ivqr),u,
     &     iparm , fparm , iwk , fwk , ier  )
          iva = ib1 
         ireq = 3
        iretlb = 2034
        go to 1500
 2034   continue
          call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , i
     &er )
         if (ijob .eq. -1) go to 900
*        *---end revcom matvec----------
          call zwsub ( fwk(ib1),
     &    b, fwk(ib1),  iparm , fparm , iwk , fwk , ier  )
          call zwdot (fwk(istab+ird), fwk(ib1),
     &    fwk(ib1),  iparm , fparm , iwk , fwk , ier  )
*        *---begin revcom precon----------
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5688
                 go to 5689
 5688  continue
          call zwset (fwk(ib2),fwk(ib1),
     &     iparm , fparm , iwk , fwk , ier  )
                go to 5690
 5689 continue
          iva = ib1 
          ivql = ib2 
         ireq = 5
        iretlb = 2035
        go to 1500
 2035   continue
         if (ijob .eq. -1) go to 900
 5690 continue
*        *---end revcom precon----------
          call zwdot (fwk(istab+ilrd ), fwk(ib2),
     &    fwk(ib2),  iparm , fparm , iwk , fwk , ier  )
          call zwset (fwk(iu1),fwk(ib2),
     &     iparm , fparm , iwk , fwk , ier  )
*        *---begin revcom precon----------
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5691
                 go to 5692
 5691  continue
          call zwset (fwk(iu2),fwk(iu1),
     &     iparm , fparm , iwk , fwk , ier  )
                go to 5693
 5692 continue
          ivql = iu1 
          ivqr = iu2 
         ireq = 7
        iretlb = 2036
        go to 1500
 2036   continue
         if (ijob .eq. -1) go to 900
 5693 continue
*        *---end revcom precon----------
          call zwdot (fwk(istab+irlrd), fwk(iu2),
     &    fwk(iu2),  iparm , fparm , iwk , fwk , ier  )
        if (iparm(19) .eq. 1) then
          if (fwk(istab1+ied) .eq. (0d0,0.0d0)) then
                ier = 4
                call xersho ( ier, 'zgmchr' , iparm ,
     &                        'Value of RELERR may be inaccurate' )
          else
            fparm(8) = sqrt(fwk(istab2+ied)/fwk(istab1+ied))
          endif
        endif
        if (fwk(istab1+ird) .eq. (0d0,0.0d0)) then
                ier = 4
                call xersho ( ier, 'zgmchr' , iparm ,
     &                        'Value of RELRSD may be inaccurate' )
        else
          fparm(7) = sqrt(fwk(istab2+ird)/fwk(istab1+ird))
        endif
        if (iparm(2) .ge. 4) then
        write (iparm(1),'()') 
        write (iparm(1),'(1x, 60(''*''))') 
        write (iparm(1),'()') 
        write (iparm(1),'(5x,''Final iteration statistics'')') 
        write (iparm(1),'()') 
        if (iparm(19) .eq. 1) then
        write (iparm(1),'(5x,''2-norm of uexact            '',1pg20.10,2
     &x,1pg20.10)') sqrt(fwk(istab+iud))
        endif
        write (iparm(1),'(5x,''2-norm of b                 '',1pg20.10,2
     &x,1pg20.10)') sqrt(fwk(istab+ibd))
        write (iparm(1),'(5x,''2-norm of Ql*b              '',1pg20.10,2
     &x,1pg20.10)') sqrt(fwk(istab+ilbd))
        write (iparm(1),'(5x,''2-norm of Qr*Ql*b           '',1pg20.10,2
     &x,1pg20.10)') sqrt(fwk(istab+irlbd))
        if (iparm(19) .eq. 1) then
        write (iparm(1),'(5x,''2-norm of error             '',1pg20.10,2
     &x,1pg20.10)') sqrt(fwk(istab+ied))
        endif
        write (iparm(1),'(5x,''2-norm of residual          '',1pg20.10,2
     &x,1pg20.10)') sqrt(fwk(istab+ird))
        write (iparm(1),'(5x,''2-norm of Ql*r              '',1pg20.10,2
     &x,1pg20.10)') sqrt(fwk(istab+ilrd))
        write (iparm(1),'(5x,''2-norm of Qr*Ql*r           '',1pg20.10,2
     &x,1pg20.10)') sqrt(fwk(istab+irlrd))
        if (iparm(19) .eq. 1) then
        write (iparm(1),'(5x,''Relative 2-norm of error    '',1pg20.10,2
     &x,1pg20.10)') fparm(8)
        endif
        write (iparm(1),'(5x,''Relative 2-norm of residual '',1pg20.10,2
     &x,1pg20.10)') fparm(7)
        write (iparm(1),'()') 
        write (iparm(1),'(1x, 60(''*''))') 
        write (iparm(1),'()') 
        endif
        call zffre (iu2,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
        call zffre (iu1,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
        call zffre (ib2,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
        call zffre (ib1,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
        call zffre (istab2,8, iparm , fparm , iwk , fwk , ier )
        call zffre (istab1,8, iparm , fparm , iwk , fwk , ier )
                go to 5681
 5680 continue
 5681 continue
*     *---end iteration statistics calculation----------
*
      call zffre (idiag ,(ns  ), iparm , fparm , iwk , fwk , ier )
      call zffre (itmpv ,(ns  ), iparm , fparm , iwk , fwk , ier )
      call zffre (iwkla ,((5*ns)+5*ns ), iparm , fparm , iwk , fwk , ier
     & )
      call zffre (ismat ,(ns  ), iparm , fparm , iwk , fwk , ier )
      call zffre (ivtmat,(ns  )*(ns  ), iparm , fparm , iwk , fwk , ier 
     &)
      call zffre (ihpr  ,(ns  ), iparm , fparm , iwk , fwk , ier )
      call zffre (ihpph ,(ns  )*(ns  ), iparm , fparm , iwk , fwk , ier 
     &)
      call zffre (ihpp  ,(ns  )*(ns+1), iparm , fparm , iwk , fwk , ier 
     &)
      call zffre (ipp   ,(ns+1)*(ns+1), iparm , fparm , iwk , fwk , ier 
     &)
      call zffre (ipzold,ns+1, iparm , fparm , iwk , fwk , ier )
      call zffre (ipz   ,ns+1, iparm , fparm , iwk , fwk , ier )
      call zffre (iyold ,ns, iparm , fparm , iwk , fwk , ier )
      call zffre (iy    ,ns, iparm , fparm , iwk , fwk , ier )
      call zffre (izc   ,ns+1, iparm , fparm , iwk , fwk , ier )
      call zffre (iu    ,ns*(ns+1), iparm , fparm , iwk , fwk , ier )
      call zffre (ih    ,ns*(ns+1), iparm , fparm , iwk , fwk , ier )
      call zffre (isin  ,ns+1, iparm , fparm , iwk , fwk , ier )
      call zffre (icos  ,ns+1, iparm , fparm , iwk , fwk , ier )
      call zffre (ipf   ,ns+1, iparm , fparm , iwk , fwk , ier )
      call zffre (iv2 ,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      call zffre (iv1 ,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      if (((iparm(13).eq.2).or.(iparm(13).eq.3))) then
        call zffre (iqrqlp,iparm(3)*(ns+1), iparm , fparm , iwk , fwk , 
     &ier )
        call zffre (iqrqlr,iparm(3)*(1), iparm , fparm , iwk , fwk , ier
     & )
      endif
      if (((iparm(13).eq.1).or.(iparm(13).eq.3)) .and. ((mod(iparm(15)/2
     &,2).eq.1)     .or.
     &    iparm(12).eq.2     .or. iparm(12).eq.6)) then
        call zffre (ip,iparm(3)*(ns+1), iparm , fparm , iwk , fwk , ier 
     &)
        call zffre (ir,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      endif
      call zffre (iqlp,iparm(3)*(ns+1), iparm , fparm , iwk , fwk , ier 
     &)
      call zffre (iqlr,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
*
*
*
*
      call zffre (iftab,19, iparm , fparm , iwk , fwk , ier )
      call xifre (iitab,55, iparm , fparm , iwk , fwk , ier )
*
      if (itimer .eq. 1) then
        call xtimer (tc2,tr2,iparm(22),0)
        fparm(1) = fparm(1) + (tc2-tc1)
        fparm(2) = fparm(2) + (tr2-tr1)
      endif
*
 910  continue
      if (iparm(27) .eq. 0) then
        call zifppr ( -1 , iparm, fparm, ier )
      endif
*
*     *---reset iclev----------
      if (iparm(27) .eq. 0) iparm(27) = 0
      iparm(11) = (iparm(6))
      ireq        = -1
      return
*
*-------------------------------brancher--------------------------------
*
*
 1000 continue
*
      ipme  = 0
      nproc = 1
      iitab = iparm(11)
      if (iitab.eq.(iparm(6)) .and. ijob.eq.-1) go to 910
*
      if (iparm(11) .eq. (iparm(6))) then
                ier = -4
                call xersho ( ier, 'zgmchr' , iparm ,
     &                        'Values of ijob and iparm(iptr) incompatib
     &le' )
      endif
*
      call xtimer (tc1,tr1,iparm(22),0)
*
*     *---remember parameters out of tables----------
        iitab  = iwk(iitab+0)
        iftab  = iwk(iitab+1)
        istab  = iwk(iitab+2)
        iwkstg = iwk(iitab+3)
        iclstg = iwk(iitab+4)
        in     = iwk(iitab+5)
        is     = iwk(iitab+6)
        itimer = iwk(iitab+7)
        iretlb = iwk(iitab+8)
        ivitmp = iwk(iitab+9)
        ivotmp = iwk(iitab+10)
        iv1tst = iwk(iitab+11)
        iv2tst = iwk(iitab+12)
        istab1 = iwk(iitab+13)
        istab2 = iwk(iitab+14)
        ib1    = iwk(iitab+15)
        ib2    = iwk(iitab+16)
        iu1    = iwk(iitab+17)
        iu2    = iwk(iitab+18)
        ir     = iwk(iitab+19)
        iqlr   = iwk(iitab+20)
        iqrqlr = iwk(iitab+21)
        ip     = iwk(iitab+22)
        iqlp   = iwk(iitab+23)
        iqrqlp = iwk(iitab+24)
        iv1    = iwk(iitab+25)
        iv2    = iwk(iitab+26)
        ipf    = iwk(iitab+27)
        icos   = iwk(iitab+28)
        isin   = iwk(iitab+29)
        ih     = iwk(iitab+30)
        iu     = iwk(iitab+31)
        izc    = iwk(iitab+32)
        iy     = iwk(iitab+33)
        iyold  = iwk(iitab+34)
        ipz    = iwk(iitab+35)
        ipzold = iwk(iitab+36)
        ns     = iwk(iitab+37)
        ipp    = iwk(iitab+38)
        ihpp   = iwk(iitab+39)
        ihpph  = iwk(iitab+40)
        ihpr   = iwk(iitab+41)
        ivtmat = iwk(iitab+42)
        ismat  = iwk(iitab+43)
        iwkla  = iwk(iitab+44)
        infola = iwk(iitab+45)
        itmpv  = iwk(iitab+46)
        idiag  = iwk(iitab+47)
        nrank  = iwk(iitab+48)
        iww    = iwk(iitab+49)
        iwr    = iwk(iitab+50)
        iwi    = iwk(iitab+51)
        nm     = iwk(iitab+52)
        i      = iwk(iitab+53)
        j      = iwk(iitab+54)
        top = fwk(iftab+0)
        bot = fwk(iftab+1)
        rdot   = fwk(iftab+2)
        rnorm  = fwk(iftab+3)
        r0norm = fwk(iftab+4)
        coe1   = fwk(iftab+5)
        coe2   = fwk(iftab+6)
        coe3   = fwk(iftab+7)
        d      = fwk(iftab+8)
        c2     = fwk(iftab+9)
        gam    = fwk(iftab+10)
        h10    = fwk(iftab+11)
        epsm   = fwk(iftab+12)
        vrmin  = fwk(iftab+13)
        vrmax  = fwk(iftab+14)
        vimin  = fwk(iftab+15)
        vimax  = fwk(iftab+16)
        al     = fwk(iftab+17)
        be     = fwk(iftab+18)
*
      if (ijob.ne.3 .and. ijob.ne.-1) then
                ier = -4
                call xersho ( ier, 'zgmchr' , iparm ,
     &                        'Argument ijob invalid' )
      endif
*
*     *---perform the branch to where we left off----------
      go to (2000,2001,2002,2003,2004,2005,2006,2007,2008,2009,2010,2011
     &,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022,2023,2024,
     &2025,2026,2027,2028,2029,2030,2031,2032,2033,2034,2035,2036),
     &       iretlb - 2000 + 1
*
*---------------------------request of caller---------------------------
*
*
 1500 continue
      iparm(11) = iitab
*     *---save tables----------
        iwk(iitab+0) = iitab 
        iwk(iitab+1) = iftab 
        iwk(iitab+2) = istab 
        iwk(iitab+3) = iwkstg
        iwk(iitab+4) = iclstg
        iwk(iitab+5) = in    
        iwk(iitab+6) = is    
        iwk(iitab+7) = itimer
        iwk(iitab+8) = iretlb
        iwk(iitab+9) = ivitmp
        iwk(iitab+10) = ivotmp
        iwk(iitab+11) = iv1tst
        iwk(iitab+12) = iv2tst
        iwk(iitab+13) = istab1
        iwk(iitab+14) = istab2
        iwk(iitab+15) = ib1   
        iwk(iitab+16) = ib2   
        iwk(iitab+17) = iu1   
        iwk(iitab+18) = iu2   
        iwk(iitab+19) = ir    
        iwk(iitab+20) = iqlr  
        iwk(iitab+21) = iqrqlr
        iwk(iitab+22) = ip    
        iwk(iitab+23) = iqlp  
        iwk(iitab+24) = iqrqlp
        iwk(iitab+25) = iv1   
        iwk(iitab+26) = iv2   
        iwk(iitab+27) = ipf   
        iwk(iitab+28) = icos  
        iwk(iitab+29) = isin  
        iwk(iitab+30) = ih    
        iwk(iitab+31) = iu    
        iwk(iitab+32) = izc   
        iwk(iitab+33) = iy    
        iwk(iitab+34) = iyold 
        iwk(iitab+35) = ipz   
        iwk(iitab+36) = ipzold
        iwk(iitab+37) = ns    
        iwk(iitab+38) = ipp   
        iwk(iitab+39) = ihpp  
        iwk(iitab+40) = ihpph 
        iwk(iitab+41) = ihpr  
        iwk(iitab+42) = ivtmat
        iwk(iitab+43) = ismat 
        iwk(iitab+44) = iwkla 
        iwk(iitab+45) = infola
        iwk(iitab+46) = itmpv 
        iwk(iitab+47) = idiag 
        iwk(iitab+48) = nrank 
        iwk(iitab+49) = iww   
        iwk(iitab+50) = iwr   
        iwk(iitab+51) = iwi   
        iwk(iitab+52) = nm    
        iwk(iitab+53) = i     
        iwk(iitab+54) = j     
        fwk(iftab+0) = top
        fwk(iftab+1) = bot
        fwk(iftab+2) = rdot  
        fwk(iftab+3) = rnorm 
        fwk(iftab+4) = r0norm
        fwk(iftab+5) = coe1  
        fwk(iftab+6) = coe2  
        fwk(iftab+7) = coe3  
        fwk(iftab+8) = d     
        fwk(iftab+9) = c2    
        fwk(iftab+10) = gam   
        fwk(iftab+11) = h10   
        fwk(iftab+12) = epsm  
        fwk(iftab+13) = vrmin 
        fwk(iftab+14) = vrmax 
        fwk(iftab+15) = vimin 
        fwk(iftab+16) = vimax 
        fwk(iftab+17) = al    
        fwk(iftab+18) = be    
      call xtimer (tc2,tr2,iparm(22),0)
      fparm(1) = fparm(1) + (tc2-tc1)
      fparm(2) = fparm(2) + (tr2-tr1)
      return
*
*
*--------------------------------end------------------------------------
*
  500 format ( 1x, a, 1pg20.10,2x,1pg20.10 )
*
      end
