************************************************************************
*
* PCG: Preconditioned Conjugate Gradient Package
* Version: f77
*
************************************************************************
************************************************************************
*# 
*#                      COPYRIGHT/DISCLAIMER NOTICE
*#
*# This program was prepared by the Regents of the University of
*# California at Los Alamos National Laboratory under Contract No.
*# W-7405-ENG-36 with the U.S. Department of Energy (DOE), and by
*# the University of Texas at Austin under ARPA Contract No.
*# DABT63-92-C-0024.
*# 
*# The University of California and the University of Texas at
*# Austin have certain rights in the program pursuant to these
*# contracts.
*# 
*# Permission is hereby granted to use the program for the user's
*# own internal use.  The user is not granted the right to reproduce,
*# prepare derivative works, or redistribute the program without
*# prior permission of the University of California or the University
*# of Texas at Austin.
*# 
*# All rights in the program are reserved by the University of
*# California and the University of Texas at Austin.
*# 
*# Portions of this material resulted from work developed under a
*# U.S. Government Contract and are subject to the following
*# license: the Government is granted for itself and others acting
*# on its behalf a paid-up, nonexclusive, irrevocable worldwide
*# license in this computer software to reproduce, prepare derivative
*# works, and perform publicly and display publicly.
*# 
*# Neither the U.S. Government, the University of California nor
*# the University of Texas at Austin, nor any of their employees,
*# makes any warranty, express or implied, or assumes any liability
*# or responsibility for the use of this software.
*# 
*# Copyright (c) 1992-1995 the University of California and the
*# University of Texas at Austin.
*# 
************************************************************************
************************************************************************
************************************************************************
** ^ROUTINE: WMVGRW - Matrix-vector product "workhorse" routine, for
**                    the regular grid-based matrix storage format
**                    (GR), as well as the union of rectangles matrix
**                    storage format, for scalar/mimd machine cases. 
**                       (^)
**
************************************************************************
**
      subroutine cmvgrw ( urflag,  ijob , ia , ja , a , vi , vo ,
     &       iwk , fwk , iparm , fparm , ier  )
**
************************************************************************
**
** ^DESCRIPTION:
**
**   This routine performs a matrix-vector multiply for two matrix
**   formats associated with regular grids: the GR rectangular grid
**   format, and the UR union of rectangles format (see test drivers
**   testt2d.fm4, test_UR.fm4).
**
**   The current strategy for how the operation is performed is as
**   follows.  For a given subgrid, each stencil point displacement, as
**   defined in the array ia, specifies a shifted subgrid, which
**   typically overlaps (up to 2**ndim) other subgrids.  Handling these
**   overlaps (referred to as slices) via routines like xslice and
**   islice is one of the key tasks.
**
**   To perform the off-processor work, one takes the box defined as the
**   minimal circumscribing box around the union of the shifted
**   subgrids.  This box defines all processors from which it is
**   necessary to obtain information.  For each processor subgrid
**   (except for the home processor), that processor subgrid is
**   intersected with the big box to determine what information to send.
**   This is buffered, sent, received.
**
**   To perform the in-processor part, one finds the maximal inscribing
**   box within the intersection of shifted subgrids, and also within
**   the processor.  This defines an area common to all subgrids,
**   for which highly optimized code can be written.
**
**   There is a third region, the region in-processor but not in the
**   intersection.  This is handled as a special case.
**
**   The off-processor slices are dealt with by locating the necessary
**   receive buffer and then finding the location of that slice within
**   the buffer.
**
**   Several numbering systems are used to describe a gridpoint's
**   location.  One is a zero-based (x varies fastest) natural ordering
**   on subgrids (NO_sub). Fore example, on a 4x4 2D subgrfid, the
**   gridpoint in the 3rd column of the 2nd row is 6. The other is a
**   zero-based (x varies fastest) natural ordering on a slice (NO_sl).
**   Another, called multiindex, is sort of a (zero-based) cartesian
**   coordinate of the gridpoint on the subgrid (MI_sub). Thus the
**   above described point would be (3,2). The last is multiindex
**   numbering, but on a slice (MI_sl).
**
**   The slices are also numbered with a zero-based (x varies fastest)
**   natural ordering. The slice dimensions and location are determined
**   in subroutine xslice.  Integer function islice is used to return a
**   NO_sub location on the destination subgrid given a NO_sl location
**   on a slice.
**
**   One should note that multiindexes and element numbers (e.g.
**   ielt, islice) are zero-based, while arrat elements (e.g. via
**   MAXPGR) are 1-based.
**
**   All operations on elements of the subgrids are processed in terms
**   of simple rectangular boxes which are subsets of the subgrid.
**   Furthermore, every box is processed by planes (nondegenerate
**   ones when they exist).  This decreases overhead and allows
**   optimization of the tight inner loops.
**
** ---------------------------------------------------------------------
**
**  Concepts
**  --------
**
**   Processor subgrid:
**     The subgrid resident on a processor.  Note that the subgrid
**     number and the processor number are related via the maps igrid
**     and iproc.
**
**   Shifted subgrid:
**     A stencil point's offset vector can be thought of as shifting a
**     processor subgrid by that displacement in NDIM-space.  Note that
**     in the transpose case, the offset vector is negated.
**
**   Circumscribing rectangle:
**     As defined by iminc and imaxc, the coordinates in the processor
**     subgrid coordinate system of the extremal corners of the
**     rectangle which is defined as the smallest circumscribing
**     rectangle containing all the shifted subgrids.
**
**   Inscribing rectangle:
**     As defined by imini and imaxi, the coordinates in the processor
**     subgrid coordinate system of the extremal corners of the
**     rectangle which is defined as the intersection of all the
**     shifted subgrids.  Note it may be empty, but this is unlikely
**     for practical problems.
**
**   Communication active processors:
**     As defined by ncpmin, ncpmax, ncp and nccprd, the rectangular
**     subset of processor subgrids which are the ones containing some
**     part of the circumscribing rectangle.  The processor you are on
**     is taken as reference and numbered (0,0,...,0).  The subgrids
**     are numbered lexicographically from 0 to ncpprd-1 by the index
**     iprc.
**
**   Communication buffers:
**     Each subgrid in the set of communication-active processors'
**     subgrids intersects the circumscribing rectangle to form a
**     subrectangle.  The size of this subrectangle defines the size
**     of the relevant communication buffer.  Note that the elements
**     of the subrectangle are numbered lexicographically and stored
**     in the buffer in memory as such.  There are actually two types
**     of buffers: vector buffers for the off-processor matvec
**     portions, and matrix buffers for the part of the matrix needed
**     for the transpose matvec.  Both involve the same grid
**     subrectangles.
**
**   Needed buffers:
**     The buffer associated with the processor I am on should not be
**     allocated.  Others may be skipped as well, if the information
**     from that buffer is never used by any stencil point.
**
**   Subgrid number vs. buffer number:
**     The variable iprc is the subgrid number in the rectangle of
**     communication-active processor subgrids.  The associated buffer
**     denoted by ibuf.  For the non-transpose case, ibuf=iprc;
**     for the transpose case, a mapping is made so that ibuf is the
**     same as the iproc value associated with the non-transpose buffer
**     of the same size.  This mapping is done so that buffers of the
**     desired size can be preallocated for use either with the
**     transpose or non-transpose.
**
**   Planes:
**     Operations on subrectangles involve an odometer-type looping
**     over all the elements in the subrectangle lexicographically,
**     amounting to the equivalent of NDIM nested loops.  For
**     efficiency, in all cases the lowest two nontrivial dimensions
**     of the subrectangle are stripped off and handled by explicit
**     loops.  The routine xplane finds these axes and also calculates
**     the relevant memory strides for vectors and matrix when looping
**     over such planes.
**
**   Grid numbering on a subrectangle:
**     The routine islice is used to convert the lexicographical
**     element number of an element in a subrectangle to a global
**     lexicographical element number in a rectangle which contains
**     that subrectangle.
**
**   Source and destination vector:
**     The position of a subrectangle in the source (VI) and
**     destination (VO) vectors differs by the offset vector defined by
**     the stencil point in question.  The position in the array is the
**     same as that of VO in the non-transpose case, and the same as
**     that of VI in the transpose case.
**
**   Plane pointers:
**     The values ivip, iap, ivop are the memory pointers to the first
**     element in the relevant plane being processed.  In the specific
**     case in which the result is being calculated on the inscribed
**     rectangle, these are pre-stored in a table for all NSTEN points
**     of the stencil.
**
**   A rectangle minus a rectangle:
**     To process a region which is a rectangle with a contained
**     rectangle removed, the method is to loop over the axes and for
**     each axis process the part of the region above and below the
**     smaller rectangle, and then make note of the deletion of that
**     processed part.  In many cases the below or above part may be
**     empty.
**
**   Slices:
**     A shifted processor subgrid cuts the original unshifted subgrid
**     into at most 2**NDIM subrectangles.  These subrectangles are
**     numbered lexicographically by the variable islno.  The routine
**     xslice is currently used to calculate the coordinates of this
**     subrectangle and what processor it is on.
**
**   Subrectangles within buffers:
**     When one of these slices is off-processor, then it will reside
**     entirely within a single communication buffer, since the buffer
**     contains all information needed from that processor by my
**     processor.  However, it may not be the full buffer.  Thus,
**     it is necessary to get the coordinates of the slice within that
**     buffer.
**
** ---------------------------------------------------------------------
**
** ^AUTHOR:    wdj@beta.lanl.gov
**
** ^MODIFIED:  wdj@lithos.c3.lanl.gov  Thu Mar  4 13:57:02 MST 1993
**
** ^ARGUMENTS: see *Subroutine Arguments* below.
**
** ^REQUIREMENTS:
**    Subroutines: See External declerations below
**
** ^SIDE_EFFECTS:
**
** ^ALGORITHM:
**   - initialize:
**       precommunicate necessary parts of transpose matrix for
**       transpose ops
**   - general case:
**       compute inscribing/circumscribing rectangles
**       post sends
**       perform matvec on inscribing-rect part
**       perform receives
**       set to zero in-processor off-inscribing-rect part
**       perform matvec on in-proc off-inscribing-rect part
**       perform matvec from off-proc part
**   - terminate:
**       deallocate transpose buffers
**
** ^REFERENCES:
**      author="M. Bromley and Steve Heller and Tim McNerny and Guy
**              Steele",
**      key="Bromley et. al.",
**      title="Fortran at Ten {Gigaflops:  The Connection Machine}
**              Convolution Compiler",
**      booktitle="Proceedings of ACM SIGPLAN 1991 Conference on
**              Programming Language Design and Implementation",
**      publisher="ACM Press",
**      year="1991",
**      pages=" "
**
** ^DOCUMENTATION:
**
** ^MACROS:
**
************************************************************************
*
*     implicit character*1 (a-z)
*
************************************************************************
*#    MVFFAL             -
*#                        (^./src/m4defs/defs_arglists.m4)
      integer ijob
      integer ier
      integer iwk(*)
      integer iparm(*)
      complex       fwk(*)
      complex       fparm(*)
      integer ia(*)
      integer ja(*)
      complex a(*)
      complex vi(*)
      complex vo(*)
      integer urflag
*
************************************************************************
*
        integer ipme
        integer nproc
      external ximini
      external ximal
      external xifre
      external cfmini
      external cfmal
      external cffre
      integer  nv
      integer  ndim
      integer  nsten
      integer  nb
      integer  ns
      integer  imaxgr
      external imaxgr
      integer  ivaxgr
      external ivaxgr
      integer isten
      integer islno
      integer iaxis
      integer jaxis
      integer ielt
      integer nelt
      integer ibr, ibc
      integer iin1, iin2
      integer nin1, nin2
      integer istrd1
      integer istrd2
      integer ivstr1
      integer ivstr2
      integer ivstrr
      integer imstr1
      integer imstr2
      integer imstrr
      integer imstrc
      integer ivop
      integer ivip
      integer iap
      integer imbufp
      integer ivbufp
      integer ivipt, iapt
      integer ijaptr
      integer iaxin1, iaxin2
      integer isgnt
      integer ilohi
      integer i
        integer ivi1
        integer ia1
        integer ivi2
        integer ia2
        integer ivi3
        integer ia3
        integer ivi4
        integer ia4
        integer ivi5
        integer ia5
      integer isteno, iin1o, nsteni, nin1i, iin1i, ivo0
      integer imini(32)
      integer imaxi(32)
      integer iminc(32)
      integer imaxc(32)
      integer imins(32)
      integer imind(32)
      integer imaxd(32)
      integer inelt(32)
      integer imi  (32)
      integer imi2 (32)
*
************************************************************************
      integer  imodf
      external imodf
      external ii2mi
      integer  islice
      external islice
      external xslur
      external xslice
      external xplane
      external chad0
      external chad
      external cchad0
      external cchad
*
************************************************************************
*     iceil(i,j) = -ifloor(-i,j)
*
************************************************************************
*
**#   IOFST - stencil offset extracted from ja
*
**#   NSBGR - number of subgrid points along an axis
*
**#   NPROCS - number of processors along an axis
*
**#   NTAB - size of table of pointers stored across calls
*
**#   OFFSTN - magnitude of maximum offset along -ve axial direction
**#   in user's definition of neighborhood information
*
**#   OFFSTP - magnitude of maximum offset along +ve axial direction
**#   in user's definition of neighborhood information.
*
**#   NBR - processor number of processor in the $1 position in the
**#    neighborhood of the $2 processor as defined by the user. 
*
*=======================================================================
*#    APLANE(   ibr,ibc,iin1,iin2)
*#    APLANEIND(ibr,ibc,iin1,iin2)
*#                      - reference an element of a in a plane.
*#                        indices are 0-based for speed.
*=======================================================================
*#    VIPLANE(   ib,iin1,iin2)
*#    VIPLANEIND(ib,iin1,iin2)
*#                      - reference an element of vi in a plane.
*#                        indices are 0-based for speed.
*=======================================================================
*#    VOPLANE(   ib,iin1,iin2)
*#    VOPLANEIND(ib,iin1,iin2)
*#                      - reference an element of vo in a plane.
*#                        indices are 0-based for speed.
*=======================================================================
*#    MBUFPLANE(ibr,ibc,iin1,iin2)
*#                      - reference an element of a matrix buffer
*#                        in a plane.  indices are 0-based for speed.
*=======================================================================
*#    VBUFPLANE(ibr,iin1,iin2)
*#                      - reference an element of a vector buffer
*#                        in a plane.  indices are 0-based for speed.
*=======================================================================
*#    GET_IN_RECT(ipm)
*#                      - fill out imini/imaxi, the extents of the
*#                        inscribed rectangle, in grid points.
*#                        ipm=1 regular case, -1 transpose case.
*#                        (^)
*=======================================================================
*#    GET_CIRCUM_RECT(ipm)
*#                      - fill out iminc/imaxc, the extents of the
*#                        circumscribing rectangle, in grid points.
*#                        ipm=1 regular case, -1 transpose case.
*#                        (^)
*=======================================================================
*#    GET_CA_PROCS(ipm)
*#                  - fill out ncpmin/max, ncp, ncpprd, the information
*#                    on the communication active processors.
*#                    (^)
*=======================================================================
*#    GET_IPSEND_IPRECV(ipm)
*#                  - fill out ipsend/iprecv based on iprc and ncp*.
*#                    Also fill out iprcmi, the multiindex of iprc in
*#                    the ncpmin/max coord system.
*#                    imi2/3 temporaries.
*#                    (^)
*=======================================================================
*#    CHECK_WHETHER_BUFFER_NEEDED(ipm,iline)
*#                      - figure out whether we will really need to use
*#                        anything from the processor in question.
*#                        Assumes iprcmi.
*#                        (^)
*=======================================================================
*#    GET_SUBGRID_INTERSECT_CIRCUM_RECT(ipm)
*#                  - fill out imins/imaxs, the coords of the rectangle
*#                    that intersects the circumscribed rectangle and
*#                    the subgrid specified in iprcmi.  Also nelt.
*#                    (^)
*
****^^******************************************************************
*     $Modified: wdj@lanl.gov Mon Aug 28 19:20:55 MDT 1995
*     $Id: mvgrw.fm4,v 1.13 1994/11/22 05:19:39 joubert Exp $
*     $Revision: 1.0 $
************************************************************************
*
*-----------------------------------------------------------------------
*-------------------------general initializations-----------------------
*-----------------------------------------------------------------------
*
      ipme  = 0
      nproc = 1
      ivipt = (iparm(6))
      iapt = (iparm(6))
*
      nv     = iparm(3     )
      ndim   = ia   (1 )
      nsten  = ia   (2)
      nb     = ia   (3   )
      ns     = nv / nb
          if ((ijob) .eq. (3)) then
           isgnt =  1
          else
           isgnt = -1
          endif
*
*-----------------------------------------------------------------------
*--------------------------initialization case--------------------------
*-----------------------------------------------------------------------
*
      if (ijob.eq.1) then
        iparm(11) = (iparm(6))
        go to 900
      endif
*
*-----------------------------------------------------------------------
*----------------------------termination case---------------------------
*-----------------------------------------------------------------------
*
      if (ijob.eq.-1) then
        go to 900
      endif
*
*-----------------------------------------------------------------------
*------------------------------other cases------------------------------
*-----------------------------------------------------------------------
*
      if (ijob.ne.3 .and. ijob.ne.4) then
                ier = -4
                call xersho ( ier, 'cmvgrw' , iparm , ' ' )
                go to 900
      endif
*
*-----------------------------------------------------------------------
*  ---simpler (but very slow) code for uniprocessor case----------------
*
*-----------------------------------------------------------------------
*  ---compute information for inscribed and circumscribed rectangles----
*
          do 5500 iaxis = 1, ndim
            iminc(iaxis) = 0              +(isgnt)*ja(iaxis+ndim*(1    -
     &1))
       do 5501 isten = 2, nsten
       iminc(iaxis) = min(iminc(iaxis),
     &             0                      +(isgnt)*ja(iaxis+ndim*(isten-
     &1)))
 5501  continue
            imaxc(iaxis) = ja(iaxis+ndim*nsten)-1+(isgnt)*ja(iaxis+ndim*
     &(1    -1))
       do 5502 isten = 2, nsten
       imaxc(iaxis) = max(imaxc(iaxis),
     &             ja(iaxis+ndim*nsten)-1        +(isgnt)*ja(iaxis+ndim*
     &(isten-1)))
 5502  continue
5500  continue
*
          do 5503 iaxis = 1, ndim
            imini(iaxis) = 0              +(isgnt)*ja(iaxis+ndim*(1    -
     &1))
       do 5504 isten = 2, nsten
       imini(iaxis) = max(imini(iaxis),
     &                     0              +(isgnt)*ja(iaxis+ndim*(isten-
     &1)))
 5504  continue
            imaxi(iaxis) = ja(iaxis+ndim*nsten)-1+(isgnt)*ja(iaxis+ndim*
     &(1    -1))
       do 5505 isten = 2, nsten
       imaxi(iaxis) = min(imaxi(iaxis),
     &                     ja(iaxis+ndim*nsten)-1+(isgnt)*ja(iaxis+ndim*
     &(isten-1)))
 5505  continue
5503  continue
*
*-----------------------------------------------------------------------
*  ---perform sends-----------------------------------------------------
*
*
*-----------------------------------------------------------------------
*  ---perform matvec on inscribed rect----------------------------------
*
       do 5506 iaxis = 1, ndim
       inelt(iaxis) = max(0,imaxi(iaxis)-imini(iaxis)+1)
 5506  continue
      nelt = 1
       do 5507 iaxis = 1, ndim
       nelt = nelt * inelt(iaxis)
 5507  continue
      if (nelt .eq. 0) go to 101
*
      call ximal (ivipt,nsten, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call ximal (iapt ,nsten, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
*
      call xplane ( ja(1+ndim*nsten), inelt, ia, iparm,
     &              iaxin1, iaxin2, istrd1, istrd2, nin1, nin2,
     &              ivstrr,ivstr1,ivstr2, imstrr,imstrc,imstr1,imstr2)
*
      do 5508 ielt = 0, nelt-1, nin1*nin2
        ivop =    ivaxgr(1,1+islice(ielt,ndim,ja(1+ndim*nsten),inelt,imi
     &ni),ia(5),nb,ns,nsten)
        do 5509 isten = 1, nsten
       do 5510 iaxis = 1, ndim
       imins(iaxis) = imini(iaxis) + isgnt*ja(iaxis+ndim*(isten-1))
 5510  continue
          ivip =  ivaxgr(1,1+islice(ielt,ndim,ja(1+ndim*nsten),inelt,imi
     &ns),ia(5),nb,ns,nsten)
          if (ijob .eq. 3) then
            iap = imaxgr(1,1,isten,1+islice(ielt,ndim,ja(1+ndim*nsten),i
     &nelt,imini),ia(4),nb,ns,nsten)
          else
            iap = imaxgr(1,1,isten,1+islice(ielt,ndim,ja(1+ndim*nsten),i
     &nelt,imins),ia(4),nb,ns,nsten)
          endif
          iwk(ivipt-1+isten) = ivip
          iwk(iapt -1+isten) = iap
5509  continue
        if (ijob .eq. 3) then
          if (nb .eq. 1) then
            do 5511 iin1o = 0, nin1-1, 100
              do 5512 iin2 = 0, nin2-1
                do 5513 isteno = 1, nsten, 5
                    isten = isteno + 0
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap  = iwk(iapt -1+isten)
                      ivi1 = ivip+(0)*ivstrr
     &                          +(iin1o)*ivstr1+(iin2)*ivstr2
                      ia1  =  iap+(0)*imstrr+(0)*imstrc
     &                       +(iin1o)*imstr1+(iin2)*imstr2
                    endif
                    isten = isteno + 1
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap  = iwk(iapt -1+isten)
                      ivi2 = ivip+(0)*ivstrr
     &                          +(iin1o)*ivstr1+(iin2)*ivstr2
                      ia2  =  iap+(0)*imstrr+(0)*imstrc
     &                       +(iin1o)*imstr1+(iin2)*imstr2
                    endif
                    isten = isteno + 2
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap  = iwk(iapt -1+isten)
                      ivi3 = ivip+(0)*ivstrr
     &                          +(iin1o)*ivstr1+(iin2)*ivstr2
                      ia3  =  iap+(0)*imstrr+(0)*imstrc
     &                       +(iin1o)*imstr1+(iin2)*imstr2
                    endif
                    isten = isteno + 3
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap  = iwk(iapt -1+isten)
                      ivi4 = ivip+(0)*ivstrr
     &                          +(iin1o)*ivstr1+(iin2)*ivstr2
                      ia4  =  iap+(0)*imstrr+(0)*imstrc
     &                       +(iin1o)*imstr1+(iin2)*imstr2
                    endif
                    isten = isteno + 4
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap  = iwk(iapt -1+isten)
                      ivi5 = ivip+(0)*ivstrr
     &                          +(iin1o)*ivstr1+(iin2)*ivstr2
                      ia5  =  iap+(0)*imstrr+(0)*imstrc
     &                       +(iin1o)*imstr1+(iin2)*imstr2
                    endif
                  ivo0 = ivop+(0)*ivstrr
     &                          +(iin1o)*ivstr1+(iin2)*ivstr2
                  nsteni = min(5,nsten-isteno+1)
                  nin1i  = min(100,nin1-iin1o)
                  go to (701,702,703,704,705), nsteni
 701        continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5514 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                            a(ia1 +(iin1i)       )*
     &                           vi(ivi1+(iin1i)       ) 
5514  continue
                        else
      do 5515 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                            a(ia1 +(iin1i)*imstr1)*
     &                           vi(ivi1+(iin1i)*ivstr1) 
5515  continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5516 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                           vo(ivo0  +(iin1i)       ) 
     &                         +  a(ia1 +(iin1i)       )*
     &                           vi(ivi1+(iin1i)       ) 
5516  continue
                        else
      do 5517 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                           vo(ivo0  +(iin1i)*ivstr1) 
     &                         +  a(ia1 +(iin1i)*imstr1)*
     &                           vi(ivi1+(iin1i)*ivstr1) 
5517  continue
                        endif
                      endif
                    go to 700
 702        continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5518 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                            a(ia1 +(iin1i)       )*
     &                           vi(ivi1+(iin1i)       ) 
     &                         +  a(ia2 +(iin1i)       )*
     &                           vi(ivi2+(iin1i)       ) 
5518  continue
                        else
      do 5519 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                            a(ia1 +(iin1i)*imstr1)*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                         +  a(ia2 +(iin1i)*imstr1)*
     &                           vi(ivi2+(iin1i)*ivstr1) 
5519  continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5520 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                           vo(ivo0  +(iin1i)       ) 
     &                         +  a(ia1 +(iin1i)       )*
     &                           vi(ivi1+(iin1i)       ) 
     &                         +  a(ia2 +(iin1i)       )*
     &                           vi(ivi2+(iin1i)       ) 
5520  continue
                        else
      do 5521 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                           vo(ivo0  +(iin1i)*ivstr1) 
     &                         +  a(ia1 +(iin1i)*imstr1)*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                         +  a(ia2 +(iin1i)*imstr1)*
     &                           vi(ivi2+(iin1i)*ivstr1) 
5521  continue
                        endif
                      endif
                    go to 700
 703        continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5522 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                            a(ia1 +(iin1i)       )*
     &                           vi(ivi1+(iin1i)       ) 
     &                         +  a(ia2 +(iin1i)       )*
     &                           vi(ivi2+(iin1i)       ) 
     &                         +  a(ia3 +(iin1i)       )*
     &                           vi(ivi3+(iin1i)       ) 
5522  continue
                        else
      do 5523 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                            a(ia1 +(iin1i)*imstr1)*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                         +  a(ia2 +(iin1i)*imstr1)*
     &                           vi(ivi2+(iin1i)*ivstr1) 
     &                         +  a(ia3 +(iin1i)*imstr1)*
     &                           vi(ivi3+(iin1i)*ivstr1) 
5523  continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5524 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                           vo(ivo0  +(iin1i)       ) 
     &                         +  a(ia1 +(iin1i)       )*
     &                           vi(ivi1+(iin1i)       ) 
     &                         +  a(ia2 +(iin1i)       )*
     &                           vi(ivi2+(iin1i)       ) 
     &                         +  a(ia3 +(iin1i)       )*
     &                           vi(ivi3+(iin1i)       ) 
5524  continue
                        else
      do 5525 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                           vo(ivo0  +(iin1i)*ivstr1) 
     &                         +  a(ia1 +(iin1i)*imstr1)*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                         +  a(ia2 +(iin1i)*imstr1)*
     &                           vi(ivi2+(iin1i)*ivstr1) 
     &                         +  a(ia3 +(iin1i)*imstr1)*
     &                           vi(ivi3+(iin1i)*ivstr1) 
5525  continue
                        endif
                      endif
                    go to 700
 704        continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5526 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                            a(ia1 +(iin1i)       )*
     &                           vi(ivi1+(iin1i)       ) 
     &                         +  a(ia2 +(iin1i)       )*
     &                           vi(ivi2+(iin1i)       ) 
     &                         +  a(ia3 +(iin1i)       )*
     &                           vi(ivi3+(iin1i)       ) 
     &                         +  a(ia4 +(iin1i)       )*
     &                           vi(ivi4+(iin1i)       ) 
5526  continue
                        else
      do 5527 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                            a(ia1 +(iin1i)*imstr1)*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                         +  a(ia2 +(iin1i)*imstr1)*
     &                           vi(ivi2+(iin1i)*ivstr1) 
     &                         +  a(ia3 +(iin1i)*imstr1)*
     &                           vi(ivi3+(iin1i)*ivstr1) 
     &                         +  a(ia4 +(iin1i)*imstr1)*
     &                           vi(ivi4+(iin1i)*ivstr1) 
5527  continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5528 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                           vo(ivo0  +(iin1i)       ) 
     &                         +  a(ia1 +(iin1i)       )*
     &                           vi(ivi1+(iin1i)       ) 
     &                         +  a(ia2 +(iin1i)       )*
     &                           vi(ivi2+(iin1i)       ) 
     &                         +  a(ia3 +(iin1i)       )*
     &                           vi(ivi3+(iin1i)       ) 
     &                         +  a(ia4 +(iin1i)       )*
     &                           vi(ivi4+(iin1i)       ) 
5528  continue
                        else
      do 5529 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                           vo(ivo0  +(iin1i)*ivstr1) 
     &                         +  a(ia1 +(iin1i)*imstr1)*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                         +  a(ia2 +(iin1i)*imstr1)*
     &                           vi(ivi2+(iin1i)*ivstr1) 
     &                         +  a(ia3 +(iin1i)*imstr1)*
     &                           vi(ivi3+(iin1i)*ivstr1) 
     &                         +  a(ia4 +(iin1i)*imstr1)*
     &                           vi(ivi4+(iin1i)*ivstr1) 
5529  continue
                        endif
                      endif
                    go to 700
 705        continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5530 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                            a(ia1 +(iin1i)       )*
     &                           vi(ivi1+(iin1i)       ) 
     &                         +  a(ia2 +(iin1i)       )*
     &                           vi(ivi2+(iin1i)       ) 
     &                         +  a(ia3 +(iin1i)       )*
     &                           vi(ivi3+(iin1i)       ) 
     &                         +  a(ia4 +(iin1i)       )*
     &                           vi(ivi4+(iin1i)       ) 
     &                         +  a(ia5 +(iin1i)       )*
     &                           vi(ivi5+(iin1i)       ) 
5530  continue
                        else
      do 5531 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                            a(ia1 +(iin1i)*imstr1)*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                         +  a(ia2 +(iin1i)*imstr1)*
     &                           vi(ivi2+(iin1i)*ivstr1) 
     &                         +  a(ia3 +(iin1i)*imstr1)*
     &                           vi(ivi3+(iin1i)*ivstr1) 
     &                         +  a(ia4 +(iin1i)*imstr1)*
     &                           vi(ivi4+(iin1i)*ivstr1) 
     &                         +  a(ia5 +(iin1i)*imstr1)*
     &                           vi(ivi5+(iin1i)*ivstr1) 
5531  continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5532 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                           vo(ivo0  +(iin1i)       ) 
     &                         +  a(ia1 +(iin1i)       )*
     &                           vi(ivi1+(iin1i)       ) 
     &                         +  a(ia2 +(iin1i)       )*
     &                           vi(ivi2+(iin1i)       ) 
     &                         +  a(ia3 +(iin1i)       )*
     &                           vi(ivi3+(iin1i)       ) 
     &                         +  a(ia4 +(iin1i)       )*
     &                           vi(ivi4+(iin1i)       ) 
     &                         +  a(ia5 +(iin1i)       )*
     &                           vi(ivi5+(iin1i)       ) 
5532  continue
                        else
      do 5533 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                           vo(ivo0  +(iin1i)*ivstr1) 
     &                         +  a(ia1 +(iin1i)*imstr1)*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                         +  a(ia2 +(iin1i)*imstr1)*
     &                           vi(ivi2+(iin1i)*ivstr1) 
     &                         +  a(ia3 +(iin1i)*imstr1)*
     &                           vi(ivi3+(iin1i)*ivstr1) 
     &                         +  a(ia4 +(iin1i)*imstr1)*
     &                           vi(ivi4+(iin1i)*ivstr1) 
     &                         +  a(ia5 +(iin1i)*imstr1)*
     &                           vi(ivi5+(iin1i)*ivstr1) 
5533  continue
                        endif
                      endif
                    go to 700
 700              continue
5513  continue
5512  continue
5511  continue
          else
            do 5534 isten = 1, nsten
              ivip = iwk(ivipt-1+isten)
              iap  = iwk(iapt -1+isten)
              do 5535 ibc  = 0, nb-1
                if (isten.eq.1 .and. ibc.eq.0) then
       do 5536 iin2 = 0, nin2-1
       do 5537 ibr = 0, nb-1
       call chad0 (nin1, vo(ivop+(ibr)*ivstrr
     &                          +(0)*ivstr1+(iin2)*ivstr2), ivstr1,
     &                                      a(iap+(ibr)*imstrr+(ibc)*ims
     &trc
     &                       +(0)*imstr1+(iin2)*imstr2), imstr1,
     &                                     vi(ivip+(ibc)*ivstrr
     &                          +(0)*ivstr1+(iin2)*ivstr2), ivstr1)
 5537  continue
 5536  continue
**#               DOALL([VOPLANE(ibr,    iin1,iin2) = 
**#  &                    APLANE(ibr,ibc,iin1,iin2)*
**#  &                   VIPLANE(    ibc,iin1,iin2)],
**#                     iin1,0,nin1-1,iin2,0,nin2-1,ibr,0,nb-1)
                else
       do 5538 iin2 = 0, nin2-1
       do 5539 ibr = 0, nb-1
       call chad  (nin1, vo(ivop+(ibr)*ivstrr
     &                          +(0)*ivstr1+(iin2)*ivstr2), ivstr1,
     &                                      a(iap+(ibr)*imstrr+(ibc)*ims
     &trc
     &                       +(0)*imstr1+(iin2)*imstr2), imstr1,
     &                                     vi(ivip+(ibc)*ivstrr
     &                          +(0)*ivstr1+(iin2)*ivstr2), ivstr1)
 5539  continue
 5538  continue
**#               DOALL([VOPLANE(ibr,    iin1,iin2) = 
**#  &                   VOPLANE(ibr,    iin1,iin2) +
**#  &                    APLANE(ibr,ibc,iin1,iin2)*
**#  &                   VIPLANE(    ibc,iin1,iin2)],
**#                       iin1,0,nin1-1,iin2,0,nin2-1,ibr,0,nb-1)
                endif
5535  continue
5534  continue
          endif
        else if (iparm(42) .eq. 1) then
          if (nb .eq. 1) then
            do 5540 iin1o = 0, nin1-1, 100
              do 5541 iin2 = 0, nin2-1
                do 5542 isteno = 1, nsten, 5
                    isten = isteno + 0
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap  = iwk(iapt -1+isten)
                      ivi1 = ivip+(0)*ivstrr
     &                          +(iin1o)*ivstr1+(iin2)*ivstr2
                      ia1  =  iap+(0)*imstrr+(0)*imstrc
     &                       +(iin1o)*imstr1+(iin2)*imstr2
                    endif
                    isten = isteno + 1
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap  = iwk(iapt -1+isten)
                      ivi2 = ivip+(0)*ivstrr
     &                          +(iin1o)*ivstr1+(iin2)*ivstr2
                      ia2  =  iap+(0)*imstrr+(0)*imstrc
     &                       +(iin1o)*imstr1+(iin2)*imstr2
                    endif
                    isten = isteno + 2
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap  = iwk(iapt -1+isten)
                      ivi3 = ivip+(0)*ivstrr
     &                          +(iin1o)*ivstr1+(iin2)*ivstr2
                      ia3  =  iap+(0)*imstrr+(0)*imstrc
     &                       +(iin1o)*imstr1+(iin2)*imstr2
                    endif
                    isten = isteno + 3
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap  = iwk(iapt -1+isten)
                      ivi4 = ivip+(0)*ivstrr
     &                          +(iin1o)*ivstr1+(iin2)*ivstr2
                      ia4  =  iap+(0)*imstrr+(0)*imstrc
     &                       +(iin1o)*imstr1+(iin2)*imstr2
                    endif
                    isten = isteno + 4
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap  = iwk(iapt -1+isten)
                      ivi5 = ivip+(0)*ivstrr
     &                          +(iin1o)*ivstr1+(iin2)*ivstr2
                      ia5  =  iap+(0)*imstrr+(0)*imstrc
     &                       +(iin1o)*imstr1+(iin2)*imstr2
                    endif
                  ivo0 = ivop+(0)*ivstrr
     &                          +(iin1o)*ivstr1+(iin2)*ivstr2
                  nsteni = min(5,nsten-isteno+1)
                  nin1i  = min(100,nin1-iin1o)
                  go to (601,602,603,604,605), nsteni
 601        continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5543 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                      conjg(a(ia1 +(iin1i)       ))*
     &                           vi(ivi1+(iin1i)       ) 
5543  continue
                        else
      do 5544 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                      conjg(a(ia1 +(iin1i)*imstr1))*
     &                           vi(ivi1+(iin1i)*ivstr1) 
5544  continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5545 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                           vo(ivo0  +(iin1i)       ) 
     &                   +  conjg(a(ia1 +(iin1i)       ))*
     &                           vi(ivi1+(iin1i)       ) 
5545  continue
                        else
      do 5546 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                           vo(ivo0  +(iin1i)*ivstr1) 
     &                   +  conjg(a(ia1 +(iin1i)*imstr1))*
     &                           vi(ivi1+(iin1i)*ivstr1) 
5546  continue
                        endif
                      endif
                    go to 600
 602        continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5547 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                      conjg(a(ia1 +(iin1i)       ))*
     &                           vi(ivi1+(iin1i)       ) 
     &                   +  conjg(a(ia2 +(iin1i)       ))*
     &                           vi(ivi2+(iin1i)       ) 
5547  continue
                        else
      do 5548 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                      conjg(a(ia1 +(iin1i)*imstr1))*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                   +  conjg(a(ia2 +(iin1i)*imstr1))*
     &                           vi(ivi2+(iin1i)*ivstr1) 
5548  continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5549 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                           vo(ivo0  +(iin1i)       ) 
     &                   +  conjg(a(ia1 +(iin1i)       ))*
     &                           vi(ivi1+(iin1i)       ) 
     &                   +  conjg(a(ia2 +(iin1i)       ))*
     &                           vi(ivi2+(iin1i)       ) 
5549  continue
                        else
      do 5550 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                           vo(ivo0  +(iin1i)*ivstr1) 
     &                   +  conjg(a(ia1 +(iin1i)*imstr1))*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                   +  conjg(a(ia2 +(iin1i)*imstr1))*
     &                           vi(ivi2+(iin1i)*ivstr1) 
5550  continue
                        endif
                      endif
                    go to 600
 603        continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5551 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                      conjg(a(ia1 +(iin1i)       ))*
     &                           vi(ivi1+(iin1i)       ) 
     &                   +  conjg(a(ia2 +(iin1i)       ))*
     &                           vi(ivi2+(iin1i)       ) 
     &                   +  conjg(a(ia3 +(iin1i)       ))*
     &                           vi(ivi3+(iin1i)       ) 
5551  continue
                        else
      do 5552 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                      conjg(a(ia1 +(iin1i)*imstr1))*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                   +  conjg(a(ia2 +(iin1i)*imstr1))*
     &                           vi(ivi2+(iin1i)*ivstr1) 
     &                   +  conjg(a(ia3 +(iin1i)*imstr1))*
     &                           vi(ivi3+(iin1i)*ivstr1) 
5552  continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5553 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                           vo(ivo0  +(iin1i)       ) 
     &                   +  conjg(a(ia1 +(iin1i)       ))*
     &                           vi(ivi1+(iin1i)       ) 
     &                   +  conjg(a(ia2 +(iin1i)       ))*
     &                           vi(ivi2+(iin1i)       ) 
     &                   +  conjg(a(ia3 +(iin1i)       ))*
     &                           vi(ivi3+(iin1i)       ) 
5553  continue
                        else
      do 5554 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                           vo(ivo0  +(iin1i)*ivstr1) 
     &                   +  conjg(a(ia1 +(iin1i)*imstr1))*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                   +  conjg(a(ia2 +(iin1i)*imstr1))*
     &                           vi(ivi2+(iin1i)*ivstr1) 
     &                   +  conjg(a(ia3 +(iin1i)*imstr1))*
     &                           vi(ivi3+(iin1i)*ivstr1) 
5554  continue
                        endif
                      endif
                    go to 600
 604        continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5555 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                      conjg(a(ia1 +(iin1i)       ))*
     &                           vi(ivi1+(iin1i)       ) 
     &                   +  conjg(a(ia2 +(iin1i)       ))*
     &                           vi(ivi2+(iin1i)       ) 
     &                   +  conjg(a(ia3 +(iin1i)       ))*
     &                           vi(ivi3+(iin1i)       ) 
     &                   +  conjg(a(ia4 +(iin1i)       ))*
     &                           vi(ivi4+(iin1i)       ) 
5555  continue
                        else
      do 5556 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                      conjg(a(ia1 +(iin1i)*imstr1))*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                   +  conjg(a(ia2 +(iin1i)*imstr1))*
     &                           vi(ivi2+(iin1i)*ivstr1) 
     &                   +  conjg(a(ia3 +(iin1i)*imstr1))*
     &                           vi(ivi3+(iin1i)*ivstr1) 
     &                   +  conjg(a(ia4 +(iin1i)*imstr1))*
     &                           vi(ivi4+(iin1i)*ivstr1) 
5556  continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5557 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                           vo(ivo0  +(iin1i)       ) 
     &                   +  conjg(a(ia1 +(iin1i)       ))*
     &                           vi(ivi1+(iin1i)       ) 
     &                   +  conjg(a(ia2 +(iin1i)       ))*
     &                           vi(ivi2+(iin1i)       ) 
     &                   +  conjg(a(ia3 +(iin1i)       ))*
     &                           vi(ivi3+(iin1i)       ) 
     &                   +  conjg(a(ia4 +(iin1i)       ))*
     &                           vi(ivi4+(iin1i)       ) 
5557  continue
                        else
      do 5558 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                           vo(ivo0  +(iin1i)*ivstr1) 
     &                   +  conjg(a(ia1 +(iin1i)*imstr1))*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                   +  conjg(a(ia2 +(iin1i)*imstr1))*
     &                           vi(ivi2+(iin1i)*ivstr1) 
     &                   +  conjg(a(ia3 +(iin1i)*imstr1))*
     &                           vi(ivi3+(iin1i)*ivstr1) 
     &                   +  conjg(a(ia4 +(iin1i)*imstr1))*
     &                           vi(ivi4+(iin1i)*ivstr1) 
5558  continue
                        endif
                      endif
                    go to 600
 605        continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5559 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                      conjg(a(ia1 +(iin1i)       ))*
     &                           vi(ivi1+(iin1i)       ) 
     &                   +  conjg(a(ia2 +(iin1i)       ))*
     &                           vi(ivi2+(iin1i)       ) 
     &                   +  conjg(a(ia3 +(iin1i)       ))*
     &                           vi(ivi3+(iin1i)       ) 
     &                   +  conjg(a(ia4 +(iin1i)       ))*
     &                           vi(ivi4+(iin1i)       ) 
     &                   +  conjg(a(ia5 +(iin1i)       ))*
     &                           vi(ivi5+(iin1i)       ) 
5559  continue
                        else
      do 5560 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                      conjg(a(ia1 +(iin1i)*imstr1))*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                   +  conjg(a(ia2 +(iin1i)*imstr1))*
     &                           vi(ivi2+(iin1i)*ivstr1) 
     &                   +  conjg(a(ia3 +(iin1i)*imstr1))*
     &                           vi(ivi3+(iin1i)*ivstr1) 
     &                   +  conjg(a(ia4 +(iin1i)*imstr1))*
     &                           vi(ivi4+(iin1i)*ivstr1) 
     &                   +  conjg(a(ia5 +(iin1i)*imstr1))*
     &                           vi(ivi5+(iin1i)*ivstr1) 
5560  continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5561 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                           vo(ivo0  +(iin1i)       ) 
     &                   +  conjg(a(ia1 +(iin1i)       ))*
     &                           vi(ivi1+(iin1i)       ) 
     &                   +  conjg(a(ia2 +(iin1i)       ))*
     &                           vi(ivi2+(iin1i)       ) 
     &                   +  conjg(a(ia3 +(iin1i)       ))*
     &                           vi(ivi3+(iin1i)       ) 
     &                   +  conjg(a(ia4 +(iin1i)       ))*
     &                           vi(ivi4+(iin1i)       ) 
     &                   +  conjg(a(ia5 +(iin1i)       ))*
     &                           vi(ivi5+(iin1i)       ) 
5561  continue
                        else
      do 5562 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                           vo(ivo0  +(iin1i)*ivstr1) 
     &                   +  conjg(a(ia1 +(iin1i)*imstr1))*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                   +  conjg(a(ia2 +(iin1i)*imstr1))*
     &                           vi(ivi2+(iin1i)*ivstr1) 
     &                   +  conjg(a(ia3 +(iin1i)*imstr1))*
     &                           vi(ivi3+(iin1i)*ivstr1) 
     &                   +  conjg(a(ia4 +(iin1i)*imstr1))*
     &                           vi(ivi4+(iin1i)*ivstr1) 
     &                   +  conjg(a(ia5 +(iin1i)*imstr1))*
     &                           vi(ivi5+(iin1i)*ivstr1) 
5562  continue
                        endif
                      endif
                    go to 600
 600              continue
5542  continue
5541  continue
5540  continue
          else
            do 5563 isten = 1, nsten
              ivip = iwk(ivipt-1+isten)
              iap  = iwk(iapt -1+isten)
              do 5564 ibc  = 0, nb-1
                if (isten.eq.1 .and. ibc.eq.0) then
       do 5565 iin2 = 0, nin2-1
       do 5566 ibr = 0, nb-1
       call cchad0 (nin1, vo(ivop+(ibr)*ivstrr
     &                          +(0)*ivstr1+(iin2)*ivstr2), ivstr1,
     &                                       a(iap+(ibc)*imstrr+(ibr)*im
     &strc
     &                       +(0)*imstr1+(iin2)*imstr2), imstr1,
     &                                      vi(ivip+(ibc)*ivstrr
     &                          +(0)*ivstr1+(iin2)*ivstr2), ivstr1)
 5566  continue
 5565  continue
                else
       do 5567 iin2 = 0, nin2-1
       do 5568 ibr = 0, nb-1
       call cchad  (nin1, vo(ivop+(ibr)*ivstrr
     &                          +(0)*ivstr1+(iin2)*ivstr2), ivstr1,
     &                                       a(iap+(ibc)*imstrr+(ibr)*im
     &strc
     &                       +(0)*imstr1+(iin2)*imstr2), imstr1,
     &                                      vi(ivip+(ibc)*ivstrr
     &                          +(0)*ivstr1+(iin2)*ivstr2), ivstr1)
 5568  continue
 5567  continue
                endif
5564  continue
5563  continue
          endif   
        else
          if (nb .eq. 1) then
            do 5569 iin1o = 0, nin1-1, 100
              do 5570 iin2 = 0, nin2-1
                do 5571 isteno = 1, nsten, 5
                    isten = isteno + 0
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap  = iwk(iapt -1+isten)
                      ivi1 = ivip+(0)*ivstrr
     &                          +(iin1o)*ivstr1+(iin2)*ivstr2
                      ia1  =  iap+(0)*imstrr+(0)*imstrc
     &                       +(iin1o)*imstr1+(iin2)*imstr2
                    endif
                    isten = isteno + 1
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap  = iwk(iapt -1+isten)
                      ivi2 = ivip+(0)*ivstrr
     &                          +(iin1o)*ivstr1+(iin2)*ivstr2
                      ia2  =  iap+(0)*imstrr+(0)*imstrc
     &                       +(iin1o)*imstr1+(iin2)*imstr2
                    endif
                    isten = isteno + 2
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap  = iwk(iapt -1+isten)
                      ivi3 = ivip+(0)*ivstrr
     &                          +(iin1o)*ivstr1+(iin2)*ivstr2
                      ia3  =  iap+(0)*imstrr+(0)*imstrc
     &                       +(iin1o)*imstr1+(iin2)*imstr2
                    endif
                    isten = isteno + 3
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap  = iwk(iapt -1+isten)
                      ivi4 = ivip+(0)*ivstrr
     &                          +(iin1o)*ivstr1+(iin2)*ivstr2
                      ia4  =  iap+(0)*imstrr+(0)*imstrc
     &                       +(iin1o)*imstr1+(iin2)*imstr2
                    endif
                    isten = isteno + 4
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap  = iwk(iapt -1+isten)
                      ivi5 = ivip+(0)*ivstrr
     &                          +(iin1o)*ivstr1+(iin2)*ivstr2
                      ia5  =  iap+(0)*imstrr+(0)*imstrc
     &                       +(iin1o)*imstr1+(iin2)*imstr2
                    endif
                  ivo0 = ivop+(0)*ivstrr
     &                          +(iin1o)*ivstr1+(iin2)*ivstr2
                  nsteni = min(5,nsten-isteno+1)
                  nin1i  = min(100,nin1-iin1o)
                  go to (801,802,803,804,805), nsteni
 801        continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5572 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                            a(ia1 +(iin1i)       )*
     &                           vi(ivi1+(iin1i)       ) 
5572  continue
                        else
      do 5573 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                            a(ia1 +(iin1i)*imstr1)*
     &                           vi(ivi1+(iin1i)*ivstr1) 
5573  continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5574 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                           vo(ivo0  +(iin1i)       ) 
     &                         +  a(ia1 +(iin1i)       )*
     &                           vi(ivi1+(iin1i)       ) 
5574  continue
                        else
      do 5575 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                           vo(ivo0  +(iin1i)*ivstr1) 
     &                         +  a(ia1 +(iin1i)*imstr1)*
     &                           vi(ivi1+(iin1i)*ivstr1) 
5575  continue
                        endif
                      endif
                    go to 800
 802        continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5576 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                            a(ia1 +(iin1i)       )*
     &                           vi(ivi1+(iin1i)       ) 
     &                         +  a(ia2 +(iin1i)       )*
     &                           vi(ivi2+(iin1i)       ) 
5576  continue
                        else
      do 5577 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                            a(ia1 +(iin1i)*imstr1)*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                         +  a(ia2 +(iin1i)*imstr1)*
     &                           vi(ivi2+(iin1i)*ivstr1) 
5577  continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5578 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                           vo(ivo0  +(iin1i)       ) 
     &                         +  a(ia1 +(iin1i)       )*
     &                           vi(ivi1+(iin1i)       ) 
     &                         +  a(ia2 +(iin1i)       )*
     &                           vi(ivi2+(iin1i)       ) 
5578  continue
                        else
      do 5579 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                           vo(ivo0  +(iin1i)*ivstr1) 
     &                         +  a(ia1 +(iin1i)*imstr1)*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                         +  a(ia2 +(iin1i)*imstr1)*
     &                           vi(ivi2+(iin1i)*ivstr1) 
5579  continue
                        endif
                      endif
                    go to 800
 803        continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5580 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                            a(ia1 +(iin1i)       )*
     &                           vi(ivi1+(iin1i)       ) 
     &                         +  a(ia2 +(iin1i)       )*
     &                           vi(ivi2+(iin1i)       ) 
     &                         +  a(ia3 +(iin1i)       )*
     &                           vi(ivi3+(iin1i)       ) 
5580  continue
                        else
      do 5581 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                            a(ia1 +(iin1i)*imstr1)*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                         +  a(ia2 +(iin1i)*imstr1)*
     &                           vi(ivi2+(iin1i)*ivstr1) 
     &                         +  a(ia3 +(iin1i)*imstr1)*
     &                           vi(ivi3+(iin1i)*ivstr1) 
5581  continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5582 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                           vo(ivo0  +(iin1i)       ) 
     &                         +  a(ia1 +(iin1i)       )*
     &                           vi(ivi1+(iin1i)       ) 
     &                         +  a(ia2 +(iin1i)       )*
     &                           vi(ivi2+(iin1i)       ) 
     &                         +  a(ia3 +(iin1i)       )*
     &                           vi(ivi3+(iin1i)       ) 
5582  continue
                        else
      do 5583 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                           vo(ivo0  +(iin1i)*ivstr1) 
     &                         +  a(ia1 +(iin1i)*imstr1)*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                         +  a(ia2 +(iin1i)*imstr1)*
     &                           vi(ivi2+(iin1i)*ivstr1) 
     &                         +  a(ia3 +(iin1i)*imstr1)*
     &                           vi(ivi3+(iin1i)*ivstr1) 
5583  continue
                        endif
                      endif
                    go to 800
 804        continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5584 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                            a(ia1 +(iin1i)       )*
     &                           vi(ivi1+(iin1i)       ) 
     &                         +  a(ia2 +(iin1i)       )*
     &                           vi(ivi2+(iin1i)       ) 
     &                         +  a(ia3 +(iin1i)       )*
     &                           vi(ivi3+(iin1i)       ) 
     &                         +  a(ia4 +(iin1i)       )*
     &                           vi(ivi4+(iin1i)       ) 
5584  continue
                        else
      do 5585 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                            a(ia1 +(iin1i)*imstr1)*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                         +  a(ia2 +(iin1i)*imstr1)*
     &                           vi(ivi2+(iin1i)*ivstr1) 
     &                         +  a(ia3 +(iin1i)*imstr1)*
     &                           vi(ivi3+(iin1i)*ivstr1) 
     &                         +  a(ia4 +(iin1i)*imstr1)*
     &                           vi(ivi4+(iin1i)*ivstr1) 
5585  continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5586 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                           vo(ivo0  +(iin1i)       ) 
     &                         +  a(ia1 +(iin1i)       )*
     &                           vi(ivi1+(iin1i)       ) 
     &                         +  a(ia2 +(iin1i)       )*
     &                           vi(ivi2+(iin1i)       ) 
     &                         +  a(ia3 +(iin1i)       )*
     &                           vi(ivi3+(iin1i)       ) 
     &                         +  a(ia4 +(iin1i)       )*
     &                           vi(ivi4+(iin1i)       ) 
5586  continue
                        else
      do 5587 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                           vo(ivo0  +(iin1i)*ivstr1) 
     &                         +  a(ia1 +(iin1i)*imstr1)*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                         +  a(ia2 +(iin1i)*imstr1)*
     &                           vi(ivi2+(iin1i)*ivstr1) 
     &                         +  a(ia3 +(iin1i)*imstr1)*
     &                           vi(ivi3+(iin1i)*ivstr1) 
     &                         +  a(ia4 +(iin1i)*imstr1)*
     &                           vi(ivi4+(iin1i)*ivstr1) 
5587  continue
                        endif
                      endif
                    go to 800
 805        continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5588 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                            a(ia1 +(iin1i)       )*
     &                           vi(ivi1+(iin1i)       ) 
     &                         +  a(ia2 +(iin1i)       )*
     &                           vi(ivi2+(iin1i)       ) 
     &                         +  a(ia3 +(iin1i)       )*
     &                           vi(ivi3+(iin1i)       ) 
     &                         +  a(ia4 +(iin1i)       )*
     &                           vi(ivi4+(iin1i)       ) 
     &                         +  a(ia5 +(iin1i)       )*
     &                           vi(ivi5+(iin1i)       ) 
5588  continue
                        else
      do 5589 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                            a(ia1 +(iin1i)*imstr1)*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                         +  a(ia2 +(iin1i)*imstr1)*
     &                           vi(ivi2+(iin1i)*ivstr1) 
     &                         +  a(ia3 +(iin1i)*imstr1)*
     &                           vi(ivi3+(iin1i)*ivstr1) 
     &                         +  a(ia4 +(iin1i)*imstr1)*
     &                           vi(ivi4+(iin1i)*ivstr1) 
     &                         +  a(ia5 +(iin1i)*imstr1)*
     &                           vi(ivi5+(iin1i)*ivstr1) 
5589  continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
      do 5590 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)       ) =
     &                           vo(ivo0  +(iin1i)       ) 
     &                         +  a(ia1 +(iin1i)       )*
     &                           vi(ivi1+(iin1i)       ) 
     &                         +  a(ia2 +(iin1i)       )*
     &                           vi(ivi2+(iin1i)       ) 
     &                         +  a(ia3 +(iin1i)       )*
     &                           vi(ivi3+(iin1i)       ) 
     &                         +  a(ia4 +(iin1i)       )*
     &                           vi(ivi4+(iin1i)       ) 
     &                         +  a(ia5 +(iin1i)       )*
     &                           vi(ivi5+(iin1i)       ) 
5590  continue
                        else
      do 5591 iin1i = 0, nin1i-1
        vo(ivo0  +(iin1i)*ivstr1) =
     &                           vo(ivo0  +(iin1i)*ivstr1) 
     &                         +  a(ia1 +(iin1i)*imstr1)*
     &                           vi(ivi1+(iin1i)*ivstr1) 
     &                         +  a(ia2 +(iin1i)*imstr1)*
     &                           vi(ivi2+(iin1i)*ivstr1) 
     &                         +  a(ia3 +(iin1i)*imstr1)*
     &                           vi(ivi3+(iin1i)*ivstr1) 
     &                         +  a(ia4 +(iin1i)*imstr1)*
     &                           vi(ivi4+(iin1i)*ivstr1) 
     &                         +  a(ia5 +(iin1i)*imstr1)*
     &                           vi(ivi5+(iin1i)*ivstr1) 
5591  continue
                        endif
                      endif
                    go to 800
 800              continue
5571  continue
5570  continue
5569  continue
          else
            do 5592 isten = 1, nsten
              ivip = iwk(ivipt-1+isten)
              iap  = iwk(iapt -1+isten)
              do 5593 ibc  = 0, nb-1
                if (isten.eq.1 .and. ibc.eq.0) then
       do 5594 iin2 = 0, nin2-1
       do 5595 ibr = 0, nb-1
       call chad0  (nin1, vo(ivop+(ibr)*ivstrr
     &                          +(0)*ivstr1+(iin2)*ivstr2), ivstr1,
     &                                       a(iap+(ibc)*imstrr+(ibr)*im
     &strc
     &                       +(0)*imstr1+(iin2)*imstr2), imstr1,
     &                                      vi(ivip+(ibc)*ivstrr
     &                          +(0)*ivstr1+(iin2)*ivstr2), ivstr1)
 5595  continue
 5594  continue
                else
       do 5596 iin2 = 0, nin2-1
       do 5597 ibr = 0, nb-1
       call chad   (nin1, vo(ivop+(ibr)*ivstrr
     &                          +(0)*ivstr1+(iin2)*ivstr2), ivstr1,
     &                                       a(iap+(ibc)*imstrr+(ibr)*im
     &strc
     &                       +(0)*imstr1+(iin2)*imstr2), imstr1,
     &                                      vi(ivip+(ibc)*ivstrr
     &                          +(0)*ivstr1+(iin2)*ivstr2), ivstr1)
 5597  continue
 5596  continue
                endif
5593  continue
5592  continue
          endif
        endif
5508  continue
*
      call xifre (iapt ,nsten, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call xifre (ivipt,nsten, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
*
 101  continue
*
*-----------------------------------------------------------------------
*  ---perform receives--------------------------------------------------
*
*
*-----------------------------------------------------------------------
*  ---set to zero the in-processor off-intersection part----------------
*
*
       do 5598 iaxis = 1, ndim
       imind(iaxis) = 0
 5598  continue
       do 5599 iaxis = 1, ndim
       imaxd(iaxis) = ja(iaxis+ndim*nsten) - 1
 5599  continue
*
      do 5600 iaxis = 1, ndim
        do 5601 ilohi = 1, 2
          if (ilohi .eq. 1) then
            if (imind(iaxis) .ge. imini(iaxis)) goto 10
       do 5602 jaxis = 1, ndim
       imi  (jaxis) = imind(jaxis)
 5602  continue
       do 5603 jaxis = 1, ndim
       inelt(jaxis) =
     &             max(0,    imaxd(jaxis)                -imi(jaxis)+1)
 5603  continue
                   inelt(iaxis) =
     &             max(0,min(imaxd(iaxis),imini(iaxis)-1)-imi(iaxis)+1)
          else
            if (imaxd(iaxis) .le. imaxi(iaxis)) goto 10
       do 5604 jaxis = 1, ndim
       imi  (jaxis) = imind(jaxis)
 5604  continue
                   imi  (iaxis) = max(imind(iaxis),imaxi(iaxis)+1)
       do 5605 jaxis = 1, ndim
       inelt(jaxis) = max(0,imaxd(jaxis)    -imi(jaxis)+1)
 5605  continue
          endif
          nelt = 1
       do 5606 jaxis = 1, ndim
       nelt = nelt * inelt(jaxis)
 5606  continue
          if (nelt .eq. 0) goto 10
          call xplane ( ja(1+ndim*nsten), inelt, ia, iparm,
     &                  iaxin1, iaxin2, istrd1, istrd2, nin1, nin2,
     &                  ivstrr,ivstr1,ivstr2, imstrr,imstrc,imstr1,imstr
     &2)
          do 5607 ielt = 0, nelt-1, nin1*nin2
            ivop = ivaxgr(1,1+islice(ielt,ndim,ja(1+ndim*nsten),inelt,im
     &i),ia(5),nb,ns,nsten)
       do 5608 iin1 = 0, nin1-1
       do 5609 iin2 = 0, nin2-1
       do 5610 ibr = 0, nb-1
       vo(ivop+(ibr)*ivstrr
     &                          +(iin1)*ivstr1+(iin2)*ivstr2) = (0e0,0.0
     &e0)
 5610  continue
 5609  continue
 5608  continue
5607  continue
 10       continue
5601  continue
        imind(iaxis) = max(imind(iaxis),imini(iaxis))
        imaxd(iaxis) = min(imaxd(iaxis),imaxi(iaxis))
5600  continue
*
*-----------------------------------------------------------------------
*  ---Handle parts off the inscribed rect but using on-proc data--------
*
*
      do 5611 isten = 1, nsten
        do 5612 islno = 0, 2**ndim-1
          if(urflag.eq.1)then
            call xslur ( ia, ja, isten, islno, ijob, imins, imind, inelt
     &,
     &        nelt  )
          else
            call xslice ( ia, ja, isten, islno, ijob, imins, imind, inel
     &t,
     &        nelt  )
          endif
          if (nelt .eq. 0) goto 20
*
*
       do 5613 iaxis = 1, ndim
       imaxd(iaxis) = imind(iaxis)+inelt(iaxis)-1
 5613  continue
*
          do 5614 iaxis = 1, ndim
            do 5615 ilohi = 1, 2
              if (ilohi .eq. 1) then
                if (imind(iaxis) .ge. imini(iaxis)) goto 30
       do 5616 jaxis = 1, ndim
       imi  (jaxis) = imind(jaxis)
 5616  continue
       do 5617 jaxis = 1, ndim
       imi2 (jaxis) = imins(jaxis)
 5617  continue
       do 5618 jaxis = 1, ndim
       inelt(jaxis) =
     &                 max(0,    imaxd(jaxis)                -imi(jaxis)
     &+1)
 5618  continue
                       inelt(iaxis) =
     &                 max(0,min(imaxd(iaxis),imini(iaxis)-1)-imi(iaxis)
     &+1)
              else
                if (imaxd(iaxis) .le. imaxi(iaxis)) goto 30
       do 5619 jaxis = 1, ndim
       imi  (jaxis) =     imind(jaxis)
 5619  continue
                       imi  (iaxis) = max(imind(iaxis),imaxi(iaxis)+1)
       do 5620 jaxis = 1, ndim
       imi2 (jaxis) = imins(jaxis)
 5620  continue
                       imi2 (iaxis) = imins(iaxis)-(imind(iaxis)-imi(iax
     &is))
       do 5621 jaxis = 1, ndim
       inelt(jaxis) = max(0,imaxd(jaxis)    -imi(jaxis)+1)
 5621  continue
              endif
              nelt = 1
       do 5622 jaxis = 1, ndim
       nelt = nelt * inelt(jaxis)
 5622  continue
              if (nelt .eq. 0) goto 30
              call xplane ( ja(1+ndim*nsten), inelt, ia, iparm,
     &                      iaxin1, iaxin2, istrd1, istrd2, nin1, nin2,
     &                      ivstrr,ivstr1,ivstr2, imstrr,imstrc,imstr1,i
     &mstr2)
              do 5623 ielt = 0, nelt-1, nin1*nin2
                ivop =  ivaxgr(1,1+islice(ielt,ndim,ja(1+ndim*nsten),ine
     &lt,imi),ia(5),nb,ns,nsten)
                ivip =  ivaxgr(1,1+islice(ielt,ndim,ja(1+ndim*nsten),ine
     &lt,imi2),ia(5),nb,ns,nsten)
                if (ijob .eq. 3) then
                  iap = imaxgr(1,1,isten,1+islice(ielt,ndim,ja(1+ndim*ns
     &ten),inelt,imi),ia(4),nb,ns,nsten)
                else
                  iap = imaxgr(1,1,isten,1+islice(ielt,ndim,ja(1+ndim*ns
     &ten),inelt,imi2),ia(4),nb,ns,nsten)
                endif
                if (ijob .eq. 3) then
       do 5624 iin2 = 0, nin2-1
       do 5625 ibr = 0, nb-1
       do 5626 ibc = 0, nb-1
       call chad (nin1, vo(ivop+(ibr)*ivstrr
     &                          +(0)*ivstr1+(iin2)*ivstr2), ivstr1,
     &                                   a(iap+(ibr)*imstrr+(ibc)*imstrc
     &                       +(0)*imstr1+(iin2)*imstr2), imstr1,
     &                                  vi(ivip+(ibc)*ivstrr
     &                          +(0)*ivstr1+(iin2)*ivstr2), ivstr1)
 5626  continue
 5625  continue
 5624  continue
                else if (iparm(42) .eq. 1) then
       do 5627 iin2 = 0, nin2-1
       do 5628 ibr = 0, nb-1
       do 5629 ibc = 0, nb-1
       call cchad (nin1, vo(ivop+(ibr)*ivstrr
     &                          +(0)*ivstr1+(iin2)*ivstr2), ivstr1,
     &                                    a(iap+(ibc)*imstrr+(ibr)*imstr
     &c
     &                       +(0)*imstr1+(iin2)*imstr2), imstr1,
     &                                   vi(ivip+(ibc)*ivstrr
     &                          +(0)*ivstr1+(iin2)*ivstr2), ivstr1)
 5629  continue
 5628  continue
 5627  continue
                else
       do 5630 iin2 = 0, nin2-1
       do 5631 ibr = 0, nb-1
       do 5632 ibc = 0, nb-1
       call chad  (nin1, vo(ivop+(ibr)*ivstrr
     &                          +(0)*ivstr1+(iin2)*ivstr2), ivstr1,
     &                                    a(iap+(ibc)*imstrr+(ibr)*imstr
     &c
     &                       +(0)*imstr1+(iin2)*imstr2), imstr1,
     &                                   vi(ivip+(ibc)*ivstrr
     &                          +(0)*ivstr1+(iin2)*ivstr2), ivstr1)
 5632  continue
 5631  continue
 5630  continue
                endif
5623  continue
 30           continue
5615  continue
            imins(iaxis) = imins(iaxis) -
     &                    (imind(iaxis)-max(imind(iaxis),imini(iaxis)))
            imind(iaxis) = max(imind(iaxis),imini(iaxis))
            imaxd(iaxis) = min(imaxd(iaxis),imaxi(iaxis))
5614  continue
*
*-----------------------------------------------------------------------
*  ---Handle parts off-proc---------------------------------------------
*
*
 20       continue
5612  continue
5611  continue
*
*-----------------------------------------------------------------------
*  ---Flop count--------------------------------------------------------
*
       fparm(13) = fparm(13) + (iparm(3)*2*(2.*2*nb*nsten-1))
*
*-----------------------------------------------------------------------
*  ---done--------------------------------------------------------------
*-----------------------------------------------------------------------
*
 900  continue
*
      if (iparm(11).eq.(iparm(6))) go to 910
*
      if (ijob.eq.-1 .or. ier.lt.0) then
        iparm(11) = (iparm(6))
      endif
*
 910  continue
      return
      end
*-----------------------------------------------------------------------
