************************************************************************
**  ^ROUTINE: WCGR   - Reverse communication level routine for
**                     conjugate gradient acceleration. (^)
**
      subroutine zcgr ( ijob , ireq , u , uexact , b , iva , ivql , 
     &   ivqr , iwk , fwk , iparm , fparm , ier )
**
**  ^DESCRIPTION:
**    Basic preconditioned conjugate gradient routine.  Left, right and
**    split preconditioning are allowed.  In all cases, the quantity
**    (u^{(n)}-u,A(u^{(n)}-u)) is minimized over the relevant Krylov
**    space.  The matrices A and (Q_R Q_L) should be Hermitian positive
**    definite to insure that the method is well-behaved.
**
**  ^AUTHOR:   wdj@beta.lanl.gov
**
**  ^MODIFIED: wdj@lithos.c3.lanl.gov Thu May  6 13:12:38 MDT 1993
**
**  ^ARGUMENTS: see *Subroutine Arguments* below.
**
**  ^REQUIREMENTS:
**    Common Blocks: none
**    Subroutines:   see below.
**
**  ^SIDE_EFFECTS:
**    See the definition of METHRDCLS to see which parameters
**    are modified
**
**  ^ALGORITHM:
**   $$ p^{(0)}  = Q_R Q_L r^{(0)} $$
**   $$ p^{(n)}  = Q_R Q_L r^{(n)} + \alpha_n p^{(n-1)}, \qquad n>0 $$
**   $$ \alpha_n = {( r^{(n  )}, Q_R Q_L r^{(n  )}) \over
**                  ( r^{(n-1)}, Q_R Q_L r^{(n-1)})} $$
**   $$ u^{(n+1)} = u^{(n)} + \lambda_n   p^{(n)}
**      r^{(n+1)} = r^{(n)} - \lambda_n A p^{(n)} $$
**   $$ \lambda_n = {( r^{(n)}, Q_R Q_L r^{(n)}) \over
**                   ( p^{(n)}, A       p^{(n)})} $$
**
**  ^REFERENCES:
**    M. R. Hestenes and E. L. Stiefel, ``Methods of Conjugate Gradients
**    for Solving Linear Systems,'' {\sl J. Res. Nat. Bur. Standards},
**    vol. 49, 1952, pp. 409-436.
**
**   Steven F. Ashby, Thomas A. Manteuffel and Paul E. Saylor,
**   ``A Taxonomy for Conjugate Gradient Methods,'' {\sl SIAM J. Numer.
**   Anal.}, 27 (1990), pp. 1542--1568.
**
**  ^DOCUMENTATION:
**
************************************************************************
*
      implicit none
          Include 'fcube.h'
*         Include 'veclib.h'
*
*#    METHRAL        - (^)
      integer ijob
      integer ireq
      integer ier
       integer iwk(*)
       integer iparm(*)
       double complex fwk(*)
       double complex fparm(*)
       double complex u(*)
       double complex uexact(*)
       double complex b(*)
      integer ivql
      integer iva
      integer ivqr
*
        integer itimer
        double precision tc1, tr1, tc2, tr2
      integer iitab
      integer iftab
      integer iwkstg
      integer iclstg
      integer in
      integer is
      integer iretlb
      integer ivitmp, ivotmp
      integer iv1tst, iv2tst
      integer istab1, istab2, istab
      integer ib1,ib2, iu1,iu2
      integer iud
      parameter (iud =0)
      integer ibd
      parameter (ibd =1)
      integer ilbd
      parameter (ilbd =2)
      integer irlbd
      parameter (irlbd=3)
      integer ied
      parameter (ied =4)
      integer ird
      parameter (ird =5)
      integer ilrd
      parameter (ilrd =6)
      integer irlrd
      parameter (irlrd=7)
      double complex urndf
      double complex top
      double complex bot
      parameter (urndf=.1)
      double complex mydot1, mydot2
      external zargck
      external zckstg
      external zwrnd
      external zifppr
      external zitput
      external xtimer
      integer ipme
      integer iphost
      integer log2np
      integer nproc
      integer iom
      character*72 errstr
      external ximini
      external ximal
      external xifre
      external zfmini
      external zfmal
      external zffre
*     integer imv1, imv2        ! temporaries - for matvec
      integer ir
      integer ip
      integer iap
      integer iqlr
      integer iqrqlr
      integer iv1
      integer i
      double complex alphan
      double complex dectr
      double complex pap
      double complex rnorm
      double complex r0norm
      double complex rr
      double complex rrold
      double complex vlamda
      double complex uudot, uexdt
*
*
****^^******************************************************************
*     $Modified: spencer@navier.ae.utexas.edu on Thu May 16 11:59:51 1996 $
*     $Id: cgr.fm4,v 1.35 1994/11/22 05:20:14 joubert Exp $
*     $Revision: 1.35 $
************************************************************************
*
      iitab = (iparm(6))
      iftab = (iparm(6))
      ip = (iparm(6))
      iap = (iparm(6))
      ir = (iparm(6))
      iv1 = (iparm(6))
*
*
      if (ijob .ne. 1) go to 1000
*
      iclstg = 1
      itimer = 0
      in = 0
      iparm(5) = in
      is = 0
*
      iu1 = (iparm(6))
      iu2 = (iparm(6))
      ib1 = (iparm(6))
      ib2 = (iparm(6))
      istab1 = (iparm(6))
      istab2 = (iparm(6))
*
      ipme = mynode ( )
      iphost = myhost ( )
      log2np = nodedim ( )
      nproc = 2**log2np
      if (iparm(27) .eq. 0) iparm(27) = 0
      if (iparm(27) .eq. 0) then
        ier = 0
        if (ijob.eq.1 .or. ijob.eq.2 .or. ijob.eq.0) then
      call ximini ( iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmini ( iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        endif
        call zargck ( iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        call xtimer (tc1,tr1,iparm(22),1)
      endif
*
      call xtimer (tc1,tr1,iparm(22),0)
      itimer = 1
*
      if (iparm(27) .eq. 0) then
        call zifppr ( 1 , iparm, fparm, ier)
           if (ier .lt. 0) go to 900
      endif
*
      call ximal (iitab,21, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (iftab,10, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
*
*
      if (iparm(2) .ge. 3) then
        if (ipme .eq. 0) then
          write (iparm(1),'(/''     Conjugate Gradient Method.''/)')
        endif
      endif
*
      call zfmal (ip ,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (iap,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (ir ,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      if (.not.((iparm(13).eq.2).or.(iparm(13).eq.3))) then
        if (.not.((iparm(13).eq.1).or.(iparm(13).eq.3))) then
          iqlr = ir
          iqrqlr = iqlr
        else
          iqlr = iap
          iqrqlr = iqlr
        endif
      else
        if (.not.((iparm(13).eq.1).or.(iparm(13).eq.3))) then
          iqlr = ir
          iqrqlr = iap
        else
          iqlr = iap
      call zfmal (iqrqlr,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        endif
      endif
      call zfmal (iv1,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
*
      if (iparm(14) .eq. -2 .or. iparm(14) .eq. -1) then
          call zwfil (u,(0d0,0.0d0), iparm , fparm , iwk , fwk , ier )
      endif
*
      if (iparm(14) .eq. 2 ) go to 5500
                 go to 5501
 5500  continue
          call zwrdot (mydot1,b, b, iparm , fparm , iwk , fwk , ier )
        call zwrnd (u, iparm , fparm , iwk , fwk , ier )
      call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call zwset (fwk(ivqr),u, iparm , fparm , iwk , fwk , ier )
          iva = ir
       ireq = 3
      iretlb = 2000
      go to 1500
 2000 continue
      call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
          call zwrdot (mydot2,fwk(ir), fwk(ir), iparm , fparm , iwk , 
     &       fwk , ier )
          call zwscl ( u, u, urndf*sqrt(mydot1/mydot2), iparm , fparm , 
     &       iwk , fwk , ier )
                go to 5502
 5501 continue
 5502 continue
*
      if (iparm(14) .eq. 3) go to 5503
                 go to 5504
 5503  continue
      call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call zwset (fwk(ivqr),u, iparm , fparm , iwk , fwk , ier )
          iva = ir
       ireq = 3
      iretlb = 2001
      go to 1500
 2001 continue
      call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
          call zwsub ( fwk(ir), b, fwk(ir), iparm , fparm , iwk , fwk , 
     &       ier )
          call zwrdot (mydot1,fwk(ir), fwk(ir), iparm , fparm , iwk , 
     &       fwk , ier )
          call zwset (fwk(ir),u, iparm , fparm , iwk , fwk , ier )
        call zwrnd (u, iparm , fparm , iwk , fwk , ier )
      call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call zwset (fwk(ivqr),u, iparm , fparm , iwk , fwk , ier )
          iva = ip
       ireq = 3
      iretlb = 2002
      go to 1500
 2002 continue
      call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
          call zwrdot (mydot2,fwk(ip), fwk(ip), iparm , fparm , iwk , 
     &       fwk , ier )
          call zwsax (u,fwk(ir), u, -urndf*sqrt(mydot1/mydot2), iparm , 
     &       fparm , iwk , fwk , ier )
                go to 5505
 5504 continue
 5505 continue
*
      if (iparm(21) .eq. 1) go to 5506
                 go to 5507
 5506  continue
      call zfmal (istab1,8, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (istab2,8, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (ib1,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (ib2,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (iu1,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (iu2,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        istab = istab1
        if (iparm(19) .eq. 1) then
          call zwdot (fwk(istab+iud), uexact, uexact, iparm , fparm , 
     &       iwk , fwk , ier )
        endif
          call zwdot (fwk(istab+ibd ), b, b, iparm , fparm , iwk , fwk ,
     &        ier )
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5509
                 go to 5510
 5509  continue
          call zwset (fwk(ib2),b, iparm , fparm , iwk , fwk , ier )
                go to 5511
 5510 continue
      call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          iva = ivitmp
          call zwset (fwk(iva),b, iparm , fparm , iwk , fwk , ier )
          ivql = ib2
       ireq = 5
      iretlb = 2003
      go to 1500
 2003 continue
      call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
 5511 continue
          call zwdot (fwk(istab+ilbd), fwk(ib2), fwk(ib2), iparm , 
     &       fparm , iwk , fwk , ier )
          call zwset (fwk(iu1),fwk(ib2), iparm , fparm , iwk , fwk , 
     &       ier )
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5512
                 go to 5513
 5512  continue
          call zwset (fwk(iu2),fwk(iu1), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5514
 5513 continue
          ivql = iu1
          ivqr = iu2
       ireq = 7
      iretlb = 2004
      go to 1500
 2004 continue
       if (ijob .eq. -1) go to 900
 5514 continue
          call zwdot (fwk(istab+irlbd), fwk(iu2), fwk(iu2), iparm , 
     &       fparm , iwk , fwk , ier )
        if (iparm(19) .eq. 1) then
          call zwsub ( fwk(iu1), u, uexact, iparm , fparm , iwk , fwk , 
     &       ier )
          call zwdot (fwk(istab+ied), fwk(iu1), fwk(iu1), iparm , fparm 
     &       , iwk , fwk , ier )
        endif
      call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call zwset (fwk(ivqr),u, iparm , fparm , iwk , fwk , ier )
          iva = ib1
       ireq = 3
      iretlb = 2005
      go to 1500
 2005 continue
      call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
          call zwsub ( fwk(ib1), b, fwk(ib1), iparm , fparm , iwk , fwk 
     &       , ier )
          call zwdot (fwk(istab+ird), fwk(ib1), fwk(ib1), iparm , fparm 
     &       , iwk , fwk , ier )
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5515
                 go to 5516
 5515  continue
          call zwset (fwk(ib2),fwk(ib1), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5517
 5516 continue
          iva = ib1
          ivql = ib2
       ireq = 5
      iretlb = 2006
      go to 1500
 2006 continue
       if (ijob .eq. -1) go to 900
 5517 continue
          call zwdot (fwk(istab+ilrd ), fwk(ib2), fwk(ib2), iparm , 
     &       fparm , iwk , fwk , ier )
          call zwset (fwk(iu1),fwk(ib2), iparm , fparm , iwk , fwk , 
     &       ier )
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5518
                 go to 5519
 5518  continue
          call zwset (fwk(iu2),fwk(iu1), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5520
 5519 continue
          ivql = iu1
          ivqr = iu2
       ireq = 7
      iretlb = 2007
      go to 1500
 2007 continue
       if (ijob .eq. -1) go to 900
 5520 continue
          call zwdot (fwk(istab+irlrd), fwk(iu2), fwk(iu2), iparm , 
     &       fparm , iwk , fwk , ier )
        if (iparm(2) .ge. 4) then
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(1x, 60(''*''))')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''Initial iteration statistics'')')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (iparm(19) .eq. 1) then
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of uexact            '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+iud))
        endif
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of b                 '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+ibd))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Ql*b              '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+ilbd))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Qr*Ql*b           '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+irlbd))
        endif
        if (iparm(19) .eq. 1) then
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of error             '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+ied))
        endif
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of residual          '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+ird))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Ql*r              '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+ilrd))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Qr*Ql*r           '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+irlrd))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(1x, 60(''*''))')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        endif
      call zffre (iu2,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zffre (iu1,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zffre (ib2,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zffre (ib1,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
                go to 5508
 5507 continue
 5508 continue
*
      if (((iparm(14).eq.0).or.(iparm(14).eq.-2)) .or. iparm(14).eq.-1) 
     &   go to 5521
                 go to 5522
 5521  continue
          call zwset (fwk(ir),b, iparm , fparm , iwk , fwk , ier )
                go to 5523
 5522 continue
      call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call zwset (fwk(ivqr),u, iparm , fparm , iwk , fwk , ier )
          iva = ir
       ireq = 3
      iretlb = 2008
      go to 1500
 2008 continue
      call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
          call zwsub ( fwk(ir), b, fwk(ir), iparm , fparm , iwk , fwk , 
     &       ier )
 5523 continue
*
      if (((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5524
                 go to 5525
 5524  continue
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5527
                 go to 5528
 5527  continue
          call zwset (fwk(iqlr),fwk(ir), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5529
 5528 continue
          iva = ir
          ivql = iqlr
       ireq = 5
      iretlb = 2009
      go to 1500
 2009 continue
       if (ijob .eq. -1) go to 900
 5529 continue
                go to 5526
 5525 continue
 5526 continue
      if (((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5530
                 go to 5531
 5530  continue
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5533
                 go to 5534
 5533  continue
          call zwset (fwk(iqrqlr),fwk(iqlr), iparm , fparm , iwk , fwk ,
     &        ier )
                go to 5535
 5534 continue
          ivql = iqlr
          ivqr = iqrqlr
       ireq = 7
      iretlb = 2010
      go to 1500
 2010 continue
       if (ijob .eq. -1) go to 900
 5535 continue
                go to 5532
 5531 continue
 5532 continue
*
          call zwrdot (dectr,fwk(ir), fwk(iqrqlr), iparm , fparm , iwk ,
     &        fwk , ier )
      rnorm = sqrt(abs(dectr))
      r0norm = rnorm
      if (iparm(20) .eq. 1) then
        rr = dectr
      else
          call zwcdot (rr, fwk(ir), fwk(iqrqlr), iparm(20), iparm , 
     &       fparm , iwk , fwk , ier )
      endif
*
*---------------------------iteration loop------------------------------
*
  100 continue
*
        if (iparm(2) .ge. 5) then
        if (ipme .eq. 0) then
          write (iparm(1),500) '(QR QL)^1/2 r norm  ', rnorm
        endif
        endif
        if (iparm(2) .ge. 5) then
        if (ipme .eq. 0) then
          write (iparm(1),500) '(QL*r,r)            ', rr
        endif
        endif
*
        if (iparm(12).eq.-3) go to 5536
                 go to 5537
 5536  continue
          if ((mod(iparm(15)/1,2).eq.1)) then
          endif
          if ((mod(iparm(15)/2,2).eq.1)) then
          iva = ir
          endif
          if ((mod(iparm(15)/4,2).eq.1)) then
          ivql = iqlr
          endif
          if ((mod(iparm(15)/8,2).eq.1)) then
          ivqr = iqrqlr
          endif
          ireq = 9
      iretlb = 2011
      go to 1500
 2011 continue
          if ((mod(iparm(15)/8,2).eq.1)) then
          endif
          if ((mod(iparm(15)/4,2).eq.1)) then
          endif
          if ((mod(iparm(15)/2,2).eq.1)) then
          endif
          if ((mod(iparm(15)/1,2).eq.1)) then
          endif
          if (ier .ge. 0) then
            call zitput ( in , is , iparm , fparm , iwk , fwk , ier )
          endif
          if (ijob .eq. -1) then
            if (iparm(2).ge.3 .and. iparm(27).eq.0) then
        if (ipme .eq. 0) then
          write (iparm(1),'(/a/)') ' User-requested termination of itera
     &tive method.'
        endif
            endif
            go to 900
          endif
                go to 5538
 5537 continue
 5538 continue
        if (iparm(12) .gt. 1) go to 5539
                 go to 5540
 5539  continue
          if (iparm(12) .eq. 1 ) then
            if (iparm(5) .eq. 0) then
              if (iparm(19) .eq. 0) then
                ier = -5
                call xersho ( ier, 'zcgr' , iparm , '19 and 12 incompati
     &ble' )
                go to 900
              endif
          call zwdot (bot, uexact, uexact, iparm , fparm , iwk , fwk , 
     &       ier )
              top = bot
            else
      call zfmal (iv1tst,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          call zwsub ( fwk(iv1tst), u, uexact, iparm , fparm , iwk , 
     &       fwk , ier )
          call zwdot (top, fwk(iv1tst), fwk(iv1tst), iparm , fparm , 
     &       iwk , fwk , ier )
      call zffre (iv1tst,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
            endif
          endif
          if (iparm(12) .eq. 2 ) then
            if (iparm(5) .eq. 0) then
          call zwdot (bot, b, b, iparm , fparm , iwk , fwk , ier )
              top = bot
            else
          call zwdot (top, fwk(ir), fwk(ir), iparm , fparm , iwk , fwk ,
     &        ier )
            endif
          endif
          if (iparm(12) .eq. 3) go to 5542
                 go to 5543
 5542  continue
            if (iparm(5) .eq. 0) go to 5545
                 go to 5546
 5545  continue
              if (((iparm(14).eq.0).or.(iparm(14).eq.-2))) go to 5548
                 go to 5549
 5548  continue
          call zwdot (bot, fwk(iqlr), fwk(iqlr), iparm , fparm , iwk , 
     &       fwk , ier )
                go to 5550
 5549 continue
      call zfmal (iv1tst,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5551
                 go to 5552
 5551  continue
          call zwset (fwk(iv1tst),b, iparm , fparm , iwk , fwk , ier )
                go to 5553
 5552 continue
      call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          iva = ivitmp
          call zwset (fwk(iva),b, iparm , fparm , iwk , fwk , ier )
          ivql = iv1tst
       ireq = 5
      iretlb = 2012
      go to 1500
 2012 continue
      call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
 5553 continue
          call zwdot (bot, fwk(iv1tst), fwk(iv1tst), iparm , fparm , 
     &       iwk , fwk , ier )
      call zffre (iv1tst,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
 5550 continue
              top = bot
                go to 5547
 5546 continue
          call zwdot (top, fwk(iqlr), fwk(iqlr), iparm , fparm , iwk , 
     &       fwk , ier )
 5547 continue
                go to 5544
 5543 continue
 5544 continue
          if (iparm(12) .eq. 4) go to 5554
                 go to 5555
 5554  continue
            if (iparm(5) .eq. 0) go to 5557
                 go to 5558
 5557  continue
              if (((iparm(14).eq.0).or.(iparm(14).eq.-2))) go to 5560
                 go to 5561
 5560  continue
          call zwdot (bot, fwk(iqrqlr), fwk(iqrqlr), iparm , fparm , 
     &       iwk , fwk , ier )
                go to 5562
 5561 continue
      call zfmal (iv1tst,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (iv2tst,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5563
                 go to 5564
 5563  continue
          call zwset (fwk(iv1tst),b, iparm , fparm , iwk , fwk , ier )
                go to 5565
 5564 continue
      call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          iva = ivitmp
          call zwset (fwk(iva),b, iparm , fparm , iwk , fwk , ier )
          ivql = iv1tst
       ireq = 5
      iretlb = 2013
      go to 1500
 2013 continue
      call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
 5565 continue
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5566
                 go to 5567
 5566  continue
          call zwset (fwk(iv2tst),fwk(iv1tst), iparm , fparm , iwk , 
     &       fwk , ier )
                go to 5568
 5567 continue
          ivql = iv1tst
          ivqr = iv2tst
       ireq = 7
      iretlb = 2014
      go to 1500
 2014 continue
       if (ijob .eq. -1) go to 900
 5568 continue
          call zwdot (bot, fwk(iv2tst), fwk(iv2tst), iparm , fparm , 
     &       iwk , fwk , ier )
      call zffre (iv2tst,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zffre (iv1tst,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
 5562 continue
              top = bot
                go to 5559
 5558 continue
          call zwdot (top, fwk(iqrqlr), fwk(iqrqlr), iparm , fparm , 
     &       iwk , fwk , ier )
 5559 continue
                go to 5556
 5555 continue
 5556 continue
          if (iparm(12) .eq. 1 ) then
            if (iparm(19) .eq. 0) then
                ier = -5
                call xersho ( ier, 'zcgr' , iparm , '19 and 12 incompati
     &ble' )
                go to 900
            endif
            if (((iparm(14).eq.0).or.(iparm(14).eq.-2))) then
          call zwdot (top, uexact, uexact, iparm , fparm , iwk , fwk , 
     &       ier )
            else
      call zfmal (iv1tst,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          call zwsub ( fwk(iv1tst), u, uexact, iparm , fparm , iwk , 
     &       fwk , ier )
          call zwdot (top, fwk(iv1tst), fwk(iv1tst), iparm , fparm , 
     &       iwk , fwk , ier )
      call zffre (iv1tst,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
            endif
            if (iparm(5) .eq. 0) bot = top
          endif
          if (iparm(12) .eq. 6) then
          call zwdot (top, fwk(ir), fwk(ir), iparm , fparm , iwk , fwk ,
     &        ier )
            if (iparm(5) .eq. 0) bot = top
          endif
          if (iparm(12) .eq. 7) then
          call zwdot (top, fwk(iqlr), fwk(iqlr), iparm , fparm , iwk , 
     &       fwk , ier )
            if (iparm(5) .eq. 0) bot = top
          endif
          if (iparm(12) .eq. 8) then
          call zwdot (top, fwk(iqrqlr), fwk(iqrqlr), iparm , fparm , 
     &       iwk , fwk , ier )
            if (iparm(5) .eq. 0) bot = top
          endif
          if (dble(top) .lt. dble((0d0,0.0d0))) then
                ier = -10
                call xersho ( ier, 'zcgr' , iparm , 'Attempt to take sqr
     &t of negative number' )
                go to 900
          endif
          if (dble(bot) .le. dble((0d0,0.0d0))) then
                ier = -10
                call xersho ( ier, 'zcgr' , iparm , 'Attempt to divide b
     &y zero' )
                go to 900
          endif
          fparm(5) = sqrt(dble(top)/dble(bot))
          call zitput ( in , is , iparm , fparm , iwk , fwk , ier )
          if (dble(fparm(5)) .le. dble(fparm(4))) go to 2500
                go to 5541
 5540 continue
 5541 continue
*
        if (iparm(12) .eq. -1) then
          if (r0norm .eq. (0d0,0.0d0)) then
                ier = -10
                call xersho ( ier, 'zcgr' , iparm , 'Attempt to divide b
     &y zero' )
                go to 900
          endif
          fparm(5) = rnorm/r0norm
          call zitput ( in , is , iparm , fparm , iwk , fwk , ier )
          if (dble(fparm(5)) .le. dble(fparm(4))) go to 2500
        endif
*
        if (iparm(12) .eq. -2) then
          call zwsub ( fwk(iv1), u, uexact, iparm , fparm , iwk , fwk , 
     &       ier )
          call zwrdot (uudot,fwk(iv1), fwk(iv1), iparm , fparm , iwk , 
     &       fwk , ier )
          call zwrdot (uexdt,uexact, uexact, iparm , fparm , iwk , fwk ,
     &        ier )
          if (dble(uudot) .lt. dble((0d0,0.0d0))) then
                ier = -10
                call xersho ( ier, 'zcgr' , iparm , 'Attempt to take sqr
     &t of negative number' )
                go to 900
          endif
          if (dble(uexdt) .le. dble((0d0,0.0d0))) then
                ier = -10
                call xersho ( ier, 'zcgr' , iparm , 'Attempt to divide b
     &y zero' )
                go to 900
          endif
          fparm(5) = sqrt(dble(uudot)/dble(uexdt))
          call zitput ( in , is , iparm , fparm , iwk , fwk , ier )
          if (dble(fparm(5)) .le. dble(fparm(4))) go to 2500
        endif
*
        if (in .ge. iparm(4)) then
                ier = 2
                call xersho ( ier, 'zcgr' , iparm , ' ' )
          go to 900
        endif
*
        call zckstg ( iclstg, iwkstg, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
*
*-----------------------proceed with iteration--------------------------
*
        if (in .eq. 0) then
          call zwset (fwk(ip),fwk(iqrqlr), iparm , fparm , iwk , fwk , 
     &       ier )
        else
          if (rrold .eq. (0d0,0.0d0)) then
                ier = -7
                call xersho ( ier, 'zcgr' , iparm , ' ' )
                go to 900
          endif
          alphan = rr / rrold
          call zwsax (fwk(ip),fwk(iqrqlr), fwk(ip), alphan, iparm , 
     &       fparm , iwk , fwk , ier )
          if (iparm(2) .ge. 5) then
        if (ipme .eq. 0) then
          write (iparm(1),500) 'alpha               ', alphan
        endif
          endif
        endif
*
          ivqr = ip
          iva = iap
       ireq = 3
      iretlb = 2015
      go to 1500
 2015 continue
       if (ijob .eq. -1) go to 900
*
          call zwcdot (pap, fwk(ip), fwk(iap), iparm(20), iparm , fparm 
     &       , iwk , fwk , ier )
        if (iparm(2) .ge. 5) then
        if (ipme .eq. 0) then
          write (iparm(1),500) '(p,ap)  ', pap
        endif
        endif
        if (pap .eq. (0d0,0.0d0)) then
                ier = -6
                call xersho ( ier, 'zcgr' , iparm , ' ' )
                go to 900
        endif
        vlamda = rr / pap
        if (iparm(2) .ge. 5) then
        if (ipme .eq. 0) then
          write (iparm(1),500) 'lambda              ', vlamda
        endif
        endif
*
          call zwsax (u,u, fwk(ip), vlamda , iparm , fparm , iwk , fwk ,
     &        ier )
          call zwsax (fwk(ir),fwk(ir), fwk(iap), (-vlamda), iparm , 
     &       fparm , iwk , fwk , ier )
        if (((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5569
                 go to 5570
 5569  continue
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5572
                 go to 5573
 5572  continue
          call zwset (fwk(iqlr),fwk(ir), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5574
 5573 continue
          iva = ir
          ivql = iqlr
       ireq = 5
      iretlb = 2016
      go to 1500
 2016 continue
       if (ijob .eq. -1) go to 900
 5574 continue
                go to 5571
 5570 continue
 5571 continue
        if (((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5575
                 go to 5576
 5575  continue
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5578
                 go to 5579
 5578  continue
          call zwset (fwk(iqrqlr),fwk(iqlr), iparm , fparm , iwk , fwk ,
     &        ier )
                go to 5580
 5579 continue
          ivql = iqlr
          ivqr = iqrqlr
       ireq = 7
      iretlb = 2017
      go to 1500
 2017 continue
       if (ijob .eq. -1) go to 900
 5580 continue
                go to 5577
 5576 continue
 5577 continue
*
          call zwrdot (dectr,fwk(ir), fwk(iqrqlr), iparm , fparm , iwk ,
     &        fwk , ier )
        rnorm = sqrt(abs(dectr))
        rrold = rr
        if (iparm(20) .eq. 1) then
          rr = dectr
        else
          call zwcdot (rr, fwk(ir), fwk(iqrqlr), iparm(20), iparm , 
     &       fparm , iwk , fwk , ier )
        endif
*
        in = in + 1
        iparm(5) = in
        is = is + 1
*
      go to 100
*
 2500 continue
      if (iparm(2) .ge. 3) then
        if (ipme .eq. 0) then
          write (iparm(1),'(/a,i7,a/)') '     Conjugate Gradient Method 
     &has converged in ', iparm(5), ' iterations.'
        endif
      endif
*
*-------------------------------terminate-------------------------------
*
  900 continue
*
      if (iclstg .ne. 1) then
        call zckstg (-1, iwkstg, iparm , fparm , iwk , fwk , ier )
      endif
*
      if (iparm(21).eq.1 .and. ier.ge.0) go to 5581
                 go to 5582
 5581  continue
      call zfmal (ib1,iparm(3), iparm , fparm , iwk , fwk , ier )
      call zfmal (ib2,iparm(3), iparm , fparm , iwk , fwk , ier )
      call zfmal (iu1,iparm(3), iparm , fparm , iwk , fwk , ier )
      call zfmal (iu2,iparm(3), iparm , fparm , iwk , fwk , ier )
        istab = istab2
        if (iparm(19) .eq. 1) then
          call zwdot (fwk(istab+iud), uexact, uexact, iparm , fparm , 
     &       iwk , fwk , ier )
        endif
          call zwdot (fwk(istab+ibd ), b, b, iparm , fparm , iwk , fwk ,
     &        ier )
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5584
                 go to 5585
 5584  continue
          call zwset (fwk(ib2),b, iparm , fparm , iwk , fwk , ier )
                go to 5586
 5585 continue
      call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
          iva = ivitmp
          call zwset (fwk(iva),b, iparm , fparm , iwk , fwk , ier )
          ivql = ib2
       ireq = 5
      iretlb = 2018
      go to 1500
 2018 continue
      call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
       if (ijob .eq. -1) go to 900
 5586 continue
          call zwdot (fwk(istab+ilbd), fwk(ib2), fwk(ib2), iparm , 
     &       fparm , iwk , fwk , ier )
          call zwset (fwk(iu1),fwk(ib2), iparm , fparm , iwk , fwk , 
     &       ier )
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5587
                 go to 5588
 5587  continue
          call zwset (fwk(iu2),fwk(iu1), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5589
 5588 continue
          ivql = iu1
          ivqr = iu2
       ireq = 7
      iretlb = 2019
      go to 1500
 2019 continue
       if (ijob .eq. -1) go to 900
 5589 continue
          call zwdot (fwk(istab+irlbd), fwk(iu2), fwk(iu2), iparm , 
     &       fparm , iwk , fwk , ier )
        if (iparm(19) .eq. 1) then
          call zwsub ( fwk(iu1), u, uexact, iparm , fparm , iwk , fwk , 
     &       ier )
          call zwdot (fwk(istab+ied), fwk(iu1), fwk(iu1), iparm , fparm 
     &       , iwk , fwk , ier )
        endif
      call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
          ivqr = ivitmp
          call zwset (fwk(ivqr),u, iparm , fparm , iwk , fwk , ier )
          iva = ib1
       ireq = 3
      iretlb = 2020
      go to 1500
 2020 continue
      call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
       if (ijob .eq. -1) go to 900
          call zwsub ( fwk(ib1), b, fwk(ib1), iparm , fparm , iwk , fwk 
     &       , ier )
          call zwdot (fwk(istab+ird), fwk(ib1), fwk(ib1), iparm , fparm 
     &       , iwk , fwk , ier )
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5590
                 go to 5591
 5590  continue
          call zwset (fwk(ib2),fwk(ib1), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5592
 5591 continue
          iva = ib1
          ivql = ib2
       ireq = 5
      iretlb = 2021
      go to 1500
 2021 continue
       if (ijob .eq. -1) go to 900
 5592 continue
          call zwdot (fwk(istab+ilrd ), fwk(ib2), fwk(ib2), iparm , 
     &       fparm , iwk , fwk , ier )
          call zwset (fwk(iu1),fwk(ib2), iparm , fparm , iwk , fwk , 
     &       ier )
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5593
                 go to 5594
 5593  continue
          call zwset (fwk(iu2),fwk(iu1), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5595
 5594 continue
          ivql = iu1
          ivqr = iu2
       ireq = 7
      iretlb = 2022
      go to 1500
 2022 continue
       if (ijob .eq. -1) go to 900
 5595 continue
          call zwdot (fwk(istab+irlrd), fwk(iu2), fwk(iu2), iparm , 
     &       fparm , iwk , fwk , ier )
        if (iparm(19) .eq. 1) then
          if (fwk(istab1+ied) .eq. (0d0,0.0d0)) then
                ier = 4
                call xersho ( ier, 'zcgr' , iparm , 'Value of RELERR may
     & be inaccurate' )
          else
            fparm(8) = sqrt(fwk(istab2+ied)/fwk(istab1+ied))
          endif
        endif
        if (fwk(istab1+ird) .eq. (0d0,0.0d0)) then
                ier = 4
                call xersho ( ier, 'zcgr' , iparm , 'Value of RELRSD may
     & be inaccurate' )
        else
          fparm(7) = sqrt(fwk(istab2+ird)/fwk(istab1+ird))
        endif
        if (iparm(2) .ge. 4) then
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(1x, 60(''*''))')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''Final iteration statistics'')')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (iparm(19) .eq. 1) then
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of uexact            '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+iud))
        endif
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of b                 '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+ibd))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Ql*b              '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+ilbd))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Qr*Ql*b           '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+irlbd))
        endif
        if (iparm(19) .eq. 1) then
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of error             '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+ied))
        endif
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of residual          '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+ird))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Ql*r              '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+ilrd))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Qr*Ql*r           '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+irlrd))
        endif
        if (iparm(19) .eq. 1) then
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''Relative 2-norm of error    '',1pg20.10
     &,2x,1pg20.10)') fparm(8)
        endif
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''Relative 2-norm of residual '',1pg20.10
     &,2x,1pg20.10)') fparm(7)
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(1x, 60(''*''))')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        endif
      call zffre (iu2,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      call zffre (iu1,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      call zffre (ib2,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      call zffre (ib1,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      call zffre (istab2,8, iparm , fparm , iwk , fwk , ier )
      call zffre (istab1,8, iparm , fparm , iwk , fwk , ier )
                go to 5583
 5582 continue
 5583 continue
*
      call zffre (iv1,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      if (.not.((iparm(13).eq.2).or.(iparm(13).eq.3))) then
      else
        if (.not.((iparm(13).eq.1).or.(iparm(13).eq.3))) then
        else
      call zffre (iqrqlr,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
        endif
      endif
      call zffre (ir ,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      call zffre (iap,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      call zffre (ip ,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
*
*
*
*
      call zffre (iftab,10, iparm , fparm , iwk , fwk , ier )
      call xifre (iitab,21, iparm , fparm , iwk , fwk , ier )
*
      if (itimer .eq. 1) then
        call xtimer (tc2,tr2,iparm(22),0)
        fparm(1) = fparm(1) + (tc2-tc1)
        fparm(2) = fparm(2) + (tr2-tr1)
      endif
*
  910 continue
      if (iparm(27) .eq. 0) then
        call zifppr ( -1 , iparm, fparm, ier )
      endif
*
      if (iparm(27) .eq. 0) iparm(27) = 0
      iparm(11) = (iparm(6))
      ireq = -1
      return
*
*-------------------------------brancher--------------------------------
*
*
 1000 continue
*
      ipme = mynode ( )
      iphost = myhost ( )
      log2np = nodedim ( )
      nproc = 2**log2np
      iitab = iparm(11)
      if (iitab.eq.(iparm(6)) .and. ijob.eq.-1) go to 910
*
      if (iparm(11) .eq. (iparm(6))) then
                ier = -4
                call xersho ( ier, 'zcgr' , iparm , 'Values of ijob and 
     &iparm(iptr) incompatible' )
      endif
*
      call xtimer (tc1,tr1,iparm(22),0)
*
        iitab = iwk(iitab+0)
        iftab = iwk(iitab+1)
        istab = iwk(iitab+2)
        iwkstg = iwk(iitab+3)
        iclstg = iwk(iitab+4)
        in = iwk(iitab+5)
        is = iwk(iitab+6)
        itimer = iwk(iitab+7)
        iretlb = iwk(iitab+8)
        ivitmp = iwk(iitab+9)
        ivotmp = iwk(iitab+10)
        iv1tst = iwk(iitab+11)
        iv2tst = iwk(iitab+12)
        istab1 = iwk(iitab+13)
        istab2 = iwk(iitab+14)
        ir = iwk(iitab+15)
        ip = iwk(iitab+16)
        iap = iwk(iitab+17)
        iqlr = iwk(iitab+18)
        iqrqlr = iwk(iitab+19)
        iv1 = iwk(iitab+20)
        top = fwk(iftab+0)
        bot = fwk(iftab+1)
        alphan = fwk(iftab+2)
        dectr = fwk(iftab+3)
        pap = fwk(iftab+4)
        rnorm = fwk(iftab+5)
        r0norm = fwk(iftab+6)
        rr = fwk(iftab+7)
        rrold = fwk(iftab+8)
        vlamda = fwk(iftab+9)
*
      if (ijob.ne.3 .and. ijob.ne.-1) then
                ier = -4
                call xersho ( ier, 'zcgr' , iparm , 'Argument ijob inval
     &id' )
      endif
*
      go to (2000,2001,2002,2003,2004,2005,2006,2007,2008,2009,2010,
     &   2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022), 
     &   iretlb - 2000 + 1
*
*---------------------------request of caller---------------------------
*
*
 1500 continue
      iparm(11) = iitab
        iwk(iitab+0) = iitab
        iwk(iitab+1) = iftab
        iwk(iitab+2) = istab
        iwk(iitab+3) = iwkstg
        iwk(iitab+4) = iclstg
        iwk(iitab+5) = in
        iwk(iitab+6) = is
        iwk(iitab+7) = itimer
        iwk(iitab+8) = iretlb
        iwk(iitab+9) = ivitmp
        iwk(iitab+10) = ivotmp
        iwk(iitab+11) = iv1tst
        iwk(iitab+12) = iv2tst
        iwk(iitab+13) = istab1
        iwk(iitab+14) = istab2
        iwk(iitab+15) = ir
        iwk(iitab+16) = ip
        iwk(iitab+17) = iap
        iwk(iitab+18) = iqlr
        iwk(iitab+19) = iqrqlr
        iwk(iitab+20) = iv1
        fwk(iftab+0) = top
        fwk(iftab+1) = bot
        fwk(iftab+2) = alphan
        fwk(iftab+3) = dectr
        fwk(iftab+4) = pap
        fwk(iftab+5) = rnorm
        fwk(iftab+6) = r0norm
        fwk(iftab+7) = rr
        fwk(iftab+8) = rrold
        fwk(iftab+9) = vlamda
      call xtimer (tc2,tr2,iparm(22),0)
      fparm(1) = fparm(1) + (tc2-tc1)
      fparm(2) = fparm(2) + (tr2-tr1)
      return
*
*
*--------------------------------end------------------------------------
*
  500 format ( 1x, a, 1pg20.10,2x,1pg20.10 )
*
      end
