************************************************************************
**  ^ROUTINE: WCGNER - Reverse communication level routine for
**                     conjugate gradient on the normal equations
**                     (minimal error version). (^)
**
      subroutine zcgner ( ijob , ireq , u , uexact , b , iva , ivql , 
     &   ivqr , iwk , fwk , iparm , fparm , ier )
**
**  ^DESCRIPTION:
**    Preconditioned CG / normal equations algorithm, error minimizing
**    version, a.k.a. Craig's method.  This algorithm applies a variant
**    of the conjugate gradient method to the normal equations:
**    (Q_L A Q_R) (Q_L A Q_R)^* (Q_L A Q_R)^{-*} Q_R^{-1} u
**     = Q_L b.
**    The variant given here minimizes the quantity
**    Q_R^{-1}(u^{(n)}-u).
**
**  ^AUTHOR:   wdj@beta.lanl.gov
**
**  ^MODIFIED: wdj@lanl.gov Tue Sep 28 08:27:54 MDT 1993
**
**  ^ARGUMENTS: see *Subroutine Arguments* below.
**
**  ^REQUIREMENTS:
**    Common Blocks: none
**    Subroutines:   see below.
**
**  ^SIDE_EFFECTS:
**    See the definition of METHRDCLS to see which parameters
**    are modified
**
**  ^ALGORITHM:
**    $$ p^{(n)} = Q_R^* A^* Q_L^* Q_L r^{(n)} + \alpha_n p^{(n-1)} $$
**    $$ \alpha_n  ={( Q_L r^{(n  )}, Q_L r^{(n  )}) \over
**                   ( Q_L r^{(n-1)}, Q_L r^{(n-1)})} $$
**    $$      u^{(n+1)} =     u^{(n)} + \lambda_n   Q_R p^{(n)}$$
**    $$  Q_L r^{(n+1)} = Q_L r^{(n)} - \lambda_n Q_L A Q_R p^{(n)}$$
**    $$ \lambda_n = {( Q_L r^{(n)}, Q_L r^{(n)}) \over
**                    (     p^{(n)},     p^{(n)})} $$
**
**  ^REFERENCES:
**   Howard **   Nonsymmetric Systems of Linear Equations,'' Research Report 229,
**   Yale University, Department of Computer Science, April 1982.
**
**   Christopher **   for Sparse Linear Equations and Sparse Least Squares,'' ACM
**   Transactions on Mathematical Software, vol. 8., no. 1, March 1982,
**   pp. 43--71.
**
**   Steven F. Ashby, Thomas A. Manteuffel and Paul E. Saylor,
**   ``A Taxonomy for Conjugate Gradient Methods,'' {\sl SIAM J. Numer.
**   Anal.}, 27 (1990), pp. 1542--1568.
**
**  ^DOCUMENTATION:
**
************************************************************************
*
      implicit none
          Include 'fcube.h'
*         Include 'veclib.h'
*
*#    METHRAL        - (^)
      integer ijob
      integer ireq
      integer ier
       integer iwk(*)
       integer iparm(*)
       double complex fwk(*)
       double complex fparm(*)
       double complex u(*)
       double complex uexact(*)
       double complex b(*)
      integer ivql
      integer iva
      integer ivqr
*
        integer itimer
        double precision tc1, tr1, tc2, tr2
      integer iitab
      integer iftab
      integer iwkstg
      integer iclstg
      integer in
      integer is
      integer iretlb
      integer ivitmp, ivotmp
      integer iv1tst, iv2tst
      integer istab1, istab2, istab
      integer ib1,ib2, iu1,iu2
      integer iud
      parameter (iud =0)
      integer ibd
      parameter (ibd =1)
      integer ilbd
      parameter (ilbd =2)
      integer irlbd
      parameter (irlbd=3)
      integer ied
      parameter (ied =4)
      integer ird
      parameter (ird =5)
      integer ilrd
      parameter (ilrd =6)
      integer irlrd
      parameter (irlrd=7)
      double complex urndf
      double complex top
      double complex bot
      parameter (urndf=.1)
      double complex mydot1, mydot2
      external zargck
      external zckstg
      external zwrnd
      external zifppr
      external zitput
      external xtimer
      integer ipme
      integer iphost
      integer log2np
      integer nproc
      integer iom
      character*72 errstr
      external ximini
      external ximal
      external xifre
      external zfmini
      external zfmal
      external zffre
*     integer imv1, imv2        ! temporaries - for matvec
      integer ir
      integer iqlr
      integer iqrqlr
      integer ip, irp, iarp, ilarp
      integer iv1, iv2, iv3
      double complex rdot
      double complex rnorm
      double complex r0norm
      double complex alpha
      double complex beta
      double complex vlamda
      double complex rr, rrold, pap
      double complex uudot, uexdt
*
*
****^^******************************************************************
*     $Modified: spencer@navier.ae.utexas.edu on Thu May 16 11:59:53 1996 $
*     $Id: cgner.fm4,v 1.7 1994/11/22 05:20:10 joubert Exp $
*     $Revision: 1.7 $
*     $Log: cgner.fm4,v $
*     Revision 1.7  1994/11/22  05:20:10  joubert
*     misc changes for release 1.0
*
*     Revision 1.6  1994/10/06  16:24:06  joubert
*     fixed Intel flop count problem; ifparm param name changes
*
*     Revision 1.5  1994/09/28  19:24:35  joubert
*     mods to stats and ntest options
*
*     Revision 1.4  1994/07/06  05:52:13  joubert
*     mods to it meths
*
*     Revision 1.3  1994/07/05  22:20:33  joubert
*     lanczos/orthores acceleration; some cleanup of iterative methods
*
*     Revision 1.2  1994/06/24  18:20:30  joubert
*     t3d version mods; installed iom it meth
*
*     Revision 1.1  1994/06/14  16:31:20  joubert
*     added normal equations methods
*
************************************************************************
*
      iitab = (iparm(6))
      iftab = (iparm(6))
      ir = (iparm(6))
      iqlr = (iparm(6))
      iqrqlr = (iparm(6))
      ip = (iparm(6))
      irp = (iparm(6))
      iarp = (iparm(6))
      ilarp = (iparm(6))
      iv1 = (iparm(6))
      iv2 = (iparm(6))
      iv3 = (iparm(6))
*
*
      if (ijob .ne. 1) go to 1000
*
      iclstg = 1
      itimer = 0
      in = 0
      iparm(5) = in
      is = 0
*
      iu1 = (iparm(6))
      iu2 = (iparm(6))
      ib1 = (iparm(6))
      ib2 = (iparm(6))
      istab1 = (iparm(6))
      istab2 = (iparm(6))
*
      ipme = mynode ( )
      iphost = myhost ( )
      log2np = nodedim ( )
      nproc = 2**log2np
      if (iparm(27) .eq. 0) iparm(27) = 0
      if (iparm(27) .eq. 0) then
        ier = 0
        if (ijob.eq.1 .or. ijob.eq.2 .or. ijob.eq.0) then
      call ximini ( iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmini ( iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        endif
        call zargck ( iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        call xtimer (tc1,tr1,iparm(22),1)
      endif
*
      call xtimer (tc1,tr1,iparm(22),0)
      itimer = 1
*
      if (iparm(27) .eq. 0) then
        call zifppr ( 1 , iparm, fparm, ier)
           if (ier .lt. 0) go to 900
      endif
*
      call ximal (iitab,25, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (iftab,10, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
*
*
      if (iparm(2) .ge. 3) then
        if (ipme .eq. 0) then
          write (iparm(1),'(/''     CG on the Normal Equations: Craig'''
     &'s method.''/)')
        endif
      endif
*
      call zfmal (iqlr,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (ip,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      if (((iparm(13).eq.1).or.(iparm(13).eq.3))) then
        if (((mod(iparm(15)/2,2).eq.1) .or. iparm(12).eq.2 .or. iparm(
     &     12).eq.6)) then
      call zfmal (ir,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        endif
      else
        ir = iqlr
      endif
      if (((iparm(13).eq.2).or.(iparm(13).eq.3))) then
        if (((mod(iparm(15)/8,2).eq.1) .or. iparm(12).eq.4 .or. iparm(
     &     12).eq.8)) then
      call zfmal (iqrqlr,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        endif
      else
        iqrqlr = iqlr
      endif
      call zfmal (iv1 ,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (iv2 ,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (iv3 ,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      iarp = iv2
      if (((iparm(13).eq.1).or.(iparm(13).eq.3))) then
        ilarp = iv3
      else
        ilarp = iv2
      endif
      if (((iparm(13).eq.2).or.(iparm(13).eq.3))) then
        irp = iv1
      else
        irp = ip
      endif
*
      if (iparm(14) .eq. -2 .or. iparm(14) .eq. -1) then
          call zwfil (u,(0d0,0.0d0), iparm , fparm , iwk , fwk , ier )
      endif
*
      if (iparm(14) .eq. 2 ) go to 5500
                 go to 5501
 5500  continue
          call zwrdot (mydot1,b, b, iparm , fparm , iwk , fwk , ier )
        call zwrnd (u, iparm , fparm , iwk , fwk , ier )
      call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call zwset (fwk(ivqr),u, iparm , fparm , iwk , fwk , ier )
          iva = iv1
       ireq = 3
      iretlb = 2000
      go to 1500
 2000 continue
      call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
          call zwrdot (mydot2,fwk(iv1), fwk(iv1), iparm , fparm , iwk , 
     &       fwk , ier )
          call zwscl ( u, u, urndf*sqrt(mydot1/mydot2), iparm , fparm , 
     &       iwk , fwk , ier )
                go to 5502
 5501 continue
 5502 continue
*
      if (iparm(14) .eq. 3) go to 5503
                 go to 5504
 5503  continue
      call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call zwset (fwk(ivqr),u, iparm , fparm , iwk , fwk , ier )
          iva = iv1
       ireq = 3
      iretlb = 2001
      go to 1500
 2001 continue
      call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
          call zwsub ( fwk(iv1), b, fwk(iv1), iparm , fparm , iwk , fwk 
     &       , ier )
          call zwrdot (mydot1,fwk(iv1), fwk(iv1), iparm , fparm , iwk , 
     &       fwk , ier )
          call zwset (fwk(iv1),u, iparm , fparm , iwk , fwk , ier )
        call zwrnd (u, iparm , fparm , iwk , fwk , ier )
      call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call zwset (fwk(ivqr),u, iparm , fparm , iwk , fwk , ier )
          iva = iv2
       ireq = 3
      iretlb = 2002
      go to 1500
 2002 continue
      call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
          call zwrdot (mydot2,fwk(iv2), fwk(iv2), iparm , fparm , iwk , 
     &       fwk , ier )
          call zwsax (u,fwk(iv1), u, -urndf*sqrt(mydot1/mydot2), iparm ,
     &        fparm , iwk , fwk , ier )
                go to 5505
 5504 continue
 5505 continue
*
      if (iparm(21) .eq. 1) go to 5506
                 go to 5507
 5506  continue
      call zfmal (istab1,8, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (istab2,8, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (ib1,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (ib2,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (iu1,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (iu2,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        istab = istab1
        if (iparm(19) .eq. 1) then
          call zwdot (fwk(istab+iud), uexact, uexact, iparm , fparm , 
     &       iwk , fwk , ier )
        endif
          call zwdot (fwk(istab+ibd ), b, b, iparm , fparm , iwk , fwk ,
     &        ier )
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5509
                 go to 5510
 5509  continue
          call zwset (fwk(ib2),b, iparm , fparm , iwk , fwk , ier )
                go to 5511
 5510 continue
      call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          iva = ivitmp
          call zwset (fwk(iva),b, iparm , fparm , iwk , fwk , ier )
          ivql = ib2
       ireq = 5
      iretlb = 2003
      go to 1500
 2003 continue
      call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
 5511 continue
          call zwdot (fwk(istab+ilbd), fwk(ib2), fwk(ib2), iparm , 
     &       fparm , iwk , fwk , ier )
          call zwset (fwk(iu1),fwk(ib2), iparm , fparm , iwk , fwk , 
     &       ier )
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5512
                 go to 5513
 5512  continue
          call zwset (fwk(iu2),fwk(iu1), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5514
 5513 continue
          ivql = iu1
          ivqr = iu2
       ireq = 7
      iretlb = 2004
      go to 1500
 2004 continue
       if (ijob .eq. -1) go to 900
 5514 continue
          call zwdot (fwk(istab+irlbd), fwk(iu2), fwk(iu2), iparm , 
     &       fparm , iwk , fwk , ier )
        if (iparm(19) .eq. 1) then
          call zwsub ( fwk(iu1), u, uexact, iparm , fparm , iwk , fwk , 
     &       ier )
          call zwdot (fwk(istab+ied), fwk(iu1), fwk(iu1), iparm , fparm 
     &       , iwk , fwk , ier )
        endif
      call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call zwset (fwk(ivqr),u, iparm , fparm , iwk , fwk , ier )
          iva = ib1
       ireq = 3
      iretlb = 2005
      go to 1500
 2005 continue
      call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
          call zwsub ( fwk(ib1), b, fwk(ib1), iparm , fparm , iwk , fwk 
     &       , ier )
          call zwdot (fwk(istab+ird), fwk(ib1), fwk(ib1), iparm , fparm 
     &       , iwk , fwk , ier )
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5515
                 go to 5516
 5515  continue
          call zwset (fwk(ib2),fwk(ib1), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5517
 5516 continue
          iva = ib1
          ivql = ib2
       ireq = 5
      iretlb = 2006
      go to 1500
 2006 continue
       if (ijob .eq. -1) go to 900
 5517 continue
          call zwdot (fwk(istab+ilrd ), fwk(ib2), fwk(ib2), iparm , 
     &       fparm , iwk , fwk , ier )
          call zwset (fwk(iu1),fwk(ib2), iparm , fparm , iwk , fwk , 
     &       ier )
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5518
                 go to 5519
 5518  continue
          call zwset (fwk(iu2),fwk(iu1), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5520
 5519 continue
          ivql = iu1
          ivqr = iu2
       ireq = 7
      iretlb = 2007
      go to 1500
 2007 continue
       if (ijob .eq. -1) go to 900
 5520 continue
          call zwdot (fwk(istab+irlrd), fwk(iu2), fwk(iu2), iparm , 
     &       fparm , iwk , fwk , ier )
        if (iparm(2) .ge. 4) then
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(1x, 60(''*''))')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''Initial iteration statistics'')')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (iparm(19) .eq. 1) then
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of uexact            '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+iud))
        endif
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of b                 '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+ibd))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Ql*b              '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+ilbd))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Qr*Ql*b           '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+irlbd))
        endif
        if (iparm(19) .eq. 1) then
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of error             '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+ied))
        endif
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of residual          '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+ird))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Ql*r              '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+ilrd))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Qr*Ql*r           '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+irlrd))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(1x, 60(''*''))')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        endif
      call zffre (iu2,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zffre (iu1,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zffre (ib2,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zffre (ib1,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
                go to 5508
 5507 continue
 5508 continue
*
      if (((iparm(14).eq.0).or.(iparm(14).eq.-2)) .or. iparm(14).eq.-1) 
     &   go to 5521
                 go to 5522
 5521  continue
        if (((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5524
                 go to 5525
 5524  continue
          if (((mod(iparm(15)/2,2).eq.1) .or. iparm(12).eq.2 .or. iparm(
     &       12).eq.6)) go to 5527
                 go to 5528
 5527  continue
          call zwset (fwk(ir),b, iparm , fparm , iwk , fwk , ier )
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5530
                 go to 5531
 5530  continue
          call zwset (fwk(iqlr),fwk(ir), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5532
 5531 continue
          iva = ir
          ivql = iqlr
       ireq = 5
      iretlb = 2008
      go to 1500
 2008 continue
       if (ijob .eq. -1) go to 900
 5532 continue
                go to 5529
 5528 continue
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5533
                 go to 5534
 5533  continue
          call zwset (fwk(iqlr),b, iparm , fparm , iwk , fwk , ier )
                go to 5535
 5534 continue
      call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          iva = ivitmp
          call zwset (fwk(iva),b, iparm , fparm , iwk , fwk , ier )
          ivql = iqlr
       ireq = 5
      iretlb = 2009
      go to 1500
 2009 continue
      call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
 5535 continue
 5529 continue
                go to 5526
 5525 continue
          call zwset (fwk(iqlr),b, iparm , fparm , iwk , fwk , ier )
 5526 continue
                go to 5523
 5522 continue
        if (((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5536
                 go to 5537
 5536  continue
          if (((mod(iparm(15)/2,2).eq.1) .or. iparm(12).eq.2 .or. iparm(
     &       12).eq.6)) go to 5539
                 go to 5540
 5539  continue
      call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call zwset (fwk(ivqr),u, iparm , fparm , iwk , fwk , ier )
          iva = ir
       ireq = 3
      iretlb = 2010
      go to 1500
 2010 continue
      call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
          call zwsub ( fwk(ir), b, fwk(ir), iparm , fparm , iwk , fwk , 
     &       ier )
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5542
                 go to 5543
 5542  continue
          call zwset (fwk(iqlr),fwk(ir), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5544
 5543 continue
          iva = ir
          ivql = iqlr
       ireq = 5
      iretlb = 2011
      go to 1500
 2011 continue
       if (ijob .eq. -1) go to 900
 5544 continue
                go to 5541
 5540 continue
      call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call zwset (fwk(ivqr),u, iparm , fparm , iwk , fwk , ier )
          iva = iv1
       ireq = 3
      iretlb = 2012
      go to 1500
 2012 continue
      call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
          call zwsub ( fwk(iv1), b, fwk(iv1), iparm , fparm , iwk , fwk 
     &       , ier )
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5545
                 go to 5546
 5545  continue
          call zwset (fwk(iqlr),fwk(iv1), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5547
 5546 continue
          iva = iv1
          ivql = iqlr
       ireq = 5
      iretlb = 2013
      go to 1500
 2013 continue
       if (ijob .eq. -1) go to 900
 5547 continue
 5541 continue
                go to 5538
 5537 continue
      call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call zwset (fwk(ivqr),u, iparm , fparm , iwk , fwk , ier )
          iva = iqlr
       ireq = 3
      iretlb = 2014
      go to 1500
 2014 continue
      call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
          call zwsub ( fwk(iqlr), b, fwk(iqlr), iparm , fparm , iwk , 
     &       fwk , ier )
 5538 continue
 5523 continue
      if (((iparm(13).eq.2).or.(iparm(13).eq.3)) .and. ((mod(iparm(15)/
     &   8,2).eq.1) .or. iparm(12).eq.4 .or. iparm(12).eq.8)) go to 5548
                 go to 5549
 5548  continue
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5551
                 go to 5552
 5551  continue
          call zwset (fwk(iqrqlr),fwk(iqlr), iparm , fparm , iwk , fwk ,
     &        ier )
                go to 5553
 5552 continue
          ivql = iqlr
          ivqr = iqrqlr
       ireq = 7
      iretlb = 2015
      go to 1500
 2015 continue
       if (ijob .eq. -1) go to 900
 5553 continue
                go to 5550
 5549 continue
 5550 continue
*
          call zwrdot (rdot,fwk(iqlr), fwk(iqlr), iparm , fparm , iwk , 
     &       fwk , ier )
      rnorm = sqrt(abs(rdot))
      r0norm = rnorm
*
*---------------------------iteration loop------------------------------
*
  100 continue
*
        if (iparm(2) .ge. 5) then
        if (ipme .eq. 0) then
          write (iparm(1),500) 'QL r norm           ', rnorm
        endif
        endif
*
        if (iparm(12).eq.-3) go to 5554
                 go to 5555
 5554  continue
          if ((mod(iparm(15)/1,2).eq.1)) then
          endif
          if ((mod(iparm(15)/2,2).eq.1)) then
          iva = ir
          endif
          if ((mod(iparm(15)/4,2).eq.1)) then
          ivql = iqlr
          endif
          if ((mod(iparm(15)/8,2).eq.1)) then
          ivqr = iqrqlr
          endif
          ireq = 9
      iretlb = 2016
      go to 1500
 2016 continue
          if ((mod(iparm(15)/8,2).eq.1)) then
          endif
          if ((mod(iparm(15)/4,2).eq.1)) then
          endif
          if ((mod(iparm(15)/2,2).eq.1)) then
          endif
          if ((mod(iparm(15)/1,2).eq.1)) then
          endif
          if (ier .ge. 0) then
            call zitput ( in , is , iparm , fparm , iwk , fwk , ier )
          endif
          if (ijob .eq. -1) then
            if (iparm(2).ge.3 .and. iparm(27).eq.0) then
        if (ipme .eq. 0) then
          write (iparm(1),'(/a/)') ' User-requested termination of itera
     &tive method.'
        endif
            endif
            go to 900
          endif
                go to 5556
 5555 continue
 5556 continue
        if (iparm(12) .gt. 1) go to 5557
                 go to 5558
 5557  continue
          if (iparm(12) .eq. 1 ) then
            if (iparm(5) .eq. 0) then
              if (iparm(19) .eq. 0) then
                ier = -5
                call xersho ( ier, 'zcgner' , iparm , '19 and 12 incompa
     &tible' )
                go to 900
              endif
          call zwdot (bot, uexact, uexact, iparm , fparm , iwk , fwk , 
     &       ier )
              top = bot
            else
      call zfmal (iv1tst,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          call zwsub ( fwk(iv1tst), u, uexact, iparm , fparm , iwk , 
     &       fwk , ier )
          call zwdot (top, fwk(iv1tst), fwk(iv1tst), iparm , fparm , 
     &       iwk , fwk , ier )
      call zffre (iv1tst,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
            endif
          endif
          if (iparm(12) .eq. 2 ) then
            if (iparm(5) .eq. 0) then
          call zwdot (bot, b, b, iparm , fparm , iwk , fwk , ier )
              top = bot
            else
          call zwdot (top, fwk(ir), fwk(ir), iparm , fparm , iwk , fwk ,
     &        ier )
            endif
          endif
          if (iparm(12) .eq. 3) go to 5560
                 go to 5561
 5560  continue
            if (iparm(5) .eq. 0) go to 5563
                 go to 5564
 5563  continue
              if (((iparm(14).eq.0).or.(iparm(14).eq.-2))) go to 5566
                 go to 5567
 5566  continue
          call zwdot (bot, fwk(iqlr), fwk(iqlr), iparm , fparm , iwk , 
     &       fwk , ier )
                go to 5568
 5567 continue
      call zfmal (iv1tst,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5569
                 go to 5570
 5569  continue
          call zwset (fwk(iv1tst),b, iparm , fparm , iwk , fwk , ier )
                go to 5571
 5570 continue
      call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          iva = ivitmp
          call zwset (fwk(iva),b, iparm , fparm , iwk , fwk , ier )
          ivql = iv1tst
       ireq = 5
      iretlb = 2017
      go to 1500
 2017 continue
      call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
 5571 continue
          call zwdot (bot, fwk(iv1tst), fwk(iv1tst), iparm , fparm , 
     &       iwk , fwk , ier )
      call zffre (iv1tst,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
 5568 continue
              top = bot
                go to 5565
 5564 continue
          call zwdot (top, fwk(iqlr), fwk(iqlr), iparm , fparm , iwk , 
     &       fwk , ier )
 5565 continue
                go to 5562
 5561 continue
 5562 continue
          if (iparm(12) .eq. 4) go to 5572
                 go to 5573
 5572  continue
            if (iparm(5) .eq. 0) go to 5575
                 go to 5576
 5575  continue
              if (((iparm(14).eq.0).or.(iparm(14).eq.-2))) go to 5578
                 go to 5579
 5578  continue
          call zwdot (bot, fwk(iqrqlr), fwk(iqrqlr), iparm , fparm , 
     &       iwk , fwk , ier )
                go to 5580
 5579 continue
      call zfmal (iv1tst,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zfmal (iv2tst,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5581
                 go to 5582
 5581  continue
          call zwset (fwk(iv1tst),b, iparm , fparm , iwk , fwk , ier )
                go to 5583
 5582 continue
      call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          iva = ivitmp
          call zwset (fwk(iva),b, iparm , fparm , iwk , fwk , ier )
          ivql = iv1tst
       ireq = 5
      iretlb = 2018
      go to 1500
 2018 continue
      call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
 5583 continue
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5584
                 go to 5585
 5584  continue
          call zwset (fwk(iv2tst),fwk(iv1tst), iparm , fparm , iwk , 
     &       fwk , ier )
                go to 5586
 5585 continue
          ivql = iv1tst
          ivqr = iv2tst
       ireq = 7
      iretlb = 2019
      go to 1500
 2019 continue
       if (ijob .eq. -1) go to 900
 5586 continue
          call zwdot (bot, fwk(iv2tst), fwk(iv2tst), iparm , fparm , 
     &       iwk , fwk , ier )
      call zffre (iv2tst,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call zffre (iv1tst,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
 5580 continue
              top = bot
                go to 5577
 5576 continue
          call zwdot (top, fwk(iqrqlr), fwk(iqrqlr), iparm , fparm , 
     &       iwk , fwk , ier )
 5577 continue
                go to 5574
 5573 continue
 5574 continue
          if (iparm(12) .eq. 1 ) then
            if (iparm(19) .eq. 0) then
                ier = -5
                call xersho ( ier, 'zcgner' , iparm , '19 and 12 incompa
     &tible' )
                go to 900
            endif
            if (((iparm(14).eq.0).or.(iparm(14).eq.-2))) then
          call zwdot (top, uexact, uexact, iparm , fparm , iwk , fwk , 
     &       ier )
            else
      call zfmal (iv1tst,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          call zwsub ( fwk(iv1tst), u, uexact, iparm , fparm , iwk , 
     &       fwk , ier )
          call zwdot (top, fwk(iv1tst), fwk(iv1tst), iparm , fparm , 
     &       iwk , fwk , ier )
      call zffre (iv1tst,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
            endif
            if (iparm(5) .eq. 0) bot = top
          endif
          if (iparm(12) .eq. 6) then
          call zwdot (top, fwk(ir), fwk(ir), iparm , fparm , iwk , fwk ,
     &        ier )
            if (iparm(5) .eq. 0) bot = top
          endif
          if (iparm(12) .eq. 7) then
          call zwdot (top, fwk(iqlr), fwk(iqlr), iparm , fparm , iwk , 
     &       fwk , ier )
            if (iparm(5) .eq. 0) bot = top
          endif
          if (iparm(12) .eq. 8) then
          call zwdot (top, fwk(iqrqlr), fwk(iqrqlr), iparm , fparm , 
     &       iwk , fwk , ier )
            if (iparm(5) .eq. 0) bot = top
          endif
          if (dble(top) .lt. dble((0d0,0.0d0))) then
                ier = -10
                call xersho ( ier, 'zcgner' , iparm , 'Attempt to take s
     &qrt of negative number' )
                go to 900
          endif
          if (dble(bot) .le. dble((0d0,0.0d0))) then
                ier = -10
                call xersho ( ier, 'zcgner' , iparm , 'Attempt to divide
     & by zero' )
                go to 900
          endif
          fparm(5) = sqrt(dble(top)/dble(bot))
          call zitput ( in , is , iparm , fparm , iwk , fwk , ier )
          if (dble(fparm(5)) .le. dble(fparm(4))) go to 2500
                go to 5559
 5558 continue
 5559 continue
*
        if (iparm(12) .eq. -1) then
          if (r0norm .eq. (0d0,0.0d0)) then
                ier = -10
                call xersho ( ier, 'zcgner' , iparm , 'Attempt to divide
     & by zero' )
                go to 900
          endif
          fparm(5) = rnorm/r0norm
          call zitput ( in , is , iparm , fparm , iwk , fwk , ier )
          if (dble(fparm(5)) .le. dble(fparm(4))) go to 2500
        endif
*
        if (iparm(12) .eq. -2) then
          call zwsub ( fwk(iv1), u, uexact, iparm , fparm , iwk , fwk , 
     &       ier )
          call zwrdot (uudot,fwk(iv1), fwk(iv1), iparm , fparm , iwk , 
     &       fwk , ier )
          call zwrdot (uexdt,uexact, uexact, iparm , fparm , iwk , fwk ,
     &        ier )
          if (dble(uudot) .lt. dble((0d0,0.0d0))) then
                ier = -10
                call xersho ( ier, 'zcgner' , iparm , 'Attempt to take s
     &qrt of negative number' )
                go to 900
          endif
          if (dble(uexdt) .le. dble((0d0,0.0d0))) then
                ier = -10
                call xersho ( ier, 'zcgner' , iparm , 'Attempt to divide
     & by zero' )
                go to 900
          endif
          fparm(5) = sqrt(dble(uudot)/dble(uexdt))
          call zitput ( in , is , iparm , fparm , iwk , fwk , ier )
          if (dble(fparm(5)) .le. dble(fparm(4))) go to 2500
        endif
*
        if (in .ge. iparm(4)) then
                ier = 2
                call xersho ( ier, 'zcgner' , iparm , ' ' )
          go to 900
        endif
*
        call zckstg ( iclstg, iwkstg, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
*
*-----------------------proceed with iteration--------------------------
*
        rrold = rr
        rr = rdot
        if (iparm(2) .ge. 5) then
        if (ipme .eq. 0) then
          write (iparm(1),500) '(r,r)               ', rdot
        endif
        endif
**#     VRDOT(rr,qlr,,qlr,)
        if (in .eq. 0) go to 5587
                 go to 5588
 5587  continue
          if (.not.((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5590
                 go to 5591
 5590  continue
            if (.not.((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5593
                 go to 5594
 5593  continue
          iva = iqlr
          ivqr = ip
       ireq = 4
      iretlb = 2020
      go to 1500
 2020 continue
       if (ijob .eq. -1) go to 900
                go to 5595
 5594 continue
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5596
                 go to 5597
 5596  continue
          call zwset (fwk(iv1),fwk(iqlr), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5598
 5597 continue
          ivql = iqlr
          iva = iv1
       ireq = 6
      iretlb = 2021
      go to 1500
 2021 continue
       if (ijob .eq. -1) go to 900
 5598 continue
          iva = iv1
          ivqr = ip
       ireq = 4
      iretlb = 2022
      go to 1500
 2022 continue
       if (ijob .eq. -1) go to 900
 5595 continue
                go to 5592
 5591 continue
            if (.not.((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5599
                 go to 5600
 5599  continue
          iva = iqlr
          ivqr = iv1
       ireq = 4
      iretlb = 2023
      go to 1500
 2023 continue
       if (ijob .eq. -1) go to 900
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5602
                 go to 5603
 5602  continue
          call zwset (fwk(ip),fwk(iv1), iparm , fparm , iwk , fwk , ier 
     &       )
                go to 5604
 5603 continue
          ivqr = iv1
          ivql = ip
       ireq = 8
      iretlb = 2024
      go to 1500
 2024 continue
       if (ijob .eq. -1) go to 900
 5604 continue
                go to 5601
 5600 continue
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5605
                 go to 5606
 5605  continue
          call zwset (fwk(iv2),fwk(iqlr), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5607
 5606 continue
          ivql = iqlr
          iva = iv2
       ireq = 6
      iretlb = 2025
      go to 1500
 2025 continue
       if (ijob .eq. -1) go to 900
 5607 continue
          iva = iv2
          ivqr = iv1
       ireq = 4
      iretlb = 2026
      go to 1500
 2026 continue
       if (ijob .eq. -1) go to 900
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5608
                 go to 5609
 5608  continue
          call zwset (fwk(ip),fwk(iv1), iparm , fparm , iwk , fwk , ier 
     &       )
                go to 5610
 5609 continue
          ivqr = iv1
          ivql = ip
       ireq = 8
      iretlb = 2027
      go to 1500
 2027 continue
       if (ijob .eq. -1) go to 900
 5610 continue
 5601 continue
 5592 continue
                go to 5589
 5588 continue
          if (.not.((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5611
                 go to 5612
 5611  continue
            if (.not.((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5614
                 go to 5615
 5614  continue
          iva = iqlr
          ivqr = iv2
       ireq = 4
      iretlb = 2028
      go to 1500
 2028 continue
       if (ijob .eq. -1) go to 900
                go to 5616
 5615 continue
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5617
                 go to 5618
 5617  continue
          call zwset (fwk(iv1),fwk(iqlr), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5619
 5618 continue
          ivql = iqlr
          iva = iv1
       ireq = 6
      iretlb = 2029
      go to 1500
 2029 continue
       if (ijob .eq. -1) go to 900
 5619 continue
          iva = iv1
          ivqr = iv2
       ireq = 4
      iretlb = 2030
      go to 1500
 2030 continue
       if (ijob .eq. -1) go to 900
 5616 continue
                go to 5613
 5612 continue
            if (.not.((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5620
                 go to 5621
 5620  continue
          iva = iqlr
          ivqr = iv1
       ireq = 4
      iretlb = 2031
      go to 1500
 2031 continue
       if (ijob .eq. -1) go to 900
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5623
                 go to 5624
 5623  continue
          call zwset (fwk(iv2),fwk(iv1), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5625
 5624 continue
          ivqr = iv1
          ivql = iv2
       ireq = 8
      iretlb = 2032
      go to 1500
 2032 continue
       if (ijob .eq. -1) go to 900
 5625 continue
                go to 5622
 5621 continue
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5626
                 go to 5627
 5626  continue
          call zwset (fwk(iv2),fwk(iqlr), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5628
 5627 continue
          ivql = iqlr
          iva = iv2
       ireq = 6
      iretlb = 2033
      go to 1500
 2033 continue
       if (ijob .eq. -1) go to 900
 5628 continue
          iva = iv2
          ivqr = iv1
       ireq = 4
      iretlb = 2034
      go to 1500
 2034 continue
       if (ijob .eq. -1) go to 900
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5629
                 go to 5630
 5629  continue
          call zwset (fwk(iv2),fwk(iv1), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5631
 5630 continue
          ivqr = iv1
          ivql = iv2
       ireq = 8
      iretlb = 2035
      go to 1500
 2035 continue
       if (ijob .eq. -1) go to 900
 5631 continue
 5622 continue
 5613 continue
          if (rr .eq. (0d0,0.0d0)) then
                ier = -7
                call xersho ( ier, 'zcgner' , iparm , ' ' )
                go to 900
          endif
          beta = rr / rrold
          if (iparm(2) .ge. 5) then
        if (ipme .eq. 0) then
          write (iparm(1),500) 'beta                ', beta
        endif
          endif
          call zwsax (fwk(ip),fwk(iv2), fwk(ip), beta, iparm , fparm , 
     &       iwk , fwk , ier )
 5589 continue
*
        if (.not.((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5632
                 go to 5633
 5632  continue
          if (.not.((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5635
                 go to 5636
 5635  continue
          ivqr = ip
          iva = iarp
       ireq = 3
      iretlb = 2036
      go to 1500
 2036 continue
       if (ijob .eq. -1) go to 900
                go to 5637
 5636 continue
          ivqr = ip
          iva = iarp
       ireq = 3
      iretlb = 2037
      go to 1500
 2037 continue
       if (ijob .eq. -1) go to 900
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5638
                 go to 5639
 5638  continue
          call zwset (fwk(ilarp),fwk(iarp), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5640
 5639 continue
          iva = iarp
          ivql = ilarp
       ireq = 5
      iretlb = 2038
      go to 1500
 2038 continue
       if (ijob .eq. -1) go to 900
 5640 continue
 5637 continue
                go to 5634
 5633 continue
          if (.not.((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5641
                 go to 5642
 5641  continue
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5644
                 go to 5645
 5644  continue
          call zwset (fwk(irp),fwk(ip), iparm , fparm , iwk , fwk , ier 
     &       )
                go to 5646
 5645 continue
          ivql = ip
          ivqr = irp
       ireq = 7
      iretlb = 2039
      go to 1500
 2039 continue
       if (ijob .eq. -1) go to 900
 5646 continue
          ivqr = irp
          iva = iarp
       ireq = 3
      iretlb = 2040
      go to 1500
 2040 continue
       if (ijob .eq. -1) go to 900
                go to 5643
 5642 continue
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5647
                 go to 5648
 5647  continue
          call zwset (fwk(irp),fwk(ip), iparm , fparm , iwk , fwk , ier 
     &       )
                go to 5649
 5648 continue
          ivql = ip
          ivqr = irp
       ireq = 7
      iretlb = 2041
      go to 1500
 2041 continue
       if (ijob .eq. -1) go to 900
 5649 continue
          ivqr = irp
          iva = iarp
       ireq = 3
      iretlb = 2042
      go to 1500
 2042 continue
       if (ijob .eq. -1) go to 900
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5650
                 go to 5651
 5650  continue
          call zwset (fwk(ilarp),fwk(iarp), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5652
 5651 continue
          iva = iarp
          ivql = ilarp
       ireq = 5
      iretlb = 2043
      go to 1500
 2043 continue
       if (ijob .eq. -1) go to 900
 5652 continue
 5643 continue
 5634 continue
*
          call zwrdot (pap,fwk(ip), fwk(ip), iparm , fparm , iwk , fwk ,
     &        ier )
        if (iparm(2) .ge. 5) then
        if (ipme .eq. 0) then
          write (iparm(1),500) '(p,p)               ', pap
        endif
        endif
        if (pap .eq. (0d0,0.0d0)) then
                ier = -6
                call xersho ( ier, 'zcgner' , iparm , ' ' )
                go to 900
        endif
        vlamda = rr / pap
        if (iparm(2) .ge. 5) then
        if (ipme .eq. 0) then
          write (iparm(1),500) 'lambda              ', vlamda
        endif
        endif
*
          call zwsax (u,u, fwk(irp), vlamda, iparm , fparm , iwk , fwk ,
     &        ier )
        if (((iparm(13).eq.1).or.(iparm(13).eq.3)) .and. ((mod(iparm(15)
     &     /2,2).eq.1) .or. iparm(12).eq.2 .or. iparm(12).eq.6)) then
          call zwsax (fwk(ir),fwk(ir), fwk(iarp), -vlamda, iparm , 
     &       fparm , iwk , fwk , ier )
        endif
          call zwsax (fwk(iqlr),fwk(iqlr), fwk(ilarp), -vlamda, iparm , 
     &       fparm , iwk , fwk , ier )
        if (((iparm(13).eq.2).or.(iparm(13).eq.3)) .and. ((mod(iparm(15)
     &     /8,2).eq.1) .or. iparm(12).eq.4 .or. iparm(12).eq.8)) go to 
     &     5653
                 go to 5654
 5653  continue
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5656
                 go to 5657
 5656  continue
          call zwset (fwk(iqrqlr),fwk(iqlr), iparm , fparm , iwk , fwk ,
     &        ier )
                go to 5658
 5657 continue
          ivql = iqlr
          ivqr = iqrqlr
       ireq = 7
      iretlb = 2044
      go to 1500
 2044 continue
       if (ijob .eq. -1) go to 900
 5658 continue
                go to 5655
 5654 continue
 5655 continue
*
          call zwrdot (rdot,fwk(iqlr), fwk(iqlr), iparm , fparm , iwk , 
     &       fwk , ier )
        rnorm = sqrt(abs(rdot))
*
        in = in + 1
        iparm(5) = in
        is = is + 1
*
      go to 100
*
 2500 continue
      if (iparm(2) .ge. 3) then
        if (ipme .eq. 0) then
          write (iparm(1),'(/a,i7,a/)') '     Craig''s Method has conver
     &ged in ', iparm(5), ' iterations.'
        endif
      endif
*
*-------------------------------terminate-------------------------------
*
  900 continue
*
      if (iclstg .ne. 1) then
        call zckstg (-1, iwkstg, iparm , fparm , iwk , fwk , ier )
      endif
*
      if (iparm(21).eq.1 .and. ier.ge.0) go to 5659
                 go to 5660
 5659  continue
      call zfmal (ib1,iparm(3), iparm , fparm , iwk , fwk , ier )
      call zfmal (ib2,iparm(3), iparm , fparm , iwk , fwk , ier )
      call zfmal (iu1,iparm(3), iparm , fparm , iwk , fwk , ier )
      call zfmal (iu2,iparm(3), iparm , fparm , iwk , fwk , ier )
        istab = istab2
        if (iparm(19) .eq. 1) then
          call zwdot (fwk(istab+iud), uexact, uexact, iparm , fparm , 
     &       iwk , fwk , ier )
        endif
          call zwdot (fwk(istab+ibd ), b, b, iparm , fparm , iwk , fwk ,
     &        ier )
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5662
                 go to 5663
 5662  continue
          call zwset (fwk(ib2),b, iparm , fparm , iwk , fwk , ier )
                go to 5664
 5663 continue
      call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
          iva = ivitmp
          call zwset (fwk(iva),b, iparm , fparm , iwk , fwk , ier )
          ivql = ib2
       ireq = 5
      iretlb = 2045
      go to 1500
 2045 continue
      call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
       if (ijob .eq. -1) go to 900
 5664 continue
          call zwdot (fwk(istab+ilbd), fwk(ib2), fwk(ib2), iparm , 
     &       fparm , iwk , fwk , ier )
          call zwset (fwk(iu1),fwk(ib2), iparm , fparm , iwk , fwk , 
     &       ier )
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5665
                 go to 5666
 5665  continue
          call zwset (fwk(iu2),fwk(iu1), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5667
 5666 continue
          ivql = iu1
          ivqr = iu2
       ireq = 7
      iretlb = 2046
      go to 1500
 2046 continue
       if (ijob .eq. -1) go to 900
 5667 continue
          call zwdot (fwk(istab+irlbd), fwk(iu2), fwk(iu2), iparm , 
     &       fparm , iwk , fwk , ier )
        if (iparm(19) .eq. 1) then
          call zwsub ( fwk(iu1), u, uexact, iparm , fparm , iwk , fwk , 
     &       ier )
          call zwdot (fwk(istab+ied), fwk(iu1), fwk(iu1), iparm , fparm 
     &       , iwk , fwk , ier )
        endif
      call zfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
          ivqr = ivitmp
          call zwset (fwk(ivqr),u, iparm , fparm , iwk , fwk , ier )
          iva = ib1
       ireq = 3
      iretlb = 2047
      go to 1500
 2047 continue
      call zffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
       if (ijob .eq. -1) go to 900
          call zwsub ( fwk(ib1), b, fwk(ib1), iparm , fparm , iwk , fwk 
     &       , ier )
          call zwdot (fwk(istab+ird), fwk(ib1), fwk(ib1), iparm , fparm 
     &       , iwk , fwk , ier )
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5668
                 go to 5669
 5668  continue
          call zwset (fwk(ib2),fwk(ib1), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5670
 5669 continue
          iva = ib1
          ivql = ib2
       ireq = 5
      iretlb = 2048
      go to 1500
 2048 continue
       if (ijob .eq. -1) go to 900
 5670 continue
          call zwdot (fwk(istab+ilrd ), fwk(ib2), fwk(ib2), iparm , 
     &       fparm , iwk , fwk , ier )
          call zwset (fwk(iu1),fwk(ib2), iparm , fparm , iwk , fwk , 
     &       ier )
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5671
                 go to 5672
 5671  continue
          call zwset (fwk(iu2),fwk(iu1), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5673
 5672 continue
          ivql = iu1
          ivqr = iu2
       ireq = 7
      iretlb = 2049
      go to 1500
 2049 continue
       if (ijob .eq. -1) go to 900
 5673 continue
          call zwdot (fwk(istab+irlrd), fwk(iu2), fwk(iu2), iparm , 
     &       fparm , iwk , fwk , ier )
        if (iparm(19) .eq. 1) then
          if (fwk(istab1+ied) .eq. (0d0,0.0d0)) then
                ier = 4
                call xersho ( ier, 'zcgner' , iparm , 'Value of RELERR m
     &ay be inaccurate' )
          else
            fparm(8) = sqrt(fwk(istab2+ied)/fwk(istab1+ied))
          endif
        endif
        if (fwk(istab1+ird) .eq. (0d0,0.0d0)) then
                ier = 4
                call xersho ( ier, 'zcgner' , iparm , 'Value of RELRSD m
     &ay be inaccurate' )
        else
          fparm(7) = sqrt(fwk(istab2+ird)/fwk(istab1+ird))
        endif
        if (iparm(2) .ge. 4) then
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(1x, 60(''*''))')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''Final iteration statistics'')')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (iparm(19) .eq. 1) then
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of uexact            '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+iud))
        endif
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of b                 '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+ibd))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Ql*b              '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+ilbd))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Qr*Ql*b           '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+irlbd))
        endif
        if (iparm(19) .eq. 1) then
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of error             '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+ied))
        endif
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of residual          '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+ird))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Ql*r              '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+ilrd))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Qr*Ql*r           '',1pg20.10
     &,2x,1pg20.10)') sqrt(fwk(istab+irlrd))
        endif
        if (iparm(19) .eq. 1) then
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''Relative 2-norm of error    '',1pg20.10
     &,2x,1pg20.10)') fparm(8)
        endif
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''Relative 2-norm of residual '',1pg20.10
     &,2x,1pg20.10)') fparm(7)
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(1x, 60(''*''))')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        endif
      call zffre (iu2,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      call zffre (iu1,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      call zffre (ib2,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      call zffre (ib1,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      call zffre (istab2,8, iparm , fparm , iwk , fwk , ier )
      call zffre (istab1,8, iparm , fparm , iwk , fwk , ier )
                go to 5661
 5660 continue
 5661 continue
*
      call zffre (iv3 ,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      call zffre (iv2 ,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      call zffre (iv1 ,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      if (((iparm(13).eq.2).or.(iparm(13).eq.3))) then
        if (((mod(iparm(15)/8,2).eq.1) .or. iparm(12).eq.4 .or. iparm(
     &     12).eq.8)) then
      call zffre (iqrqlr,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
        endif
      endif
      if (((iparm(13).eq.1).or.(iparm(13).eq.3))) then
        if (((mod(iparm(15)/2,2).eq.1) .or. iparm(12).eq.2 .or. iparm(
     &     12).eq.6)) then
      call zffre (ir,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
        endif
      endif
      call zffre (ip,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      call zffre (iqlr,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
*
*
*
*
      call zffre (iftab,10, iparm , fparm , iwk , fwk , ier )
      call xifre (iitab,25, iparm , fparm , iwk , fwk , ier )
*
      if (itimer .eq. 1) then
        call xtimer (tc2,tr2,iparm(22),0)
        fparm(1) = fparm(1) + (tc2-tc1)
        fparm(2) = fparm(2) + (tr2-tr1)
      endif
*
  910 continue
      if (iparm(27) .eq. 0) then
        call zifppr ( -1 , iparm, fparm, ier )
      endif
*
      if (iparm(27) .eq. 0) iparm(27) = 0
      iparm(11) = (iparm(6))
      ireq = -1
      return
*
*-------------------------------brancher--------------------------------
*
*
 1000 continue
*
      ipme = mynode ( )
      iphost = myhost ( )
      log2np = nodedim ( )
      nproc = 2**log2np
      iitab = iparm(11)
      if (iitab.eq.(iparm(6)) .and. ijob.eq.-1) go to 910
*
      if (iparm(11) .eq. (iparm(6))) then
                ier = -4
                call xersho ( ier, 'zcgner' , iparm , 'Values of ijob an
     &d iparm(iptr) incompatible' )
      endif
*
      call xtimer (tc1,tr1,iparm(22),0)
*
        iitab = iwk(iitab+0)
        iftab = iwk(iitab+1)
        istab = iwk(iitab+2)
        iwkstg = iwk(iitab+3)
        iclstg = iwk(iitab+4)
        in = iwk(iitab+5)
        is = iwk(iitab+6)
        itimer = iwk(iitab+7)
        iretlb = iwk(iitab+8)
        ivitmp = iwk(iitab+9)
        ivotmp = iwk(iitab+10)
        iv1tst = iwk(iitab+11)
        iv2tst = iwk(iitab+12)
        istab1 = iwk(iitab+13)
        istab2 = iwk(iitab+14)
        ir = iwk(iitab+15)
        iqlr = iwk(iitab+16)
        iqrqlr = iwk(iitab+17)
        ip = iwk(iitab+18)
        irp = iwk(iitab+19)
        iarp = iwk(iitab+20)
        ilarp = iwk(iitab+21)
        iv1 = iwk(iitab+22)
        iv2 = iwk(iitab+23)
        iv3 = iwk(iitab+24)
        top = fwk(iftab+0)
        bot = fwk(iftab+1)
        rdot = fwk(iftab+2)
        rnorm = fwk(iftab+3)
        r0norm = fwk(iftab+4)
        alpha = fwk(iftab+5)
        vlamda = fwk(iftab+6)
        rr = fwk(iftab+7)
        rrold = fwk(iftab+8)
        pap = fwk(iftab+9)
*
      if (ijob.ne.3 .and. ijob.ne.-1) then
                ier = -4
                call xersho ( ier, 'zcgner' , iparm , 'Argument ijob inv
     &alid' )
      endif
*
      go to (2000,2001,2002,2003,2004,2005,2006,2007,2008,2009,2010,
     &   2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022,
     &   2023,2024,2025,2026,2027,2028,2029,2030,2031,2032,2033,2034,
     &   2035,2036,2037,2038,2039,2040,2041,2042,2043,2044,2045,2046,
     &   2047,2048,2049), iretlb - 2000 + 1
*
*---------------------------request of caller---------------------------
*
*
 1500 continue
      iparm(11) = iitab
        iwk(iitab+0) = iitab
        iwk(iitab+1) = iftab
        iwk(iitab+2) = istab
        iwk(iitab+3) = iwkstg
        iwk(iitab+4) = iclstg
        iwk(iitab+5) = in
        iwk(iitab+6) = is
        iwk(iitab+7) = itimer
        iwk(iitab+8) = iretlb
        iwk(iitab+9) = ivitmp
        iwk(iitab+10) = ivotmp
        iwk(iitab+11) = iv1tst
        iwk(iitab+12) = iv2tst
        iwk(iitab+13) = istab1
        iwk(iitab+14) = istab2
        iwk(iitab+15) = ir
        iwk(iitab+16) = iqlr
        iwk(iitab+17) = iqrqlr
        iwk(iitab+18) = ip
        iwk(iitab+19) = irp
        iwk(iitab+20) = iarp
        iwk(iitab+21) = ilarp
        iwk(iitab+22) = iv1
        iwk(iitab+23) = iv2
        iwk(iitab+24) = iv3
        fwk(iftab+0) = top
        fwk(iftab+1) = bot
        fwk(iftab+2) = rdot
        fwk(iftab+3) = rnorm
        fwk(iftab+4) = r0norm
        fwk(iftab+5) = alpha
        fwk(iftab+6) = vlamda
        fwk(iftab+7) = rr
        fwk(iftab+8) = rrold
        fwk(iftab+9) = pap
      call xtimer (tc2,tr2,iparm(22),0)
      fparm(1) = fparm(1) + (tc2-tc1)
      fparm(2) = fparm(2) + (tr2-tr1)
      return
*
*
*--------------------------------end------------------------------------
*
  500 format ( 1x, a, 1pg20.10,2x,1pg20.10 )
*
      end
