************************************************************************
** ^ROUTINE: WMVGRW - Matrix-vector product "workhorse" routine, for
**                    the regular grid-based matrix storage format
**                    (GR), as well as the union of rectangles matrix
**                    storage format, for scalar/mimd machine cases.
**                       (^)
      subroutine smvgrw ( urflag, ijob , ia , ja , a , vi , vo , iwk , 
     &   fwk , iparm , fparm , ier )
**
** ^DESCRIPTION:
**
**   This routine performs a matrix-vector multiply for two matrix formats
**   associated with regular grids: the GR rectangular grid format,
**   and the UR union of rectangles format (see test drivers testt2d.fm4,
**   test_UR.fm4).
**
**   The current strategy for how the operation is performed is as
**   follows.  For a given subgrid, each stencil point displacement, as
**   defined in the array ia, specifies a shifted subgrid, which typically
**   overlaps (up to 2**ndim) other subgrids.  Handling these overlaps
**   (referred to as slices) via routines like xslice and islice is one
**   of the key tasks.
**
**   To perform the off-processor work, one takes the box defined as the
**   minimal circumscribing box around the union of the shifted subgrids.
**   This box defines all processors from which it is necessary to obtain
**   information.  For each processor subgrid (except for the home
**   processor), that processor subgrid is intersected with the big box to
**   determine what information to send.  This is buffered, sent, received.
**
**   To perform the in-processor part, one finds the maximal inscribing
**   box within the intersection of shifted subgrids, and also within the
**   processor.  This defines an area common to all subgrids, for which
**   highly optimized code can be written.
**
**   There is a third region, the region in-processor but not in the
**   intersection.  This is handled as a special case.
**
**   The off-processor slices are dealt with by locating the necessary
**   receive buffer and then finding the location of that slice within
**   the buffer.
**
**   Several numbering systems are used to describe a gridpoint's
**   location.  One is a zero-based (x varies fastest) natural ordering
**   on subgrids (NO_sub). Fore example, on a 4x4 2D subgrfid, the
**   gridpoint in the 3rd column of the 2nd row is 6. The other is a
**   zero-based (x varies fastest) natural ordering on a slice (NO_sl).
**   Another, called multiindex, is sort of a (zero-based) cartesian
**   coordinate of the gridpoint on the subgrid (MI_sub). Thus the
**   above described point would be (3,2). The last is multiindex
**   numbering, but on a slice (MI_sl).
**
**   The slices are also numbered with a zero-based (x varies fastest)
**   natural ordering. The slice dimensions and location are determined
**   in subroutine xslice.  Integer function islice is used to return a
**   NO_sub location on the destination subgrid given a NO_sl location
**   on a slice.
**
**   One should note that multiindexes and element numbers (e.g.
**   ielt, islice) are zero-based, while arrat elements (e.g. via
**   MAXPGR) are 1-based.
**
**   All operations on elements of the subgrids are processed in terms
**   of simple rectangular boxes which are subsets of the subgrid.
**   Furthermore, every box is processed by planes (nondegenerate
**   ones when they exist).  This decreases overhead and allows
**   optimization of the tight inner loops.
**
** ----------------------------------------------------------------------------
**
**  Concepts
**  --------
**
**   Processor subgrid:
**     The subgrid resident on a processor.  Note that the subgrid number
**     and the processor number are related via the maps igrid and iproc.
**
**   Shifted subgrid:
**     A stencil point's offset vector can be thought of as shifting a
**     processor subgrid by that displacement in NDIM-space.  Note that
**     in the transpose case, the offset vector is negated.
**
**   Circumscribing rectangle:
**     As defined by iminc and imaxc, the coordinates in the processor
**     subgrid coordinate system of the extremal corners of the rectangle
**     which is defined as the smallest circumscribing rectangle
**     containing all the shifted subgrids.
**
**   Inscribing rectangle:
**     As defined by imini and imaxi, the coordinates in the processor
**     subgrid coordinate system of the extremal corners of the rectangle
**     which is defined as the intersection of all the shifted subgrids.
**     Note it may be empty, but this is unlikely for practical problems.
**
**   Communication active processors:
**     As defined by ncpmin, ncpmax, ncp and nccprd, the rectangular
**     subset of processor subgrids which are the ones containing some
**     part of the circumscribing rectangle.  The processor you are on
**     is taken as reference and numbered (0,0,...,0).  The subgrids are
**     numbered lexicographically from 0 to ncpprd-1 by the index iprc.
**
**   Communication buffers:
**     Each subgrid in the set of communication-active processors' subgrids
**     intersects the circumscribing rectangle to form a subrectangle.
**     The size of this subrectangle defines the size of the relevant
**     communication buffer.  Note that the elements of the subrectangle
**     are numbered lexicographically and stored in the buffer in memory
**     as such.  There are actually two types of buffers: vector buffers
**     for the off-processor matvec portions, and matrix buffers for the
**     part of the matrix needed for the transpose matvec.  Both involve
**     the same grid subrectangles.
**
**   Needed buffers:
**     The buffer associated with the processor I am on should not be
**     allocated.  Others may be skipped as well, if the information
**     from that buffer is never used by any stencil point.
**
**   Subgrid number vs. buffer number:
**     The variable iprc is the subgrid number in the rectangle of
**     communication-active processor subgrids.  The associated buffer
**     denoted by ibuf.  For the non-transpose case, ibuf=iprc;
**     for the transpose case, a mapping is made so that ibuf is the
**     same as the iproc value associated with the non-transpose buffer
**     of the same size.  This mapping is done so that buffers of the
**     desired size can be preallocated for use either with the transpose
**     or non-transpose.
**
**   Planes:
**     Operations on subrectangles involve an odometer-type looping
**     over all the elements in the subrectangle lexicographically,
**     amounting to the equivalent of NDIM nested loops.  For efficiency,
**     in all cases the lowest two nontrivial dimensions of the
**     subrectangle are stripped off and handled by explicit loops.
**     The routine xplane finds these axes and also calculates the
**     relevant memory strides for vectors and matrix when looping over
**     such planes.
**
**   Grid numbering on a subrectangle:
**     The routine islice is used to convert the lexicographical element
**     number of an element in a subrectangle to a global lexicographical
**     element number in a rectangle which contains that subrectangle.
**
**   Source and destination vector:
**     The position of a subrectangle in the source (VI) and destination
**     (VO) vectors differs by the offset vector defined by the stencil
**     point in question.  The position in the array is the same as that
**     of VO in the non-transpose case, and the same as that of VI in the
**     transpose case.
**
**   Plane pointers:
**     The values ivip, iap, ivop are the memory pointers to the first
**     element in the relevant plane being processed.  In the specific
**     case in which the result is being calculated on the inscribed
**     rectangle, these are pre-stored in a table for all NSTEN points of
**     the stencil.
**
**   A rectangle minus a rectangle:
**     To process a region which is a rectangle with a contained rectangle
**     removed, the method is to loop over the axes and for each axis
**     process the part of the region above and below the smaller
**     rectangle, and then make note of the deletion of that processed part.
**     In many cases the below or above part may be empty.
**
**   Slices:
**     A shifted processor subgrid cuts the original unshifted subgrid
**     into at most 2**NDIM subrectangles.  These subrectangles are
**     numbered lexicographically by the variable islno.  The routine
**     xslice is currently used to calculate the coordinates of this
**     subrectangle and what processor it is on.
**
**   Subrectangles within buffers:
**     When one of these slices is off-processor, then it will reside
**     entirely within a single communication buffer, since the buffer
**     contains all information needed from that processor by my
**     processor.  However, it may not be the full buffer.  Thus,
**     it is necessary to get the coordinates of the slice within that
**     buffer.
**
** ----------------------------------------------------------------------------
**
** ^AUTHOR:    wdj@beta.lanl.gov
**
** ^MODIFIED: spencer@navier.ae.utexas.edu on Thu May 16 12:00:23 1996 $
**
** ^ARGUMENTS: see *Subroutine Arguments* below.
**
** ^REQUIREMENTS:
**    Subroutines: See External declerations below
**
** ^SIDE_EFFECTS:
**
** ^ALGORITHM:
**   - initialize:
**       precommunicate necessary parts of transpose matrix for transpose ops
**   - general case:
**       compute inscribing/circumscribing rectangles
**       post sends
**       perform matvec on inscribing-rect part
**       perform receives
**       set to zero in-processor off-inscribing-rect part
**       perform matvec on in-proc off-inscribing-rect part
**       perform matvec from off-proc part
**   - terminate:
**       deallocate transpose buffers
**
** ^REFERENCES:
**      author="M. Bromley and Steve Heller and Tim McNerny and Guy Steele",
**      key="Bromley et. al.",
**      title="Fortran at Ten {Gigaflops:  The Connection Machine}
**              Convolution Compiler",
**      booktitle="Proceedings of ACM SIGPLAN 1991 Conference on
**              Programming Language Design and Implementation",
**      publisher="ACM Press",
**      year="1991",
**      pages=" "
**
** ^DOCUMENTATION:
**
************************************************************************
*
      implicit none
          Include 'fcube.h'
*         Include 'veclib.h'
*
************************************************************************
*#    MVFFAL             - An argument list which is common to nearly all
*#                        internal routines of the package.  ALLAL is used in
*#                        calls to fortran routines.
*#
*#                        Integer Scalars:
*#                        ier - (int) error return value.
*#
*#                        Integer Arrays:
*#                        ia - (int) used to store indexing information for
*#                            non-zero elements of matrix stored in a.
*#                        iparm - (int) used to pass integer parameters to and
*#                            from the package.
*#                        iq - (int) used along with floating point array q
*#                            and subroutine subq for preconditioning.
*#                        iwk - (int) integer workspace
*#
*#                        Float Arrays:
*#                        a - (float) array used to store non-zero elements
*#                            of the matrix A.
*#                        b - (float) right hand side of the linear system.
*#                        fparm - (float) used to pass floating point
*#                            parameters to and from the package.
*#                        fwk - (float) floating work array space
*#                        q - (float) used along with integer array iq and
*#                            subroutine subq for preconditioning
*#                        u - (float) solution vector. On input it contains
*#                            the initial guess.
*#                        ubar - (float) exact answer (if known)
*#                        (^./src/m4defs/defs_arglists.m4)
      integer ijob
      integer ier
       integer iwk(*)
       integer iparm(*)
       real fwk(*)
       real fparm(*)
       integer ia(*)
       integer ja(*)
       real a(*)
       real vi(*)
       real vo(*)
      integer urflag
*
************************************************************************
*
      integer ipme
      integer iphost
      integer log2np
      integer nproc
      integer iom
      character*72 errstr
      external ximini
      external ximal
      external xifre
      external sfmini
      external sfmal
      external sffre
      integer nv
      integer ndim
      integer nsten
      integer nb
      integer ns
      integer imaxgr
      external imaxgr
      integer ivaxgr
      external ivaxgr
      integer isten
      integer islno
      integer iaxis
      integer jaxis
      integer ielt
      integer nelt
      integer ibr, ibc
      integer iin1, iin2
      integer nin1, nin2
      integer istrd1
      integer istrd2
      integer ivstr1
      integer ivstr2
      integer ivstrr
      integer imstr1
      integer imstr2
      integer imstrr
      integer imstrc
          integer inelrp
          integer ninb1, ninb2
          integer ibstrr
          integer ibstrc
          integer ibstr1
          integer ibstr2
      integer ivop
      integer ivip
      integer iap
      integer imbufp
      integer ivbufp
      integer ivipt, iapt
      integer ijaptr
      integer iaxin1, iaxin2
      integer isgnt
      integer ilohi
      integer i
          integer ncpprd
          integer isbufp, irbufp
          integer ibufl
          integer itbufp
          integer itbufl
          integer iprc
          integer ibuf
          integer ipsend, iprecv
          integer msg0
          integer iprdax
          integer itab
          integer nbors
          integer ip_ns
          integer ip_nr
        integer ivi1
        integer ia1
        integer ivi2
        integer ia2
        integer ivi3
        integer ia3
        integer ivi4
        integer ia4
        integer ivi5
        integer ia5
      integer isteno, iin1o, nsteni, nin1i, iin1i, ivo0
      integer imini(32)
      integer imaxi(32)
      integer iminc(32)
      integer imaxc(32)
      integer imins(32)
      integer imind(32)
      integer imaxd(32)
      integer inelt(32)
      integer imi (32)
      integer imi2 (32)
          integer iminr (32)
          integer imaxr (32)
          integer inelr (32)
          integer iprcmi(32)
          integer ncpmin(32)
          integer ncpmax(32)
          integer ncp (32)
          integer imin2 (32)
          integer imax2 (32)
          integer imi3 (32)
          integer imaxs (32)
          integer rng (32)
*
************************************************************************
      integer imodf
      external imodf
      external ii2mi
      integer islice
      external islice
      external xslur
      external xslice
      external xplane
      external shad0
      external shad
      external schad0
      external schad
          integer imi2i
          external imi2i
          integer ifloor
          external ifloor
          integer iproc
          external iproc
          integer igrid
          external igrid
*
************************************************************************
*     iceil(i,j) = -ifloor(-i,j)
*
************************************************************************
*
**#   IOFST - stencil offset extracted from ja
*
**#   NSBGR - number of subgrid points along an axis
*
**#   NPROCS - number of processors along an axis
*
**#   NTAB - size of table of pointers stored across calls
*
**#   OFFSTN - magnitude of maximum offset along -ve axial direction in user's
**    definition of neighborhood information
*
**#   OFFSTP - magnitude of maximum offset along +ve axial direction in user's
**#    definition of neighborhood information.
*
**#   NBR - processor number of processor in the $1 position in the
**#    neighborhood of the $2 processor as defined by the user.
*
*=======================================================================
*#    APLANE(   ibr,ibc,iin1,iin2)
*#    APLANEIND(ibr,ibc,iin1,iin2)
*#                      - reference an element of a in a plane.
*#                        indices are 0-based for speed.
*=======================================================================
*#    VIPLANE(   ib,iin1,iin2)
*#    VIPLANEIND(ib,iin1,iin2)
*#                      - reference an element of vi in a plane.
*#                        indices are 0-based for speed.
*=======================================================================
*#    VOPLANE(   ib,iin1,iin2)
*#    VOPLANEIND(ib,iin1,iin2)
*#                      - reference an element of vo in a plane.
*#                        indices are 0-based for speed.
*=======================================================================
*#    MBUFPLANE(ibr,ibc,iin1,iin2)
*#                      - reference an element of a matrix buffer
*#                        in a plane.  indices are 0-based for speed.
*=======================================================================
*#    VBUFPLANE(ibr,iin1,iin2)
*#                      - reference an element of a vector buffer
*#                        in a plane.  indices are 0-based for speed.
*=======================================================================
*#    GET_IN_RECT(ipm)
*#                      - fill out imini/imaxi, the extents of the
*#                        inscribed rectangle, in grid points.
*#                        ipm=1 regular case, -1 transpose case.
*#                        (^)
*=======================================================================
*#    GET_CIRCUM_RECT(ipm)
*#                      - fill out iminc/imaxc, the extents of the
*#                        circumscribing rectangle, in grid points.
*#                        ipm=1 regular case, -1 transpose case.
*#                        (^)
*=======================================================================
*#    GET_CA_PROCS(ipm)
*#                      - fill out ncpmin/max, ncp, ncpprd, the information
*#                        on the communication active processors.
*#                        (^)
*=======================================================================
*#    GET_IPSEND_IPRECV(ipm)
*#                      - fill out ipsend/iprecv based on iprc and ncp*.
*#                        Also fill out iprcmi, the multiindex of iprc in the
*#                        ncpmin/max coord system.
*#                        imi2/3 temporaries.
*#                        (^)
*=======================================================================
*#    CHECK_WHETHER_BUFFER_NEEDED(ipm)
*#                      - figure out whether we will really need to use
*#                        anything from the processor in question.
*#                        Assumes iprcmi.
*#                        (^)
*=======================================================================
*#    GET_SUBGRID_INTERSECT_CIRCUM_RECT(ipm)
*#                      - fill out imins/imaxs, the coords of the rectangle
*#                        that intersects the circumscribed rectangle and
*#                        the subgrid specified in iprcmi.  Also nelt.
*#                        (^)
************************************************************************
*#    VFIL2(v,value)   - Fill a Vector with a value
*#                        (^)
****^^******************************************************************
*     $Modified: spencer@navier.ae.utexas.edu on Thu May 16 12:00:23 1996 $
*     $Id: mvgrw.fm4,v 1.13 1994/11/22 05:19:39 joubert Exp $
*     $Revision: 1.13 $
************************************************************************
*
*-----------------------------------------------------------------------
*-------------------------general initializations-----------------------
*-----------------------------------------------------------------------
*
      ipme = mynode ( )
      iphost = myhost ( )
      log2np = nodedim ( )
      nproc = 2**log2np
      ivipt = (iparm(6))
      iapt = (iparm(6))
      itab = (iparm(6))
      isbufp = (iparm(6))
      irbufp = (iparm(6))
      ibufl = (iparm(6))
      itbufp = (iparm(6))
      itbufl = (iparm(6))
      ijaptr = (iparm(6))
*
      nv = iparm(3 )
      ndim = ia (1 )
      nsten = ia (2)
      nb = ia (3 )
      ns = nv / nb
          if ((ijob) .eq. (3)) then
           isgnt = 1
          else
           isgnt = -1
          endif
*
*-----------------------------------------------------------------------
*--------------------------initialization case--------------------------
*-----------------------------------------------------------------------
*
      if (ijob.eq.1) then
        iparm(11) = (iparm(6))
*
          do 8511 iaxis = 1, ndim
            iminc(iaxis) = 0 +(-1)*ja(iaxis+ndim*(1 -1))
             do 8513 isten = 2, nsten
      iminc(iaxis) = min(iminc(iaxis), 0 +(-1)*ja(iaxis+ndim*(isten-1)))
 8513 continue
            imaxc(iaxis) = ja(iaxis+ndim*nsten)-1+(-1)*ja(iaxis+ndim*(1 
     &         -1))
             do 8515 isten = 2, nsten
      imaxc(iaxis) = max(imaxc(iaxis), ja(iaxis+ndim*nsten)-1 +(-1)*ja(
     &   iaxis+ndim*(isten-1)))
 8515 continue
 8511     continue
*
          ncpprd = 1
          do 8517 iaxis = 1, ndim
            ncpmin(iaxis) = ifloor(iminc(iaxis),ja(iaxis+ndim*nsten))
            ncpmax(iaxis) = ifloor(imaxc(iaxis),ja(iaxis+ndim*nsten))
            ncp (iaxis) = max(0,ncpmax(iaxis)-ncpmin(iaxis)+1)
            ncpprd = ncpprd * ncp(iaxis)
 8517     continue
*
      call ximal (itab,7, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
              iparm(11) = itab
                                      iwk(itab ) = ncpprd
      call ximal (itbufp,ncpprd, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          iwk(itab+1) = itbufp
      call ximal (itbufl,ncpprd, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          iwk(itab+2) = itbufl
         do 8519 ibuf = 0, ncpprd-1
      iwk(itbufp+ibuf) = (iparm(6))
 8519 continue
      call ximal (isbufp,ncpprd, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          iwk(itab+3) = isbufp
      call ximal (irbufp,ncpprd, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          iwk(itab+4) = irbufp
      call ximal (ibufl ,ncpprd, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          iwk(itab+5) = ibufl
         do 8521 ibuf = 0, ncpprd-1
      iwk(isbufp+ibuf) = (iparm(6))
      iwk(irbufp+ibuf) = (iparm(6))
 8521 continue
*
      call ximal (ijaptr,ncpprd, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        iwk(itab+6) = ijaptr
        if (urflag .eq. 0) then
*******
        else
           do 8523 iprc = 0, ncpprd-1
      iwk(ijaptr+iprc) = ja(1+iprc)
 8523 continue
****** MUST FIX THE RIGHT WAY
        endif
*
         do 8525 iprc = 0, ncpprd-1
           if(urflag.eq.1)then
             nbors = 1
             do 8527 iaxis = 1,ndim
               rng(iaxis) = ja(iaxis + ndim*(nsten+4)) - ja(iaxis + 
     &            ndim*(nsten+3)) + 1
               nbors = nbors*rng(iaxis)
               imi3(iaxis) = -ja(iaxis + ndim*(nsten+3))
 8527        continue
             call ii2mi(iprcmi,iprc,ndim,ncp)
              do 8529 iaxis = 1, ndim
      iprcmi(iaxis) = iprcmi(iaxis) + ncpmin(iaxis)
 8529 continue
              do 8531 iaxis = 1, ndim
      imi2(iaxis) = imi3(iaxis) + iprcmi(iaxis)
 8531 continue
             ip_ns = imi2i(imi2,ndim,rng)
             ipsend = ja(1 + ip_ns + (nsten+5)*ndim)
              do 8533 iaxis = 1, ndim
      imi2(iaxis) = imi3(iaxis) - iprcmi(iaxis)
 8533 continue
             ip_nr = imi2i(imi2,ndim,rng)
             iprecv = ja(1 + ip_nr + (nsten+5)*ndim)
           else
             call ii2mi (iprcmi,iprc,ndim,ncp)
              do 8535 iaxis = 1, ndim
      iprcmi(iaxis) = iprcmi(iaxis) + ncpmin(iaxis)
 8535 continue
             call ii2mi (imi2,igrid(ipme,ndim,ja(1+ndim*(nsten+1))),
     &          ndim,ja(1+ndim*(nsten+1)))
              do 8537 iaxis = 1, ndim
      imi3(iaxis) = imodf(imi2(iaxis)+iprcmi(iaxis), ja(iaxis+ndim*(
     &   nsten+1)))
 8537 continue
             ipsend = iproc(imi2i(imi3,ndim,ja(1+ndim*(nsten+1))),ndim,
     &          ja(1+ndim*(nsten+1)))
              do 8539 iaxis = 1, ndim
      imi3(iaxis) = imodf(imi2(iaxis)-iprcmi(iaxis), ja(iaxis+ndim*(
     &   nsten+1)))
 8539 continue
             iprecv = iproc(imi2i(imi3,ndim,ja(1+ndim*(nsten+1))),ndim,
     &          ja(1+ndim*(nsten+1)))
             if (-1 .eq. +1) then
                do 8541 iaxis = 1, ndim
      imi2(iaxis) = iprcmi(iaxis) - ncpmin(iaxis)
 8541 continue
             else
                do 8543 iaxis = 1, ndim
      imi2(iaxis) = ncpmax(iaxis) - iprcmi(iaxis)
 8543 continue
             endif
             ibuf = imi2i (imi2,ndim,ncp)
           endif
           if (ipsend .eq. ipme) go to 8525
           do 8545 isten = 1, nsten
             do 8547 iaxis = 1, ndim
               if (max(min((-1)*ja(iaxis+ndim*(isten-1)),1),-1) .ne. 
     &            max(min(iprcmi(iaxis) ,1),-1) ) go to 5500
 8547        continue
             go to 5501
 5500       continue
 8545      continue
           go to 8525
 5501      continue
           inelrp = 1
           do 8549 iaxis = 1, ndim
             iminr(iaxis) = max(0 , iminc(iaxis)+ja(iaxis+ndim*nsten)*
     &          iprcmi(iaxis))
             imaxr(iaxis) = min(ja(iaxis+ndim*nsten)-1, imaxc(iaxis)+ja(
     &          iaxis+ndim*nsten)*iprcmi(iaxis))
             inelr(iaxis) = max(0,imaxr(iaxis)-iminr(iaxis)+1)
             inelrp = inelrp * inelr(iaxis)
 8549      continue
           iwk( itbufl+ibuf) = inelrp*nb*nb*nsten
      call sfmal (iwk(itbufp+ibuf),iwk(itbufl+ibuf), iparm , fparm , 
     &   iwk , fwk , ier )
           if (ier .lt. 0) go to 900
 8525    continue
*
      if (iparm(26) + (ncpprd) + 0 - 1 .gt. iparm(25)) then
        if (iparm(26) + 0 - 1 .gt. iparm(25)) then
                ier = -12
                call xersho ( ier, 'smvgrw' , iparm , 'Unable to perform
     & synchronization' )
                go to 900
        endif
      call gsync ()
        iparm(26) = iparm(24)
      endif
         do 8551 iprc = 0, ncpprd-1
           if(urflag.eq.1)then
             nbors = 1
             do 8553 iaxis = 1,ndim
               rng(iaxis) = ja(iaxis + ndim*(nsten+4)) - ja(iaxis + 
     &            ndim*(nsten+3)) + 1
               nbors = nbors*rng(iaxis)
               imi3(iaxis) = -ja(iaxis + ndim*(nsten+3))
 8553        continue
             call ii2mi(iprcmi,iprc,ndim,ncp)
              do 8555 iaxis = 1, ndim
      iprcmi(iaxis) = iprcmi(iaxis) + ncpmin(iaxis)
 8555 continue
              do 8557 iaxis = 1, ndim
      imi2(iaxis) = imi3(iaxis) + iprcmi(iaxis)
 8557 continue
             ip_ns = imi2i(imi2,ndim,rng)
             ipsend = ja(1 + ip_ns + (nsten+5)*ndim)
              do 8559 iaxis = 1, ndim
      imi2(iaxis) = imi3(iaxis) - iprcmi(iaxis)
 8559 continue
             ip_nr = imi2i(imi2,ndim,rng)
             iprecv = ja(1 + ip_nr + (nsten+5)*ndim)
           else
             call ii2mi (iprcmi,iprc,ndim,ncp)
              do 8561 iaxis = 1, ndim
      iprcmi(iaxis) = iprcmi(iaxis) + ncpmin(iaxis)
 8561 continue
             call ii2mi (imi2,igrid(ipme,ndim,ja(1+ndim*(nsten+1))),
     &          ndim,ja(1+ndim*(nsten+1)))
              do 8563 iaxis = 1, ndim
      imi3(iaxis) = imodf(imi2(iaxis)+iprcmi(iaxis), ja(iaxis+ndim*(
     &   nsten+1)))
 8563 continue
             ipsend = iproc(imi2i(imi3,ndim,ja(1+ndim*(nsten+1))),ndim,
     &          ja(1+ndim*(nsten+1)))
              do 8565 iaxis = 1, ndim
      imi3(iaxis) = imodf(imi2(iaxis)-iprcmi(iaxis), ja(iaxis+ndim*(
     &   nsten+1)))
 8565 continue
             iprecv = iproc(imi2i(imi3,ndim,ja(1+ndim*(nsten+1))),ndim,
     &          ja(1+ndim*(nsten+1)))
             if (-1 .eq. +1) then
                do 8567 iaxis = 1, ndim
      imi2(iaxis) = iprcmi(iaxis) - ncpmin(iaxis)
 8567 continue
             else
                do 8569 iaxis = 1, ndim
      imi2(iaxis) = ncpmax(iaxis) - iprcmi(iaxis)
 8569 continue
             endif
             ibuf = imi2i (imi2,ndim,ncp)
           endif
           if (ipsend .eq. ipme) go to 8551
           do 8571 isten = 1, nsten
             do 8573 iaxis = 1, ndim
               if (max(min((-1)*ja(iaxis+ndim*(isten-1)),1),-1) .ne. 
     &            max(min(iprcmi(iaxis) ,1),-1) ) go to 5502
 8573        continue
             go to 5503
 5502       continue
 8571      continue
           go to 8551
 5503      continue
           inelrp = 1
           do 8575 iaxis = 1, ndim
             iminr(iaxis) = max(0 , iminc(iaxis)+ja(iaxis+ndim*nsten)*
     &          iprcmi(iaxis))
             imaxr(iaxis) = min(ja(iaxis+ndim*nsten)-1, imaxc(iaxis)+ja(
     &          iaxis+ndim*nsten)*iprcmi(iaxis))
             inelr(iaxis) = max(0,imaxr(iaxis)-iminr(iaxis)+1)
             inelrp = inelrp * inelr(iaxis)
 8575      continue
           call xplane ( ja(1+ndim*nsten), inelr, ia, iparm, iaxin1, 
     &        iaxin2, istrd1, istrd2, nin1, nin2, ivstrr,ivstr1,ivstr2, 
     &        imstrr,imstrc,imstr1,imstr2)
           ibstrr = 1
           ibstrc = nb*inelrp
           ibstr1 = nb
           ibstr2 = nb*nin1
           do 8577 ielt = 0, inelrp-1, nin1*nin2
             do 8579 isten = 1, nsten
               iap = imaxgr(1,1,isten,1+islice(ielt,ndim,ja(1+ndim*
     &            nsten),inelr,iminr),ia(4),nb,ns,nsten)
               imbufp = iwk(itbufp+ibuf) + nb*(ielt+inelrp*nb*(isten-1))
                do 8581 iin2 = 0, nin2-1
       do 8583 iin1 = 0, nin1-1
       do 8585 ibr = 0, nb-1
       do 8587 ibc = 0, nb-1
      fwk(imbufp+(ibr)*ibstrr+(ibc)*ibstrc +(iin1)*ibstr1+(iin2)*ibstr2)
     &    = a(iap+(ibr)*imstrr+(ibc)*imstrc +(iin1)*imstr1+(iin2)*
     &   imstr2)
 8587 continue
 8585 continue
 8583 continue
 8581 continue
 8579        continue
 8577      continue
           if(ipsend.ne.-1)then
      call csend ((iparm(26)), fwk(iwk(itbufp+ibuf)), 4*(iwk(itbufl+
     &   ibuf)), (ipsend), 0)
           endif
           if(iprecv.ne.-1)then
      call crecv ((iparm(26)), fwk(iwk(itbufp+ibuf)), 4*(iwk(itbufl+
     &   ibuf)))
           endif
           iparm(26) = iparm(26) + 1
 8551    continue
*
         do 8589 iprc = 0, ncpprd-1
           if(urflag.eq.1)then
             nbors = 1
             do 8591 iaxis = 1,ndim
               rng(iaxis) = ja(iaxis + ndim*(nsten+4)) - ja(iaxis + 
     &            ndim*(nsten+3)) + 1
               nbors = nbors*rng(iaxis)
               imi3(iaxis) = -ja(iaxis + ndim*(nsten+3))
 8591        continue
             call ii2mi(iprcmi,iprc,ndim,ncp)
              do 8593 iaxis = 1, ndim
      iprcmi(iaxis) = iprcmi(iaxis) + ncpmin(iaxis)
 8593 continue
              do 8595 iaxis = 1, ndim
      imi2(iaxis) = imi3(iaxis) + iprcmi(iaxis)
 8595 continue
             ip_ns = imi2i(imi2,ndim,rng)
             ipsend = ja(1 + ip_ns + (nsten+5)*ndim)
              do 8597 iaxis = 1, ndim
      imi2(iaxis) = imi3(iaxis) - iprcmi(iaxis)
 8597 continue
             ip_nr = imi2i(imi2,ndim,rng)
             iprecv = ja(1 + ip_nr + (nsten+5)*ndim)
           else
             call ii2mi (iprcmi,iprc,ndim,ncp)
              do 8599 iaxis = 1, ndim
      iprcmi(iaxis) = iprcmi(iaxis) + ncpmin(iaxis)
 8599 continue
             call ii2mi (imi2,igrid(ipme,ndim,ja(1+ndim*(nsten+1))),
     &          ndim,ja(1+ndim*(nsten+1)))
              do 8601 iaxis = 1, ndim
      imi3(iaxis) = imodf(imi2(iaxis)+iprcmi(iaxis), ja(iaxis+ndim*(
     &   nsten+1)))
 8601 continue
             ipsend = iproc(imi2i(imi3,ndim,ja(1+ndim*(nsten+1))),ndim,
     &          ja(1+ndim*(nsten+1)))
              do 8603 iaxis = 1, ndim
      imi3(iaxis) = imodf(imi2(iaxis)-iprcmi(iaxis), ja(iaxis+ndim*(
     &   nsten+1)))
 8603 continue
             iprecv = iproc(imi2i(imi3,ndim,ja(1+ndim*(nsten+1))),ndim,
     &          ja(1+ndim*(nsten+1)))
             if (1 .eq. +1) then
                do 8605 iaxis = 1, ndim
      imi2(iaxis) = iprcmi(iaxis) - ncpmin(iaxis)
 8605 continue
             else
                do 8607 iaxis = 1, ndim
      imi2(iaxis) = ncpmax(iaxis) - iprcmi(iaxis)
 8607 continue
             endif
             ibuf = imi2i (imi2,ndim,ncp)
           endif
           if (ipsend .eq. ipme) go to 8589
           do 8609 isten = 1, nsten
             do 8611 iaxis = 1, ndim
               if (max(min((1)*ja(iaxis+ndim*(isten-1)),1),-1) .ne. max(
     &            min(iprcmi(iaxis) ,1),-1) ) go to 5504
 8611        continue
             go to 5505
 5504       continue
 8609      continue
           go to 8589
 5505      continue
           inelrp = 1
           do 8613 iaxis = 1, ndim
             iminr(iaxis) = max(0 , iminc(iaxis)+ja(iaxis+ndim*nsten)*
     &          iprcmi(iaxis))
             imaxr(iaxis) = min(ja(iaxis+ndim*nsten)-1, imaxc(iaxis)+ja(
     &          iaxis+ndim*nsten)*iprcmi(iaxis))
             inelr(iaxis) = max(0,imaxr(iaxis)-iminr(iaxis)+1)
             inelrp = inelrp * inelr(iaxis)
 8613      continue
           iwk( ibufl +ibuf) = inelrp*nb
      call sfmal (iwk(isbufp+ibuf),iwk(ibufl+ibuf), iparm , fparm , iwk 
     &   , fwk , ier )
           if (ier .lt. 0) go to 900
      call sfmal (iwk(irbufp+ibuf),iwk(ibufl+ibuf), iparm , fparm , iwk 
     &   , fwk , ier )
           if (ier .lt. 0) go to 900
 8589    continue
*
        go to 900
      else
        itab = iparm(11)
        if (itab .eq. (iparm(6))) goto 900
        ncpprd = iwk(itab )
        itbufp = iwk(itab+1)
        itbufl = iwk(itab+2)
        isbufp = iwk(itab+3)
        irbufp = iwk(itab+4)
        ibufl = iwk(itab+5)
        ijaptr = iwk(itab+6)
      endif
*
*-----------------------------------------------------------------------
*----------------------------termination case---------------------------
*-----------------------------------------------------------------------
*
      if (ijob.eq.-1) then
        go to 900
      endif
*
*-----------------------------------------------------------------------
*------------------------------other cases------------------------------
*-----------------------------------------------------------------------
*
      if (ijob.ne.3 .and. ijob.ne.4) then
                ier = -4
                call xersho ( ier, 'smvgrw' , iparm , ' ' )
                go to 900
      endif
*
*-----------------------------------------------------------------------
*  ---simpler (but very slow) code for uniprocessor case----------------
**#   do i = 1, nv
**#     vo(i) = ZERO
**#   enddo
**#   do isten = 1, nsten
**#     do islice = 0, 2**ndim-1
**#       call xslur ( ia, ja, isten, islice, ijob, imins, imind, inelt,
**#  &                 ncelts )
**#       if (ncelts .eq. 0) next
**#       do ibr = 1, nb
**#         do ibc = 1, nb
**#           do ielt = 0, ncelts
**#             vo[]VAXPGR(ibr,1+islice(ielt,ndim,NSBGR(1),inelt,imind)) =
**#  &          vo[]VAXPGR(ibr,1+islice(ielt,ndim,NSBGR(1),inelt,imind))
**#  &         + a[]MAXPGR(ibr,ibc,isten,
**#  &                          1+islice(ielt,ndim,NSBGR(1),inelt,imind))*
**#  &          vi[]VAXPGR(ibc,1+islice(ielt,ndim,NSBGR(1),inelt,imins))
**#           enddo
**#         enddo
**#       enddo
**#     enddo
**#   enddo
*
*-----------------------------------------------------------------------
*  ---compute information for inscribed and circumscribed rectangles----
*
          do 8615 iaxis = 1, ndim
            iminc(iaxis) = 0 +(isgnt)*ja(iaxis+ndim*(1 -1))
             do 8617 isten = 2, nsten
      iminc(iaxis) = min(iminc(iaxis), 0 +(isgnt)*ja(iaxis+ndim*(isten-
     &   1)))
 8617 continue
            imaxc(iaxis) = ja(iaxis+ndim*nsten)-1+(isgnt)*ja(iaxis+ndim*
     &         (1 -1))
             do 8619 isten = 2, nsten
      imaxc(iaxis) = max(imaxc(iaxis), ja(iaxis+ndim*nsten)-1 +(isgnt)*
     &   ja(iaxis+ndim*(isten-1)))
 8619 continue
 8615     continue
*
          do 8621 iaxis = 1, ndim
            imini(iaxis) = 0 +(isgnt)*ja(iaxis+ndim*(1 -1))
             do 8623 isten = 2, nsten
      imini(iaxis) = max(imini(iaxis), 0 +(isgnt)*ja(iaxis+ndim*(isten-
     &   1)))
 8623 continue
            imaxi(iaxis) = ja(iaxis+ndim*nsten)-1+(isgnt)*ja(iaxis+ndim*
     &         (1 -1))
             do 8625 isten = 2, nsten
      imaxi(iaxis) = min(imaxi(iaxis), ja(iaxis+ndim*nsten)-1+(isgnt)*
     &   ja(iaxis+ndim*(isten-1)))
 8625 continue
 8621     continue
*
*-----------------------------------------------------------------------
*  ---perform sends-----------------------------------------------------
*
          ncpprd = 1
          do 8627 iaxis = 1, ndim
            ncpmin(iaxis) = ifloor(iminc(iaxis),ja(iaxis+ndim*nsten))
            ncpmax(iaxis) = ifloor(imaxc(iaxis),ja(iaxis+ndim*nsten))
            ncp (iaxis) = max(0,ncpmax(iaxis)-ncpmin(iaxis)+1)
            ncpprd = ncpprd * ncp(iaxis)
 8627     continue
*
*
      if (iparm(26) + (ncpprd) + 0 - 1 .gt. iparm(25)) then
        if (iparm(26) + 0 - 1 .gt. iparm(25)) then
                ier = -12
                call xersho ( ier, 'smvgrw' , iparm , 'Unable to perform
     & synchronization' )
                go to 900
        endif
      call gsync ()
        iparm(26) = iparm(24)
      endif
         msg0 = iparm(26)
         do 8629 iprc = 0, ncpprd-1
           if(urflag.eq.1)then
             nbors = 1
             do 8631 iaxis = 1,ndim
               rng(iaxis) = ja(iaxis + ndim*(nsten+4)) - ja(iaxis + 
     &            ndim*(nsten+3)) + 1
               nbors = nbors*rng(iaxis)
               imi3(iaxis) = -ja(iaxis + ndim*(nsten+3))
 8631        continue
             call ii2mi(iprcmi,iprc,ndim,ncp)
              do 8633 iaxis = 1, ndim
      iprcmi(iaxis) = iprcmi(iaxis) + ncpmin(iaxis)
 8633 continue
              do 8635 iaxis = 1, ndim
      imi2(iaxis) = imi3(iaxis) + iprcmi(iaxis)
 8635 continue
             ip_ns = imi2i(imi2,ndim,rng)
             ipsend = ja(1 + ip_ns + (nsten+5)*ndim)
              do 8637 iaxis = 1, ndim
      imi2(iaxis) = imi3(iaxis) - iprcmi(iaxis)
 8637 continue
             ip_nr = imi2i(imi2,ndim,rng)
             iprecv = ja(1 + ip_nr + (nsten+5)*ndim)
           else
             call ii2mi (iprcmi,iprc,ndim,ncp)
              do 8639 iaxis = 1, ndim
      iprcmi(iaxis) = iprcmi(iaxis) + ncpmin(iaxis)
 8639 continue
             call ii2mi (imi2,igrid(ipme,ndim,ja(1+ndim*(nsten+1))),
     &          ndim,ja(1+ndim*(nsten+1)))
              do 8641 iaxis = 1, ndim
      imi3(iaxis) = imodf(imi2(iaxis)+iprcmi(iaxis), ja(iaxis+ndim*(
     &   nsten+1)))
 8641 continue
             ipsend = iproc(imi2i(imi3,ndim,ja(1+ndim*(nsten+1))),ndim,
     &          ja(1+ndim*(nsten+1)))
              do 8643 iaxis = 1, ndim
      imi3(iaxis) = imodf(imi2(iaxis)-iprcmi(iaxis), ja(iaxis+ndim*(
     &   nsten+1)))
 8643 continue
             iprecv = iproc(imi2i(imi3,ndim,ja(1+ndim*(nsten+1))),ndim,
     &          ja(1+ndim*(nsten+1)))
             if (isgnt .eq. +1) then
                do 8645 iaxis = 1, ndim
      imi2(iaxis) = iprcmi(iaxis) - ncpmin(iaxis)
 8645 continue
             else
                do 8647 iaxis = 1, ndim
      imi2(iaxis) = ncpmax(iaxis) - iprcmi(iaxis)
 8647 continue
             endif
             ibuf = imi2i (imi2,ndim,ncp)
           endif
           if (ipsend .eq. ipme) go to 8629
           do 8649 isten = 1, nsten
             do 8651 iaxis = 1, ndim
               if (max(min((isgnt)*ja(iaxis+ndim*(isten-1)),1),-1) .ne. 
     &            max(min(iprcmi(iaxis) ,1),-1) ) go to 5506
 8651        continue
             go to 5507
 5506       continue
 8649      continue
           go to 8629
 5507      continue
           inelrp = 1
           do 8653 iaxis = 1, ndim
             iminr(iaxis) = max(0 , iminc(iaxis)+ja(iaxis+ndim*nsten)*
     &          iprcmi(iaxis))
             imaxr(iaxis) = min(ja(iaxis+ndim*nsten)-1, imaxc(iaxis)+ja(
     &          iaxis+ndim*nsten)*iprcmi(iaxis))
             inelr(iaxis) = max(0,imaxr(iaxis)-iminr(iaxis)+1)
             inelrp = inelrp * inelr(iaxis)
 8653      continue
           call xplane ( ja(1+ndim*nsten), inelr, ia, iparm, iaxin1, 
     &        iaxin2, istrd1, istrd2, nin1, nin2, ivstrr,ivstr1,ivstr2, 
     &        imstrr,imstrc,imstr1,imstr2)
           ibstrr = 1
           ibstrc = nb*inelrp
           ibstr1 = nb
           ibstr2 = nb*nin1
           do 8655 ielt = 0, inelrp-1, nin1*nin2
             ivbufp = iwk(isbufp+ibuf) + nb*ielt
             ivip = ivaxgr(1,1+islice(ielt,ndim,ja(1+ndim*nsten),inelr,
     &          iminr),ia(5),nb,ns,nsten)
              do 8657 iin2 = 0, nin2-1
       do 8659 iin1 = 0, nin1-1
       do 8661 ibr = 0, nb-1
      fwk(ivbufp+(ibr)*ibstrr +(iin1)*ibstr1+(iin2)*ibstr2) = vi(ivip+(
     &   ibr)*ivstrr +(iin1)*ivstr1+(iin2)*ivstr2)
 8661 continue
 8659 continue
 8657 continue
 8655      continue
           if(ipsend.ne.-1)then
      call csend ((iparm(26)), fwk(iwk(isbufp+ibuf)), 4*(iwk(ibufl+ibuf)
     &   ), (ipsend), 0)
           endif
           iparm(26) = iparm(26) + 1
 8629    continue
*
*-----------------------------------------------------------------------
*  ---perform matvec on inscribed rect----------------------------------
*
       do 8663 iaxis = 1, ndim
      inelt(iaxis) = max(0,imaxi(iaxis)-imini(iaxis)+1)
 8663 continue
      nelt = 1 
      do 8665 iaxis = 1, ndim
      nelt = nelt * inelt(iaxis)
 8665 continue
      if (nelt .eq. 0) go to 101
      call xplane ( ja(1+ndim*nsten), inelt, ia, iparm, iaxin1, iaxin2, 
     &   istrd1, istrd2, nin1, nin2, ivstrr,ivstr1,ivstr2, imstrr,
     &   imstrc,imstr1,imstr2)
      call ximal (ivipt,nsten, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call ximal (iapt ,nsten, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      do 8667 ielt = 0, nelt-1, nin1*nin2
        ivop = ivaxgr(1,1+islice(ielt,ndim,ja(1+ndim*nsten),inelt,imini)
     &     ,ia(5),nb,ns,nsten)
        do 8669 isten = 1, nsten
           do 8671 iaxis = 1, ndim
      imins(iaxis) = imini(iaxis) + isgnt*ja(iaxis+ndim*(isten-1))
 8671 continue
          ivip = ivaxgr(1,1+islice(ielt,ndim,ja(1+ndim*nsten),inelt,
     &       imins),ia(5),nb,ns,nsten)
          if (ijob .eq. 3) then
            iap = imaxgr(1,1,isten,1+islice(ielt,ndim,ja(1+ndim*nsten),
     &         inelt,imini),ia(4),nb,ns,nsten)
          else
            iap = imaxgr(1,1,isten,1+islice(ielt,ndim,ja(1+ndim*nsten),
     &         inelt,imins),ia(4),nb,ns,nsten)
          endif
          iwk(ivipt-1+isten) = ivip
          iwk(iapt -1+isten) = iap
 8669   continue
        if (ijob .eq. 3) then
          if (nb .eq. 1) then
            do 8673 iin1o = 0, nin1-1, 100
              do 8675 iin2 = 0, nin2-1
                do 8677 isteno = 1, nsten, 5
                    isten = isteno + 0
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap = iwk(iapt -1+isten)
                      ivi1 = ivip+(0)*ivstrr +(iin1o)*ivstr1+(iin2)*
     &                   ivstr2
                      ia1 = iap+(0)*imstrr+(0)*imstrc +(iin1o)*imstr1+(
     &                   iin2)*imstr2
                    endif
                    isten = isteno + 1
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap = iwk(iapt -1+isten)
                      ivi2 = ivip+(0)*ivstrr +(iin1o)*ivstr1+(iin2)*
     &                   ivstr2
                      ia2 = iap+(0)*imstrr+(0)*imstrc +(iin1o)*imstr1+(
     &                   iin2)*imstr2
                    endif
                    isten = isteno + 2
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap = iwk(iapt -1+isten)
                      ivi3 = ivip+(0)*ivstrr +(iin1o)*ivstr1+(iin2)*
     &                   ivstr2
                      ia3 = iap+(0)*imstrr+(0)*imstrc +(iin1o)*imstr1+(
     &                   iin2)*imstr2
                    endif
                    isten = isteno + 3
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap = iwk(iapt -1+isten)
                      ivi4 = ivip+(0)*ivstrr +(iin1o)*ivstr1+(iin2)*
     &                   ivstr2
                      ia4 = iap+(0)*imstrr+(0)*imstrc +(iin1o)*imstr1+(
     &                   iin2)*imstr2
                    endif
                    isten = isteno + 4
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap = iwk(iapt -1+isten)
                      ivi5 = ivip+(0)*ivstrr +(iin1o)*ivstr1+(iin2)*
     &                   ivstr2
                      ia5 = iap+(0)*imstrr+(0)*imstrc +(iin1o)*imstr1+(
     &                   iin2)*imstr2
                    endif
                  ivo0 = ivop+(0)*ivstrr +(iin1o)*ivstr1+(iin2)*ivstr2
                  nsteni = min(5,nsten-isteno+1)
                  nin1i = min(100,nin1-iin1o)
                  go to (701,702,703,704,705), nsteni
  701       continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
                          do 8679 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i) ) = a(ia1 +(iin1i) )* vi(
     &                       ivi1+(iin1i) )
 8679                     continue
                        else
                          do 8681 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i)*ivstr1) = a(ia1 +(iin1i)*
     &                       imstr1)* vi(ivi1+(iin1i)*ivstr1)
 8681                     continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
                          do 8683 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i) ) = vo(ivo0 +(iin1i) ) + a(
     &                       ia1 +(iin1i) )* vi(ivi1+(iin1i) )
 8683                     continue
                        else
                          do 8685 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i)*ivstr1) = vo(ivo0 +(iin1i)*
     &                       ivstr1) + a(ia1 +(iin1i)*imstr1)* vi(ivi1+(
     &                       iin1i)*ivstr1)
 8685                     continue
                        endif
                      endif
                    go to 700
  702       continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
                          do 8687 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i) ) = a(ia1 +(iin1i) )* vi(
     &                       ivi1+(iin1i) ) + a(ia2 +(iin1i) )* vi(ivi2+
     &                       (iin1i) )
 8687                     continue
                        else
                          do 8689 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i)*ivstr1) = a(ia1 +(iin1i)*
     &                       imstr1)* vi(ivi1+(iin1i)*ivstr1) + a(ia2 +(
     &                       iin1i)*imstr1)* vi(ivi2+(iin1i)*ivstr1)
 8689                     continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
                          do 8691 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i) ) = vo(ivo0 +(iin1i) ) + a(
     &                       ia1 +(iin1i) )* vi(ivi1+(iin1i) ) + a(ia2 +
     &                       (iin1i) )* vi(ivi2+(iin1i) )
 8691                     continue
                        else
                          do 8693 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i)*ivstr1) = vo(ivo0 +(iin1i)*
     &                       ivstr1) + a(ia1 +(iin1i)*imstr1)* vi(ivi1+(
     &                       iin1i)*ivstr1) + a(ia2 +(iin1i)*imstr1)* 
     &                       vi(ivi2+(iin1i)*ivstr1)
 8693                     continue
                        endif
                      endif
                    go to 700
  703       continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
                          do 8695 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i) ) = a(ia1 +(iin1i) )* vi(
     &                       ivi1+(iin1i) ) + a(ia2 +(iin1i) )* vi(ivi2+
     &                       (iin1i) ) + a(ia3 +(iin1i) )* vi(ivi3+(
     &                       iin1i) )
 8695                     continue
                        else
                          do 8697 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i)*ivstr1) = a(ia1 +(iin1i)*
     &                       imstr1)* vi(ivi1+(iin1i)*ivstr1) + a(ia2 +(
     &                       iin1i)*imstr1)* vi(ivi2+(iin1i)*ivstr1) + 
     &                       a(ia3 +(iin1i)*imstr1)* vi(ivi3+(iin1i)*
     &                       ivstr1)
 8697                     continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
                          do 8699 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i) ) = vo(ivo0 +(iin1i) ) + a(
     &                       ia1 +(iin1i) )* vi(ivi1+(iin1i) ) + a(ia2 +
     &                       (iin1i) )* vi(ivi2+(iin1i) ) + a(ia3 +(
     &                       iin1i) )* vi(ivi3+(iin1i) )
 8699                     continue
                        else
                          do 8701 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i)*ivstr1) = vo(ivo0 +(iin1i)*
     &                       ivstr1) + a(ia1 +(iin1i)*imstr1)* vi(ivi1+(
     &                       iin1i)*ivstr1) + a(ia2 +(iin1i)*imstr1)* 
     &                       vi(ivi2+(iin1i)*ivstr1) + a(ia3 +(iin1i)*
     &                       imstr1)* vi(ivi3+(iin1i)*ivstr1)
 8701                     continue
                        endif
                      endif
                    go to 700
  704       continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
                          do 8703 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i) ) = a(ia1 +(iin1i) )* vi(
     &                       ivi1+(iin1i) ) + a(ia2 +(iin1i) )* vi(ivi2+
     &                       (iin1i) ) + a(ia3 +(iin1i) )* vi(ivi3+(
     &                       iin1i) ) + a(ia4 +(iin1i) )* vi(ivi4+(
     &                       iin1i) )
 8703                     continue
                        else
                          do 8705 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i)*ivstr1) = a(ia1 +(iin1i)*
     &                       imstr1)* vi(ivi1+(iin1i)*ivstr1) + a(ia2 +(
     &                       iin1i)*imstr1)* vi(ivi2+(iin1i)*ivstr1) + 
     &                       a(ia3 +(iin1i)*imstr1)* vi(ivi3+(iin1i)*
     &                       ivstr1) + a(ia4 +(iin1i)*imstr1)* vi(ivi4+(
     &                       iin1i)*ivstr1)
 8705                     continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
                          do 8707 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i) ) = vo(ivo0 +(iin1i) ) + a(
     &                       ia1 +(iin1i) )* vi(ivi1+(iin1i) ) + a(ia2 +
     &                       (iin1i) )* vi(ivi2+(iin1i) ) + a(ia3 +(
     &                       iin1i) )* vi(ivi3+(iin1i) ) + a(ia4 +(
     &                       iin1i) )* vi(ivi4+(iin1i) )
 8707                     continue
                        else
                          do 8709 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i)*ivstr1) = vo(ivo0 +(iin1i)*
     &                       ivstr1) + a(ia1 +(iin1i)*imstr1)* vi(ivi1+(
     &                       iin1i)*ivstr1) + a(ia2 +(iin1i)*imstr1)* 
     &                       vi(ivi2+(iin1i)*ivstr1) + a(ia3 +(iin1i)*
     &                       imstr1)* vi(ivi3+(iin1i)*ivstr1) + a(ia4 +(
     &                       iin1i)*imstr1)* vi(ivi4+(iin1i)*ivstr1)
 8709                     continue
                        endif
                      endif
                    go to 700
  705       continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
                          do 8711 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i) ) = a(ia1 +(iin1i) )* vi(
     &                       ivi1+(iin1i) ) + a(ia2 +(iin1i) )* vi(ivi2+
     &                       (iin1i) ) + a(ia3 +(iin1i) )* vi(ivi3+(
     &                       iin1i) ) + a(ia4 +(iin1i) )* vi(ivi4+(
     &                       iin1i) ) + a(ia5 +(iin1i) )* vi(ivi5+(
     &                       iin1i) )
 8711                     continue
                        else
                          do 8713 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i)*ivstr1) = a(ia1 +(iin1i)*
     &                       imstr1)* vi(ivi1+(iin1i)*ivstr1) + a(ia2 +(
     &                       iin1i)*imstr1)* vi(ivi2+(iin1i)*ivstr1) + 
     &                       a(ia3 +(iin1i)*imstr1)* vi(ivi3+(iin1i)*
     &                       ivstr1) + a(ia4 +(iin1i)*imstr1)* vi(ivi4+(
     &                       iin1i)*ivstr1) + a(ia5 +(iin1i)*imstr1)* 
     &                       vi(ivi5+(iin1i)*ivstr1)
 8713                     continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
                          do 8715 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i) ) = vo(ivo0 +(iin1i) ) + a(
     &                       ia1 +(iin1i) )* vi(ivi1+(iin1i) ) + a(ia2 +
     &                       (iin1i) )* vi(ivi2+(iin1i) ) + a(ia3 +(
     &                       iin1i) )* vi(ivi3+(iin1i) ) + a(ia4 +(
     &                       iin1i) )* vi(ivi4+(iin1i) ) + a(ia5 +(
     &                       iin1i) )* vi(ivi5+(iin1i) )
 8715                     continue
                        else
                          do 8717 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i)*ivstr1) = vo(ivo0 +(iin1i)*
     &                       ivstr1) + a(ia1 +(iin1i)*imstr1)* vi(ivi1+(
     &                       iin1i)*ivstr1) + a(ia2 +(iin1i)*imstr1)* 
     &                       vi(ivi2+(iin1i)*ivstr1) + a(ia3 +(iin1i)*
     &                       imstr1)* vi(ivi3+(iin1i)*ivstr1) + a(ia4 +(
     &                       iin1i)*imstr1)* vi(ivi4+(iin1i)*ivstr1) + 
     &                       a(ia5 +(iin1i)*imstr1)* vi(ivi5+(iin1i)*
     &                       ivstr1)
 8717                     continue
                        endif
                      endif
                    go to 700
  700             continue
 8677           continue
 8675         continue
 8673       continue
          else
            do 8719 isten = 1, nsten
              ivip = iwk(ivipt-1+isten)
              iap = iwk(iapt -1+isten)
              do 8721 ibc = 0, nb-1
                if (isten.eq.1 .and. ibc.eq.0) then
                   do 8723 iin2 = 0, nin2-1
       do 8725 ibr = 0, nb-1
      call shad0 (nin1, vo(ivop+(ibr)*ivstrr +(0)*ivstr1+(iin2)*ivstr2),
     &    ivstr1, a(iap+(ibr)*imstrr+(ibc)*imstrc +(0)*imstr1+(iin2)*
     &   imstr2), imstr1, vi(ivip+(ibc)*ivstrr +(0)*ivstr1+(iin2)*
     &   ivstr2), ivstr1)
 8725 continue
 8723 continue
**#               DOALL([VOPLANE(ibr,    iin1,iin2) =
**#  &                    APLANE(ibr,ibc,iin1,iin2)*
**#  &                   VIPLANE(    ibc,iin1,iin2)],
**#                     iin1,0,nin1-1,iin2,0,nin2-1,ibr,0,nb-1)
                else
                   do 8727 iin2 = 0, nin2-1
       do 8729 ibr = 0, nb-1
      call shad (nin1, vo(ivop+(ibr)*ivstrr +(0)*ivstr1+(iin2)*ivstr2), 
     &   ivstr1, a(iap+(ibr)*imstrr+(ibc)*imstrc +(0)*imstr1+(iin2)*
     &   imstr2), imstr1, vi(ivip+(ibc)*ivstrr +(0)*ivstr1+(iin2)*
     &   ivstr2), ivstr1)
 8729 continue
 8727 continue
**#               DOALL([VOPLANE(ibr,    iin1,iin2) =
**#  &                   VOPLANE(ibr,    iin1,iin2) +
**#  &                    APLANE(ibr,ibc,iin1,iin2)*
**#  &                   VIPLANE(    ibc,iin1,iin2)],
**#                       iin1,0,nin1-1,iin2,0,nin2-1,ibr,0,nb-1)
                endif
 8721         continue
 8719       continue
          endif
        else
          if (nb .eq. 1) then
            do 8731 iin1o = 0, nin1-1, 100
              do 8733 iin2 = 0, nin2-1
                do 8735 isteno = 1, nsten, 5
                    isten = isteno + 0
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap = iwk(iapt -1+isten)
                      ivi1 = ivip+(0)*ivstrr +(iin1o)*ivstr1+(iin2)*
     &                   ivstr2
                      ia1 = iap+(0)*imstrr+(0)*imstrc +(iin1o)*imstr1+(
     &                   iin2)*imstr2
                    endif
                    isten = isteno + 1
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap = iwk(iapt -1+isten)
                      ivi2 = ivip+(0)*ivstrr +(iin1o)*ivstr1+(iin2)*
     &                   ivstr2
                      ia2 = iap+(0)*imstrr+(0)*imstrc +(iin1o)*imstr1+(
     &                   iin2)*imstr2
                    endif
                    isten = isteno + 2
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap = iwk(iapt -1+isten)
                      ivi3 = ivip+(0)*ivstrr +(iin1o)*ivstr1+(iin2)*
     &                   ivstr2
                      ia3 = iap+(0)*imstrr+(0)*imstrc +(iin1o)*imstr1+(
     &                   iin2)*imstr2
                    endif
                    isten = isteno + 3
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap = iwk(iapt -1+isten)
                      ivi4 = ivip+(0)*ivstrr +(iin1o)*ivstr1+(iin2)*
     &                   ivstr2
                      ia4 = iap+(0)*imstrr+(0)*imstrc +(iin1o)*imstr1+(
     &                   iin2)*imstr2
                    endif
                    isten = isteno + 4
                    if (isten .le. nsten) then
                      ivip = iwk(ivipt-1+isten)
                      iap = iwk(iapt -1+isten)
                      ivi5 = ivip+(0)*ivstrr +(iin1o)*ivstr1+(iin2)*
     &                   ivstr2
                      ia5 = iap+(0)*imstrr+(0)*imstrc +(iin1o)*imstr1+(
     &                   iin2)*imstr2
                    endif
                  ivo0 = ivop+(0)*ivstrr +(iin1o)*ivstr1+(iin2)*ivstr2
                  nsteni = min(5,nsten-isteno+1)
                  nin1i = min(100,nin1-iin1o)
                  go to (801,802,803,804,805), nsteni
  801       continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
                          do 8737 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i) ) = a(ia1 +(iin1i) )* vi(
     &                       ivi1+(iin1i) )
 8737                     continue
                        else
                          do 8739 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i)*ivstr1) = a(ia1 +(iin1i)*
     &                       imstr1)* vi(ivi1+(iin1i)*ivstr1)
 8739                     continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
                          do 8741 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i) ) = vo(ivo0 +(iin1i) ) + a(
     &                       ia1 +(iin1i) )* vi(ivi1+(iin1i) )
 8741                     continue
                        else
                          do 8743 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i)*ivstr1) = vo(ivo0 +(iin1i)*
     &                       ivstr1) + a(ia1 +(iin1i)*imstr1)* vi(ivi1+(
     &                       iin1i)*ivstr1)
 8743                     continue
                        endif
                      endif
                    go to 800
  802       continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
                          do 8745 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i) ) = a(ia1 +(iin1i) )* vi(
     &                       ivi1+(iin1i) ) + a(ia2 +(iin1i) )* vi(ivi2+
     &                       (iin1i) )
 8745                     continue
                        else
                          do 8747 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i)*ivstr1) = a(ia1 +(iin1i)*
     &                       imstr1)* vi(ivi1+(iin1i)*ivstr1) + a(ia2 +(
     &                       iin1i)*imstr1)* vi(ivi2+(iin1i)*ivstr1)
 8747                     continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
                          do 8749 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i) ) = vo(ivo0 +(iin1i) ) + a(
     &                       ia1 +(iin1i) )* vi(ivi1+(iin1i) ) + a(ia2 +
     &                       (iin1i) )* vi(ivi2+(iin1i) )
 8749                     continue
                        else
                          do 8751 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i)*ivstr1) = vo(ivo0 +(iin1i)*
     &                       ivstr1) + a(ia1 +(iin1i)*imstr1)* vi(ivi1+(
     &                       iin1i)*ivstr1) + a(ia2 +(iin1i)*imstr1)* 
     &                       vi(ivi2+(iin1i)*ivstr1)
 8751                     continue
                        endif
                      endif
                    go to 800
  803       continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
                          do 8753 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i) ) = a(ia1 +(iin1i) )* vi(
     &                       ivi1+(iin1i) ) + a(ia2 +(iin1i) )* vi(ivi2+
     &                       (iin1i) ) + a(ia3 +(iin1i) )* vi(ivi3+(
     &                       iin1i) )
 8753                     continue
                        else
                          do 8755 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i)*ivstr1) = a(ia1 +(iin1i)*
     &                       imstr1)* vi(ivi1+(iin1i)*ivstr1) + a(ia2 +(
     &                       iin1i)*imstr1)* vi(ivi2+(iin1i)*ivstr1) + 
     &                       a(ia3 +(iin1i)*imstr1)* vi(ivi3+(iin1i)*
     &                       ivstr1)
 8755                     continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
                          do 8757 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i) ) = vo(ivo0 +(iin1i) ) + a(
     &                       ia1 +(iin1i) )* vi(ivi1+(iin1i) ) + a(ia2 +
     &                       (iin1i) )* vi(ivi2+(iin1i) ) + a(ia3 +(
     &                       iin1i) )* vi(ivi3+(iin1i) )
 8757                     continue
                        else
                          do 8759 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i)*ivstr1) = vo(ivo0 +(iin1i)*
     &                       ivstr1) + a(ia1 +(iin1i)*imstr1)* vi(ivi1+(
     &                       iin1i)*ivstr1) + a(ia2 +(iin1i)*imstr1)* 
     &                       vi(ivi2+(iin1i)*ivstr1) + a(ia3 +(iin1i)*
     &                       imstr1)* vi(ivi3+(iin1i)*ivstr1)
 8759                     continue
                        endif
                      endif
                    go to 800
  804       continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
                          do 8761 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i) ) = a(ia1 +(iin1i) )* vi(
     &                       ivi1+(iin1i) ) + a(ia2 +(iin1i) )* vi(ivi2+
     &                       (iin1i) ) + a(ia3 +(iin1i) )* vi(ivi3+(
     &                       iin1i) ) + a(ia4 +(iin1i) )* vi(ivi4+(
     &                       iin1i) )
 8761                     continue
                        else
                          do 8763 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i)*ivstr1) = a(ia1 +(iin1i)*
     &                       imstr1)* vi(ivi1+(iin1i)*ivstr1) + a(ia2 +(
     &                       iin1i)*imstr1)* vi(ivi2+(iin1i)*ivstr1) + 
     &                       a(ia3 +(iin1i)*imstr1)* vi(ivi3+(iin1i)*
     &                       ivstr1) + a(ia4 +(iin1i)*imstr1)* vi(ivi4+(
     &                       iin1i)*ivstr1)
 8763                     continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
                          do 8765 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i) ) = vo(ivo0 +(iin1i) ) + a(
     &                       ia1 +(iin1i) )* vi(ivi1+(iin1i) ) + a(ia2 +
     &                       (iin1i) )* vi(ivi2+(iin1i) ) + a(ia3 +(
     &                       iin1i) )* vi(ivi3+(iin1i) ) + a(ia4 +(
     &                       iin1i) )* vi(ivi4+(iin1i) )
 8765                     continue
                        else
                          do 8767 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i)*ivstr1) = vo(ivo0 +(iin1i)*
     &                       ivstr1) + a(ia1 +(iin1i)*imstr1)* vi(ivi1+(
     &                       iin1i)*ivstr1) + a(ia2 +(iin1i)*imstr1)* 
     &                       vi(ivi2+(iin1i)*ivstr1) + a(ia3 +(iin1i)*
     &                       imstr1)* vi(ivi3+(iin1i)*ivstr1) + a(ia4 +(
     &                       iin1i)*imstr1)* vi(ivi4+(iin1i)*ivstr1)
 8767                     continue
                        endif
                      endif
                    go to 800
  805       continue
                      if (isteno .eq. 1) then
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
                          do 8769 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i) ) = a(ia1 +(iin1i) )* vi(
     &                       ivi1+(iin1i) ) + a(ia2 +(iin1i) )* vi(ivi2+
     &                       (iin1i) ) + a(ia3 +(iin1i) )* vi(ivi3+(
     &                       iin1i) ) + a(ia4 +(iin1i) )* vi(ivi4+(
     &                       iin1i) ) + a(ia5 +(iin1i) )* vi(ivi5+(
     &                       iin1i) )
 8769                     continue
                        else
                          do 8771 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i)*ivstr1) = a(ia1 +(iin1i)*
     &                       imstr1)* vi(ivi1+(iin1i)*ivstr1) + a(ia2 +(
     &                       iin1i)*imstr1)* vi(ivi2+(iin1i)*ivstr1) + 
     &                       a(ia3 +(iin1i)*imstr1)* vi(ivi3+(iin1i)*
     &                       ivstr1) + a(ia4 +(iin1i)*imstr1)* vi(ivi4+(
     &                       iin1i)*ivstr1) + a(ia5 +(iin1i)*imstr1)* 
     &                       vi(ivi5+(iin1i)*ivstr1)
 8771                     continue
                        endif
                      else
                        if (ivstr1.eq.1 .and. imstr1.eq.1) then
                          do 8773 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i) ) = vo(ivo0 +(iin1i) ) + a(
     &                       ia1 +(iin1i) )* vi(ivi1+(iin1i) ) + a(ia2 +
     &                       (iin1i) )* vi(ivi2+(iin1i) ) + a(ia3 +(
     &                       iin1i) )* vi(ivi3+(iin1i) ) + a(ia4 +(
     &                       iin1i) )* vi(ivi4+(iin1i) ) + a(ia5 +(
     &                       iin1i) )* vi(ivi5+(iin1i) )
 8773                     continue
                        else
                          do 8775 iin1i = 0, nin1i-1
                          vo(ivo0 +(iin1i)*ivstr1) = vo(ivo0 +(iin1i)*
     &                       ivstr1) + a(ia1 +(iin1i)*imstr1)* vi(ivi1+(
     &                       iin1i)*ivstr1) + a(ia2 +(iin1i)*imstr1)* 
     &                       vi(ivi2+(iin1i)*ivstr1) + a(ia3 +(iin1i)*
     &                       imstr1)* vi(ivi3+(iin1i)*ivstr1) + a(ia4 +(
     &                       iin1i)*imstr1)* vi(ivi4+(iin1i)*ivstr1) + 
     &                       a(ia5 +(iin1i)*imstr1)* vi(ivi5+(iin1i)*
     &                       ivstr1)
 8775                     continue
                        endif
                      endif
                    go to 800
  800             continue
 8735           continue
 8733         continue
 8731       continue
          else
            do 8777 isten = 1, nsten
              ivip = iwk(ivipt-1+isten)
              iap = iwk(iapt -1+isten)
              do 8779 ibc = 0, nb-1
                if (isten.eq.1 .and. ibc.eq.0) then
                   do 8781 iin2 = 0, nin2-1
       do 8783 ibr = 0, nb-1
      call shad0 (nin1, vo(ivop+(ibr)*ivstrr +(0)*ivstr1+(iin2)*ivstr2),
     &    ivstr1, a(iap+(ibc)*imstrr+(ibr)*imstrc +(0)*imstr1+(iin2)*
     &   imstr2), imstr1, vi(ivip+(ibc)*ivstrr +(0)*ivstr1+(iin2)*
     &   ivstr2), ivstr1)
 8783 continue
 8781 continue
                else
                   do 8785 iin2 = 0, nin2-1
       do 8787 ibr = 0, nb-1
      call shad (nin1, vo(ivop+(ibr)*ivstrr +(0)*ivstr1+(iin2)*ivstr2), 
     &   ivstr1, a(iap+(ibc)*imstrr+(ibr)*imstrc +(0)*imstr1+(iin2)*
     &   imstr2), imstr1, vi(ivip+(ibc)*ivstrr +(0)*ivstr1+(iin2)*
     &   ivstr2), ivstr1)
 8787 continue
 8785 continue
                endif
 8779         continue
 8777       continue
          endif
        endif
 8667 continue
*
      call xifre (iapt ,nsten, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call xifre (ivipt,nsten, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
*
  101 continue
*
*-----------------------------------------------------------------------
*  ---perform receives--------------------------------------------------
*
         iparm(26) = msg0
         do 8789 iprc = 0, ncpprd-1
           if (iwk(irbufp+iprc) .eq. (iparm(6))) go to 8789
           if(urflag.eq.1)then
             nbors = 1
             do 8791 iaxis = 1,ndim
               rng(iaxis) = ja(iaxis + ndim*(nsten+4)) - ja(iaxis + 
     &            ndim*(nsten+3)) + 1
               nbors = nbors*rng(iaxis)
               imi3(iaxis) = -ja(iaxis + ndim*(nsten+3))
 8791        continue
             call ii2mi(iprcmi,iprc,ndim,ncp)
              do 8793 iaxis = 1, ndim
      iprcmi(iaxis) = iprcmi(iaxis) + ncpmin(iaxis)
 8793 continue
              do 8795 iaxis = 1, ndim
      imi2(iaxis) = imi3(iaxis) + iprcmi(iaxis)
 8795 continue
             ip_ns = imi2i(imi2,ndim,rng)
             ipsend = ja(1 + ip_ns + (nsten+5)*ndim)
              do 8797 iaxis = 1, ndim
      imi2(iaxis) = imi3(iaxis) - iprcmi(iaxis)
 8797 continue
             ip_nr = imi2i(imi2,ndim,rng)
             iprecv = ja(1 + ip_nr + (nsten+5)*ndim)
           else
             call ii2mi (iprcmi,iprc,ndim,ncp)
              do 8799 iaxis = 1, ndim
      iprcmi(iaxis) = iprcmi(iaxis) + ncpmin(iaxis)
 8799 continue
             call ii2mi (imi2,igrid(ipme,ndim,ja(1+ndim*(nsten+1))),
     &          ndim,ja(1+ndim*(nsten+1)))
              do 8801 iaxis = 1, ndim
      imi3(iaxis) = imodf(imi2(iaxis)+iprcmi(iaxis), ja(iaxis+ndim*(
     &   nsten+1)))
 8801 continue
             ipsend = iproc(imi2i(imi3,ndim,ja(1+ndim*(nsten+1))),ndim,
     &          ja(1+ndim*(nsten+1)))
              do 8803 iaxis = 1, ndim
      imi3(iaxis) = imodf(imi2(iaxis)-iprcmi(iaxis), ja(iaxis+ndim*(
     &   nsten+1)))
 8803 continue
             iprecv = iproc(imi2i(imi3,ndim,ja(1+ndim*(nsten+1))),ndim,
     &          ja(1+ndim*(nsten+1)))
             if (isgnt .eq. +1) then
                do 8805 iaxis = 1, ndim
      imi2(iaxis) = iprcmi(iaxis) - ncpmin(iaxis)
 8805 continue
             else
                do 8807 iaxis = 1, ndim
      imi2(iaxis) = ncpmax(iaxis) - iprcmi(iaxis)
 8807 continue
             endif
             ibuf = imi2i (imi2,ndim,ncp)
           endif
           if(iprecv.ne.-1)then
      call crecv ((iparm(26)), fwk(iwk(irbufp+ibuf)), 4*(iwk(ibufl+ibuf)
     &   ))
           endif
           iparm(26) = iparm(26) + 1
 8789    continue
*
*-----------------------------------------------------------------------
*  ---set to zero the in-processor off-intersection part----------------
*
*
       do 8809 iaxis = 1, ndim
      imind(iaxis) = 0
 8809 continue
       do 8811 iaxis = 1, ndim
      imaxd(iaxis) = ja(iaxis+ndim*nsten) - 1
 8811 continue
*
      do 8813 iaxis = 1, ndim
        do 8815 ilohi = 1, 2
          if (ilohi .eq. 1) then
            if (imind(iaxis) .ge. imini(iaxis)) go to 8815
             do 8817 jaxis = 1, ndim
      imi (jaxis) = imind(jaxis)
 8817 continue
             do 8819 jaxis = 1, ndim
      inelt(jaxis) = max(0, imaxd(jaxis) -imi(jaxis)+1)
 8819 continue
                   inelt(iaxis) = max(0,min(imaxd(iaxis),imini(iaxis)-1)
     &                -imi(iaxis)+1)
          else
            if (imaxd(iaxis) .le. imaxi(iaxis)) go to 8815
             do 8821 jaxis = 1, ndim
      imi (jaxis) = imind(jaxis)
 8821 continue
                   imi (iaxis) = max(imind(iaxis),imaxi(iaxis)+1)
             do 8823 jaxis = 1, ndim
      inelt(jaxis) = max(0,imaxd(jaxis) -imi(jaxis)+1)
 8823 continue
          endif
          nelt = 1 
          do 8825 jaxis = 1, ndim
      nelt = nelt * inelt(jaxis)
 8825 continue
          if (nelt .eq. 0) go to 8815
          call xplane ( ja(1+ndim*nsten), inelt, ia, iparm, iaxin1, 
     &       iaxin2, istrd1, istrd2, nin1, nin2, ivstrr,ivstr1,ivstr2, 
     &       imstrr,imstrc,imstr1,imstr2)
          do 8827 ielt = 0, nelt-1, nin1*nin2
            ivop = ivaxgr(1,1+islice(ielt,ndim,ja(1+ndim*nsten),inelt,
     &         imi),ia(5),nb,ns,nsten)
             do 8829 iin1 = 0, nin1-1
       do 8831 iin2 = 0, nin2-1
       do 8833 ibr = 0, nb-1
      vo(ivop+(ibr)*ivstrr +(iin1)*ivstr1+(iin2)*ivstr2) = 0e0
 8833 continue
 8831 continue
 8829 continue
 8827     continue
 8815   continue
        imind(iaxis) = max(imind(iaxis),imini(iaxis))
        imaxd(iaxis) = min(imaxd(iaxis),imaxi(iaxis))
 8813 continue
*
*-----------------------------------------------------------------------
*  ---Handle parts off the inscribed rect but using on-proc data--------
*
*
      do 8835 isten = 1, nsten
        do 8837 islno = 0, 2**ndim-1
          if(urflag.eq.1)then
            call xslur ( ia, ja, isten, islno, ijob, imins, imind, 
     &         inelt, nelt , ipme, ipsend, iprecv, iprcmi )
          else
            call xslice ( ia, ja, isten, islno, ijob, imins, imind, 
     &         inelt, nelt , ipme, ipsend, iprecv, iprcmi )
          endif
          if (nelt .eq. 0) go to 8837
*
          if (iprecv .eq. ipme) then
*
           do 8839 iaxis = 1, ndim
      imaxd(iaxis) = imind(iaxis)+inelt(iaxis)-1
 8839 continue
*
          do 8841 iaxis = 1, ndim
            do 8843 ilohi = 1, 2
              if (ilohi .eq. 1) then
                if (imind(iaxis) .ge. imini(iaxis)) go to 8843
                 do 8845 jaxis = 1, ndim
      imi (jaxis) = imind(jaxis)
 8845 continue
                 do 8847 jaxis = 1, ndim
      imi2 (jaxis) = imins(jaxis)
 8847 continue
                 do 8849 jaxis = 1, ndim
      inelt(jaxis) = max(0, imaxd(jaxis) -imi(jaxis)+1)
 8849 continue
                       inelt(iaxis) = max(0,min(imaxd(iaxis),imini(
     &                    iaxis)-1)-imi(iaxis)+1)
              else
                if (imaxd(iaxis) .le. imaxi(iaxis)) go to 8843
                 do 8851 jaxis = 1, ndim
      imi (jaxis) = imind(jaxis)
 8851 continue
                       imi (iaxis) = max(imind(iaxis),imaxi(iaxis)+1)
                 do 8853 jaxis = 1, ndim
      imi2 (jaxis) = imins(jaxis)
 8853 continue
                       imi2 (iaxis) = imins(iaxis)-(imind(iaxis)-imi(
     &                    iaxis))
                 do 8855 jaxis = 1, ndim
      inelt(jaxis) = max(0,imaxd(jaxis) -imi(jaxis)+1)
 8855 continue
              endif
              nelt = 1 
              do 8857 jaxis = 1, ndim
      nelt = nelt * inelt(jaxis)
 8857 continue
              if (nelt .eq. 0) go to 8843
              call xplane ( ja(1+ndim*nsten), inelt, ia, iparm, iaxin1, 
     &           iaxin2, istrd1, istrd2, nin1, nin2, ivstrr,ivstr1,
     &           ivstr2, imstrr,imstrc,imstr1,imstr2)
              do 8859 ielt = 0, nelt-1, nin1*nin2
                ivop = ivaxgr(1,1+islice(ielt,ndim,ja(1+ndim*nsten),
     &             inelt,imi),ia(5),nb,ns,nsten)
                ivip = ivaxgr(1,1+islice(ielt,ndim,ja(1+ndim*nsten),
     &             inelt,imi2),ia(5),nb,ns,nsten)
                if (ijob .eq. 3) then
                  iap = imaxgr(1,1,isten,1+islice(ielt,ndim,ja(1+ndim*
     &               nsten),inelt,imi),ia(4),nb,ns,nsten)
                else
                  iap = imaxgr(1,1,isten,1+islice(ielt,ndim,ja(1+ndim*
     &               nsten),inelt,imi2),ia(4),nb,ns,nsten)
                endif
                if (ijob .eq. 3) then
                 do 8861 iin2 = 0, nin2-1
       do 8863 ibr = 0, nb-1
       do 8865 ibc = 0, nb-1
      call shad (nin1, vo(ivop+(ibr)*ivstrr +(0)*ivstr1+(iin2)*ivstr2), 
     &   ivstr1, a(iap+(ibr)*imstrr+(ibc)*imstrc +(0)*imstr1+(iin2)*
     &   imstr2), imstr1, vi(ivip+(ibc)*ivstrr +(0)*ivstr1+(iin2)*
     &   ivstr2), ivstr1)
 8865 continue
 8863 continue
 8861 continue
                else
                 do 8867 iin2 = 0, nin2-1
       do 8869 ibr = 0, nb-1
       do 8871 ibc = 0, nb-1
      call shad (nin1, vo(ivop+(ibr)*ivstrr +(0)*ivstr1+(iin2)*ivstr2), 
     &   ivstr1, a(iap+(ibc)*imstrr+(ibr)*imstrc +(0)*imstr1+(iin2)*
     &   imstr2), imstr1, vi(ivip+(ibc)*ivstrr +(0)*ivstr1+(iin2)*
     &   ivstr2), ivstr1)
 8871 continue
 8869 continue
 8867 continue
                endif
 8859         continue
 8843       continue
            imins(iaxis) = imins(iaxis) - (imind(iaxis)-max(imind(iaxis)
     &         ,imini(iaxis)))
            imind(iaxis) = max(imind(iaxis),imini(iaxis))
            imaxd(iaxis) = min(imaxd(iaxis),imaxi(iaxis))
 8841     continue
*
*-----------------------------------------------------------------------
*  ---Handle parts off-proc---------------------------------------------
*
          else
*
*
             do 8873 iaxis = 1, ndim
      imi2(iaxis) = iprcmi(iaxis) - ncpmin(iaxis)
 8873 continue
            iprc = imi2i (imi2,ndim,ncp)
            call ii2mi (iprcmi,iprc,ndim,ncp)
             do 8875 iaxis = 1, ndim
      iprcmi(iaxis) = iprcmi(iaxis) + ncpmin(iaxis)
 8875 continue
            ibuf = iprc
             if (isgnt .eq. +1) then
                do 8877 iaxis = 1, ndim
      imi2(iaxis) = iprcmi(iaxis) - ncpmin(iaxis)
 8877 continue
             else
                do 8879 iaxis = 1, ndim
      imi2(iaxis) = ncpmax(iaxis) - iprcmi(iaxis)
 8879 continue
             endif
             ibuf = imi2i (imi2,ndim,ncp)
*
            call xplane ( ja(1+ndim*nsten), inelt, ia, iparm, iaxin1, 
     &         iaxin2, istrd1, istrd2, nin1, nin2, ivstrr,ivstr1,ivstr2,
     &          imstrr,imstrc,imstr1,imstr2)
*
           inelrp = 1
           do 8881 iaxis = 1, ndim
             iminr(iaxis) = max(0 , iminc(iaxis)+ja(iaxis+ndim*nsten)*
     &          iprcmi(iaxis))
             imaxr(iaxis) = min(ja(iaxis+ndim*nsten)-1, imaxc(iaxis)+ja(
     &          iaxis+ndim*nsten)*iprcmi(iaxis))
             inelr(iaxis) = max(0,imaxr(iaxis)-iminr(iaxis)+1)
             inelrp = inelrp * inelr(iaxis)
 8881      continue
*
            iprdax = 1
            do 8883 iaxis = 1, ndim
              if (iaxis .eq. iaxin1) ninb1 = iprdax
              if (iaxis .eq. iaxin2) ninb2 = iprdax
              iprdax = iprdax * inelr(iaxis)
 8883       continue
*
            ibstrr = 1
            ibstrc = nb*inelrp
            ibstr1 = nb*ninb1
            ibstr2 = nb*ninb2
*
             do 8885 iaxis = 1, ndim
      imi(iaxis) = imins(iaxis) - iminr(iaxis)
 8885 continue
*
            if (ijob .eq. 3) then
              do 8887 ielt = 0, nelt-1, nin1*nin2
                ivop = ivaxgr(1,1+islice(ielt,ndim,ja(1+ndim*nsten),
     &             inelt,imind),ia(5),nb,ns,nsten)
                ivbufp = iwk(irbufp+ibuf) + nb*islice(ielt,ndim,inelr, 
     &             inelt,imi )
                iap = imaxgr(1,1,isten,1+islice(ielt,ndim,ja(1+ndim*
     &             nsten),inelt,imind),ia(4),nb,ns,nsten)
**#             DOALL([VOPLANE(ibr,    iin1,iin2) = VOPLANE(ibr,iin1,iin2) +
**#  &                 APLANE(ibr,ibc,iin1,iin2)*VBUFPLANE(ibc,iin1,iin2)],
**#                   iin2,0,nin2-1,iin1,0,nin1-1,ibr,0,nb-1,ibc,0,nb-1)
                 do 8889 iin2 = 0, nin2-1
       do 8891 ibr = 0, nb-1
       do 8893 ibc = 0, nb-1
      call shad (nin1, vo(ivop+(ibr)*ivstrr +(0)*ivstr1+(iin2)*ivstr2), 
     &   ivstr1, a(iap+(ibr)*imstrr+(ibc)*imstrc +(0)*imstr1+(iin2)*
     &   imstr2), imstr1, fwk(ivbufp+(ibc)*ibstrr +(0)*ibstr1+(iin2)*
     &   ibstr2), ibstr1)
 8893 continue
 8891 continue
 8889 continue
 8887         continue
            else
              do 8895 ielt = 0, nelt-1, nin1*nin2
                ivop = ivaxgr(1,1+islice(ielt,ndim,ja(1+ndim*nsten),
     &             inelt,imind),ia(5),nb,ns,nsten)
                ivbufp = iwk(irbufp+ibuf) + nb*islice(ielt,ndim,inelr, 
     &             inelt,imi )
                imbufp = iwk(itbufp+ibuf) + nb*islice(ielt,ndim,inelr, 
     &             inelt,imi ) + nb*inelrp*nb*(isten-1)
                 do 8897 iin2 = 0, nin2-1
       do 8899 ibr = 0, nb-1
       do 8901 ibc = 0, nb-1
      call shad (nin1, vo(ivop+(ibr)*ivstrr +(0)*ivstr1+(iin2)*ivstr2), 
     &   ivstr1, fwk(imbufp+(ibc)*ibstrr+(ibr)*ibstrc +(0)*ibstr1+(iin2)
     &   *ibstr2), ibstr1, fwk(ivbufp+(ibc)*ibstrr +(0)*ibstr1+(iin2)*
     &   ibstr2), ibstr1)
 8901 continue
 8899 continue
 8897 continue
 8895         continue
            endif
          endif
*
 8837   continue
 8835 continue
*
*-----------------------------------------------------------------------
*  ---Flop count--------------------------------------------------------
*
       fparm(13) = fparm(13) + (iparm(3) *(2.* nb*nsten-1))
*
*-----------------------------------------------------------------------
*  ---done--------------------------------------------------------------
*-----------------------------------------------------------------------
*
  900 continue
*
      if (iparm(11).eq.(iparm(6))) go to 910
*
      if (ijob.eq.-1 .or. ier.lt.0) then
        if (irbufp .ne. (iparm(6))) then
          do 8903 iaxis = 1, ndim
            iminc(iaxis) = 0 +(isgnt)*ja(iaxis+ndim*(1 -1))
             do 8905 isten = 2, nsten
      iminc(iaxis) = min(iminc(iaxis), 0 +(isgnt)*ja(iaxis+ndim*(isten-
     &   1)))
 8905 continue
            imaxc(iaxis) = ja(iaxis+ndim*nsten)-1+(isgnt)*ja(iaxis+ndim*
     &         (1 -1))
             do 8907 isten = 2, nsten
      imaxc(iaxis) = max(imaxc(iaxis), ja(iaxis+ndim*nsten)-1 +(isgnt)*
     &   ja(iaxis+ndim*(isten-1)))
 8907 continue
 8903     continue
          ncpprd = 1
          do 8909 iaxis = 1, ndim
            ncpmin(iaxis) = ifloor(iminc(iaxis),ja(iaxis+ndim*nsten))
            ncpmax(iaxis) = ifloor(imaxc(iaxis),ja(iaxis+ndim*nsten))
            ncp (iaxis) = max(0,ncpmax(iaxis)-ncpmin(iaxis)+1)
            ncpprd = ncpprd * ncp(iaxis)
 8909     continue
          do 8911 iprc = (ncpprd-1), 0, -1
           if(urflag.eq.1)then
             nbors = 1
             do 8913 iaxis = 1,ndim
               rng(iaxis) = ja(iaxis + ndim*(nsten+4)) - ja(iaxis + 
     &            ndim*(nsten+3)) + 1
               nbors = nbors*rng(iaxis)
               imi3(iaxis) = -ja(iaxis + ndim*(nsten+3))
 8913        continue
             call ii2mi(iprcmi,iprc,ndim,ncp)
              do 8915 iaxis = 1, ndim
      iprcmi(iaxis) = iprcmi(iaxis) + ncpmin(iaxis)
 8915 continue
              do 8917 iaxis = 1, ndim
      imi2(iaxis) = imi3(iaxis) + iprcmi(iaxis)
 8917 continue
             ip_ns = imi2i(imi2,ndim,rng)
             ipsend = ja(1 + ip_ns + (nsten+5)*ndim)
              do 8919 iaxis = 1, ndim
      imi2(iaxis) = imi3(iaxis) - iprcmi(iaxis)
 8919 continue
             ip_nr = imi2i(imi2,ndim,rng)
             iprecv = ja(1 + ip_nr + (nsten+5)*ndim)
           else
             call ii2mi (iprcmi,iprc,ndim,ncp)
              do 8921 iaxis = 1, ndim
      iprcmi(iaxis) = iprcmi(iaxis) + ncpmin(iaxis)
 8921 continue
             call ii2mi (imi2,igrid(ipme,ndim,ja(1+ndim*(nsten+1))),
     &          ndim,ja(1+ndim*(nsten+1)))
              do 8923 iaxis = 1, ndim
      imi3(iaxis) = imodf(imi2(iaxis)+iprcmi(iaxis), ja(iaxis+ndim*(
     &   nsten+1)))
 8923 continue
             ipsend = iproc(imi2i(imi3,ndim,ja(1+ndim*(nsten+1))),ndim,
     &          ja(1+ndim*(nsten+1)))
              do 8925 iaxis = 1, ndim
      imi3(iaxis) = imodf(imi2(iaxis)-iprcmi(iaxis), ja(iaxis+ndim*(
     &   nsten+1)))
 8925 continue
             iprecv = iproc(imi2i(imi3,ndim,ja(1+ndim*(nsten+1))),ndim,
     &          ja(1+ndim*(nsten+1)))
             if (isgnt .eq. +1) then
                do 8927 iaxis = 1, ndim
      imi2(iaxis) = iprcmi(iaxis) - ncpmin(iaxis)
 8927 continue
             else
                do 8929 iaxis = 1, ndim
      imi2(iaxis) = ncpmax(iaxis) - iprcmi(iaxis)
 8929 continue
             endif
             ibuf = imi2i (imi2,ndim,ncp)
           endif
      call sffre (iwk(irbufp+ibuf),iwk(ibufl+ibuf), iparm , fparm , iwk 
     &   , fwk , ier )
      call sffre (iwk(isbufp+ibuf),iwk(ibufl+ibuf), iparm , fparm , iwk 
     &   , fwk , ier )
 8911     continue
        endif
        if (itbufp .ne. (iparm(6))) then
          do 8931 iaxis = 1, ndim
            iminc(iaxis) = 0 +(-1)*ja(iaxis+ndim*(1 -1))
             do 8933 isten = 2, nsten
      iminc(iaxis) = min(iminc(iaxis), 0 +(-1)*ja(iaxis+ndim*(isten-1)))
 8933 continue
            imaxc(iaxis) = ja(iaxis+ndim*nsten)-1+(-1)*ja(iaxis+ndim*(1 
     &         -1))
             do 8935 isten = 2, nsten
      imaxc(iaxis) = max(imaxc(iaxis), ja(iaxis+ndim*nsten)-1 +(-1)*ja(
     &   iaxis+ndim*(isten-1)))
 8935 continue
 8931     continue
          ncpprd = 1
          do 8937 iaxis = 1, ndim
            ncpmin(iaxis) = ifloor(iminc(iaxis),ja(iaxis+ndim*nsten))
            ncpmax(iaxis) = ifloor(imaxc(iaxis),ja(iaxis+ndim*nsten))
            ncp (iaxis) = max(0,ncpmax(iaxis)-ncpmin(iaxis)+1)
            ncpprd = ncpprd * ncp(iaxis)
 8937     continue
          do 8939 iprc = (ncpprd-1), 0, -1
           if(urflag.eq.1)then
             nbors = 1
             do 8941 iaxis = 1,ndim
               rng(iaxis) = ja(iaxis + ndim*(nsten+4)) - ja(iaxis + 
     &            ndim*(nsten+3)) + 1
               nbors = nbors*rng(iaxis)
               imi3(iaxis) = -ja(iaxis + ndim*(nsten+3))
 8941        continue
             call ii2mi(iprcmi,iprc,ndim,ncp)
              do 8943 iaxis = 1, ndim
      iprcmi(iaxis) = iprcmi(iaxis) + ncpmin(iaxis)
 8943 continue
              do 8945 iaxis = 1, ndim
      imi2(iaxis) = imi3(iaxis) + iprcmi(iaxis)
 8945 continue
             ip_ns = imi2i(imi2,ndim,rng)
             ipsend = ja(1 + ip_ns + (nsten+5)*ndim)
              do 8947 iaxis = 1, ndim
      imi2(iaxis) = imi3(iaxis) - iprcmi(iaxis)
 8947 continue
             ip_nr = imi2i(imi2,ndim,rng)
             iprecv = ja(1 + ip_nr + (nsten+5)*ndim)
           else
             call ii2mi (iprcmi,iprc,ndim,ncp)
              do 8949 iaxis = 1, ndim
      iprcmi(iaxis) = iprcmi(iaxis) + ncpmin(iaxis)
 8949 continue
             call ii2mi (imi2,igrid(ipme,ndim,ja(1+ndim*(nsten+1))),
     &          ndim,ja(1+ndim*(nsten+1)))
              do 8951 iaxis = 1, ndim
      imi3(iaxis) = imodf(imi2(iaxis)+iprcmi(iaxis), ja(iaxis+ndim*(
     &   nsten+1)))
 8951 continue
             ipsend = iproc(imi2i(imi3,ndim,ja(1+ndim*(nsten+1))),ndim,
     &          ja(1+ndim*(nsten+1)))
              do 8953 iaxis = 1, ndim
      imi3(iaxis) = imodf(imi2(iaxis)-iprcmi(iaxis), ja(iaxis+ndim*(
     &   nsten+1)))
 8953 continue
             iprecv = iproc(imi2i(imi3,ndim,ja(1+ndim*(nsten+1))),ndim,
     &          ja(1+ndim*(nsten+1)))
             if (-1 .eq. +1) then
                do 8955 iaxis = 1, ndim
      imi2(iaxis) = iprcmi(iaxis) - ncpmin(iaxis)
 8955 continue
             else
                do 8957 iaxis = 1, ndim
      imi2(iaxis) = ncpmax(iaxis) - iprcmi(iaxis)
 8957 continue
             endif
             ibuf = imi2i (imi2,ndim,ncp)
           endif
      call sffre (iwk(itbufp+ibuf),iwk(itbufl+ibuf), iparm , fparm , 
     &   iwk , fwk , ier )
 8939     continue
        endif
      call xifre (ijaptr,ncpprd, iparm , fparm , iwk , fwk , ier )
      call xifre (ibufl,ncpprd, iparm , fparm , iwk , fwk , ier )
      call xifre (irbufp,ncpprd, iparm , fparm , iwk , fwk , ier )
      call xifre (isbufp,ncpprd, iparm , fparm , iwk , fwk , ier )
      call xifre (itbufl,ncpprd, iparm , fparm , iwk , fwk , ier )
      call xifre (itbufp,ncpprd, iparm , fparm , iwk , fwk , ier )
      call xifre (itab,7, iparm , fparm , iwk , fwk , ier )
        iparm(11) = (iparm(6))
      endif
*
  910 continue
      return
      end
*-----------------------------------------------------------------------
