************************************************************************
**  ^ROUTINE: WCGSR - Reverse communication level routine for conjugate
**                    gradient squared acceleration. (^)
**
      subroutine scgsr ( ijob , ireq , u , uexact , b , iva , ivql , 
     &   ivqr , iwk , fwk , iparm , fparm , ier )
**
**  ^DESCRIPTION:
**    Preconditioned biconjugate gradient squared routine.  This
**    transpose-free algorithm is based on the biconjugate gradient
**    algorithm but may converge up to twice as fast.
**
**  ^AUTHOR:   wdj@beta.lanl.gov
**
**  ^MODIFIED: wdj@lithos.c3.lanl.gov Thu May  6 13:12:38 MDT 1993
**
**  ^ARGUMENTS: see *Subroutine Arguments* below.
**
**  ^REQUIREMENTS:
**    Common Blocks: none
**    Subroutines:   see below.
**
**  ^SIDE_EFFECTS:
**    See the definition of METHRDCLS to see which parameters
**    are modified
**
**  ^ALGORITHM:
**    $$ \tilde r^{(0)} = Q_L r^{(0)} $$
**    $$ p^{(0)}        = f^{(0)} = r^{(0)}$$
**    $$ u^{(n+1)} = u^{(n)} +
**                   \lambda_n   Q_R Q_L (f^{(n)}+h^{(n+1)}) $$
**    $$ r^{(n+1)} = r^{(n)} -
**                   \lambda_n A Q_R Q_L (f^{(n)}+h^{(n+1)}) $$
**    $$ h^{(n+1)} = f^{(n)} - \lambda_n A Q_R Q_L p^{(n)}   $$
**    $$ f^{(n+1)} = r^{(n+1)} + \alpha_{n+1} h^{(n+1)}      $$
**    $$ p^{(n+1)} = f^{(n+1)} +
**                 \alpha_{n+1} (h^{(n+1)}+\alpha_{n+1}p^{(n)}) $$
**    $$ \lambda_n    ={(\tilde r^{(0)},Q_L           r^{(n )}) \over
**                      (\tilde r^{(0)},Q_L A Q_R Q_L p^{(n )})} $$
**    $$ \alpha_{n+1} ={(\tilde r^{(0)},Q_L           r^{(n+1)}) \over
**                      (\tilde r^{(0)},Q_L           r^{(n  )})}$$
**
**  ^REFERENCES:
**   Peter Sonneveld, ``CGS, A Fast Lanczos-type Solver for
**   Nonsymmetric Linear Systems,'' {\sl SIAM J. Sci. Stat. Comput.},
**   10 (1989), pp. 36--52.
**
**  ^DOCUMENTATION:
**
************************************************************************
*
      implicit none
          Include 'fcube.h'
*         Include 'veclib.h'
*
*#    METHRAL        - (^)
      integer ijob
      integer ireq
      integer ier
       integer iwk(*)
       integer iparm(*)
       real fwk(*)
       real fparm(*)
       real u(*)
       real uexact(*)
       real b(*)
      integer ivql
      integer iva
      integer ivqr
*
        integer itimer
        double precision tc1, tr1, tc2, tr2
      integer iitab
      integer iftab
      integer iwkstg
      integer iclstg
      integer in
      integer is
      integer iretlb
      integer ivitmp, ivotmp
      integer iv1tst, iv2tst
      integer istab1, istab2, istab
      integer ib1,ib2, iu1,iu2
      integer iud
      parameter (iud =0)
      integer ibd
      parameter (ibd =1)
      integer ilbd
      parameter (ilbd =2)
      integer irlbd
      parameter (irlbd=3)
      integer ied
      parameter (ied =4)
      integer ird
      parameter (ird =5)
      integer ilrd
      parameter (ilrd =6)
      integer irlrd
      parameter (irlrd=7)
      real urndf
      real top
      real bot
      parameter (urndf=.1)
      real mydot1, mydot2
      external sargck
      external sckstg
      external swrnd
      external sifppr
      external sitput
      external xtimer
      integer ipme
      integer iphost
      integer log2np
      integer nproc
      integer iom
      character*72 errstr
      external ximini
      external ximal
      external xifre
      external sfmini
      external sfmal
      external sffre
*     integer imv1, imv2        ! temporaries - for matvec
      integer ir
      integer iqlr
      integer iqrqlr
      integer irt
      integer iqrqlp
      integer iqlf
      integer iqrqlf
      integer iqlh
      integer iqrqlh
      integer iv1
      integer iv2
      real alphan
      real rdot
      real pap
      real rnorm
      real r0norm
      real rr
      real rrold
      real vlamda
      real uudot, uexdt
*
*
****^^******************************************************************
*     $Modified: spencer@navier.ae.utexas.edu on Thu May 16 12:00:41 1996 $
*     $Id: cgsr.fm4,v 1.16 1994/11/22 05:20:18 joubert Exp $
*     $Revision: 1.16 $
*     $Log: cgsr.fm4,v $
*     Revision 1.16  1994/11/22  05:20:18  joubert
*     misc changes for release 1.0
*
*     Revision 1.15  1994/10/06  16:24:12  joubert
*     fixed Intel flop count problem; ifparm param name changes
*
*     Revision 1.14  1994/09/28  19:24:44  joubert
*     mods to stats and ntest options
*
*     Revision 1.13  1994/07/06  05:52:24  joubert
*     mods to it meths
*
*     Revision 1.12  1994/07/05  22:20:41  joubert
*     lanczos/orthores acceleration; some cleanup of iterative methods
*
*     Revision 1.11  1994/06/24  18:20:40  joubert
*     t3d version mods; installed iom it meth
*
*     Revision 1.10  1994/06/07  17:40:10  joubert
*     installed orthores iterative method
*
*     Revision 1.9  1994/06/02  21:55:05  joubert
*     installed CM random vector routine
*
*     Revision 1.8  1994/04/30  04:41:44  joubert
*     improved error handling
*
*     Revision 1.7  1994/03/26  01:41:34  joubert
*     added gmres acceleration
*
*     Revision 1.6  1994/03/25  08:45:07  joubert
*     added orthomin acceleration
*
*     Revision 1.5  1994/03/24  03:30:58  joubert
*     added cgstab acceleration
*
*     Revision 1.4  1994/03/23  21:15:10  joubert
*     cgs method implemented; itmeth macros developed
*
*     Revision 1.3  1994/02/18  08:50:53  joubert
*     mpi, t3dpvm versions completed; misc other changes
*
*     Revision 1.1  1994/01/05  20:33:18  joubert
*     installed CG Squared routines, not fully functional yet
*
************************************************************************
*
      iitab = (iparm(6))
      iftab = (iparm(6))
      ir = (iparm(6))
      iqlr = (iparm(6))
      iqrqlr = (iparm(6))
      irt = (iparm(6))
      iqrqlp = (iparm(6))
      iqlf = (iparm(6))
      iqrqlf = (iparm(6))
      iqlh = (iparm(6))
      iqrqlh = (iparm(6))
      iv1 = (iparm(6))
      iv2 = (iparm(6))
*
*
      if (ijob .ne. 1) go to 1000
*
      iclstg = 1
      itimer = 0
      in = 0
      iparm(5) = in
      is = 0
*
      iu1 = (iparm(6))
      iu2 = (iparm(6))
      ib1 = (iparm(6))
      ib2 = (iparm(6))
      istab1 = (iparm(6))
      istab2 = (iparm(6))
*
      ipme = mynode ( )
      iphost = myhost ( )
      log2np = nodedim ( )
      nproc = 2**log2np
      if (iparm(27) .eq. 0) iparm(27) = 0
      if (iparm(27) .eq. 0) then
        ier = 0
        if (ijob.eq.1 .or. ijob.eq.2 .or. ijob.eq.0) then
      call ximini ( iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call sfmini ( iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        endif
        call sargck ( iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        call xtimer (tc1,tr1,iparm(22),1)
      endif
*
      call xtimer (tc1,tr1,iparm(22),0)
      itimer = 1
*
      if (iparm(27) .eq. 0) then
        call sifppr ( 1 , iparm, fparm, ier)
           if (ier .lt. 0) go to 900
      endif
*
      call ximal (iitab,26, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call sfmal (iftab,10, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
*
*
      if (iparm(2) .ge. 3) then
        if (ipme .eq. 0) then
          write (iparm(1),'(/''     Conjugate Gradient Squared Method.''
     &/)')
        endif
      endif
*
      call sfmal (iqlr,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      if (((iparm(13).eq.1).or.(iparm(13).eq.3))) then
        if (((mod(iparm(15)/2,2).eq.1) .or. iparm(12).eq.2 .or. iparm(
     &     12).eq.6)) then
      call sfmal (ir,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        endif
      else
        ir = iqlr
      endif
      if (((iparm(13).eq.2).or.(iparm(13).eq.3))) then
      call sfmal (iqrqlr,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      else
        iqrqlr = iqlr
      endif
      call sfmal (irt,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call sfmal (iqrqlp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call sfmal (iqlf,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call sfmal (iqlh,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      if (((iparm(13).eq.2).or.(iparm(13).eq.3))) then
      call sfmal (iqrqlf,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call sfmal (iqrqlh,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      else
        iqrqlf = iqlf
        iqrqlh = iqlh
      endif
      call sfmal (iv1,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call sfmal (iv2,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
*
      if (iparm(14) .eq. -2 ) then
          call swfil (u,0e0, iparm , fparm , iwk , fwk , ier )
      endif
*
      if (iparm(14) .eq. 2 .or. iparm(14) .eq. -1) go to 5500
                 go to 5501
 5500  continue
          call swrdot (mydot1,b, b, iparm , fparm , iwk , fwk , ier )
        call swrnd (u, iparm , fparm , iwk , fwk , ier )
      call sfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call swset (fwk(ivqr),u, iparm , fparm , iwk , fwk , ier )
          iva = iv1
       ireq = 3
      iretlb = 2000
      go to 1500
 2000 continue
      call sffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
          call swrdot (mydot2,fwk(iv1), fwk(iv1), iparm , fparm , iwk , 
     &       fwk , ier )
          call swscl ( u, u, urndf*sqrt(mydot1/mydot2), iparm , fparm , 
     &       iwk , fwk , ier )
                go to 5502
 5501 continue
 5502 continue
*
      if (iparm(14) .eq. 3) go to 5503
                 go to 5504
 5503  continue
      call sfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call swset (fwk(ivqr),u, iparm , fparm , iwk , fwk , ier )
          iva = iv1
       ireq = 3
      iretlb = 2001
      go to 1500
 2001 continue
      call sffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
          call swsub ( fwk(iv1), b, fwk(iv1), iparm , fparm , iwk , fwk 
     &       , ier )
          call swrdot (mydot1,fwk(iv1), fwk(iv1), iparm , fparm , iwk , 
     &       fwk , ier )
          call swset (fwk(iv1),u, iparm , fparm , iwk , fwk , ier )
        call swrnd (u, iparm , fparm , iwk , fwk , ier )
      call sfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call swset (fwk(ivqr),u, iparm , fparm , iwk , fwk , ier )
          iva = iv2
       ireq = 3
      iretlb = 2002
      go to 1500
 2002 continue
      call sffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
          call swrdot (mydot2,fwk(iv2), fwk(iv2), iparm , fparm , iwk , 
     &       fwk , ier )
          call swsax (u,fwk(iv1), u, -urndf*sqrt(mydot1/mydot2), iparm ,
     &        fparm , iwk , fwk , ier )
                go to 5505
 5504 continue
 5505 continue
*
      if (iparm(21) .eq. 1) go to 5506
                 go to 5507
 5506  continue
      call sfmal (istab1,8, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call sfmal (istab2,8, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call sfmal (ib1,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call sfmal (ib2,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call sfmal (iu1,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call sfmal (iu2,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
        istab = istab1
        if (iparm(19) .eq. 1) then
          call swdot (fwk(istab+iud), uexact, uexact, iparm , fparm , 
     &       iwk , fwk , ier )
        endif
          call swdot (fwk(istab+ibd ), b, b, iparm , fparm , iwk , fwk ,
     &        ier )
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5509
                 go to 5510
 5509  continue
          call swset (fwk(ib2),b, iparm , fparm , iwk , fwk , ier )
                go to 5511
 5510 continue
      call sfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          iva = ivitmp
          call swset (fwk(iva),b, iparm , fparm , iwk , fwk , ier )
          ivql = ib2
       ireq = 5
      iretlb = 2003
      go to 1500
 2003 continue
      call sffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
 5511 continue
          call swdot (fwk(istab+ilbd), fwk(ib2), fwk(ib2), iparm , 
     &       fparm , iwk , fwk , ier )
          call swset (fwk(iu1),fwk(ib2), iparm , fparm , iwk , fwk , 
     &       ier )
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5512
                 go to 5513
 5512  continue
          call swset (fwk(iu2),fwk(iu1), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5514
 5513 continue
          ivql = iu1
          ivqr = iu2
       ireq = 7
      iretlb = 2004
      go to 1500
 2004 continue
       if (ijob .eq. -1) go to 900
 5514 continue
          call swdot (fwk(istab+irlbd), fwk(iu2), fwk(iu2), iparm , 
     &       fparm , iwk , fwk , ier )
        if (iparm(19) .eq. 1) then
          call swsub ( fwk(iu1), u, uexact, iparm , fparm , iwk , fwk , 
     &       ier )
          call swdot (fwk(istab+ied), fwk(iu1), fwk(iu1), iparm , fparm 
     &       , iwk , fwk , ier )
        endif
      call sfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call swset (fwk(ivqr),u, iparm , fparm , iwk , fwk , ier )
          iva = ib1
       ireq = 3
      iretlb = 2005
      go to 1500
 2005 continue
      call sffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
          call swsub ( fwk(ib1), b, fwk(ib1), iparm , fparm , iwk , fwk 
     &       , ier )
          call swdot (fwk(istab+ird), fwk(ib1), fwk(ib1), iparm , fparm 
     &       , iwk , fwk , ier )
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5515
                 go to 5516
 5515  continue
          call swset (fwk(ib2),fwk(ib1), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5517
 5516 continue
          iva = ib1
          ivql = ib2
       ireq = 5
      iretlb = 2006
      go to 1500
 2006 continue
       if (ijob .eq. -1) go to 900
 5517 continue
          call swdot (fwk(istab+ilrd ), fwk(ib2), fwk(ib2), iparm , 
     &       fparm , iwk , fwk , ier )
          call swset (fwk(iu1),fwk(ib2), iparm , fparm , iwk , fwk , 
     &       ier )
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5518
                 go to 5519
 5518  continue
          call swset (fwk(iu2),fwk(iu1), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5520
 5519 continue
          ivql = iu1
          ivqr = iu2
       ireq = 7
      iretlb = 2007
      go to 1500
 2007 continue
       if (ijob .eq. -1) go to 900
 5520 continue
          call swdot (fwk(istab+irlrd), fwk(iu2), fwk(iu2), iparm , 
     &       fparm , iwk , fwk , ier )
        if (iparm(2) .ge. 4) then
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(1x, 60(''*''))')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''Initial iteration statistics'')')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (iparm(19) .eq. 1) then
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of uexact            '',1pg20.10
     &)') sqrt(fwk(istab+iud))
        endif
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of b                 '',1pg20.10
     &)') sqrt(fwk(istab+ibd))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Ql*b              '',1pg20.10
     &)') sqrt(fwk(istab+ilbd))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Qr*Ql*b           '',1pg20.10
     &)') sqrt(fwk(istab+irlbd))
        endif
        if (iparm(19) .eq. 1) then
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of error             '',1pg20.10
     &)') sqrt(fwk(istab+ied))
        endif
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of residual          '',1pg20.10
     &)') sqrt(fwk(istab+ird))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Ql*r              '',1pg20.10
     &)') sqrt(fwk(istab+ilrd))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Qr*Ql*r           '',1pg20.10
     &)') sqrt(fwk(istab+irlrd))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(1x, 60(''*''))')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        endif
      call sffre (iu2,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call sffre (iu1,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call sffre (ib2,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call sffre (ib1,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
                go to 5508
 5507 continue
 5508 continue
*
      if (((iparm(14).eq.0).or.(iparm(14).eq.-2))) go to 5521
                 go to 5522
 5521  continue
        if (((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5524
                 go to 5525
 5524  continue
          if (((mod(iparm(15)/2,2).eq.1) .or. iparm(12).eq.2 .or. iparm(
     &       12).eq.6)) go to 5527
                 go to 5528
 5527  continue
          call swset (fwk(ir),b, iparm , fparm , iwk , fwk , ier )
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5530
                 go to 5531
 5530  continue
          call swset (fwk(iqlr),fwk(ir), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5532
 5531 continue
          iva = ir
          ivql = iqlr
       ireq = 5
      iretlb = 2008
      go to 1500
 2008 continue
       if (ijob .eq. -1) go to 900
 5532 continue
                go to 5529
 5528 continue
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5533
                 go to 5534
 5533  continue
          call swset (fwk(iqlr),b, iparm , fparm , iwk , fwk , ier )
                go to 5535
 5534 continue
      call sfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          iva = ivitmp
          call swset (fwk(iva),b, iparm , fparm , iwk , fwk , ier )
          ivql = iqlr
       ireq = 5
      iretlb = 2009
      go to 1500
 2009 continue
      call sffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
 5535 continue
 5529 continue
                go to 5526
 5525 continue
          call swset (fwk(iqlr),b, iparm , fparm , iwk , fwk , ier )
 5526 continue
                go to 5523
 5522 continue
        if (((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5536
                 go to 5537
 5536  continue
          if (((mod(iparm(15)/2,2).eq.1) .or. iparm(12).eq.2 .or. iparm(
     &       12).eq.6)) go to 5539
                 go to 5540
 5539  continue
      call sfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call swset (fwk(ivqr),u, iparm , fparm , iwk , fwk , ier )
          iva = ir
       ireq = 3
      iretlb = 2010
      go to 1500
 2010 continue
      call sffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
          call swsub ( fwk(ir), b, fwk(ir), iparm , fparm , iwk , fwk , 
     &       ier )
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5542
                 go to 5543
 5542  continue
          call swset (fwk(iqlr),fwk(ir), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5544
 5543 continue
          iva = ir
          ivql = iqlr
       ireq = 5
      iretlb = 2011
      go to 1500
 2011 continue
       if (ijob .eq. -1) go to 900
 5544 continue
                go to 5541
 5540 continue
      call sfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call swset (fwk(ivqr),u, iparm , fparm , iwk , fwk , ier )
          iva = iv1
       ireq = 3
      iretlb = 2012
      go to 1500
 2012 continue
      call sffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
          call swsub ( fwk(iv1), b, fwk(iv1), iparm , fparm , iwk , fwk 
     &       , ier )
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5545
                 go to 5546
 5545  continue
          call swset (fwk(iqlr),fwk(iv1), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5547
 5546 continue
          iva = iv1
          ivql = iqlr
       ireq = 5
      iretlb = 2013
      go to 1500
 2013 continue
       if (ijob .eq. -1) go to 900
 5547 continue
 5541 continue
                go to 5538
 5537 continue
      call sfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          ivqr = ivitmp
          call swset (fwk(ivqr),u, iparm , fparm , iwk , fwk , ier )
          iva = iqlr
       ireq = 3
      iretlb = 2014
      go to 1500
 2014 continue
      call sffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
          call swsub ( fwk(iqlr), b, fwk(ir), iparm , fparm , iwk , fwk 
     &       , ier )
 5538 continue
 5523 continue
      if (((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5548
                 go to 5549
 5548  continue
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5551
                 go to 5552
 5551  continue
          call swset (fwk(iqrqlr),fwk(iqlr), iparm , fparm , iwk , fwk ,
     &        ier )
                go to 5553
 5552 continue
          ivql = iqlr
          ivqr = iqrqlr
       ireq = 7
      iretlb = 2015
      go to 1500
 2015 continue
       if (ijob .eq. -1) go to 900
 5553 continue
                go to 5550
 5549 continue
 5550 continue
*
          call swrdot (rdot,fwk(iqlr), fwk(iqlr), iparm , fparm , iwk , 
     &       fwk , ier )
      rnorm = sqrt(abs(rdot))
      r0norm = rnorm
*
*---------------------------iteration loop------------------------------
*
  100 continue
*
        if (iparm(2) .ge. 5) then
        if (ipme .eq. 0) then
          write (iparm(1),500) 'QL r norm           ', rnorm
        endif
        endif
*
        if (iparm(12).eq.-3) go to 5554
                 go to 5555
 5554  continue
          if ((mod(iparm(15)/1,2).eq.1)) then
          endif
          if ((mod(iparm(15)/2,2).eq.1)) then
          iva = ir
          endif
          if ((mod(iparm(15)/4,2).eq.1)) then
          ivql = iqlr
          endif
          if ((mod(iparm(15)/8,2).eq.1)) then
          ivqr = iqrqlr
          endif
          ireq = 9
      iretlb = 2016
      go to 1500
 2016 continue
          if ((mod(iparm(15)/8,2).eq.1)) then
          endif
          if ((mod(iparm(15)/4,2).eq.1)) then
          endif
          if ((mod(iparm(15)/2,2).eq.1)) then
          endif
          if ((mod(iparm(15)/1,2).eq.1)) then
          endif
          if (ier .ge. 0) then
            call sitput ( in , is , iparm , fparm , iwk , fwk , ier )
          endif
          if (ijob .eq. -1) then
            if (iparm(2).ge.3 .and. iparm(27).eq.0) then
        if (ipme .eq. 0) then
          write (iparm(1),'(/a/)') ' User-requested termination of itera
     &tive method.'
        endif
            endif
            go to 900
          endif
                go to 5556
 5555 continue
 5556 continue
        if (iparm(12) .gt. 1) go to 5557
                 go to 5558
 5557  continue
          if (iparm(12) .eq. 1 ) then
            if (iparm(5) .eq. 0) then
              if (iparm(19) .eq. 0) then
                ier = -5
                call xersho ( ier, 'scgsr' , iparm , '19 and 12 incompat
     &ible' )
                go to 900
              endif
          call swdot (bot, uexact, uexact, iparm , fparm , iwk , fwk , 
     &       ier )
              top = bot
            else
      call sfmal (iv1tst,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          call swsub ( fwk(iv1tst), u, uexact, iparm , fparm , iwk , 
     &       fwk , ier )
          call swdot (top, fwk(iv1tst), fwk(iv1tst), iparm , fparm , 
     &       iwk , fwk , ier )
      call sffre (iv1tst,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
            endif
          endif
          if (iparm(12) .eq. 2 ) then
            if (iparm(5) .eq. 0) then
          call swdot (bot, b, b, iparm , fparm , iwk , fwk , ier )
              top = bot
            else
          call swdot (top, fwk(ir), fwk(ir), iparm , fparm , iwk , fwk ,
     &        ier )
            endif
          endif
          if (iparm(12) .eq. 3) go to 5560
                 go to 5561
 5560  continue
            if (iparm(5) .eq. 0) go to 5563
                 go to 5564
 5563  continue
              if (((iparm(14).eq.0).or.(iparm(14).eq.-2))) go to 5566
                 go to 5567
 5566  continue
          call swdot (bot, fwk(iqlr), fwk(iqlr), iparm , fparm , iwk , 
     &       fwk , ier )
                go to 5568
 5567 continue
      call sfmal (iv1tst,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5569
                 go to 5570
 5569  continue
          call swset (fwk(iv1tst),b, iparm , fparm , iwk , fwk , ier )
                go to 5571
 5570 continue
      call sfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          iva = ivitmp
          call swset (fwk(iva),b, iparm , fparm , iwk , fwk , ier )
          ivql = iv1tst
       ireq = 5
      iretlb = 2017
      go to 1500
 2017 continue
      call sffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
 5571 continue
          call swdot (bot, fwk(iv1tst), fwk(iv1tst), iparm , fparm , 
     &       iwk , fwk , ier )
      call sffre (iv1tst,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
 5568 continue
              top = bot
                go to 5565
 5564 continue
          call swdot (top, fwk(iqlr), fwk(iqlr), iparm , fparm , iwk , 
     &       fwk , ier )
 5565 continue
                go to 5562
 5561 continue
 5562 continue
          if (iparm(12) .eq. 4) go to 5572
                 go to 5573
 5572  continue
            if (iparm(5) .eq. 0) go to 5575
                 go to 5576
 5575  continue
              if (((iparm(14).eq.0).or.(iparm(14).eq.-2))) go to 5578
                 go to 5579
 5578  continue
          call swdot (bot, fwk(iqrqlr), fwk(iqrqlr), iparm , fparm , 
     &       iwk , fwk , ier )
                go to 5580
 5579 continue
      call sfmal (iv1tst,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call sfmal (iv2tst,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5581
                 go to 5582
 5581  continue
          call swset (fwk(iv1tst),b, iparm , fparm , iwk , fwk , ier )
                go to 5583
 5582 continue
      call sfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          iva = ivitmp
          call swset (fwk(iva),b, iparm , fparm , iwk , fwk , ier )
          ivql = iv1tst
       ireq = 5
      iretlb = 2018
      go to 1500
 2018 continue
      call sffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
       if (ijob .eq. -1) go to 900
 5583 continue
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5584
                 go to 5585
 5584  continue
          call swset (fwk(iv2tst),fwk(iv1tst), iparm , fparm , iwk , 
     &       fwk , ier )
                go to 5586
 5585 continue
          ivql = iv1tst
          ivqr = iv2tst
       ireq = 7
      iretlb = 2019
      go to 1500
 2019 continue
       if (ijob .eq. -1) go to 900
 5586 continue
          call swdot (bot, fwk(iv2tst), fwk(iv2tst), iparm , fparm , 
     &       iwk , fwk , ier )
      call sffre (iv2tst,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
      call sffre (iv1tst,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
 5580 continue
              top = bot
                go to 5577
 5576 continue
          call swdot (top, fwk(iqrqlr), fwk(iqrqlr), iparm , fparm , 
     &       iwk , fwk , ier )
 5577 continue
                go to 5574
 5573 continue
 5574 continue
          if (iparm(12) .eq. 1 ) then
            if (iparm(19) .eq. 0) then
                ier = -5
                call xersho ( ier, 'scgsr' , iparm , '19 and 12 incompat
     &ible' )
                go to 900
            endif
            if (((iparm(14).eq.0).or.(iparm(14).eq.-2))) then
          call swdot (top, uexact, uexact, iparm , fparm , iwk , fwk , 
     &       ier )
            else
      call sfmal (iv1tst,iparm(3), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
          call swsub ( fwk(iv1tst), u, uexact, iparm , fparm , iwk , 
     &       fwk , ier )
          call swdot (top, fwk(iv1tst), fwk(iv1tst), iparm , fparm , 
     &       iwk , fwk , ier )
      call sffre (iv1tst,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
            endif
            if (iparm(5) .eq. 0) bot = top
          endif
          if (iparm(12) .eq. 6) then
          call swdot (top, fwk(ir), fwk(ir), iparm , fparm , iwk , fwk ,
     &        ier )
            if (iparm(5) .eq. 0) bot = top
          endif
          if (iparm(12) .eq. 7) then
          call swdot (top, fwk(iqlr), fwk(iqlr), iparm , fparm , iwk , 
     &       fwk , ier )
            if (iparm(5) .eq. 0) bot = top
          endif
          if (iparm(12) .eq. 8) then
          call swdot (top, fwk(iqrqlr), fwk(iqrqlr), iparm , fparm , 
     &       iwk , fwk , ier )
            if (iparm(5) .eq. 0) bot = top
          endif
          if ((top) .lt. (0e0)) then
                ier = -10
                call xersho ( ier, 'scgsr' , iparm , 'Attempt to take sq
     &rt of negative number' )
                go to 900
          endif
          if ((bot) .le. (0e0)) then
                ier = -10
                call xersho ( ier, 'scgsr' , iparm , 'Attempt to divide 
     &by zero' )
                go to 900
          endif
          fparm(5) = sqrt((top)/(bot))
          call sitput ( in , is , iparm , fparm , iwk , fwk , ier )
          if ((fparm(5)) .le. (fparm(4))) go to 2500
                go to 5559
 5558 continue
 5559 continue
*
        if (iparm(12) .eq. -1) then
          if (r0norm .eq. 0e0) then
                ier = -10
                call xersho ( ier, 'scgsr' , iparm , 'Attempt to divide 
     &by zero' )
                go to 900
          endif
          fparm(5) = rnorm/r0norm
          call sitput ( in , is , iparm , fparm , iwk , fwk , ier )
          if ((fparm(5)) .le. (fparm(4))) go to 2500
        endif
*
        if (iparm(12) .eq. -2) then
          call swsub ( fwk(iv1), u, uexact, iparm , fparm , iwk , fwk , 
     &       ier )
          call swrdot (uudot,fwk(iv1), fwk(iv1), iparm , fparm , iwk , 
     &       fwk , ier )
          call swrdot (uexdt,uexact, uexact, iparm , fparm , iwk , fwk ,
     &        ier )
          if ((uudot) .lt. (0e0)) then
                ier = -10
                call xersho ( ier, 'scgsr' , iparm , 'Attempt to take sq
     &rt of negative number' )
                go to 900
          endif
          if ((uexdt) .le. (0e0)) then
                ier = -10
                call xersho ( ier, 'scgsr' , iparm , 'Attempt to divide 
     &by zero' )
                go to 900
          endif
          fparm(5) = sqrt((uudot)/(uexdt))
          call sitput ( in , is , iparm , fparm , iwk , fwk , ier )
          if ((fparm(5)) .le. (fparm(4))) go to 2500
        endif
*
        if (in .ge. iparm(4)) then
                ier = 2
                call xersho ( ier, 'scgsr' , iparm , ' ' )
          go to 900
        endif
*
        call sckstg ( iclstg, iwkstg, iparm , fparm , iwk , fwk , ier )
           if (ier .lt. 0) go to 900
*
*-----------------------proceed with iteration--------------------------
*
        if (in .eq. 0) then
          call swset (fwk(irt ),fwk(iqlr), iparm , fparm , iwk , fwk , 
     &       ier )
          call swset (fwk(iqlf),fwk(iqlr), iparm , fparm , iwk , fwk , 
     &       ier )
          if (((iparm(13).eq.2).or.(iparm(13).eq.3))) then
          call swset (fwk(iqrqlp),fwk(iqrqlr), iparm , fparm , iwk , 
     &       fwk , ier )
          call swset (fwk(iqrqlf),fwk(iqrqlr), iparm , fparm , iwk , 
     &       fwk , ier )
          else
          call swset (fwk(iqrqlp),fwk(iqlr), iparm , fparm , iwk , fwk ,
     &        ier )
          endif
          if (iparm(20) .eq. 1) then
            rr = rdot
          else
          call swcdot (rr, fwk(irt), fwk(iqlr), iparm(20), iparm , 
     &       fparm , iwk , fwk , ier )
          endif
          if (iparm(2) .ge. 5) then
        if (ipme .eq. 0) then
          write (iparm(1),500) '(rtilde0,r)         ', rr
        endif
          endif
        else
          rrold = rr
          if (rrold .eq. 0e0) then
                ier = -7
                call xersho ( ier, 'scgsr' , iparm , ' ' )
                go to 900
          endif
          call swcdot (rr, fwk(irt), fwk(iqlr), iparm(20), iparm , 
     &       fparm , iwk , fwk , ier )
          if (iparm(2) .ge. 5) then
        if (ipme .eq. 0) then
          write (iparm(1),500) '(rtilde0,r)         ', rr
        endif
          endif
          alphan = rr / rrold
          if (iparm(2) .ge. 5) then
        if (ipme .eq. 0) then
          write (iparm(1),500) 'alpha               ', alphan
        endif
          endif
          call swsax (fwk(iqlf),fwk(iqlr), fwk(iqlh), alphan , iparm , 
     &       fparm , iwk , fwk , ier )
          if (((iparm(13).eq.2).or.(iparm(13).eq.3))) then
          call swsax (fwk(iqrqlf),fwk(iqrqlr), fwk(iqrqlh), alphan , 
     &       iparm , fparm , iwk , fwk , ier )
          endif
          call swsax (fwk(iqrqlp),fwk(iqrqlf), fwk(iqrqlp), alphan**2, 
     &       iparm , fparm , iwk , fwk , ier )
          call swsax (fwk(iqrqlp),fwk(iqrqlp), fwk(iqrqlh), alphan , 
     &       iparm , fparm , iwk , fwk , ier )
        endif
*
        go to (501,502,503,504), iparm(13)+1
  501   continue
          ivqr = iqrqlp
          iva = iv1
       ireq = 3
      iretlb = 2020
      go to 1500
 2020 continue
       if (ijob .eq. -1) go to 900
        go to 510
  502   continue
          ivqr = iqrqlp
          iva = iv2
       ireq = 3
      iretlb = 2021
      go to 1500
 2021 continue
       if (ijob .eq. -1) go to 900
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5587
                 go to 5588
 5587  continue
          call swset (fwk(iv1),fwk(iv2), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5589
 5588 continue
          iva = iv2
          ivql = iv1
       ireq = 5
      iretlb = 2022
      go to 1500
 2022 continue
       if (ijob .eq. -1) go to 900
 5589 continue
        go to 510
  503   continue
          ivqr = iqrqlp
          iva = iv2
       ireq = 3
      iretlb = 2023
      go to 1500
 2023 continue
       if (ijob .eq. -1) go to 900
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5590
                 go to 5591
 5590  continue
          call swset (fwk(iv1),fwk(iv2), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5592
 5591 continue
          ivql = iv2
          ivqr = iv1
       ireq = 7
      iretlb = 2024
      go to 1500
 2024 continue
       if (ijob .eq. -1) go to 900
 5592 continue
        go to 510
  504   continue
          ivqr = iqrqlp
          iva = iv1
       ireq = 3
      iretlb = 2025
      go to 1500
 2025 continue
       if (ijob .eq. -1) go to 900
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5593
                 go to 5594
 5593  continue
          call swset (fwk(iv2),fwk(iv1), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5595
 5594 continue
          iva = iv1
          ivql = iv2
       ireq = 5
      iretlb = 2026
      go to 1500
 2026 continue
       if (ijob .eq. -1) go to 900
 5595 continue
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5596
                 go to 5597
 5596  continue
          call swset (fwk(iv1),fwk(iv2), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5598
 5597 continue
          ivql = iv2
          ivqr = iv1
       ireq = 7
      iretlb = 2027
      go to 1500
 2027 continue
       if (ijob .eq. -1) go to 900
 5598 continue
  510   continue
*
        if (((iparm(13).eq.2).or.(iparm(13).eq.3))) then
          call swcdot (pap, fwk(irt), fwk(iv2), iparm(20), iparm , 
     &       fparm , iwk , fwk , ier )
        else
          call swcdot (pap, fwk(irt), fwk(iv1), iparm(20), iparm , 
     &       fparm , iwk , fwk , ier )
        endif
        if (iparm(2) .ge. 5) then
        if (ipme .eq. 0) then
          write (iparm(1),500) '(rtilde0,ap)        ', pap
        endif
        endif
        if (pap .eq. 0e0) then
                ier = -6
                call xersho ( ier, 'scgsr' , iparm , ' ' )
                go to 900
        endif
        vlamda = rr / pap
        if (iparm(2) .ge. 5) then
        if (ipme .eq. 0) then
          write (iparm(1),500) 'lambda              ', vlamda
        endif
        endif
*
          call swsax (fwk(iqrqlh),fwk(iqrqlf), fwk(iv1), (-vlamda), 
     &       iparm , fparm , iwk , fwk , ier )
        if (((iparm(13).eq.2).or.(iparm(13).eq.3))) then
          call swsax (fwk(iqlh),fwk(iqlf), fwk(iv2), (-vlamda), iparm , 
     &       fparm , iwk , fwk , ier )
        endif
*
          call swadd (fwk(iv1), fwk(iqrqlf), fwk(iqrqlh), iparm , fparm 
     &       , iwk , fwk , ier )
*
          call swsax (u,u, fwk(iv1), vlamda , iparm , fparm , iwk , fwk 
     &       , ier )
        if (((iparm(13).eq.1).or.(iparm(13).eq.3)) .and. ((mod(iparm(15)
     &     /2,2).eq.1) .or. iparm(12).eq.2 .or. iparm(12).eq.6)) go to 
     &     5599
                 go to 5600
 5599  continue
          ivqr = iv1
          iva = iv2
       ireq = 3
      iretlb = 2028
      go to 1500
 2028 continue
       if (ijob .eq. -1) go to 900
          call swsax (fwk(ir),fwk(ir), fwk(iv2), (-vlamda), iparm , 
     &       fparm , iwk , fwk , ier )
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5602
                 go to 5603
 5602  continue
          call swset (fwk(iv1),fwk(iv2), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5604
 5603 continue
          iva = iv2
          ivql = iv1
       ireq = 5
      iretlb = 2029
      go to 1500
 2029 continue
       if (ijob .eq. -1) go to 900
 5604 continue
          call swsax (fwk(iqlr),fwk(iqlr), fwk(iv1), (-vlamda), iparm , 
     &       fparm , iwk , fwk , ier )
                go to 5601
 5600 continue
          if (((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5605
                 go to 5606
 5605  continue
          ivqr = iv1
          iva = iv2
       ireq = 3
      iretlb = 2030
      go to 1500
 2030 continue
       if (ijob .eq. -1) go to 900
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5608
                 go to 5609
 5608  continue
          call swset (fwk(iv1),fwk(iv2), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5610
 5609 continue
          iva = iv2
          ivql = iv1
       ireq = 5
      iretlb = 2031
      go to 1500
 2031 continue
       if (ijob .eq. -1) go to 900
 5610 continue
          call swsax (fwk(iqlr),fwk(iqlr), fwk(iv1), (-vlamda), iparm , 
     &       fparm , iwk , fwk , ier )
                go to 5607
 5606 continue
          ivqr = iv1
          iva = iv2
       ireq = 3
      iretlb = 2032
      go to 1500
 2032 continue
       if (ijob .eq. -1) go to 900
          call swsax (fwk(iqlr),fwk(iqlr), fwk(iv2), (-vlamda), iparm , 
     &       fparm , iwk , fwk , ier )
 5607 continue
 5601 continue
        if (((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5611
                 go to 5612
 5611  continue
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5614
                 go to 5615
 5614  continue
          call swset (fwk(iqrqlr),fwk(iqlr), iparm , fparm , iwk , fwk ,
     &        ier )
                go to 5616
 5615 continue
          ivql = iqlr
          ivqr = iqrqlr
       ireq = 7
      iretlb = 2033
      go to 1500
 2033 continue
       if (ijob .eq. -1) go to 900
 5616 continue
                go to 5613
 5612 continue
 5613 continue
*
          call swrdot (rdot,fwk(iqlr), fwk(iqlr), iparm , fparm , iwk , 
     &       fwk , ier )
        rnorm = sqrt(abs(rdot))
*
        in = in + 1
        iparm(5) = in
        is = is + 1
*
      go to 100
*
 2500 continue
      if (iparm(2) .ge. 3) then
        if (ipme .eq. 0) then
          write (iparm(1),'(/a,i7,a/)') '     Conjugate Gradient Squared
     & Method has converged in ', iparm(5), ' iterations.'
        endif
      endif
*
*-------------------------------terminate-------------------------------
*
  900 continue
*
      if (iclstg .ne. 1) then
        call sckstg (-1, iwkstg, iparm , fparm , iwk , fwk , ier )
      endif
*
      if (iparm(21).eq.1 .and. ier.ge.0) go to 5617
                 go to 5618
 5617  continue
      call sfmal (ib1,iparm(3), iparm , fparm , iwk , fwk , ier )
      call sfmal (ib2,iparm(3), iparm , fparm , iwk , fwk , ier )
      call sfmal (iu1,iparm(3), iparm , fparm , iwk , fwk , ier )
      call sfmal (iu2,iparm(3), iparm , fparm , iwk , fwk , ier )
        istab = istab2
        if (iparm(19) .eq. 1) then
          call swdot (fwk(istab+iud), uexact, uexact, iparm , fparm , 
     &       iwk , fwk , ier )
        endif
          call swdot (fwk(istab+ibd ), b, b, iparm , fparm , iwk , fwk ,
     &        ier )
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5620
                 go to 5621
 5620  continue
          call swset (fwk(ib2),b, iparm , fparm , iwk , fwk , ier )
                go to 5622
 5621 continue
      call sfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
          iva = ivitmp
          call swset (fwk(iva),b, iparm , fparm , iwk , fwk , ier )
          ivql = ib2
       ireq = 5
      iretlb = 2034
      go to 1500
 2034 continue
      call sffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
       if (ijob .eq. -1) go to 900
 5622 continue
          call swdot (fwk(istab+ilbd), fwk(ib2), fwk(ib2), iparm , 
     &       fparm , iwk , fwk , ier )
          call swset (fwk(iu1),fwk(ib2), iparm , fparm , iwk , fwk , 
     &       ier )
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5623
                 go to 5624
 5623  continue
          call swset (fwk(iu2),fwk(iu1), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5625
 5624 continue
          ivql = iu1
          ivqr = iu2
       ireq = 7
      iretlb = 2035
      go to 1500
 2035 continue
       if (ijob .eq. -1) go to 900
 5625 continue
          call swdot (fwk(istab+irlbd), fwk(iu2), fwk(iu2), iparm , 
     &       fparm , iwk , fwk , ier )
        if (iparm(19) .eq. 1) then
          call swsub ( fwk(iu1), u, uexact, iparm , fparm , iwk , fwk , 
     &       ier )
          call swdot (fwk(istab+ied), fwk(iu1), fwk(iu1), iparm , fparm 
     &       , iwk , fwk , ier )
        endif
      call sfmal (ivitmp,iparm(3), iparm , fparm , iwk , fwk , ier )
          ivqr = ivitmp
          call swset (fwk(ivqr),u, iparm , fparm , iwk , fwk , ier )
          iva = ib1
       ireq = 3
      iretlb = 2036
      go to 1500
 2036 continue
      call sffre (ivitmp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
       if (ijob .eq. -1) go to 900
          call swsub ( fwk(ib1), b, fwk(ib1), iparm , fparm , iwk , fwk 
     &       , ier )
          call swdot (fwk(istab+ird), fwk(ib1), fwk(ib1), iparm , fparm 
     &       , iwk , fwk , ier )
         if (.not. ((iparm(13).eq.1).or.(iparm(13).eq.3))) go to 5626
                 go to 5627
 5626  continue
          call swset (fwk(ib2),fwk(ib1), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5628
 5627 continue
          iva = ib1
          ivql = ib2
       ireq = 5
      iretlb = 2037
      go to 1500
 2037 continue
       if (ijob .eq. -1) go to 900
 5628 continue
          call swdot (fwk(istab+ilrd ), fwk(ib2), fwk(ib2), iparm , 
     &       fparm , iwk , fwk , ier )
          call swset (fwk(iu1),fwk(ib2), iparm , fparm , iwk , fwk , 
     &       ier )
         if (.not. ((iparm(13).eq.2).or.(iparm(13).eq.3))) go to 5629
                 go to 5630
 5629  continue
          call swset (fwk(iu2),fwk(iu1), iparm , fparm , iwk , fwk , 
     &       ier )
                go to 5631
 5630 continue
          ivql = iu1
          ivqr = iu2
       ireq = 7
      iretlb = 2038
      go to 1500
 2038 continue
       if (ijob .eq. -1) go to 900
 5631 continue
          call swdot (fwk(istab+irlrd), fwk(iu2), fwk(iu2), iparm , 
     &       fparm , iwk , fwk , ier )
        if (iparm(19) .eq. 1) then
          if (fwk(istab1+ied) .eq. 0e0) then
                ier = 4
                call xersho ( ier, 'scgsr' , iparm , 'Value of RELERR ma
     &y be inaccurate' )
          else
            fparm(8) = sqrt(fwk(istab2+ied)/fwk(istab1+ied))
          endif
        endif
        if (fwk(istab1+ird) .eq. 0e0) then
                ier = 4
                call xersho ( ier, 'scgsr' , iparm , 'Value of RELRSD ma
     &y be inaccurate' )
        else
          fparm(7) = sqrt(fwk(istab2+ird)/fwk(istab1+ird))
        endif
        if (iparm(2) .ge. 4) then
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(1x, 60(''*''))')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''Final iteration statistics'')')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (iparm(19) .eq. 1) then
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of uexact            '',1pg20.10
     &)') sqrt(fwk(istab+iud))
        endif
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of b                 '',1pg20.10
     &)') sqrt(fwk(istab+ibd))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Ql*b              '',1pg20.10
     &)') sqrt(fwk(istab+ilbd))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Qr*Ql*b           '',1pg20.10
     &)') sqrt(fwk(istab+irlbd))
        endif
        if (iparm(19) .eq. 1) then
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of error             '',1pg20.10
     &)') sqrt(fwk(istab+ied))
        endif
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of residual          '',1pg20.10
     &)') sqrt(fwk(istab+ird))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Ql*r              '',1pg20.10
     &)') sqrt(fwk(istab+ilrd))
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''2-norm of Qr*Ql*r           '',1pg20.10
     &)') sqrt(fwk(istab+irlrd))
        endif
        if (iparm(19) .eq. 1) then
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''Relative 2-norm of error    '',1pg20.10
     &)') fparm(8)
        endif
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(5x,''Relative 2-norm of residual '',1pg20.10
     &)') fparm(7)
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'(1x, 60(''*''))')
        endif
        if (ipme .eq. 0) then
          write (iparm(1),'()')
        endif
        endif
      call sffre (iu2,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      call sffre (iu1,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      call sffre (ib2,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      call sffre (ib1,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      call sffre (istab2,8, iparm , fparm , iwk , fwk , ier )
      call sffre (istab1,8, iparm , fparm , iwk , fwk , ier )
                go to 5619
 5618 continue
 5619 continue
*
      call sffre (iv2 ,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      call sffre (iv1 ,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      if (((iparm(13).eq.2).or.(iparm(13).eq.3))) then
      call sffre (iqrqlh,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      call sffre (iqrqlf,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      endif
      call sffre (iqlh,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      call sffre (iqlf,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      call sffre (iqrqlp,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      call sffre (irt ,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      if (((iparm(13).eq.2).or.(iparm(13).eq.3))) then
      call sffre (iqrqlr,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      endif
      if (((mod(iparm(15)/2,2).eq.1) .or. iparm(12).eq.2 .or. iparm(12)
     &   .eq.6) .and. ((iparm(13).eq.1).or.(iparm(13).eq.3))) then
      call sffre (ir,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
      endif
      call sffre (iqlr,iparm(3)*(1), iparm , fparm , iwk , fwk , ier )
*
*
*
*
      call sffre (iftab,10, iparm , fparm , iwk , fwk , ier )
      call xifre (iitab,26, iparm , fparm , iwk , fwk , ier )
*
      if (itimer .eq. 1) then
        call xtimer (tc2,tr2,iparm(22),0)
        fparm(1) = fparm(1) + (tc2-tc1)
        fparm(2) = fparm(2) + (tr2-tr1)
      endif
*
  910 continue
      if (iparm(27) .eq. 0) then
        call sifppr ( -1 , iparm, fparm, ier )
      endif
*
      if (iparm(27) .eq. 0) iparm(27) = 0
      iparm(11) = (iparm(6))
      ireq = -1
      return
*
*-------------------------------brancher--------------------------------
*
*
 1000 continue
*
      ipme = mynode ( )
      iphost = myhost ( )
      log2np = nodedim ( )
      nproc = 2**log2np
      iitab = iparm(11)
      if (iitab.eq.(iparm(6)) .and. ijob.eq.-1) go to 910
*
      if (iparm(11) .eq. (iparm(6))) then
                ier = -4
                call xersho ( ier, 'scgsr' , iparm , 'Values of ijob and
     & iparm(iptr) incompatible' )
      endif
*
      call xtimer (tc1,tr1,iparm(22),0)
*
        iitab = iwk(iitab+0)
        iftab = iwk(iitab+1)
        istab = iwk(iitab+2)
        iwkstg = iwk(iitab+3)
        iclstg = iwk(iitab+4)
        in = iwk(iitab+5)
        is = iwk(iitab+6)
        itimer = iwk(iitab+7)
        iretlb = iwk(iitab+8)
        ivitmp = iwk(iitab+9)
        ivotmp = iwk(iitab+10)
        iv1tst = iwk(iitab+11)
        iv2tst = iwk(iitab+12)
        istab1 = iwk(iitab+13)
        istab2 = iwk(iitab+14)
        ir = iwk(iitab+15)
        iqlr = iwk(iitab+16)
        iqrqlr = iwk(iitab+17)
        irt = iwk(iitab+18)
        iqrqlp = iwk(iitab+19)
        iqlf = iwk(iitab+20)
        iqrqlf = iwk(iitab+21)
        iqlh = iwk(iitab+22)
        iqrqlh = iwk(iitab+23)
        iv1 = iwk(iitab+24)
        iv2 = iwk(iitab+25)
        top = fwk(iftab+0)
        bot = fwk(iftab+1)
        alphan = fwk(iftab+2)
        rdot = fwk(iftab+3)
        pap = fwk(iftab+4)
        rnorm = fwk(iftab+5)
        r0norm = fwk(iftab+6)
        rr = fwk(iftab+7)
        rrold = fwk(iftab+8)
        vlamda = fwk(iftab+9)
*
      if (ijob.ne.3 .and. ijob.ne.-1) then
                ier = -4
                call xersho ( ier, 'scgsr' , iparm , 'Argument ijob inva
     &lid' )
      endif
*
      go to (2000,2001,2002,2003,2004,2005,2006,2007,2008,2009,2010,
     &   2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022,
     &   2023,2024,2025,2026,2027,2028,2029,2030,2031,2032,2033,2034,
     &   2035,2036,2037,2038), iretlb - 2000 + 1
*
*---------------------------request of caller---------------------------
*
*
 1500 continue
      iparm(11) = iitab
        iwk(iitab+0) = iitab
        iwk(iitab+1) = iftab
        iwk(iitab+2) = istab
        iwk(iitab+3) = iwkstg
        iwk(iitab+4) = iclstg
        iwk(iitab+5) = in
        iwk(iitab+6) = is
        iwk(iitab+7) = itimer
        iwk(iitab+8) = iretlb
        iwk(iitab+9) = ivitmp
        iwk(iitab+10) = ivotmp
        iwk(iitab+11) = iv1tst
        iwk(iitab+12) = iv2tst
        iwk(iitab+13) = istab1
        iwk(iitab+14) = istab2
        iwk(iitab+15) = ir
        iwk(iitab+16) = iqlr
        iwk(iitab+17) = iqrqlr
        iwk(iitab+18) = irt
        iwk(iitab+19) = iqrqlp
        iwk(iitab+20) = iqlf
        iwk(iitab+21) = iqrqlf
        iwk(iitab+22) = iqlh
        iwk(iitab+23) = iqrqlh
        iwk(iitab+24) = iv1
        iwk(iitab+25) = iv2
        fwk(iftab+0) = top
        fwk(iftab+1) = bot
        fwk(iftab+2) = alphan
        fwk(iftab+3) = rdot
        fwk(iftab+4) = pap
        fwk(iftab+5) = rnorm
        fwk(iftab+6) = r0norm
        fwk(iftab+7) = rr
        fwk(iftab+8) = rrold
        fwk(iftab+9) = vlamda
      call xtimer (tc2,tr2,iparm(22),0)
      fparm(1) = fparm(1) + (tc2-tc1)
      fparm(2) = fparm(2) + (tr2-tr1)
      return
*
*
*--------------------------------end------------------------------------
*
  500 format ( 1x, a, 1pg20.10 )
*
      end
