/*
 * This is a recursive row ordering strategy routine
 *
 * The overall principle of operation of this routine is described in
 * the report ``OptimQR, Discrete Optimization of the Sparse QR
 * Factorization'' Which can be found at
 *              http://ostenfeld.dk/~jakob/OptimQR/
 *
 *
 * In order for the heuristic to be able to give a qualified guess at
 * the possible minimal cost of eliminating some partially ordered
 * system, if the system does get fully ordered, it needs to obtain a
 * system with all rows ordered in a way so that the system is valid
 * (has no diagonal non-zeros), and even better, in a way so that the
 * cost of eliminating the system is minimized.
 *
 * The row-ordering routine produces a row-ordering that can be used
 * by the heuristic to make it's guess.
 *
 * This routine _must_ be very efficient, since it will be called a
 * large number of times.
 *
 * The recursive-ordering routine implemented here, is the slowest of
 * the two row-ordering routines, but it is the one that produces the
 * best results.
 *
 * It should return an ordering holding the following properties:
 *
 * 1) The ordering is either valid (no diagonal non-zeros), or
 *    an error is returned.
 * 2) The ordering is the one with the minimal cost, of all possible
 *    (valid) orderings. This routine is thus not influenced by
 *    any initial ordering of the rows, only by the ordering of the
 *    rows which are allready ordered (the strongly defined rows).
 * 3) An ordering of all sub-systems is possible (this is implied
 *    allready, by the fact that the ordering found is valid, thus
 *    holds a number of perfectly valid ordered syb-systems)
 *
 * $Revision: 1.4 $
 * */

#include "optimqr.h"
#include "recorder.h"
#include <stdio.h>
#include <string.h>
#include <assert.h>

/*
 * The cost of the resulting system + transform sequence:
 *  cost = G * N_GT + N_unz
 *
 * where     G : Cost of Givens transform relative to cost of fill-in
 *        N_GT : Number of givens-transforms needed
 *       N_unz : Number of upper-triangle non-zeros
 */

/* We hold all possible orderings in the global "orderings" list variable
 */
dmh_slist *orderings = NULL;

/* We do not want to actually run a full elimination on all the
 * orderings we find, since this would take a very long time for a
 * large number of orderings.  Therefore, we only calculate the cost
 * of a fixed number of orderings found. This number is given below in
 * the COUNTORDERS define.
 *
 * Since the number of orderings found is usually very large (depends
 * on the system, but 56x56 with 11% nonzeros often produces O(10^3)
 * possible orderings), just picking some number of them is the only
 * way we can really cope with this problem.
 *
 * We must keep all the orderings while searching for them, since the
 * number of valid orderings may suddenly decrease fast, and having
 * thrown away partial orderings earlier may leave us with no
 * orderings left at all.
 *
 * Only calculating cost for 20 orderings (as is the default for
 * COUNTORDERS) may seem vague.  However, it is a lot better than only
 * taking one ordering in account as the bipartite ordering routine
 * does.
 * Also, I found that on my main system, the 56x56 11%nz, that 20 was
 * a reasonable number. Results didn't improve substantially when the
 * number was increased above this default.
 * */
#define COUNTORDERS 20

/* We may in some situations be forced to discard partial orderings
 * although this may kill the entire recursive-ordering approach
 * later.  This can happen when we find that we have too many possible
 * orderings to keep in the pre-allocated memory buffers (see
 * optimqr.h).
 * 
 * Therefore, the macro MIN_SFREE tells the ordering routine how many
 * buffers it should make sure is free, at any time in the process of
 * ordering.
 *
 * This number should decrease as we get nearer to completing the set
 * of valid orderings.
 *
 * The macro can be experimented with, but it only has effect if there
 * are too many orderings to keep in memory.  Also consider increasing
 * the number of small buffers (by increasing the MAX_MEM define in
 * optimqr.h), or simply using the bipartite ordering strategy instead
 * (by defining ROWORDER_BIPORDER in the Makefile).  */
#define MIN_SFREE ((dim1-level)*dim1*dim1)

#define OMATRIX(sys,order,i,j) (matrixA->elements[(order)[(i)]]\
				[(sys)->column_ordering[(j)]])

/* The roworder() routine takes a partial solution as argument, and
 * overwrites this argument with the same solution with the not-yet
 * ordered rows ordered.
 *
 * The routine finds the set of possible orderings of the rows that
 * are not yet ordered in the system given as argument, and finds the
 * best of a subset of those orderings by calculating the cost of
 * elimination of each ordering in that subset.
 *
 * The best ordering is written into the system given as argument to
 * the routine.
 *
 * If no valid orderings can be found (either because of memory shortage,
 * or because there are none) the value  "-2" is returned. Otherwise
 * a "0" is returned.  
 */
int roworder(Tsolution* sys)
{
  int c;
  int norderings;
  int level;
  int bestcost;
  int orders_visited = 0;
  int orders_calculated = 0;
  stype* bestordering;
  dmh_slist* trvrs;

  /* for every level of sub-system, call combinations()
   * to have the combination list updated until we stand
   * with a set of combinations that is valid at sub-level n
   * (our sub-level when called)
   */

  for(level = 1; level <= dim1-sys->ordered_pairs; level++) {
    combinations(level, sys); 
    if(!orderings) {
      return -2;
    }
  }

  /* Count orderings */
  norderings = 0;
  for(trvrs = orderings; trvrs; trvrs = trvrs->aux_next)
    norderings ++;

  /* find best sequence and remember cost */
  bestcost = OVERMUCH;
  bestordering = NULL;
  for(trvrs = orderings; trvrs; trvrs = trvrs->aux_next) {
    if((orders_visited % (norderings/COUNTORDERS+1)) == 0) {
      int cost;
      orders_calculated++;
      for(c = 0; c < dim1; c++)
	sys->row_ordering[c] = trvrs->row_ordering[c];
      cost = find_sequence(sys, 1, NULL);
      if(cost < bestcost) {
	bestcost = cost;
	bestordering = trvrs->row_ordering;
      }
    }
    orders_visited++;
  } 
  assert(orders_visited == norderings);

  /* copy ordering to result */
  for(c = 0; c < dim1; c++)
    sys->row_ordering[c] = bestordering[c];

  /* delete list of orderings */
  for(trvrs = orderings; trvrs; trvrs = trvrs->aux_next)
    dmh_sfree(trvrs);

  orderings = NULL;

  return 0;
}


/* In order to more easily operate with mappings, this
 * tiny utility routine is defined.
 * It sets up the mapping  a ~ b  in the ordering buffer
 * "order".
 */
inline void makemap(stype* order, int a, int b)
{
  stype tmp;
  /* create mapping a ~ b in system */
  tmp = order[a];
  order[a] = order[b];
  order[b] = tmp;
}

/* The combinations() routine will generate all new valid
 * orderings for some level (the level corrosponds to the number
 * of rows that are ``fixed'', plus one. Initial level is 1).
 *
 * The orderings are returned in the "orderings" global variable.
 *
 * The routine is passed the system for which we produce orderings
 * as the sole argument.
 */
/*
 * level == 1 returns orderings that give a
 *            non-zero element in the lower-right
 *            corner, taking into account that some
 *            rows are allready reserved when we are
 *            asked to order a sub-system
 * level > 1  returns orderings that give a
 *            non-zero diagonal, using the orderings
 *            that are allready validated. We will
 *            delete any orderings that can not give
 *            a new valid ordering for the extra row
 */
void combinations(int level, Tsolution* sys)
{
  int targetpos = sys->ordered_pairs + level - 1;

  if(level == 1) {
    int row;

    assert(!orderings);

    /* step thru every row that is not reserved by caller,
     * and see if it gives a non-zero in bottom right corner
     */
    for(row = sys->ordered_pairs; row < dim1; row++) {
      /* ok, we can use this row. now see if it gives
       * a valid system
       */
      if(MATRIX(sys, row, targetpos)) {
	/* it does. */
	int c;
	dmh_slist* neworder = dmh_salloc();

	/* we should copy the row and column orderings from
	 * our source solution */
	for(c = 0; c < dim1; c++) 
	  neworder->row_ordering[c] = 
	    sys->row_ordering[c];
	
	/* set result */
	makemap(neworder->row_ordering, targetpos, row);

	/* chain into list of orderings */
	neworder->aux_next = orderings;
	orderings = neworder;
      }
    }

  } else {

    dmh_slist* trvrs;
    int localorders = 0;

    /* Find source-rows that satisfy is_valid when put
     * onto some existing ordering */

    /* Iterate thru existing solutions that have to be transformed
     * or anihilated */

    dmh_slist* prvs = NULL;

    trvrs = orderings; 

    while(trvrs) {
      dmh_slist *next = trvrs->aux_next;
      int ordering_used = 0;
      int row;
      int rempivot = -1;

      /* Iterate thru rows that are not allready taken by the current
       * ordering or the initial system */
      for(row = targetpos; row < dim1; row++)
	/* if row is relevant, use it */
	if(OMATRIX(sys,trvrs->row_ordering,row,targetpos)
	   && dmh_get_num_sfree() > MIN_SFREE) {

	  /* fit row into solution then */
	  localorders++;

	  if(!ordering_used) {

	    ordering_used = 1;

	    rempivot = row;

	  } else {
	    
	    int c;

	    /* construct a new ordering and chain
	     * it onto the list */
	    
	    dmh_slist* neworder = dmh_salloc();
	    
	    /* we should copy the row and column orderings from
	     * our source solution */
	    for(c = 0; c < dim1; c++)
	      neworder->row_ordering[c] = 
		trvrs->row_ordering[c];
	    
	    makemap(neworder->row_ordering, targetpos, row);

	    neworder->aux_next = orderings;
	    orderings = neworder;
	    
	  }

	}

      /* If ordering was never used, we must delete it, since
       * it is no longer valid */

      if(!ordering_used) {
	if(!prvs) {
	  /* delete first element */
	  orderings = trvrs->aux_next;
	  dmh_sfree(trvrs);	
	} else {
	  /* delete some other element */
	  prvs->aux_next = trvrs->aux_next;	    
	  dmh_sfree(trvrs);
	}
	trvrs = prvs;
      } else {
	/* Now make the mapping we couldn't do while processing trvrs */
	assert(rempivot != -1);
	makemap(trvrs->row_ordering, targetpos, rempivot);
      }

      prvs = trvrs;
      trvrs = next;
    }  
  }
}




