/* See the LICENSE file for conditions of usage */

#ifndef _OPTIMQR
#define _OPTIMQR

#include <stdio.h>

#ifdef INSTANTIATE
#define EXTERN extern
#else
#define EXTERN 
#endif

/*************************************************************/
/* BB search constants                                       */

/* Givens is 3 times as expensive as a fill-in */
#define GIVENS_COST 3
/* The penalty for not eliminating a non-zero is the same
   as the cost for one Givens and an entire row of fill-ins */
#define PENALTY (dim2+GIVENS_COST)
/* The number of live nodes to allow before increasing the min.
 * improvmenent. */
#define LIVE_SIZE 400000

/* The number of elements to allocate for the dmh subsystem */
/* The OptimQR optimizer uses an efficient memory allocation 
 * scheme to allow extremely fast allocation and de-allocation
 * of heavily used buffers. The scheme is also friendly to swapping,
 * so it should be possible to run an optimization on a machine
 * that does not have enough physical memory to hold the entire
 * footprint of the optimizer.
 * Unfortunately, the memory subsystem cannot dynamically expand
 * it's pre-allocated buffer space. Therefore, a sufficiently large
 * buffer must be allocated at startup. The approximate size in
 * bytes of this buffer space is given below.
 * A few hundred megabytes is usually fine.  If you specify too
 * little space, the optmizer will abort with an error message.
 */
/* MAX_MEM is the number of bytes to use in total for buffer space.
 * The buffer space is made up of two kinds of buffers. Large buffers
 * (that hold partial solutions to the optimization problem), and small
 * buffers, used by the row-ordering routines.
 *
 * The number of buffers is calculated like this:
 *
 * (number of small buffers) = (MAX_MEM - (size of large buffer) * DMHSIZE) / (size of small buffer)
 *
 * The number of large buffers is directly given by the DMHSIZE define below.
 * The number of small buffers is calculated from the MAX_MEM constraint.
 */
#ifdef ROWORDER_BIPORDER
#define MAX_MEM  (1024*1024*100)
#endif
#ifdef ROWORDER_RECORDER
#define MAX_MEM  (1024*1024*100)
#endif
#define DMHSIZE  (2*LIVE_SIZE)

/* Accept factor determines how many new branches we will
 * accept as live nodes.
 * Because the heuristics we use to discard branches from our search
 * tree aren't perfect, we will loose possibly good solutions if we
 * discard anything which the heuristic deems bad (too costly).  Therefore
 * we multiply the estimated cost-of-completion from the heuristic with
 * an acceptance factor  a<1.
 * The value 0.68 has proven itself (experimentally) to yield good
 * results.  A value much lower will severely increase the run-time of
 * the optimizer, and a value much higher will discard too many good
 * branches of the search tree.
 * We accept a branch if (g() + h()*ACCEPT_FACTOR) < f(best)  */
#define ACCEPT_FACTOR 0.68

/* define a cost so large we will never reach it */
#define OVERMUCH (dim1*dim2*(GIVENS_COST+1)*2)

/*************************************************************/
/* Datatype definitions                                      */

typedef struct {
  unsigned a,b;
  unsigned f,g;
} Tgivens;

typedef struct {
  unsigned f,g,h;
  unsigned short * f_history;
  unsigned short * g_history;
  unsigned short * h_history;
  int ordered_pairs;
  short * row_ordering;
  short * column_ordering;
} Tsolution;

typedef struct _dmh_list {
  struct _dmh_list* prev;
  Tsolution* solution;
  struct _dmh_list* next;
  struct _dmh_list* aux_next; /* not used by dmh */ 
} dmh_list;

typedef struct _Vertex {
  int id;
  int level;
  struct _Vertex* mate;
} Vertex;

typedef unsigned char stype;

unsigned dmh_get_num_sfree(void);
typedef struct _dmh_slist {
  struct _dmh_slist* prev;
#ifdef ROWORDER_RECORDER
  stype * row_ordering;
#endif
#ifdef ROWORDER_BIPORDER
  Vertex *vertex;
#endif
  struct _dmh_slist* next;
  struct _dmh_slist* aux_next; /* not used by dmh */ 
} dmh_slist;

typedef struct {
  char **elements;
  int   *row_nz;
  int   *column_nz;
  int  **wspdist;
} matrix_t;

/* Global variable definitions */

EXTERN double accept_factor;
EXTERN unsigned dim1,dim2;
EXTERN matrix_t *matrixA;  /* Initial system */

#define MATRIX(s,i,j) (matrixA->elements[(s)->row_ordering[(i)]][(s)->column_ordering[(j)]])
#define SMATRIX(m,s,i,j) ((m)->elements[(s)->row_ordering[(i)]][(s)->column_ordering[(j)]])

#define MAX(a,b) ((a)>(b) ? (a) : (b))
#define MIN(a,b) ((a)<(b) ? (a) : (b))

EXTERN Tsolution* orderedsolution; /* pointer to ordered ``solution'' */
EXTERN Tsolution* bestsolution; /* pointer to best solution */

/* Prototype definitions */
 
void fatal(char*);
void costs(Tsolution*,unsigned*,unsigned*);
void loadsys(char*);
void writesys(char*, Tsolution*);
void printsys(matrix_t*,char*);
void printgt(Tsolution*,char*);

matrix_t* alloc_matrix(void);
void ordersys(char**);
void apply_givens(matrix_t*, Tsolution*, unsigned, unsigned, unsigned);
void apply_fake_givens(matrix_t*, Tsolution*, unsigned, unsigned, unsigned);
unsigned spdist(char**, unsigned, unsigned);
unsigned wspdist(matrix_t*, unsigned, unsigned);
void init_bounds(Tsolution*);
void copy_matrix(matrix_t*,matrix_t*);
void initialize_solution(Tsolution*);
void copy_solution(Tsolution*,Tsolution*);

void gillespie_olesky(void);

void bb_search(void);

void dmh_initialize(unsigned);
dmh_list* dmh_alloc(void);
void dmh_free(dmh_list*);
dmh_slist* dmh_salloc(void);
void dmh_sfree(dmh_slist*);
void dmh_stat(void);

int find_sequence(Tsolution*,int,FILE*);
int heuristic(Tsolution*);
void stat_heuristic(void);

int roworder(Tsolution*);
int order_undefined(Tsolution*);


void seqout(char*, Tsolution*);

void write_status(char*,char*,...);

extern char outputdir[1024];

#endif

