/* NOTICE:
 *
 * This file is not an active part of the OptimQR software.  It was
 * used during the research that produced the OptimQR package, but
 * was discarded.
 *
 * The file is left here in order to perhaps inspire further work.
 *
 */

/*
 * rroworder.c  $Revision: 1.6 $
 *
 * This is a recursive row ordering routine, that utilizes dynamic
 * programming to cut down execution times.
 *
 * It memorizes solutions to sub-problems, and when called, it
 * will return the memorized solution if such one exists, or it
 * will call itself on a sub-problem, and memorize the solution
 * once it's known.
 *
 * We use a search trie to memorize the solutions. This results in
 * very fast seek times (O(k) where k is the number of equations in
 * our system). Unfortunately it also results in exorbitant memory
 * usage.
 *
 */

#include "optimqr.h"
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

typedef struct _rrtrie_t {
  int is_valid;
  short *row_ordering;
  short *column_ordering;
  struct _rrcol_t **children;  
} rrtrie_t;

typedef struct _rrcol_t {
  struct _rrtrie_t **children;
} rrcol_t;


int trie_nodes = 0;
int trie_total_nodes = 0;
rrtrie_t *trie_root = NULL;
rrtrie_t *trie_next_rnode = NULL;
rrcol_t *trie_next_cnode = NULL;
rrtrie_t *trie_rbuffer = NULL;
rrcol_t  *trie_cbuffer = NULL;
short *trie_order_buffer = NULL;
rrtrie_t **trie_rchild_buffer = NULL;
rrcol_t **trie_cchild_buffer = NULL;

int trie_hits = 0;
int trie_misses = 0;

/* The number of elements to grow the trie with */
#define TRIE_GROWTH 25000


/* Utility functions */

int ordersub(Tsolution* sys, int level);

/* Trie functions */

void grow_trie_buffer(void);

void insert_order(Tsolution* sys);
rrtrie_t *extract_order(Tsolution* sys);

rrtrie_t *trie_attach_rdummy(rrcol_t*, int);
rrcol_t *trie_attach_cdummy(rrtrie_t*, int);

rrtrie_t* trie_alloc_rnode(void);
rrcol_t* trie_alloc_cnode(void);

void trie_init_rnode(rrtrie_t*);
void trie_init_cnode(rrcol_t*);

void trie_order_sys(Tsolution*);

/*
 * The rroworder routine implementations
 */


int roworder(Tsolution* sys) 
{
  rrtrie_t *order;
  int i;
  int rc;

  /* If we never did anything with the search trie before,
   * we should initialize it...
   */
  if(!trie_root) grow_trie_buffer();


  /* Order incoming system some, for trie search */

  trie_order_sys(sys);

  /*
   * See if we have a stored solution to this problem 
   * If so, return that solution, otherwise, go down another level 
   */

  order = extract_order(sys);
  if(order) {
    trie_hits++;
    for(i = sys->ordered_pairs; i < dim1; i++) {
      assert(order->row_ordering[i] != -1);
      sys->row_ordering[i] = order->row_ordering[i];
      assert(order->column_ordering[i] != -1);
      sys->column_ordering[i] = order->column_ordering[i];
    }
    printf("H");
    return 0;
  } else {
    trie_misses++; printf("M");
  }
  
  rc = ordersub(sys, sys->ordered_pairs);
  if(!rc) {
    /*
     * We succeeded in finding a formerly unknown solution
     * This should be stored appropriately in the trie 
     */
    insert_order(sys);
  }

  write_status("RR-RO","Ordering rows. TH: %i, TM: %i", 
	       trie_hits, trie_misses);

  return rc;
}


int ordersub(Tsolution* sys, int level)
{
  int i;

  /* terminate if done */
  if(level == dim1) return 0;

  /* find nonzeros in the current column */
  for(i = level; i < dim1; i++) {
    if(MATRIX(sys,i,level)) {
      /* Every row with a nonzero in this diagonal element
       * is a candidate for being the current row.
       * We'll settle for the first that fits though, just to
       * cut a little of that O(n!) complexity...
       */
      int rc;
      int tmprow;
      tmprow = sys->row_ordering[level];
      sys->row_ordering[level] = sys->row_ordering[i];
      sys->row_ordering[i] = tmprow;
      /* Try sub-ordering */
      rc = ordersub(sys, level+1);
      /* If sub-system succeeded, return zero,
       * if it failed, continue.
       */
      if(!rc) {
	/* Return success */
	return 0;
      }
      /* pivot back the rows before we continue... */
      tmprow = sys->row_ordering[level];
      sys->row_ordering[level] = sys->row_ordering[i];
      sys->row_ordering[i] = tmprow;     
    }
  }

  /* we failed */
  return -1;
}


void grow_trie_buffer(void)
{
  int i;

  /* Allocate the trie node buffer (eventually again) */
  trie_nodes = TRIE_GROWTH;
  trie_total_nodes += TRIE_GROWTH;
  printf("Trie: Grow(%i)...", trie_total_nodes); fflush(stdout);

  trie_rbuffer = (rrtrie_t*)malloc(sizeof(rrtrie_t) * trie_nodes);
  if(!trie_rbuffer) fatal("Out of memory allocating trie rbuffer!");
  if(!trie_next_rnode) trie_next_rnode = trie_rbuffer;

  trie_cbuffer = (rrcol_t*)malloc(sizeof(rrcol_t) * trie_nodes);
  if(!trie_cbuffer) fatal("Out of memory allocating trie cbuffer!");
  if(!trie_next_cnode) trie_next_cnode = trie_cbuffer;

  /* Re-allocate the ordering buffer */
  trie_order_buffer = (short*)malloc(sizeof(short) * trie_nodes * dim1 * 2);
  if(!trie_order_buffer) fatal("Out of memory re-allocating trie order buffer!");
  
  /* Re-allocate the child buffers */
  trie_rchild_buffer = (rrtrie_t**)malloc(sizeof(rrtrie_t*) * trie_nodes * dim1);
  if(!trie_rchild_buffer) fatal("Out of memory re-allocating trie rchild buffer!");
  trie_cchild_buffer = (rrcol_t**)malloc(sizeof(rrcol_t*) * trie_nodes * dim1);
  if(!trie_cchild_buffer) fatal("Out of memory re-allocating trie cchild buffer!");

  /* Initialize new nodes */
  for(i = 0; i < trie_nodes; i++) {
    trie_rbuffer[i].row_ordering = trie_order_buffer + 2*i*dim1;
    trie_rbuffer[i].column_ordering = trie_order_buffer + (2*i+1)*dim1;
    trie_rbuffer[i].children = trie_cchild_buffer + i*dim1;
    trie_cbuffer[i].children = trie_rchild_buffer + i*dim1;
  }

  trie_next_rnode = trie_rbuffer;
  trie_next_cnode = trie_cbuffer;

  /* If this is the very first time we do anything, we should also
   * initialize the root node */
  if(!trie_root) {
    trie_root = trie_rbuffer;
    trie_init_rnode(trie_root);
    trie_next_rnode++;
  }

  printf(" TH: %i, TM: %i\n", trie_hits, trie_misses);
}


/*
 * Insert the solution from <sys>, that defines an ordering
 */
void insert_order(Tsolution* sys)
{
  rrtrie_t *rtrvrs;
  rrcol_t  *ctrvrs;
  int tlevel,i;

  rrtrie_t *newsolution;

  /* Allocate new solution (ordering) */
  newsolution = trie_alloc_rnode();
  assert(newsolution);
  trie_init_rnode(newsolution);

  /* Traverse from the root and down, until we find the empty 
   * spot where we fit in 
   */

  /* Now traverse the trie */
  /* When traversing, the nodes define row, column, row, col... 
   * in a last-to-first order, eg. the n'th row and column are at the
   * top level of the trie
   */
  rtrvrs = trie_root;
  for(tlevel = dim1-1; tlevel > sys->ordered_pairs; tlevel--) {
    rrtrie_t *rnxt;
    rrcol_t *cnxt;
    assert(rtrvrs);
    cnxt = rtrvrs->children[trie_root->row_ordering[tlevel]];
    if(!cnxt) {
      cnxt = trie_attach_cdummy(rtrvrs,trie_root->row_ordering[tlevel]);
    }
    ctrvrs = cnxt;
    assert(ctrvrs);
    rnxt = ctrvrs->children[trie_root->column_ordering[tlevel]];
    if(!rnxt) {
      int defok;
      int s;
      /* 
       * We cannot just make all intermediate nodes point at the
       * solution, since the solution may contain a row or column
       * which we do not (yet) have in our row-/column set.
       *
       * Remedy:  
       *  Check if n-subset of ordering only contains the n rows and
       * columns we have defined this far, if so, make this node
       * valid and make the row-/column ordering point to that of the
       * final ordering.  If not, mark this node invalid.
       */

      /* First, attach dummy */
      rnxt = trie_attach_rdummy(ctrvrs,trie_root->column_ordering[tlevel]);     

      /* Check if we have defined the sub-ordering so far */
      defok = 1;
      for(s = tlevel; (s < dim1) && defok; s++) {
	int fnd;
	int localdef = 0;
	for(fnd = tlevel; (fnd < dim1) && !defok; fnd++) {
	  /* If we can find the current element in the ordering
	   * it's fine */
	  if(sys->row_ordering[tlevel] == trie_root->row_ordering[s])
	    localdef = 1;
	}
	if(!localdef) defok = 0;	
      }

      /* Yes we have, remember this ordering */
      if(defok) {
	rnxt->row_ordering = newsolution->row_ordering;
	assert(rnxt->row_ordering);
	rnxt->column_ordering = newsolution->column_ordering;
	assert(rnxt->column_ordering);
	rnxt->is_valid = 1;
	printf(".");
      }
    }
    rtrvrs = rnxt;
    assert(rtrvrs);
  }

  /* At this very last level, we should create the entry */

  /* If we should allocate the entry, do so */
  if(!rtrvrs->children[trie_root->row_ordering[sys->ordered_pairs]]) {
    ctrvrs = rtrvrs->children[trie_root->row_ordering[sys->ordered_pairs]] = 
      trie_alloc_cnode();
    trie_init_cnode(ctrvrs);
  } else {
    ctrvrs = rtrvrs->children[trie_root->row_ordering[sys->ordered_pairs]];
  }

  if(!ctrvrs->children[trie_root->column_ordering[sys->ordered_pairs]]) {
    rtrvrs = ctrvrs->children[trie_root->column_ordering[sys->ordered_pairs]]
      = newsolution;      
  } else {
    rrtrie_t *old = ctrvrs->children[trie_root->column_ordering[sys->ordered_pairs]];
    assert(!old->is_valid);
    /* Transfer info. from old */
    newsolution->children = old->children;
    /* And setup the new */
    rtrvrs = newsolution;
    printf("ADD(O)");
  }

  /* Set values */
  for(i = 0; i < sys->ordered_pairs; i++) {
    rtrvrs->row_ordering[i] = -1;
    rtrvrs->column_ordering[i] = -1;
  }
  for(i = sys->ordered_pairs; i < dim1; i++) {
    rtrvrs->row_ordering[i] = sys->row_ordering[i];
    rtrvrs->column_ordering[i] = sys->column_ordering[i];
  }

  rtrvrs->is_valid = 1;

  /* done. */
  
}


rrtrie_t *extract_order(Tsolution* sys)
{
  rrtrie_t *rtrvrs;
  rrcol_t  *ctrvrs;
  int tlevel;

  /* Traverse from the root and down, until we find the empty 
   * spot where we fit in 
   */

  /* Now traverse the trie */
  /* When traversing, the nodes define row, column, row, col... */
  rtrvrs = trie_root;
  for(tlevel = dim1-1; tlevel >= sys->ordered_pairs; tlevel--) {
    if(!rtrvrs) return NULL;
    ctrvrs = rtrvrs->children[trie_root->row_ordering[tlevel]];
    if(!ctrvrs) return NULL;
    rtrvrs = ctrvrs->children[trie_root->column_ordering[tlevel]];
    if(!rtrvrs) return NULL;
  }

  /* So, now we're there! */
  assert(rtrvrs);

  return rtrvrs;
}


/* Attach a dummy entry to this root, using a key */
rrtrie_t* trie_attach_rdummy(rrcol_t* root, int key)
{
  rrtrie_t* newnode;

  /* init stuff */
  newnode = trie_alloc_rnode();

  trie_init_rnode(newnode);
  
  newnode->is_valid = 0;
  root->children[key] = newnode;

  return newnode;
}

rrcol_t* trie_attach_cdummy(rrtrie_t* root, int key)
{
  rrcol_t* newnode;

  /* init stuff */
  newnode = trie_alloc_cnode();

  trie_init_cnode(newnode);

  root->children[key] = newnode;

  return newnode;
}

rrtrie_t* trie_alloc_rnode(void) 
{
  if(!trie_next_rnode || trie_next_rnode == trie_rbuffer+trie_nodes) 
    grow_trie_buffer();
  return trie_next_rnode++;
}

rrcol_t* trie_alloc_cnode(void) 
{
  if(!trie_next_cnode || trie_next_cnode == trie_cbuffer+trie_nodes) 
    grow_trie_buffer();
  return trie_next_cnode++;
}

void trie_init_rnode(rrtrie_t* newnode)
{
  int i;
  assert(newnode->children);
  for(i = 0; i < dim1; i++)
    newnode->children[i] = NULL; 
}

void trie_init_cnode(rrcol_t* newnode)
{
  int i;
  assert(newnode->children);
  for(i = 0; i < dim1; i++)
    newnode->children[i] = NULL; 
}


void trie_order_sys(Tsolution* sys)
{
  int i,j,k;

  /* Order a list of rows and columns present in the sub-system
   * we seek an ordering for.
   * We use the trie root for the book-keeping, since the row/column
   * ordering entry tables are unused there...
   *
   * Since we know the exact limits of the values we're sorting,
   * we can sort the arrys in O(n) time.
   */

  /* Initialize */
  for(i = 0; i < dim1; i++) {
    trie_root->row_ordering[i] = -1;
    trie_root->column_ordering[i] = -1;
  }

  /* Insert */
  for(i = sys->ordered_pairs; i < dim1; i++) {
    int ro,co;
    ro = sys->row_ordering[i];
    assert(ro >= 0);
    assert(ro < dim1);
    assert(trie_root->row_ordering[ro] == -1);
    trie_root->row_ordering[ro] = i;
    co = sys->column_ordering[i];
    assert(co >= 0);
    assert(co < dim1);
    assert(trie_root->column_ordering[co] == -1);
    trie_root->column_ordering[co] = i;
  }

  /* Compress */
  j = k = dim1-1;
  for(i = dim1-1; i >=0; i--) {
    if(trie_root->row_ordering[i] != -1) {
      trie_root->row_ordering[j] 
	= sys->row_ordering[trie_root->row_ordering[i]];
      j--;
    }
    if(trie_root->column_ordering[i] != -1) {
      trie_root->column_ordering[k] 
	= sys->column_ordering[trie_root->column_ordering[i]];
      k--;
    }
  }

  assert(j == sys->ordered_pairs-1);
  assert(k == sys->ordered_pairs-1);
}


void rr_stat(void)
{
  printf("Recursive Row Ordering:\n");
  printf(" Level 0 Trie Hits:   %i\n", trie_hits);
  printf(" Level 0 Trie Misses: %i\n", trie_misses);
}
