/* loadsys $Revision: 1.9 $      */

/* This will load the system matrix from a file in the
 * following format:
 *     ------------------------------
 *     m n
 *     a00 a01 a02    ...    a0(n-1)
 *     .  .  .    
 *     a(m-1)0 a(m-1)1 a(m-1)2    ...    a(m-1)(n-1)
 *     ------------------------------
 * where aij is either ``0'' or ``x'' depending on whether
 * the element is zero or non-zero
 */

#include "optimqr.h"
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

/* The loadsys routine reads a system structure description
 * file, and sets up that system in the global "matrixA" variable.
 *
 * The routine takes one argument, the filename of the system
 * description file.
 *
 * The routine allocates space for the "matrixA" variable too,
 * and sets up the dim1 and dim2 variables, which hold the system
 * dimensions.
 */
void loadsys(char* filename) {
  FILE* file;
  unsigned line,elem;

  file = fopen(filename,"r");
  if(!file) fatal("Couldn't open input file");

  fscanf(file, "%u %u\n", &dim1, &dim2);
  if(dim1 != dim2 || dim1 <= 1 || dim2 <= 2)
    fatal("Invalid dimension on input system");

  matrixA = alloc_matrix();

  memset(matrixA->column_nz, 0, sizeof(int)*dim2);

  for(line = 0; line < dim1; line++)
    for(elem = 0; elem < dim2; elem++) {
      char a;
      fscanf(file, " %c ", &a);
      matrixA->row_nz[line] = 0;
      switch (a) {
      case '0' : matrixA->elements[line][elem] = 0; break;
      case 'x' : {
	matrixA->elements[line][elem] = 1; 
	matrixA->row_nz[line] ++;
	matrixA->column_nz[elem] ++;
	break;
      }
      default: 
	printf("When reading element %u,%u we found an invalid char ('%c')\n",
	       line+1,elem+1,a);
	fatal(NULL);
      }
    }

  /* Find weighted sparsity distance */
  for(line = 0; line < dim1; line++) {
    int i;
    for(i = 0; i < dim1; i++) {
      matrixA->wspdist[line][i] = 
	matrixA->wspdist[i][line] = -1;  /* invalidate */
      matrixA->wspdist[line][i] = 
	matrixA->wspdist[i][line] = wspdist(matrixA, i, line);
    }
  }

  fclose(file);
}

/* The writesys() routine writes an overview of a system into
 * a file, for later examination by the user/developer.
 * This routine is primarily a debugging aid.
 *
 * The routine takes two arguments, the filename in which to
 * write the system description, and the actual (partial) solution
 * to write.
 *
 * A 'x' is printed for a fixed non-zero, a '0' for a zero, and
 * a '?' for a non-zero in a row/column that is not yet fixed.
 */
void writesys(char* filename, Tsolution* solution) {
  FILE* file;
  unsigned line,elem;

  file = fopen(filename,"w");
  if(!file) fatal("Couldn't create output file");

  fprintf(file, "%u %u\n", solution->ordered_pairs, solution->ordered_pairs);
  
  fprintf(file, "  ");
  for(elem = 0; elem < dim2; elem++)
    fprintf(file, "%-3u",solution->column_ordering[elem]);
  fprintf(file, "\n");

  for(line = 0; line < dim1; line++) {
    fprintf(file, "%-3u", solution->row_ordering[line]);
    for(elem = 0; elem < dim2; elem++) {
      int ndx1,ndx2;
      int defined = 0;
      char elm;
      if(solution->ordered_pairs > line) defined = 1;
      ndx1 = solution->row_ordering[line];
      if(solution->ordered_pairs > elem) defined = 1;
      ndx2 = solution->column_ordering[elem];
      elm = matrixA->elements[ndx1][ndx2] ? (defined ? 'x' : '?') : '0';
      fprintf(file, "%c  ", elm);
    }
    fprintf(file, "\n");
  }

  fprintf(file, "f:     ");
  for(elem = 0; elem < dim2; elem++)
    fprintf(file, "%6u", solution->f_history[elem]);
  fprintf(file, "\ng:     ");
  for(elem = 0; elem < dim2; elem++)
    fprintf(file, "%6u", solution->g_history[elem]);
  fprintf(file, "\nh:     ");
  for(elem = 0; elem < dim2; elem++)
    fprintf(file, "%6u", solution->h_history[elem]);
/*   fprintf(file, "\nutil:  "); */
/*   for(elem = 0; elem < dim2; elem++) */
/*     fprintf(file, "%6u", solution->utility[elem]); */
  fprintf(file, "\n");

  fclose(file);
}
