/*
 * Heuristic  $Revision: 1.11 $
 *
 * The idea here is, to use a guess at what the cost
 * would be of completing the search thru some branch
 * of our search-tree.
 * We do not nessecarily return the lower bound on the
 * cost, but rather a good guess at that bound.
 * This hopefully helps us to limit the number of branches
 * to search thru, substantially.
 *
 * Stragety is to:
 * 1) Find a good element ratio for lower/upper triangles.
 * 2) Find a good Givens Transform Sequence.
 * 3) Use this cost as the heuristic.
 *
 * We sacrifice a lot of computing power in finding h(). It is,
 * however, still done in polynomial time.
 *
 */

#include "optimqr.h"
#include <stdio.h>
#include <assert.h>

int h_calls = 0;

/* The heuristic() function takes one argument, the node
 * for which to calculate the estimated cost-of-completion
 * lower bound cost, and returns this cost.
 *
 * The puspose of this function is, to estimate a lower bound
 * on the cost of factorizing the system given, _if_ the system
 * is completely ordered (optimized).  We use this to stop optimizing
 * systems (nodes/branches) at an early stage (in the search tree).
 *
 */
int heuristic(Tsolution* sys) 
{
  int h;

  if(sys->ordered_pairs == dim1) return 0;

  h_calls ++;

  /* Order the undefined space */
  h = order_undefined(sys);
  /* If ordering fails, it returns OVERMUCH, otherwise zero */
  if(h) return h;

  /* So, now we have a nicely ordered undefined space */
  
  /* Find a sequence of Givens Transforms that is independent
   * on the actual ordering of the rows, and gives a good
   * performance (low cost)
   */
  h = find_sequence(sys,1,NULL);

  /* Done 
   */

  if(sys->ordered_pairs >= 1) 
    sys->h_history[sys->ordered_pairs-1] = h;
 
  return h;
}


/* The order_undefined() function takes one argument, the system
 * it will do it's work on, and returns either 0 or the define OVERMUCH,
 * the latter in case of an error.
 *
 * The routine will produce a column- and row-ordering for the subspace
 * of the system that has not yet been ordered by the optimizer.
 *
 * The columns are simply ordered so that the columns with the most non-zeros
 * are positioned right-most. This is not necessarily optimal, but it's a good
 * guess at what could be optimal, and is thus a feasible solution to use in
 * the heuristic.
 *
 * After the columns are fixed, we cal roworder() to order the rows.  Depending
 * on what row-ordering strategy was chosen in the Makefile, some row-ordering
 * strategy will be used to order the rows to produce a valid system (one with
 * no zeros in the diagonal).
 */
int order_undefined(Tsolution* sys)
{
  int j,k;

  /* Order the undefined space using the following strategy:
   * 
   * 1) Produce a nice column-ordering
   * 2) Call roworder() to have the rows ordered
   */

/*   writesys("pre-order",sys);  */

/*   write_status("ST-OU", "Ordering undefined subspace"); */

  /* Proceed from lower-right to upper-left diagonal element */
  for(k = dim1-1; k >= sys->ordered_pairs; k--) {
    int bestnz = -1;
    int bestj = -1;
    int tmpcol;
    for(j = sys->ordered_pairs; j <= k; j++) {
      /* Find candidates, recording the number of non-zeros they hold,
       * and keeping track of the best candidate
       */
      int nzcount; 
      nzcount = matrixA->column_nz[sys->column_ordering[j]];
      /* Keep track of better candidates */
      if(nzcount > bestnz) {
	bestnz = nzcount;
	bestj = j;
      }
    }
    
    /* There always is a candidate */

    assert(bestnz > -1);

    /* Since we have a candidate, pivot columns */

    assert(0 <= k && k < dim1);
    assert(0 <= bestj && bestj < dim2);
    /* Who are the k'th column now ? */
    tmpcol = sys->column_ordering[k];
    /* Make the j'th column the k'th column */
    sys->column_ordering[k] = sys->column_ordering[bestj];
    /* And don't forget to make the old k'th columns the j'th column */
    sys->column_ordering[bestj] = tmpcol;
  }
  
  /*
   * Now we have some sort of a column ordering.
   * Call roworder() to have the rows ordered 
   */

/*   writesys("mid-order",sys);  */

  if(roworder(sys) == -2) {
    /* we failed */
    return OVERMUCH;
  }

/*   writesys("post-order",sys); */

  return 0;
}

/* The stat_heuristic() routine prints a summary of statistics
 * from the heuristic function.
 */
void stat_heuristic(void)
{
  printf("Heuristic:\n");
  printf(" %i calls\n", h_calls);
  printf(" abort value is %i\n", OVERMUCH);
}

