/* NOTICE:
 *
 * This file is not an active part of the OptimQR software.  It was
 * used during the research that produced the OptimQR package, but
 * was discarded.
 *
 * The file is left here in order to perhaps inspire further work.
 *
 */

/*
 * groworder.c  $Revision: 1.1 $
 *
 * This routine will produce some row-ordering that holds
 * non-zeros in all diagonal elements.
 *
 * Lemma:
 *  If the system matrix is non-singular, any column ordering
 * will have at least one row-ordering with non-zeros in all
 * diagonal elements.
 *
 * Proof: Later.
 *
 * The strategy is:
 *
 * 1) Build directed graph containing all possible row-pivots.
 *    We should not consider the rows which allready have a non-zero
 *    in the diagonal element.
 * 2) Find strongly connected components in the graph
 * 3) All strongly connected components are transferred to their own
 *    non-directed graph.
 * 4) We can now easily pivot rows, as dictated by each of the non-directed
 *    graphs.
 *
 *  If we have k non-zeros in an average row, and n rows, this algorithm
 *  will run in O(k*n) time.
 *  Since k will probably be near constant for all the arial pollution
 *  simulation cases, this algorithm in fact runs in near linear time
 *  with n.
 *
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include "optimqr.h"

#define WHITE 0
#define GRAY  1
#define BLACK 2

#define DFMATRIX(i,j) ((transpose) ? MATRIX((sys),(j),(i)) \
		       : (MATRIX((sys),(i),(j))))


void df_visit(Tsolution* sys, int transpose, int vertex);

int df_f_compare(const void*, const void*);

int *df_color = NULL;      /* color */
int *df_backtrack = NULL;  /* pi    */
int *df_fwdtrack = NULL;   /* pi^-1 */
int *df_discover = NULL;   /* d     */
int *df_finish = NULL;     /* f     */
int df_time;               /* global time */
int *df_roots = NULL;      /* roots in depth first forest */
int df_n_roots;            /* number of roots */

typedef struct {
  int f;
  int u;
} dfo;

dfo *df_ordering = NULL; 

void roworder(Tsolution* sys) 
{
  int u,s;

  if(!df_color) {
    df_color = (int*)malloc(sizeof(int) * dim2);
    df_backtrack = (int*)malloc(sizeof(int) * dim2);
    df_fwdtrack = (int*)malloc(sizeof(int) * dim2);
    df_discover = (int*)malloc(sizeof(int) * dim2);
    df_finish = (int*)malloc(sizeof(int) * dim2);
    df_ordering = (dfo*)malloc(sizeof(dfo) * dim2);
    df_roots = (int*)malloc(sizeof(int) * dim2);
  }
  if(!df_color || !df_backtrack || !df_fwdtrack || !df_discover
     || !df_finish || !df_ordering || !df_roots)
    fatal("roworder(): DFS buffers not allocated properly!\n");

  /* We find strongly connected components from the following algorithm:
   *  1) Run a depth first search on G to compute finishing times f(u) for
   *     each vertex u
   *  2) Run a depth first search on G^T, but consider vertices in the
   *     order of decreasing f().
   *  3) Every tree in the depth-first forest from step 2 is now
   *     a set of strongly connected components.
   */

  /* 
   * 1) Run a depth first search on G 
   */

  printf("First DFS\n");
  
  for(u = sys->ordered_pairs; u < dim2; u++) {
/*     if(!MATRIX(sys,u,u)) { */
      df_color[u] = WHITE;
      df_backtrack[u] = -1;
    }
  df_time = 0;
  for(u = sys->ordered_pairs; u < dim2; u++) 
    /*     if(!MATRIX(sys,u,u)) { */ {
      if(df_color[u] == WHITE) df_visit(sys,0,u);
    }  

  /*
   * order f() for use in next DFS 
   */

  printf("Order f[]\n");

  for(u = 0; u < dim2; u++) {
    df_ordering[u].f = -1;
    df_ordering[u].u = u;
  }
  for(u = sys->ordered_pairs; u < dim2; u++)
/*     if(!MATRIX(sys,u,u))  */
      df_ordering[u].f = df_finish[u];

  qsort(df_ordering, dim2, sizeof(dfo), df_f_compare);

  /* 
   * 2) Run depth first search on G^T, but ordered after
   *    finish times.
   */

  printf("Second DFS\n");

  df_n_roots = 0;

  for(u = sys->ordered_pairs; u < dim2; u++) 
    /*     if(!MATRIX(sys,u,u)) { */ {
      df_color[u] = WHITE;
      df_backtrack[u] = -1;
      df_fwdtrack[u] = -1;
    }
  df_time = 0;
  for(s = 0; s < dim2; s++)
    if(df_ordering[s].f > -1) {
      u = df_ordering[s].u;
      assert(u >= sys->ordered_pairs);
/*       assert(!MATRIX(sys,u,u)); */
      if(df_color[u] == WHITE) {
	df_visit(sys,0,u);    
	/* If tree contains more than the root, register
	 * as a strongly_connected_component */
	if(df_fwdtrack[u] != -1)
	  df_roots[df_n_roots++] = u;
      }
    } 

  printf("Found %i strongly connected components \n", df_n_roots);

  if(df_n_roots == 0) return;

  /*
   * Every tree in df_roots[] is a strongly connected component
   */

  /*
   * The pivot strategy is as follows:  
   *  1)  We convert the directed graphs that represent the strongly
   *      connected components, into non-directed graphs.
   *  2)  Theese graphs are then traversed, and the rows they specify
   *      are pivoted.
   *
   *  There are possibly more than one ordering here, but any one
   *  will satisfy the non-zero-diagonal requirement.
   *
   */

  for(u = 0; u < df_n_roots; u++) {
    int i,lsti;
    printf("SCC(%i):",u);
    lsti = -1;
    i = df_roots[u];
    while(i != -1) {
      printf(" %i", i);
      lsti = i;
      i = df_fwdtrack[i];
    }
    printf("\n");
    /* lsti holds the last element... */
    i = lsti;
    while(i != -1 && df_backtrack[i] != -1) {
      int tmprow,dst;
      dst = df_backtrack[i];
      assert(df_fwdtrack[dst] == i);
      /* pivot rows  i and df_backtrack[i]  */
      tmprow = sys->row_ordering[dst];
      sys->row_ordering[dst] = sys->row_ordering[i];
      sys->row_ordering[i] = tmprow;
      /* Done, go on */
      i = df_backtrack[i];
    }
  }
}

void df_visit(Tsolution* sys, int transpose, int vertex)
{
  int u;
  df_color[vertex] = GRAY;
  df_discover[vertex] = ++df_time;
  for(u = sys->ordered_pairs; u < dim2; u++) 
    if(DFMATRIX(vertex,u)) {     /* Explore edge */
      if(df_color[u] == WHITE) {
	df_backtrack[u] = vertex;
	df_fwdtrack[vertex] = u;
	printf("%i connects to %i\n",vertex,u);
	df_visit(sys, transpose, u);
      }
    }
  df_color[vertex] = BLACK;
  df_finish[vertex] = ++df_time;
}


int df_f_compare(const void* a, const void* b)
{
  int da = ((dfo*)a)->f;
  int db = ((dfo*)b)->f;
  if(da > db) return -1;
  if(db > da) return 1;
  return 0;
}
