/*
 * $Source: /wayward/homes/moore/src/hence2/master/RCS/trace.c,v $
 * $Revision: 1.7 $
 * $Date: 1994/06/11 21:07:23 $
 * $Author: moore $
 */

#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <fcntl.h>

/*
 * define timeb structure for machines that don't have it
 * (prob. should have used gettimeofday(), but wtf...)
 */

#if defined(IMA_CRAY) || defined(IMA_SGI) || defined(IMA_SGI5)
struct timeb {
    time_t time;
    unsigned short millitm;
    short timezone;
    short dstflag;
};
#else
#include <sys/timeb.h>
#endif

#include "std.h"
#include "rb.h"
#include "dlist.h"
#include "hence.h"
#include "../htool/tfnew.h"
#include "tfwrite.h"

static int trace_on = 0;
static int trace_params = 0;
static int trace_msgs = 0;

int tf_fd = -1;
int tf_sequence = 0;

/*
 * these counts are useful when writing machine state to the trace file.
 * (since we only write out a tiny bit of machine state per record)
 */

int num_machines = 0;
int num_subrs = 0;
int num_nodes = 0;
int num_cells = 0;
int num_stats = 0;
struct stats {
    int num_running;
    int num_idle;
} *per_cell_stats;

Graph our_graph;		/* private copy of the global graph */

set_trace_msgs()
{
    trace_msgs = 1;
}

trace_flush()
{
}

set_trace_on(i)
int i;
{
    trace_on = i;
}

set_trace_params(i)
int i;
{
    trace_params = i;
}

set_trace_file(fname)
char *fname;
{
    if (fname == CNULL || strcmp(fname, "-") == 0)
	tf_fd = -1;
    else if (strcmp(fname, "-2") == 0)
	tf_fd = 2;
    else if (strcmp(fname, "-1") == 0)
	tf_fd = -1;
    else if (*fname == '-' && isdigit (fname[1]) && fname[2] == '\0') {
	tf_fd = fname[1] - '0';
    }
    else {
	tf_fd = open(fname, O_RDWR | O_CREAT | O_TRUNC, 0666);
	if (tf_fd < 0) {
	    fprintf(stderr, "ERROR: Can't open trace file: %s\n", fname);
	    exit(1);
	}
    }
}


#if defined(IMA_RIOS) || defined(IMA_CRAY) || defined(IMA_SGI) || \
    defined(IMA_ALPHA) || defined(IMA_RS6K) || defined(IMA_SGI5)
/*
 * ftime() emulation for systems that don't have it, but do have
 * gettimeofday ().  (which is probably everyone by now).
 */

#include <sys/time.h>

static void
ftime (x)
struct timeb *x;
{
    struct timeval tv;
    struct timezone tz;

    gettimeofday (&tv, &tz);
    x->time = tv.tv_sec;
    x->millitm = tv.tv_usec / 1000;
}
#endif

/*
 * get a pointer to the named node
 *
 * XXX inst is ignored because additional instances are never linked
 * into the graph once we get started.
 */

static Node
find_node (G, id, inst)
Graph G;
int id, inst;
{
    Rb_node r;
    int fnd = 0;

    r = rb_find_ikey_n (G->nlist, id, &fnd);
    if (fnd)
	return (Node) r->v.val;
    return (Node) NULL;
}

static Node
find_node_by_index (G, index)
Graph G;
int index;
{
    Rb_node r;

    rb_traverse (r, G->nlist) {
	Node n = (Node) r->v.val;
	if (n->trace_file_index == index)
	    return n;
    }
    return (Node) NULL;
}

/*
 * get a pointer to the named machine
 */

int
find_machine_number (G, name)
Graph G;
char *name;
{
    Rb_node r;
    int fnd = 0;

    if (name == CNULL)
	return -1;

    r = rb_find_key_n (G->machines, name, &fnd);
    if (fnd) {
	Machine m = (Machine) r->v.val;
	return m->trace_machine_number;
    }
    return -1;
}


/*
 * find a pointer to the named subr
 */

static Sub
find_sub (G, name)
Graph G;
char *name;
{
    Rb_node r;
    int fnd = 0;

    if (name == NULL || *name == NULL)
	return (Sub) NULL;

    r = rb_find_key_n (G->subroutines, name, &fnd);
    if (fnd) {
	Sub s = (Sub) r->v.val;
	return s;
    }
    return (Sub) NULL;
}

/*
 * given the name of a subr, find its trace number
 */

int
find_subr_number (G, name)
Graph G;
char *name;
{
    Sub s = find_sub (G, name);
    return s ? s->trace_sub_number : -1;
}


/*
 * based on the sequence number, figure out which part of the machine
 * stats to write out for this record
 */

static void
update_machine_state (e)
struct event_rec *e;
{
    int q;
    Rb_node r;

    if (!trace_on)
	return;

    e->sequence = tf_sequence++;
    q = e->sequence % num_stats;

    if (q < num_machines) {
	/* q is machine number */

	rb_traverse (r, our_graph->machines) {
	    Machine m = (Machine) r->v.val;

	    if (m->trace_machine_number == q) {
		e->num_running = m->num_running;
		e->num_idle = m->num_idle;
		return;
	    }
	}
	goto not_found;
    }
    q -= num_machines;

    if (q < num_subrs) {
	/* q is subr number */

	rb_traverse (r, our_graph->subroutines) {
	    Sub s = (Sub) r->v.val;

	    if (s->trace_sub_number == q) {
		e->num_running = s->num_running;
		e->num_idle = s->num_idle;
		return;
	    }
	}
	goto not_found;
    }
    q -= num_subrs;

    if (q < num_nodes) {
	/* q is node number */

	Node n = find_node_by_index (our_graph, q);

	if (n) {
	    e->num_running = n->num_running;
	    e->num_idle = n->num_idle;
	    return;
	}
	goto not_found;
    }
    q -= num_nodes;

    if (q < num_cells) {
	/* q is cell number */
	e->num_running = per_cell_stats[q].num_running;
	e->num_idle = per_cell_stats[q].num_running;
	return;
    }

 not_found:
    fprintf (stderr, "can't happen: q is too big (%d)\n",
	     (tf_sequence - 1) % num_stats);
    bail (CNULL);
}


/*
 * write out a trace event.  add a sequence number, current time,
 * and horizontally-encoded machine state information.
 */

static int
write_event (e)
struct event_rec *e;
{
    struct timeb now;

    ftime (&now);
    e->seconds = now.time;
    e->msec = now.millitm;
    update_machine_state (e);
    return tfwrite_event_rec (tf_fd, e);
}

void trace_event ();

trace_enroll(proc, inst)
char *proc;
int inst;
{
    struct event_rec e;

    if (!trace_on)
	return;
    
    memset (&e, '\0', sizeof (e));
    e.event_type = EV_START;
    write_event (&e);
}

/*
 * generate a trace event to mark the change in the state of a node:
 * READY -> RUNNING-> DONE -> DEAD
 * also maintain aggregate totals for machines, subrs, and all
 * instances of this particular node.
 */

void
trace_node (n, G)
Node n;
Graph G;
{
    struct event_rec e;
    Node n0;
    Sub s;
    int run_incr = 0;
    int idle_incr = 0;

    if (!trace_on)
	return;

    memset (&e, '\0', sizeof (e));
    e.event_type = EV_STATE_CHANGE;
    e.node_number = n->id;
    e.inst_number = n->inst;
    e.state = n->state;
    e.machine_number = n->mptr ? n->mptr->trace_machine_number : -1;

    switch (n->state) {
    case RUNNING:
	run_incr = 1;
	idle_incr = 0;
	break;
    case DONE:
	run_incr = -1;
	idle_incr = 1;
	break;
    case DEAD:
	run_incr = 0;
	idle_incr = -1;
	break;
    default:
	run_incr = 0;
	idle_incr = 0;
	break;
    }
    
    /*
     * update per-node counts
     */
    n0 = find_node (G, n->id, 0);
    if (n0) {
	n0->num_running += run_incr;
	n0->num_idle += idle_incr;
    }

    /*
     * update per subr counts
     */
    s = find_sub (G, n->sub_name);
    if (s) {
	s->num_running += run_incr;
	s->num_idle += idle_incr;
    }

    /*
     * update per-machine counts
     */
    if (n->mptr) {
	n->mptr->num_running += run_incr;
	n->mptr->num_idle += idle_incr;
    }

    /*
     * update per-cell counts
     */
    if (s && n->mptr) {
	int sub_number = s->trace_sub_number;
	int machine_number = n->mptr->trace_machine_number;
	int cell = sub_number + machine_number * num_subrs;

	per_cell_stats[cell].num_running += run_incr;
	per_cell_stats[cell].num_idle += idle_incr;
    }
    write_event (&e);
}

/*
 * record the expansion of a fanout or pipe
 */

trace_fan_or_pipe(n, low, high)
Node n;
int low, high;
{
    struct event_rec e;

    if (!trace_on)
	return;

    memset (&e, '\0', sizeof (e));
    e.event_type = EV_VNODE;
    e.node_number = n->id;
    e.inst_number = n->inst;
    e.state = low <= high;
    e.lower_bound = low;
    e.upper_bound = high;
    write_event (&e);
}

/*
 * trace the execution of a conditional node
 */

trace_cond(n, cond)
Node n;
int cond;
{
    struct event_rec e;

    if (!trace_on)
	return;

    memset (&e, '\0', sizeof (e));
    e.event_type = EV_VNODE;
    e.node_number = n->id;
    e.inst_number = n->inst;
    e.state = cond;
    write_event (&e);
}

/*
 * trace a loop expansion (gee, it's just like trace_cond()...)
 */

trace_loop(n, cond)
Node n;
int cond;
{
    struct event_rec e;

    if (!trace_on)
	return;

    memset (&e, '\0', sizeof (e));
    e.event_type = EV_VNODE;
    e.node_number = n->id;
    e.inst_number = n->inst;
    e.state = cond;
    write_event (&e);
}

    
/*
 * write out a FINISH record indicating we're done with the trace
 */

trace_done(t)
int t;
{
    struct event_rec e;

    if (!trace_on)
	return;

    memset (&e, '\0', sizeof (e));
    e.event_type = EV_FINISH;
    write_event (&e);
}

/*
 * write out all of the 'header' stuff of the trace file...
 * lists of machines, subrs, nodes, links, costs, and the
 * event section header.
 */

void
trace_write_file_header (G)
Graph G;
{
    int i;
    Rb_node r1, r2;

    if (!trace_on)
	return;

    /* 
     * because of the way we output machine stats in the trace file,
     * we need to poke through the graph at random times.  So we
     * keep a private pointer to the the global graph here.
     *
     * YEEECH!
     */
    our_graph = G;

    if (tf_fd < 0)
	return;

    /*
     * 1. write out file header section
     */
    tfwrite_section_hdr (tf_fd, SECT_HDR, 1, sizeof (struct hdr_rec), 0);
    tfwrite_hdr_rec (tf_fd, "2.0");	/* XXX executioner version # */

    /*
     * 2. write out machines, and initialize 'num_machines'
     */
    num_machines = 0;
    rb_traverse(r1,G->machines) {
	Machine m = (Machine) r1->v.val;
	m->trace_machine_number = num_machines++;
    }
    tfwrite_section_hdr (tf_fd, SECT_MACHINES, num_machines,
			 sizeof (struct machine_rec), 0);
    
    rb_traverse(r1, G->machines) {
	Machine m = (Machine) r1->v.val;
	tfwrite_machine_rec (tf_fd, r1->k.key);
    }
    
    /*
     * 3. write out subrs, and initialize 'num_subrs'
     */
    num_subrs = 0;
    rb_traverse(r1,  G->subroutines) {
	Sub s = (Sub) r1->v.val;
	s->trace_sub_number = num_subrs++;
    }
    tfwrite_section_hdr (tf_fd, SECT_SUBRS, num_subrs,
			 sizeof (struct subr_rec), 0);
    rb_traverse(r1,  G->subroutines) {
	Sub s = (Sub) r1->v.val;
	tfwrite_subr_rec (tf_fd, r1->k.key);
    }

    /*
     * 4. write out nodes, and initialize 'num_nodes'
     */
    num_nodes = 0;
    rb_traverse (r1,  G->nlist) {
	Node n = (Node) r1->v.val;
	n->trace_file_index = num_nodes++;
    }
    tfwrite_section_hdr (tf_fd, SECT_NODES, num_nodes,
			 sizeof (struct node_rec), 0);
    rb_traverse (r1,  G->nlist) {
	Node n = (Node) r1->v.val;
	tfwrite_node_rec (tf_fd,
			  n->id,
			  n->node_type,
			  n->xy.set ? n->xy.x : 0,
			  n->xy.set ? n->xy.y : 0,
			  find_subr_number (G, n->sub_name));
    }

    /*
     * initialize other machine totals while we're at it.
     * also allocate and initialize per-cells stats
     */
    num_cells = num_machines * num_subrs;
    num_stats = num_machines + num_subrs + num_nodes + num_cells;
    per_cell_stats = talloc (struct stats, num_cells);
    memset (per_cell_stats, '\0', num_cells * sizeof (struct stats));

    /*
     * 5. write out links
     */
    i = 0;
    rb_traverse (r1,  G->nlist) {
	Node n = (Node) r1->v.val;
	rb_traverse (r2, n->children) {
	    i++;
	}
    }
    tfwrite_section_hdr (tf_fd, SECT_LINKS, i, sizeof (struct link_rec), 0);
    rb_traverse (r1,  G->nlist) {
	Node n1 = (Node) r1->v.val;
	rb_traverse (r2, n1->children) {
	    Node n2 = (Node) r2->v.val;
	    tfwrite_link_rec (tf_fd,
			      (unsigned short) n1->id,
			      (unsigned short) n2->id);
	}
    }

    /*
     * 6. write out costs
     */
    i = 0;
    rb_traverse (r1,  G->subroutines) {
	Sub s = (Sub) r1->v.val;
	rb_traverse (r2, s->costs)
	    ++i;
    }
    tfwrite_section_hdr (tf_fd, SECT_COSTS, i, sizeof (struct cost_rec), 0);
    rb_traverse (r1,  G->subroutines) {
	Sub s = (Sub) r1->v.val;
	rb_traverse (r2, s->costs) {
	    Cost c = (Cost) r2->v.val;

	    tfwrite_cost_rec (tf_fd,
			      find_machine_number (G, c->m->name),
			      s->trace_sub_number, 	  /* subr # */
			      c->c);			  /* cost */
	}
    }
    /*
     * 7. write out header for other events
     */
    tfwrite_section_hdr (tf_fd, SECT_EVENTS, 0, sizeof (struct event_rec), 0);
}


/*
 * BELOW IS VESTIGAL STUFF THAT NEVER REALLY GOT USED...MAYBE SOMEDAY.
 */

trace_param_begin (n)
Node n;
{
#if 0
    if (!trace_params)
	return;
    fprintf(TF, "---");
    fprint_node_id(TF, n);
    fprintf(TF, "Parameters:\n");
#endif
}

trace_param(p)
Param p;
{
#if 0
    if (!trace_params)
	return;
    fprintf(TF, "  ");
    fprint_param(TF, p);
    fprintf(TF, "\n");
#endif
}

trace_arrp(arrp)
Dlist arrp;
{
#if 0
  int i, size;
  Dlist d;
  Node a;

  i = 0;
  size = 10;
  if (!trace_params) return;
  if (dl_empty(arrp)) fprintf(TF, "No array ancestor list\n");
  else {
    fprintf(TF, "Array ancestors:");
    dl_traverse(d, arrp) {
      a = (Node) d->val;
      if (i != 0) fprintf(TF, ",");
      if (i == size) {
        fprintf(TF, "\n    ");
        size += 10;
      }
      fprintf(TF, " %d/%d", a->id, a->inst);
    }
  }
  fprintf(TF, "\n");
#endif
}
      
static int tac = 0;
static Node tn, tnc;


trace_anc_list(n, nc)
Node n, nc;
{
#if 0
  tac = 1;
  fprint_node_id(stdout, n);
  fprintf(stdout, "Updating anc list of");
  fprint_node_id(stdout, nc);
  fprintf(stdout, "\n");
  (void) fflush(stdout);
  tn = n;
  tnc = nc;
#endif
}

trace_anc(a, l)
Node a;
int l;
{
#if 0
    if (!tac)
	return;
    fprintf(stdout, "  Anc:%d/%d, Level%d, AncList: ", a->id, a->inst, l);
    print_anc_list(tnc->anc);
#endif
}
 
trace_anc_off()
{
    tac = 0;
}

#ifdef PVM3
trace_msg_req(topr, totid, top, myp)
char *topr;
int totid, top, myp;
{
  if (!trace_msgs) return;

  fprintf(stderr, "%sReq. P%d from %x (%s) -> my P%d\n",
          sl_id(), top, totid, topr, myp);
}
#else
trace_msg_req(topr, toinum, top, myp)
char *topr;
int toinum, top, myp;
{
  if (!trace_msgs) return;

  fprintf(stderr, "%sReq. P%d from %s/%d -> my P%d\n",
          sl_id(), top, topr, toinum, myp);
}
#endif

#ifdef PVM3
trace_msg_send(myp, top, totid)
int totid, top, myp;
{
  if (!trace_msgs) return;

  fprintf(stderr, "%sSend. my P%d to [%x] -> their P%d\n",
          sl_id(), myp, totid, top);
}
#else
trace_msg_send(myp, top, topr, toinum)
char *topr;
int toinum, top, myp;
{
  if (!trace_msgs) return;

  fprintf(stderr, "%sSend. my P%d to %s/%d -> their P%d\n",
          sl_id(), myp, topr, toinum, top);
}
#endif
