/*
 * $Source: /wayward/homes/moore/src/hence2/master/RCS/execute.c,v $
 * $Revision: 1.5 $
 * $Date: 1994/06/11 20:57:04 $
 * $Author: moore $
 */

#include <stdio.h>
#include "std.h"
#include "dlist.h"
#include "rb.h"
#include "hence.h"
#include "htypes.h"
#include "rbhelp.h"
#include "eval.h"

#ifdef PVM3
#include <pvm3.h>
#endif

#ifdef IMA_ALPHA
#define MAX_IDS_IN_TABLE    1000
static Node unique_id_table[MAX_IDS_IN_TABLE];
static int next_unique_id = 0;
static int unique_id (Node n);
static Node node_from_unique_id (int i);
#endif

static Dlist Readyq; /* This is set to be G->readyq.  It's gross, but
                        I don't feel like passing G to every subroutine */

static fire_up ();		/* forward declaration */
static receive_finished_node ();/* forward declaration */
static finish_up_node ();	/* forward declaration */
static do_unk_type_error ();	/* forward declaration */

make_ready(n, G)
Node n;
Graph G;
{
  n->state = READY;
  trace_node(n, G);
  dl_insert_b_nd(Readyq, n);
}

static make_running(n, G)
Node n;
Graph G;
{
  n->state = RUNNING;
  reduce_params(n);
  fire_up(n, G);
}

#ifdef PVM3
/*
 * hack environment to set search path, DISPLAY for X debugging
 */

static void
hack_environment ()
{
    char **new_environ;
    extern char **environ;
    int i, j;

    for (i = 0; environ[i] != NULL; ++i);
    new_environ = (char **) malloc ((i + 3) * sizeof (char *));
    i = 0;
    j = 0;
    new_environ[j++] = "PVM_EXPORT=DISPLAY";
    new_environ[j++] = "PVM_PATH=pvm3/bin/%";
    for (i = 0; environ[i] != NULL; ++i)
	new_environ[j++] = environ[i];
    new_environ[j++] = NULL;
    environ = new_environ;
}
#endif /* PVM3 */

/*----------------------------------------------------------------------*/
/* Execute is the routine responsible for executing the HENcE graph.
 * It performs the rewriting of special-purpose nodes, fires up normal
 * nodes on the correct machines, and then waits for the nodes to finish,
 * so that it can continue processing */

execute(G)
Graph G;
{
  Rb_node r;
  Dlist d;
  Node n;
  int t;
  
  Readyq = G->readyq;
  set_master(G->alive);

  /* emit a list of machines to trace file */
#ifdef NEW_TRACE
  trace_write_file_header (G);
#else
  rb_traverse(r,G->machines) trace_machine (r->k.key);
#endif

  /* Fire up pvm */

  G->pvmproc = "master";
#ifdef PVM3
  /*
   * hack environment to set search path, DISPLAY for X debugging
   */
  hack_environment ();

  /*
   * XXX do we need to register our tid as "master" so others
   * (maybe the HeNCE executioner) can find us?
   */
  G->pvmtid = pvm_mytid ();
  if (G->pvmtid < 0) {
      fprintf (stderr, "Can't talk to pvmd: pvm_mytid() returned %d [%s]\n",
	       G->pvmtid, pvm_strerror (G->pvmtid));
      nice_bail (CNULL);
  }
  pvm_setopt (PvmOutputTid, G->pvmtid);
  pvm_setopt (PvmOutputCode, HENCE_CHILD_OUTPUT);
  trace_enroll (G->pvmproc, G->pvmtid);
  rb_traverse (r, G->machines) {
    int x;

    if ((x = add_machine (r->k.key)) < 0) {
      fprintf (stderr, "ERROR: can't add machine %s to PVM: %s\n",
	       r->k.key, pvm_strerror (x));
      pvm_exit ();
      bail (CNULL);
    }
  }
#else
  G->pvminum = enroll(G->pvmproc);
  if (G->pvminum < 0) pvm_enroll_error(G->pvmproc, G->pvminum);
  trace_enroll(G->pvmproc, G->pvminum);
#endif
  pvmon();

  /* Put the nodes from the heads list onto the ready queue */
  rb_traverse(r, G->heads) make_ready((Node) r->v.val, G);

  t = time(0);
  for (;;) {

    /* Execute from the ready Queue.  This includes taking the node off
       the ready queue, and evaluating parameters as much as is possible. */

    if (!dl_empty(G->readyq)) {
      d = first(G->readyq);
      n = (Node) d->val;
      dl_delete_node(d);
      make_running(n, G);

    /* Flag completion */

    } else if (G->numexecuting == 0) {
      master_death(G->alive);
      t = time(0) - t;
#ifdef PVM3
      pvm_exit ();
#else
      leave();
#endif
      trace_done(t);
      return;

    /* Wait for pvm nodes to complete for further processing */

    } else {
      receive_finished_node(G);
    }
  }
}

/*----------------------------------------------------------------------*/
/* Fire_up is the routine that execute() calls when it wants to execute
 * a node.  If the node is special-purpose, then special-purpose rewriting
 * routines are called.  If it is normal, then a pvm process is started, and
 * the initial message is sent.
 */

static fire_up(n, G)
Node n;
Graph G;
{
  Rb_node r, r2;
  Param p;
  int i;
  Machine get_machine_struct ();

  /* Deal with special-purpose nodes */

  if (n->node_type == COND) {
    execute_cond(n, G);
    return;
  } else if (n->node_type == FANOUT || n->node_type == PIPE) {
    execute_fan_or_pipe(n, G);
    return;
  } else if (n->node_type == LOOP) {
    execute_loop(n, G);
    return;
  } else if (n->node_type != NORMAL) {
    fprintf(stderr, "Error: Node %d.  Normal nodes only.\n", n->id);
    bail(CNULL);
  }

  /* If the subroutine is null, then it's done -- pretend it's finished */

  if (!n->arrays && strcmp(n->sub_name, "null") == 0) {
    /*
     * horrible hack.  "null" nodes need trace events like anything else.
     * in fact, they should probably be executed like anything else,
     * but for now we fake their trace events here.  finish_up_node()
     * also has special-casing for sub_name "null"
     */
    n->state = RUNNING;
    trace_node (n, G);
    n->state = DONE;
    trace_node (n, G);
    n->state = DEAD;
    trace_node (n, G);
    finish_up_node(n, G);
    return;
  }

  /* Set the machine */

#ifdef NEW_TRACE
  n->mptr = get_machine_struct (n->sub_name, G);
  if (n->mptr == (Machine) NULL) {
      fprintf(stderr, "ERROR: Subroutine %s not defined in cost matrix\n",
	      n->sub_name);
      nice_bail(CNULL);
  }
#else  /* NEW_TRACE */
  n->machine = get_machine(n->sub_name, G);
  if (n->machine == CNULL) {
    fprintf(stderr, "ERROR: Subroutine %s not defined in cost matrix\n",
            n->sub_name);
    nice_bail(CNULL);
  } 
#endif /* NEW_TRACE */
  
  /* Fire up the pvm process */

  n->ttime = time(0);

#ifdef PVM3
  /*
   * XXX to do:
   * recover when spawn fails:
   * - lack of resources -- start on another machine.
   * - no such file -- mark this machine as having infinite cost for
   *   this program, and look for another machine.
   */
  {
      int x;
      int spawn_flags = PvmTaskHost;

      if (debug_host && strcmp (n->mptr->name, debug_host) == 0 &&
	  debug_subr && strcmp (n->sub_name, debug_subr) == 0)
	  spawn_flags |= PvmTaskDebug;

      x = pvm_spawn (n->sub_name,		/* routine name */
		     (char **) NULL,		/* arguments (yeech!) */
		     spawn_flags,
#ifdef NEW_TRACE
		     n->mptr->name,
#else  /* NEW_TRACE */
		     n->machine,	/* specify host */
#endif /* NEW_TRACE */
		     1,				/* num tasks */
		     &(n->pvmtid));		/* array to recv tids */
      if (x < 0) {
	  fprintf (stderr, "ERROR %d: spawning \"%s\" on %s : %s\n",
		   n->pvmtid, n->sub_name, 
#ifdef NEW_TRACE
		   n->mptr->name,
#else  /* NEW_TRACE */
		   n->machine,
#endif /* NEW_TRACE */
		   pvm_strerror (x));
	  nice_bail (CNULL);
      }
  }
  /*
   * pvm3.0 bug: apparently pvm_spawn() always returns the number of
   * procs we wanted, not the number we got...so we check here also.
   */
  if (n->pvmtid < 0) {
      fprintf (stderr, "ERROR %d: spawning \"%s\" on %s : %s\n",
	       n->pvmtid, n->sub_name,
#ifdef NEW_TRACE
	       n->mptr->name,
#else  /* NEW_TRACE */
	       n->machine, 
#endif /* NEW_TRACE */
	       pvm_strerror (n->pvmtid));
      nice_bail (CNULL);
  }
  /* Send the initial message */

  pvm_notify (PvmTaskExit, HENCE_TASK_EXIT, 1, &n->pvmtid);
  n->state = RUNNING;
  trace_node(n, G);
  pvm_initsend (PvmDataDefault);

#else  /* PVM3 */

  n->pvminum = initiateM(n->sub_name, n->machine);
  if (n->pvminum < 0) pvm_init_error(n);
  /* Send the initial message */
  n->state = RUNNING;
  trace_node(n, G);
  initsend();

#endif /* PVM3 */

#ifdef IMA_ALPHA
  putint (unique_id (n));
#else  /* IMA_ALPHA */
  putint( (int) n );		/* XXX AArrggh! Don't DO this!
				 * slave uses this for a unique node id,
				 * but we might want to move these things
				 * around in memory some day.  That and
				 * passing pointers over the net is heinous
				 */
#endif /* IMA_ALPHA */

  putint(n->id);
  putint(n->inst);
  putlstring(n->sub_name);
  putint(G->trace_msgs);	/* Non zero if we want trace messages */
  putint(n->nparams);
  rb_traverse(r, n->params) {
    p = (Param) r->v.val;
    put_param(p);
    if (p->type == ARRAY) {
      rb_traverse(r2, p->a->p) put_param((Param) r2->v.val);
    }
  } 
  if (n->ret_param == PNULL) putint(-1); else putint(n->ret_param->num);
  putint(n->nargs);
  for(i = 0; i < n->nargs; i++) put_exp(n->args[i]);
  put_aa(n);

#ifdef PVM3
  send_or_gripe (n->sub_name, n->pvmtid, HENCE_INIT);
#else  /* PVM3 */
  send_or_gripe (n->sub_name, n->pvminum, HENCE_INIT);
#endif /* PVM3 */

  /* Flag the node as running */

  /* n->state = RUNNING; */
  G->numexecuting++;
}

#ifdef PVM3
char *
bail_task_died (G, tid)
Graph G;
int tid;
{
    Rb_node tn;

    rb_traverse(tn, G->nlist) {
	Node n = (Node) tn->v.val;
	if (n->pvmtid == tid) {
	    if (n->sub_name) {
		fprintf (stderr, "Task %x (%s) died unexpectedly\n",
			 tid, n->sub_name);
		nice_bail (CNULL);
	    }
	    else
		break;
	}
    }
    fprintf (stderr, "Task %x died unexpectedly\n", tid);
    nice_bail (CNULL);
}

Node
find_node_from_tid (G, tid)
Graph G;
int tid;
{
    Rb_node tn;
    rb_traverse(tn, G->nlist) {
	Node n = (Node) tn->v.val;
	if (n->pvmtid == tid)
	    return n;
    }
    return (Node) NULL;
}
#endif

/*----------------------------------------------------------------------*/
/* Receive_finished_node() is called when there are no nodes left on the
 * ready queue.  It waits for a message from a pvm process which indicates
 * that the process is finished.  That message contains all the finishing
 * information of the process.  That information is read into the master, 
 * which then looks at the node's children, puts itself onto their parent
 * lists, and decides whether to put them on the ready queue. */

static
receive_finished_node(G)
Graph G;
{
  Node n;
  int id, inst, nparams, nd, i;
  Rb_node r, rn;
  Param p;
  Exp e;
  int rcvtypes[4], type, len;
#ifdef PVM3
  int tid;
  int bufid;
#else
  int inum;
  char proc[64];
#endif

  /* Receive the message */

  rcvtypes[0] = HENCE_DIE;
  rcvtypes[1] = HENCE_DONE;
#ifdef PVM3
  rcvtypes[2] = HENCE_CHILD_OUTPUT;
  rcvtypes[3] = HENCE_TASK_EXIT;
  do {
      int remote_tid;
      int length;
      static char *buf = NULL;
      static int buflen = 0;

      if (buf == NULL) {
	  buflen = 4*1024;
	  buf = (char *) malloc (buflen);
      }
      bufid = retro_rcvmulti (4, rcvtypes);
      pvm_bufinfo (bufid, &len, &type, &tid);
#if 0
      printf ("len=%d type=%d tid=0x%08x\n", len, type, tid);
#endif
      if (type ==  HENCE_DIE) {
	  Node n = find_node_from_tid (G, tid);
	  if (n != (Node) NULL)
	      fprintf (stderr, "Received DIE message from %s(%d.%d) [%x]\n",
		       n->sub_name, n->id, n->inst, tid);
	  else
	      fprintf (stderr, "Received DIE message from [%x]\n", tid);
	  nice_bail (CNULL);
      }
      else if (type ==  HENCE_CHILD_OUTPUT) {
	  pvm_unpackf ("%d %d", &remote_tid, &length);
#if 0
	  printf ("remote_tid=%d length=%d\n", remote_tid, length);
#endif
	  if (length > 0) {
	      if (length < buflen) {
		  buflen = length;
		  buf = (char *) realloc ((void *) buf, buflen);
	      }
	      pvm_upkbyte (buf, length, 1);
	      fprintf (stderr, "[t%08x] %.*s\n", remote_tid, length, buf);
	      fflush (stderr);
	  }
	  else if (length == 0) {
	      /* EOF from task */
	  }
	  else if (length == -1) {
	      /* new task */
	  }
	  else if (length == -2) {
	      /* first output from task */
	  }
      }
      else if (type == HENCE_TASK_EXIT) {
#if 0
	  /*
	   * this isn't right...if a task returns a scalar and
	   * didn't modify any arrays it will exit before we get
	   * a die message.  so bailing is the wrong thing to do.
	   */
	  pvm_unpackf ("%d", &remote_tid);
	  bail_task_died (G, remote_tid);
#endif
      }
      else
	  break;
  } while (1);
#else
  rcvmulti(2, rcvtypes);
  rcvinfo(&len, &type, proc, &inum);
  if (type == HENCE_DIE) {
    fprintf(stderr, "Received DIE message from %s/%d\n", proc, inum);
    nice_bail(CNULL);
  }
#endif

  G->numexecuting--;


  /* Check to make sure that the node's information is consistent */

#ifdef IMA_ALPHA
  n = node_from_unique_id (getint ());
#else
  n = (Node) getint();
#endif
  id = getint();
  inst = getint();
  if (id != n->id || inst != n->inst) {
    fprintf(stderr, "RECV_FINISHED_NODE: id&inst don't match\n");
    fprintf(stderr, "  id=%d, n->id=%d, inst=%d, n->inst=%d\n",
            id, n->id, inst, n->inst);
    bail(CNULL);
  }
  n->ttime = time(0) - n->ttime;
  n->stime = getint();
  n->sbtime = getint();
  nparams = getint();
  if (nparams != n->nparams) {
    fprintf(stderr, "RECV_FINISHED_NODE: nparams doesn't match\n");
    fprintf(stderr, "  nparams=%d, n->nparams=%d\n", nparams, n->nparams);
    bail(CNULL);
  }
  n->state = DONE;
  trace_node(n, G);

  /* Read new parameter values for scalars.  If the parameter is > or
   * <>, then read its new value.  Otherwise, if it was injected, set
   * its new value from its injected value.  Otherwise, delete it from
   * the graph.  Nd is used to keep the parameter number contiguous
   * even though parameters are being deleted.
   */

  nd = 0;
  trace_param_begin(n);
  for (r = rb_first(n->params); r != nil(n->params); r = rn) {
    rn = rb_next(r);
    p = (Param) r->v.val;     /* p is the parameter */
    if (p->type != ARRAY) {
      e = get_exp(PNULL);       /* e is its new value.  PNULL is used because
                                 * the expression will be a constant */
      free_exp(p->val);
      p->val = ENULL;
      if (p->io.out) {
        p->val = e; 
        p->num -= nd;
        p->io.inj = 0;
        p->io.new = 0;
        trace_param(p);
      } else if (p->io.inj) {
        free_exp(e);
        push_int(p->inj);
        p->val = last_exp();
        p->io.out = 1;
        p->io.inj = 0;
        p->io.new = 0;
        trace_param(p);
      } else {
        free_exp(e);
        free_param(p);
        n->nparams--;
        nd++;
        rb_delete_node(r);
      }
    } else if (!p->io.in && p->io.out) {
      if (p->a->n_level2 > 0) 
        bail("INTE: Receive_finished_node: p->a->n_level2 > 0\n");
      for (i = 0; i < p->a->ndims; i++) {
        free_exp(p->a->dims[i]);
        p->a->dims[i] = get_exp(PNULL);
      }
      trace_param(p);
    }
  }
  subroutine_over(n->sub_name,
#ifdef NEW_TRACE
		  n->mptr->name,
#else
		  n->machine,
#endif
		  G);
  finish_up_node(n, G);
}

static
finish_up_node(n, G)
Node n;
Graph G;
{
  Rb_node r;
  Node nc;
  
  /* Traverse the list of children.  Update their ancestor list with 
   * this node's ancestor list.  If the child has no more parents left
   * to run (i.e. its ->state == 0), put it on the ready queue */

  rb_traverse(r, n->children) {
    nc = (Node) r->v.val;
    nc->state++;
    update_anc_list(nc->anc, n);
    if (nc->state == 0) {
      make_ready(nc, G);
    }
  }

  /* Delete the node's ancestor list, mark it as dead, and return */

  free_anc_list(n->anc);
  if (!n->arrays && strcmp(n->sub_name, "null") != 0) {
#ifdef PVM3
    pvm_initsend (PvmDataDefault);
#else
    initsend();
#endif
/*     printf("Master: sending DIE to %d/%d (%s/%d)\n", */
/*             n->id, n->inst, n->sub_name, n->pvminum); */
/*     fflush(stdout); */
#ifdef PVM3
    send_or_gripe (n->sub_name, n->pvmtid, HENCE_DIE);
#else
    send_or_gripe (n->sub_name, n->pvminum, HENCE_DIE);
#endif

    n->state = DEAD;
    trace_node(n, G);
  } else {
    (void) rb_inserti_nd(G->alive, n);
  }
    
}    


/*----------------------------------------------------------------------*/
/* Execute_fan_or_pipe() performs the graph rewriting for fans and pipe
 * Nodes.  They evaluate the high & low bounds, and then perform the 
 * rewriting.  If low > high, then the entire subgraph in the fan/pipe
 * is ignored.  Otherwise, copy_subgraph() is used to make the appropriate
 * number of copipes of the subgraph.  If the node is a pipe, then the 
 * subgraphs are connected in the proper way.  Finally, the special node
 * and its pair are deleted, and the rewriting is complete.
 */

execute_fan_or_pipe(n, G)
Node n;
Graph G;
{
  int low, high, i;
  Dlist *heads;
  Dlist *tails;
  Dlist d;
  Rb_node r;
  Node nd;

  /* Calculate the low bound */

  hence_eval_exp(n->args[0], 0);
  convert_exp_type(n->args[0], INT);
  low = n->args[0]->val.i;

  /* Calculate the high bound (set high to be this bound + 1) */

  hence_eval_exp(n->args[1], 0);
  convert_exp_type(n->args[0], INT);
  high = n->args[1]->val.i + 1;

  trace_fan_or_pipe(n, low, high);

  /* Remove the entire subgraph if low > high */

  if (low >= high) {  /* This is >= because high = high bound + 1 */
    ignore_subgraph(n, G);

  /* Otherwise, make all the copies using copy_subgraph.  Put the heads
   * and tails in the head and tail arrays */

#if 1
  /* It's not at all clear to me that the code below the #endif works 
   * for the case of FANOUT I = N to N.  So we have this hack. -KM
   */
  } else if (low + 1 == high) {
      rb_traverse (r, n->children) {
	  inject_param ((Node) r->v.val, n->ret_param, low);
      }
      remove_node_pair (n, G);
#endif
  } else {
    heads = talloc(Dlist, high - low);
    tails = talloc(Dlist, high - low);

    /* Make heads[0] & tails[0] by hand */

    heads[0] = make_dl();
    tails[0] = make_dl();
    rb_traverse(r, n->children) {
      nd = (Node) r->v.val;
      if (nd->nparents == 1) dl_insert_b_nd(heads[0], nd);
    }
    rb_traverse(r, n->pair->parents) {
      nd = (Node) r->v.val;
      if (nd->nchildren == 1) dl_insert_b_nd(tails[0], (Node) r->v.val);
    }

    /* Make the subgraphs.  The mirror field of each node in 
       subgraph i will point to its copy in subgraph i+1 */

    for (i = 1; i < high - low; i++) {
      heads[i] = make_dl();
      tails[i] = make_dl();
      copy_subgraph(n, heads[i-1], heads[i], tails[i], 1);
    }

    /* If this is a pipe, connect the subgraphs to each other */

    if (n->node_type == PIPE) {
      for (i = 1; i < high - low; i++) {
        connect_pipe_arcs(n, heads[i-1]);
      }
    }

    /* Inject the pipe/fan parameter into the heads of each subgraph */

    for (i = 0; i < high - low; i++) {
      dl_traverse(d, heads[i]) {
        nd = (Node) d->val;
        inject_param(nd, n->ret_param, low + i);
      } 
    }

    /* Remove the special-purpose node pair */

    remove_node_pair(n, G);

  }  
}  
 
/*----------------------------------------------------------------------*/
/* Execute_cond() executes conditional nodes.  It is quite simple:  
 * The conditional expression is evaluated.  If true, then remove the
 * conditional node pair, and the rewrite is complete.  If false, remove
 * the node pair and the subgraph. */

execute_cond(n, G)
Node n;
Graph G;
{
  hence_eval_exp(n->args[0], 0);
  convert_exp_type(n->args[0], INT);
  trace_cond(n, n->args[0]->val.i);
  if (n->args[0]->val.i) {   /* Cond == TRUE */
    /* Remove special node connect the children to the parents, and
       update the children's lists */
    remove_node_pair(n, G);
  } else {
    /* Otherwise, remove the node pair and subgraph */
    ignore_subgraph(n, G);
  }
}

/*----------------------------------------------------------------------*/
/* Execute_cond() executes loops nodes.  These are the trickiest.  
 * If there is a loop variable, then it is set to the proper expression.
 * Then a conditional is evaluated.  If the condition is false, then the
 * loop is over -- the loop nodes and subgraph are removed.  Otherwise, 
 * a copy of the loop subgraph is made, along with new loop nodes, and
 * is attached after the ENDLOOP node.  Then, the BEGINLOOP and ENDLOOP
 * nodes are removed, and the rewrite is complete.  In other words:

    BEGINLOOP--->SUBGRAPH--->ENDLOOP--->X

 * goes through the following steps for rewriting:

 1: Copy subgraph:     BEGINLOOP--->SUBGRAPH--->ENDLOOP--->X      COPY
 2: Attach subgraph in a new loop:    
    BEGINLOOP--->SUBGRAPH--->ENDLOOP--->BEGINLOOP--->COPY--->ENDLOOP--->X
 3: Remove first loop pair:
    SUBGRAPH--->BEGINLOOP--->COPY--->ENDLOOP--->X

 */

execute_loop(n, G)
Node n;
Graph G;
{
  Exp ctr_exp;
  Dlist heads, tails, newheads, newtails, d;
  Node nn, nnp, p;
  Rb_node r;

  /* Evaluate the initial expression.  When this is done, copy arg[2] to
     arg[0], so that the next node will do the right thing. */

  if (n->ret_param != PNULL && n->args[0] != ENULL) {
    hence_eval_exp(n->args[0], 0);
    convert_exp_type(n->args[0], INT);
    free_exp(n->ret_param->val);
    n->ret_param->val = n->args[0];
    n->ret_param->io.in = 1;
    n->ret_param->io.out = 1;
    n->ret_param->io.new = 0;
    n->ret_param->io.inj = 0;
    n->ret_param->type = INT;
  }
  n->args[0] = copy_exp(n->args[2], 0);

  /* Evaluate the conditional */

  ctr_exp = copy_exp(n->args[1], 0);
  hence_eval_exp(ctr_exp, 0);
  convert_exp_type(ctr_exp, INT);

  /* If the conditional is false, inject the parameter's value into it's
   * children (this is arguably correct semantics, so that the node after
   * LOOP(i = 0; i < 2; i = i + 1) completes will see a value of i = 2,
   * provided the loop's subgraph nodes do nothing fancy with i) and 
   * remove the subgraph from existance. */
 
  trace_loop(n, ctr_exp->val.i);
  
  if (!ctr_exp->val.i) {
    if (n->ret_param != PNULL) {
      rb_traverse(r, n->pair->children) {
        p = (Node) r->v.val;
        inject_param(p, n->ret_param, n->ret_param->val->val.i);
      }
    }
    ignore_subgraph(n, G);

  /* Otherwise, go through the loop rewriting gyrations: */

  } else {

    /* Make a copy of the subgraph */

    heads = make_dl();
    tails = make_dl();
    rb_traverse(r, n->children) dl_insert_b_nd(heads, (Node) r->v.val);
    rb_traverse(r, n->pair->parents) dl_insert_b_nd(tails, (Node) r->v.val);
    newheads = make_dl();
    newtails = make_dl();
    copy_subgraph(n, heads, newheads, newtails, 0);

    /* Make new loop nodes */

    nn = copy_node(n);
    nnp = copy_node(n->pair);
    nn->pair = nnp;
    nnp->pair = nn;
    n->mirror = n;
    n->pair->mirror = n->pair;

    /* Free up the new loop's return parameter value, so that it gets
     * calculated anew */

    if (nn->ret_param != PNULL) {
      free_exp(nn->ret_param->val);
      nn->ret_param->val = ENULL;
      nn->ret_param->type = -1;
      nn->ret_param->io.in = 0;
      nn->ret_param->io.out = 0;
      nn->ret_param->io.new = 0;
      nn->ret_param->io.inj = 0;
    }

    /* Attach the new loop nodes to the new subgraph */

    dl_traverse(d, newheads) {
      p = (Node) d->val;
      new_arc(nn, p, 0);
    }
    dl_traverse(d, newtails) {
      p = (Node) d->val;
      new_arc(p, nnp, 0);
    }
    
    /* Attach the new loop end node to the old loop end node's children,
     * and detatch the old loop end node. */

    while(!rb_empty(n->pair->children)) {
      r = rb_first(n->pair->children);
      p = (Node) r->v.val;
      new_arc(nnp, p, 0);
      delete_arc(n->pair, p, 0);
    }
    
    /* Attach the new loop begin to the old loop end */

    new_arc(n->pair, nn, 0);

    /* Inject the loop variable into the old loop */

    if (n->ret_param != PNULL) {
      rb_traverse(r, n->children) {
        p = (Node) r->v.val;
        if (p->nparents == 1) 
          inject_param(p, n->ret_param, n->ret_param->val->val.i);
      }
    }

    /* Remove the old loop nodes */

    remove_node_pair(n, G);
  }
}
  
/*----------------------------------------------------------------------*/
/* Ignore_subgraph() is called when a special node and its subgraph are 
 * not to be executed.  It works in three stages:  First
 * It basically deletes the begin and end nodes 
 * and attaches the begin node's parents to the end node's children.
 * It also copies the begin node's ancestor list to the and node's children,
 * as there is no other way for those ancestor lists to be passed on.
 */

ignore_subgraph(n, G) 
Node n;
Graph G;
{
  Node pair;
  Rb_node rp, rc;
  Node p, c;

  pair = n->pair;

  /* Attach the begin nodes parents to the end node's children */

  rb_traverse(rc, pair->children) {
    c = (Node) rc->v.val;
    rb_traverse(rp, n->parents) new_arc((Node) rp->v.val, c, 1);
    rp = rb_find_ndkey(c->parents, pair);
    rb_delete_node(rp);
    c->nparents--;
    c->state++;
  }
  rb_traverse(rp, n->parents) {
    p = (Node) rp->v.val;
    rc = rb_find_ndkey(p->children, n);
    rb_delete_node(rc);
    p->nchildren--;
  }
  rb_traverse(rc, pair->children) {
    c = (Node) rc->v.val;
    update_anc_list(c->anc, n);
    if (c->state == 0) make_ready(c, G);
  }
  free_anc_list(n->anc);
}

  

/*----------------------------------------------------------------------*/
/* Reduce_params gets the correct initial values for parameters.  The way
 * it works is to keep traversing the parameter list, and remove parameters
 * when it knows what to do with them.  The removed parameters are stored 
 * in the rb_tree tmp.  When n->params is empty, then all the parameters 
 * have been reduced, and n->params is set to tmp.
 */

reduce_params(n)
Node n;
{
  Rb_node r, rnxt, rap, rnp, tmp, rch, rp;
  Param p, ap;
  Node a, ch;
  int fnd, i;
  Dlist tmp_aa;

  tmp = make_rb();
  tmp_aa = make_dl();

  /* First traversal: Gets rid of new, > and injected parameters */

  number_parameters(n);
  r = rb_first(n->params);
  while(r != nil(n->params)) {
    p = (Param) r->v.val;
    p->mirror = PNULL;  /* Mirror field blanked so it can be used later */
    if (p->io.inj || p->io.new || (p->io.out && !p->io.in)) {

      /* Initialize injected parameters and propagate from non-normal nodes */

      if (p->io.inj) {
        if (!p->io.new) {
          free_exp(p->val);
          push_int(p->inj);
          p->val = last_exp();
        }
        if (n->node_type != NORMAL) {
          rb_traverse(rch, n->children) {
            ch = (Node) rch->v.val;
            if (ch->nparents == 1) inject_param(ch, p, p->inj);
          }
        }
      }

      rnxt = rb_next(r);
      rb_delete_node(r);
      p->state = 1;
      (void) rb_insert_p1(tmp, p);
      r = rnxt;
    } else r = rb_next(r);
  }

  /* Next, traverse ancestors, and until the parameter list is empty, 
     put in their correct values. For arrays, make a new parent list, 
     containing only the parents which have array parameters  */

  r = rb_first(n->anc->order);
  while (r != nil(n->anc->order) && !rb_empty(n->params)) {
    a = (Node) r->v.val;

    /* Traverse the ancestor's paremter list, and see if any of its 
       parameters match the node's */

    rb_traverse(rap, a->params) {
      ap = (Param) rap->v.val;
      rnp = rb_find_key_n(n->params, ap->name, &fnd);

      /* If so, check the parameter types */

      if (fnd) {
        p = (Param) rnp->v.val;

        /* If they don't match, flag an error */

        if (p->type != ap->type && p->type != -1) {
          do_na_types_error(n, p, a, ap);

        /* The ancestor is an array: */

        } else if (ap->type == ARRAY) {
          free_exp(p->val);
          p->val = ENULL;

          /* Set the bounds if not done so yet */
          if (p->type == -1) {
            p->type = ARRAY;
            p->a = new_parray(ap->a->type, ap->a->ndims);
	    p->io.main = 1;
            p->a->main = p;
            for (i = 0; i < p->a->ndims; i++) {
              p->a->dims[i] = copy_exp(ap->a->dims[i], 2);
            }
            n->arrays = 1;
          } else if (p->a->type == -1) {
            set_array_type(p, ap->a->type);
          }
          if (p->a->type != ap->a->type || 
              p->a->ndims != ap->a->ndims) {
            do_na_types_error(n, p, a, ap);
          }
          if (p->mirror == PNULL) p->mirror = ap; 
                        /* Here the mirror field is
                           used below to check on the size of the dimensions.
                           I think it's ok, as the node won't be copied 
                           further, so the mirror field won't be read again */
           
          /* Put the parent on the aa list */

	  add_array_anc(tmp_aa, a, p, ap);

          /* If the parent contains the whole array, 
             move the parameter to tmp */

          if (ap->io.used) {
            rb_delete_node(rnp);
            p->state = 1;
            (void) rb_insert_p1(tmp, p);
          }
            
        /* Otherwise, this is a scalar -- copy the value and
           move the parameter to tmp */
            
        } else {
          p->type = ap->type;
          free_exp(p->val);
          p->val = copy_exp(ap->val, 0);
          rb_delete_node(rnp);
          p->state = 1;
          (void) rb_insert_p1(tmp, p);
        }
      }
    }
    r = rb_next(r);
  }

  /* The remaining parameters in n->params have no ancestors (Or in 
     the case of arrays, no ancestors holding the complete value).
     Thus, they will be initialized according to their initvals or to
     "reasonable" values.  Copy them to tmp */
        
  while (!rb_empty(n->params)) {
    r = rb_first(n->params);
    p = (Param) r->v.val;
    rb_delete_node(r);
    p->state = 1;
    (void) rb_insert_p1(tmp, p);
  }

  /* Now, traverse the tmp list and initialize all parameters that need it.
   * Flag errors for untyped/uninitializable pamereters */

  trace_param_begin(n);
  rb_traverse(r, tmp) {
    p = (Param) r->v.val;
    if (p->io.in || !p->io.out) {
      if (p->type == -1) {
        do_unk_type_error(n, p);
      } else if (p->type == ARRAY) {
        eval_array_dims(n, p);
      } else if (p->val == ENULL) {
        p->val = make_null_exp(p->type);
      } else {
        hence_eval_param(p, 0);
      }
      if (!p->io.in) {
        p->io.in = 1;
        p->io.out = 1;
      }
    }
    trace_param(p);
    if (p->type == ARRAY) {
      rb_traverse(rp, p->a->p) {
        trace_param((Param) rp->v.val);
      }
    }
  }
  /* Traverse the aa list, do error checking and pruning, and make the 
     final aa list */

  make_aa_list(n, tmp_aa);
    
  /* Swap back parameters */

  n->params = tmp;
}


/* ---------------------------------------------------------------------- */
/* Error Flagging */

static
do_unk_type_error(n, p) 
Node n;
Param p;
{
  fprintf(stderr, "ERROR: "); 
  fprint_node(stderr, n);
  fprintf(stderr, "Can't deduce a type for parameter:\n  ");
  fprint_param(stderr, p);
  fprintf(stderr, "\n");
  nice_bail(CNULL);
}

do_na_types_error(n, p, a, ap)
Node n, a;
Param p, ap;
{
  fprintf(stderr, "\nERROR: Disagreeing types:\n");
  fprintf(stderr, "  Trying to execute ");
  fprint_node_id(stderr, n);
  fprintf(stderr, "Param: ");
  fprint_param(stderr, p);
  fprintf(stderr, "\n  With ancestor ");
  fprint_node_id(stderr, a);
  fprintf(stderr, "Param: ");
  fprint_param(stderr, ap);
  fprintf(stderr, "\n");
  nice_bail(CNULL);
}

      
pvm_enroll_error(proc, inum)
char *proc;
int inum;
{
  fprintf(stderr, "ERROR: Enroll failed in pvm on %s\n", proc);
  fprintf(stderr, "  Returned %d\n", inum);
  nice_bail(CNULL);
}

#ifndef PVM3
pvm_init_error(n)
Node n;
{
  fprintf(stderr, "ERROR: InitiateM failed on node %d/%d:\n", n->id, n->inst);
  fprintf(stderr, "  initiateM(%s,%s) returned %d\n",
          n->sub_name, n->machine, n->pvminum);
  nice_bail(CNULL);
}
#endif

#ifdef IMA_ALPHA
/* 20-Aug-1992 AFV: don't pass 64 bit addresses as 32 bit ints */

/* XXX when a node that is in the table is freed, it should be freed */
/* in the unique_id_table also. */

static int unique_id (n)
Node n;
{
    int	i;

    /* search for an entry in the table already */

    for (i = 0; i < next_unique_id; i++) {
	if (n == unique_id_table[i])
	    return (i);
    }

    /* search for an entry that is no longer in use */
    for (i = 0; i < next_unique_id; i++) {
	if (unique_id_table[i] == NULL)	{
	    unique_id_table[i] = n;
	    return (i);
	}
    }

    if (next_unique_id >= MAX_IDS_IN_TABLE) {
	fprintf (stderr, 
		 "Too many calls to unique_id: boost MAX_IDS_IN_TABLE (%d)\n",
		 MAX_IDS_IN_TABLE);
	exit (0);
    }

    i = next_unique_id++;
    unique_id_table[i] = n;
    return (i);
}

static Node
node_from_unique_id (u)
int u;
{
    if (u < 0 || u >= next_unique_id) {
	fprintf (stderr, 
		 "Couldn't find unique_id in table\n");
	return (NULL);
    }
    return (unique_id_table[u]);
}
#endif /* IMA_ALPHA */

