/*
 * Htool 2.x interface to pvm 3.x
 *
 * $Id: pvmglue.c,v 1.2 1994/06/11 20:45:22 moore Exp $
 *
 * $Log: pvmglue.c,v $
 * Revision 1.2  1994/06/11  20:45:22  moore
 * add compatibility code for pvm 3.3 vs 3.2 (name of struct changed)
 *
 * Revision 1.1  1994/02/17  21:34:49  moore
 * Initial revision
 *
 */

/*
 * this is a crude interface to pvm 3.
 *
 * It's crude because it blocks while pvm is doing this, and so it cannot
 * handle X events while doing so.  One fix is to fork a helper task
 * that actually does the pvm calls, and communicates with htool via
 * a socket pair.  The two big problems with this are:
 *
 * 1) In the X environment, it's hard to do an rpc.  Either the caller
 *    specifies a callback function that happens when the reply comes back,
 *    or the caller has to have its own X event-handling loop.  Sometimes
 *    it might be a good idea to "mask" certain events -- we want to handle
 *    exposes and resizes but might not want to handle new user commands --
 *    except maybe "quit".  It's difficult to know which events to mask.
 * 2) Basically we need an RPC wrapper generator to translate htool or pvm's
 *    internal data structures into a form we can send over a socket.
 *
 * ... so the spiffy interface will have to wait.
 */

/*
 * system includes
 */
#include <stdio.h>
#include <signal.h>
#include <fcntl.h>		/* XXX bondoize? */
#include <unistd.h>		/* XXX bondoize? */

/*
 * X includes
 */
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

/*
 * Bondo includes
 */
#include <Errno.h>
#include <Malloc.h>
#include <String.h>

/*
 * htool includes
 */
#include "costmat.h"
#include "global.h"
#include "mode.h"
#include "pvmglue.h"
#include "subproc.h"
#include "text_sink.h"
#include "tilde.h"

/*
 * pvm includes
 */
#include "pvm3.h"

static int mytid = 0;

extern int pvm_errno;

#define TRACE_POLL_INTERVAL 500 /* msec */

char *
pvm_strerror (x)
int x;
{
    extern int pvm_nerr;
    extern char *pvm_errlist[];
    static char buf[100];

    x = -x;
    if (x >= 0 && x < pvm_nerr)
	return pvm_errlist[x];
    sprintf (buf, "unknown pvm error %d", x);
    return buf;
}

/*
 * start up pvm if necessary.  If user specified a host file via an
 * X resource, use it.
 *
 * XXX should probably roll our own pvm_start_pvmd() and use htool's
 * subproc library so we will know if/when it exits.
 *
 * return:
 *  0 if all is well
 * -1 if startup fails
 */
 
int
pvmglue_start_pvm ()
{
    char *pvm_argv[3];
    int pvm_pid;
    char msgbuf[100];
    static int inited = 0;

    /*
     * flush X stuff so we see what's going on.
     */
    if (mytid == 0)
	msg_Format ("(starting pvm...)\n");
    XFlush (global.display);

#if 0
    /*
     * Force PVM_ROOT to $(HOME)/pvm3, and PVM_ARCH to be the
     * architecture that we were compiled with.  These variables
     * have to be set so that pvm_start_pvmd will work reliably.
     *
     * (arrgh...pvm_start_pvmd should check these envars itself,
     * rather than calling '$(PVM_ROOT)/lib/pvmd'...it already
     * does most of the work anyway).
     *
     * XXX we can be more flexible eventually if we figure out
     * how to install programs on machines where PVM_ROOT is
     * different.
     */
#endif
    if (!inited) {
#if 0
	sprintf (msgbuf, "%s/pvm3", getenv ("HOME"));
	setenv ("PVM_ROOT", msgbuf, 1);
#endif
	setenv ("PVM_ARCH", ARCHSTR, 1);
	/* XXX this is a hack so we can debug proxyd */
	setenv ("PVM_EXPORT", "DISPLAY", 1);
	inited = 1;
    }

    pvm_setopt (PvmAutoErr, 0);
    if ((mytid = pvm_mytid ()) > 0)
	return 0;

    /*
     * NB: pvm 3.1.4 needs an extra element at the end of the
     * argv array for the call to pvm_start_pvmd; it writes a NULL pointer
     * there.
     */ 

    pvm_argv[0] = defaults.pvmHostFile;
    pvm_argv[1] = NULL;

    if ((pvm_pid = pvm_start_pvmd (pvm_argv[1] ? 3 : 2, pvm_argv, 1)) < 0) {
	if (pvm_pid != PvmDupHost) {
	    sprintf (msgbuf, "pvm startup failed: %s\n",
		       pvm_strerror (pvm_errno));
	    ts_append (global.pvm_sink, msgbuf);
	    return -1;
	}
    }
    /*
     * XXX if pvm_mytid returns "can't contact local daemon", consider
     * removing pvm uid file...
     */
    if ((mytid = pvm_mytid ()) < 0) {
	sprintf (msgbuf,
		 "ERROR: pvm_mytid() returned %d (%s) after startup.\n",
		 mytid, pvm_strerror (mytid));
	ts_append (global.pvm_sink, msgbuf);
	sprintf (msgbuf, "(you might try doing \"rm /tmp/pvmd.%d\")\n",
		 getuid ());
	ts_append (global.pvm_sink, msgbuf);
	mytid = 0;
	return -1;
    }
    return 0;
}

#if 0
void
pvm_Start (x)
void *x;
{
    if (pvm_mytid () > 0) {
	ts_append (global.pvm_sink, "pvm is already running.\n");
	return;
    }
    pvmglue_start_pvm ();
}

/*
 * halt any existing pvm and start up another one.
 */

void
pvm_Restart (x)
void *x;
{
#if 0
    /*
     * XXX this doesn't work.  we can't trap signals because
     * pvmd uses SIGKILL; and even calling pvm_halt() from
     * the child sometimes causes us to die.  Eventually we will
     * set up a helper subprocess that does all pvm communication;
     * for now we do without the ability to restart the pvmd.
     */
    pvm_setopt (PvmAutoErr, 0);
    /*
     * can't call pvm_halt(), because that will tell pvmd to kill us.
     * so we leave the pvm, and call pvm_halt() in a subprocess.
     * XXX maybe we should call subproc_Call()?
     */
    if (pvm_mytid () > 0)
	pvm_exit ();
    mytid = 0;
    ts_append (global.pvm_sink, "(halting old pvm)\n");
    if (fork () == 0) {
	/* child only */
	int fd;
	int i;
	extern char **environ;

	fd = open ("/dev/null", 2);
	dup2 (fd, 0);
	dup2 (fd, 1);
	dup2 (fd, 2);
	for (i = 3; i < getdtablesize (); ++i)
	    close (i);
	pvm_setopt (PvmAutoErr, 0);
	pvm_halt ();
	exit (0);
    }
    ts_append (global.pvm_sink, "(restarting)\n");
#endif
    pvm_Start ((void *) 0);
}
#endif

/*
 * kill all processes but don't kill the machine (or us).
 */

void
pvm_KillAll (x)
void *x;
{
    int ntasks;
/*
 * UGLY HACK ALERT: the name of the taskinfo struct changed to
 * pvmtaskinfo in pvm 3.3.  Hence this grot.
 */

#ifdef PVM_STR
    struct pvmtaskinfo *tip;
#else
    struct taskinfo *tip;
#endif
    int i;

    pvm_setopt (PvmAutoErr, 0);
    if (mytid == 0)
	mytid = pvm_mytid ();
    if (mytid < 0) {
	ts_append (global.pvm_sink, "pvmd is not running\n");
	return;
    }
    pvm_tasks (0, &ntasks, &tip);
    for (i = 0; i < ntasks; ++i) {
	if (tip[i].ti_tid != mytid)
	    pvm_kill (tip[i].ti_tid);
    }
}

void
pvm_Status (x)
void *x;
{
    int i, j;
#ifdef PVM_STR
    struct pvmtaskinfo *tip;
    struct pvmhostinfo *hip;
#else
    struct taskinfo *tip;
    struct hostinfo *hip;
#endif
    int nhosts;
    int ntasks;
    char *hostname;
    char *procname;
    char msgbuf[100];

    pvm_setopt (PvmAutoErr, 0);
    mytid = pvm_mytid ();
    if (mytid < 0) {
	ts_append (global.pvm_sink, "pvmd is not running\n");
	return;
    }
    if (pvm_config (&nhosts, 0, &hip) < 0 ||
	pvm_tasks (0, &ntasks, &tip) < 0) {
	/* XXX print out some sort of error msg */
	return;
    }

    /* print out heading */
    sprintf (msgbuf, "Machine status:\n");
    ts_append (global.pvm_sink, msgbuf);
    sprintf (msgbuf, "%-24.24s %-12.12s %-8.8s %-8.8s\n",
	     "Host Name", "Pgm Name", "TID", "PTID");
    ts_append (global.pvm_sink, msgbuf);

    /* print out info for each task */
    for (i = 0; i < ntasks; ++i) {
	/* find host name */
	hostname = "(unknown)";
	for (j = 0; j < nhosts; ++j)
	    if (hip[j].hi_tid == tip[i].ti_host)
		hostname = hip[j].hi_name;
	procname = (tip[i].ti_tid == mytid) ? "(htool)" : tip[i].ti_a_out;
	if (*procname == '\0')
	    procname = "-";
	sprintf (msgbuf, "%-24.24s %-12.12s %08x %08x\n",
		 hostname, procname, tip[i].ti_tid, tip[i].ti_ptid);
	ts_append (global.pvm_sink, msgbuf);
    }
}


void
pvm_ShowConfig (x)
void *x;
{
    int i;
    int nhosts;
#ifdef PVM_STR
    struct pvmhostinfo *hip;
#else
    struct hostinfo *hip;
#endif
    char msgbuf[100];

    pvm_setopt (PvmAutoErr, 0);
    mytid = pvm_mytid ();
    if (mytid < 0) {
	ts_append (global.pvm_sink, "pvmd is not running\n");
	return;
    }
    if (pvm_config (&nhosts, 0, &hip) < 0)
	return;

    sprintf (msgbuf, "Machine configuration:\n");
    ts_append (global.pvm_sink, msgbuf);
    sprintf (msgbuf, "%-24.24s %8s %8s %5s\n",
	     "----------HOST----------", "--DTID--", "--ARCH--", "SPEED");
    ts_append (global.pvm_sink, msgbuf);
    for (i = 0; i < nhosts; ++i) {
	sprintf (msgbuf, "%-24.24s %8x %8s %5d\n",
		 hip[i].hi_name,
		 hip[i].hi_tid,
		 hip[i].hi_arch,
		 hip[i].hi_speed);
	ts_append (global.pvm_sink, msgbuf);
    }
}

/*
 * Halt the virtual machine
 */

void
pvm_Halt (x)
void *x;
{
    pvm_setopt (PvmAutoErr, 0);
    mytid = pvm_mytid ();
    if (mytid < 0) {
	ts_append (global.pvm_sink, "pvmd is not running\n");
	return;
    }
    signal (SIGTERM, SIG_IGN);
    if (pvm_halt () < 0)
	ts_append (global.pvm_sink, "halt: pvm is not responding\n");
    mytid = -1;
    pvm_exit ();
}

/*
 * make sure a host from the cost matrix is part of the machine.
 * return 0 if successful, else -1.
 *
 * called by cm_AddHostsToPvm ()
 */

int
pvm_AddHost (hostname)
char *hostname;
{
    int cc;
    int status;
    char msgbuf[1024];
    int retry = 0;

    /*
     * loop attempting to add a particular host until it's ready
     *
     * XXX need an xsleep() routine to handle events.
     */

    do {
	cc = pvm_addhosts (&hostname, 1, &status);
	if (cc == PvmAlready) {
	    ++retry;
	    sleep (1);
	}
    } while (cc == PvmAlready && retry < 45);

    if (cc < 0)
	status = cc;

#if 0
    fprintf (stderr, "addhost %s => %s\n", hostname, pvm_strerror (status));
#endif

    if (cc != 1) {
	if (status != PvmDupHost) {
	    sprintf (msgbuf, "error adding %s: %s\n",
		     hostname, pvm_strerror (status));
	    ts_append (global.pvm_sink, msgbuf);
	    return -1;
	}
    }
    return 0;
}

/***********************************************************************
 *		    executioner subprocess support                     *
 ***********************************************************************/

struct executioner_params {
    char *graph_file;
    char *cost_matrix_file;
    int trace_fd;
    int trace_state;
    XtIntervalId trace_timer_id;
};

static int executioner_pid = -1;

void
pvm_KillProgram (x)
void *x;
{
    if (executioner_pid > 0)
	kill (executioner_pid, SIGTERM);
}


void
pvm_ExitRunMode (x)
void *x;
{
    exit_run_mode ();
}

/*
 * callback for when executioner exits.
 */
static void
executioner_done (pid, exitArg, cd, howDied)
int pid;
int exitArg;
void *cd;
enum causeOfDeath howDied;
{
    struct executioner_params *ep = (struct executioner_params *) cd;
    char buf[1024];

    /*
     * read and process trace records until EOF.
     * Do this before printing cause of death.
     */
    if (ep->trace_state == 2)
	while (trace_read_next_event (ep->trace_fd) > 0)
	    trace_update_state (1);
    if (ep->trace_fd > 0)
	close (ep->trace_fd);

    if (howDied == SUBPROC_KILLED)
	sprintf (buf,
		 "(HeNCE executioner (pid %d) %s with signal %d.)\n",
		 (exitArg & 0200) ? "core dumped" : "was killed",
		 pid, exitArg & ~0200);
    else if (exitArg == 0)
	sprintf (buf, "(HeNCE executioner exited normally.)\n");
    else
	sprintf (buf, "(HeNCE executioner exited with status %d)\n",
		 exitArg);
    ts_append (global.pvm_sink, buf);
    ts_append (global.info_sink, "(run done)\n");
    subproc_Pclose (pid);

    /*
     * clean up
     */
    if (ep->graph_file) {
	unlink (ep->graph_file);
	FREE (ep->graph_file);
	ep->graph_file = NULL;
    }
    if (ep->cost_matrix_file) {
	unlink (ep->cost_matrix_file);
	FREE (ep->cost_matrix_file);
	ep->cost_matrix_file = NULL;
    }
    XtRemoveTimeOut (ep->trace_timer_id);
    FREE (ep);
    ep = NULL;
    executioner_pid = -1;
}

/*
 * callback for when executioner emits some output.
 */

static void
executioner_sink (fd, cd)
int fd;
void *cd;
{
    char buf[5120];
    int nread;
    char *ptr1, *ptr2;

    if ((nread = read (fd, buf, sizeof buf - 1)) <= 0)
	return;
    buf[nread] = '\0';

#if 0
    write (fileno (stderr), buf, nread);
#endif

    ptr1 = buf;
    while (ptr2 = strchr (ptr1, '\n')) {
	/* invariant: ptr1 is at the beginning of a line */
	ts_append_l (global.pvm_sink, ptr1, ptr2 - ptr1 + 1);
	ptr1 = ptr2 + 1;
    }
    if (ptr1 < buf + nread)
	ts_append_l (global.pvm_sink, ptr1, (buf + nread) - ptr1);
}

/*
 * callback to read/display trace events while executing
 */

static void
executioner_trace_timer (cd, x)
XtPointer cd;
XtIntervalId *x;
{
    struct executioner_params *ep = (struct executioner_params *) cd;
    int foo;

#if 0
    fprintf (stderr, "%d", ep->trace_state);
#endif
    /*
     * 1. if we haven't done so yet, read trace file header...I guess
     * it's okay if we block while doing this.
     * 2. afterward, read trace events until either the process
     * exits or we get to a FINISH event
     */
    switch (ep->trace_state) {
    case 0:
	/* try to open file */
	if ((ep->trace_fd = trace_open_file (global.traceFile)) < 0)
	    break;
	ep->trace_state = 1;
	/* FALL THROUGH */
    case 1:
	/*
	 * try to read file header.
	 * rewind in case we got partially through before
	 */
	foo = trace_read_header (ep->trace_fd);
#if 0
	fprintf (stderr, "(%d)", foo);
#endif
	switch (foo) {
	case -1:		/* premature EOF: rewind and try again later */
	    lseek (ep->trace_fd, 0L, 0);
	    goto done;
	default:
	    msg_Format ("trace_read_header() returned %d\n", foo);
	    /* FALL THROUGH */
	case -2:		/* file format error */
	    close (ep->trace_fd);
	    ep->trace_state = 4;
	    return;
	case 0:
	    ep->trace_state = 2;
	}
    case 2:
	/*
	 * got the header...now read a record
	 */
	while (1) {
	    int foo; 
	    if ((foo = trace_read_next_event (ep->trace_fd)) < 0) {
		/* set hairline to the simulated time */
		trace_set_hairline ();
		goto done;		/* EOF (for now) */
	    }
	    else if (foo == 0) {
		trace_update_state (1);
		ep->trace_state = 3;	/* saw FINISH record */
		break;
	    }
	    else
		trace_update_state (1);
	}
	/* FALL THROUGH */
    case 3:
	trace_close_file (ep->trace_fd);
	ep->trace_state = 4;
	/*
	 * since we've already seen the end of the trace file,
	 * don't schedule any more timeouts
	 */
	return;
    case 4:
	/* dead end ... don't do anything */
	return;
    }
 done:
    ep->trace_timer_id = XtAppAddTimeOut (global.context,
					  TRACE_POLL_INTERVAL,
					  executioner_trace_timer,
					  (void *) ep);
}

/*
 * execute the HeNCE program.
 *
 * 1. make sure the graph is valid
 * 2. make sure pvm is running
 * 3. make sure all subr are represented in the cost matrix,
 *    and that the required hosts are configured.
 * 4. write the graph to a temp file
 * 5. write out a temp cost matrix
 * 6. run the executioner
 *
 * TODO:
 * (a) disable changes to the graph or cost matrix while running
 */

void
pvm_RunProgram (x)
void *x;
{
    char *temp_cm;		/* temp cost matrix name */
    char *temp_gr;		/* temp graph name */
    FILE *fp_gr;		/* stream to write out graph */
    char buf[1024];		/* scratch space */
    struct executioner_params *ep = NULL;
    int foo;

    /*
     * 1. make sure there's a trace file
     *
     * XXX we could just go ahead and create a temp trace file...but
     * then the user might not realize that one wasn't being saved.
     * Maybe the thing to do is to *always* create a temp trace file,
     * then when the user asks to exit run mode, we can ask if he
     * wants to save it.
     */
    if (global.traceFile == NULL || *global.traceFile == '\0') {
	msg_Format ("Run failed: no trace file specified.\n");
	return;
    }

    /*
     * 2. make sure graph is valid  (before configuring hosts...
     * since the cm_AddHostsToPvm call might choke on a bogus
     * graph)
     */
    if (graph_critic (global.graph) != 0)
	return;

    /*
     * 3. make sure pvm is running (before configuring hosts)
     */
    if (pvmglue_start_pvm () < 0)
	return;

    /*
     * 4. make sure all subrs are represented; configure required hosts
     */
    if (cm_AddHostsToPvm (global.graph, global.costMatrix) != 0)
	return;

    /*
     * 5. We don't require that the user has saved the cost matrix
     * and the current graph.  So we generate temp file names for
     * these, and write them out before running.  This is done after
     * cm_AddHostsToPvm() because that function does some useful
     * error checking.
     */
    ep = XMALLOC (1, struct executioner_params);
    ep->trace_timer_id = 0;

    /* generate temp file names */

    sprintf (buf, "%s/htool%x.cm", defaults.tmpDir, getpid ());
    temp_cm = STRDUP (buf);
    sprintf (buf, "%s/htool%x.gr", defaults.tmpDir, getpid ());
    temp_gr = STRDUP (buf);

    /* write temp graph */

    if ((fp_gr = fopen (temp_gr, "w")) == NULL) {
	msg_Format ("ERROR: can't create %s: %s\n",
		    temp_gr, strerror (errno));
	return;
    }
    if (graph_unparse (global.graph, fp_gr) != 0) {
	msg_Format ("graph has errors. execution aborted\n");
	goto cleanup;
    }
    fclose (fp_gr);
    fp_gr = NULL;
    
     /* write temp cost matrix */

    if (cm_WriteTempFile (temp_cm, global.costMatrix) < 0)
	goto cleanup;

    /*
     * 6. nuke any trace file that already exists.  This should
     * happen before setting up the trace timer, since otherwise
     * it could read an old trace file.  But since this operation
     * can fail, do it before entering run mode.
     */
    if ((foo = open (global.traceFile, O_RDWR|O_TRUNC|O_CREAT, 0644)) > 0)
	close (foo);
    else if (unlink (global.traceFile) < 0) {
	msg_Format ("run failed: can't write to trace file %s (%s)\n",
		    global.traceFile, strerror (errno));
	goto cleanup;
    }

    /*
     * 7. set run mode before setting up timeout.  this clears the
     * graph canvas and the cost matrix, to be redrawn when
     * we start reading the trace file.  This needs to happen
     * before we set up the trace_timer, since the timer callback
     * requires some initialization that enter_run_mode() does.
     */
    set_mode (MODE_EXECUTE);
    enter_run_mode ();      /*
			     * XXX should call this as a mode hook
			     */

    /*
     * 8. set up a timer and a callback that reads events from the
     * trace file
     */
    ep->trace_timer_id = XtAppAddTimeOut (global.context,
					  TRACE_POLL_INTERVAL, /* time in msec */
					  executioner_trace_timer,
					  (void *) ep);
    ep->graph_file = temp_gr;
    ep->cost_matrix_file = temp_cm;
    ep->trace_state = 0;
    ep->trace_fd = -1;

    /*
     * 9. run the executioner.  The exit callback makes sure
     * the temp files are deleted (and the storage space for
     * the filenames is freed), when the executioner exits.
     */

    /* XXX add debug flags */
    msg_Format ("running %s/pvm3/bin/%s/master %s -tf %s -cm -%s\n",
		getenv ("HOME"), ARCHSTR, temp_gr, global.traceFile, temp_cm);

    if (defaults.debugMaster) {
	if (global.traceFile && *global.traceFile != '\0')
	    sprintf (buf,
		     "xterm -e %s/pvm3/bin/%s/master.debug %s -tf %s -cm %s",
		     getenv ("HOME"), ARCHSTR, temp_gr, global.traceFile,
		     temp_cm);
	else
	    sprintf (buf, "xterm -e %s/pvm3/bin/%s/master.debug %s -cm %s",
		     getenv ("HOME"), ARCHSTR, temp_gr, temp_cm);
    }
    else {
	if (global.traceFile && *global.traceFile != '\0')
	    sprintf (buf, "%s/pvm3/bin/%s/master %s -tf %s -cm %s",
		     getenv ("HOME"), ARCHSTR, temp_gr, global.traceFile,
		     temp_cm);
	else
	    sprintf (buf, "%s/pvm3/bin/%s/master %s -cm %s",
		     getenv ("HOME"), ARCHSTR, temp_gr, temp_cm);
    }

#if 0
    fprintf (stderr, "running '%s'\n", buf);
#endif

    executioner_pid = subproc_Popen (buf,
				     executioner_done,
				     executioner_done,
				     executioner_sink,
				     ep);
    if (executioner_pid < 0) {
	exit_run_mode ();
	msg_Format ("Couldn't run the HeNCE executioner: %s\n",
		    strerror (errno));
	goto cleanup;
    }
    ts_append (global.pvm_sink, "running HeNCE executioner...\n");
    return;

    /*
     * come here if we abort before running the executioner
     */
 cleanup:
    set_mode (MODE_COMPOSE);
    if (ep) {
	if (ep->trace_timer_id != 0) {
	    XtRemoveTimeOut (ep->trace_timer_id);
	    ep->trace_timer_id = 0;
	}
	FREE (ep);
    }
    if (fp_gr)
	fclose (fp_gr);
    if (temp_gr) {
	unlink (temp_gr);
	FREE (temp_gr);
    }
    if (temp_cm) {
	unlink (temp_cm);
	FREE (temp_cm);
    }
    return;
}


int
pvmglue_spawn (cmd, host)
char *cmd;
char *host;
{
    int tid;
#if 0
    if (strcmp (cmd, "proxyd") == 0 && strcmp (host, "thud") == 0) {
	(void) pvm_spawn (cmd, NULL, PvmTaskHost|PvmTaskDebug, host, 1, &tid);
	return tid;
    }
#endif
    (void) pvm_spawn (cmd, NULL, PvmTaskHost, host, 1, &tid);
    return tid;
}

/*
 * requests to be notified on taskexit are queued by pvm, even if we
 * make the same request multiple times.  So we maintain a list of
 * pending taskexit requests to make sure we don't ask for any particular
 * notification more than once.
 *
 * we assume the number of outstanding hosts is small, else we will
 * need to use a tree rather than a linear list.
 */

static struct foo {
    int tid;
    int msgtype;
    struct foo *next;
} *pending_requests = NULL;

int
pvmglue_notify_taskexit (tid, msgtype)
int tid;
int msgtype;
{
    struct foo *ptr;
    int x;

    /*
     * if we've already issued such a request, don't issue another
     * one.
     */
    for (ptr = pending_requests; ptr; ptr = ptr->next) {
	if (ptr->tid == tid && ptr->msgtype == msgtype)
	    return 0;
    }
    /*
     * if request is honored, remember it.
     */
    if ((x = pvm_notify (PvmTaskExit, msgtype, 1, &tid)) >= 0) {
	ptr = TALLOC (1, struct foo);
	if (ptr) {
	    ptr->tid = tid;
	    ptr->msgtype = msgtype;
	    ptr->next = pending_requests;
	    pending_requests = ptr;
	}
    }
    return x;
}


static int
free_pending_requests ()
{
    struct foo *ptr;

    while (pending_requests) {
	ptr = pending_requests;
	pending_requests = ptr->next;
	FREE (ptr);
    }
}

/*
 * ask pvm to notify us when a host has exited.
 *
 * XXX need to remember when we've already asked, so we don't ask twice.
 * pvm queues as many requests as we ask for.
 */

int
pvmglue_notify_hostexit (hostname, msgtype)
char *hostname;
int msgtype;
{
    int hosttid;
#ifdef PVM_STR
    struct pvmhostinfo *hostp;
#else
    struct hostinfo *hostp;
#endif
    int nhost;
    int status;
    int i;

    /*
     * find pvmd-tid for host
     */
    if ((status = pvm_config (&nhost, (int *) NULL, &hostp)) < 0)
	return status;
    /*
     * XXX in pvm3.2, pvm_config sometimes returns nhost == 0.  so
     * we retry.
     */
    if (nhost == 0) {
	if ((status = pvm_config (&nhost, (int *) NULL, &hostp)) < 0)
	    return status;
    }
    for (i = 0; i < nhost; ++i) {
	if (strcasecmp (hostname, hostp[i].hi_name) == 0) {
	    hosttid = hostp[i].hi_tid;
	    break;
	}
    }
    if (i == nhost) {
	msg_Format ("error in notify_hostexit: can't find host %s in config\n",
		    hostname);
	return PvmNoHost;
    }
    if ((status = pvm_notify (PvmHostDelete, msgtype, 1, &hosttid)) < 0) {
	msg_Format ("pvm_notify (PvmHostDelete, %d, 1, [%08x]) failed:%s\n",
		    msgtype, hosttid, pvm_strerror (status));
    }
    return hosttid;
}

int
pvmglue_kill (tid)
{
    return pvm_kill (tid);
}

char *
pvmglue_get_arch (hostname)
char *hostname;
{
#ifdef PVM_STR
    struct pvmhostinfo *hostp;
#else
    struct hostinfo *hostp;
#endif
    int nhost;
    int i;

    if (pvm_config (&nhost, 0, &hostp) < 0)
	return NULL;
    /*
     * pvm3.2 sometimes returns nhost == 0.  so retry
     */
    if (nhost == 0) {
	if (pvm_config (&nhost, 0, &hostp) < 0)
	    return NULL;
    }
    for (i = 0; i < nhost; ++i)
	if (strcasecmp (hostname, hostp[i].hi_name) == 0)
	    return hostp[i].hi_arch;
    return NULL;
}

struct recvargs {
    int srctid;
    int msgtag;
    int flag;
    int status;
};

static void
pvmglue_intr (p, fd, id)
XtPointer p;
int *fd;
XtInputId *id;
{
    struct recvargs *q = (struct recvargs *) p;
    int x;
    XClientMessageEvent pvm_done_event;

    if ((x = pvm_nrecv (q->srctid, q->msgtag)) != 0) {
#if 0
	fprintf (stderr, "pvm_nrecv: got one\n");
#endif
	/*
	 * send a dummy event so we will break out of the
	 * wait loop in pvmglue_recv().
	 */
	pvm_done_event.format = 32;
	pvm_done_event.data.l[0] = x;
	q->flag = 1;
	q->status = x;
    }
}

int
pvmglue_recv (srctid, msgtag)
int srctid, msgtag;
{
    XEvent event;
    XtInputId *input_ids;
    static struct recvargs x;
    int num_fds;
    int i;
    int *fds;

    XFlush (global.display);

    /*
     * if pvm already has a message ready, get it
     */

    if ((i = pvm_nrecv (srctid, msgtag)) != 0)
	return i;

    /*
     * initialize pvm_nrecv() arguments for callback
     */
    x.srctid = srctid;
    x.msgtag = msgtag;
    x.flag = 0;
    x.status = 0;

    /*
     * make sure we're waiting on all fds used by pvm
     */
    num_fds = pvm_getfds (&fds);
    input_ids = TALLOC (num_fds, XtInputId);
    for (i = 0; i < num_fds; ++i) {
#if 0
	fprintf (stderr, "XtAppAddInput (%d)\n", fds[i]);
#endif
	input_ids[i] = XtAppAddInput(global.context,
				     fds[i],
				     (XtPointer) XtInputReadMask,
				     pvmglue_intr,
				     (XtPointer) &x);
    }
    /*
     * process all X events until we get a message in
     */
	
    while (x.flag == 0)
	XtAppProcessEvent (global.context, XtIMXEvent | XtIMAlternateInput);

    /*
     * cancel all of the pending input events...
     */
    for (i = 0; i < num_fds; ++i) {
#if 0
	fprintf (stderr, "XtRemoveInput (%d)\n", fds[i]);
#endif
	XtRemoveInput (input_ids[i]);
    }
    FREE (input_ids);

    return x.status;
}
