/*
 * routines for unparsing HeNCE graphs into text form
 *
 * $Id: graph_unparse.c,v 1.1 1994/02/17 20:24:16 moore Exp $
 *
 * $Log: graph_unparse.c,v $
 * Revision 1.1  1994/02/17  20:24:16  moore
 * Initial revision
 *
 */

#include <stdio.h>
#include "rb.h"
#include "param.h"
#include "exp.h"
#include "graph.h"
#include "graph_unparse.h"

#include <Malloc.h>

static FILE *output_file;
static int unparse_errors;

/*
 * Print all of the arcs in the graph.
 * This is done after all node decls are printed, because an arc
 * decl must appear after all of the nodes it references.  It is
 * too hard to place them close to their corresponding node decls.
 */

static void
unparse_arcs (g)
Graph g;
{
    rbNode n1, n2;

    for (n1 = rb_First (g->nlist); !rb_Done (n1, g->nlist); n1 = rb_Next(n1)) {
	Node parent = rb_Value (n1);
	for (n2 = rb_First (parent->children); !rb_Done (n2, parent->children);
	     n2 = rb_Next (n2)) {
	    Node child = (Node) rb_Value (n2);
	    fprintf (output_file, "ARC %d %d\n", parent->nk.id,
		     child->nk.id);
	}
    }
}

/*
 * print out the name of a parameter, and any associated array
 * dimensions
 */

static int unparse_expr ();

static void
unparse_param (p)
Param p;
{
    fprintf (output_file, "%s", p->name);
    if (p->type == TYPE_ARRAY) {
	int i;
	/*
	 * XXX should this be p->a.ndims or p->a.nadims?
	 */
	for ( i = 0; i < p->a->ndims ; ++i) {
	    switch (p->a->dims[i]->elt_type) {
	    case OP_DIM:
	    case OP_EMPTY_DIM:
	    case OP_ARR_SECTION:
		unparse_expr (p->a->dims[i], 0);
		break;
	    default:
		msg_Format ("unparse_param: bogus type %d for dimension expr\n",
			    p->a->dims[i]->elt_type);
		unparse_errors++;
		break;
	    }
	}
    }
}

/*
 * print out the literal value of a constant
 */

static void
unparse_const (e)
Expr e;
{
    switch (e->type) {
    case TYPE_INT:
	fprintf (output_file, "%d", e->val.i);
	break;
    case TYPE_CHAR:
	/* XXX really not sure about this... */
	fprintf (output_file, "'%c'", e->val.c);
	break;
    case TYPE_FLOAT:
	fprintf (output_file, "%f", (double) e->val.f);
	break;
    case TYPE_DOUBLE:
	fprintf (output_file, "%f", e->val.d);
	break;
    case TYPE_STRING:
	/*
	 * XXX doesn't handle '"' inside a string, nor does
	 * it handle characters that need escapes (like \n)
	 */
	fprintf (output_file, "\"%s\"", e->val.s);
	break;
    case TYPE_ARRAY:
	/* XXX not sure how to handle this... */
	break;
    default:
	msg_Format ("unknown expr type %d in unparse_const\n", e->type);
	unparse_errors++;
	break;
    }
}

static struct {
    int op_code;
    int nargs;
    char *op_text;
    int precedence;
} op_table[] = {
    { OP_COND_EXPR,	3,  "",	    1,	},
    { OP_LOGICAL_OR,	2,  "||",   2,	},
    { OP_LOGICAL_AND,	2,  "&&",   3,	},
    { OP_LOGICAL_CMPL,	1,  "!",    12, },
    { OP_BIT_OR,	2,  "|",    4,	},
    { OP_BIT_AND,	2,  "&",    6,	},
    { OP_BIT_XOR,	2,  "^",    5,	},
    { OP_BIT_CMPL,	1,  "~",    12, },
    { OP_COMP_NE,	2,  "!=",   7,	},
    { OP_COMP_EQ,	2,  "==",   7,	},
    { OP_COMP_LE,	2,  "<=",   8,	},
    { OP_COMP_GE,	2,  ">=",   8,	},
    { OP_COMP_LT,	2,  "<",    8,	},
    { OP_COMP_GT,	2,  ">",    8,	},
    { OP_LSHIFT,	2,  "<<",   9,	},
    { OP_RSHIFT,	2,  ">>",   9,	},
    { OP_MULT,		2,  "*",    11, },
    { OP_DIV,		2,  "/",    11, },
    { OP_MOD,		2,  "%",    11, },
    { OP_ADD,		2,  "+",    10, },
    { OP_SUBTRACT,	2,  "-",    10, },
    { OP_NEG,		1,  "-",    13, },
    { OP_INDIR,		1,  "*",    13, },
    { OP_ADDRESS,	1,  "&",    13, },
    { OP_EMPTY_DIM,	0,  "",	    -1, },
    { OP_DIM,		1,  "",	    -1, },
    { OP_ARR_SECTION,	2,  "",	    -1, },
    { OP_PARAM,		0,  "",	    -1, },
    { OP_CONST,		0,  "",	    -1, },
};

#define SIZE(arr) (sizeof(arr)/sizeof(*arr))

static int
parens_reqd (parent_op, child_expr)
int parent_op;
Expr child_expr;
{
    int child = -1, parent = -1;
    int i;

    for (i = 0; i < SIZE(op_table); ++i) {
	if (child_expr->elt_type == op_table[i].op_code)
	    child = i;
	if (parent_op == op_table[i].op_code)
	    parent = i;
	if (child > 0 && parent > 0)
	    break;
    }
    /*
     * if the precedcence of the child operator is higher, than its
     * parent, we don't need parenthesis around the child expression.
     */
    if (op_table[child].precedence > op_table[parent].precedence)
	return 0;
    return 1;
}


/*
 * unparse an expression
 */

static int
unparse_expr (e, need_parens)
Expr e;
int need_parens;
{
    int i;

    if (e == (Expr) NULL) {
	msg_Format ("NULL expression\n");
	unparse_errors++;
	return 0;
    }

    for (i = 0; i < SIZE(op_table); ++i) {
	if (e->elt_type == op_table[i].op_code)
	    break;
    }
    if (i == SIZE (op_table)) {
	msg_Format ("unknown expr elt_type %d\n", e->elt_type);
	unparse_errors++;
	return 0;
    }
    if (op_table[i].nargs != e->nargs) {
	msg_Format ("wrong # of arguments for elt_type '%c'\n",
		    e->elt_type);
	unparse_errors++;
	return 0;
    }

    /*
     * eventually we will use associativity and precedence rules
     * to determine when to add parens.	 For the time being
     * we add them sometimes when they are not necessary.  However,
     * it is never necessary to parenthesize an atom, and we can
     * detect this case easily.
     */
    if (op_table[i].nargs == 0)
	need_parens = 0;

    if (need_parens)
	fprintf (output_file, "(");

    switch (e->elt_type) {
    case OP_EMPTY_DIM:
	fprintf (output_file, "[]");
	break;
    case OP_DIM:
	fprintf (output_file, "[");
	unparse_expr (e->args[0], 0);
	fprintf (output_file, "]");
	break;
    case OP_ARR_SECTION:
	fprintf (output_file, "[");
	unparse_expr (e->args[0], 0);
	fprintf (output_file, ":");		/* was " TO " */
	unparse_expr (e->args[1], 0);
	fprintf (output_file, "]");
	break;
    case OP_COND_EXPR:		/* ? : */
	unparse_expr (e->args[0], 1);
	fprintf (output_file, " ? ");
	unparse_expr (e->args[1], 1);
	fprintf (output_file, " : ");
	unparse_expr (e->args[2], 1);
	break;
    case OP_PARAM:
	unparse_param (e->val.p);
	break;
    case OP_CONST:
	unparse_const (e);
	break;
    default:
	if (op_table[i].nargs == 2) {
	    /* infix binary ops */
	    unparse_expr (e->args[0], parens_reqd (e->elt_type, e->args[0]));
	    fprintf (output_file, "%s", op_table[i].op_text);
	    unparse_expr (e->args[1], parens_reqd (e->elt_type, e->args[1]));
	    break;
	}
	else if (op_table[i].nargs == 1) {
	    /* prefix unary ops */
	    fprintf (output_file, "%s", op_table[i].op_text);
	    unparse_expr (e->args[0], parens_reqd (e->elt_type, e->args[0]));
	    break;
	}
	else
	    fprintf (output_file, "bogus: unrecognized op %d\n", e->elt_type);
    }
    if (need_parens)
	fprintf (output_file, ")");
    return 0;
}

/*
 * unparse an output declaration, i.e., print all non-input declarations
 * and ignore those with the input bit set.
 * called after printing out the sub call and its args.
 */

static int
unparse_output_decl (tn)
rbNode tn;
{
    Param p = (Param) rb_Value (tn);
    int type;

    if (p->io.in == 1 || p->io.out == 0)
	return 0;

#if 0
    if (p->io.new)
	fprintf (output_file, "NEW ");
#endif
    fprintf (output_file, "    > ");
    type = p->type == TYPE_ARRAY ? p->a->type : p->type;
    switch (type) {
    case TYPE_INHERITED:	/* leave out type name */
	break;
    case TYPE_INT:
	fprintf (output_file, "int ");
	break;
    case TYPE_FLOAT:
	fprintf (output_file, "float ");
	break;
    case TYPE_DOUBLE:
	fprintf (output_file, "double ");
	break;
    case TYPE_CHAR:
	fprintf (output_file, "char ");
	break;
    default:
	msg_Format ("output_decl: unknown parameter type %d for param %s\n",
		    p->type, p->name ? p->name : "(null)");
	unparse_errors++;
	break;
    }
    unparse_param (p);
    fprintf (output_file, ";\n");
    return 0;
}

/*
 * unparse all output declarations for this node
 */

static void
unparse_output_decl_list (n)
Node n;
{
    rbNode x;
    for (x = rb_First (n->params); !rb_Done (x, n->params); x = rb_Next (x))
	unparse_output_decl (x);
}

/*
 * unparse an input decl (that is, one that has the input bit set)
 * (i.e. ignore output-only decls)
 */

static int
unparse_input_decl (tn)
rbNode tn;
{
    Param p = (Param) rb_Value (tn);
    int type;

    if (p->io.in == 0)
	return 0;

    fprintf (output_file, "    ");
    if (p->io.new)
	fprintf (output_file, "NEW ");
    fprintf (output_file, p->io.out ? "<> " : "< ");
    type = p->type == TYPE_ARRAY ? p->a->type : p->type;
    switch (type) {
    case TYPE_INHERITED: 
	break;
    case TYPE_INT:
	fprintf (output_file, "int ");
	break;
    case TYPE_FLOAT:
	fprintf (output_file, "float ");
	break;
    case TYPE_DOUBLE:
	fprintf (output_file, "double ");
	break;
    case TYPE_CHAR:
	fprintf (output_file, "char ");
	break;
    default:
	msg_Format ("input_decl: unknown parameter type %d for param %s\n",
		    p->type, p->name ? p->name : "(null)");
	unparse_errors++;
	break;
    }
    unparse_param (p);
    if (p->val != (Expr) NULL) {
	fprintf (output_file, "=");
	unparse_expr (p->val, 0);
    }
    fprintf (output_file, ";\n");
    return 0;
}

/*
 * unparse all input decls for this particular node
 */

static void
unparse_input_decl_list (n)
Node n;
{
    rbNode x;

    for (x = rb_First (n->params); !rb_Done (x, n->params); x = rb_Next (x))
	unparse_input_decl (x);
}

/*
 * unparse a sub call, its return value (if any), and its args
 */

static void
unparse_sub_call (n)
Node n;
{
    int i;
    
    fprintf (output_file, "    ");
    if (n->ret_param) {
	unparse_param (n->ret_param);
	fprintf (output_file, " = ");
    }
    fprintf (output_file, " %s(", n->sub_name);
    for (i = 0; i < n->nargs; ++i) {
	unparse_expr (n->args[i], 0);
	if (i != n->nargs - 1)
	    fprintf (output_file, ", ");
    }
    fprintf (output_file, ");\n");
}

/*
 * unparse a node declaration
 */

static int
unparse_single_node (n)
Node n;
{
#define FP(z) fprintf(output_file, "%s", z)

    /* node keyword */
    switch (n->node_type) {
    case NODE_NORMAL:  FP("NODE ");	break;
    case NODE_COND:    FP("BEGINSWITCH ");break;
    case NODE_ENDCOND: FP("ENDSWITCH ");break;
    case NODE_LOOP:    FP("BEGINLOOP ");break;
    case NODE_ENDLOOP: FP("ENDLOOP ");	break;
    case NODE_FANOUT:  FP("FANOUT ");	break;
    case NODE_FANIN:   FP("FANIN ");	break;
    case NODE_PIPE:    FP("BEGINPIPE ");break;
    case NODE_ENDPIPE: FP("ENDPIPE ");	break;
    default:
	msg_Format ("node %d: unknown node type %d\n",
		    n->nk.id, n->node_type);
	unparse_errors++;
	return 0;
    }

    if (n->xy.set)
	fprintf (output_file, "[ %d %d ] ", n->xy.x, n->xy.y);
    fprintf (output_file, "%d", n->nk.id);

    switch (n->node_type) {
    case NODE_NORMAL:
	if (rb_Empty (n->params) && n->sub_name == (char *) NULL) {
	    FP("\n");
	    FP("/* replace the UNDEFINED keyword with the node program */\n");
	    FP("/*\n");
	    FP(" * Format: any input-only or input-output declarations go first, e.g:\n");
	    FP(" *	   NEW <> foo[100];\n");
	    FP(" * followed by a subr call, e.g:\n");
	    FP(" *	   x = xyzzy (a, b);\n");
	    FP(" * then any output-only declarations:\n");
	    FP(" *	   > bar[42];\n");
	    FP(" */\n");
	    FP("    UNDEFINED;\n");
	}
	else {
	    FP("\n");
	    unparse_input_decl_list (n);
	    unparse_sub_call (n);
	    unparse_output_decl_list (n);
	}
	break;
    case NODE_COND:
	if (n->nargs == 0) {
	    FP("\n");
	    FP("/*\n");
	    FP(" * Replace the UNDEFINED keyword with an expression in parenthesis. \n");
	    FP(" * If at run-time, the expression evaluates to nonzero, the subgraph\n");
	    FP(" * enclosed by this node and its matching node will be executed.\n");
	    FP(" */\n");
	    FP("    UNDEFINED;\n");
	}
	else {
	    FP("\n    ");
	    FP("(");
	    unparse_expr (n->args[0], 0);
	    FP(");\n");
	}
	break;
    case NODE_ENDCOND:
    case NODE_ENDLOOP:
    case NODE_ENDPIPE:
    case NODE_FANIN:
	FP("\n	  ;\n");
	break;
    case NODE_FANOUT:
    case NODE_PIPE:
	if (n->ret_param == (Param) NULL) {
	    FP("\n");
	    FP("/*\n");
	    FP(" * replace the UNDEFINED keyword with something like:\n");
	    FP(" *     variable = initial_value TO final_value\n");
	    FP(" * where \"variable\" is a scalar integer variable\n");
	    FP(" * and \"initial_value\" and \"final_value\" are expressions\n");
	    FP(" * (also, make sure the TO keyword is in upper case)\n");
	    FP(" */\n");
	    FP("    UNDEFINED;\n");
	}
	else {
	    FP("\n    ");
	    unparse_param (n->ret_param);
	    FP(" = ");
	    unparse_expr (n->args[0], 0);
	    FP(" TO ");
	    unparse_expr (n->args[1], 0);
	    FP(";\n");
	}
	break;
    case NODE_LOOP:	
	if (n->ret_param == (Param) NULL) {
	    FP("\n");
	    FP("/* replace the UNDEFINED keyword with: \n");
	    FP(" * ( variable = initial_value ; ");
	    FP(" *   termination_condition ; ");
	    FP(" *   variable = next_value ) \n");
	    FP(" * where \"variable\" is a integer.\n");
	    FP(" * NB: The same variable name MUST be used for the result of\n");
	    FP(" * both the \"initial_value\" and \"next_value\" expressions.\n");
	    FP(" * The \"termination_condition\" can be any expression.\n");
	    FP(" */\n");
	    FP("    UNDEFINED;\n");
	}
	else {	    
	    FP("\n    ");
	    FP("(");
	    unparse_param (n->ret_param);
	    FP("=");
	    unparse_expr (n->args[0], 0);
	    FP("; ");
	    unparse_expr (n->args[1], 0);
	    FP("; ");
	    unparse_param (n->ret_param);
	    FP("=");
	    unparse_expr (n->args[2], 0);
	    FP(");\n");
	}
	break;
    }
    return 0;

#undef FP
}


static void
unparse_nodes (g)
Graph g;
{
    rbNode x;

    for (x = rb_First (g->nlist); !rb_Done (x, g->nlist); x = rb_Next (x)) {
	Node n = rb_Value (x);
	unparse_single_node (n);
    }
}

int
graph_unparse (g, output)
Graph g;
FILE *output;
{
    if (output != (FILE *) NULL)
	output_file = output;
    else
	output_file = stdout;

    unparse_errors = 0;
    fprintf (output_file, "PROGRAM\n");
    unparse_nodes (g);
    unparse_arcs (g);
    return unparse_errors;
}

int
graph_unparse_node (n, output)
Node n;
FILE *output;
{
    if (output != (FILE *) NULL)
	output_file = output;
    else
	output_file = stdout;

    unparse_errors = 0;
    unparse_single_node (n);
    return unparse_errors;
}
