/*
 *	HeNCE Tool
 *
 *	graph_print.c - Postscript graph routines.
 *
 *	originally from "psgraph.c", written Jun 1991 by Robert Manchek.
 *
 * $Id: graph_print.c,v 1.1 1994/02/17 20:24:03 moore Exp $
 *
 * $Log: graph_print.c,v $
 * Revision 1.1  1994/02/17  20:24:03  moore
 * Initial revision
 *
 *
 */

#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <pwd.h>
#include <sys/types.h>
#include "rb.h"
#include "param.h"
#include "exp.h"		/* XXX Yuk! */
#include "graph.h"
#include <Malloc.h>
#include <String.h>

/* these define the arrowhead size and proximity to node */

#define	NODERADIUS 10		/* size of graph node circle */
#define	ARROWX	NODERADIUS/2
#define	ARROWY	(NODERADIUS/4)
#define	ARROWPROX (NODERADIUS*5/4)

extern char *psheader[];	/* from psheader.c */

static char *psnodenames[] = {
    "RegularNode",
    "BeginSwitchNode",
    "EndSwitchNode",
    "BeginLoopNode",
    "EndLoopNode",
    "BeginFanNode",
    "EndFanNode",
    "BeginPipeNode",
    "EndPipeNode",
    0
};

/*
 *	Write graph to a text file in encapsulated postscript.
 */

void
graph_print (ff, grf, title)
FILE *ff;
Graph grf;
char *title;
{
    rbTree tr = grf->nlist;
    rbTree tra;
    rbNode tn, tna;
    Node n, n2;
    time_t now = time ((long *) 0);
    int bbx1 = 0, bby1 = 0, bbx2 = 0, bby2 = 0;
    int x, y;
    char **pp;
    struct passwd *pe;
    char *loginname = STRDUP ("anonymous");
    char hostname[1024];

    if (pe = getpwuid (getuid ())) {
	FREE (loginname);
	loginname = STRDUP (pe->pw_name);
    }
    endpwent();

    gethostname (hostname, sizeof hostname);

    /* get bbx of graph */

    if (!rb_Empty (tr)) {
	n = (Node) rb_Value (tn = rb_First (tr));
	bbx1 = bbx2 = n->xy.x;
	bby1 = bby2 = n->xy.y;

	while (!rb_Done ((tn = rb_Next (tn)), tr)) {
	    n = (Node) rb_Value(tn);
	    if (n->xy.x < bbx1) bbx1 = n->xy.x;
	    if (n->xy.x > bbx2) bbx2 = n->xy.x;
	    if (n->xy.y < bby1) bby1 = n->xy.y;
	    if (n->xy.y > bby2) bby2 = n->xy.y;
	}
    }
    bbx1 -= NODERADIUS*2;
    bby1 -= NODERADIUS*2;
    bbx2 += NODERADIUS*2;
    bby2 += NODERADIUS*2;

    /* write eps header */

    fprintf(ff, "%%!\n");
    fprintf(ff, "%%%%Title: %s\n", title);
    fprintf(ff, "%%%%Creator: %s\n", "HeNCE tool v2.0");
    fprintf(ff, "%%%%CreationDate: %s", ctime (&now));
    fprintf(ff, "%%%%For: %s@%s\n", loginname, hostname);
    fprintf(ff, "%%%%Pages: 0\n");
    fprintf(ff, "%%%%BoundingBox: %d %d %d %d\n", bbx1, bby1, bbx2, bby2);
    fprintf(ff, "%%%%EndComments\n");

    /* write postscript function defs */
    for (pp = psheader; *pp; pp++)
	fputs (*pp, ff);

    fprintf (ff, "/NodeRad %d def\n", NODERADIUS);
    fprintf (ff, "/NodeRad2 NodeRad 2 div def\n");
    fprintf (ff, "/ArrowX %d def\n", ARROWX);
    fprintf (ff, "/ArrowY %d def\n", ARROWY);
    fprintf (ff, "/ArrowProx %d def\n", ARROWPROX);

    /* draw all the node outlines */

    fprintf (ff, "/Helvetica findfont %d scalefont setfont\n",
	    NODERADIUS*2);
    for (tn = rb_First (tr); !rb_Done (tn, tr); tn = rb_Next (tn)) {
	n = (Node) rb_Value (tn);
	x = n->xy.x;
	y = bby2 - n->xy.y + bby1;
	fprintf (ff, "%d %d %s\n", x, y, psnodenames[n->node_type]);
    }

    /* draw all the arcs */

    for (tn = rb_First (tr); !rb_Done (tn, tr); tn = rb_Next (tn)) {
	n = (Node) rb_Value (tn);
	tra = n->children;
	for (tna = rb_First (tra); !rb_Done (tna, tra); tna = rb_Next (tna)) {
	    n2 = (Node) rb_Value (tna);
	    fprintf (ff, "%d %d %d %d ConnectArc\n",
		     n->xy.x, bby2 - n->xy.y + bby1,
		     n2->xy.x, bby2 - n2->xy.y + bby1);
	}
    }

    /* label all the nodes */

    fprintf (ff, "/Helvetica findfont %d scalefont setfont\n",
	     NODERADIUS);
    for (tn = rb_First (tr); !rb_Done (tn, tr); tn = rb_Next (tn)) {
	n = (Node) rb_Value (tn);
	x = n->xy.x;
	y = bby2 - n->xy.y + bby1;
	fprintf(ff, "%d %d (%d) LabelNodeId\n", x, y, n->nk.id);

	switch (n->node_type) {
	case NODE_NORMAL:
	    if (n->sub_name) {
		fprintf (ff, "%d %d (%s()) LabelNodeParam\n",
			 x, y, n->sub_name);
	    }
	    break;

#if 0
	case NTYPE_BP:
	case NTYPE_BL:
	case NTYPE_FO:
	    fprintf (ff, "%d %d (%s; (%s)) LabelNodeParam\n",
		     x, y, (n->fun ? n->fun : ""), (n->par ? n->par : ""));
	    break;

	case NTYPE_BS:
	    fprintf (ff, "%d %d ((%s)) LabelNodeParam\n",
		    x, y, (n->par ? n->par : ""));
	    break;
#endif

	default:
	    break;
	}
    }

    fprintf (ff, "showpage\n");
}

