/*
 * htool version 2.x
 * routines to manage the graph panel
 * Keith Moore
 *
 * $Id: graph_panel.c,v 1.1 1994/02/17 20:23:11 moore Exp $
 *
 * $Log: graph_panel.c,v $
 * Revision 1.1  1994/02/17  20:23:11  moore
 * Initial revision
 *
 */

#include <stdio.h>
#include <varargs.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#if 0
#include <X11/IntrinsicP.h>
#include <X11/CoreP.h>
#endif

#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/Sme.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/Toggle.h>

/*
 * from bondo...
 */
#include <Errno.h>
#include <String.h>

#include "Canvas.h"
#include "FileSelect.h"
#include "filenames.h"
#include "global.h"
#include "graph.h"
#include "graph_critic.h"
#include "graph_draw.h"
#include "graph_edit.h"
#include "graph_panel.h"
#include "graph_parse.h"
#include "graph_unparse.h"
#include "help.h"
#include "menu.h"
#include "mode.h"
#include "trace_new.h"
#include "xtglue.h"

#include "bitmaps/fwd.xbm"
#include "bitmaps/fwdstep.xbm"
#include "bitmaps/rev.xbm"
#include "bitmaps/revstep.xbm"
#include "bitmaps/stop.xbm"
#include "bitmaps/beginloop.xbm"
#include "bitmaps/endloop.xbm"
#include "bitmaps/beginswitch.xbm"
#include "bitmaps/endswitch.xbm"
#include "bitmaps/beginpipe.xbm"
#include "bitmaps/endpipe.xbm"
#include "bitmaps/fanout.xbm"
#include "bitmaps/fanin.xbm"
#include "bitmaps/node.xbm"
#ifdef SCISSORS
#include "bitmaps/scissors.xbm"
#endif
#include "bitmaps/lines.xbm"

#include <Malloc.h>

static Widget graphCanvas;
Widget trace_play_button;	/* scribbled on by trace_new.c */

#ifdef max
#undef max
#endif
#define max(a,b) ((a) > (b) ? (a) : (b))
#ifdef min
#undef min
#endif
#define min(a,b) ((a) < (b) ? (a) : (b))

#define HELP_POPUP 1

#ifdef HELP_POPUP
/* XXX sigh...this causes my DrekWindows server to crash */

static int help_visible = 0;
static Widget help_popup = NULL;
static int help_page = 0;
static Widget help_canvas;
#define HELP_HEIGHT 300
#define HELP_WIDTH  400

static void
help_expose (w, cli, xev, rem)
Widget w;
XtPointer cli;
XEvent *xev;
Boolean *rem;
{
    if (xev->type != Expose || xev->xexpose.count != 0)
	return;

    switch (help_page) {
    default:
	display_help_page (w, help_page);
    }
}

#define XtVaCMW XtVaCreateManagedWidget

static void
new_page (w, cli, cd)
Widget w;
XtPointer cli;
XtPointer cd;
{
    char *f = (char *) cli;
    switch (*f) {
    case 'n':
	help_page++;
	display_help_page (help_canvas, help_page);
	break;
    case 'p':
	if (--help_page < 0)
	    help_page = 0;
	display_help_page (help_canvas, help_page);
	break;
    case 'd':
	XtPopdown (help_popup);
	help_visible = 0;
	break;
    }
}

void
help_button (unused, cli, cd)
Widget unused;
XtPointer cli;
XtPointer cd;
{
    Widget form;

    if (help_visible = !help_visible) {
	Widget w;

	if (help_popup == NULL) {
	    help_popup = XtVaCreatePopupShell ("help",
					       topLevelShellWidgetClass,
					       global.topLevel,
					       XtNtitle, "HeNCE Help",
					       NULL);
	    
	    w = XtVaCMW ("form",
			 formWidgetClass,
			 help_popup,
			 XtNwidth, HELP_WIDTH,
			 XtNheight, HELP_HEIGHT,
			 NULL);
	    form = w;
	    w = XtVaCMW ("next",
			 commandWidgetClass,
			 form,
			 XtNlabel, "Next",
			 /* form constraints */
			 XtNtop, XtChainTop,
			 XtNbottom, XtChainTop,
			 XtNleft, XtChainLeft,
			 XtNright, XtChainLeft,
			 XtNfromVert, NULL,
			 XtNfromHoriz, NULL,
			 NULL);
	    XtAddCallback (w, XtNcallback, (XtCallbackProc) new_page,
			   (XtPointer) "next");

	    w = XtVaCMW ("prev",
			 commandWidgetClass,
			 form,
			 XtNlabel, "Prev",
			 /* form constraints */
			 XtNtop, XtChainTop,
			 XtNbottom, XtChainTop,
			 XtNleft, XtChainLeft,
			 XtNright, XtChainLeft,
			 XtNfromVert, NULL,
			 XtNfromHoriz, w,
			 NULL);
	    XtAddCallback (w, XtNcallback, (XtCallbackProc) new_page,
			   (XtPointer) "prev");

	    w = XtVaCMW ("done",
			 commandWidgetClass,
			 form,
			 XtNlabel, "Done",
			 /* form constraints */
			 XtNtop, XtChainTop,
			 XtNbottom, XtChainTop,
			 XtNleft, XtChainLeft,
			 XtNright, XtChainLeft,
			 XtNfromVert, NULL,
			 XtNfromHoriz, w,
			 NULL);
	    XtAddCallback (w, XtNcallback, (XtCallbackProc) new_page,
			   (XtPointer) "done");

	    w = XtVaCMW ("canvas",
			 canvasWidgetClass,
			 form,
			 XtNwidth, HELP_WIDTH,
			 XtNheight, HELP_HEIGHT - height_of (w),
			 /* form constraints */
			 XtNtop, XtChainTop,
			 XtNbottom, XtChainBottom,
			 XtNleft, XtChainLeft,
			 XtNright, XtChainRight,
			 XtNfromVert, w,
			 XtNfromHoriz, NULL,
			 NULL);
	    help_canvas = w;
	    XtPopup (help_popup, XtGrabNone);
	    XtAddEventHandler (help_canvas, ExposureMask, False,
			       help_expose, NULL);
	}
	else
	    XtPopup (help_popup, XtGrabNone);
    }
    else
	XtPopdown (help_popup);
}
#endif /* HELP_POPUP */

/*
 * utility function to redraw the entire graph window
 *
 * XXX might want to modify this to pass a region to minimize the
 * amount of net traffic to the server.
 */

void
redraw_graph_window (clearfirst)
int clearfirst;
{
    if (clearfirst)
	XClearWindow (global.display, XtWindow (graphCanvas));

    if (global.mode == MODE_EXECUTE || global.mode == MODE_TRACE) {
	draw_graph (graphCanvas, global.traceGraph, 1);
	draw_time (graphCanvas, &global.traceClock, global.traceSequence);
    }
    else
	draw_graph (graphCanvas, global.graph, 0);
}


/*
 * handle events on the graphCanvas window
 */

static void
graph_event (w, cli, xev, rem)
Widget w;
XtPointer cli;
XEvent *xev;
Boolean *rem;
{
    XButtonEvent *bev = (XButtonEvent*)xev;
    XConfigureEvent *cev = (XConfigureEvent*)xev;
    XExposeEvent *eev = (XExposeEvent*)xev;
    XMotionEvent *mev = (XMotionEvent*)xev;
    XCrossingEvent *rev = (XCrossingEvent*)xev;

    switch (xev->type) {

    case Expose:
	if (eev->count == 0) {
	    if (global.graph)
		redraw_graph_window (0);
	}
	break;

    case ConfigureNotify:
	break;

    }
}

/*
 * function to handle the "Load Graph" menu item.
 *
 * Can be called from any mode, but not
 *
 * while a trace is in progress (XXX fix this)
 * while a program is running (XXX)
 * while building an application (XXX)
 */

static void
load_graph (w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    char graphFile[1024];
    FILE *fp;
    int fgetc();

    strcpy (graphFile, make_pretty (global.graphFile));
    if (fileVerify ("Load HeNCE graph from...",
		    graphFile, sizeof (graphFile), "*.gr",
		    FS_RegularFile | FS_MustExist | FS_MustBeReadable) == 0) {
	if ((fp = fopen (graphFile, "r")) == NULL) {
	    msg_Format ("can't open \"%s\": %s\n", graphFile,
			strerror (errno));
	    return;
	}
	/*
	 * process graph file 
	 */
	if ((global.graph = parse_Program (fgetc, fp, graphFile)) == NULL) {
	    msg_Format ("couldn't parse graph: %s\n", graphFile);
	    global.graph = gr_New ();
	    *global.graphFile = '\0';
	}
	else {
	    msg_Format ("graph file %s loaded.\n", graphFile);
	    strcpy (global.graphFile, graphFile);
	    gr_Free (global.traceGraph);
	    global.traceGraph = gr_CopyGraph (global.graph);
	    /* new_graph_file (); */
	}
	fclose (fp);
	set_graphfile_label (make_pretty (global.graphFile), 0);
	/* scale_graph (graphCanvas, global.graph); */
	redraw_graph_window (1);
	global.graph->modified = 0;
    }
}

/*
 * "Save as" menu item for graph window.
 * Save graph after prompting the user for a file name.
 * Also changes the name of the file for the next "save".
 */

static void
save_graph_as (w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    char graphFile[1024];
    FILE *fp;
    int fgetc();

    strcpy (graphFile, make_pretty (global.graphFile));
    if (fileVerify ("Save a copy of the HeNCE graph to...",
		    graphFile, sizeof (graphFile), "*.gr",
		    FS_RegularFile | FS_MustBeWritable) == 0) {
	if ((fp = fopen (graphFile, "w")) == NULL) {
	    msg_Format ("can't open \"%s\" for writing: %s\n", graphFile,
			strerror (errno));
	    return;
	}
	/*
	 * process graph file 
	 */
	if (graph_unparse (global.graph, fp) != 0)
	    msg_Format ("warning: errors in graph prevent saving.\n");
	else {
	    msg_Format ("graph saved to \"%s\".\n", graphFile);
	    strcpy (global.graphFile, make_pretty (graphFile));
	    set_graphfile_label (global.graphFile, 0);
	    global.graph->modified = 0;
	}
	fclose (fp);
    }
}


/*
 * "Save" menu item for graph window.
 * If there is no default file name, do a "Save As".
 */

static void
save_graph (w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    char graphFile[1024];
    FILE *fp;
    int fgetc();

    if (global.graphFile == NULL || *global.graphFile == '\0') {
	save_graph_as (w, client_data, call_data);
	return;
    }
    strcpy (graphFile, make_pretty (global.graphFile));
    if ((fp = fopen (graphFile, "w")) == NULL) {
	msg_Format ("can't open \"%s\" for writing: %s\n", graphFile,
		    strerror (errno));
	return;
    }
    /*
     * process graph file 
     */
    if (graph_unparse (global.graph, fp) != 0) {
	msg_Format ("warning: errors in graph prevent saving.\n");
    }
    else {
	msg_Format ("graph written to \"%s\".\n", graphFile);
	global.graph->modified = 0;
    }
    fclose (fp);
}

static void
print_graph (w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    char tmpFile[1024];
    char command[1024];
    FILE *fp;
    int pid;

    strcpy (tmpFile, defaults.tmpDir);
    strcat (tmpFile, "/htoolXXXXXX");
    mktemp (tmpFile);

    if ((fp = fopen (tmpFile, "w")) == NULL) {
	msg_Format ("can't create temp file \"%s\": %s\n",
		    tmpFile, strerror (errno));
	return;
    }
    graph_print (fp, global.graph, (global.graphFile ?
				    make_pretty (global.graphFile) :
				    "untitled"));
    fclose (fp);
    sprintf (command, defaults.printCommand, tmpFile);
    if (subproc_Spawn (command, NULL, NULL, NULL) < 0) {
	msg_Format ("command failed: \"%s\"\n", command);
	return;
    }
    return;
}


static void
save_postscript (w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    char graphFile[1024];
    FILE *fp;
    int fgetc();

    strcpy (graphFile, make_pretty (global.graphFile));
    if (fileVerify ("Save a PostScript picture of the graph to...",
		    graphFile, sizeof (graphFile), "*.ps",
		    FS_RegularFile | FS_MustBeWritable) == 0) {
	if ((fp = fopen (graphFile, "w")) == NULL) {
	    msg_Format ("can't open \"%s\" for writing: %s\n", graphFile,
			strerror (errno));
	    return;
	}
	/*
	 * process graph file 
	 */
	graph_print (fp, global.graph, (global.graphFile ?
					make_pretty (global.graphFile) :
					"untitled"));
	fclose (fp);
    }
}


/*
 * create a new graph.  Compose mode only.
 *
 * XXX need to give luser a chance to save if modified
 */

static void
new_graph (w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    if (global.mode == MODE_COMPOSE) {
	global.graph = gr_New ();
	redraw_graph_window (1);
    }
    *global.graphFile = '\0';
    set_graphfile_label ("", 0);
}

struct menu_entry graph_file_menu[] = {
    { "c",    	"New",        new_graph },
    { "c",	"Load",       load_graph },
    { "c",	"Save",       save_graph },
    { "c",      "Save as...", save_graph_as },
    { "c", 	"Print",      print_graph },
    { "c", 	"Write PostScript to...", save_postscript },
    { NULL, NULL },
};


/*
 * check graph for errors.
 */

static void
check_graph (w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    int e;
    Graph g = global.graph;

    e = graph_critic (g);
    if (e == 0)
	e = subdefs_check_graph (g, defaults.subDefsFile);
    subdefs_nuke_decls ();

    if (e)
	msg_Format ("%d errors found.\n", e);
    else
	msg_Format ("no errors found.\n");
    XClearWindow (global.display, XtWindow (graphCanvas));
    redraw_graph_window (0);
}


/*
 * scale graph to fit bounds of windows.  modifies graph.
 *
 * XXX instead of modifying graph, change the bounds and scale
 * used to display the graph.
 */

static void
scale_graph_to_fit (w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    if (global.mode == MODE_TRACE || global.mode == MODE_EXECUTE)
	scale_graph (graphCanvas, global.traceGraph);
    else
	scale_graph (graphCanvas, global.graph);
    redraw_graph_window (1);
    if (global.graph->modified)
	set_graphfile_label (make_pretty (global.graphFile), 1);
}


/*
 * clear graph.
 *
 * XXX need to either ask first if graph is not saved, or allow undo.
 *
 * XXX what's the difference between clear and new?
 */

static void
clear_graph (w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    if (global.mode == MODE_COMPOSE) {
	global.graph = gr_New ();
	redraw_graph_window (1);
    }
}

static void
cleanup_graph (w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    /*
     * XXX if nodes are being edited, we can't clean up.
     * otherwise, make sure the routine that splices the editing
     * stuff back into the graph ignores the coordinates in the
     * file and uses the coordinates in the graph
     */

    if (global.mode == MODE_COMPOSE) {
	Dimension width, height;
	
	if (graph_has_cycles (global.graph)) {
	    msg_Format ("can't cleanup: graph has a loop.\n");
	    return;
	}
	XtVaGetValues (graphCanvas,
		       XtNwidth, &width,
		       XtNheight, &height,
		       NULL);
	graph_place_nodes (global.graph, width, height);
	redraw_graph_window (1);
    }
}

struct menu_entry graph_edit_menu[] = {
#ifdef XXX
    { "c", "Undo", NULL, },
#endif
    { "c", 	"Scale to Fit", scale_graph_to_fit },
    { "c",     	"Cleanup",      cleanup_graph },
    { "c", 	"Critic",       check_graph },
#if 0
    { "c",     	"Clear",        clear_graph },
#endif
    { NULL, NULL },
};

static void
load_trace_file (w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    char traceFile[1024];
    FILE *fp;

    strcpy (traceFile, make_pretty (global.traceFile));
    if (fileVerify ("Load HeNCE trace file from...",
		    traceFile, sizeof (traceFile), "*.trace",
		    FS_RegularFile | FS_MustExist | FS_MustBeReadable) == 0) {

	/*
	 * process trace file 
	 */
	if (trace_load (traceFile) < 0) {
	    return;
	}
	else {
	    msg_Format ("trace file %s loaded.\n", traceFile);
	    strcpy (global.traceFile, traceFile);
	    /* new_trace_file (); */	/* XXX */
	}
	set_tracefile_label (make_pretty (global.traceFile), 0);
    }
}

#if 0
static void
set_trace_file (w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    char traceFile[1024];
    FILE *fp;

    strcpy (traceFile, make_pretty (global.traceFile));
    if (fileVerify ("Save HeNCE execution trace data in...",
		    traceFile, sizeof (traceFile), "*.trace",
		    FS_RegularFile | FS_MustBeWritable) == 0) {
	strcpy (global.traceFile, traceFile);
	set_tracefile_label (make_pretty (global.traceFile), 0);
    }
}
#endif

extern void exit_trace_mode ();	/* in trace_new.c */

struct menu_entry graph_trace_menu[] = {
#if 0
    { "c",   	"Set Trace File",  set_trace_file },
#endif
    { "c", 	"Load Trace File", load_trace_file },
    { "t", 	"Exit Trace Mode", exit_trace_mode },
    { NULL, NULL },
};

/*
 * XXX to be really cute, change cursor in graphCanvas whenever we
 * change the palette.  Especially when doing scissors.
 */

static void
select_palette (w, cli, cd)
Widget w;
XtPointer cli;
XtPointer cd;
{
    int node_type = (int) cli;

#ifdef SCISSORS
    /*
     * XXX need to detect (node_type == -1) and cut node or arc, on button
     * press, rather than adding one.
     */
#endif

    edit_select_node_type (node_type);
}


#define XtVaCMW XtVaCreateManagedWidget
#define XtGCFSW XtGlueCreateFormSubWidget

void
make_graph_panel (parent, height, width)
Widget parent;
int height, width;
{
    Widget graphTitle;
    Widget nodePalette;
    int defaultDistance = 4;	/* XXX get this from parent */
    int borderWidth = 1;	/* XXX get this from parent */
    int menuWidth = 0;
    int menuHeight = 0;
    int traceWidth = 0;
    int traceHeight = 0;
    int paletteWidth = 0;
    int paletteHeight = 0;
    char *modes;
    XtTranslations radioGroupTranslations = XtParseTranslationTable ("\
<EnterWindow>: highlight(Always)\n\
<LeaveWindow>: unhighlight()\n\
<Btn1Down>,<Btn1Up>: set()notify()\n");

    /*
     * title bar across the top
     */

    graphTitle = XtGCFSW ("title", labelWidgetClass, parent, North,
			  XtNlabel, "HeNCE graph",
			  XtNwidth, (width -
				     2 * defaultDistance -
				     2 * borderWidth),
			  XtNresize, False,
			  XtNresizable, False,
			  XtNforeground, defaults.titleForeground,
			  XtNbackground, defaults.titleBackground,
			  NULL);

    /*
     * now draw menu bar underneath title, to the left.
     */

    {
	Widget above = graphTitle;
	Widget w = NULL;

	menuWidth = defaultDistance;
	menuHeight = 0;

	w = XtGCFSW ("menu.help", commandWidgetClass, parent, NorthWest,
		     XtNlabel, "Help",
		     XtNfromVert, above,
		     XtNfromHoriz, w,
		     NULL);
#ifdef HELP_POPUP
	XtAddCallback (w, XtNcallback, (XtCallbackProc) help_button,
		       (XtPointer) NULL);
#endif

	menuWidth += width_of (w) + defaultDistance;
	menuHeight = max (menuHeight, height_of (w));

	modes = make_menu ("graphFileMenu", graph_file_menu);

	w = XtGCFSW ("menu.file", menuButtonWidgetClass, parent, NorthWest,
		     XtNlabel, "File",
		     XtNmenuName, "graphFileMenu",
		     XtNfromVert, above,
		     XtNfromHoriz, w,
		     NULL);
	mode_mark_widget (w, modes);
	menuWidth += width_of (w) + defaultDistance;
	menuHeight = max (menuHeight, height_of (w));

	modes = make_menu ("graphEditMenu", graph_edit_menu);

	w = XtGCFSW ("menu.edit", menuButtonWidgetClass, parent, NorthWest,
		     XtNlabel, "Edit",
		     XtNmenuName, "graphEditMenu",
		     XtNfromVert, above,
		     XtNfromHoriz, w,
		     NULL);
	mode_mark_widget (w, modes);
	menuWidth += width_of (w) + defaultDistance;
	menuHeight = max (menuHeight, height_of (w));

	modes = make_menu ("graphTraceMenu", graph_trace_menu);

	w = XtGCFSW ("menu.trace", menuButtonWidgetClass, parent, NorthWest,
		     XtNlabel, "Trace",
		     XtNmenuName, "graphTraceMenu",
		     XtNfromVert, above,
		     XtNfromHoriz, w,
		     NULL);
	mode_mark_widget (w, modes);
	menuWidth += width_of (w) + defaultDistance;
	menuHeight = max (menuHeight, height_of (w));
    }

    /*
     * trace buttons go under title, but to the far right.
     * we have to figure out how wide they are before we can
     * set the x coordinate of the leftmost trace button.
     * With the form widget, we set the initial X coordinate
     * by setting the fromHoriz resource to NULL and the
     * horizDistance resource to the initial value for X.
     */

    {
	Widget above = graphTitle;
	Widget w = NULL;
	Widget leftMost;

	traceWidth = 0;
	traceHeight = 0;

	w = XtGCFSW ("trace.rewind", commandWidgetClass, parent, NorthEast,
		     XtNbitmap, XtGlueMakePixmap (rev_bits,
						  rev_width,
						  rev_height),
		     XtNfromVert, above,
		     XtNfromHoriz, w,
		     NULL);
	XtAddCallback (w, XtNcallback, trace_rwd_button, NODE_NORMAL);
	mode_mark_widget (w, "t"); /* trace mode only */

	traceWidth += width_of (w) + defaultDistance;
	traceHeight = max (traceHeight, height_of (w));
	leftMost = w;

	w = XtGCFSW ("trace.rewindStep", commandWidgetClass, parent, NorthEast,
		     XtNbitmap, XtGlueMakePixmap (revstep_bits,
						  revstep_width,
						  revstep_height),
		     XtNfromVert, above,
		     XtNfromHoriz, w,
		     NULL);
	XtAddCallback (w, XtNcallback, trace_rwdstep_button, NODE_NORMAL);
	mode_mark_widget (w, "t"); /* trace mode only */

	traceWidth += width_of (w) + defaultDistance;
	traceHeight = max (traceHeight, height_of (w));

	w = XtGCFSW ("trace.stop", commandWidgetClass, parent, NorthEast,
		     XtNbitmap, XtGlueMakePixmap (stop_bits,
						  stop_width,
						  stop_height),
		     XtNstate, True,
		     XtNfromVert, above,
		     XtNfromHoriz, w,
		     NULL);
	XtAddCallback (w, XtNcallback, trace_stop_button, NODE_NORMAL);
	mode_mark_widget (w, "t"); /* trace mode only */

	traceWidth += width_of (w) + defaultDistance;
	traceHeight = max (traceHeight, height_of (w));

	w = XtGCFSW ("trace.forwardStep", commandWidgetClass, parent,
		     NorthEast,
		     XtNbitmap, XtGlueMakePixmap (fwdstep_bits,
						  fwdstep_width,
						  fwdstep_height),
		     XtNfromVert, above,
		     XtNfromHoriz, w,
		     NULL);
	XtAddCallback (w, XtNcallback, trace_fwdstep_button, NODE_NORMAL);
	mode_mark_widget (w, "t"); /* trace mode only */

	traceWidth += width_of (w) + defaultDistance;
	traceHeight = max (traceHeight, height_of (w));

	w = XtGCFSW ("trace.forward", toggleWidgetClass, parent, NorthEast,
		     XtNbitmap, XtGlueMakePixmap (fwd_bits,
						  fwd_width,
						  fwd_height),
		     XtNstate, False,
		     XtNfromVert, above,
		     XtNfromHoriz, w,
		     NULL);
	XtAddCallback (w, XtNcallback, trace_fwd_button, NODE_NORMAL);
	mode_mark_widget (w, "t"); /* trace mode only */
	trace_play_button = w;

	traceWidth += width_of (w) + defaultDistance;
	traceHeight = max (traceHeight, height_of (w));

	XtVaSetValues (leftMost,
		       XtNfromHoriz, NULL,
		       XtNhorizDistance, width - traceWidth,
		       NULL);
    }

    /*
     * Now draw the node palette buttons.  These go on the left,
     * underneath the menu and trace buttons.
     */

    {
	Widget w = graphTitle;
	Widget left = NULL;

	paletteWidth = 0;
	paletteHeight = 0;

	w = XtGCFSW ("palette.node", toggleWidgetClass, parent, NorthWest,
		     XtNbitmap, XtGlueMakePixmap (node_bits,
						  node_width,
						  node_height),
		     XtNstate, True,
		     XtNtranslations, radioGroupTranslations,
		     XtNfromHoriz, left,
		     XtNfromVert, w,
		     XtNvertDistance, (max (menuHeight, traceHeight) +
				       2 * defaultDistance),
		     NULL);
	XtAddCallback (w, XtNcallback, select_palette, NODE_NORMAL);
	/*
	 * functions only in compose mode
	 */
	mode_mark_widget (w, "c");
	paletteWidth = max (paletteWidth, width_of (w));
	paletteHeight += height_of (w) + defaultDistance;

	w = XtGCFSW ("palette.endLoop", toggleWidgetClass, parent, NorthWest,
		     XtNbitmap,  XtGlueMakePixmap (endloop_bits,
						   endloop_width,
						   endloop_height),
		     XtNtranslations, radioGroupTranslations,
		     XtNradioGroup, w,
		     XtNfromHoriz, left,
		     XtNfromVert, w,
		     NULL);
	XtAddCallback (w, XtNcallback, select_palette,
		       (XtPointer) NODE_ENDLOOP);
	/*
	 * functions only in compose mode
	 */
	mode_mark_widget (w, "c");
	paletteWidth = max (paletteWidth, width_of (w));
	paletteHeight += height_of (w) + defaultDistance;

	w = XtGCFSW ("palette.beginLoop", toggleWidgetClass, parent, NorthWest,
		     XtNbitmap,  XtGlueMakePixmap (beginloop_bits,
						   beginloop_width,
						   beginloop_height),
		     XtNtranslations, radioGroupTranslations,
		     XtNradioGroup, w,
		     XtNfromHoriz, left,
		     XtNfromVert, w,
		     NULL);
	XtAddCallback (w, XtNcallback, select_palette,
		       (XtPointer) NODE_LOOP);
	/*
	 * functions only in compose mode
	 */
	mode_mark_widget (w, "c");
	paletteWidth = max (paletteWidth, width_of (w));
	paletteHeight += height_of (w) + defaultDistance;

	w = XtGCFSW ("palette.fanIn", toggleWidgetClass, parent, NorthWest,
		     XtNbitmap, XtGlueMakePixmap (fanin_bits,
						  fanin_width,
						  fanin_height),
		     XtNtranslations, radioGroupTranslations,
		     XtNradioGroup, w,
		     XtNfromHoriz, left,
		     XtNfromVert, w,
		     NULL);
	XtAddCallback (w, XtNcallback, select_palette,
		       (XtPointer) NODE_FANIN);
	/*
	 * functions only in compose mode
	 */
	mode_mark_widget (w, "c");
	paletteWidth = max (paletteWidth, width_of (w));
	paletteHeight += height_of (w) + defaultDistance;

	w = XtGCFSW ("palette.fanOut", toggleWidgetClass, parent, NorthWest,
		     XtNbitmap, XtGlueMakePixmap (fanout_bits,
						  fanout_width,
						  fanout_height),
		     XtNtranslations, radioGroupTranslations,
		     XtNradioGroup, w,
		     XtNfromHoriz, left,
		     XtNfromVert, w,
		     NULL);
	XtAddCallback (w, XtNcallback, select_palette,
		       (XtPointer) NODE_FANOUT);
	/*
	 * functions only in compose mode
	 */
	mode_mark_widget (w, "c");
	paletteWidth = max (paletteWidth, width_of (w));
	paletteHeight += height_of (w) + defaultDistance;

	w = XtGCFSW ("palette.endPipe", toggleWidgetClass, parent, NorthWest,
		     XtNbitmap, XtGlueMakePixmap (endpipe_bits,
						  endpipe_width,
						  endpipe_height),
		     XtNtranslations, radioGroupTranslations,
		     XtNradioGroup, w,
		     XtNfromHoriz, left,
		     XtNfromVert, w,
		     NULL);
	XtAddCallback (w, XtNcallback, select_palette,
		       (XtPointer) NODE_ENDPIPE);
	/*
	 * functions only in compose mode
	 */
	mode_mark_widget (w, "c");
	paletteWidth = max (paletteWidth, width_of (w));
	paletteHeight += height_of (w) + defaultDistance;

	w = XtGCFSW ("palette.beginPipe", toggleWidgetClass, parent, NorthWest,
		     XtNbitmap, XtGlueMakePixmap (beginpipe_bits,
						  beginpipe_width,
						  beginpipe_height),
		     XtNtranslations, radioGroupTranslations,
		     XtNradioGroup, w,
		     XtNfromHoriz, left,
		     XtNfromVert, w,
		     NULL);
	XtAddCallback (w, XtNcallback, select_palette,
		       (XtPointer) NODE_PIPE);
	/*
	 * functions only in compose mode
	 */
	mode_mark_widget (w, "c");
	paletteWidth = max (paletteWidth, width_of (w));
	paletteHeight += height_of (w) + defaultDistance;

	w = XtGCFSW ("palette.endCond", toggleWidgetClass, parent, NorthWest,
		     XtNbitmap, XtGlueMakePixmap (endswitch_bits,
						  endswitch_width,
						  endswitch_height),
		     XtNtranslations, radioGroupTranslations,
		     XtNradioGroup, w,
		     XtNfromHoriz, left,
		     XtNfromVert, w,
		     NULL);
	XtAddCallback (w, XtNcallback, select_palette,
		       (XtPointer) NODE_ENDCOND);
	/*
	 * functions only in compose mode
	 */
	mode_mark_widget (w, "c");
	paletteWidth = max (paletteWidth, width_of (w));
	paletteHeight += height_of (w) + defaultDistance;
	
	w = XtGCFSW ("palette.beginCond", toggleWidgetClass, parent, NorthWest,
		     XtNbitmap, XtGlueMakePixmap (beginswitch_bits,
						  beginswitch_width,
						  beginswitch_height),
		     XtNtranslations, radioGroupTranslations,
		     XtNradioGroup, w,
		     XtNfromHoriz, left,
		     XtNfromVert, w,
		     NULL);
	XtAddCallback (w, XtNcallback, select_palette,
		       (XtPointer) NODE_COND);
	/*
	 * functions only in compose mode
	 */
	mode_mark_widget (w, "c");
	paletteWidth = max (paletteWidth, width_of (w));
	paletteHeight += height_of (w) + defaultDistance;

#ifdef SCISSORS
	w = XtGCFSW ("palette.scissors", toggleWidgetClass, parent, NorthWest,
		     XtNbitmap, XtGlueMakePixmap (scissors_bits,
						  scissors_width,
						  scissors_height),
		     XtNtranslations, radioGroupTranslations,
		     XtNradioGroup, w,
		     XtNfromHoriz, left,
		     XtNfromVert, w,
		     NULL);
	XtAddCallback (w, XtNcallback, select_palette, -1);
	mode_mark_widget (w, "c"); /* only in compose mode */
	paletteWidth = max (paletteWidth, width_of (w));
	paletteHeight += height_of (w) + defaultDistance;
#endif

	w = XtGCFSW ("speed", scrollbarWidgetClass, parent, West,
		     XtNwidth, paletteWidth - 2 * borderWidth,
		     XtNheight, (height - 2 * borderWidth -
				 max (menuHeight, traceHeight) -
				 paletteHeight),
		     XtNlength, (height - 2 * borderWidth -
				 max (menuHeight, traceHeight) -
				 paletteHeight),
		     XtNbackgroundPixmap,
		     XCreatePixmapFromBitmapData (global.display,
						  global.rootWindow,
						  lines_bits,
						  lines_width,
						  lines_height,
						  global.blackPixel,
						  global.whitePixel,
						  global.depth),
		     XtNminimumThumb, 14,
		     XtNorientation, XtorientVertical,
		     XtNfromHoriz, left,
		     XtNfromVert, w,
		     NULL);
	mode_mark_widget (w, "t"); /* only in trace mode */
	XawScrollbarSetThumb (w, 0.5, -1.0);
	XtAddCallback (w, XtNjumpProc, trace_speed_jump,
		       (XtPointer) -1);
	/* XXX need to add scrollproc also */
    }

    {
	Widget horizScroll;
	Widget vertScroll;
	Widget w;
	int canvasWidth = 0;
	int canvasHeight = 0;
	int canvasHorizDistance;
	int canvasVertDistance;

	/*
	 * create scrollbars - vertical with a dummy height and
	 * horiz with a dummy width - so we an get the thickness
	 * of each to compute the size of the canvas.
	 */

	w = XtGCFSW ("horizScroll", scrollbarWidgetClass, parent, South,
		     XtNorientation, XtorientHorizontal,
		     XtNwidth, 10,
		     XtNheight, 14,
		     NULL);
	XawScrollbarSetThumb (w, 0.0, 1.0); /* XXX until we do scaling */
	horizScroll = w;

	w = XtGCFSW ("vertScroll", scrollbarWidgetClass, parent, East,
		     XtNorientation, XtorientVertical,
		     XtNheight, 10,
		     XtNwidth, 14,
		     NULL);
	XawScrollbarSetThumb (w, 0.0, 1.0); /* XXX until we do scaling */
	vertScroll = w;
			   
	canvasWidth = (width -			/* width of panel */
		       2 * borderWidth - 	/* borders of canvas itself */
		       paletteWidth -		/* width of palette */
		       16 -			/* width of vertScroll */
		       3 * defaultDistance);
	canvasHeight = (height -		/* height of panel */
			2 * borderWidth -	/* borders of canvas */
			16 -			/* height of horizScroll  */
			max (menuHeight, traceHeight));

	canvasHorizDistance = paletteWidth + 2 * defaultDistance;
	canvasVertDistance = (max (menuHeight, traceHeight) +
			      2 * defaultDistance);

	/*
	 * graph canvas.  We have to fudge on the width.  If the luser
	 * changes borderWidth or spacing between windows, this won't
	 * look right.  Also, all of the bitmaps in the palette must
	 * be the same width.
	 */

	graphCanvas = XtGCFSW ("canvas", canvasWidgetClass, parent, Center,
			       XtNwidth, canvasWidth,
			       XtNheight, canvasHeight,
			       XtNresizable, False,
			       XtNfromHoriz, NULL,
			       XtNfromVert, graphTitle,
			       XtNhorizDistance, canvasHorizDistance,
			       XtNvertDistance, canvasVertDistance,
			       NULL);

	/*
	 * now set up size and constraint information for the scrollbars
	 */

	XtVaSetValues (horizScroll,
		       XtNwidth, canvasWidth,
		       XtNfromHoriz, NULL,
		       XtNhorizDistance, canvasHorizDistance,
		       XtNfromVert, graphCanvas,
		       XtNvertDistance, 0,
		       NULL);

	XtVaSetValues (vertScroll,
		       XtNheight, canvasHeight,
		       XtNfromHoriz, graphCanvas,
		       XtNhorizDistance, 0,
		       XtNfromVert, graphTitle,
		       XtNvertDistance, canvasVertDistance,
		       NULL);

    }

    draw_init (graphCanvas);
    edit_init (graphCanvas);
    trace_init (graphCanvas);
    XtAddEventHandler(graphCanvas,
		      StructureNotifyMask|ExposureMask, False,
		      graph_event, (XtPointer)0);
}
