/*
 * routines for drawing HeNCE graphs
 *
 * $Id: graph_draw.c,v 1.1 1994/02/17 20:22:43 moore Exp $
 *
 * $Log: graph_draw.c,v $
 * Revision 1.1  1994/02/17  20:22:43  moore
 * Initial revision
 *
 */

#include <stdio.h>
#include <math.h>

#include <X11/IntrinsicP.h>
#include <X11/CoreP.h>

#include "global.h"
#include "rb.h"
#include "graph.h"
#include "graph_draw.h"
#include "xtglue.h"
#include "CanvasP.h"

#include "bitmaps/node.xbm"
#include "bitmaps/ready.xbm"
#include "bitmaps/run1.xbm"
#include "bitmaps/run2.xbm"
#include "bitmaps/done.xbm"
#include "bitmaps/dead.xbm"
#include "bitmaps/warn.xbm"
#include "bitmaps/error.xbm"

/*
 * XXX these are used both by graph_panel and graph_draw.
 * Should probably be read in in one place only.
 */

#include "bitmaps/beginloop.xbm"
#include "bitmaps/endloop.xbm"
#include "bitmaps/beginswitch.xbm"
#include "bitmaps/endswitch.xbm"
#include "bitmaps/beginpipe.xbm"
#include "bitmaps/endpipe.xbm"
#include "bitmaps/fanout.xbm"
#include "bitmaps/fanin.xbm"

/*
 * Bondo
 */
#include <Malloc.h>

#ifdef max
#undef max
#endif
#define max(a,b) ((a) > (b) ? (a) : (b))
#ifdef min
#undef min
#endif
#define min(a,b) ((a) < (b) ? (a) : (b))

/* these define the arrowhead size and proximity to node */

#define	ARROWX	NODERADIUS/2
#define	ARROWY	(NODERADIUS/4)
#define	ARROWPROX ((NODERADIUS*5)/4)

#define MAXPICKNODEDIST (NODERADIUS*NODERADIUS*25)
#define MAXPICKARCDIST (NODERADIUS*NODERADIUS*25)

static struct {
    GC normal;
    GC inverse;
    GC clear;
    GC xor;

    GC done;
    GC dead;
    GC error;
    GC notReady;
    GC ready;
    GC running;
    GC warning;
} gcs;

static GC state_gcs[ST_NUM_STATES];

Pix node_pix[NODE_NUM_TYPES];
Pix normal_node_pix[ST_NUM_STATES];
int text_height;
struct pix pix;

static GC
make_gc (cw, function, fg, bg)
CanvasWidget cw;
int function;
Pixel fg, bg;
{
    XGCValues v;

    v.function = function;
    v.foreground = fg;
    v.background = bg;
    v.font = cw->canvas.fontinfo->fid;
    return XCreateGC (global.display, global.rootWindow,
		      GCFunction|GCFont|GCForeground|GCBackground, &v);
}

static Pix
make_pix (bits, width, height)
char *bits;
int width, height;
{
    Pix p = (Pix) MALLOC (sizeof (*p));

    p->pm = XtGlueMakePixmap (bits, width, height);
    p->width = width;
    p->height = height;
    return p;
}

static void
draw_pix (win, pix, gc, x, y)
Window win;
Pix pix;
GC gc;
int x, y;
{
    XCopyPlane (global.display,	/* display */
		pix->pm,	/* src drawable */
		win,		/* dst drawable */
		gc,		/* gc */
		0,		/* src x */
		0,		/* src y */
		pix->width,	/* width */
		pix->height,	/* height */
		x - pix->width / 2, /* dst x */
		y - pix->height / 2, /* dst y */
		1);
}

static void
draw_text (win, gc, x, y, text)
Window win;
GC gc;
int x, y;
char *text;
{
    XDrawImageString (global.display, win, gc, x, y, text, strlen (text));
}

static int
string_width (w, s)
Widget w;
char *s;
{
    CanvasWidget cw = (CanvasWidget) w;
    return XTextWidth (cw->canvas.fontinfo, s, strlen (s));
}

/*	draw_init ()
 *
 *	Must be called once at startup time
 *	Creates pixmaps and GCs used for drawing graphs.
 */

void
draw_init (w)
Widget w;
{
    CanvasWidget cw = (CanvasWidget) w;
    Pixel fgPixel = cw->canvas.foreground;
    Pixel bgPixel = cw->core.background_pixel;

    /* init GCs */
    gcs.normal = make_gc (cw, GXcopy, fgPixel, bgPixel);
    gcs.inverse = make_gc (cw, GXcopy, bgPixel, fgPixel);
    gcs.clear = make_gc (cw, GXcopy, bgPixel, bgPixel);
    /*
     * XXX this won't work for all displays.  Need to specifically
     * allocate color map cells so that that XORing values will work.
     */
    gcs.xor = make_gc (cw, GXxor, fgPixel ^ bgPixel, 0);
    gcs.dead = make_gc (cw, GXcopy, defaults.deadColor, bgPixel);
    gcs.done = make_gc (cw, GXcopy, defaults.doneColor, bgPixel);
    gcs.error = make_gc (cw, GXcopy, defaults.errorColor, bgPixel);
    gcs.notReady = make_gc (cw, GXcopy, defaults.notReadyColor, bgPixel);
    gcs.ready = make_gc (cw, GXcopy, defaults.readyColor, bgPixel);
    gcs.running = make_gc (cw, GXcopy, defaults.runningColor, bgPixel);
    gcs.warning = make_gc (cw, GXcopy, defaults.warningColor, bgPixel);

    state_gcs[ST_NOT_BEGUN] = gcs.notReady;
    state_gcs[ST_READY] = gcs.ready;
    state_gcs[ST_RUNNING] = gcs.running;
    state_gcs[ST_DONE] = gcs.done;
    state_gcs[ST_DEAD] = gcs.dead;
    state_gcs[ST_ERROR] = gcs.error;
    state_gcs[ST_WARNING] = gcs.warning;

    pix.notReady = make_pix (node_bits, node_width, node_height);
    pix.ready = make_pix (ready_bits, ready_width, ready_height);
    pix.running1 = make_pix (run1_bits, run1_width, run1_height);
    pix.running2 = make_pix (run2_bits, run2_width, run2_height);
    pix.done = make_pix (done_bits, done_width, done_height);
    pix.dead = make_pix (dead_bits, dead_width, dead_height);
    pix.warning = make_pix (warn_bits, warn_width, warn_height);
    pix.error = make_pix (error_bits, error_width, error_height);

    pix.beginLoop = make_pix (beginloop_bits, beginloop_width,
			      beginloop_height);
    pix.endLoop = make_pix (endloop_bits, endloop_width, endloop_height);
    pix.beginSwitch = make_pix (beginswitch_bits, beginswitch_width,
				beginswitch_height);
    pix.endSwitch = make_pix (endswitch_bits, endswitch_width,
			      endswitch_height);
    pix.beginPipe = make_pix (beginpipe_bits, beginpipe_width,
			      beginpipe_height);
    pix.endPipe = make_pix (endpipe_bits, endpipe_width, endpipe_height);
    pix.fanOut = make_pix (fanout_bits, fanout_width, fanout_height);
    pix.fanIn = make_pix (fanin_bits, fanin_width, fanin_height);

    node_pix[NODE_NORMAL] = pix.notReady;
    node_pix[NODE_COND] = pix.beginSwitch;
    node_pix[NODE_ENDCOND] = pix.endSwitch;
    node_pix[NODE_LOOP] = pix.beginLoop;
    node_pix[NODE_ENDLOOP] = pix.endLoop;
    node_pix[NODE_FANOUT] = pix.fanOut;
    node_pix[NODE_FANIN] = pix.fanIn;
    node_pix[NODE_PIPE] = pix.beginPipe;
    node_pix[NODE_ENDPIPE] = pix.endPipe;

    normal_node_pix[ST_NOT_BEGUN] = pix.notReady;
    normal_node_pix[ST_READY] = pix.ready;
    normal_node_pix[ST_RUNNING] = pix.running1;
    normal_node_pix[ST_DONE] = pix.done;
    normal_node_pix[ST_DEAD] = pix.dead;
    normal_node_pix[ST_ERROR] = pix.error;
    normal_node_pix[ST_WARNING] = pix.warning;

    text_height = cw->canvas.fontinfo->ascent + cw->canvas.fontinfo->descent;
}


/*
 *	draw_node ()
 *
 *	Draw a node icon in a window.  abs(how) is
 *		0 to erase
 *		1 to draw
 *		2 to draw via xor
 *
 * If how is negative, it means to print trace info.
 */

void
draw_node (w, node, how)
Widget w;
Node node;
int how;
{
    int x = node->xy.x;
    int y = node->xy.y;
    int t = node->node_type;
    GC tgc;
    GC bgc;
    Pix npm;
    int tracing = 0;
    Window win = XtWindow (w);
    static int click = 0;
    char crud[1024];

    if (how < 0) {
	how = -how;
	tracing = 1;
    }

    if (how != 2) {
	if (how) {
	    tgc = gcs.normal;
	    if (global.inColor)
		bgc = state_gcs[node->state];	/* XXX check bounds! */
	    else
		bgc = gcs.normal;	/* XXX todo b&w */
	    
	}
	else {
	    tgc = gcs.clear;
	    bgc = gcs.clear;
	}
	
	if (t != NODE_NORMAL)
	    npm = node_pix[t];
	else {
	    switch (node->state) {
	    case ST_READY:
	    case ST_DONE:
	    case ST_DEAD:
	    case ST_WARNING:
	    case ST_ERROR:
		npm = normal_node_pix[node->state];
		break;
	    case ST_RUNNING:
		if (click)
		    npm = pix.running1;
		else
		    npm = pix.running2;
		click = !click;
		break;
	    case ST_NOT_BEGUN:
	    default:
		/*
		 * note: all states < 0 are considered not ready
		 */
		bgc = gcs.notReady;
		npm = pix.notReady;
		break;
	    }
	}
	draw_pix (win, npm, bgc, x, y);

	if (node->nk.id >= 0) {
	    if (tracing) {
		if (node->nk.inst)
		    (void) sprintf (crud, "%s%s%s%d/%d",
				    node->sub_name ? "(" : "",
				    node->sub_name ? node->sub_name : "",
				    node->sub_name ? ") " : "",
				    node->nk.id, node->nk.inst);
		else
		    (void) sprintf (crud, "%s%s%s%d",
				    node->sub_name ? "(" : "",
				    node->sub_name ? node->sub_name : "",
				    node->sub_name ? ") " : "",
				    node->nk.id);
		
		draw_text (win, tgc,
			   x - NODERADIUS - 4 - string_width (w, crud),
			   y + text_height / 2,
			   crud);

		if (node->node_type == NODE_NORMAL) {
		    sprintf (crud, "%dr,%di  ", node->numRunning,
			     node->numIdle);
		    draw_text (win, tgc,
			       x + NODERADIUS + 4,
			       y + text_height / 2,
			       crud);
		}
	    }
	    else {
		if (node->nk.inst)
		    (void) sprintf (crud, "%d/%d", node->nk.id, node->nk.inst);
		else
		    (void) sprintf (crud, "%d", node->nk.id);
		
		draw_text (win, tgc,
			   x - NODERADIUS - 4 - string_width (w, crud),
			   y +  text_height / 2,
			   crud);

		if (node->node_type == NODE_NORMAL) {
		    if (node->sub_name) {
			draw_text (win, tgc,
				   x + NODERADIUS + 4,
				   y + text_height / 2,
				   node->sub_name);
		    }
		}
	    }
	}
    }
    else {	/* rubber drawing */
	draw_pix (win, node_pix[t], gcs.xor, x, y);
    }
}

void
draw_line (w, gc, x1, y1, x2, y2)
Widget w;
GC gc;
int x1, y1, x2, y2;
{
    XDrawLine (global.display, XtWindow (w), gc, x1, y1, x2, y2);
}

/*
 *	draw_arc ()
 *
 *	Draw arc between nodes in window.  How is
 *		0 to erase
 *		1 to draw
 *		2 to draw via xor
 */

void
draw_arc (w, n1, n2, how)
Widget w;
Node n1, n2;
int how;
{
    int delta_x, delta_y;
    double v, vx, vy;
    int x1, y1, x2, y2;
    GC gc;

    switch (how) {
    case 0:
	gc = gcs.clear;
	break;
    case 1:
	gc = gcs.normal;
	break;
    case 2:
	gc = gcs.xor;
	break;
    }
		
    delta_x = n2->xy.x - n1->xy.x;
    delta_y = n2->xy.y - n1->xy.y;
    if (delta_x == 0 && delta_y == 0)		/* sanity check */
	return;
    v = sqrt((double)(delta_x * delta_x + delta_y * delta_y));
    vx = (double) delta_x / v;
    vy = (double) delta_y / v;
    x1 = n1->xy.x + ARROWPROX * vx;
    y1 = n1->xy.y + ARROWPROX * vy;
    x2 = n2->xy.x - ARROWPROX * vx;
    y2 = n2->xy.y - ARROWPROX * vy;
    draw_line (w, gc, x1, y1, x2, y2);
    
    x1 = x2 - ARROWX * vx + ARROWY * vy;
    y1 = y2 - ARROWY * vx - ARROWX * vy;
    draw_line (w, gc, x1, y1, x2, y2);
    
    x1 = x2 - ARROWX * vx - ARROWY * vy;
    y1 = y2 + ARROWY * vx - ARROWX * vy;
    draw_line (w, gc, x1, y1, x2, y2);
}

/*
 *	draw_graph
 *
 *	Redraw all nodes and arcs in a graph
 */

void
draw_graph (w, g, tracemode)
Widget w;
Graph g;
int tracemode;
{
    rbTree ntree = g->nlist;
    rbTree atree;
    rbNode tn, atn;
    Node n1, n2;

    for (tn = rb_First (ntree); !rb_Done (tn, ntree) ; tn = rb_Next (tn)) {
	draw_node (w, (Node) rb_Value (tn), tracemode ? -1 : 1);
    }
    for (tn = rb_First (ntree); !rb_Done (tn, ntree); tn = rb_Next (tn)) {
	n1 = (Node) rb_Value (tn);
	atree = n1->children;
	for (atn = rb_First (atree); !rb_Done (atn, atree) ;
	     atn = rb_Next (atn)) {
	    n2 = (Node) rb_Value (atn);
	    draw_arc (w, n1, n2, 1);
	}
    }
#if 0
    /* XXX for debugging pair-finding */
    for (tn = rb_First (ntree); !rb_Done (tn, ntree); tn = rb_Next (tn)) {
	n1 = (Node) rb_Value (tn);
	if (n2 = n1->pair) {
	    if (n1->xy.set && n2->xy.set)
		draw_line (w, gcs.normal,
			   n1->xy.x + 10, n1->xy.y, n2->xy.x, n2->xy.y);
	}
    }
#endif
}

void
redraw_graph (w, g)
Widget w;
Graph g;
{
    XClearWindow (global.display, XtWindow (w));
    draw_graph (w, g, 0);
}


/*
 * find_nearest_node ()
 *
 * Return pointer to node nearest a point.
 *
 * XXX should this be in graph.c?
 */

Node
find_nearest_node (grf, x, y)
Graph grf;
int x, y;
{
    rbTree ntree = grf->nlist;
    rbNode tn;
    Node n, nn = 0;
    int nd = -1, dx, dy, dd;

    for (tn = rb_First (ntree); !rb_Done (tn, ntree); tn = rb_Next (tn)) {
	n = (Node) rb_Value (tn);
	dx = n->xy.x - x;
	dy = n->xy.y - y;
	dd = dx * dx + dy * dy;
	if (dd < MAXPICKNODEDIST && (nd < 0 || dd < nd)) {
	    nd = dd;
	    nn = n;
	}
    }
    return nn;
}

/*
 * find_nearest_arc ()
 *
 * Return the nodes at the ends of the nearest arc to a point,
 * endpoint which is closest first.
 */

void
find_nearest_arc (grf, x, y, n1p, n2p)
Graph grf;
int x, y;
Node *n1p, *n2p;
{
    rbTree ntree = grf->nlist;
    rbTree atree;
    rbNode tn, atn;
    Node n1, n2;		/* gp */
    int ax, ay;			/* delta */
    int d1, d2;			/* dist to endpoints */
    double v, vx, vy;		/* gp */
    int x1, y1, x2, y2;		/* arc bbx */
    int md = -1;		/* min dist found any arc */
    int mda;			/* min dist this arc */
    Node nna1 = 0, nna2 = 0;	/* nodes of nearest arc */

    for (tn = rb_First (ntree); !rb_Done (tn, ntree); tn = rb_Next (tn)) {
	n1 = (Node) rb_Value (tn);
	atree = n1->children;
	for (atn = rb_First (atree); !rb_Done (atn, atree);
	     atn = rb_Next (atn)) {
	    n2 = (Node) rb_Value(atn);

	    /* compute endpoints of arc */

	    ax = n2->xy.x - n1->xy.x;
	    ay = n2->xy.y - n1->xy.y;
	    if (!ax && !ay)		/* sanity check */
		continue;
	    v = sqrt((double)(ax * ax + ay * ay));
	    vx = ax / v;
	    vy = ay / v;
	    x1 = n1->xy.x + ARROWPROX * vx;
	    y1 = n1->xy.y + ARROWPROX * vy;
	    x2 = n2->xy.x - ARROWPROX * vx;
	    y2 = n2->xy.y - ARROWPROX * vy;

	    /* get dist to epts */
	    ax = x - x1;
	    ay = y - y1;
	    d1 = ax * ax + ay * ay;
	    ax = x - x2;
	    ay = y - y2;
	    d2 = ax * ax + ay * ay;

	    /* get nearest pt on line */

	    ax = x2 - x1;
	    ay = y2 - y1;
	    v = (double)(ax * (x - x1) + ay * (y - y1)) / (ax * ax + ay * ay);
	    ax = x1 + ax * v;
	    ay = y1 + ay * v;

	    if (ax < min(x1, x2) || ax > max(x1, x2)
		|| ay < min(y1, y2) || ay > max(y1, y2)) {	/* not in seg */
		mda = min(d1, d2);
	    } else {	/* in ray segment */
		ax -= x;
		ay -= y;
		mda = ax * ax + ay * ay;
	    }

	    /* compare distance to old closest arc */

	    if (mda < MAXPICKARCDIST && (md == -1 || mda < md)) {
		md = mda;
		if (d1 < d2) {
		    nna1 = n1;
		    nna2 = n2;
		}
		else {
		    nna1 = n2;
		    nna2 = n1;
		}
	    }
	}
    }
    *n1p = nna1;
    *n2p = nna2;
/*
  fprintf(stderr, "(%d-%d)\n", nna1->nk.id, nna2->nk.id);
*/
}

void
draw_time (w, t, seq)
Widget w;
struct event_time *t;
int seq;
{
    char buf[100];
    unsigned int days;
    unsigned int hours;
    unsigned int mins;
    unsigned int secs;
    unsigned int foo = t->seconds;

    secs = foo % 60;
    foo /= 60;
    mins = foo % 60;
    foo /= 60;
    hours = foo % 24;
    foo /= 24;
    days = foo;

#if 1
    if (t->msec > 1000)
	fprintf (stderr, "illegal time %d %d\n",
		 t->seconds, t->msec);
#endif
	
    sprintf (buf, "%2d %2d:%02d:%02d.%03d",
	     days, hours, mins, secs, t->msec);
    if (days == 0)
	buf[0] = buf[1] = ' ';
    if (hours == 0)
	buf[3] = buf[4] = buf[5] = ' ';
    if (seq > 0)
	sprintf (buf + 15, "  [%d]         ", seq);
    else
	sprintf (buf + 15, "                       ");
    /*
     * XXX probably better to draw this at top right than top left.
     */
    draw_text (XtWindow(w), gcs.normal, 0, text_height, buf);
}

#ifdef NOT_USED_YET

/*
 *	xgr_GetSubNames()
 *
 *	Return a symbol table containing a list of the function names
 *	used in the graph, exactly once each.
 */

rbTree
xgr_GetSubNames(g)
Graph g;
{
    rbTree ntree = g->nlist;
    rbTree st = st_New();
    rbNode tn;
    Node n;

    for (tn = rb_First (ntree); !rb_Done(tn,ntree); tn = rb_Next (tn)) {
	n = (Node) rb_Value (tn);
	if (n->node_type == NODE_NORMAL && n->sub_name && *n->sub_name)
	    st_Insert (st, n->sub_name);
    }
    return st;
}

/*
 *	xgr_drawLegend()
 */

void
xgr_DrawLegend (w)
Window w;
{
    static Node n = 0;
    int tx = NODERADIUS * 6;

    if (!n)
	n = gr_NewNode(-1);

    n->xy.x = NODERADIUS*4;
    n->xy.y = NODERADIUS*2;
    n->node_type = NODE_NORMAL;
    n->state = ST_NOT_BEGUN;
    xgr_DrawNode(w, n, 1);
    xdraw_Text (w, state_gcs[GCNORMAL], tx, n->xy.y, "Not Ready");

    n->xy.y += NODERADIUS*3;
    n->state = ST_READY;
    xgr_DrawNode(w, n, 1);
    xdraw_Text (w, state_gcs[GCNORMAL], tx, n->xy.y, "Node Started");

    n->xy.y += NODERADIUS*3;
    n->state = ST_RUNNING;
    xgr_DrawNode(w, n, 1);
    xdraw_Text (w, state_gcs[GCNORMAL], tx, n->xy.y, "Function Executing");

    n->xy.y += NODERADIUS*3;
    n->state = ST_DONE;
    xgr_DrawNode(w, n, 1);
    xdraw_Text (w, state_gcs[GCNORMAL], tx, n->xy.y, "Function Completed");

#if 0
    XDrawLine(xDisp, w, state_gcs[GCNORMAL],
		0, n->xy.y + (int)(NODERADIUS * 1.5), 400, 
		n->xy.y + (int)(NODERADIUS * 1.5));
#endif

    n->xy.y += NODERADIUS*3;
    n->state = ST_DEAD;
    xgr_DrawNode(w, n, 1);
    xdraw_Text (w, state_gcs[GCNORMAL], tx, n->xy.y, "Node Exited");

#if 1
    XDrawLine(xDisp, w, state_gcs[GCNORMAL],
	      0, n->xy.y + (int)(NODERADIUS * 1.5), 400, 
	      n->xy.y + (int)(NODERADIUS * 1.5));
#endif

    n->xy.y += NODERADIUS*3;
    n->state = ST_WARNING;
    xgr_DrawNode(w, n, 1);
    xdraw_Text (w, state_gcs[GCNORMAL], tx, n->xy.y, "Warning in Graph Program");
    
    n->xy.y += NODERADIUS*3;
    n->state = ST_ERROR;
    xgr_DrawNode(w, n, 1);
    xdraw_Text (w, state_gcs[GCNORMAL], tx, n->xy.y, "Error in Graph Program");
}

/*
 *  xgr_drawComposeHelp()
 */

void
xgr_drawComposeHelp (w)
Window w;
{
    XCopyPlane(xDisp, compose_help_pm, w, state_gcs[GCNORMAL],
	       0,0, compose_help_width, compose_help_height, 0,0, 1);
}

#endif /* NOT_USED_YET */
