/*
 * architecture-specific HeNCE function calling
 * (mostly to interface C with FORTRAN)
 *
 * $Id: arches.c,v 1.2 1994/03/07 20:57:49 moore Exp $
 *
 * $Log: arches.c,v $
 * Revision 1.2  1994/03/07  20:57:49  moore
 * fix translate_subr_name to return a valid name for FORTRAN subprogs.
 *
 * Revision 1.1  1994/02/17  20:17:54  moore
 * Initial revision
 *
 */

#include <stdio.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#include <Malloc.h>
#include <String.h>

#include "arches.h"
#include "dot_htool.h"
#include "exp.h"
#include "global.h"
#include "rb.h"

struct arch_specific {
    char *flags;		/* arch-specific flags */
    char *cc;			/* c compile command, printf style */
    char *fc;			/* fortran compile command, printf style */
};

/*
 * encoding of CC and FC strings:
 *
 * %{something}	include "something" only if debugging
 * %[something]	include "something" only if NOT debugging
 * %D		-D preprocessor defines go here (-DIMA_XXXX)
 * %I		-I includes go here
 * %L		libraries go here (-Ldirectory and -llibname)
 * %o		name of object file (foo)
 * %s		names of source file (foo.c or foo.f)
 * %w		name of wrapper (cw_foo.c or fw_foo.c)
 * %W		name of compiled wrapper (cw_foo.o or fw_foo.o)
 *
 * encoding of flags:
 *
 * _		append an underscore to function name to call from FORTRAN
 * U		uppercase function names to call from FORTRAN
 * l		add extra args at the end of arg lists to give the length
 *		of FORTRAN CHARACTER arrays
 * C		special CRAY hacks for fortran strings (unimplemented)
 * T		special TITAN hacks for fortran strings (unimplemented)
 */

#define DEFAULT_FLAGS	"_l"
#define DEFAULT_CC	"cc -o %o %{-g}%[-O] %D %I %s %w %L -lm"
#define DEFAULT_FC	"cc %{-g}%[-O] %D %I -c %w && f77 -o %o %{-g}%[-O] fmain.f %W %s %L -lm"

static struct arch_specific *
find_arch (name)
char *name;
{
    char buf[1024];
    static struct arch_specific foo;

    if (name == NULL)
	return NULL;

    sprintf (buf, "%s_BUILD_FLAGS", name);
    foo.flags = dot_htool_get_value (buf);
    sprintf (buf, "%s_CC", name);
    foo.cc = dot_htool_get_value (buf);
    sprintf (buf, "%s_FC", name);
    foo.fc = dot_htool_get_value (buf);

    if (foo.flags == NULL)
	foo.flags = DEFAULT_FLAGS;
    if (foo.cc == NULL)
	foo.cc = DEFAULT_CC;
    if (foo.fc == NULL)
	foo.fc = DEFAULT_FC;

    return &foo;
}


char *
translate_subr_name (name, language, arch)
char *name;
int language;
char *arch;
{
    struct arch_specific *foo = find_arch (arch);
    static char buf[256];
    char *src, *dst;
    int fold_to_upper_case = strchr (foo->flags, 'U') != NULL;
    int append_underscore = strchr (foo->flags, '_') != NULL;

    if (language == LANG_FORTRAN) {
	src = name;
	dst = buf;
	while (*src)
	    if (fold_to_upper_case && *src >= 'a' && *src <= 'z')
		*dst++ = *src++ - 'a' + 'A';
	    else
		*dst++ = *src++;
	if (append_underscore)
	    *dst++ = '_';
	*dst++ = '\0';
	return buf;
    }
    else
	return name;
}



/*
 * given a parameter in a local C variable (named by 'prefix' and 'i'),
 * do whatever has to be done to pass its value as a formal parameter
 * to a subroutine written in 'language'.
 *
 * This routine is called *twice* for each parameter, once with pass == 1,
 * the next time with pass == 2.  This is for generating extra args for
 * strings and function paramaters, on those machines that need it.
 *
 * return NULL if we don't need an extra parameter
 */

char *
translate_actual_param (prefix, i, type, indirs, language, arch, pass)
char *prefix;
int i;
int type;
int indirs;
int language;
char *arch;
int pass;
{
    static char buf[128];
    struct arch_specific *foo = find_arch (arch);

    /*
     * generate default args for convenience
     */
    if (pass == 1)
	sprintf (buf, "%s%d", prefix, i);
    else
	sprintf (buf, "strlen (%s%d)", prefix, i);


    switch (language) {
    case LANG_C:
	/*
	 * When calling C routines from HeNCE...
	 * + scalars are passed by value
	 * + arrays are passed by reference 
	 * both of these happen without modifying the local reference.
	 *
	 * (XXX THIS IS BROKEN, because an IN/OUT parameter MUST be passed
	 * by reference to a C subr so that it can be modified.  Probably
	 * best to just pass everything by reference.)
	 */
	return pass == 1 ? buf : NULL;

    case LANG_FORTRAN:
	/*
	 * + strings require special code; either an extra length
	 *   parameter on pass 2, or passing a special structure.
	 */
	if (indirs == 1 && type == TYPE_CHAR) {
	    if (strchr (foo->flags, 'l'))
		return buf;
	    else if (strchr (foo->flags, 'C')) {
		/* XXX CRAY strings not implemented yet */
		return "???";
	    }
	    else if (strchr (foo->flags, 'T')) {
		/* XXX TITAN strings not implemented yet */
		return "???";
	    }
	}
	/*
	 * + arrays are passed by reference (no prefix is required)
	 */
	else if (indirs > 0) {
	    return pass == 1 ? buf : NULL;
	}
	/*	
	 * + scalars are passed by reference (which requires an '&' prefix)
	 */
	else {
	    if (pass == 1) {
		sprintf (buf, "&%s%d", prefix, i);
		return buf;
	    }
	    else
		return NULL;
	}

    default:
	msg_Format ("htool bug: unknown language %d\n", language);
    }
}


char *
translate_type_name (type, indirs, language, arch)
int type;
int indirs;
int language;
char *arch;
{
    char *basetype;
    static char buf[100];
    char *ptr;
    struct arch_specific *foo = find_arch (arch);

    /*
     * special hacks for FORTRAN strings.
     */
    if (language == LANG_FORTRAN && type == TYPE_CHAR && indirs == 1) {
	if (strchr (foo->flags, 'C')) {
	    /* this is declared in <fortran.h> on a CRAY */
	    return "_fcd ";
	}
	else if (strchr (foo->flags, 'T')) {
	    /* on a TITN, we define this structure in the prologue */
	    return "FSD ";
	}
	else {
	    return "char *";
	}
    }

    /*
     * default code for other data types
     */
    switch (type) {
    case TYPE_VOID:
	basetype = "void";
	break;
    case TYPE_INT:
	basetype = "int";
	break;
    case TYPE_FLOAT:
	basetype = "float";
	break;
    case TYPE_DOUBLE:
	basetype = "double";
	break;
    case TYPE_CHAR:
	basetype = "char";
	break;
    default:
	basetype = "bogus_type";
	break;
    }
    ptr = buf;
    while (*basetype)
	*ptr++ = *basetype++;
    *ptr++ = ' ';
    while (indirs-- > 0)
	*ptr++ = '*';
    *ptr = '\0';
    return buf;
}

/*
 * XXX collapse all hence slave libs into one file
 */

#define LIBS "-lslave3 -lhence3 -lrb -ldl -lalloc -lpvm3"

char *
arch_get_compile_cmd (arch, language, output, source, wrapper, debug)
char *arch;
int language;
char *output;
char *source;
char *wrapper;
int debug;
{
    static char buf[1024];
    static char wrapper_obj[1024];
    char *src, *dst, *src2;
    struct arch_specific *foo = find_arch (arch);
    
    strcpy (wrapper_obj, wrapper);
    dst = strrchr (wrapper_obj, '.');
    if (dst) {
	dst[1] = 'o';
	dst[2] = '\0';
    }

    src = (language == LANG_FORTRAN) ? foo->fc : foo->cc;
    dst = buf;

    while (*src) {
	if (*src == '%') {
	    ++src;
	    switch (*src) {
	    case '[':
		++src;		/* skip over '[' */
		while (*src && *src != ']') {
		    if (!debug)
			*dst++ = *src;
		    ++src;
		}
		break;
	    case '{':
		++src;		/* skip over '{' */
		while (*src && *src != '}') {
		    if (debug)
			*dst++ = *src;
		    ++src;
		}
		break;
	    case 'D':
		src2 = "-DIMA_$(PVM_ARCH)";
		while (*src2)
		    *dst++ = *src2++;
		break;
	    case 'I':
		src2 = "-I$(PVM_ROOT)/include";
		while (*src2)
		    *dst++ = *src2++;
		break;
	    case 'L':
		src2 = "-L$(PVM_ROOT)/lib/$(PVM_ARCH)";
		while (*src2)
		    *dst++ = *src2++;
		*dst++ = ' ';
		src2 = LIBS;
		while (*src2)
		    *dst++ = *src2++;
		break;
	    case 'o':
		src2 = output;
		while (*src2)
		    *dst++ = *src2++;
		break;
	    case 's':
		src2 = source;
		while (*src2)
		    *dst++ = *src2++;
		break;
	    case 'w':
		src2 = wrapper;
		while (*src2)
		    *dst++ = *src2++;
		break;
	    case 'W':
		src2 = wrapper_obj;
		while (*src2)
		    *dst++ = *src2++;
		break;
	    }
	    if (*src)
		++src;
	}
	else
	    *dst++ = *src++;
    }
    *dst++ = '\0';
    return buf;
}
