/*
 *   Matrix.c - Matrix Widget
 *
 *   Jul 1991 Robert Manchek  manchek@CS.UTK.EDU.
 *
 * $Id: Matrix.c,v 1.1 1994/02/17 20:17:18 moore Exp $
 *
 * $Log: Matrix.c,v $
 * Revision 1.1  1994/02/17  20:17:18  moore
 * Initial revision
 *
 */


#include <stdio.h>

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>

#include <X11/Xmu/Misc.h>

#include <X11/Xaw/XawInit.h>
#include "MatrixP.h"

#ifdef DEMO
#include "global.h"
#endif

#include <Malloc.h>
#include <String.h>

#define BITESIZE    1	/* danger will robinson probably don't wanna change */

/***************************
*  Class record constants  *
*			   *
***************************/

/* Private Data */

#ifdef DEMO
static char defTran[] = "\
    <BtnDown>: recolor() \n\
    <Motion>: highlight() \n\
    <Leave>: highlight() \n\
    <Key>Tab: highlight(r) \n\
    <Key>Return: highlight(d) \n\
    <Key>BackSpace: edit(del)notify() \n\
    <Key>Delete: edit(del)notify() \n\
    <Key>: edit(add)notify() ";
#else
static char defTran[] =
    "<Motion>: highlight() \n\
    <Leave>: highlight() \n\
    <Key>Tab: highlight(r) \n\
    <Key>Return: highlight(d) \n\
    <Key>BackSpace: edit(del)notify() \n\
    <Key>Delete: edit(del)notify() \n\
    <Key>: edit(add)notify() ";
#endif

#define OFST(field) XtOffset(MatrixWidget, field)
static XtResource resources[] = {
    {HgwNnumeric, HgwCnumeric, XtRBoolean, sizeof(Boolean),
	OFST(matrix.numeric), XtRString, "False"},
    {HgwNeditable, HgwCeditable, XtRBoolean, sizeof(Boolean),
	OFST(matrix.editable), XtRString, "True"},
    {HgwNbase, HgwCbase, XtRPointer, sizeof(XtPointer),
	OFST(matrix.base), XtRImmediate, 0},
    {HgwNrows, HgwCrows, XtRInt, sizeof(int),
	OFST(matrix.rows), XtRImmediate, 0},
    {HgwNcols, HgwCcols, XtRInt, sizeof(int),
	OFST(matrix.cols), XtRImmediate, 0},
    {HgwNrowSpacing, HgwCrowSpacing, XtRInt, sizeof(int),
	OFST(matrix.row_spc), XtRImmediate, 0},
    {HgwNcolSpacing, HgwCcolSpacing, XtRInt, sizeof(int),
	OFST(matrix.col_spc), XtRImmediate, 0},
    {HgwNrowOffset, HgwCrowOffset, XtRInt, sizeof(int),
	OFST(matrix.rowoff), XtRImmediate, 0},
    {HgwNcolOffset, HgwCcolOffset, XtRInt, sizeof(int),
	OFST(matrix.coloff), XtRImmediate, 0},
    {XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
	OFST(matrix.foreground), XtRString, XtDefaultForeground},
    {XtNfont,  XtCFont, XtRFontStruct, sizeof(XFontStruct *),
	OFST(matrix.font),XtRString, XtDefaultFont},
    {XtNcallback, XtCCallback, XtRCallback, sizeof(XtPointer),
	OFST(matrix.callbacks), XtRCallback, (XtPointer)NULL},
    {HgwNcolorBase, HgwCcolorBase, XtRPointer, sizeof (XtPointer),
	 OFST(matrix.color_base), XtRImmediate, 0 },
    {HgwNcolors, HgwCcolors, XtRPointer, sizeof (XtPointer),
	 OFST(matrix.color_pixels), XtRImmediate, 0 },
    {HgwNnumColors, HgwCnumColors, XtRInt, sizeof(int),
	 OFST(matrix.ncolors), XtRImmediate, 0 },
};
#undef OFST

static void Highlight(), Edit(), Notify();
static void Initialize(), ClassInitialize(), Destroy();
static void Redisplay();
static Boolean SetValues();

#ifdef DEMO
static void ReColor();
#endif

static XtActionsRec actList[] = {
    {"highlight", Highlight},
    {"edit", Edit},
    {"notify", Notify},
#ifdef DEMO
    {"recolor", ReColor},
#endif
};

#define superClass  (&simpleClassRec)

MatrixClassRec matrixClassRec = {
    {
	(WidgetClass) superClass,	/* superclass */
	"Matrix",			/* class_name */
	sizeof(MatrixRec),		/* size */
	ClassInitialize,		/* class_initialize */
	NULL,				/* class_part_initialize  */
	FALSE,				/* class_inited */
	Initialize,			/* initialize */
	NULL,				/* initialize_hook */
	XtInheritRealize,		/* realize */
	actList,			/* actions */
	XtNumber(actList),		/* num_actions */
	resources,			/* resources */
	XtNumber(resources),		/* resource_count */
	NULLQUARK,			/* xrm_class */
	FALSE,				/* compress_motion */
	TRUE,				/* compress_exposure */
	TRUE,				/* compress_enterleave */
	FALSE,				/* visible_interest */
	Destroy,			/* destroy */
	XtInheritResize,		/* resize */
	Redisplay,			/* expose */
	SetValues,			/* set_values */
	NULL,				/* set_values_hook */
	XtInheritSetValuesAlmost,	/* set_values_almost */
	NULL,				/* get_values_hook */
	NULL,				/* accept_focus */
	XtVersion,			/* version */
	NULL,				/* callback_private */
	defTran,			/* tm_table */
	XtInheritQueryGeometry,		/* query_geometry */
	XtInheritDisplayAccelerator,	/* display_accelerator */
	NULL				/* extension */
    },	/* CoreClass fields initialization */
    {
	XtInheritChangeSensitive	/* change_sensitive */
    },	/* SimpleClass fields initialization */
    {
	0,				       /* field not used    */
    },	/* MatrixClass fields initialization */
};

/* for public consumption */

WidgetClass matrixWidgetClass = (WidgetClass) &matrixClassRec;

/***********************
*  Private Procedures  *
*		       *
***********************/

#ifdef DEBUG
/*  printmystate()
*
*   Private state dump.
*/

static void
printmystate(mmw)
    MatrixWidget mmw;
{
    (void)fprintf(stderr, "Matrix widget %x\n", (u_long)mmw);
    (void)fprintf(stderr,
	"  WinSiz=%dx%d	 CelSpc=%dx%d  MatSiz=%dx%d  Ofst=%dx%d	 HiLit=%dx%d\n",
	mmw->core.width,
	mmw->core.height,
	mmw->matrix.col_spc,
	mmw->matrix.row_spc,
	mmw->matrix.cols,
	mmw->matrix.rows,
	mmw->matrix.coloff,
	mmw->matrix.rowoff,
	mmw->matrix.hl_col,
	mmw->matrix.hl_row);
    (void)fprintf(stderr, "  MatAddr=%x, %s %s %s\n",
	mmw->matrix.base,
	((mmw->matrix.flag & IOwnMatrix) ? "OwnMatrix" : "ClientMatrix"),
	(mmw->matrix.numeric ? "Numeric" : "String"),
	(mmw->matrix.editable ? "Editable" : "NonEditable"));
}
#endif /* DEBUG */

/*
 *  Get_GC()
 *
 *   Alloc a GC with specified fg and bg pixels, also whatever
 *   font is current.
 */

static GC
Get_GC (mmw, fg, bg)
MatrixWidget mmw;
Pixel fg, bg;
{
    XGCValues	gcvs;

    gcvs.foreground   = fg;
    gcvs.background = bg;
    gcvs.font	    = mmw->matrix.font->fid;

    return XtGetGC ((Widget)mmw,
		    GCForeground|GCBackground|GCFont,
		    &gcvs);
}

/* XXX this is broken, only returns values for the first font */
static int
get_strwid(w, gc, s)
MatrixWidget w;
GC gc;
char *s;
{
    static XFontStruct *font_info = 0;
    int wd;

    if (!font_info)
	font_info = XQueryFont (XtDisplay(w), XGContextFromGC(gc));
    wd = XTextWidth (font_info, s, strlen(s));
    return wd;
}

/*
 *  paint_cell()
 *
 *   Draw a matrix cell (in reverse if highlighted).
 */

static void
paint_cell(mmw, row, col)
MatrixWidget mmw;
int row, col;
{
    GC bgc, tgc;
    int r = row - mmw->matrix.rowoff;
    int c = col - mmw->matrix.coloff;
    int nc = mmw->matrix.cols;
    int cw = mmw->matrix.col_spc;
    int rh = mmw->matrix.row_spc;
    char *str;
    int val;
    static char buf[16];

    if (mmw->matrix.base == NULL) /* sanity check */
	return;

    if (mmw->matrix.color_base != NULL && mmw->matrix.ncolors > 0) {
	int color_index = mmw->matrix.color_base[col + row * nc];

	if (color_index >= mmw->matrix.ncolors)
	    color_index = 0;

	if (col == mmw->matrix.hl_col && row == mmw->matrix.hl_row) {
	    tgc = mmw->matrix.color_GCs[color_index].inverse;
	    bgc = mmw->matrix.color_GCs[color_index].normal;
	}
	else {
	    tgc = mmw->matrix.color_GCs[color_index].normal;
	    bgc = mmw->matrix.color_GCs[color_index].inverse;
	}
    }
    else {
	if (col == mmw->matrix.hl_col && row == mmw->matrix.hl_row) {
	    tgc = mmw->matrix.inverse_GC;
	    bgc = mmw->matrix.normal_GC;
	}
	else {
	    tgc = mmw->matrix.normal_GC;
	    bgc = mmw->matrix.inverse_GC;
	}
    }

    XFillRectangle (XtDisplay (mmw), mmw->core.window, bgc,
		    c * cw + 1, r * rh + 1, cw - 3, rh - 3);

    if (mmw->matrix.numeric) {
	if (val = ((int*)(mmw->matrix.base))[col + row * nc]) {
	    sprintf (buf, "%d", val);
	    XDrawString (XtDisplay (mmw), mmw->core.window, tgc,
			 (c + 1) * cw - 2 - get_strwid(mmw, tgc, buf),
			 (r + 1) * rh - 2 - mmw->matrix.font->descent,
			 buf, strlen(buf));
	}

    }
    else {
	if ((str = ((char**)(mmw->matrix.base))[col + row * nc]) && *str) {
	    XDrawString (XtDisplay(mmw), mmw->core.window, tgc,
			 (c + 1) * cw - 2 - get_strwid(mmw, tgc, str),
			 (r + 1) * rh - 2 - mmw->matrix.font->descent,
			 str, strlen(str));
	}
    }
}

/*  lightup()
*
*   Select new highlighted cell.
*/

static void
lightup(mmw, row, col)
MatrixWidget mmw;
int row, col;	    /* cel coords */
{
    int oc, or;

    if (mmw->matrix.editable) {
	if (col < 0 || row < 0 || col >= mmw->matrix.cols
	    || row >= mmw->matrix.rows)
	    col = row = -1;
	if (col != mmw->matrix.hl_col || row != mmw->matrix.hl_row) {
	    oc = mmw->matrix.hl_col;
	    or = mmw->matrix.hl_row;
	    mmw->matrix.hl_col = col;
	    mmw->matrix.hl_row = row;
	    if (oc != -1)
		paint_cell (mmw, or, oc);
	    if (col != -1)
		paint_cell (mmw, row, col);
	}
    }
}


/*
 *
 *  Various Methods 
 *
 */

/*
 *  Initialize()
 *
 *   Creating new widget.  Check resources and allocate private stuff.
 */

static void
Initialize (request, new, args, num_args)
Widget request, new;
ArgList args;			/* unused */
Cardinal *num_args;		/* unused */
{
    MatrixWidget mmw = (MatrixWidget)new;
    int i;

    mmw->matrix.flag = 0;

    mmw->matrix.normal_GC = Get_GC (mmw, mmw->matrix.foreground,
				    mmw->core.background_pixel);
    mmw->matrix.inverse_GC = Get_GC (mmw, mmw->core.background_pixel,
				     mmw->matrix.foreground);


    /* bail if no rows or columns */
    
    if (mmw->matrix.rows < 1)
	XtError ("Matrix Widget has < 1 row");
    if (mmw->matrix.cols < 1)
	XtError ("Matrix Widget has < 1 col");

    /* set default spacing */

    if (mmw->matrix.col_spc < 1)
	mmw->matrix.col_spc = 100;
    if (mmw->matrix.row_spc < 1)
	mmw->matrix.row_spc = mmw->matrix.font->ascent
	    + mmw->matrix.font->descent + 4;

    /* determine width and height if not given */

    if (mmw->core.width < 1)
	mmw->core.width = mmw->matrix.cols * mmw->matrix.col_spc;
    if (mmw->core.height < 1)
	mmw->core.height = mmw->matrix.rows * mmw->matrix.row_spc;

    mmw->matrix.hl_row = -1;
    mmw->matrix.hl_col = -1;

    /* grab some arry if it's not given */

    if (!mmw->matrix.base) {
	if (mmw->matrix.numeric)
	    mmw->matrix.base = (XtPointer) CALLOC (mmw->matrix.rows *
						   mmw->matrix.cols,
						   sizeof(int));
	else
	    mmw->matrix.base = (XtPointer) CALLOC (mmw->matrix.rows *
						   mmw->matrix.cols,
						   sizeof(char*));
	mmw->matrix.flag |= IOwnMatrix;
    }

    /* clip bogus row, column offsets */

    if (mmw->matrix.rowoff < 0)
	mmw->matrix.rowoff = 0;
    if (mmw->matrix.rowoff >= mmw->matrix.rows)
	mmw->matrix.rowoff = mmw->matrix.rows - 1;
    if (mmw->matrix.coloff < 0)
	mmw->matrix.coloff = 0;
    if (mmw->matrix.coloff >= mmw->matrix.cols)
	mmw->matrix.coloff = mmw->matrix.cols - 1;

#ifdef DEBUG
    printmystate (mmw);
#endif
}

/*
 *  Redisplay()
 *
 *   Redraw the matrix for a screen refresh.
 */

static void
Redisplay(w, xev, region)
Widget w;
XEvent *xev;
Region region;
{
    MatrixWidget mmw = (MatrixWidget) w;
    GC gc = mmw->matrix.normal_GC;

    switch (xev->type) {
    case Expose:
	if (xev->xexpose.count == 0) {
	    int rs = mmw->matrix.row_spc;
	    int cs = mmw->matrix.col_spc;
	    int mr, mc;	    /* max visible row, column */
	    int mx, my;	    /* max x, y of visible active matrix */
	    int x, y;

	    mr = mmw->matrix.rowoff + (mmw->core.height + rs - 3) / rs;
	    if (mr > mmw->matrix.rows)
		mr = mmw->matrix.rows;
	    mc = mmw->matrix.coloff + (mmw->core.width + cs - 3) / cs;
	    if (mc > mmw->matrix.cols)
		mc = mmw->matrix.cols;
	    my = (mr - mmw->matrix.rowoff) * rs;
	    mx = (mc - mmw->matrix.coloff) * cs;

	    for (y = rs - 1; y < my; y += rs) {
		XDrawLine (XtDisplay (w), mmw->core.window, gc, 0, y, mx, y);
		for (x = cs - 1; x < mx; x += cs)
		    XDrawLine (XtDisplay (w), mmw->core.window, gc,
			x, y - 3, x, y);
	    }

	    for (x = mmw->matrix.coloff; x < mc; x++)
		for (y = mmw->matrix.rowoff; y < mr; y++)
		    paint_cell (mmw, y, x);
	}
	break;
    }
}


/*
 *  Destroy()
 *
 *   Widget being destroyed.  Free any private stuff.
 */

static void
Destroy (w)
Widget w;
{
    MatrixWidget mmw = (MatrixWidget)w;

/*
    fputs("Destroy()\n", stderr);
*/
    XtReleaseGC (w, mmw->matrix.inverse_GC);
    XtReleaseGC (w, mmw->matrix.normal_GC);

    /* XXX release GCs for colors if present */

    if (mmw->matrix.flag & IOwnMatrix)
	FREE ((char *) mmw->matrix.base);
}

static Boolean
SetValues (current, request, new)
Widget current, request, new;
{
    MatrixWidget omw = (MatrixWidget)current;
    MatrixWidget rmw = (MatrixWidget)request;
    MatrixWidget mmw = (MatrixWidget)new;
    Boolean redisplay = False;

/*
    fprintf(stderr, "SetValues(%x, %x, %x)\n", current, request, new);
*/

    if (omw->core.sensitive != mmw->core.sensitive && !mmw->core.sensitive) {
	/* about to become insensitive */
	/* XXX? */
    }


    /*
     * make new GCs if colors or font change
     */
    if ((omw->matrix.color_pixels != mmw->matrix.color_pixels) ||
	(omw->core.background_pixel != mmw->core.background_pixel) ||
	(omw->matrix.foreground != mmw->matrix.foreground) ||
	(omw->matrix.ncolors != mmw->matrix.ncolors) ||
	(omw->matrix.font != mmw->matrix.font)) {

	int i;

	XtReleaseGC (new, omw->matrix.inverse_GC);
	XtReleaseGC (new, omw->matrix.normal_GC);

	mmw->matrix.normal_GC = Get_GC (mmw, mmw->matrix.foreground,
					mmw->core.background_pixel);
	mmw->matrix.inverse_GC = Get_GC (mmw, mmw->core.background_pixel,
					 mmw->matrix.foreground);


	if (omw->matrix.color_GCs) {
	    for (i = 0; i < omw->matrix.ncolors; ++i) {
		XtReleaseGC (new, omw->matrix.color_GCs[i].normal);
		XtReleaseGC (new, omw->matrix.color_GCs[i].inverse);
		FREE ((char *) omw->matrix.color_GCs);
	    }
	}

	if (mmw->matrix.ncolors > 0) {
	    mmw->matrix.color_GCs = TALLOC (mmw->matrix.ncolors,
					    struct gc_pair);

	    for (i = 0; i < mmw->matrix.ncolors; ++i) {
		mmw->matrix.color_GCs[i].normal =
		    Get_GC (mmw, mmw->matrix.foreground,
			    mmw->matrix.color_pixels[i]);
		mmw->matrix.color_GCs[i].inverse =
		    Get_GC (mmw, mmw->matrix.color_pixels[i],
			    mmw->matrix.foreground);
	    }
	}
	

	redisplay = True;
    }

    /*
     * check and redraw if matrix size change
     * reallocate if we own matrix
     */

    if (omw->matrix.rows != mmw->matrix.rows
	|| omw->matrix.cols != mmw->matrix.cols) {
	if (mmw->matrix.rows < 1)
	    XtError ("Matrix Widget has < 1 row");
	if (mmw->matrix.cols < 1)
	    XtError ("Matrix Widget has < 1 col");
	
	mmw->matrix.hl_row = mmw->matrix.hl_col = -1;
	
	redisplay = True;
    }

    /*
     * check and redraw if offset changes
     */

    if (omw->matrix.rowoff != mmw->matrix.rowoff
	|| omw->matrix.coloff != mmw->matrix.coloff) {
	if (mmw->matrix.rowoff < 0)
	    mmw->matrix.rowoff = 0;
	if (mmw->matrix.rowoff >= mmw->matrix.rows)
	    mmw->matrix.rowoff = mmw->matrix.rows - 1;
	if (mmw->matrix.coloff < 0)
	    mmw->matrix.coloff = 0;
	if (mmw->matrix.coloff >= mmw->matrix.cols)
	    mmw->matrix.coloff = mmw->matrix.cols - 1;
	redisplay = True;
    }

    /*
     * free any old and redraw if matrix changes
     */

    if (omw->matrix.base != mmw->matrix.base) {
	if (omw->matrix.flag & IOwnMatrix)
	    FREE ((char *) omw->matrix.base);
	if (!mmw->matrix.base) {
	    if (mmw->matrix.numeric)
		mmw->matrix.base =
		    (XtPointer) CALLOC (mmw->matrix.rows * mmw->matrix.cols,
					sizeof(int));
	    else
		mmw->matrix.base =
		    (XtPointer) CALLOC (mmw->matrix.rows * mmw->matrix.cols,
					sizeof(char*));
	    mmw->matrix.flag |= IOwnMatrix;
	}
	else
	    mmw->matrix.flag &= ~IOwnMatrix;
	mmw->matrix.hl_row = mmw->matrix.hl_col = -1;
	redisplay = True;
    }

    if (omw->matrix.editable != mmw->matrix.editable && omw->matrix.editable)
	redisplay = True;

#ifdef DEBUG
    printmystate(mmw);
#endif
    return redisplay;
}

static void
ClassInitialize()
{
/*
    fputs("ClassInitialize()\n", stderr);
*/
    XawInitializeWidgetSet();
}


/**********************
*  Action Procedures  *
*		      *
**********************/

/*
 *  Notify()
 *
 *   Notify any callbacks that a cell value has changed.
 */

static void
Notify(w, ev, par, nump)
Widget w;
XEvent *ev;
String *par;
int *nump;
{
    MatrixWidget mmw = (MatrixWidget)w;
    static struct {
	char *base;
	int row;
	int col;
    } posn;

    if (mmw->matrix.hl_row != -1) {
	posn.base = mmw->matrix.base;
	posn.row = mmw->matrix.hl_row;
	posn.col = mmw->matrix.hl_col;
	XtCallCallbackList(w, mmw->matrix.callbacks, &posn);
    }
}

#ifdef DEMO
static void
ReColor (w, xev, par, nump)
Widget w;
XEvent *xev;
String *par;
int *nump;
{
    MatrixWidget mmw = (MatrixWidget)w;
    int hc = mmw->matrix.hl_col;    /* current highlighted column */
    int hr = mmw->matrix.hl_row;
    int ro = mmw->matrix.rowoff;    /* row offset */
    int co = mmw->matrix.coloff;

    if (++(mmw->matrix.color_base[hc + hr * mmw->matrix.cols]) >=
	mmw->matrix.ncolors)
	mmw->matrix.color_base[hc + hr * mmw->matrix.cols] = 0;
    fprintf (stderr, "color[%d][%d] = %d\n",
	     hc, hr, mmw->matrix.color_base[hc + hr * mmw->matrix.cols]);
    paint_cell (mmw, ro, co);
}
#endif

/*
 *  Highlight()
 *
 *   Highlight cel that pointer is in.
 */

static void
Highlight(w, xev, par, nump)
Widget w;
XEvent *xev;
String *par;
int *nump;
{
    MatrixWidget mmw = (MatrixWidget)w;
    int hc = mmw->matrix.hl_col;    /* current highlighted column */
    int hr = mmw->matrix.hl_row;
    int ro = mmw->matrix.rowoff;    /* row offset */
    int co = mmw->matrix.coloff;
    int mxr, mxc;		    /* max visible row, column */

/*
    fprintf(stderr, "Highlight(%c)\n", *nump ? par[0][0] : '*');
*/

    mxc = (mmw->core.width - 1) / mmw->matrix.col_spc + co;
    if (mxc >= mmw->matrix.cols)
	mxc = mmw->matrix.cols - 1;
    mxr = (mmw->core.height - 1) / mmw->matrix.row_spc + ro;
    if (mxr >= mmw->matrix.rows)
	mxr = mmw->matrix.rows - 1;

    switch (*nump ? par[0][0] : ' ') {
    case 'h':
	hc = co;
	hr = ro;
	break;

    case 'r':
	if (++hc > mxc) {
	    hc = co;
	    if (++hr > mxr)
		hr = ro;
	}
	break;

    case 'l':
	if (--hc < co) {
	    hc = mxc;
	    if (--hr < ro)
		hr = mxr;
	}
	break;

    case 'u':
	if (--hr < ro) {
	    hr = mxr;
	    if (--hc < co)
		hc = mxc;
	}
	break;

    case 'd':
	if (++hr > mxr) {
	    hr = ro;
	    if (++hc > mxc)
		hc = co;
	}
	break;

    default:
	switch (xev->type) {
	case MotionNotify:
	    lightup (mmw, ro + xev->xmotion.y / mmw->matrix.row_spc,
		     co + xev->xmotion.x / mmw->matrix.col_spc);
	    break;

	case LeaveNotify:
	    lightup (mmw, -1, -1);
	    break;
	}
	return;
	break;
    }
    lightup (mmw, hr, hc);
}

/* 
 *  Edit()
 *
 *   Change the value in the highlighted cel.
 */

static void
Edit(w, xev, par, nump)
Widget w;
XEvent *xev;
String *par;
int *nump;
{
    MatrixWidget mmw = (MatrixWidget)w;
    XKeyEvent *kev = (XKeyEvent*)xev;
    char **celp;
    int *valp;
    char keys[2];
    char c;
    int l;
    int nc = mmw->matrix.cols;
    int hc = mmw->matrix.hl_col;
    int hr = mmw->matrix.hl_row;

    if (xev->type == KeyPress && mmw->matrix.hl_row != -1
	&& mmw->matrix.hl_col != -1) {
	celp = &(((char**)(mmw->matrix.base))[hc + hr * nc]);
	valp = &(((int*)(mmw->matrix.base))[hc + hr * nc]);
	
	if (*nump && par[0][0] == 'd') {
	    if (mmw->matrix.numeric) {
		if (*valp) {
		    *valp /= 10;
		    paint_cell (mmw, hr, hc);
		}
	    }
	    else {
		if (*celp) {
		    if ((l = strlen (*celp)) > 0)
			(*celp)[--l] = 0;
		    if (!l) {
/*
			fprintf (stderr, "free(celp)\n");
*/
			FREE (*celp);
			*celp = 0;
		    }
		    paint_cell (mmw, hr, hc);
		}
	    }
	    return;
	}

	if (XLookupString (kev, keys, sizeof(keys), 0, 0) == 1) {
	    c = keys[0];
	    if (mmw->matrix.numeric) {
		if (c >= '0' && c <= '9') {
		    *valp = *valp * 10 + c - '0';
		    paint_cell(mmw, hr, hc);
		}
	    }
	    else {
		if (c >= ' ' && c <= '~') {
		    if (*celp) {
			l = strlen (*celp);
			if ((l + 1) / BITESIZE != l / BITESIZE) {
			    char *s;

/*
			    fprintf (stderr, "malloc(%d)\n",
			    (((l + 1) / BITESIZE) + 1) * BITESIZE);
*/
			    s = (char*) MALLOC((((l + 1) / BITESIZE) + 1) *
					      BITESIZE);
			    bcopy (*celp, s, l);
			    FREE (*celp);
			    *celp = s;
			}
			(*celp)[l] = c;
			(*celp)[l + 1] = 0;
		    }
		    else {
#if (BITESIZE > 1)
			*celp = (char*)MALLOC(BITESIZE);
#else
			*celp = (char*)MALLOC(2);
#endif
			(*celp)[0] = c;
			(*celp)[1] = 0;
		    }
		    paint_cell (mmw, hr, hc);
		}
	    }
	}
    }
}


void
MatrixRepaintCell (w, row, col)
Widget w;
int row, col;
{
    MatrixWidget mmw = (MatrixWidget) w;

    if (row < 0 || row >= mmw->matrix.rows ||
	col < 0 || col >= mmw->matrix.cols)
	return;
    paint_cell (mmw, row, col);
}
