/*----------------------------------------------------------------------*
 * Bounds Checking for GCC.						*
 * Copyright (C) 1995 Richard W.M. Jones <rwmj@doc.ic.ac.uk>.		*
 *----------------------------------------------------------------------*
 * This program is free software; you can redistribute it and/or modify	*
 * it under the terms of the GNU General Public License as published by	*
 * the Free Software Foundation; either version 2 of the License, or	*
 * (at your option) any later version.					*
 *									*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	*
 * GNU General Public License for more details.				*
 *									*
 * You should have received a copy of the GNU General Public License	*
 * along with this program; if not, write to the Free Software		*
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		*
 *----------------------------------------------------------------------*
 * File:
 *	treestats/readfile.c
 * Summary:
 *	Read a particular splay tree dump file into memory.
 * Other notes:
 *	
 * Author      	Date		Notes
 * RWMJ		28/5/95		Initial implementation.
 *----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>

#include "ext-tree.h"
#include "objects.h"
#include "readfile.h"

static int getsubtree (FILE *, object **);
static int compressed;
static void close_file (FILE *);
static FILE *open_file (char *);

int
readfile (char *filename, tree **t_return, struct ext_tree **et_return)
{
  FILE *fp;
  struct ext_tree *et = malloc (sizeof (struct ext_tree));
  tree *t = malloc (sizeof (tree));
  object *root;

  fp = open_file (filename);
  if (fp == NULL)
    {
      fprintf (stderr, "readfile: error opening `%s'.\n", filename);
      return -1;
    }
  if (fread (et, sizeof (struct ext_tree), 1, fp) != 1)
    {
      fprintf (stderr, "readfile: error reading header.\n");
      close_file (fp);
      return -1;
    }
  if (strcmp (et->magic, EXT_TREE_MAGIC) != 0)
    {
      fprintf (stderr, "readfile: dump file has incorrect magic number.\n");
      close_file (fp);
      return -1;
    }

  if (t_return != NULL)
    {
      if (getsubtree (fp, &root) != 0)
	{
	  close_file (fp);
	  return -1;
	}

      t->root = root;
      if (t_return)
	*t_return = t;
    }

  if (et_return)
    *et_return = et;

  close_file (fp);
  return 0;
}

static int
getsubtree (FILE *fp, object **obj_return)
{
  int present;

  if (fread (&present, sizeof (int), 1, fp) != 1)
    {
      fprintf (stderr, "readfile: error reading `present' flag.\n");
      return -1;
    }

  /* First word is a flag indicating whether this node is NULL or not.
   */
  if (present)
    {
      object *obj = malloc (sizeof (object));

      if (fread (obj, sizeof (object), 1, fp) != 1)
	{
	  fprintf (stderr, "readfile: error reading tree node.\n");
	  return -1;
	}
      if (getsubtree (fp, &(obj->left)) != 0
	  || getsubtree (fp, &(obj->right)) != 0)
	return -1;

      /* Zero out entries that aren't actually present in the file.
       */
      obj->filename = obj->name = obj->alloca_function = 0;
      obj->line = 0;

      /* Connect up parent properly.
       */
      if (obj->left)
	{
	  obj->left->parent = obj;
	  obj->left->left_of_parent = 1;
	}
      if (obj->right)
	{
	  obj->right->parent = obj;
	  obj->right->left_of_parent = 0;
	}

      /* Return object with current parent unset. */
      obj->parent = 0;

      if (obj_return)
	*obj_return = obj;
    }
  else
    {
      if (obj_return)
	*obj_return = NULL;
    }

  return 0;
}

/* Open and close a file, even if it is compressed.
 */
static FILE *
open_file (char *filename)
{
  char *t = strrchr (filename, '.');
  char cmd [strlen (filename) + 64];

  if (t != NULL &&
      (strcmp (t, ".gz") == 0 ||
       strcmp (t, ".Z") == 0 ||
       strcmp (t, ".z") == 0))
    {
      compressed = 1;
      sprintf (cmd, "zcat %s", filename);
      return popen (cmd, "r");
    }
  else
    {
      compressed = 0;
      return fopen (filename, "r");
    }
}

static void
close_file (FILE *fp)
{
  if (compressed)
    pclose (fp);
  else
    fclose (fp);
}
