/*----------------------------------------------------------------------*
 * Bounds Checking for GCC.						*
 * Copyright (C) 1995 Richard W.M. Jones <rwmj@doc.ic.ac.uk>.		*
 *----------------------------------------------------------------------*
 * This program is free software; you can redistribute it and/or modify	*
 * it under the terms of the GNU General Public License as published by	*
 * the Free Software Foundation; either version 2 of the License, or	*
 * (at your option) any later version.					*
 *									*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	*
 * GNU General Public License for more details.				*
 *									*
 * You should have received a copy of the GNU General Public License	*
 * along with this program; if not, write to the Free Software		*
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		*
 *----------------------------------------------------------------------*
 * File:
 *	treestats/mount-graphs.c
 * Summary:
 *	Mount tree images generated by st-graph.c into a single image.
 * Other notes:
 *	
 * Author      	Date		Notes
 * RWMJ		28/5/95		Initial implementation.
 *----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>

static int verbose = 0, N = 0;

static void usage (void) __attribute__ ((noreturn));
static void do_command (char *);

int
main (int argc, char *argv[])
{
  int i, Nx, Ny;
  double x = 72, y = 72,
    width = 8*72 - 144, height = 11*72 - 144,
    x_inc, y_inc, x_orig;
  char *graphs[argc];
  char command [2048];
  char *outputfile = NULL;

  if (argc < 3)
    usage ();

  Nx = atoi (argv[1]);

  for (i = 2; i < argc; ++i)
    {
      if (argv[i][0] == '-')
	{
	  switch (argv[i][1])
	    {
	    case 'v':
	      verbose = 1;
	      break;
	    case 'p':
	      if (i >= argc-4)
		usage ();
	      x = atoi (argv[++i]);
	      y = atoi (argv[++i]);
	      width = atoi (argv[++i]);
	      height = atoi (argv[++i]);
	      if (x < 0 || x >= width ||
		  y < 0 || y >= height ||
		  width <= 0 || height <= 0)
		usage ();
	      break;
	    case 'o':
	      if (outputfile || i >= argc-1)
		usage ();
	      outputfile = argv[++i];
	      break;
	    default:
	      usage ();
	    }
	}
      else
	graphs[N++] = argv[i];
    }

  if (N == 0 || Nx <= 0)
    usage ();
  if (!outputfile)
    outputfile = "output.ps";

  unlink (outputfile);

  Ny = N / Nx;
  if (N % Nx > 0) Ny ++;

  x_inc = width / Nx;
  y_inc = height / Ny;
  x_orig = x;
  y += (Ny - 1) * y_inc;

  for (i = 0; i < N; ++i)
    {
      sprintf (command, "st-graph -a -b -o %s -p %d %d %d %d %s",
	       outputfile,
	       (int) x, (int) y, (int) (x_inc * 0.9), (int) (y_inc * 0.9),
	       graphs[i]);
      do_command (command);
      if (i % Nx < Nx - 1)
	x += x_inc;
      else
	{
	  x = x_orig;
	  y -= y_inc;
	}
    }

  exit (0);
}

static void
do_command (char *cmd)
{
  int r;

  if (verbose)
    puts (cmd);
  r = system (cmd);
  if (r != 0)
    exit (r);
}

static void
usage (void)
{
  fprintf (stderr,
	   "mount-graphs: Mount trees from st-graph into a single image.\n"
	   "Usage:\n"
	   "  mount-graphs wide [-options] [-o outputfile] dumpfile [dumpfile [...]]\n"
	   "  where `wide' is number of images across page\n"
	   "Options:\n"
	   "  -o outputfile       Name file for resulting image (default: output.ps)\n"
	   "  -p x y width height Specify size of resulting image\n"
	   "  -v                  Echo commands before executing them\n");
  exit (1);
}
