#include "Bdef.h"

void RepTreeComb(BLACSCONTEXT *ctxt, char scope, int N, int length,
                 BLACBUFF *bp, BLACBUFF *bp2, int rdest1, int cdest1,
                 VVFUNPTR Xvvop)
/*
 *  -- V1.0 BLACS routine --
 *  University of Tennessee, February 28, 1995
 *  Written by Clint Whaley.
 *
 *  Purpose
 *  =======
 *  Perform a element-by-element combine on vectors.
 *  If rdest1 = -1, the answer will be left on all participating processes.
 *  Otherwise, only the process at grid coordinates {rdest1, cdest1} will
 *  have the final answer.  Other Processes will have intermediate (useless)
 *  values.
 *
 *  Arguments
 *  =========
 *  CTXT      (input) pointer to BLACSCONTEXT
 *            The BLACS context where operation is taking place.
 *
 *  SCOPE     (input) char
 *            Limit the scope of the operation.
 *            = 'r' :   Operation is performed by a process row.
 *            = 'c' :   Operation is performed by a process column.
 *            = 'a' :   Operation is performed by all processes in grid.
 *
 *  N         (input) int
 *            The number of elements in the vector.  N >= 0.
 *
 *  LENGTH    (input) int
 *            The length, in bytes, of the vector to be combined.
 *
 *  BP        (input/output) pointer to BLACBUFF.
 *            BLACBUFF is a special data type used by the BLACS to control
 *            buffers and the asynchronous operations coming out of them.
 *            This BLACBUFF should have a buffer who's first N elements
 *            contain the data to be combined. Additional space may be
 *            required, depending upon what combine is being performed.
 *            In any case, the buffer's length should be LENGTH.
 *
 *  BP2       (workspace) pointer to BLACBUFF.
 *            This BLACBUFF is used to receive information for combining with
 *            this process's information.  It should point to a buffer of size
 *            LENGTH.
 *
 *  RDEST1    (input) int
 *            Process row coordinate of node to receive the answer.
 *            If RDEST1 == -1, all nodes in scope receive the answer.
 *
 *  CDEST1    (input) int
 *            Process column coordinate of node to receive the answer.
 *            If RDEST1 == -1, CDEST is ignored.
 *  Xvvop     (input) pointer to typed operation function
 *            Points to a typed function which performs the required operation
 *            (e.g. summation) on the two N-element vectors.
 *
 * ------------------------------------------------------------------------
 */
{
   void UpdateBuffs(BLACBUFF *);
   BLACBUFF *getbuff(int);
   int BuffIsFree(BLACBUFF *, int);
   void Asend2d00(BLACSCONTEXT *, BLACBUFF *, int, int, int, int);
   void Srecv2d00(BLACSCONTEXT *, char *, int, int);

   char *buff, *buff2;
   int nnodes, msgid, dest, rdest, cdest, i, j;
   int nrcvs=0;	  /* Number of ReCeiVeS to do */
   int REBS;	  /* should info be RE-BroadcaSt? */
   int rightedge; /* right-most receiving node */
   int mydist;    /* my distance from destination node */
   extern int BLACOP, AOPDONE;

   if ( (REBS = (rdest1 == -1)) ) rdest1 = cdest1 = 0;

   switch (scope)
   {
   case 'r':
      nnodes = ctxt->npcol;
      if (nnodes < 2) return;
      mydist = (nnodes+ctxt->mycol-cdest1)%nnodes;
      break;
   case 'c':
      nnodes = ctxt->nprow;
      if (nnodes < 2) return;
      mydist = (nnodes+ctxt->myrow-rdest1)%nnodes;
      break;
   case 'a':
      nnodes = ctxt->Ng;
      if (nnodes < 2) return;
      dest = Mvkpnum(ctxt, rdest1, cdest1);
      mydist = (nnodes+ctxt->vIam-dest)%nnodes;
      break;
   default :
      return;
   }

   buff = bp->Buff;
   buff2 = bp2->Buff;
   rdest = ctxt->myrow;
   cdest = ctxt->mycol;

   for (i=1; (i < nnodes); i <<= 1)
   {
      switch (scope)
      {
      case 'r':
         cdest = (cdest1 + (mydist-1)*i) % nnodes;
         msgid = Mrid(ctxt);
         break;
      case 'c':
         rdest = (rdest1 + (mydist-1)*i) % nnodes;
         msgid = Mcid(ctxt);
         break;
      case 'a':
         j = (dest + (mydist-1)*i) % nnodes;
         Mvpcoord(ctxt, j, rdest, cdest);
         msgid = Maid(ctxt);
      }
      if (mydist - ((mydist >> 1) << 1) )  /* sending nodes */
      {
         Asend2d00(ctxt, bp, length, rdest, cdest, msgid);
/*
 *       Keep msgids in sync
 */
         switch (scope)
         {
         case 'r':
            for (i <<= 1; i < nnodes; i <<= 1) { msgid = Mrid(ctxt); }
            break;
         case 'c':
            for (i <<= 1; i < nnodes; i <<= 1) { msgid = Mcid(ctxt); }
            break;
         case 'a':
            for (i <<= 1; i < nnodes; i <<= 1) { msgid = Maid(ctxt); }
            break;
         }
	 break;		/* I'm done */
      }
      else
      {
         if ((nnodes+i-1)/i - mydist - 1)
         {
            Srecv2d00(ctxt, buff2, length, msgid);
            Xvvop(N, buff, buff2);
         }
         mydist >>= 1;
      }
   }

/*
 * Broadcast answer to everyone if RDEST == -1
 */
   if (REBS)
   {
      if (mydist == 0) i = Stree_bs(ctxt, scope, bp->Buff, length, 2);
      else
      {
         BuffIsFree(bp, 1);     /* wait for any sends to complete */
         i = Stree_br(ctxt, scope, bp->Buff, length, 2, rdest1, cdest1);
      }
   }
} /* end tree_comb */
