/*
 * Copyright 1991 Pei-Yuan Wei.  All rights reserved.
 *
 * Permission to use, copy, and/or distribute for any purpose and
 * without fee is hereby granted, provided that both the above copyright
 * notice and this permission notice appear in all copies and derived works.
 * Fees for distribution or use of this software or derived works may only
 * be charged with express written permission of the copyright holder.
 * This software is provided ``as is'' without express or implied warranty.
 */
%{
#define YYDEBUG 0
/*#define mydebug(x,y) {printf("%s...\n",x); PrintAST(y);}*/

#include <stdio.h>
#include "ast.h"

char 	*yyscript;
int	yyscriptidx;
char	*yyobjcontext;
int	yyscriptcontext;

%}

%{
/* Reserved words */
%}

%token	INC DEC
%token	INT CHAR STRING
%token 	AND OR
%token 	EQ NE LE GE
%token 	IF ELSE
%token	WHILE DO
%token	FOR
%token	PLUS_ASSERT MINUS_ASSERT MULT_ASSERT DIV_ASSERT MOD_ASSERT
%token	SWITCH CASE DEFAULT BREAK
%token	RETURN 
%token	PERSISTENT

%{
/* Constants */
%}

%token <i>	IDENT
%token <s>	STRINGCONST
%token <c>	CHARCONST
%token <i>	INTCONST 
%token <f>	FLOATCONST

%{
/* Precedence definition */	
%}

%left		AND OR
%left		EQ NE
%left		'>' '<' ';' ',' '(' ')'
%left		IDENT
%right		'=' PLUS_ASSERT MINUS_ASSERT MULT_ASSERT DIV_ASSERT MOD_ASSERT
%left		LT LE GT GE 
%left		'+' '-' 
%left		'*' '/' '%'
%left		'!'
%left 		UMINUS
%nonassoc 	INC DEC 

%{
/* Semantic stack definition */
%}

%union {
	struct AST	*p;		/* an AST node	*/
	int		i;
	float		f;
	char		c;
	char		*s;
}

%{
/* Nonterminals declearation */
%}

%type <p> 	prog
%type <p> 	persistent
%type <p> 	stmt stmts flow flows block
%type <p> 	body
%type <p> 	if_flow elseif
%type <p> 	for_flow exprs_stmts
%type <p> 	do_flow
%type <p> 	while_flow
%type <p>	switch_flow case_clause default_clause case_clauses 
%type <p> 	case_block case_blocks
%type <p> 	assert_stmt
%type <p> 	call_stmt
%type <p> 	return_stmt
%type <p> 	expr exprs term list_term
%type <p> 	integer float char string ident idents 

%{
/* Reduction code declarations */

extern YYSTYPE yylval;
AST *theAST = NULL;
AST *persistentAST = NULL;
AST *ast;

%}

%%

/*****************************************************************************
 * Start production rules
 */

prog 	:	persistent ';' block
		{
			persistentAST = $1;
			$$ = makeAST(AST_BODY);
			$$->children = $1;
			theAST = $$;
		}
	|	block
		{
			$$ = makeAST(AST_BODY);
			$$->children = $1;
			theAST = $$;
		}
	;

/*****************************************************************************
 * Persistent variables
 */
persistent :	PERSISTENT idents
		{
			$$ = makeAST(AST_PERSISTENT);
			$$->children = $2;
		}
	;

/*****************************************************************************
 * Statements 
 */
flow	:	do_flow
	|	for_flow
	|	if_flow
	|	switch_flow
	|	while_flow
	;

flows	:	flow
	|	flow flows
		{
			$$ = $1;
			$1->next = $2;
		}
	;

stmt	:	assert_stmt
	|	call_stmt
	|	return_stmt
	;

stmts	:	stmt ';'
	|	stmt ';' stmts	
		{
			$$ = $1;
			$1->next = $3;
		}
	;

block	:	stmts
	|	flows
	|	stmts block
		{
			if ($2) {
				for (ast = $1; ast->next; ast = ast->next);
				ast->next = $2;
			}
		}
	|	flows block
		{
			if ($2) {
				for (ast = $1; ast->next; ast = ast->next);
				ast->next = $2;
			}
		}
	|	'{' block '}'
		{
			$$ = $2;
		}
	|	block ';'
		{
			$$ = $1;
		}
	|	';'
		{
			$$ = NULL;
		}
	;

assert_stmt :	ident '=' expr
		{
			$$ = $1;
			$$->type = AST_MOVTO;
			$$->children = $3;
		}
	|	list_term '=' expr
		{
			$$ = $1;
			$$->type = AST_MOVTO_LIST;
			$$->children->next = $3;
		}
	;

call_stmt :	ident '(' exprs ')'
		{
			$$ = $1;
			$$->type = AST_CALL;
			$$->children = $3;
		}
	|	ident '(' ')'
		{
			$$ = $1;
			$$->type = AST_CALL;
		}
	;

body	:	stmt ';'
	|	flow
	|	'{' block '}'
		{ 
			$$ = makeAST(AST_BODY);
			$$->children = $2;
		}
	|	'{' '}'
		{ 
			$$ = makeAST(AST_BODY);
		}
	|	body ';'
		{
			$$ = $1;
		}
	;

elseif	:	ELSE IF expr body
		{
			ast = $4;
			if (ast->type != AST_BODY) {
				ast = makeAST(AST_BODY);
				ast->children = $4;
			}
			$$ = $3;
			$$->next = ast;
		}
	|	ELSE IF expr body elseif
		{
			ast = $4;
			if (ast->type != AST_BODY) {
				ast = makeAST(AST_BODY);
				ast->children = $4;
			}
			$$ = $3;
			$$->next = ast;
			ast->next = $5;
		}
	|	ELSE body
		{
			$$ = $2;
			if ($$->type != AST_BODY) {
				$$ = makeAST(AST_BODY);
				$$->children = $2;
			}
		}
	;

if_flow :	IF expr body
		{
			ast = $3;
			if (ast->type != AST_BODY) {
				ast = makeAST(AST_BODY);
				ast->children = $3;
			}
			$$ = makeAST(AST_IF);
			$$->children = $2;
			$2->next = ast;
		}
	|	IF expr body elseif
		{
			ast = $3;
			if (ast->type != AST_BODY) {
				ast = makeAST(AST_BODY);
				ast->children = $3;
			}
			$$ = makeAST(AST_IF);
			$$->children = $2;
			$2->next = ast;
			ast->next = $4;
		}
	;

while_flow :	WHILE expr body
		{
			ast = $3;
			if (ast->type != AST_BODY) {
				ast = makeAST(AST_BODY);
				ast->children = $3;
			}
			$$ = makeAST(AST_WHILE);
			$$->children = $2;
			$2->next = ast;
		}
	;

do_flow :	DO expr body WHILE expr ';'
		{
			ast = $3;
			if (ast->type != AST_BODY) {
				ast = makeAST(AST_BODY);
				ast->children = $3;
			}
			$$ = makeAST(AST_DO);
			$$->children = $2;
			$2->next = ast;
			ast->next = $5;
		}
	;

/* could this bad? */
exprs_stmts :	'('       ';'      ';'       ')'
		{
			$$ = makeAST(AST_EXPR);
			$$->next = makeAST(AST_EXPR);
			$$->next->next = makeAST(AST_EXPR);
		}
	|	'('       ';'      ';' exprs ')'
		{
			$$ = makeAST(AST_EXPR);
			$$->next = makeAST(AST_EXPR);
			$$->next->next = $4;
		}
	|	'('	  ';' expr ';'       ')'
		{
			$$ = makeAST(AST_EXPR);
			$$->next = $3;
			$$->next->next = makeAST(AST_EXPR);
		}
	|	'('	  ';' expr ';' exprs ')'
		{
			$$ = makeAST(AST_EXPR);
			$$->next = $3;
			$$->next->next = $5;
		}
	|	'(' exprs ';'      ';'	     ')'
		{
			$$ = $2;
			$$->next = makeAST(AST_EXPR);
			$$->next->next = makeAST(AST_EXPR);
		}
	|	'(' exprs ';'      ';' exprs ')'
		{
			$$ = $2;
			$$->next = makeAST(AST_EXPR);
			$$->next->next = $5;
		}
	|	'(' exprs ';' expr ';'	     ')'
		{
			$$ = $2;
			$$->next = $4;
			$$->next->next = makeAST(AST_EXPR);
		}
	|	'(' exprs ';' expr ';' exprs ')'
		{
			$$ = $2;
			$$->next = $4;
			$$->next->next = $6;
		}
	;

for_flow :	FOR exprs_stmts body
		{
			$$ = makeAST(AST_FOR);
			$$->children = $2;
			$$->children->next->next->next = $3;
		}
	|	FOR exprs_stmts ';'
		{
			$$ = makeAST(AST_FOR);
			$$->children = $2;
		}
	;

case_clause :	CASE exprs ':' 
		{
			$$ = makeAST(AST_CASE_CLAUSE);
			$$->children = $2;
		}
	;

default_clause : DEFAULT ':' 
		{
			$$ = makeAST(AST_DEFAULT_CLAUSE);
		}
	;

case_clauses :	case_clause case_clauses
		{
			$$ = $1;
			$1->next = $2;
		}
	|	default_clause case_clauses
		{
			$$ = $1;
			$1->next = $2;
		}
	|	case_clause
	|	default_clause
	;

case_block :	case_clauses block BREAK ';'
		{
			$$ = makeAST(AST_CASE_SEG);
			$$->children = makeAST(AST_CASE);
			$$->children->children = $1;
			$$->children->next = makeAST(AST_BODY);
			$$->children->next->children = $2;
			$$->children->next->next = makeAST(AST_BREAK);
		}
	|	case_clauses BREAK ';'
		{
			$$ = makeAST(AST_CASE_SEG);
			$$->children = makeAST(AST_CASE);
			$$->children->children = $1;
			$$->children->next = makeAST(AST_BODY);
			$$->children->next->next = makeAST(AST_BREAK);
		}
	;

case_blocks :	case_block case_blocks
		{
			$$ = $1;
			$$->next = $2;
		}
	|	case_block
	;

switch_flow :	SWITCH expr '{' case_blocks '}'
		{
			$$ = makeAST(AST_SWITCH);
			$$->children = $2;
			$2->next = $4;
		}
	;

return_stmt :	RETURN stmt
		{
			$$ = makeAST(AST_RETURN);
			$$->children = $2;
		}
	|	RETURN expr
		{
			$$ = makeAST(AST_RETURN);
			$$->children = $2;
		}
	|	RETURN
		{
			$$ = makeAST(AST_RETURN);
		}
	;

/*****************************************************************************
 * Simple expressions
 */

term	:	ident
	|	float
	|	integer
	|	char
	|	string
	|	list_term
	;

list_term :	ident '[' exprs ']'
		{
			$$ = $1;
			$$->type = AST_LIST;
			$$->children = $3;
		}
	;

expr	:	'(' expr ')'
		{
			$$ = $2;
		}
	|	'-' expr	%prec UMINUS
		{ 
			$$ = makeAST(AST_INC_PRE);
			$$->children = $2;
		}
	|	INC expr
		{ 
			$$ = makeAST(AST_INC_PRE);
			$$->children = $2;
		}
	|	expr INC
		{ 
			$$ = makeAST(AST_INC_POST);
			$$->children = $1;
		}
	|	DEC expr
		{ 
			$$ = makeAST(AST_DEC_PRE);
			$$->children = $2;
		}
	|	expr DEC
		{ 
			$$ = makeAST(AST_DEC_POST);
			$$->children = $1;
		}
	|	expr EQ expr
		{ 
			$$ = makeAST(AST_EXPR_EQ);
			$$->children = $1;
			$1->next = $3;
		}
	|	expr NE expr
		{ 
			$$ = makeAST(AST_EXPR_NE);
			$$->children = $1;
			$1->next = $3;
		}
	|	expr LE expr	
		{ 
			$$ = makeAST(AST_EXPR_LE);
			$$->children = $1;
			$1->next = $3;
		}
	|	expr GE expr	
		{ 
			$$ = makeAST(AST_EXPR_GE);
			$$->children = $1;
			$1->next = $3;
		}
	|	expr '+' expr
		{ 
			$$ = makeAST(AST_EXPR_PLUS);
			$$->children = $1;
			$1->next = $3;
		}
	|	expr '-' expr
		{ 
			$$ = makeAST(AST_EXPR_MINUS);
			$$->children = $1;
			$1->next = $3;
		}
	|	expr '*' expr
		{ 
			$$ = makeAST(AST_EXPR_MULT);
			$$->children = $1;
			$1->next = $3;
		}
	|	expr '/' expr
		{ 
			$$ = makeAST(AST_EXPR_DIV);
			$$->children = $1;
			$1->next = $3;
		}
	|	expr '>' expr
		{ 
			$$ = makeAST(AST_EXPR_GT);
			$$->children = $1;
			$1->next = $3;
		}
	|	expr '<' expr
		{ 
			$$ = makeAST(AST_EXPR_LT);
			$$->children = $1;
			$1->next = $3;
		}
	|	stmt
	|	term
	|	ident '[' ']'
		{
			$$ = $1;
			$$->type = AST_LISTC;
		}
	;

exprs : 	expr 
	|	expr ',' exprs
		{
			$$ = $1;
			$1->next = $3;
		}
	;

/*****************************************************************************
 * Constants
 */

ident 	:	IDENT	
		{
			$$ = makeAST(AST_REF);
			$$->attribute.info.i = $1;
		}
	;

idents	:	ident ',' idents
		{
			$$ = $1;
			$1->next = $3;
		}
	|	ident
	;

integer :	INTCONST
		{
			$$ = makeAST(AST_INTEGER);
			$$->attribute.info.i = $1;
		}
	;

float :		FLOATCONST
		{
			$$ = makeAST(AST_FLOAT);
			$$->attribute.info.f = $1;
		}
	;

char	:	CHARCONST
		{
			$$ = makeAST(AST_CHAR);
			$$->attribute.info.c = $1;
		}
	;

string	:	STRINGCONST
		{
			$$ = makeAST(AST_STRING);
			$$->attribute.info.s = $1;
		}
	;

%%

yyerror() {
	reportError();
}

yywrap() {
	return 1;
}




