/*
 * scanutil.c
 */
/*
 * Copyright 1991 Pei-Yuan Wei.  All rights reserved.
 *
 * Permission to use, copy, and/or distribute for any purpose and
 * without fee is hereby granted, provided that both the above copyright
 * notice and this permission notice appear in all copies and derived works.
 * Fees for distribution or use of this software or derived works may only
 * be charged with express written permission of the copyright holder.
 * This software is provided ``as is'' without express or implied warranty.
 */
#include <stdio.h>
#include <ctype.h>
#include "y.tab.h"
#include "hash.h"
#include "mystrings.h"
#include "ident.h"
#include "ast.h"

extern YYSTYPE	yylval;

#ifdef notusedheap
#define HEAP_SIZE 5000
char *stringHeap;  /* pointer to a HEAP_SIZE memory for string table */
int stringHeapIdx = 0;   /* index to current position in string table heap */
int strID = 0;
#endif

/* starting identifier for program identifiers (see ident.h) */
int identStrID = STR_LAST_OF_THE_PREDEFINED; 
int varStrID = 0;

/*
 * Reserved-word/token table 
 */
typedef struct strNumPair {
	char 	*s;
	int	i;
} strNumPair;

/*
 * Initialize scanner
 */
int init_scanutils()
{
	int i;
	return 1;
}

/* 
 * Stores identifier in hashtable
 */
int storeIdent(identStr)
	char *identStr;
{
	HashEntry *entry;

	if (!(entry = getHashEntry(symStr2ID, (long)identStr))) {
		entry = putHashEntry(symStr2ID, (long)identStr, ++identStrID);
		putHashEntry(symID2Str, identStrID, (long)identStr);
	}

	return entry->val;
}

int getIdent(identStr)
	char *identStr;
{
	HashEntry *entry;

	if (entry = getHashEntry(symStr2ID, (long)identStr))
		return entry->val;

	return NULL;
}

/*
 * Convert a "back-slash code" character representation to a character
 * i.e. "a" or "\n" or "\253"
 */
char convertEscChar()
{
	char c;

	if (input() == '\\') {  /* deal with escape character */
		switch (c = input()) {
		case 'n': 
			return '\n'; 	/* newline */
		case 't':
			return '\t'; 	/* horizontal tab */
		case 'v': 
			return '\v'; 	/* vertal tab */
		case 'b': 
			return '\b'; 	/* backspace */
		case 'r': 
			return '\r';	/* carriage return */
		case 'f': 
			return '\f'; 	/* form feed */
		case '\\': 
			return '\\'; 	/* backslash */
		case '\'': 
			return '\''; 	/* single quote */
		case '\"': 
			return '\"';	/* double quote */
		default:
			unput(c);	/* number after '\' is the ascii # */
			return (char)scanIntNDigits(3); 
		}
	}
	return c;
}

/*
 * Scan a character string, enclosed in two delimeter chars, into yytext
 * and return token and semantic value.
 * Converts back-slash representations to char. ie. "\n" => '\n'
 * NOTE: ascii codes must be three digits, unless it is at the end of 
 * the string, or the next character is anything other than a digit.
 * example: "\65B" => "AB"   "\0652" => "A2"   "\65\66" => "AB"
 */
void scanStringBlock(delimeter)
	char delimeter;
{
	char c;
	short i=0;

L:	if ((c = input()) == '\\') {
		unput(c);
		yytext[i++] = convertEscChar();
		goto L;
	} else if (c == delimeter) {
		yytext[i] = '\0';
		return;
	} else {
		yytext[i++] = c;
	}
	goto L;
}

/*
 * Scan a character constant into yytext
 * and return token and semantic value.
 */
int scanChar()
{
	scanStringBlock('\''); /* extract and convert the escape-char */
	yylval.c = yytext[0]; /* representation */

	return CHARCONST;
}

/*
 * Scan a character string into yytext, store it in string table,
 * and return token and semantic value.
 */
int scanString()
{
	scanStringBlock('\"');	/* process escape characters */
	yylval.s = SaveString(yytext); /* should use headp... */

	return STRINGCONST;
}

/*
 * Scan an identifier.
 */
int scanIdentifier(sp)
	char *sp;
{
	HashEntry *entry;

	/* store in symbolic table */
	yylval.i = storeIdent(SaveString(sp));
	return IDENT;
}

/*
 * Scan for integer
 * Return integer value.
 */
int scanInt()
{
	int i = 0;
/*
	char c;

	while (isdigit(c = input())) buff[i++] = c;
	unput(c);
	buff[i] = '\0';
	i = atoi(buff);
*/
	i = (int)atoi(yytext);

	return i;
}

/*
 * Scan for integer, maximum of n digits.
 * Return integer value.
 */
int scanIntNDigits(n)
	int n;
{
	int i = 0;
	char c;
	while (isdigit(c = input()) && (n-- > 0)) buff[i++] = c;
	unput(c);
	buff[i] = '\0';
	i = (int)atoi(buff);

	return i;
}

/*
 * Scan for either '/' token or drow away comment line. 
 * No nested comments allowed.
 */
char scanComment()
{
	char c = yylval.c = input();

	unput(c);
	if (c != '*') return '/';
L:	if ((c = input()) == '*') goto L;
	if (c == '/') return NULL;
	while (input() != '*');
	goto L;
}

int dumpProximityErrorLine(buff, i)
	char *buff;
	int *i;
{
	char c, j;

	for (j = 0; c = yyscript[(*i)++]; j++) {
		if (c == '\n') {
			buff[j] = '\0';
			return 1;
		}
		buff[j] = c;
	}
	buff[j] = '\0';
	return 0;
}

reportError()
{
	int i = 0, ln = 0, hasmore = 1;
	char c;

	if (yyscriptcontext == SCRIPT_ORIGIN_OBJ_SCRIPT) {
		fprintf(stderr, 
			"error at line %d... (obj=%s, in script)\n", 
			yylineno, yyobjcontext);
	} else if (yyscriptcontext == SCRIPT_ORIGIN_CLASS_SCRIPT) {
		fprintf(stderr,
			"error at line %d... (obj=%s, in classScript)\n", 
			yylineno, yyobjcontext);
	} else {
		fprintf(stderr,
			"error at line %d... (obj=%s, in eval)\n", 
			yylineno, yyobjcontext);
		fprintf(stderr, 
			"script = {%s}\n", 
			yyscript);
	}

	for (;;) {
		c = yyscript[i];
		if (c == '\0') return;
		if (c == '\n') {
			ln++;
			if (ln >= yylineno - 2) break;
		}
		i++;
	}

	hasmore = dumpProximityErrorLine(buff, &i);
	printf(" %d:\t%s\n", ln++, buff);
	if (!hasmore) return;
	hasmore = dumpProximityErrorLine(buff, &i);
	printf(" %d:\t%s\n", ln++, buff);
	if (!hasmore) return;
	hasmore = dumpProximityErrorLine(buff, &i);
	printf("*%d:\t%s\n", ln++, buff);
	if (!hasmore) return;
	hasmore = dumpProximityErrorLine(buff, &i);
	printf(" %d:\t%s\n", ln++, buff);
	if (!hasmore) return;
	hasmore = dumpProximityErrorLine(buff, &i);
	printf(" %d:\t%s\n", ln++, buff);
	if (!hasmore) return;

}

