/*
 * This file is part of Crossbow.
 *
 * Crossbow is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation,
 * either version 3 of the License, or (at your option) any
 * later version.
 *
 * Crossbow is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with Crossbow.  If not, see
 * <https://www.gnu.org/licenses/>.
 */
#include "unittest.h"
#include "util.h"

#include <err.h>
#include <fcntl.h>
#include <stdarg.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>
#include <stdlib.h>

#include "persist_dir.h"
#include "persist_file.h"
#include "persist_items.h"

enum
{
    max_items = 4096,   /* matching the constant in persist_file.c */
};

static persist_dir_t *temp_persist_dir(util_ctx_t ctx)
{
    int tempdir;
    persist_dir_t *pd;

    tempdir = util_get_tempdir(ctx);
    if (tempdir == -1)
        return NULL;

    pd = persist_dir_new(tempdir);
    if (!pd)
        close(tempdir);
    return pd;
}

static int write_persist(persist_dir_t *pd,
                         const struct iovec *iov,
                         int iovlen,
                         const char *fname)
{
    int fd, e;

    fd = openat(persist_dir_fd(pd),
        fname,
        O_CREAT | O_WRONLY | O_TRUNC,
        S_IRWXU
    );

    if (fd == -1) {
        warn("openat %d test_file_1", persist_dir_fd(pd));
        return -1;
    }

    e = writev(fd, iov, iovlen) == -1 ? -1 : 0;
    if (fd != -1 && close(fd))
        warn("close");
    return e;
}

static int items_are(persist_file_t *pf, unsigned n, ...)
{
    va_list ap;
    int e = -1;
    const persist_items_t *items;

    va_start(ap, n);

    items = persist_file_get_items(pf);
    EXPECT_OR_EXIT(items);

    EXPECT_OR_EXIT(persist_items_size(items) == n);
    for (unsigned i = 0; i < n; i ++) {
        const persist_item_t *item;
        const char *expected;

        item = persist_items_get(items, i);
        expected = va_arg(ap, const char *);

        EXPECT_OR_EXIT(strncmp(expected, item->data, item->len) == 0);
    }

    e = 0;
exit:
    va_end(ap);
    return e;
}

static int test_persist_items(util_ctx_t ctx, intptr_t opaque)
{
    persist_items_t *items;
    int e = -1;

    int n_items = 111;
    int *vals = reallocarray(NULL, n_items, sizeof(*vals));
    EXPECT(vals);

    items = persist_items_new(0);
    EXPECT(items);

    persist_items_set_store(items, vals, free);

    for (int i = 0; i < n_items; i ++) {
        vals[i] = i;
        e = persist_items_add(items, &(persist_item_t){
            .data = &vals[i],
            .len = sizeof(*vals),
        });
        EXPECT_OR_EXIT(!e);
    }

    for (int i = 0; i < n_items; i ++) {
        const persist_item_t *item;
        int *val;

        item = persist_items_get(items, i);
        val = item->data;

        EXPECT_OR_EXIT(item->len == sizeof(*vals));
        EXPECT_OR_EXIT(*val == val - vals);
    }

    e = 0;
exit:
    persist_items_del(items);
    return e;
}

static int test_file_status_detection(util_ctx_t ctx, intptr_t opaque)
{
    persist_dir_t *pd;
    persist_file_t *pf;
    int e = -1;
    const char *fname = "testfile";

    /* this data type must kept in sync with the corresponding type in
     * persist_file.c */
    const size_t N = sizeof(short);

    /* This structure is initially configured to be a working persist
     * file, with two items stored ("hello" and "world"). */
    unsigned char magic_revision = 1;
    unsigned short items_count = 2;
    const struct iovec iovec[] = {
        { .iov_len = 8, .iov_base = "crossbow" },
        { .iov_len = 1, .iov_base = &magic_revision },
        { .iov_len = N, .iov_base = &(unsigned short){0} },
        { .iov_len = N, .iov_base = &items_count },
        { .iov_len = N, .iov_base = &(unsigned short){5} },
        { .iov_len = 5, .iov_base = "hello" },
        { .iov_len = N, .iov_base = &(unsigned short){5} },
        { .iov_len = 5, .iov_base = "world" },
    };

    const unsigned whole = sizeof(iovec) / sizeof(*iovec);

    pd = temp_persist_dir(ctx);
    EXPECT(pd);
    pf = persist_file_new(pd, fname);
    EXPECT_OR_EXIT(pf);

    EXPECT_OR_EXIT(write_persist(pd, iovec, whole, fname) == 0);
    EXPECT_OR_EXIT(persist_file_load(pf) == 0);
    EXPECT_OR_EXIT(persist_file_status(pf) == pfs_regular);
    EXPECT_OR_EXIT(items_are(pf, 2, "hello", "world") == 0);

    /* Recognized but wrong version. */
    magic_revision = 2;
    EXPECT_OR_EXIT(write_persist(pd, iovec, 2, fname) == 0);
    EXPECT_OR_EXIT(persist_file_load(pf) == 0);
    EXPECT_OR_EXIT(persist_file_status(pf) == pfs_incompat);
    magic_revision = 1;

    /* Good version, file too short, detected as corrupt. */
    for (int i = 2; i < whole; ++i) {
        EXPECT_OR_EXIT(write_persist(pd, iovec, i, fname) == 0);
        EXPECT_OR_EXIT(persist_file_load(pf) == 0);
        EXPECT_OR_EXIT(persist_file_status(pf) == pfs_corrupt);
    }

    /* Good version, but items count does not reflect the effective number
     * of items: the file is corrupt */
    items_count = 3;
    EXPECT_OR_EXIT(write_persist(pd, iovec, whole, fname) == 0);
    EXPECT_OR_EXIT(persist_file_load(pf) == 0);
    EXPECT_OR_EXIT(persist_file_status(pf) == pfs_corrupt);
    items_count = 2;

    /* Good version, although items count is smaller than what effectively
     * serialized: the file is loaded, but less items are detected */
    items_count = 1;
    EXPECT_OR_EXIT(write_persist(pd, iovec, whole, fname) == 0);
    EXPECT_OR_EXIT(persist_file_load(pf) == 0);
    EXPECT_OR_EXIT(persist_file_status(pf) == pfs_regular);
    EXPECT_OR_EXIT(items_are(pf, 1, "hello") == 0);
    items_count = 2;

    e = 0;
exit:
    persist_dir_del(pd);
    persist_file_del(pf);
    return e;
}

static int populate(persist_dir_t *pd, const char *name, unsigned n_items, ...)
{
    persist_file_t *pf;
    persist_items_t *items;
    int e = -1;
    va_list ap;

    va_start(ap, n_items);

    pf = persist_file_new(pd, name);
    EXPECT(pf);

    EXPECT_OR_EXIT(persist_file_status(pf) == pfs_unknown);

    EXPECT_OR_EXIT(persist_file_load(pf) == 0);
    EXPECT_OR_EXIT(persist_file_status(pf) == pfs_noent);
    EXPECT_OR_EXIT(persist_file_get_items(pf) == NULL);

    items = persist_items_new(n_items);

    for (unsigned i = 0; i < n_items; ++i) {
        const char *item;

        item = va_arg(ap, const char *);
        EXPECT_OR_EXIT(
            persist_items_add(items, &(persist_item_t){
                .data = (void *)item,
                .len = strlen(item)
            }) == 0
        );
    }

    EXPECT_OR_EXIT(persist_file_swap_items(pf, items) == NULL);
    persist_file_set_incrid(pf, n_items);
    EXPECT_OR_EXIT(persist_file_write(pf) == 0);

    e = 0;
exit:
    va_end(ap);
    persist_file_del(pf);
    return e;
}

static int test_workflow(util_ctx_t ctx, intptr_t opaque)
{
    persist_dir_t *pd;
    persist_file_t *pf = NULL;
    persist_items_t *items = NULL;
    int e = -1;
    const char *fname = "feed1";

    pd = temp_persist_dir(ctx);
    EXPECT(pd);

    /* Pre-populate a file. */
    EXPECT_OR_EXIT(populate(pd, fname, 1, "one") == 0);

    /* Get a file, and load it. */
    pf = persist_file_new(pd, fname);
    EXPECT_OR_EXIT(pf);
    EXPECT_OR_EXIT(persist_file_load(pf) == 0);
    EXPECT_OR_EXIT(persist_file_status(pf) == pfs_regular);
    EXPECT_OR_EXIT(items_are(pf, 1, "one") == 0);
    EXPECT_OR_EXIT(persist_file_get_incrid(pf) == 1);

    /* Add one item and write. */
    items = persist_items_new(1);
    EXPECT_OR_EXIT(
        persist_items_add(items, &(persist_item_t){
            .data = "two",
            .len = 3,
        }) == 0
    );
    EXPECT_OR_EXIT(
        persist_items_add(items, &(persist_item_t){
            .data = "three",
            .len = 5,
        }) == 0
    );
    persist_file_set_incrid(pf, 3);
    items = persist_file_swap_items(pf, items);
    EXPECT_OR_EXIT(persist_file_write(pf) == 0);

    /* Reload: find the new item. */
    EXPECT_OR_EXIT(persist_file_load(pf) == 0);
    EXPECT_OR_EXIT(persist_file_status(pf) == pfs_regular);
    EXPECT_OR_EXIT(items_are(pf, 2, "two", "three") == 0);
    EXPECT_OR_EXIT(persist_file_get_incrid(pf) == 3);

    e = 0;
exit:
    persist_items_del(items);
    persist_file_del(pf);
    persist_dir_del(pd);
    return e;
}

static int test_max_items(util_ctx_t ctx, intptr_t opaque)
{
    int e = -1;
    persist_items_t *items;
    items = persist_items_new(opaque);

    /* Items are not allocated if the requested size goes beyond
     * the maximum. */
    if (opaque <= max_items) {
        EXPECT_OR_EXIT(items);
    } else {
        EXPECT_OR_EXIT(!items);
        goto success;
    }

    for (int i = 0; i < max_items; ++i)
        EXPECT_OR_EXIT(
            persist_items_add(items, &(persist_item_t){
                .data = &i,
                .len = sizeof(i)
            }) == 0
        );

    EXPECT_OR_EXIT(persist_items_size(items) == max_items);
    EXPECT_OR_EXIT(
        persist_items_add(items, &(persist_item_t){
            .data = &(long){ 9999 },
            .len = sizeof(long)
        }) != 0
    );

success:
    e = 0;
exit:
    persist_items_del(items);
    return e;
}

static int test_arbitrary_item_name(util_ctx_t ctx, intptr_t opaque)
{
    char buffer[256];
    persist_dir_t *pd;
    persist_file_t *pf = NULL;
    const persist_item_t *item;
    persist_items_t *items = NULL;
    int e = -1;
    const char *fname = "feed1";

    pd = temp_persist_dir(ctx);
    EXPECT_OR_EXIT(pd);
    pf = persist_file_new(pd, fname);
    EXPECT_OR_EXIT(pf);

    /* Any byte, arbitrary order.  The rationale is: every feed might define an
     * arbitrary item identifier, and we shouldn't be affected by weird choices
     * (e.g. nil-character). */
    for (int i = 0; i < sizeof(buffer); ++i)
        buffer[i] = (127 + i) % 256;

    items = persist_items_new(0);
    EXPECT_OR_EXIT(items);

    EXPECT_OR_EXIT(
        persist_items_add(items, &(persist_item_t){
            .data = buffer,
            .len = sizeof(buffer),
        }) == 0
    );
    EXPECT_OR_EXIT(persist_items_size(items) == 1);

    EXPECT_OR_EXIT(persist_file_load(pf) == 0);
    EXPECT_OR_EXIT(persist_file_status(pf) == pfs_noent);
    EXPECT_OR_EXIT(persist_file_get_items(pf) == NULL);

    items = persist_file_swap_items(pf, items);
    EXPECT_OR_EXIT(!items);
    EXPECT_OR_EXIT(persist_file_write(pf) == 0);

    EXPECT_OR_EXIT(persist_file_load(pf) == 0);
    EXPECT_OR_EXIT(persist_file_status(pf) == pfs_regular);

    items = persist_file_swap_items(pf, items);
    EXPECT_OR_EXIT(!!items);
    EXPECT_OR_EXIT(persist_items_size(items) == 1);

    item = persist_items_get(items, 0);
    EXPECT_OR_EXIT(item);
    EXPECT_OR_EXIT(item->len == sizeof(buffer));
    EXPECT_OR_EXIT(memcmp(item->data, buffer, item->len) == 0);

    e = 0;
exit:
    persist_items_del(items);
    persist_file_del(pf);
    persist_dir_del(pd);
    return e;
}

const struct test * list_test(void)
{
    static struct test tests[] = {
        TEST(1, test_persist_items, NULL),
        TEST(1, test_file_status_detection, NULL),
        TEST(1, test_workflow, NULL),
        TEST(1, test_max_items, 0),
        TEST(1, test_max_items, 10),
        TEST(1, test_max_items, max_items - 1),
        TEST(1, test_max_items, max_items),
        TEST(1, test_max_items, max_items + 1),
        TEST(1, test_max_items, max_items + 2),
        TEST(1, test_arbitrary_item_name, NULL),
        END_TESTS
    };
    return tests;
}
