/*
 * This file is part of Crossbow.
 *
 * Crossbow is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation,
 * either version 3 of the License, or (at your option) any
 * later version.
 *
 * Crossbow is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with Crossbow.  If not, see
 * <https://www.gnu.org/licenses/>.
 */
#include "persist_items.h"

#include <err.h>
#include <errno.h>
#include <stdint.h>
#include <limits.h>
#include <stdlib.h>

enum
{
    min_size = 16,
    grow_factor = 2,
    max_size = 4096,
};

struct persist_items
{
    unsigned short size, count;
    persist_item_t *array;

    void *store;
    void (*store_dtor)(void *store);
};

persist_items_t * persist_items_new(unsigned size)
{
    persist_items_t *items;

    if (size > max_size) {
        warnx("persist_items declared beyond max size");
        return NULL;
    }

    items = malloc(sizeof(*items));
    if (!items) {
        warn("malloc");
        return NULL;
    }

    if (size == 0)
        size = min_size;

    *items = (persist_items_t){
        .size = size,
        .array = reallocarray(NULL, size, sizeof(persist_items_t)),
    };

    if (!items->array) {
        warn("reallocarray");
        persist_items_del(items);
        return NULL;
    }

    return items;
}

void persist_items_del(persist_items_t *items)
{
    if (!items)
        return;

    if (items->store_dtor)
        items->store_dtor(items->store);
    free(items->array);
    free(items);
}

static int maybe_grow(persist_items_t *items)
{
    unsigned long new_size;
    persist_item_t *new_array;

    if (items->count < items->size)
        return 0;

    new_size = items->size * grow_factor;
    if (new_size > max_size)
        new_size = max_size;

    if (items->count >= new_size) {
        warnx("persist_items grown beyond max size");
        return -1;
    }

    new_array = reallocarray(
        items->array,
        new_size,
        sizeof(persist_item_t)
    );

    if (!new_array) {
        warn("reallocarray");
        return -1;
    }

    items->array = new_array;
    items->size = new_size;
    return 0;
}

int persist_items_add(persist_items_t *items, const persist_item_t *item)
{
    if (maybe_grow(items))
        return -1;

    items->array[items->count++] = *item;
    return 0;
}

void persist_items_set_store(
        persist_items_t *items,
        void *store,
        void (*dtor)(void *))
{
    items->store = store;
    items->store_dtor = dtor;
}

const persist_item_t * persist_items_get(
        const persist_items_t *items,
        unsigned idx)
{
    if (idx >= items->count)
        return NULL;

    return &items->array[idx];
}

unsigned persist_items_size(const persist_items_t *items)
{
    return items->count;
}
