/*
 * This file is part of Crossbow.
 *
 * Crossbow is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation,
 * either version 3 of the License, or (at your option) any
 * later version.
 *
 * Crossbow is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with Crossbow.  If not, see
 * <https://www.gnu.org/licenses/>.
 */

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sysexits.h>
#include <unistd.h>

#include "logging.h"
#include "persist_dir.h"
#include "persist_file.h"
#include "persist_items.h"
#include "savefile.h"
#include "savefile_aux.h"
#include "util.h"

enum
{
    permission = S_IRUSR | S_IWUSR | S_IRGRP
               | S_IWGRP | S_IROTH | S_IWOTH,
};

struct context
{
    struct {
        savefile_t *savefile;
    } v1;
    struct {
        persist_dir_t *persist_dir;
        FILE *out_config;
    } v2;
    int backup_dirfd;
};

static int home_path(char *dst, size_t len, const char *path)
{
    const char *home;
    int wrote;

    home = getenv("HOME");
    if (!home || !*home) {
        warnx("Undefined env: $HOME");
        return -1;
    }

    wrote = snprintf(dst, len, "%s/%s", home, path);
    if (wrote > len) {
        warnx("path truncated: %.*s", (int)len, dst);
        return -1;
    }

    return 0;
}

static int backup_dir(struct context *ctx)
{
    char path[PATH_MAX];
    int dirfd;

    dirfd = ctx->backup_dirfd;
    if (dirfd != -1)
        return dirfd;

    if (home_path(path, sizeof(path), "crossbow.backup"))
        return -1;

    if (mkdir(path, 0755) && errno != EEXIST) {
        warn("mkdir %s", path);
        return -1;
    }

    notice("Backing up old persist files in %s", path);
    dirfd = open(path, O_CLOEXEC | O_DIRECTORY);
    if (dirfd == -1)
        warn("open %s", path);
    else
        ctx->backup_dirfd = dirfd;

    return dirfd;
}

static FILE *new_config_file(void)
{
    char path[PATH_MAX];
    FILE *out;
    int fd;

    if (home_path(path, sizeof(path), ".crossbow.conf"))
        return NULL;

    fd = open(path, O_CREAT | O_EXCL | O_CLOEXEC | O_WRONLY, permission);
    if (fd == -1) {
        if (errno == EEXIST)
            warn("Refusing to overwrite %s", path);
        else
            warn("open %s", path);
        goto fail;
    }

    out = fdopen(fd, "w");
    if (out) {
        notice("Created config file: %s", path);
        return out;
    }
    warn("fopen %s", path);

fail:
    if (fd != -1) {
        if (close(fd))
            warn("close %d", fd);
    }
    return NULL;
}

static persist_dir_t * new_persist_dir(void)
{
    int fd;
    char path[PATH_MAX];
    persist_dir_t *out;

    if (home_path(path, sizeof(path), ".crossbow"))
        return NULL;

    if (mkdir(path, 0755) && errno != EEXIST) {
        warn("mkdir %s", path);
        return NULL;
    }

    fd = open(path, O_CLOEXEC | O_DIRECTORY);
    if (fd == -1) {
        warn("open %s", path);
        return NULL;
    }

    out = persist_dir_new(fd);
    if (out)
        notice("New persist directory: %s", path);
    else if (close(fd))
        warn("close %d", fd);
    return out;
}

static int print_config(struct context *ctx, const feed_t *feed)
{
    const char *name;
    FILE *out = ctx->v2.out_config;
    str_t str;
    feed_output_mode_t mode;

    name = feed_get_name(feed);
    fprintf(out, "feed %s\n", name);

    str = feed_get_effective_url(feed);
    if (str.len) {
        fprintf(out, " url %s%.*s\n",
            feed_get_url_type(feed) == feed_ut_local ? "file://" : "",
            STR_FMT(&str)
        );
    }

    mode = feed_get_output_mode(feed);

    switch (mode) {
    case feed_om_subproc:
    case feed_om_pipe:
        str = feed_get_subproc_chdir(feed);
        if (str.len)
            fprintf(out, " chdir %.*s\n", STR_FMT(&str));
    case feed_om_pretty:
    case feed_om_print:
        break;
    }

    str = feed_get_outfmt(feed);
    switch (mode) {
    case feed_om_pretty:
        if (str.len)
            fprintf(out, " format %.*s\n", STR_FMT(&str));
        else
            warnx("feed %s uses 'pretty' mode without format", name);
    case feed_om_print:
        break;

    case feed_om_pipe:
        if (str.len)
            fprintf(out, " handler pipe\n command %.*s\n", STR_FMT(&str));
        else
            warnx("feed %s uses 'pipe' mode without ", name);
        break;

    case feed_om_subproc:
        if (str.len)
            fprintf(out, " handler exec\n command %.*s\n", STR_FMT(&str));
        else
            warnx("feed %s uses 'pipe' mode without ", name);
        break;
    }

    fputc('\n', out);
    return 0;
}

static int backup_v1_feed(struct context *ctx, const feed_t *feed)
{
    const char *file_name;
    int e;

    file_name = feed_get_file_path(feed);
    e = renameat(
        savefile_get_dirfd(ctx->v1.savefile),
        file_name,
        backup_dir(ctx),
        file_name
    );

    if (e)
        warn("rename %s", file_name);
    else
        notice("Backed up %s", file_name);
    return e;
}

static int migrate_persist_file(struct context *ctx, const feed_t *feed)
{
    int e = -1;
    struct {
        persist_items_t *items;
        persist_file_t *pfile;
    } v2 = {};
    struct {
        feed_items_t items;
    } v1 = {
        .items = feed_get_items(feed)
    };

    v2.items = persist_items_new(v1.items.n_items);
    if (!v2.items)
        goto fail;

    v2.pfile = persist_file_new(ctx->v2.persist_dir, feed_get_name(feed));
    if (!v2.pfile)
        goto fail;

    if (persist_file_load(v2.pfile))
        goto fail;

    switch (persist_file_status(v2.pfile)) {
    case pfs_unknown:
        panic("unknown after successful persist_file_load");
    case pfs_regular:
        warnx("Persist file (v2) for %s already exists",
            feed_get_name(feed)
        );
        goto fail;
    case pfs_noent:
        break;
    case pfs_unrecognized:
    case pfs_corrupt:
        warnx("Refusing to write persist file (v2) for %s: "
            "unrecognized or corrupt",
            feed_get_name(feed)
        );
        goto fail;
    case pfs_incompat:
        if (backup_v1_feed(ctx, feed))
            goto fail;
        if (persist_file_load(v2.pfile))
            goto fail;
    }

    if (!persist_file_is_writeable(v2.pfile))
        goto fail;

    for (unsigned int i = 0; i < v1.items.n_items; i ++) {
        persist_item_t item;

        item.data = (void *)v1.items.items[i].bytes;
        item.len = v1.items.items[i].len;
        if (persist_items_add(v2.items, &item))
            goto fail;
    }

    v2.items = persist_file_swap_items(v2.pfile, v2.items);
    persist_file_set_incrid(v2.pfile, feed_get_items_count(feed));

    if (persist_file_write(v2.pfile))
        goto fail;

    e = 0;
fail:
    persist_file_del(v2.pfile);
    persist_items_del(v2.items);
    return e;
}

static int upgrade(struct context *ctx)
{
    const feed_t *feed;
    void *aux = NULL;

    aux = NULL;
    while (feed = savefile_citer_feeds(ctx->v1.savefile, &aux), feed) {
        if (print_config(ctx, feed))
            return -1;

        if (migrate_persist_file(ctx, feed))
            warnx("Failed migration for %s", feed_get_name(feed));
    }

    return 0;
}

int main(int argc, char **argv)
{
    int err = -1;
    struct context ctx = {
        .backup_dirfd = -1,
    };

    ctx.v2.out_config = new_config_file();
    if (!ctx.v2.out_config)
        goto fail;

    ctx.v2.persist_dir = new_persist_dir();
    if (!ctx.v2.persist_dir)
        goto fail;

    ctx.v1.savefile = open_savefile(false);
    if (!ctx.v1.savefile)
        goto fail;

    if (savefile_load_all(ctx.v1.savefile) == -1)
        goto fail;

    err = upgrade(&ctx);
fail:
    savefile_free(ctx.v1.savefile);
    persist_dir_del(ctx.v2.persist_dir);
    if (ctx.v2.out_config && fclose(ctx.v2.out_config)) {
        warn("fclose (v2 config file)");
        err = -1;
    }
    if (ctx.backup_dirfd != -1) {
        notice("NOTE: I needed to back up some Crossbow 1.x persist files.");
        notice("      You can drop them if you're satisfied with the migration");
        if (close(ctx.backup_dirfd))
            warn("close %d", ctx.backup_dirfd);
    }
    return err == 0 ? EXIT_SUCCESS : EXIT_FAILURE;
}
