/*
 * This file is part of Crossbow.
 *
 * Crossbow is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation,
 * either version 3 of the License, or (at your option) any
 * later version.
 *
 * Crossbow is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with Crossbow.  If not, see
 * <https://www.gnu.org/licenses/>.
 */

#include "download.h"

#include <err.h>
#include <limits.h>
#include <stdlib.h>
#include <stdint.h>
#include <sysexits.h>
#include <unistd.h>

#include <curl/curl.h>
#include <utlist.h>

#include "config.h"
#include "filemap.h"
#include "logging.h"

typedef struct job
{
    download_cb_t callback;
    void *opaque;
    CURL *request;
    int output_file;
    struct job *prev, *next;
} job_t;

struct download
{
    CURLM *curlm;
    job_t *jobs;
    bool global_init;
};

static CURLM * setup_curl_multi(unsigned maxjobs)
{
    CURLM *curlm = NULL;
    CURLMcode err;

    curlm = curl_multi_init();
    if (!curlm) {
        warnx("curl_multi_init failed");
        return NULL;
    }

    err = curl_multi_setopt(curlm, CURLMOPT_MAX_TOTAL_CONNECTIONS, (long)maxjobs);
    if (err) {
        warnx("curl_multi_setopt CURLMOPT_MAXCONNECTS: %s",
            curl_multi_strerror(err)
        );
        goto fail;
    }

    return curlm;

fail:
    curl_multi_cleanup(curlm);
    return NULL;
}

static int open_output_file(job_t *job)
{
    int fd;
    char filename[] = ".crossbow_XXXXXX";

    fd = mkstemp(filename);
    if (fd == -1) {
        warn("mkstemp");
        return -1;
    }

    if (unlink(filename))
        warn("unlink"); /* Not critical, just warn and continue. */

    job->output_file = fd;
    return 0;
}

static size_t write_cb(void *ptr, size_t size, size_t nmemb, void *opaque)
{
    job_t *job = opaque;
    size_t wrote = 0, to_write = size * nmemb;

    if (job->output_file == -1 && open_output_file(job))
        return 0;   /* note: write_cb has the same interface as fwrite(3) */

    while (to_write) {
        ssize_t w;

        w = write(job->output_file, (uint8_t *)ptr + wrote, to_write - wrote);
        if (w <= 0) {
            warn("write -> %zd", w);
            return 0;
        }

        wrote += w;
        to_write -= w;
    }

    return size * nmemb;
}

static CURL * setup_curl_easy(download_t *dl,
                              const char *url,
                              job_t *job)
{
    CURL *request;
    CURLcode err_setopt = 0;
    CURLMcode err_multi;

    request = curl_easy_init();
    if (!request) {
        warnx("curl_easy_init failed");
        return NULL;
    }

    err_setopt = curl_easy_setopt(request, CURLOPT_URL, url);
    if (err_setopt)
        goto fail;

    err_setopt = curl_easy_setopt(request,
        CURLOPT_USERAGENT,
        PACKAGE_NAME "/" PACKAGE_VERSION
    );
    if (err_setopt)
        goto fail;

    err_setopt = curl_easy_setopt(request, CURLOPT_PROTOCOLS,
        CURLPROTO_GOPHER | CURLPROTO_HTTP | CURLPROTO_HTTPS
    );
    if (err_setopt)
        goto fail;

    err_setopt = curl_easy_setopt(request, CURLOPT_WRITEFUNCTION, write_cb);
    if (err_setopt)
        goto fail;

    err_setopt = curl_easy_setopt(request, CURLOPT_WRITEDATA, job);
    if (err_setopt)
        goto fail;

    err_setopt = curl_easy_setopt(request, CURLOPT_FAILONERROR, 1L);
    if (err_setopt)
        goto fail;

    err_setopt = curl_easy_setopt(request, CURLOPT_FOLLOWLOCATION, 1L);
    if (err_setopt)
        goto fail;

    err_setopt = curl_easy_setopt(request, CURLOPT_PRIVATE, job);
    if (err_setopt)
        goto fail;

    err_multi = curl_multi_add_handle(dl->curlm, request);
    if (err_multi) {
        warnx("curl_multi_add_handle: %s",
            curl_multi_strerror(err_multi)
        );
        goto fail;
    }

    return request;

fail:
    if (err_setopt)
        warnx("curl_easy_setopt: %s", curl_easy_strerror(err_setopt));
    curl_easy_cleanup(request);
    return NULL;
}

download_t *download_new(bool global_init, unsigned maxjobs)
{
    download_t *dl;

    dl = malloc(sizeof(*dl));
    if (!dl) {
        warn("malloc");
        return NULL;
    }
    *dl = (download_t){};

    if (global_init) {
        CURLcode err;

        err = curl_global_init(CURL_GLOBAL_ALL);
        if (err) {
            warnx("curl_global_init: %s", curl_easy_strerror(err));
            goto fail;
        }
        dl->global_init = true;
    }

    dl->curlm = setup_curl_multi(maxjobs);
    if (!dl->curlm)
        goto fail;

    return dl;

fail:
    download_del(dl);
    return NULL;
}

static void job_delete(download_t *dl, job_t *job)
{
    if (!job)
        return;

    if (job->prev || job->next)
        DL_DELETE(dl->jobs, job);

    if (job->request) {
        CURLMcode err;

        err = curl_multi_remove_handle(dl->curlm, job->request);
        if (err)
            warnx("curl_multi_remove_handle: %s", curl_multi_strerror(err));

        curl_easy_cleanup(job->request);
    }

    if (job->output_file != -1 && close(job->output_file))
        warn("close");
    free(job);
}

int download_schedule(download_t *dl,
                      const char *url,
                      download_cb_t cb,
                      void *opaque)
{
    job_t *job;

    job = malloc(sizeof(*job));
    if (!job) {
        warn("malloc");
        return -1;
    }

    *job = (job_t){
        .callback = cb,
        .opaque = opaque,
        .request = setup_curl_easy(dl, url, job),
        .output_file = -1,
    };

    if (!job->request)
        goto fail;

    DL_APPEND(dl->jobs, job);
    return 0;

fail:
    job_delete(dl, job);
    return -1;
}

static int request_failed(const job_t *job)
{
    CURLcode err;
    long protocol, code;

    err = curl_easy_getinfo(job->request, CURLINFO_PROTOCOL, &protocol);
    if (err) {
        warnx("curl_easy_getinfo CURLINFO_RESPONSE_CODE: %s",
            curl_easy_strerror(err)
        );
        return -1;
    }

    switch (protocol)
    {
        // case CURLPROTO_FILE: - TODO should we use libcurl for local
        //                        files too?
        case CURLPROTO_GOPHER:
            /* Gopher is not very good at error handling.  There's no
             * generalized way of detecting errors.  E.g. gophernicus will
             * report erros with some user-friendly text.  Assuming
             * there's no error.
             */
            return 0;

        case CURLPROTO_HTTP:
        case CURLPROTO_HTTPS:
            err = curl_easy_getinfo(
                job->request,
                CURLINFO_RESPONSE_CODE,
                &code
            );
            if (err) {
                warnx("curl_easy_getinfo CURLINFO_RESPONSE_CODE: %s",
                    curl_easy_strerror(err)
                );
                return -1;
            }
            return code < 200 || code >= 300;
    }

    /* Not knowing any better, we assume a failure if the file was never
     * assigned so far by a write operation. */
    return job->output_file != -1;
}

static int report_failure(job_t *job, download_status_t dls)
{
    return job->callback(job->opaque, dls, NULL);
}

static int deliver(job_t *job)
{
    filemap_t filemap;
    int e;

    if (fdatasync(job->output_file)) {
        warn("fdatasync");
        return report_failure(job, dl_sys_failure);
    }

    if (filemap_load_fd(&filemap, job->output_file))
        return report_failure(job, dl_sys_failure);
    job->output_file = -1; /* on success, file ownership is moved */

    e = job->callback(job->opaque, dl_complete, &filemap);
    filemap_unload(&filemap);
    return e;
}

static int flush_ready(download_t *dl)
{
    int msgq_ign;
    CURLMsg *msg;

    while (msg = curl_multi_info_read(dl->curlm, &msgq_ign), msg) {
        job_t *job;
        int cancel;
        CURLcode err;

        if (msg->msg != CURLMSG_DONE)
            continue;

        err = curl_easy_getinfo(msg->easy_handle, CURLINFO_PRIVATE, &job);
        if (err) {
            /* This should not happen, but if it does, the job delivery
             * and deletion is going to be postponed until all downloads
             * are complete.
             */
            warnx("curl_easy_getinfo CURLINFO_PRIVATE: %s",
                curl_easy_strerror(err)
            );
            continue;
        }

        if (msg->data.result == CURLE_OK) {
            switch (request_failed(job)) {
            case -1:
                cancel = report_failure(job, dl_sys_failure);
                break;

            case 0:
                cancel = deliver(job);
                break;

            case 1:
                cancel = report_failure(job, dl_fetch_failure);
                break;

            default:
                errx(EX_SOFTWARE, "unexpected request_failed value");
            }
        } else {
            warnx("curl_multi_info_read: %s",
                curl_easy_strerror(msg->data.result)
            );
            cancel = report_failure(job, dl_fetch_failure);
        }

        job_delete(dl, job);
        if (cancel)
            return -1;
    }

    return 0;
}

int download_perform(download_t *dl)
{
    int still_running;

    for (;;) {
        CURLMcode mc;

        mc = curl_multi_perform(dl->curlm, &still_running);
        if (mc) {
            warnx("curl_multi_perform failure: %s", curl_multi_strerror(mc));
            return -1;
        }

        if (flush_ready(dl)) {
            say("further transfers aborted");
            break;  /* canceled by delivery callback */
        }

        if (!still_running) {
            say("no more transfers are running");
            break;
        }

        mc = curl_multi_poll(dl->curlm, NULL, 0, 128, NULL);
        if (mc) {
            warnx("curl_multi_poll failure: %s", curl_multi_strerror(mc));
            return -1;
        }
    }

    return 0;
}

void download_del(download_t *dl)
{
    if (!dl)
        return;

    if (dl->jobs) {
        int interrupted = 0;
        job_t *job, *tmp;

        DL_FOREACH_SAFE(dl->jobs, job, tmp) {
            /* Deliver failure for all aborted downloads. */
            job->callback(job->opaque, dl_aborted, NULL);
            job_delete(dl, job);
            ++interrupted;
        }

        if (interrupted)
            say("%d download jobs were interrupted", interrupted);
    }

    curl_multi_cleanup(dl->curlm);
    if (dl->global_init)
        curl_global_cleanup();

    free(dl);
}
