/*
 * This file is part of Crossbow.
 *
 * Crossbow is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation,
 * either version 3 of the License, or (at your option) any
 * later version.
 *
 * Crossbow is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with Crossbow.  If not, see
 * <https://www.gnu.org/licenses/>.
 */

#include <assert.h>
#include <err.h>
#include <errno.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>
#include <unistd.h>

#include <mrss.h>
#include <uthash.h>

#include "config.h"
#include "download.h"
#include "hash.h"
#include "help_helper.h"
#include "logging.h"
#include "outfmt.h"
#include "placeholders.h"
#include "savefile_aux.h"
#include "text.h"
#include "str.h"
#include "subcmd_common.h"

typedef struct
{
    const char *uid;
    unsigned short max_dload_jobs;
    bool dry_mark : 1;
    bool dry_exec : 1;
    bool catch_up : 1;
} opts_t;

typedef struct
{
    opts_t opts;
    savefile_t *savefile;
    download_t *dload;
    ofmt_t ofmt;
    bool ofmt_loaded;

    struct {
        unsigned failed_feeds;
    } counters;
} global_ctx_t;

typedef struct
{
    global_ctx_t *gctx;
    feed_t *feed;
    const char *url;
} fetch_ctx_t;

typedef struct
{
    feed_t *feed;
    mrss_item_t *item;
    str_t *item_uid;
    const opts_t *opts;

    struct {
        unsigned *incremental_id;
        const char *feed_title;
        const char *feed_identifier;
    } extra_pholders;
} process_ctx_t;

static const help_flag_t flags[] = {
    {
        .optname = "-c",
        .description="catch up feeds, skip processing entirely",
    }, {
        .optname = "-D",
        .description="do not run subprocesses",
    }, {
        .optname = "-d",
        .description="do not mark items as seen",
    }, {
        .optname = "-j",
        .argname = "JOBS",
        .description="number of simultaneous download jobs"
    }, {
        .optname = "-i",
        .argname = "ID",
        .description="only fetch the given identifier"
    }, {}
};

static const struct help help = {
    .progname = "crossbow-fetch",
    .flags = flags,
};

static inline void usage(const char *progname, const char *warning)
{
    if (warning)
        warnx("%s", warning);
    fprintf(stderr, "Usage: %s [OPTIONS]\n", progname);
}

static int str_to_ushort(const char *str, unsigned short *out)
{
    char *endptr;
    unsigned long result;

    errno = 0;
    result = strtoul(str, &endptr, 10);

    if (errno)
        warn("invalid short value: '%s'", str);
    else if (*endptr != '\0')
        warnx("invalid short value: '%s'", str);
    else if (result > USHRT_MAX)
        warnx("value too big: %lu, max is %hu", result, USHRT_MAX);
    else {
        *out = result;
        return 0;
    }

    return -1;
}

static opts_t read_opts(int argc, char **argv)
{
    int opt;

    opts_t result = {
        .max_dload_jobs = 10,
    };

    while (opt = getopt(argc, argv, "cDdi:j:vh"), opt != -1)
        switch (opt) {
        case 'c':
            result.catch_up = true;
            break;

        case 'D':
            result.dry_exec = true;
            break;

        case 'd':
            result.dry_mark = true;
            break;

        case 'j':
            if (str_to_ushort(optarg, &result.max_dload_jobs))
                goto fail;
            break;

        case 'i':
            result.uid = optarg;
            break;

        case 'h':
            show_help(help_full, &help, NULL);

        case 'v':
            g_verbosity_level++;
            break;

        default:
            goto fail;
        }

    if (!result.uid && optind < argc)
        result.uid = argv[optind++];

    for (int i = optind; i < argc; ++i)
        warnx("ignoring argument: %s", argv[i]);

    return result;

fail:
    usage(argv[0], NULL);
    exit(EXIT_FAILURE);
}

static fetch_ctx_t *fetch_ctx_init(fetch_ctx_t *fctx, global_ctx_t *gctx, feed_t *feed)
{
    str_t url;
    const char *urlcpy;

    url = feed_get_provided_url(feed);
    urlcpy = strndup(url.bytes, url.len);

    if (!urlcpy) {
        warn("strndupa");
        return NULL;
    }

    if (!fctx) {
        fctx = malloc(sizeof(fetch_ctx_t));
        if (!fctx) {
            warn("malloc");
            free((void *)urlcpy);
            return NULL;
        }
    }

    *fctx = (fetch_ctx_t){
        .gctx = gctx,
        .feed = feed,
        .url = urlcpy,
    };
    return fctx;
}

static void fetch_ctx_free(fetch_ctx_t *fctx)
{
    free((void *)fctx->url);
}

static void rss_show_error(const fetch_ctx_t *fctx, mrss_error_t e)
{
    const char *reason;
    const char *kind;

    switch (e) {
        case MRSS_OK:
        case MRSS_ERR_DOWNLOAD:
            errx(EX_SOFTWARE, "unexpected mrss_error_t");
        case MRSS_ERR_POSIX:
            reason = strerror(errno);
            kind = "errno";
            break;
        default:
            reason = mrss_strerror(e);
            kind = "mrss";
    }

    str_t url = feed_get_provided_url(fctx->feed);
    warnx("cannot open feed [%.*s]: error_code=%d (%s: %s)",
          STR_FMT(&url), e, kind, reason);
}

static mrss_t *rss_open(const fetch_ctx_t *fctx, const char *file_path)
{
    mrss_t *rss;
    mrss_error_t error;

    /* mrss_parse_file requires `char *`.  Old style software, I guess. */
    error = mrss_parse_file((char *)file_path, &rss);
    if (!error)
        return rss;

    rss_show_error(fctx, error);
    return NULL;
}

static int load_seen(const feed_t *feed, hash_t *seen)
{
    say(" loading items from savefile");

    feed_items_t items = feed_get_items(feed);
    for (unsigned i = 0; i < items.n_items; ++i) {
        whisper("  known item [%.*s]", STR_FMT(&items.items[i]));
        if (hash_insert(seen, &STR_ALIAS(&items.items[i]), NULL) == -1)
            return -1;
    }
    whisper(" done loading");

    return 0;
}

static int process_item_print(const process_ctx_t *pc)
{
    char buffer[256];
    mrss_item_t *item = pc->item;

    printf("ITEM: %.*s\n", STR_FMT(pc->item_uid));
    printf(" incremental_id: %u\n", *(pc->extra_pholders.incremental_id));

    #define print_if(fmt, item) if (item) printf(fmt, item)
    print_if(" title: %s\n", item->title);
    print_if(" title_type: %s\n", item->title_type);
    print_if(" link: %s\n", item->link);
    print_if(" description: %s\n", text_short(
        buffer,
        sizeof(buffer),
        item->description,
        NULL
    ));
    print_if(" description_type: %s\n", item->description_type);
    print_if(" copyright: %s\n", item->copyright);
    print_if(" copyright_type: %s\n", item->copyright_type);
    print_if(" author: %s\n", item->author);
    print_if(" author_url: %s\n", item->author_uri);
    print_if(" author_email: %s\n", item->author_email);
    print_if(" contributor: %s\n", item->contributor);
    print_if(" contributor_uri: %s\n", item->contributor_uri);
    print_if(" contributor_email: %s\n", item->contributor_email);
    print_if(" comments: %s\n", item->comments);
    print_if(" pubDate: %s\n", item->pubDate);
    print_if(" guid: %s\n", item->guid);
    print_if(" guid_isPermaLink: %d\n", item->guid_isPermaLink);
    print_if(" source: %s\n", item->source);
    print_if(" source_url: %s\n", item->source_url);
    print_if(" enclosure: %s\n", item->enclosure);
    print_if(" enclosure_url: %s\n", item->enclosure_url);
    print_if(" enclosure_length: %d\n", item->enclosure_length);
    print_if(" enclosure_type: %s\n", item->enclosure_type);
    #undef print_if

    return 0;   /* never fails */
}

static inline int process_item_ofmt(const global_ctx_t *gctx,
                                    const process_ctx_t *pc)
{
    ofmt_evaluate_params_t params = {
        .item = (void *)pc->item,
        .dry_run = pc->opts->dry_exec,
    };
    char path[PATH_MAX];
    str_t subproc_chdir = feed_get_subproc_chdir(pc->feed);

    if (subproc_chdir.len > 0) {
        if (str_to_cstr(&subproc_chdir, path, sizeof(path)) == NULL) {
            warnx("cannot chdir to \"%.*s\": path too long",
                  STR_FMT(&subproc_chdir));
            return -1;
        }
        params.opt_subproc_chdir = path;
    }

    placeholders_set_extra(gctx->ofmt, &(const placeholder_extra_t){
        .incremental_id = *(pc->extra_pholders.incremental_id),
        .feed_identifier = pc->extra_pholders.feed_identifier,
        .feed_title = pc->extra_pholders.feed_title,
    });

    if (ofmt_evaluate(gctx->ofmt, &params) == 0)
        return 0;

    if (whisper_enabled)
        ofmt_print_error(gctx->ofmt);

    return -1;
}

static int process_item(const global_ctx_t *gctx,
                        const process_ctx_t *pc)
{
    int e = gctx->ofmt_loaded
        ? process_item_ofmt(gctx, pc)
        : process_item_print(pc);

    if (e)
        warnx("failure while fetching feed [%s]",
            feed_get_name(pc->feed)
        );

    /* Note: the value is purposely incremented even if the processing
     * failed. */
    ++(*(pc->extra_pholders.incremental_id));

    return e;
}

static void load_outfmt(fetch_ctx_t *fctx)
{
    global_ctx_t *gctx = fctx->gctx;
    bool ofmt_loaded = false;
    ofmt_mode_t om;

    str_t spec = feed_get_outfmt(fctx->feed);
    if (!spec.bytes)
        goto exit;

    if (feed_get_output_mode(fctx->feed) == feed_om_print)
        goto exit;

    if (convert_out_mode(feed_get_output_mode(fctx->feed), &om) == -1)
        goto exit;

    if (ofmt_compile(gctx->ofmt, om, spec.bytes, spec.len) == 0) {
        ofmt_loaded = true;
    } else {
        warnx("unable to compile format specification \"%.*s\"",
              STR_FMT(&spec));
        ofmt_print_error(gctx->ofmt);
        warnx("using default output format");
    }

exit:
    gctx->ofmt_loaded = ofmt_loaded;
}

static int feed_fetch_local(fetch_ctx_t *fctx, const char *file_path)
{
    enum {
        critical,
        success,
        feed_failure,
    } status = success;
    const global_ctx_t *gctx = fctx->gctx;
    hash_t seen_items = HASH_INIT;
    str_t store_items_buffer[feed_max_items] = {};
    feed_items_t store_items = {
        .items = store_items_buffer,
    };

    say("feed [%s]", feed_get_name(fctx->feed));
    mrss_t *mrss = rss_open(fctx, file_path);
    if (!mrss) {
        status = feed_failure;
        goto exit;
    }

    if (load_seen(fctx->feed, &seen_items) == -1) {
        status = critical;
        goto exit;
    }

    whisper(" processing feed");

    if (!gctx->opts.catch_up)
        load_outfmt(fctx);

    unsigned incremental_id = feed_get_items_count(fctx->feed);
    for (mrss_item_t *item = mrss->item; item; item = item->next) {
        str_t uid = STR_CAST(item->guid ?: item->link);
        if (!uid.bytes) {
            warnx("  ignored item, bad feed missing both guid and link");
            continue;
        }

        if (hash_contains(&seen_items, &uid))
            whisper("  item [%.*s] seen before", STR_FMT(&uid));
        else {
            whisper("  item [%.*s] is new", STR_FMT(&uid));

            if (gctx->opts.catch_up) {
                whisper("  marking item as seen (catch-up mode)");
                ++incremental_id;
            } else {
                process_ctx_t process_ctx = {
                    .feed = fctx->feed,
                    .item = item,
                    .item_uid = &uid,
                    .opts = &gctx->opts,
                    .extra_pholders = {
                        .incremental_id = &incremental_id,
                        .feed_title = mrss->title,
                        .feed_identifier = feed_get_name(fctx->feed),
                    },
                };

                if (process_item(gctx, &process_ctx) == -1) {
                    whisper("  processing failed, not marked as seen");
                    status = feed_failure;
                    continue;
                }
            }
        }

        store_items_buffer[store_items.n_items++] = uid;
    }

    if (incremental_id > feed_get_items_count(fctx->feed)) {
        feed_set_items_count(fctx->feed, incremental_id);

        /* having a higher incremental id means that at least one new item
         * was emitted, possibly with success */
        if (feed_set_items(fctx->feed, &store_items))
            status = feed_failure;
    }

    whisper(" done processing feed");

exit:
    hash_free(&seen_items);
    mrss_free(mrss);

    if (gctx->opts.dry_mark)
        whisper("-d mode: updates not persisted");
    else if (feed_persist(fctx->feed) == -1) {
        warnx("unable to persist feed");
        status = feed_failure;
    }

    if (status == feed_failure)
        fctx->gctx->counters.failed_feeds++;
    fetch_ctx_free(fctx);
    return status == critical ? -1 : 0;
}

static int on_downloaded(void *opaque,
                         download_status_t status,
                         const char *filename)
{
    fetch_ctx_t *fctx = opaque;
    int err = 0;

    switch (status) {
    case dl_complete:
        err = feed_fetch_local(fctx, filename);
        break;

    case dl_failed:
        fctx->gctx->counters.failed_feeds++;
        fetch_ctx_free(fctx);
        break;

    case dl_aborted:
        say("feed download aborted (%s)", fctx->url);
        fetch_ctx_free(fctx);
        break;

    }

    /* The objects passed around in the downloader are dynamically
     * allocated. */
    free(fctx);

    /* Only (err == -1) is going to interrupt all the downloads. */
    return err;
}

static int feed_fetch(global_ctx_t *gctx, feed_t *feed)
{
    switch(feed_get_url_type(feed)) {
    case feed_ut_remote:
        {
            fetch_ctx_t *fctx_p;

            fctx_p = fetch_ctx_init(NULL, gctx, feed);
            if (!fctx_p)
                return -1;

            whisper("scheduling download of [%s]", feed_get_name(feed));
            return download_schedule(
                gctx->dload,
                fctx_p->url,
                on_downloaded,
                fctx_p
            );
        }

    case feed_ut_local:
        {
            fetch_ctx_t fctx;

            if (!fetch_ctx_init(&fctx, gctx, feed))
                return -1;

            return feed_fetch_local(&fctx, fctx.url);
        }

    case feed_ut_unknown:
        {
            const str_t url = feed_get_provided_url(feed);
            warnx("Unknown URL type for \"%.*s\"", STR_FMT(&url));
            return -1;
        }
    }

    errx(EX_SOFTWARE, "unexpected feed_url_type_t");
}

static int fetch(global_ctx_t *gctx)
{
    feed_t *feed;

    if (gctx->opts.uid) {
        /* By calling savefile_get_feed, we pre-load the feed matching the
         * identifier.  The iteration via savefile_iter_feeds will extract
         * the one feed we loaded.
         */
        feed = savefile_get_feed(gctx->savefile, gctx->opts.uid);
        if (!feed)
            return -1; // FIXME: print error, factorize with crossbow-set

        if (!feed_exists(feed)) {
            warnx("feed %s does not exist", gctx->opts.uid);
            return -1;
        }

        if (feed_fetch(gctx, feed))
            return -1;
    }
    else {
        void *aux = NULL;

        /* Invoking savefile_load_all loads the save file of each feed, which
         * is a required step in order for savefile_iter_feeds to effectively
         * loop over all of them. */
        if (savefile_load_all(gctx->savefile) == -1) {
            warnx("faulty savefile load");
            return -1;
        }

        while (feed = savefile_iter_feeds(gctx->savefile, &aux), feed)
            if (feed_fetch(gctx, feed))
                return -1;
    }

    if (download_perform(gctx->dload))
        return -1;

    return gctx->counters.failed_feeds > 0 ? -1 : 0;
}

int main(int argc, char **argv)
{
    bool fail = true;
    global_ctx_t gctx = {
        .opts = read_opts(argc, argv),
    };

#ifdef HAVE_PLEDGE
    if (pledge("stdio rpath wpath cpath inet dns proc exec", NULL) == -1)
        err(1, "pledge");
#endif

    gctx.ofmt = ofmt_new();
    if (!gctx.ofmt) {
        warn("ofmt_new");
        goto exit;
    }

    gctx.dload = download_new(true, gctx.opts.max_dload_jobs);
    if (!gctx.dload)
        goto exit;

    if (placeholders_setup(gctx.ofmt) == -1)
        goto exit;

    gctx.savefile = open_savefile(false);
    if (!gctx.savefile) {
        warnx("open_savefile failed");
        goto exit;
    }

    fail = fetch(&gctx);

exit:
    savefile_free(gctx.savefile);
    download_del(gctx.dload);
    ofmt_del(gctx.ofmt);
    return fail ? EXIT_FAILURE : EXIT_SUCCESS;
}
