/*
 * This file is part of Crossbow.
 *
 * Crossbow is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation,
 * either version 3 of the License, or (at your option) any
 * later version.
 *
 * Crossbow is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with Crossbow.  If not, see
 * <https://www.gnu.org/licenses/>.
 */

#include "help_helper.h"
#include "logging.h"
#include "subcmd_common.h"

#include <err.h>
#include <stdlib.h>
#include <unistd.h>
#include <sysexits.h>

/* With this trick the enumerative is populated with all the
 * supported field names, prefixed with "fld_".
 * fld_GUARD is the guard value, that can be used for checks
 * and array dimensioning.
 */
typedef enum {
    #define PLACEHOLDER(key, field, ...) fld_##field,
    #include "placeholders_gen.h"
    #undef PLACEHOLDER
    fld_GUARD
} field_t;

struct opts
{
    const char *outfmt;
    ofmt_mode_t ofmt_mode;
    const char *values[fld_GUARD];
};

typedef struct opts opts_t;

static const struct help_flag help_flags[] = {
    {
        .optname = "-d",
        .argname = "FIELD=VALUE",
        .description = "value assignment",
    }, {
        .optname = "-o",
        .argname = "MODE",
        .description = "output mode (pipe, pretty, subproc)",
    }, {
        .optname = "-f",
        .argname = "FORMAT",
        .description = "output format",
    }, {}
};

static const struct help help = {
    .progname = "crossbow-outfmt",
    .flags = help_flags,
};

static inline ofmt_mode_t parse_ofmt_mode(const char *optarg)
{
    static const struct enum_keyval options[] = {
        {"pipe",    ofmt_mode_pipe},
        {"pretty",  ofmt_mode_print},
        {"subproc", ofmt_mode_subprocess},
        {}
    };

    return parse_enum('o', options, optarg);
}

static void read_subopts(char *optarg, opts_t *opts)
{
    static struct enum_keyval enum_keyval[] = {
        #define PLACEHOLDER(key, field, ...) \
            { .atom = #field, .enum_value = fld_##field },
        #include "placeholders_gen.h"
        #undef PLACEHOLDER
        {}
    };

    const char *value = NULL;
    field_t f;

    for (int i = 0; optarg[i]; ++i)
        if (optarg[i] == '=') {
            optarg[i] = '\0';
            value = optarg + i + 1;
            break;
        }

    if (!value)
        err(EX_USAGE, "invalid assignment: \"%s\"", optarg);

    f = parse_enum('d', enum_keyval, optarg);
    opts->values[f] = value;
}

static void read_opts(int argc, char **argv, opts_t *opts)
{
    int opt;

    while (opt = getopt(argc, argv, "d:o:f:vh"), opt != -1) {
        switch (opt) {

        case 'd':
            read_subopts(optarg, opts);
            break;

        case 'o':
            opts->ofmt_mode = parse_ofmt_mode(optarg);
            break;

        case 'f':
            opts->outfmt = optarg;
            break;

        case 'h':
            show_help(help_full, &help, NULL);

        case 'v':
            g_verbosity_level++;
            break;

        default:
            show_help(help_usage, &help, NULL);
        }
    }
}

#define PLACEHOLDER(key, field, ...) \
    static const char * resolve_ ## field (void *item, void *opaque)  \
    {                                                                 \
        return ((const opts_t *)item)->values[fld_##field];      \
    }
#include "placeholders_gen.h"
#undef PLACEHOLDER

static int mock_placeholders_setup(ofmt_t ofmt)
{
    #define PLACEHOLDER(key, field, ...) \
        if (ofmt_set_resolver(ofmt, key, resolve_ ## field)) \
            goto fail;
    #include "placeholders_gen.h"
    #undef PLACEHOLDER

    if (ofmt_set_pipe_resolver(ofmt, resolve_description))
        goto fail;

    return 0;

fail:
    ofmt_print_error(ofmt);
    return -1;
}

static void complete_opts(opts_t *opts)
{
    if (opts->outfmt)
        return;

    switch (opts->ofmt_mode) {
        case ofmt_mode_print:
            opts->outfmt = (
                #define PLACEHOLDER(key, field, ...) \
                    "\\%" key " %" key "\n"
                #include "placeholders_gen.h"
                #undef PLACEHOLDER
            );
            break;

        case ofmt_mode_subprocess:
            opts->outfmt = (
                "printf \\%s\\ \\%s\\n"
                #define PLACEHOLDER(key, field, ...) \
                    " \\%" key " %" key
                #include "placeholders_gen.h"
                #undef PLACEHOLDER
            );
            break;

        case ofmt_mode_pipe:
            opts->outfmt = "cat";
            break;
    }

    static const char * mode_names[] = {
        [ofmt_mode_pipe] = "pipe",
        [ofmt_mode_print] = "pretty",
        [ofmt_mode_subprocess] = "subproc",
    };

    say("NOTE: default output format for mode \"%s\"",
        mode_names[opts->ofmt_mode]);
    say("[[%s]]", opts->outfmt);
}

static int run(const opts_t *opts)
{
    int e = EXIT_FAILURE;
    ofmt_t ofmt = NULL;

    ofmt = ofmt_new();
    if (!ofmt) {
        warn("ofmt_new");
        goto exit;
    }

    if (mock_placeholders_setup(ofmt))
        goto exit;

    if (ofmt_compile(ofmt,
                     opts->ofmt_mode,
                     opts->outfmt,
                     strlen(opts->outfmt))) {
        ofmt_print_error(ofmt);
        e = EX_DATAERR;
        goto exit;
    }

    if (ofmt_evaluate(ofmt, (void *)opts)) {
        ofmt_print_error(ofmt);
        e = EX_SOFTWARE;
        goto exit;
    }

    e = EXIT_SUCCESS;
exit:
    ofmt_del(ofmt);
    return e;
}

int main(int argc, char **argv)
{
    opts_t opts = {
        .ofmt_mode = ofmt_mode_print,

        #define PLACEHOLDER(key, field, ...) \
            .values[fld_##field] = #field,
        #include "placeholders_gen.h"
        #undef PLACEHOLDER
    };

    read_opts(argc, argv, &opts);
    complete_opts(&opts);
    return run(&opts);
}
