/* vortex.c 15.1 06/08/90 10:07:57 */

/* Original code by Daryl Poe.
 *
 * Copyright (c) 1989 Daryl Poe
 *
 * Ported to X11 by Jim Andreas.
 * Tractors, treaties, and other features by Norm Gee.
 * Damage window and shield bitmaps by Tom LaStrange.
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting
 * documentation.  No representations are made about the
 * suitability of this software for any purpose.  It is
 * provided "as is" without express or implied warranty.
 */

#include <stdio.h>
#include "defs.h"
#include "weapon.h"
#include "system.h"
#include "ship.h"
#include "stats.h"
#include "player.h"
#include "torp.h"
#include "status.h"
#include "planet.h"
#include "phaser.h"
#include "message.h"
#include "shmem.h"
#include "data.h"

#define MAX_SINGLE_CASE 59
#define MAX_EFFECT (MAX_SINGLE_CASE+55)
#define PV_EXPLODE_DIST 10600

static void teleport_player(player,x,y)
PLAYER *player;
int x,y;
{
	/* randomize a bit */
	x += (GWIDTH/40 - INTRAND(GWIDTH/20));
	y += (GWIDTH/40 - INTRAND(GWIDTH/20));

	if (x < 10) x = 10;
	else if (x > (GWIDTH-10)) x = (GWIDTH-10);
	if (y < 10) y = 10;
	else if (y > (GWIDTH-10)) y = (GWIDTH-10);

	player->p_flags &= ~(PFBOMB|PFORBIT|PFBEAMUP|PFBEAMDOWN);
	player->p_x = x;
	player->p_y = y;
	player->p_app_x = x + player->p_ecm_dx;
	player->p_app_y = y + player->p_ecm_dy;
	player->p_dir = player->p_desdir = (COURSE) INTRAND(256);
	player->p_desspeed = 0;
}


static void hit_player_with_vortex(player,x,y,effect)
PLAYER *player;
int x,y,effect;
{
	SHIP *ship;
	char msg[MSG_STRLEN],str[MSG_STRLEN];
	int i,delta;

	ship = &player->p_ship;

	logmsg5("Vortex affects player %d at %d,%d; effect %d\n",
		player->p_no,x,y,effect);

	if (effect < MAX_SINGLE_CASE) {
		switch (effect) {
			case 0:
			case 1:
				ship->s_torpspeed = ship->s_torpspeed * 4 / 3;
				sprintf(msg,"Torp Speed increased to %d",
					ship->s_torpspeed);
				break;
			case 2:
				ship->s_torpspeed = ship->s_torpspeed * 3 / 4;
				sprintf(msg,"Torp Speed decreased to %d",
					ship->s_torpspeed);
				break;
			case 3:
			case 4:
				ship->s_torpdamage = ship->s_torpdamage * 4 / 3;
				compute_fields(ship);
				sprintf(msg,"Torp Damage increased to %d",
					ship->s_torpdamage);
				break;
			case 5:
				ship->s_torpdamage = ship->s_torpdamage * 3 / 4;
				compute_fields(ship);
				sprintf(msg,"Torp Damage decreased to %d",
					ship->s_torpdamage);
				break;
			case 6:
			case 7:
				ship->s_beamrange = ship->s_beamrange * 4 / 3;
				compute_fields(ship);
				sprintf(msg,"Beam Range increased to %d",
					ship->s_beamrange);
				break;
			case 9:
				ship->s_beamrange = ship->s_beamrange * 3 / 4;
				compute_fields(ship);
				sprintf(msg,"Beam Range decreased to %d",
					ship->s_beamrange);
				break;
			case 10:
				ship->s_beamdamage = ship->s_beamdamage * 4 / 3;
				compute_fields(ship);
				sprintf(msg,"Beam Damage increased to %d",
					ship->s_beamdamage);
				break;
			case 11:
				ship->s_beamdamage = ship->s_beamdamage * 3 / 4;
				compute_fields(ship);
				sprintf(msg,"Beam Damage decreased to %d",
					ship->s_beamdamage);
				break;
			case 12:
			case 13:
				if ((ship->s_maxspeed += INTRAND(2) + 1) > MAXSPEED) {
					ship->s_maxspeed = MAXSPEED;
				}
				player->p_stats_dirty = TRUE;
				sprintf(msg,"Max Warp increased to %d",
					ship->s_maxspeed);
				break;
			case 14:
				if ((ship->s_maxspeed -= (INTRAND(2) + 1)) < 0) {
					ship->s_maxspeed = 0;
				}
				player->p_stats_dirty = TRUE;
				sprintf(msg,"Max Warp decreased to %d",
					ship->s_maxspeed);
				break;
			case 15:
			case 16:
				if ((--ship->s_warpcost) < 0) ship->s_warpcost = 0;
				sprintf(msg,"Warp Fuel cost reduced.");
				break;
			case 17:
				++ship->s_warpcost;
				sprintf(msg,"Warp Fuel cost increased.");
				break;
			case 18:
			case 19:
				ship->s_turns = ship->s_turns * 4 / 3;
				sprintf(msg,"Maneuverability increased 33%%");
				break;
			case 20:
				ship->s_turns = ship->s_turns * 3 / 4;
				sprintf(msg,"Maneuverability decreased 33%%");
				break;
			case 21:
			case 22:
				ship->s_accint = ship->s_accint * 4 / 3;
				sprintf(msg,"Acceleration increased 33%%");
				break;
			case 23:
				ship->s_accint = ship->s_accint * 3 / 4;
				sprintf(msg,"Acceleration decreased 33%%");
				break;
			case 24:
			case 25:
				ship->s_decint = ship->s_decint * 4 / 3;
				sprintf(msg,"Deceleration increased 33%%");
				break;
			case 26:
				ship->s_decint = ship->s_decint * 3 / 4;
				sprintf(msg,"Deceleration decreased 33%%");
				break;
			case 27:
			case 28:
				++ship->s_recharge;
				sprintf(msg,"Dilithium effectiveness increased.");
				break;
			case 29:
				--ship->s_recharge;
				sprintf(msg,"Dilithium effectiveness decreased.");
				break;
			case 30:
			case 31:
				ship->s_repair = ship->s_repair * 4 / 3;
				for (i=0; i<ship->s_numsys; ++i) {
					delta = ship->s_system[i].repair_rate / 3;
					ship->s_system[i].repair_rate        += delta;
					ship->s_system[i].planet_repair_rate += delta;
				}
				sprintf(msg,"Repair increased 33%%");
				break;
			case 32:
				ship->s_repair = ship->s_repair * 3 / 4;
				for (i=0; i<ship->s_numsys; ++i) {
					delta = ship->s_system[i].repair_rate / 4;
					ship->s_system[i].repair_rate        -= delta;
					ship->s_system[i].planet_repair_rate -= delta;
				}
				sprintf(msg,"Repair decreased 33%%");
				break;
			case 33:
			case 34:
				ship->s_cargo = ship->s_cargo * 4 / 3;
				player->p_damage_dirty = TRUE;
				sprintf(msg,"Cargo increased to %d",
					ship->s_cargo);
				break;
			case 35:
				ship->s_cargo = ship->s_cargo * 3 / 4;
				player->p_damage_dirty = TRUE;
				sprintf(msg,"Cargo decreased to %d",
					ship->s_cargo);
				break;
			case 36:
				ship->s_shield[FRONT_SHIELD] =
					ship->s_shield[FRONT_SHIELD] * 4 / 3;
				player->p_stats_dirty = TRUE;
				sprintf(msg,"Front shield strength increased to %d",
					ship->s_shield[FRONT_SHIELD]);
				break;
			case 37:
				ship->s_shield[FRONT_SHIELD] = 
					ship->s_shield[FRONT_SHIELD] * 3 / 4;
				player->p_stats_dirty = TRUE;
				sprintf(msg,"Front shield strength decreased to %d",
					ship->s_shield[FRONT_SHIELD]);
				break;
			case 38:
				ship->s_shield[RIGHT_SHIELD] =
					ship->s_shield[RIGHT_SHIELD] * 4 / 3;
				player->p_stats_dirty = TRUE;
				sprintf(msg,"Right shield strength increased to %d",
					ship->s_shield[RIGHT_SHIELD]);
				break;
			case 39:
				ship->s_shield[RIGHT_SHIELD] =
					ship->s_shield[RIGHT_SHIELD] * 3 / 4;
				player->p_stats_dirty = TRUE;
				sprintf(msg,"Right shield strength decreased to %d",
					ship->s_shield[RIGHT_SHIELD]);
				break;
			case 40:
				ship->s_shield[REAR_SHIELD] =
					ship->s_shield[REAR_SHIELD] * 4 / 3;
				player->p_stats_dirty = TRUE;
				sprintf(msg,"Rear shield strength increased to %d",
					ship->s_shield[REAR_SHIELD]);
				break;
			case 41:
				ship->s_shield[REAR_SHIELD] =
					ship->s_shield[REAR_SHIELD] * 3 / 4;
				player->p_stats_dirty = TRUE;
				sprintf(msg,"Rear shield strength decreased to %d",
					ship->s_shield[REAR_SHIELD]);
				break;
			case 42:
				ship->s_shield[LEFT_SHIELD] =
					ship->s_shield[LEFT_SHIELD] * 4 / 3;
				player->p_stats_dirty = TRUE;
				sprintf(msg,"Left shield strength increased to %d",
					ship->s_shield[LEFT_SHIELD]);
				break;
			case 43:
				ship->s_shield[LEFT_SHIELD] =
					ship->s_shield[LEFT_SHIELD] * 3 / 4;
				player->p_stats_dirty = TRUE;
				sprintf(msg,"Left shield strength decreased to %d",
					ship->s_shield[LEFT_SHIELD]);
				break;
			case 44:
			case 45:
				ship->s_armor += 100;
				ship->s_system[0].current_size   += 100;
				ship->s_system[0].undamaged_size += 100;
				player->p_damage_dirty = TRUE;
				sprintf(msg,"Armor strengthened by 100");
				break;
			case 46:
				if ((ship->s_armor -= 50) <= 1) {
					ship->s_armor = 1;
					ship->s_system[0].current_size   = 0;
					ship->s_system[0].undamaged_size = 1;
					sprintf(msg,"Armor destroyed");
				}
				else {
					ship->s_system[0].current_size   -= 50;
					ship->s_system[0].undamaged_size -= 50;
					sprintf(msg,"Armor weakened by 50");
				}
				player->p_damage_dirty = TRUE;
				break;
			case 47:
				ship->s_hull_str = ship->s_hull_str * 4 / 3;
				ship->s_system[1].current_size =
					ship->s_system[1].current_size * 4 / 3;
				ship->s_system[1].undamaged_size =
					ship->s_system[1].undamaged_size * 4 / 3;
				player->p_damage_dirty = TRUE;
				sprintf(msg,"Hull Strength increased by 33%%");
				break;
			case 48:
				ship->s_hull_str = ship->s_hull_str * 3 / 4;
				ship->s_system[1].current_size =
					ship->s_system[1].current_size * 3 / 4;
				ship->s_system[1].undamaged_size =
					ship->s_system[1].undamaged_size * 3 / 4;
				player->p_damage_dirty = TRUE;
				sprintf(msg,"Hull Strength decreased by 33%%");
				break;
			case 49:
			case 50:
				ship->s_maxfuel = ship->s_maxfuel * 4 / 3;
				player->p_stats_dirty = TRUE;
				player->p_damage_dirty = TRUE;
				sprintf(msg,"Fuel Capacity increased by 33%%");
				break;
			case 51:
				ship->s_maxfuel = ship->s_maxfuel * 3 / 4;
				player->p_stats_dirty = TRUE;
				player->p_damage_dirty = TRUE;
				sprintf(msg,"Fuel Capacity decreased by 33%%");
				break;
			case 52:
				/* teleport to nearest planet */
				{
					int mindist,plno;

					if ((plno = closest_planet_dist(
							player->p_x,player->p_y,
							&mindist,TARG_PLANET,0)) != -1) {
						teleport_player(player,
							planets[plno].pl_x,planets[plno].pl_y);
					}
					else {
						teleport_player(player,
							INTRAND(GWIDTH),INTRAND(GWIDTH));
					}
				}
				sprintf(msg,"We\'ve been thrown through a space warp.");
				break;
			case 53:
				/* teleport to nearest repair planet */
				{
					int mindist,plno;

					if ((plno = closest_planet_dist(
							player->p_x,player->p_y,
							&mindist,TARG_PLANET|TARG_PLFLAGS_AND,
							PLMASK_REPAIR)) != -1) {
						teleport_player(player,
							planets[plno].pl_x,planets[plno].pl_y);
					}
					else {
						teleport_player(player,
							INTRAND(GWIDTH),INTRAND(GWIDTH));
					}
				}
				sprintf(msg,"We\'ve been thrown through a space warp.");
				break;
			case 54:
				/* teleport to nearest fuel planet */
				{
					int mindist,plno;

					if ((plno = closest_planet_dist(
							player->p_x,player->p_y,
							&mindist,TARG_PLANET|TARG_PLFLAGS_AND,
							PLMASK_FUEL)) != -1) {
						teleport_player(player,
							planets[plno].pl_x,planets[plno].pl_y);
					}
					else {
						teleport_player(player,
							INTRAND(GWIDTH),INTRAND(GWIDTH));
					}
				}
				sprintf(msg,"We\'ve been thrown through a space warp.");
				break;
			case 55:
				/* teleport to nearest home planet */
				{
					int mindist,plno;

					if ((plno = closest_planet_dist(
							player->p_x,player->p_y,
							&mindist,TARG_PLANET|TARG_PLFLAGS_AND,
							PLMASK_HOME)) != -1) {
						teleport_player(player,
							planets[plno].pl_x,planets[plno].pl_y);
					}
					else {
						teleport_player(player,
							INTRAND(GWIDTH),INTRAND(GWIDTH));
					}
				}
				sprintf(msg,"We\'ve been thrown through a space warp.");
				break;
			case 56:
				/* teleport to nearest black hole */
				{
					int mindist,plno;

					if ((plno = closest_planet_dist(
							player->p_x,player->p_y,
							&mindist,TARG_PLANET|TARG_PLFLAGS_AND,
							PLMASK_BLACKHOLE)) != -1) {
						teleport_player(player,
							planets[plno].pl_x,planets[plno].pl_y);
					}
					else {
						teleport_player(player,
							INTRAND(GWIDTH),INTRAND(GWIDTH));
					}
				}
				sprintf(msg,"We\'ve been thrown through a space warp.");
				break;
			case 57:
				/* teleport to nearest pulsar */
				{
					int mindist,plno;

					if ((plno = closest_planet_dist(
							player->p_x,player->p_y,
							&mindist,TARG_PLANET|TARG_PLFLAGS_AND,
							PLMASK_PULSAR)) != -1) {
						teleport_player(player,
							planets[plno].pl_x,planets[plno].pl_y);
					}
					else {
						teleport_player(player,
							INTRAND(GWIDTH),INTRAND(GWIDTH));
					}
				}
				sprintf(msg,"We\'ve been thrown through a space warp.");
				break;
			case 58:
				/* teleport to nearest nebula */
				{
					int mindist,plno;

					if ((plno = closest_planet_dist(
							player->p_x,player->p_y,
							&mindist,TARG_PLANET|TARG_PLFLAGS_AND,
							PLMASK_NEBULA)) != -1) {
						teleport_player(player,
							planets[plno].pl_x,planets[plno].pl_y);
					}
					else {
						teleport_player(player,
							INTRAND(GWIDTH),INTRAND(GWIDTH));
					}
				}
				sprintf(msg,"We\'ve been thrown through a space warp.");
				break;
		}
	}
	else if (effect < (MAX_SINGLE_CASE+5)) {
		/* teleport the critter */
		teleport_player(player,INTRAND(GWIDTH),INTRAND(GWIDTH));
		sprintf(msg,"We\'ve been thrown through a space warp.");
	}
	else if (effect < (MAX_SINGLE_CASE+10)) {
		/* reduce shields to zero */
		player->p_shield[0] = 
			player->p_shield[1] = 
			player->p_shield[2] = 
			player->p_shield[3] = 
			0;
		sprintf(msg,"The shields have been drained.");
	}
	else if (effect < (MAX_SINGLE_CASE+15)) {
		/* double shields */
		player->p_shield[0] = ship->s_shield[0] * 2;
		player->p_shield[1] = ship->s_shield[1] * 2;
		player->p_shield[2] = ship->s_shield[2] * 2;
		player->p_shield[3] = ship->s_shield[3] * 2;
		sprintf(msg,"The shields have been doubled.");
	}
	else if (effect < (MAX_SINGLE_CASE+20)) {
		player->p_fuel = 0;
		sprintf(msg,"All fuel drained.");
	}
	else if (effect < (MAX_SINGLE_CASE+25)) {
		player->p_fuel = ship->s_maxfuel * 2;
		sprintf(msg,"Fuel doubled.");
	}
	else if (effect < (MAX_SINGLE_CASE+30)) {
		player->p_etemp = 1200;
		sprintf(msg,"Engines overheated.");
	}
	else if (effect < (MAX_SINGLE_CASE+40)) {
		int shield,damage;

		damage = 20 + INTRAND(180);
		sprintf(msg,"Vortex exploded for %d",damage);
		shield = shield_hit(Atan2(y - player->p_y,
			player->p_x - x),player->p_dir);
		do_damage(NULL,player,damage,WPN_PROB_VORTEX,shield);
	}
	else if (effect < (MAX_SINGLE_CASE+45)) {
		int damage;

		damage = 10 + INTRAND(50);
		sprintf(msg,"Vortex radiation caused %d damage",damage);
		do_damage(NULL,player,damage,WPN_PROB_VORTEX_NS,0);
	}
	else {
		/* teleport near another player */
		int pno,mindist;

		if ((pno = closest_player_dist(INTRAND(GWIDTH),INTRAND(GWIDTH),
				&mindist,TARG_PLAYER,0)) != -1) {
			teleport_player(player,players[pno].p_x,players[pno].p_y);
		}
		else {
			teleport_player(player,INTRAND(GWIDTH),INTRAND(GWIDTH));
		}
		sprintf(msg,"We\'ve been thrown through a space warp.");
	}

	/* Tell everyone he was hit */
	sprintf(str,"%s (%s) Hit by Probability Vortex",
		player->p_name, player->p_mapchars);
	pmessage(str,0,MALL,"Game->ALL");

	/* Tell him the effect */
	pmessage(msg,player->p_no,MINDIV,"Eng.->Cpt.");
}


void explode_vortex(player,torp)
PLAYER *player;
TORP *torp;
{
	int effect,dx,dy;

	logmsg1("VORTEX\n");
	effect = INTRAND(MAX_EFFECT);

#ifdef VORTEX_TESTING
	/******************
	{
		static char *numpath="/usr/games/lib/dtrek/dtrek.effect";
		FILE *numfile;

		if (access(numpath,04) == 0) {
			numfile = fopen(numpath,"r");
			fscanf(numfile,"%d",&effect);
			fclose(numfile);
		}
	}
	********************/
#endif

	if (player != NULL) {
		hit_player_with_vortex(player,torp->t_x,torp->t_y,effect);
	}
	else {
		/* hit everyone around */
		for (player = players; player < players+MAXPLAYER; ++player) {
			if (player->p_status != PALIVE) continue;
			if ((dx = ABS(player->p_x - torp->t_x)) > PV_EXPLODE_DIST)
				continue;
			if ((dy = ABS(player->p_y - torp->t_y)) > PV_EXPLODE_DIST)
				continue;
			if ((dx*dx + dy*dy) > (PV_EXPLODE_DIST*PV_EXPLODE_DIST))
				continue;
			hit_player_with_vortex(player,torp->t_x,torp->t_y,effect);
		}
	}

	torp->t_status = TEXPLODE;
	torp->t_fuse   = 10;
}

