/* shipmenu.c 15.1 06/08/90 10:07:09 */

/* Original code by Daryl Poe.
 * Widgetless and enhanced overhaul by Jonathon Lemon.
 * Smaller archive and anti-cheating by Daryl Poe
 *
 * Copyright (c) 1989 Daryl Poe
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting
 * documentation.  No representations are made about the
 * suitability of this software for any purpose.  It is
 * provided "as is" without express or implied warranty.
 */

#include <stdio.h>
#include <sys/time.h>
#include <sys/param.h>
#include <fcntl.h>
#include <pwd.h>
#include <ctype.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>
#include <X11/Xutil.h>
#include "defs.h"
#include "weapon.h"
#include "system.h"
#include "ship.h"
#include "stats.h"
#include "player.h"
#include "torp.h"
#include "status.h"
#include "planet.h"
#include "phaser.h"
#include "message.h"
#include "shmem.h"
#include "data.h"
#include "daemon.h"

#define ALL_HULLS		-1
#define USABLE_HULLS	-2

#define MAXSHIPS		200	/* maximum number of ships permitted */

#define NO_ACTION			0
#define USE_OLD_SHIP		1
#define EDIT_OLD_SHIP		2
#define VIEW_OLD_SHIP		3
#define	COPY_OLD_SHIP		4
#define DELETE_OLD_SHIP		5
#define CONSTRUCT_NEW_SHIP	6
#define GET_DEFAULT			7
#define SET_HULL_SIZE		8
#define REDO_OLD_SHIP		9	

#define NEW_VALUES_MASK		0x01
#define NEED_DVAL_MASK		0x02

#define LABELSIZE		20
#define UNITSIZE		2
#define LABELFORMAT		"%-*s %9g %6d%*s"

#define CURRENT_VERSION	9
#define BLANK_ENTRY 	(NUM_ADJ_SYS+1)
#define HEIGHT 			20		/* KLUDGE - height of all boxes */
#define WIDTH 			250
#define COLUMN1 		10
#define COLUMN2 		500
#define STARTROW 		10
#define YSPACING 		(HEIGHT+5)
#define XBTNSPACING 	(HEIGHT+5)
#define XSPACING 		(HEIGHT+15)

static char *_version = "@(#)15.1 06/08/90 shipbuilding program for dtrek";
FILE *logfile = NULL;

/* what order should these be displayed in?  */
static int display_order[2][(NUM_ADJ_SYS/2)+1] = {
{       TITLE_BAR,
        TORP_NUM,
        TORP_SPEED,
        TORP_DAMAGE,
        TORP_RELOAD,
        TORP_DURATION,
        BEAM_NUM,
        BEAM_RANGE,
        BEAM_DAMAGE,
        BEAM_ACCURACY,
        BEAM_RELOAD,
        TRACTOR_BEAM,
        ENGINE,
        MAX_SPEED,
        WARP_COST,
        MANEUVER,
        ACCELERATION,
        DECELERATION,
		},
		{
		TITLE_BAR,
        DILITHIUM,
        DAMAGE_CONTROL,
        CARGO,
        FRONT_SHIELD_SYS,
        SIDE_SHIELD_SYS,
        REAR_SHIELD_SYS,
        ARMOR,
        HULL,
        DAMAGE_ABS,
        CLOAK,
        ECM,
        IMP_SENS,
        HYPERSPACE,
        SPHERE,
        AFC,
        BLANK_ENTRY,
        FUEL,
		}
};


/* what order should these be printed in?  */
static int print_order[NUM_ADJ_SYS] = {
	TITLE_BAR,
	TORP_NUM,
	TORP_SPEED,
	TORP_DAMAGE,
	TORP_DURATION,
	TORP_RELOAD,
	BEAM_NUM,
	BEAM_RANGE,
	BEAM_DAMAGE,
	BEAM_ACCURACY,
	BEAM_RELOAD,
	CLOAK,
	IMP_SENS,
	ECM,
	HYPERSPACE,
	SPHERE,
	AFC,
	DAMAGE_ABS,
	TRACTOR_BEAM,
	ENGINE,
	MAX_SPEED,
	WARP_COST,
	MANEUVER,
	ACCELERATION,
	DECELERATION,
	DILITHIUM,
	DAMAGE_CONTROL,
	CARGO,
	FRONT_SHIELD_SYS,
	SIDE_SHIELD_SYS,
	REAR_SHIELD_SYS,
	ARMOR,
	HULL,
	FUEL
};

/* this is a list of special technology numbers and their masks */
typedef struct {
	int system_no,mask;
} STPAIR;
static STPAIR stlist[] = {
	{ CLOAK, STMASK_CLOAK },
	{ IMP_SENS, STMASK_IMP_SENS },
	{ ECM, STMASK_ECM },
	{ HYPERSPACE, STMASK_HYPERSPACE },
	{ SPHERE, STMASK_SPHERE },
	{ AFC, STMASK_AFC },
	{ DAMAGE_ABS, STMASK_DAMAGE_ABS }
};

typedef struct {
	int index;		/* which system is this, anyway? */
	int dval;		/* displayed value */
	float size;		/* how many units the system takes */
	float dsize;		/* displayed size value */
	int *field;		/* pointer to the field in ship structure */
	char label[24],		/* the item's label */
		 units[8];	/* in what units are the value, e.g. "%" */
	int min,		/* minimum "value" */
		max;		/* maximum "value" */
	Boolean	plus5_on,	/* is the plus5 pushbutton selectable? */
		plus_on,	/* is the plus  pushbutton selectable? */
		minus_on,	/* is the minus pushbutton selectable? */
		minus5_on;	/* is the minus5 pushbutton selectable? */
	int show_sys;	/* is this system even available? */
	int	modifiable;	/* is the system modifiable, (not figured)? */
	int repair_rate,	/* in 1/256's of a kton per turn */
		planet_repair_rate;
} MENU_ITEM;

static SHIP ship;
static int junk;
static char shiparchive[MAXPATHLEN];

typedef struct {
	int value[NUM_ADJ_SYS];
	int reserved[32];
	int team_number;
	int torpclass;
	int beamclass;
	int version_number;
	char shipname[SHIPNAMELEN];
} SAVED_STUFF;

SAVED_STUFF saved_stuff;
int *value = saved_stuff.value;

typedef struct {
	char filename[256];
	char short_shipname[10];
	int  team_no;
	int  hull_size;
} SHIPFILENAMESTRUCT;

static MENU_ITEM sys[NUM_ADJ_SYS] = {
	{ TORP_NUM, 0, 111.0, 111.0, &ship.s_torps,
		"111", "  ",
	 	0, MAXTORP, 
		True, True, False, False,
		TRUE, TRUE, 15, 45
	},
	{ TORP_SPEED, 111, 0.0, 111.0, &ship.s_torpspeed,
		"Torpedo Speed","  ",
	 	0, 111,
		True, True, False, False,
		TRUE, TRUE, 0, 0
	},
	{ TORP_DAMAGE, 111, 0.0, 111.0, &ship.s_torpdamage,
		"Torpedo Damage", "  ",
		0, 111,
		True, True, False, False,
		TRUE, TRUE, 0, 0
	},
	{ TORP_DURATION, 111, 0.0, 111.0, &ship.s_torpduration,
		"Torpedo Duration", " s",
		0, 111,
		True, True, False, False,
		TRUE, TRUE, 0, 0
	},
	{ TORP_RELOAD, 111, 0.0, 111.0, &ship.s_torpreload,
		"Torpedo Reload", " s",
		0, 111,
		True, True, False, False,
		TRUE, TRUE, 0, 0
	},
	{ BEAM_NUM, 0, 111.0, 111.0, &ship.s_beams,
		"111", "  ",
		0, MAXBEAM,
		True, True, False, False,
		TRUE, TRUE, 30, 90
	},
	{ BEAM_RANGE, 111, 0.0, 111.0, &ship.s_beamrange,
		"Beam Range", "  ",
		0, 111,
		True, True, False, False,
		TRUE, TRUE, 0, 0
	},
	{ BEAM_DAMAGE, 111, 0.0, 111.0, &ship.s_beamdamage,
		"Beam Damage", "  ",
		0, 111,
		True, True, False, False,
		TRUE, TRUE, 0, 0
	},
	{ BEAM_ACCURACY, 111, 0.0, 111.0, &ship.s_beamaccuracy,
		"Beam Accuracy", "  ",
		0, 111,
		True, True, False, False,
		TRUE, TRUE, 0, 0
	},
	{ BEAM_RELOAD, 111, 0.0, 111.0, &ship.s_beamreload,
		"Beam Reload", " s",
		0, 111,
		True, True, False, False,
		TRUE, TRUE, 0, 0
	},
	{ TRACTOR_BEAM, 0, 3.0, 3.0, &ship.s_tractor,
		"Tractor Beams", "  ",
		0, 4,
		True, True, False, False,
		TRUE, TRUE, 5, 10
	},
	{ ENGINE, 0, 1.0, 1.0, &ship.s_engines,
		"Engines", "  ",
		0, 1234,
		True, True, False, False,
		TRUE, TRUE, 25, 75
	},
	{ MAX_SPEED, 0, 0.0, 1.0, &ship.s_maxspeed,
		"Max Warp", "  ",
		0, MAXSPEED,
		True, True, False, False,
		TRUE, FALSE, 0, 0
	},
	{ WARP_COST, 0, 6.0, 6.0, &ship.s_warpcost,
		"Warp Fuel Cost", " %",
		0, 3,
		True, True, False, False,
		TRUE, TRUE, 0, 0
	},
	{ MANEUVER, 0, 1.0, 1.0, &ship.s_turns,
		"Maneuverability", " %",
		0, 1234,
		True, True, False, False,
		TRUE, TRUE, 0, 0
	},
	{ ACCELERATION, 0, 1.0, 1.0, &ship.s_accint,
		"Acceleration", " %",
		0, 1234,
		True, True, False, False,
		TRUE, TRUE, 0, 0
	},
	{ DECELERATION, 0, 1.0, 1.0, &ship.s_decint,
		"Deceleration", " %",
		0, 1234,
		True, True, False, False,
		TRUE, TRUE, 0, 0
	},
	{ DILITHIUM, 0, 2.0, 2.0, &ship.s_recharge,
		"Dilithium Crystals", " %",
		0, 6,
		True, True, False, False,
		TRUE, TRUE, 7, 25
	},
	{ DAMAGE_CONTROL, 0, 2.0, 2.0, &ship.s_repair,
		"Damage Control", " %",
		0, 8,
		True, True, False, False,
		TRUE, TRUE, 0, 0
	},
	{ CARGO, 0, 1.0, 1.0, &ship.s_cargo,
		"Cargo", "  ",
		0, 20,
		True, True, False, False,
		TRUE, TRUE, 30, 75
	},
	{ FRONT_SHIELD_SYS, 0, 1.0, 1.0, &ship.s_shield[FRONT_SHIELD],
		"Front Shield", " %",
		0, 1234,
		True, True, False, False,
		TRUE, TRUE, 0, 0
	},
	{ ARMOR, 0, 1.0, 0.25, &ship.s_armor,
		"Armor", "  ",
		0, 50,
		True, True, False, False,
		TRUE, TRUE, 0, 250
	},
	{ HULL, 0, 1.0, 1.0, &ship.s_hull_str,
		"Hull Strength", " %",
		0, 1234,
		True, True, False, False,
		TRUE, TRUE, 100, 500
	},
	{ FUEL, 0, 1.0, 1.0/500.0, &ship.s_maxfuel,
		"Fuel Capacity", "  ",
		0, 1234,
		True, True, False, False,
		TRUE, FALSE, 20, 75
	},
	{ TITLE_BAR, 0, 0.0, 1.0, &junk,
		"MENU_ITEM", "  ",
		0, 1,
		False, False, False, False,
		TRUE, FALSE, 0, 0
	},
	{ SIDE_SHIELD_SYS, 0, 1.0, 1.0, &ship.s_shield[RIGHT_SHIELD],
		"Side Shields", " %",
		0, 1234,
		True, True, False, False,
		TRUE, TRUE, 0, 0
	},
	{ REAR_SHIELD_SYS, 0, 1.0, 1.0, &ship.s_shield[REAR_SHIELD],
		"Rear Shield", " %",
		0, 1234,
		True, True, False, False,
		TRUE, TRUE, 0, 0
	},
	{ CLOAK, 0, 14.0, 14.0, &junk,
		"Cloaking Device", "  ",
		0, 1,
		True, True, False, False,
		TRUE, TRUE, 20, 50
	},
	{ IMP_SENS, 0, 5.0, 5.0, &junk,
		"Improved Sensors", "  ",
		0, 1,
		True, True, False, False,
		TRUE, TRUE, 10, 25
	},
	{ ECM, 0, 8.0, 8.0, &junk,
		"ECM", "  ",
		0, 1,
		True, True, False, False,
		TRUE, TRUE, 15, 40
	},
	{ HYPERSPACE, 0, 15.0, 15.0, &junk,
		"Hyperspace Drive", "  ",
		0, 1,
		True, True, False, False,
		TRUE, TRUE, 20, 50
	},
	{ SPHERE, 0, 100.0, 100.0, &junk,
		"Black Sphere", "  ",
		0, 1,
		True, True, False, False,
		FALSE, FALSE, 80, 200
	},
	{ AFC, 0, 8.0, 8.0, &junk,
		"Adv. Fire Control", "  ",
		0, 1,
		True, True, False, False,
		FALSE, FALSE, 10, 20
	},
	{ DAMAGE_ABS, 0, 10.0, 10.0, &junk,
		"Damage Absorbers", "  ",
		0, 1,
		True, True, False, False,
		TRUE, TRUE, 20, 50
	}
};

typedef struct {
	int min,dflt,max,attr_inc,tonnage_inc;
} SUBWEAPON_ATTRS;

typedef struct {
	char *weapon_name;
	int fuel_multiplier;
	SUBWEAPON_ATTRS speed_or_range;
	SUBWEAPON_ATTRS damage;
	SUBWEAPON_ATTRS duration_or_accuracy;
	SUBWEAPON_ATTRS reload;
	int base_size;
} WEAPON_PROTO;

WEAPON_PROTO *torpptr,*beamptr;

/* ALL SIZES & SUCH MULTIPLIED BY 4 */
static WEAPON_PROTO weapon_proto[] = {
	/* TORPEDO WEAPONS */
	{ "Photon Torpedo", 10,
		{   WPN_PHOTON_MIN_SPEED,  WPN_PHOTON_DEF_SPEED,
			WPN_PHOTON_MAX_SPEED, 2,  3 },
		{   WPN_PHOTON_MIN_DAMAGE,  WPN_PHOTON_DEF_DAMAGE,
			WPN_PHOTON_MAX_DAMAGE, 6,  5 },
		{   WPN_PHOTON_MIN_DURATION,   WPN_PHOTON_DEF_DURATION,
			WPN_PHOTON_MAX_DURATION, 1,  3 },
		{   WPN_PHOTON_MIN_RELOAD,   WPN_PHOTON_DEF_RELOAD,
			WPN_PHOTON_MAX_RELOAD, 1, -5 },
		8 },
	{ "Missile", 10,
		{   WPN_MISSILE_MIN_SPEED,   WPN_MISSILE_DEF_SPEED,
			WPN_MISSILE_MAX_SPEED, 1,  5 },
		{   WPN_MISSILE_MIN_DAMAGE, WPN_MISSILE_DEF_DAMAGE,
			WPN_MISSILE_MAX_DAMAGE, 3,  8 },
		{   WPN_MISSILE_MIN_DURATION,   WPN_MISSILE_DEF_DURATION,
			WPN_MISSILE_MAX_DURATION, 2,  9 },
		{   WPN_MISSILE_MIN_RELOAD,   WPN_MISSILE_DEF_RELOAD,
			WPN_MISSILE_MAX_RELOAD, 1, -6 },
		12 },
	{ "Plasma Torpedo", 10,
		{   WPN_PLASMA_MIN_SPEED,   WPN_PLASMA_DEF_SPEED,
			WPN_PLASMA_MAX_SPEED,  1,  30 },
		{   WPN_PLASMA_MIN_DAMAGE,   WPN_PLASMA_DEF_DAMAGE,
			WPN_PLASMA_MAX_DAMAGE, 10,  35 },
		{   0,   0,   0,  1,   1 },
		{   WPN_PLASMA_MIN_RELOAD,   WPN_PLASMA_DEF_RELOAD,
			WPN_PLASMA_MAX_RELOAD,  1, -25 },
		100 },
	{ "Chaff", 5,
		{   0,   0,   0, 1,  1 },
		{   WPN_CHAFF_MIN_DAMAGE, WPN_CHAFF_DEF_DAMAGE,
			WPN_CHAFF_MAX_DAMAGE, 2,  4 },
		{   WPN_CHAFF_MIN_DURATION,   WPN_CHAFF_DEF_DURATION,
			WPN_CHAFF_MAX_DURATION, 2,  5 },
		{   WPN_CHAFF_MIN_RELOAD,   WPN_CHAFF_DEF_RELOAD,
			WPN_CHAFF_MAX_RELOAD, 1, -6 },
		20 },
	{ "Mine", 1,
		{   0,   0,   0,  1,  1 },
		{   WPN_MINE_MIN_DAMAGE,   WPN_MINE_DEF_DAMAGE,
			WPN_MINE_MAX_DAMAGE,  5, 10 },
		{   WPN_MINE_MIN_DURATION,   WPN_MINE_DEF_DURATION,
			WPN_MINE_MAX_DURATION,  2,  3 },
		{   WPN_MINE_MIN_RELOAD,   WPN_MINE_DEF_RELOAD,
			WPN_MINE_MAX_RELOAD,  1, -5 },
		15 },

	/* BEAM WEAPONS */
	{ "Phaser", 8,
		{ 	WPN_PHASER_MIN_RANGE,  WPN_PHASER_DEF_RANGE,
			WPN_PHASER_MAX_RANGE, 1500,  4 },
		{   WPN_PHASER_MIN_DAMAGE,    WPN_PHASER_DEF_DAMAGE,
			WPN_PHASER_MAX_DAMAGE,    5,  6 },
		{   WPN_PHASER_MIN_ACCURACY,     WPN_PHASER_DEF_ACCURACY,
			WPN_PHASER_MAX_ACCURACY,    1,  2 },
		{   WPN_PHASER_MIN_RELOAD,     WPN_PHASER_DEF_RELOAD,
			WPN_PHASER_MAX_RELOAD,    1, -5 },
		15 },
	{ "Disrupter", 10,
		{ 	WPN_DISRUPTER_MIN_RANGE,  	WPN_DISRUPTER_DEF_RANGE,
			WPN_DISRUPTER_MAX_RANGE, 1500,  2 },
		{   WPN_DISRUPTER_MIN_DAMAGE,    WPN_DISRUPTER_DEF_DAMAGE,
			WPN_DISRUPTER_MAX_DAMAGE,    3,  7 },
		{   WPN_DISRUPTER_MIN_ACCURACY,     WPN_DISRUPTER_DEF_ACCURACY,
			WPN_DISRUPTER_MAX_ACCURACY,    1,  3 },
		{   WPN_DISRUPTER_MIN_RELOAD,     WPN_DISRUPTER_DEF_RELOAD,
			WPN_DISRUPTER_MAX_RELOAD,    1, -5 },
		20 },
	{ "Tachyon", 50,
		{   WPN_TACHYON_MIN_RANGE,    WPN_TACHYON_DEF_RANGE,
			WPN_TACHYON_MAX_RANGE, 1000,  5 },
		{   WPN_TACHYON_MIN_DAMAGE,     WPN_TACHYON_DEF_DAMAGE,
			WPN_TACHYON_MAX_DAMAGE,    1,  5 },
		{   WPN_TACHYON_MIN_ACCURACY,     WPN_TACHYON_DEF_ACCURACY,
			WPN_TACHYON_MAX_ACCURACY,    1,  3 },
		{   WPN_TACHYON_MIN_RELOAD,     WPN_TACHYON_DEF_RELOAD,
			WPN_TACHYON_MAX_RELOAD,    2, -16 },
		25 },
	{ "Fusion", 10,
		{   WPN_FUSION_MIN_RANGE,    WPN_FUSION_DEF_RANGE,
			WPN_FUSION_MAX_RANGE, 1000,  3 },
		{   WPN_FUSION_MIN_DAMAGE,    WPN_FUSION_DEF_DAMAGE,
			WPN_FUSION_MAX_DAMAGE,    5,  8 },
		{   WPN_FUSION_MIN_ACCURACY,     WPN_FUSION_DEF_ACCURACY,
			WPN_FUSION_MAX_ACCURACY,    2,  2 },
		{   WPN_FUSION_MIN_RELOAD,     WPN_FUSION_DEF_RELOAD,
			WPN_FUSION_MAX_RELOAD,    1, -5 },
		12 }
};


typedef struct {
	int torp,beam,sptechmask;
} RACE_WEAPON;

static RACE_WEAPON rw[] = {
	/* IND */ { WPN_PHOTON,	WPN_PHASER, 0 }, 
	/* FED */ { WPN_PHOTON,	WPN_PHASER, 0 },
	/* ROM */ { WPN_PLASMA, WPN_FUSION, 0 },
	/* KLI */ { WPN_MISSILE, WPN_PHASER, 0 },
	/* ORI */ { WPN_PHOTON, WPN_DISRUPTER, 0 },
	/* JUG */ { WPN_MISSILE, WPN_PHASER, 0 }, 
	/* MON */ { WPN_MISSILE, WPN_PHASER, 0 }, 
};

/* convenience defines */
#define Engines (value[ENGINE])
#define Engptr  (&sys[ENGINE])
#define Fuelptr	(&sys[FUEL])
#define Fuel	(value[FUEL])

/* GLOBALS */
static int hull;
static int buildship = TRUE,robotship = FALSE, maxsize = FALSE;
static int printit; /* print in readable format */
static char *reqshipname;       /* shipname requested */
static XSizeHints size_hints;
static int req_x, req_y, req_flags;
static Window parent;
static Display *mydisplay;
static int myscreen;
static Window base;
static GC gc, gcerase;

set_class(window, class, name)
Window window;
char *class, *name;
{
	XClassHint xch;     /* Class hint for window manager */

	xch.res_class = class;
	xch.res_name = name;

	XSetClassHint(mydisplay, window, &xch);
}



static void initialize()
{
	int i;

	for (i=0; i<9; ++i) {
		/* compensate for effects of decreasing size with incr. relod */
		weapon_proto[i].base_size -= 
			(weapon_proto[i].reload.max - weapon_proto[i].reload.min)
			/ weapon_proto[i].reload.attr_inc
			* weapon_proto[i].reload.tonnage_inc;
	}

	seedrandom();
	construct_filenames();
}


/* This routine maps the "value" into the value the user sees. */
static int fvalue(item,val)
int item,val;
{
	int i;

	switch (item) {
		case TORP_NUM:
		case BEAM_NUM:
		case CLOAK:
		case IMP_SENS:
		case ECM:
		case HYPERSPACE:
		case SPHERE:
		case AFC:
		case DAMAGE_ABS:
		case ENGINE:
		case TRACTOR_BEAM:
		case TITLE_BAR:
			return(val);
		case ARMOR:
			return(val*4);
		case WARP_COST:
			if (hull > STD_HULL_SIZE) {
			  return((hull+STD_HULL_SIZE)*50/STD_HULL_SIZE
				- 25*val);
			}
			else  {
				return(hull*100/STD_HULL_SIZE - 
					(25*STD_HULL_SIZE)*val/hull);
			}
		case MAX_SPEED:
			i = 36*Engines/hull;
			if (i > sys[MAX_SPEED].max) return (sys[MAX_SPEED].max);
			else return(i);
		case MANEUVER:
			i = (500000/hull*Engines+val*8000)/1280;
			if (i > 400) return(400);
			else return(i);
		case ACCELERATION:
			i = 400*Engines/hull+val*20;
			if (i > 400) return(400);
			else return(i);
		case DECELERATION:
			i = 400*Engines/hull+val*30;
			if (i > 400) return(400);
			else return(i);
		case DILITHIUM:
			return(60+val*10);
		case DAMAGE_CONTROL:
			return(60+12*val);
		case CARGO:
			return(hull*3/STD_HULL_SIZE+val);
		case FRONT_SHIELD_SYS:
		case REAR_SHIELD_SYS:
			return(50 + val*12);
		case SIDE_SHIELD_SYS:
			return(50 + val*6);
		case HULL:
			return((hull*213/STD_HULL_SIZE + 12*val)*100/256);
		case FUEL:
			return(val*500);
		case TORP_SPEED:
			return(val * torpptr->speed_or_range.attr_inc
				+ torpptr->speed_or_range.min);
		case TORP_DAMAGE:
			return(val * torpptr->damage.attr_inc
				+ torpptr->damage.min);
		case TORP_DURATION:
			return(val * torpptr->duration_or_accuracy.attr_inc
				+ torpptr->duration_or_accuracy.min);
		case TORP_RELOAD:
			return(val * torpptr->reload.attr_inc
				+ torpptr->reload.min);
		case BEAM_RANGE:
			return(val * beamptr->speed_or_range.attr_inc
				+ beamptr->speed_or_range.min);
		case BEAM_DAMAGE:
			return(val * beamptr->damage.attr_inc
				+ beamptr->damage.min);
		case BEAM_ACCURACY:
			return(val * beamptr->duration_or_accuracy.attr_inc
				+ beamptr->duration_or_accuracy.min);
		case BEAM_RELOAD:
			return(val * beamptr->reload.attr_inc
				+ beamptr->reload.min);
		default:
			fprintf(stderr,"bad sys# for fvalue(%d,%d).\n",
				item,val);
			return(0);
	}
}


		
/* This routine maps the "value" into the field value */
static int ffield(item,val)
int item,val;
{
	int i;

	switch (item) {
		case TORP_NUM:
		case BEAM_NUM:
		case CLOAK:
		case IMP_SENS:
		case ECM:
		case HYPERSPACE:
		case SPHERE:
		case AFC:
		case DAMAGE_ABS:
		case ENGINE:
		case TRACTOR_BEAM:
		case TITLE_BAR:
			return(val);
		case ARMOR:
			return(4*val+1);
		case WARP_COST:
			if (hull > STD_HULL_SIZE) {
				return((hull+STD_HULL_SIZE)*2/STD_HULL_SIZE - val);
			}
			else {
				return(4*hull/STD_HULL_SIZE - STD_HULL_SIZE*val/hull);
			}
		case MANEUVER:
			i = 500000/hull*Engines+val*8000;
			if (i > 512000) return(512000);
			else return(i);
		case ACCELERATION:
			i = 400*Engines/hull+val*20;
			if (i > 400) return(400);
			else return(i);
		case DECELERATION:
			i = 800*Engines/hull+val*60;
			if (i > 800) return(800);
			else return(i);
		case DILITHIUM:
			return(6+val);
		case DAMAGE_CONTROL:
			return(150+30*val);
		case HULL:
			return(hull*213/STD_HULL_SIZE + 12*val);
		case MAX_SPEED:
		case CARGO:
		case FRONT_SHIELD_SYS:
		case REAR_SHIELD_SYS:
		case SIDE_SHIELD_SYS:
		case FUEL:
		case TORP_SPEED:
		case TORP_DAMAGE:
		case BEAM_RANGE:
		case BEAM_DAMAGE:
		case BEAM_ACCURACY:
			return(fvalue(item,val));
		case TORP_DURATION:
		case TORP_RELOAD:
		case BEAM_RELOAD:
			return(fvalue(item,val)*10);
		default:
			fprintf(stderr,"bad sys# for ffield(%d,%d).\n",
				item,val);
			return(0);
	}
}


static float torpsize()
{
	return((torpptr->base_size + 
		value[TORP_SPEED]
			* torpptr->speed_or_range.tonnage_inc +
		value[TORP_DAMAGE]
			* torpptr->damage.tonnage_inc +
		value[TORP_DURATION]
			* torpptr->duration_or_accuracy.tonnage_inc +
		value[TORP_RELOAD]
			* torpptr->reload.tonnage_inc 
		) / 8.0);
}

static float beamsize()
{
	return((beamptr->base_size + 
		value[BEAM_RANGE]
			* beamptr->speed_or_range.tonnage_inc +
		value[BEAM_DAMAGE]
			* beamptr->damage.tonnage_inc +
		value[BEAM_ACCURACY]
			* beamptr->duration_or_accuracy.tonnage_inc +
		value[BEAM_RELOAD]
			* beamptr->reload.tonnage_inc 
		) / 8.0);
}


static Boolean update_value(sysptr)
MENU_ITEM *sysptr;
{
	int i,j,c,r;
	char str[80];

	/* do system-dependent tasks */
	switch (sysptr->index) {
		case ENGINE:
			/* also update some others */
			update_value(&sys[MAX_SPEED]);
			update_value(&sys[MANEUVER]);
			update_value(&sys[ACCELERATION]);
			update_value(&sys[DECELERATION]);
			break;
		case TORP_SPEED:
		case TORP_DAMAGE:
		case TORP_DURATION:
		case TORP_RELOAD:
			{	float oldsize,newsize;
				int delta;

				/* insure there's enough fuel */
				oldsize = sys[TORP_NUM].size;
				newsize = torpsize();
				delta  = (int) (newsize*value[TORP_NUM]);
				delta -= (int) (oldsize*value[TORP_NUM]);
				if (Fuel < delta) return(False);
				/* else */
				sys[TORP_NUM].size = sys[TORP_NUM].dsize = newsize;
				value[FUEL] -= delta;

				/* update label/size/value widget for torp */
				update_value(&sys[TORP_NUM]);
			}
			break;
		case BEAM_RANGE:
		case BEAM_DAMAGE:
		case BEAM_ACCURACY:
		case BEAM_RELOAD:
			{	float oldsize,newsize;
				int delta;

				/* insure there's enough fuel */
				oldsize = sys[BEAM_NUM].size;
				newsize = beamsize();
				delta  = (int) (newsize*value[BEAM_NUM]);
				delta -= (int) (oldsize*value[BEAM_NUM]);
				if (Fuel < delta) return(False);
				/* else */
				sys[BEAM_NUM].size = sys[BEAM_NUM].dsize = newsize;
				value[FUEL] -= delta;

				/* update label/size/value widget for beam */
				update_value(&sys[BEAM_NUM]);
			}
			break;
		default:
			break;
	}

	/* if here, everything's okay */
	sysptr->dval = fvalue(sysptr->index,value[sysptr->index]);

	/* update label/size/value widget */
	sprintf(str,LABELFORMAT,
		LABELSIZE,sysptr->label,
		sysptr->dsize,
		sysptr->dval,
		UNITSIZE,sysptr->units);
	for (i = 1; i >= 0; --i) {
		for (j = (NUM_ADJ_SYS/2); j >= 0; --j)  {
			if (display_order[i][j] == sysptr->index) {
				c = i;
				r = j;
				i = -1;	/* break out of outer loop */
				break;
			} 
		}
	}
	i = ((COLUMN2*c) - (COLUMN1*(c-1)) + (XSPACING + 3*XBTNSPACING));
	j = (STARTROW + (r*YSPACING));
	XClearArea(mydisplay, base,
		i+3, j+3, WIDTH-6, HEIGHT-6, FALSE);
	XDrawString(mydisplay, base, gc,
		(i+5), (j+13), str, strlen(str));

	return(True);
}


static void complete_ship()
{
	MENU_ITEM *sp;
	int i,j,rmult,prmult;
	SYSTEM_UNIT *suptr,*suptr2;

	/* call ffield to evaluate each system */
	for (i=0,sp=sys;i<NUM_ADJ_SYS;++i,++sp) 
		*sp->field = ffield(i,value[sp->index]);

	/* finish torps */
	/* torpdecay completed in getship */
	ship.s_torpcost = 
		torpptr->fuel_multiplier * ship.s_torpdamage;
	strncpy(ship.s_torpname,torpptr->weapon_name,
		sizeof(ship.s_torpname));

	/* finish beams */
	ship.s_beamcost =
		beamptr->fuel_multiplier * ship.s_beamdamage;
	/* beamatten completed in getship */
	strncpy(ship.s_beamname,beamptr->weapon_name,
		sizeof(ship.s_beamname));

	/* details, details */
	ship.s_shield[LEFT_SHIELD] = 
		ship.s_shield[RIGHT_SHIELD];

	/* fill hit locations list */
	prmult = REPAIRFUSE / 5;
	rmult = ffield(DAMAGE_CONTROL,value[DAMAGE_CONTROL]) * prmult;

	suptr = ship.s_system;
	ship.s_numsys = 0;

	/* ARMOR MUST BE SYSTEM UNIT #0 */
	suptr->item     		= ARMOR;
	suptr->subitem  		= 0;
	suptr->current_size		= suptr->undamaged_size	= ship.s_armor;
	suptr->repair_rate      = (sys[ARMOR].repair_rate * rmult) >> 8;
	suptr->planet_repair_rate  = sys[ARMOR].planet_repair_rate * prmult;
	++suptr; ++ship.s_numsys;

	/* HULL MUST BE SYSTEM UNIT #1 */
	suptr->item     		= HULL;
	suptr->subitem  		= 0;
	suptr->current_size		= suptr->undamaged_size = 
		ship.s_hull * ship.s_hull_str *
		100 / STD_HULL_SIZE / 256;
	suptr->repair_rate      = (sys[HULL].repair_rate * rmult) >> 8;
	suptr->planet_repair_rate  = sys[HULL].planet_repair_rate * prmult;
	++suptr; ++ship.s_numsys;

	/* OTHER SYSTEM's UNDAMAGED SIZES & REPAIRS ARE INITIALIZED LATER */
	suptr->item				= ENGINE;
	suptr->subitem			= 0;
	suptr->current_size		= value[ENGINE] * sys[ENGINE].size +
		  value[MANEUVER] * sys[MANEUVER].size +
		  value[ACCELERATION] * sys[ACCELERATION].size +
		  value[DECELERATION] * sys[DECELERATION].size +
		  value[WARP_COST] * sys[WARP_COST].size;
	++suptr; ++ship.s_numsys;

	suptr->item				= FUEL;
	suptr->subitem			= 0;
	suptr->current_size		= Fuel;
	++suptr; ++ship.s_numsys;

	for (i=0;i<value[BEAM_NUM];++i) {
		suptr->item				= BEAM_NUM;
		suptr->subitem			= i;
		suptr->current_size		= (int) (sys[BEAM_NUM].size + 0.5);
		++suptr; ++ship.s_numsys;
	}

	/* SP TECH */
	for (i=0;i<sizeof(stlist)/sizeof(STPAIR);++i) {
		j = stlist[i].system_no;
		if (value[j] > 0) {
			ship.s_sptechmask |= stlist[i].mask;
			suptr->item		  	= j;
			suptr->subitem		= 0;
			suptr->current_size = (int) (sys[j].size);
			++suptr; ++ship.s_numsys;
		}
		else {
			ship.s_sptechmask &= ~stlist[i].mask;
		}
	}

	for (i=0;i<value[TORP_NUM];++i) {
		suptr->item				= TORP_NUM;
		suptr->subitem			= i;
		suptr->current_size		= (int) (sys[TORP_NUM].size + 0.5);
		++suptr; ++ship.s_numsys;
	}

	for (i=0;i<value[TRACTOR_BEAM];++i) {
		suptr->item				= TRACTOR_BEAM;
		suptr->subitem			= i;
		suptr->current_size		= (int) sys[TRACTOR_BEAM].size;
		++suptr; ++ship.s_numsys;
	}

	suptr->item				= DILITHIUM;
	suptr->subitem			= 0;
	suptr->current_size		= ffield(DILITHIUM,value[DILITHIUM])-2;
	++suptr; ++ship.s_numsys;

	suptr->item				= CARGO;
	suptr->subitem			= 0;
	suptr->current_size		= ffield(CARGO,value[CARGO]);
	++suptr; ++ship.s_numsys;

	/* all systems start undamaged */
	ship.s_systemsize = 0;
	for (suptr2=ship.s_system+2;suptr2<suptr;++suptr2) {
		ship.s_systemsize += suptr2->current_size;
		suptr2->undamaged_size = suptr2->current_size;
		suptr2->repair_rate = 
			(sys[suptr2->item].repair_rate * rmult) >> 8;
		suptr2->planet_repair_rate = 
			sys[suptr2->item].planet_repair_rate * prmult;
	}
}


static void draw_buttons(item,x,y)
MENU_ITEM *item;
int x,y;
{
	if (item->modifiable) {
		if (value[item->index] <= item->max-5) item->plus5_on = True;
		else item->plus5_on = False;
		if (value[item->index] < item->max) item->plus_on = True;
		else item->plus_on = False;
		if (value[item->index] > item->min) item->minus_on = True;
		else item->minus_on = False;
		if (value[item->index] >= item->min+5) item->minus5_on = True;
		else item->minus5_on = False;
	}
	else {
		item->plus5_on =
			item->plus_on =
			item->minus_on =
			item->minus5_on =
			False;
	}
	
	if (item->plus5_on) {
		XDrawRectangle(mydisplay, base, gc, x, y, HEIGHT, HEIGHT);
		XDrawString(mydisplay, base, gc, (x+5), (y+13), "+5", 2);
	}
	x += XBTNSPACING; 
	if(item->plus_on) {
		XDrawRectangle(mydisplay, base, gc, x, y, HEIGHT, HEIGHT);
		XDrawString(mydisplay, base, gc, (x+5), (y+13), "+1", 2);
	}
	x += XBTNSPACING; 
	if (item->minus_on) {
		XDrawRectangle(mydisplay, base, gc, x, y, HEIGHT, HEIGHT);
		XDrawString(mydisplay, base, gc, (x+5), (y+13), "-1", 2);
	}
	x += XBTNSPACING; 
	if (item->minus5_on) {
		XDrawRectangle(mydisplay, base, gc, x, y, HEIGHT, HEIGHT);
		XDrawString(mydisplay, base, gc, (x+5), (y+13), "-5", 2);
	}
}
 

static void draw_buildmenu(fontinfo,exit_verify)
XFontStruct *fontinfo;
int exit_verify;
{
	int i,j,x,y,w;
	char str[80];
	MENU_ITEM *m;

	for (i=0; i<2; i++) {
		for (j=0; j<((NUM_ADJ_SYS/2)+1); j++) {
			x = (COLUMN2*i)-(COLUMN1*(i-1));
			y = STARTROW+(j*YSPACING);
			if ((display_order[i][j]) != BLANK_ENTRY) {
				m = &(sys[(display_order[i][j])]);
				if (m->show_sys) {
					draw_buttons(m,x,y);
					x += XSPACING+(3*XBTNSPACING);
					XDrawRectangle(mydisplay, base, gc,
						x, y, WIDTH, HEIGHT);
					if (display_order[i][j]==TITLE_BAR) {
						sprintf(str,"%-*s   SIZE    VALUE ",
							LABELSIZE,m->label);
						XDrawString(mydisplay, base, gc,
							(x+5), (y+13),
							str, strlen(str));
	         		}
					else {
						m->dval = fvalue(m->index,value[m->index]);
						sprintf(str,LABELFORMAT,
							LABELSIZE,m->label,
							m->dsize,m->dval,
							UNITSIZE,m->units);
						XDrawString(mydisplay, base, gc,
							(x+5), (y+13),
							str, strlen(str));
					}
				} 
			}             
		}
	}

	/* exit button */
	x = COLUMN1;
	y = STARTROW+(((NUM_ADJ_SYS/2)+1)*YSPACING);
	w = COLUMN2+WIDTH+(3*XBTNSPACING)+XSPACING-COLUMN1;
	if (exit_verify) {
		sprintf(str,"Confirm Exit");
		XClearArea(mydisplay, base,
			x+3, y+3,
			w-6, HEIGHT-6, FALSE);
		XDrawString(mydisplay, base, gc, 
			(w/2)-(XTextWidth(fontinfo, str, strlen(str))/2), 13+y,
			str, strlen(str));
		XDrawRectangle(mydisplay, base, gc,
			x, y,
			w, HEIGHT);
	}
	else {
		sprintf(str,"Exit Shipmenu");
		XClearArea(mydisplay, base,
			x+3, y+3,
			w-6, HEIGHT-6, FALSE);
		XDrawString(mydisplay, base, gc,
			(w/2)-(XTextWidth(fontinfo, str, strlen(str))/2), 13+y,
			str, strlen(str));
		XDrawRectangle(mydisplay, base, gc,
			x, y,
			w, HEIGHT);
	}
}


static void setup_weaponchars(wno,swptr,mode)
int wno;
SUBWEAPON_ATTRS *swptr;
int mode;
{
	if (swptr->min == swptr->max) {
		sys[wno].show_sys = sys[wno].modifiable = FALSE;
	}

	if (mode & NEW_VALUES_MASK) {
		value[wno] = (swptr->dflt - swptr->min) / swptr->attr_inc;
	}

	if (mode & NEED_DVAL_MASK) {
		sys[wno].dval  = fvalue(sys[wno].index,value[wno]);
		sys[wno].dsize = (float) ((int) (50.0 *
			ABS((float)swptr->tonnage_inc/swptr->attr_inc)))
			/100.0;
	}

	sys[wno].min   = 0;
	sys[wno].max   = (swptr->max  - swptr->min) / swptr->attr_inc;
	sys[wno].size  = 0.0;
}


static void setup_racials(torp,beam,sptechmask,mode)
int torp,beam,sptechmask,mode;
{
	int i;

	torpptr = &weapon_proto[torp];
	beamptr = &weapon_proto[beam];

	ship.s_torpclass  = torp;
	setup_weaponchars(TORP_SPEED,&torpptr->speed_or_range,mode);
	setup_weaponchars(TORP_DAMAGE,&torpptr->damage,mode);
	setup_weaponchars(TORP_DURATION,&torpptr->duration_or_accuracy,mode);
	setup_weaponchars(TORP_RELOAD,&torpptr->reload,mode);
	strcpy(sys[TORP_NUM].label,torpptr->weapon_name);
	sys[TORP_NUM].size = sys[TORP_NUM].dsize = torpsize();

	ship.s_beamclass = beam;
	setup_weaponchars(BEAM_RANGE,&beamptr->speed_or_range,mode);
	setup_weaponchars(BEAM_DAMAGE,&beamptr->damage,mode);
	setup_weaponchars(BEAM_ACCURACY,&beamptr->duration_or_accuracy,mode);
	setup_weaponchars(BEAM_RELOAD,&beamptr->reload,mode);
	strcpy(sys[BEAM_NUM].label,beamptr->weapon_name);
	sys[BEAM_NUM].size = sys[BEAM_NUM].dsize = beamsize();

	/* turn on special technologies available */
	for (i=0;i<sizeof(stlist)/sizeof(STPAIR);++i)
		if (sptechmask & stlist[i].mask) {
			sys[stlist[i].system_no].show_sys = 
				sys[stlist[i].system_no].modifiable =
				TRUE;
		}
		else {
			sys[stlist[i].system_no].show_sys =
				sys[stlist[i].system_no].modifiable =
				FALSE;
		}
}


void setup_ship(hull_size,race,mode,mytorp,mybeam)
int hull_size,race,mode,mytorp,mybeam;
{
	int sptechmask,i;

	if (mode & NEW_VALUES_MASK) {
		for (i=0; i<NUM_ADJ_SYS; ++i) 
			if ((i != HULL) && (i != FUEL)) value[i] = 0;     
		value[FUEL] = hull_size;
	}

	hull = 
		ship.s_hull =
		sys[ENGINE].max = 
		sys[MANEUVER].max = 
		sys[ACCELERATION].max = 
		sys[DECELERATION].max = 
		sys[FRONT_SHIELD_SYS].max = 
		sys[SIDE_SHIELD_SYS].max = 
		sys[REAR_SHIELD_SYS].max = 
		sys[FUEL].max =
		hull_size;

	if (mode & NEED_DVAL_MASK) {
		sys[FUEL].dval = fvalue(FUEL,value[FUEL]);
	}

	if (mytorp == -1) mytorp = rw[race].torp;
	if (mybeam == -1) mybeam = rw[race].beam;
	sptechmask = (STMASK_CLOAK | STMASK_IMP_SENS | STMASK_ECM |
		STMASK_HYPERSPACE | STMASK_DAMAGE_ABS);
	sptechmask |= rw[race].sptechmask;

	if (!(mode & NEW_VALUES_MASK))
		sptechmask |= ship.s_sptechmask;

	setup_racials(mytorp,mybeam,sptechmask,mode);

	if (race == PLANETKILLER) {
		sptechmask |= STMASK_SPHERE;
		sys[SPHERE].modifiable   = TRUE;
		sys[SPHERE].show_sys     = TRUE;
		sys[TORP_NUM].modifiable = FALSE;
		sys[MAX_SPEED].max       = 3;
		sys[ARMOR].repair_rate   = 10;
		sys[BEAM_DAMAGE].max     = 15;
		sys[BEAM_RANGE].max      = sys[BEAM_RANGE].min;
		value[BEAM_RANGE]        = sys[BEAM_RANGE].min;
		sys[BEAM_RANGE].modifiable = FALSE;
		sys[BEAM_RELOAD].max      = sys[BEAM_RELOAD].min;
		value[BEAM_RELOAD]        = sys[BEAM_RELOAD].min;
		sys[BEAM_RELOAD].modifiable = FALSE;
		sys[TRACTOR_BEAM].max    = 6;
		sys[DILITHIUM].max       = 20;
	}
}


#define BUFSIZE 256
static void get_current_list(list)
SHIPFILENAMESTRUCT *list[];
{
	int count,pipemask,readit,len;
	int pipefd[2];
	char buf[BUFSIZE],*cptr;
	struct timeval waittime;

	if (access(shiparchive,4) == -1) return;
	/* else */

	pipe(pipefd);
	if (fork() == 0) {
		pipedetach(pipefd[1]);
		/* do "ar t filename" */
		execl("/bin/ar","ar","t",shiparchive,0);
		/* should never get here anyway */
		exit(0);
	}
	else {
		/* parent only */
		/* read from ar command */
		wait((int *)0);
		count = 0;
		pipemask = 1<<pipefd[0];
		waittime.tv_sec  = 10;
		waittime.tv_usec = 0;
		/* is pipe output ready? */
		if (select(pipefd[0]+1,&pipemask,0,0,&waittime) <= 0) return;
		waittime.tv_sec  = 1;
		do {
			cptr = buf;
			do {
				if (select(pipefd[0]+1,&pipemask,0,0,&waittime) <= 0) {
					readit = FALSE;
					break;
				}
				/* else */
				readit = read(pipefd[0],cptr++,1);
			} while (*(cptr-1) != '\012');
			if (readit) {
				*(cptr-1) = '\0';
				strcpy(list[count]->filename,buf);
				sscanf(cptr-6,"%1d%4d",
  					&(list[count]->team_no),&(list[count]->hull_size));
				if ((len = strlen(buf)-5) > 9) len = 9;
				strncpy(list[count]->short_shipname,buf,len);
				list[count]->short_shipname[len] = '\0';
				++count;
			}
		} while (readit);

		close(pipefd[0]);
		close(pipefd[1]);
	}
}


/* Is this ship unmodified (no cheating!) and not out-of-date? */
static int ship_verified(team_no,hull_size)
int team_no,hull_size;
{
	int i,sum;

	/* check version number */
	if (!robotship
			&& (saved_stuff.version_number < CURRENT_VERSION)) {
		fprintf(stderr,"shiparchive version number mismatch.\n");
		return(FALSE);
	}

	/* check for bad weapon types */
	if ((team_no >= FEDERATION) && (team_no <= ORION)) {
		if (ship.s_torpclass != rw[team_no].torp) {
			fprintf(stderr,"torpclass mismatch.\n");
			return(FALSE);
		}
		if (ship.s_beamclass != rw[team_no].beam) {
			fprintf(stderr,"beamclass mismatch.\n");
			return(FALSE);
		}
	}
	
	/* invalid values & fields */
	sum = 0;
	for (i=0; i<NUM_ADJ_SYS; ++i) {
		if (value[sys[i].index] < sys[i].min) {
			fprintf(stderr,"value < minimum\n");
			return(FALSE);
		}
		if (value[sys[i].index] > sys[i].max) {
			fprintf(stderr,"value > minimum\n");
			return(FALSE);
		}
		sum += (int) (value[sys[i].index] * sys[i].size);
	}

	/* doesn't add up */
	if (sum > hull_size) {
		fprintf(stderr,"sum of components doesn't match hull size.\n");
		return(FALSE);
	}

	return(TRUE);
}


/* return value: is a good ship returned? */
static int retrieve_shipfile(shipname,team_no,hull_size,mode)
char *shipname;
int team_no,hull_size,mode;
{
	int i;
	SAVED_STUFF tmpstuff;

	if (ar_read_entry(shiparchive,shipname,(char *) &tmpstuff,
			(unsigned) sizeof(SAVED_STUFF)) == FALSE) {
		return(FALSE);
	}

	if (!robotship
			&& (tmpstuff.version_number < CURRENT_VERSION)) {
		/* don't alter the array */
		return(FALSE);
	}
	else {
		/* used the saved value array */
		/* else */
		if (tmpstuff.team_number != team_no) return(FALSE);
		memcpy((char *) &saved_stuff,(char *) &tmpstuff,
			sizeof(SAVED_STUFF));
		ship.s_torpclass = saved_stuff.torpclass;
		ship.s_beamclass = saved_stuff.beamclass;
		strcpy(ship.s_shipname,saved_stuff.shipname);
		ship.s_hull = hull_size;
		mode &= ~NEW_VALUES_MASK;
		/* setup racial stuff and whatnot */
		setup_ship(hull_size,team_no,mode,
			ship.s_torpclass,ship.s_beamclass);
		if (mode & NEED_DVAL_MASK) {
			/* recompute dvals */
			for (i=0; i<NUM_ADJ_SYS; ++i) {
				sys[i].dval = fvalue(sys[i].index,value[i]);
			}
		}
		if (!ship_verified(team_no,hull_size)) return(FALSE);
	}

	return(TRUE);
}


#define UNCHOOSE_SHIP(win,width,i,j) \
{ \
	XDrawRectangle(mydisplay, (win), gcerase,  \
		4+((width)+3)*(i), 4+(3+HEIGHT)*(j), (width)-2, HEIGHT-2); \
}


#define CHOOSE_SHIP(win,width,i,j) \
{ \
	XDrawRectangle(mydisplay, (win), gc,  \
		4+((width)+3)*(i), 4+(3+HEIGHT)*(j), (width)-2, HEIGHT-2); \
}


#define BTNUNLIGHT(win,col,i,j) \
{ \
	XDrawRectangle(mydisplay, (win), gcerase,  \
		((col)*(COLUMN2-COLUMN1))+12+(HEIGHT+5)*(i), \
		12+(5+HEIGHT)*(j), HEIGHT-4, HEIGHT-4); \
}

#define BTNLIGHT(win,col,i,j) \
{ \
	XDrawRectangle(mydisplay, (win), gc,  \
		((col)*(COLUMN2-COLUMN1))+12+(HEIGHT+5)*(i), \
		12+(5+HEIGHT)*(j), HEIGHT-4, HEIGHT-4); \
}

#define UNLIGHT(win,width,i,j) \
{ \
	XDrawRectangle(mydisplay, (win), gcerase,  \
		5+((width)+3)*(i), 5+(3+HEIGHT)*(j), (width)-4, HEIGHT-4); \
}

#define HIGHLIGHT(win,width,i,j) \
{ \
	XDrawRectangle(mydisplay, (win), gc,  \
		5+((width)+3)*(i), 5+(3+HEIGHT)*(j), (width)-4, HEIGHT-4); \
}

    
static void draw_select_w(list,select_w,fontinfo,
	width,listsize,hull_size,show_size, eff_size,delete_verify)
SHIPFILENAMESTRUCT *list[];
Window select_w;
XFontStruct *fontinfo;
int width,listsize,hull_size,show_size,eff_size;
Boolean delete_verify;
{
	int j,k;
	char str[80];

	/* create "hull size" window */
	sprintf(str,"Usable hull size: %d ktons",eff_size);
	XDrawString(mydisplay, select_w, gc, 
		3+(width/2)-(XTextWidth(fontinfo, str, strlen(str))/2),
		13+3+(3+HEIGHT)*0, str, strlen(str));
	XDrawRectangle(mydisplay, select_w, gc, 
		3, 3+(3+HEIGHT)*0, width, HEIGHT);
	XDrawRectangle(mydisplay, select_w, gc, 
		4, 4+(3+HEIGHT)*0, (width-2), (HEIGHT-2));
	XDrawRectangle(mydisplay, select_w, gc, 
		5, 5+(3+HEIGHT)*0, (width-4), (HEIGHT-4));

	/* create "current hull size" window */
	if (show_size == ALL_HULLS) {
		sprintf(str,"Show ALL hulls");
	}
	else {
		if (show_size == USABLE_HULLS) {
			sprintf(str,"Show USABLE hulls");
		}
		else {
			sprintf(str,"Show %d kton hulls",show_size);
		}
	}

	XDrawString(mydisplay, select_w, gc, 
		3+3+width+(width/2)-(XTextWidth(fontinfo, str, strlen(str))/2),
		13+3+(3+HEIGHT)*0, str, strlen(str));
	XDrawRectangle(mydisplay, select_w, gc, 
		3+3+width, 3+(3+HEIGHT)*0, width, HEIGHT);
	XDrawRectangle(mydisplay, select_w, gc, 
		4+3+width, 4+(3+HEIGHT)*0, (width-2), (HEIGHT-2));
	XDrawRectangle(mydisplay, select_w, gc, 
		5+3+width, 5+(3+HEIGHT)*0, (width-4), (HEIGHT-4));

	/* do buttons for all current selectable ships */
	for (j=0,k=1; j<listsize; ++j,k++) {
		sprintf(str,"\"%.9s\" (%d ktons)",
			list[j]->short_shipname,list[j]->hull_size);
		XDrawString(mydisplay, select_w, gc, 
			3+(width/2)-(XTextWidth(fontinfo, str, strlen(str))/2),
			13+3+(3+HEIGHT)*k, str, strlen(str));
		XDrawRectangle(mydisplay, select_w, gc, 
			3, 3+(3+HEIGHT)*k, width, HEIGHT);
	}

	/* create "Use" button */
	sprintf(str,"Use");
	XDrawString(mydisplay, select_w, gc, 
		3+width+3+(width/2)-(XTextWidth(fontinfo, str, strlen(str))/2),
		13+3+(3+HEIGHT)*1, str, strlen(str));
	XDrawRectangle(mydisplay, select_w, gc, 
		3+width+3, 3+(3+HEIGHT)*1, width, HEIGHT);

	/* create "Edit" button */
	sprintf(str,"Edit");
	XDrawString(mydisplay, select_w, gc, 
		3+width+3+(width/2)-(XTextWidth(fontinfo, str, strlen(str))/2),
		13+3+(3+HEIGHT)*2, str, strlen(str));
	XDrawRectangle(mydisplay, select_w, gc, 
		3+width+3, 3+(3+HEIGHT)*2, width, HEIGHT);

	/* create "View" button */
	sprintf(str,"View");
	XDrawString(mydisplay, select_w, gc, 
		3+width+3+(width/2)-(XTextWidth(fontinfo, str, strlen(str))/2),
		13+3+(3+HEIGHT)*3, str, strlen(str));
	XDrawRectangle(mydisplay, select_w, gc, 
		3+width+3, 3+(3+HEIGHT)*3, width, HEIGHT);

	/* create "Copy" button */
	sprintf(str,"Copy");
	XDrawString(mydisplay, select_w, gc, 
		3+width+3+(width/2)-(XTextWidth(fontinfo, str, strlen(str))/2),
		13+3+(3+HEIGHT)*4, str, strlen(str));
	XDrawRectangle(mydisplay, select_w, gc, 
		3+width+3, 3+(3+HEIGHT)*4, width, HEIGHT);

	/* create "Delete" button */
	if (delete_verify) {
		sprintf(str,"Confirm Delete");
		XClearArea(mydisplay, select_w,
			3+width+6, 6+(3+HEIGHT)*5, width-6, HEIGHT-6, FALSE);
		XDrawString(mydisplay, select_w, gc, 
			3+width+3+(width/2)-(XTextWidth(fontinfo, str,
				strlen(str))/2), 13+3+(3+HEIGHT)*5, str, strlen(str));
		XDrawRectangle(mydisplay, select_w, gc, 
			3+width+3, 3+(3+HEIGHT)*5, width, HEIGHT);
	}
	else {
		sprintf(str,"Delete");
		XClearArea(mydisplay, select_w,
			3+width+6, 6+(3+HEIGHT)*5, width-6, HEIGHT-6, FALSE);
		XDrawString(mydisplay, select_w, gc, 
			3+width+3+(width/2)-(XTextWidth(fontinfo, str,
				strlen(str))/2), 13+3+(3+HEIGHT)*5, str, strlen(str));
		XDrawRectangle(mydisplay, select_w, gc, 
			3+width+3, 3+(3+HEIGHT)*5, width, HEIGHT);
	}

	/* create "New Ship" button */
	sprintf(str,"Construct %d kton ship",hull_size);
	XDrawString(mydisplay, select_w, gc, 
		3+width+3+(width/2)-(XTextWidth(fontinfo, str, strlen(str))/2),
		13+3+(3+HEIGHT)*6, str, strlen(str));
	XDrawRectangle(mydisplay, select_w, gc, 
		3+width+3, 3+(3+HEIGHT)*6, width, HEIGHT);

	/* create "Default Ship" button */
	sprintf(str,"Get a default ship");
	XDrawString(mydisplay, select_w, gc, 
		3+width+3+(width/2)-(XTextWidth(fontinfo, str, strlen(str))/2),
		13+3+(3+HEIGHT)*7, str, strlen(str));
	XDrawRectangle(mydisplay, select_w, gc, 
		3+width+3, 3+(3+HEIGHT)*7, width, HEIGHT);

	/* create "Set hull Size" button */
	sprintf(str,"Set hull Size");
	XDrawString(mydisplay, select_w, gc, 
		3+width+3+(width/2)-(XTextWidth(fontinfo, str, strlen(str))/2),
		13+3+(3+HEIGHT)*8, str, strlen(str));
	XDrawRectangle(mydisplay, select_w, gc, 
		3+width+3, 3+(3+HEIGHT)*8, width, HEIGHT);
}


static void show_text(name_w, width, str, num, fontinfo)
Window name_w;
int width;
char str[];
int num;
XFontStruct *fontinfo;
{
	XClearWindow(mydisplay, name_w);
	XDrawRectangle(mydisplay, name_w, gc, 
		3 + XTextWidth(fontinfo, "Ship Name:", 10),
		3,
		width - XTextWidth(fontinfo, "Ship Name:", 10),
		HEIGHT);
	XDrawString(mydisplay, name_w, gc,
		3 + XTextWidth(fontinfo, "Ship Name: ", 11),
		3 + 13,
		str, num);
	XDrawString(mydisplay, name_w, gc, 3, 3 + 13, "Ship Name:", 10);
}


static int select_ship(list,listsize,hull_size,eff_size,show_size,action)
SHIPFILENAMESTRUCT *list[];
int listsize,hull_size,eff_size,show_size,*action;
{
	int i,x,y,choicex,choicey,lastx,lasty;
	Boolean selected, ship_chosen, delete_verify;
	int width;
	Window select_w;
	XFontStruct *fontinfo;
	XSetWindowAttributes att;
	XEvent next_e, *next_p;

	if (!buildship) {
		*action = USE_OLD_SHIP;
		if (listsize == 0) {
			fprintf(stderr,"shipmenu: no ships match criteria. %s",
				"Cannot select random ship.  Aborting.\n");
			exit(1);
		}
		/* else */
		if (reqshipname == NULL) {
			/* return a random ship */
			if (maxsize) {
				int *good,max;

				if ((good = (int *) malloc(listsize*sizeof(int)))
						== NULL) {
					/* just return one at random */
					return(INTRAND(listsize));
				}

				/* find the biggest size */
				max = 0;
				for (i=0; i<listsize; ++i)
					if (list[i]->hull_size > max)
						max = list[i]->hull_size;
			
				/* mark all ships that size as good */
				for (i=0; i<listsize; ++i)
					if (list[i]->hull_size == max) good[i] = TRUE;
					else good[i] = FALSE;

				/* now pick a good ship */
				do {
					i = INTRAND(listsize);
				} while (!good[i]);

				free((char *) good);
				return(i);
			}
			else {
				/* just return one at random */
				return(INTRAND(listsize));
			}
		}
		else {
			/* find the requested ship */
			for (i=0; i<listsize; ++i)
				if (strncmp(reqshipname,list[i]->short_shipname,9) == 0)
					return(i);
			/* If I'm here, I couldn't find it */
			fprintf(stderr,"shipmenu: no ships match criteria. %s",
				"Cannot select random ship.  Aborting.\n");
			exit(1);
		}		
	}
	/* else  INTERACTIVE */

	fontinfo = XQueryFont(mydisplay,(XID)gc->gid);
	width = 30*XTextWidth(fontinfo, "X", 1);

	i = 0;
	select_w = XCreateSimpleWindow(mydisplay, parent,
		10, 10,
		9+2*width, 			/* 9 = 3 + 3 + 3 buffers */
		(((listsize+1)*(HEIGHT+3)+3) > (9*(HEIGHT+3)+3) ?
			(listsize+1)*(HEIGHT+3)+3  :  /* 3 = buffer btwn entries */
               (9*(HEIGHT+3)+3)),
		2,
		XBlackPixel(mydisplay, myscreen),
		XWhitePixel(mydisplay, myscreen));

	if (select_w == NULL) {
		fprintf(stderr,"Could not create window.\n");
		exit(1);
	}

	if ((req_flags & XValue) && (req_flags & XNegative)) { 
		size_hints.x = WidthOfScreen(DefaultScreenOfDisplay(mydisplay))
			- (9+2*width); 
	}

	if ((req_flags & YValue) && (req_flags & YNegative)) {
		size_hints.y = WidthOfScreen(DefaultScreenOfDisplay(mydisplay))
			-  (((listsize+1)*(HEIGHT+3)+3) > (8*(HEIGHT+3)+3) ?
			(listsize+1)*(HEIGHT+3)+3  : /* 3 = buffer btwn entries */
				(8*(HEIGHT+3)+3));
	}

	XSetStandardProperties(mydisplay, select_w, "RCMenu",
		"RCMenu", None, NULL, 0, &size_hints);
	set_class(select_w, "SHipMenu", "RCMenu");
	XMapRaised(mydisplay, select_w);

	delete_verify = FALSE;
	draw_select_w(list,select_w,fontinfo,width,listsize,hull_size,
		show_size,eff_size,delete_verify);

	/* wait for a selection */
	att.event_mask = ExposureMask | ButtonPressMask | PointerMotionMask;
	XChangeWindowAttributes(mydisplay, select_w, CWEventMask, &att);
	ship_chosen = FALSE;
	selected = FALSE;
	lastx= -1;lasty= -1;
	choicex = -2; choicey = -2;

	*action = NO_ACTION;
	next_p = &next_e;
	do {
		XNextEvent(mydisplay, &next_e);
		switch ((int)next_p->type) {
			case Expose:
				if (((XExposeEvent *)next_p)->count == 0) {
					draw_select_w(list,select_w,fontinfo,
						width,listsize,hull_size,
						show_size,eff_size,delete_verify);
				}
				if (selected) {
					HIGHLIGHT(select_w,width,lastx,lasty);
				}
				break;
			case MotionNotify:
				y = ((XMotionEvent *)next_p)->y;
				x = ((XMotionEvent *)next_p)->x;
				if ((y > 3) 
						&& ((y/23) == ((y-3)/23))
						&& ((y/23) > 0)
						&& (x > 3)
						&& ((x/(width+3))==((x-3)/(width+3)))) {
					/* in a valid subwindow */
					if (!((lastx == x/(width+3)) && (lasty == y/23))) {
						UNLIGHT(select_w,width,lastx,lasty);
						lastx = x/(width+3);
						lasty = y/23;
						if ((lastx == 1) && (lasty > 8)) {
							selected = FALSE;
							lastx= -1; lasty= -1;
						}
						else {
							if ((lastx == 0) && (lasty > listsize)) {
								selected = FALSE;
								lastx= -1; lasty= -1;
							}
							else {
								selected = TRUE;
								HIGHLIGHT(select_w,width,lastx,lasty);
							}
						}
					}
				}
				else if (selected) {
						selected = FALSE;
						UNLIGHT(select_w,width,lastx,lasty);
						lastx= -1; lasty= -1;
				}
				break;
			case ButtonPress:
				if ((((XButtonEvent *)next_p)->button == Button1)
						&& (selected)) {
					if (lastx == 0) {
						/* SHIPS column */
						if (ship_chosen) {
							if (choicey == lasty) {
								if (list[(choicey-1)]->hull_size
										<= eff_size) {
									*action = USE_OLD_SHIP;
								}
							}
							else {
								UNCHOOSE_SHIP(select_w,width,choicex,choicey);
							}
						}
						delete_verify = FALSE;
						ship_chosen = TRUE;
						choicex = lastx; choicey = lasty;
						CHOOSE_SHIP(select_w,width,choicex,choicey);
						draw_select_w(list,select_w,fontinfo,
							width,listsize,hull_size,
							show_size,eff_size,delete_verify);
					}
					else switch (lasty) {
						case CONSTRUCT_NEW_SHIP:
							choicey = listsize;
							*action  = CONSTRUCT_NEW_SHIP;
							break;
						case GET_DEFAULT:
							*action = GET_DEFAULT;
							break;
						case SET_HULL_SIZE:
							*action = SET_HULL_SIZE;
							break;
						case USE_OLD_SHIP:
						case EDIT_OLD_SHIP:
						case VIEW_OLD_SHIP:
						case COPY_OLD_SHIP:
							if ((ship_chosen)
								&& ((lasty != USE_OLD_SHIP)
										|| (list[(choicey-1)]->hull_size
											<= eff_size))) {
								*action = lasty;
							}
							break;
						case DELETE_OLD_SHIP:
							if (delete_verify) {
								*action = DELETE_OLD_SHIP;
							} 
							else {
								delete_verify = TRUE; 
								draw_select_w(list,select_w,
								fontinfo,width,listsize,
								hull_size,show_size,
								eff_size,delete_verify);
							} 
							break;
					}
				}
				break;
		}
	} while (*action == NO_ACTION);

	XDestroyWindow(mydisplay, select_w);
	XFlush(mydisplay);
	return((choicey-1));
}



static void name_ship(str,maxlen)
char *str;
int maxlen;
{
	int len, i,j, width, done;
	char ch;
	Window name_w;
	XFontStruct *fontinfo;
	XSetWindowAttributes att;
	XEvent next_e, *next_p;
	char *cptr;
	char name[80];

	fontinfo = XQueryFont(mydisplay,(XID)gc->gid);
	width = (maxlen+strlen("Ship Name: ><"))
		*XTextWidth(fontinfo, "X", 1);
	 /* KLUDGE - total width of maxlen chars, in pixels */

	name_w = XCreateSimpleWindow(mydisplay, parent, 10, 10,
		6+width, 			/* 6 = 3 + 3 buffers */
		HEIGHT+6,
		2,
		XBlackPixel(mydisplay, myscreen),
		XWhitePixel(mydisplay, myscreen));
	if (name_w == NULL) {
		fprintf(stderr,"Could not create window.\n");
		exit(1);
	}

	if ((req_flags & XValue) && (req_flags & XNegative)) {
		size_hints.x = WidthOfScreen(DefaultScreenOfDisplay(mydisplay))
			-  (6+width);
	}
                   
	if ((req_flags & YValue) && (req_flags & YNegative)) {
		size_hints.y = WidthOfScreen(DefaultScreenOfDisplay(mydisplay))
			- (HEIGHT+6);
	}

	XSetStandardProperties(mydisplay, name_w, "NewName",
		"NewName", None, NULL, 0, &size_hints);
	set_class(name_w, "SHipMenu", "NewName");
	XMapRaised(mydisplay, name_w);
	name[0] = '>';
	name[1] = '<';
	len = 2;
	show_text(name_w, width, name, len, fontinfo);

	/* wait for a selection */
	att.event_mask = ExposureMask | KeyPressMask;
	XChangeWindowAttributes(mydisplay, name_w, CWEventMask, &att);
	done = FALSE;

	next_p = &next_e;
	do {
		XNextEvent(mydisplay, &next_e);
		switch ((int)next_p->type) {
			case Expose:
				if (((XExposeEvent *)next_p)->count == 0)
					show_text(name_w, width, name, len, fontinfo);
				break;
			case KeyPress: 
				if( XLookupString((XKeyPressedEvent *)next_p,
						&ch, 1, (KeySym *)NULL,
						(XComposeStatus *)NULL) != 0) {
					switch (ch) {
						case '\b':
							if (len > 2) {
								len--;
								name[(len-1)]='<';
							}
							break;
						case '\r':
							done = TRUE;
							break;
						case '\000':
							break;
						default:
							if (!iscntrl(ch)) {
								if (len<maxlen+2) {
									name[len]='<';
									name[(len-1)]=ch;
									len++;
								}
							}
							break;
					}
					show_text(name_w, width, name, len, fontinfo);
				} 
				break;
			default:
				break;
		}
	} while (!done);

	*str = '\0';
	i = 0;
	cptr = name;
	for ( ; *cptr != '<' ; ++cptr)
		if ((*cptr >= 'A' && *cptr <= 'Z')
				|| (*cptr >= 'a' && *cptr <= 'z')
				|| (*cptr >= '0' && *cptr <= '9')
				|| (*cptr == '.')
				|| (*cptr == '_')) {
			name[i++] = *cptr;
			if (i >= sizeof(name)) break;
		}
        for (j=0;j<maxlen;j++)
          ship.s_shipname[j]='\0';
	strncpy(str,name,maxlen);
	if (i < maxlen) str[i] = '\0';
	else str[maxlen] = '\0';
	j = sizeof(ship.s_shipname)-1;
	if (j > i) j = i;
	strncpy(ship.s_shipname,str,j);
	ship.s_shipname[j+1] = '\0';

	XDestroyWindow(mydisplay, name_w);
	XFlush(mydisplay);
}


static void delete_ship(shipname)
char *shipname;
{
	char str[80];

	sprintf(str,"/bin/ar d %s %s",shiparchive,shipname);
	system(str);
	chmod(shiparchive,0666);
	chown(shiparchive,getuid(),getgid());
}


static char *save_builtship(team_no,hull_size,newship)
int team_no,hull_size,newship;
{
	char str[80];
	static char filename[256];

	if ((newship == CONSTRUCT_NEW_SHIP) || (newship == GET_DEFAULT)) {
		/* get the ship's name */
		name_ship(str,40);
		strncpy(saved_stuff.shipname,str,SHIPNAMELEN);
		saved_stuff.shipname[SHIPNAMELEN-1] = '\0';
		str[9] = '\0';
		sprintf(filename,"%s%1d%04d",str,team_no,hull_size);
	}
	else {
		strncpy(str,saved_stuff.shipname,SHIPNAMELEN);
		str[9] = '\0';
		sprintf(filename,"%s%1d%04d",str,team_no,hull_size);
	}

	saved_stuff.version_number = CURRENT_VERSION;
	saved_stuff.torpclass   = ship.s_torpclass;
	saved_stuff.beamclass   = ship.s_beamclass;
	saved_stuff.team_number = team_no;

	/* save it in the "ar" file */
	ar_write_entry(shiparchive,filename,(char *) &saved_stuff,
			(unsigned) sizeof(SAVED_STUFF));
	chmod(shiparchive,0666);
	chown(shiparchive,(int) getuid(),(int) getgid());
	return(filename);
}


static int rebuild_lists(num_ships, hull_size, show_size, team_no,
	goodlist, list)
int num_ships, hull_size, show_size, team_no;
SHIPFILENAMESTRUCT *goodlist[],*list[];
{
	int i,j;

	j = 0;
	for (i=0; i<num_ships; ++i) {
		if ((list[i]->team_no == team_no)
			&& (!robotship ||
				(reqshipname != NULL) ||
					!(strncmp(list[i]->short_shipname,"Starbase",8) == 0 ||
						strncmp(list[i]->short_shipname,"Defense_base",9) == 0))) {
             if (list[i]->hull_size > 0) {
				if (show_size == ALL_HULLS) {
					goodlist[j++] = list[i];
				}
				else {
					if (show_size == USABLE_HULLS) {
						if (list[i]->hull_size <= hull_size) {
							goodlist[j++] = list[i];
						}
					}
					else {
						if (list[i]->hull_size == hull_size) {
							goodlist[j++] = list[i];
						}
					}
				}
			}
		}
	}
	return(j);
}


static int read_names_from_ar(num_ships, list)
int *num_ships;
SHIPFILENAMESTRUCT *list[];
{
	int i;
	if (*num_ships = ar_table_of_contents_size(shiparchive)) {
		if (*num_ships > MAXSHIPS) {
			fprintf(stderr,"Exceeded MAXSHIPS.  num_ships=%d\n",*num_ships);
			exit(1);
		}
		for (i=0; i<(*num_ships); i++) {
			list[i] = (SHIPFILENAMESTRUCT *)
				malloc(sizeof(SHIPFILENAMESTRUCT));
		}
		get_current_list(list);
	}
}


static void draw_hulls(edit_w,fontinfo,width)
Window edit_w;
XFontStruct *fontinfo;
int width;
{
	int i,j,k;
	char str[80];

	/* do buttons for all reasonable hull sizes */
	k=135;
	for (i=0; i<3; i++) {
		for (j=0; j<3; j++,k+=15) {
			sprintf(str,"%d ktons",k);
			XDrawString(mydisplay, edit_w, gc, 
				5+(width/2)+((3+width)*i)-(XTextWidth(fontinfo,
					str, strlen(str))/2), 13+3+(3+HEIGHT)*j,
				str, strlen(str));
			XDrawRectangle(mydisplay, edit_w, gc, 
				3+((width+3)*i), 3+(3+HEIGHT)*j, width, HEIGHT);
		}
	}

	sprintf(str,"ALL Hulls",k);
	XDrawString(mydisplay, edit_w, gc, 
		5+(width/2)+((3+width)*3)-(XTextWidth(fontinfo,
			str, strlen(str))/2), 13+3+(3+HEIGHT)*0,
		str, strlen(str));
	XDrawRectangle(mydisplay, edit_w, gc, 
		3+((width+3)*3), 3+(3+HEIGHT)*0, width, HEIGHT);

	sprintf(str,"USABLE Hulls",k);
	XDrawString(mydisplay, edit_w, gc, 
		5+(width/2)+((3+width)*3)-(XTextWidth(fontinfo,
			str, strlen(str))/2),
		13+3+(3+HEIGHT)*1, str, strlen(str));
	XDrawRectangle(mydisplay, edit_w, gc, 
		3+((width+3)*3), 3+(3+HEIGHT)*1, width, HEIGHT);
}


static int select_hulls()
{
	int k,x,y,dx,dy,done;
	Boolean selected;
	int width;
	Window edit_w;
	XFontStruct *fontinfo;
	XSetWindowAttributes att;
	XEvent next_e, *next_p;

	fontinfo = XQueryFont(mydisplay,(XID)gc->gid);
	width = 15*XTextWidth(fontinfo, "X", 1);
	/* KLUDGE - total width of 30 chars, in pixels */

	edit_w = XCreateSimpleWindow(mydisplay, parent, 10, 10,
		15+4*width, 		/* 9 = 3 + 3 + 3 buffers */
		(3*(HEIGHT+3)+3),
		2,
		XBlackPixel(mydisplay, myscreen),
		XWhitePixel(mydisplay, myscreen));
	if (edit_w == NULL) {
		fprintf(stderr,"Could not create window.\n");
		exit(1);
	}

	if ((req_flags & XValue) && (req_flags & XNegative)) {
		size_hints.x = WidthOfScreen(DefaultScreenOfDisplay(mydisplay))
			- (15+4*width); 
	}
                  
	if ((req_flags & YValue) && (req_flags & YNegative)) {
		size_hints.y = WidthOfScreen(DefaultScreenOfDisplay(mydisplay))
			- (3*(HEIGHT+3)+3);
	}

	XSetStandardProperties(mydisplay, edit_w, "HullMenu", "HullMenu",
		None, NULL, 0, &size_hints);
	set_class(edit_w, "SHipMenu", "HullMenu");
	XMapRaised(mydisplay, edit_w);
	draw_hulls(edit_w,fontinfo,width);

	/* wait for a selection */
	done = FALSE;
	att.event_mask = ExposureMask | ButtonPressMask | PointerMotionMask;
	XChangeWindowAttributes(mydisplay, edit_w, CWEventMask, &att);
   	selected = FALSE;
	dx= -1; dy= -1;
	next_p = &next_e;

	do {
		XNextEvent(mydisplay, &next_e);
		switch ((int)next_p->type) {
			case Expose:
				if (((XExposeEvent *)next_p)->count == 0)
					draw_hulls(edit_w,fontinfo,width);
				if (selected) HIGHLIGHT(edit_w,width,dx,dy);
				break;
			case MotionNotify:
				y = ((XMotionEvent *)next_p)->y;
				x = ((XMotionEvent *)next_p)->x;
				if ((y > 3) 
						&& ((y/23) == ((y-3)/23))
						&& (x > 3)) {
					if ((x/(width+3)) == ((x-3)/(width+3))) {
						if (!((dx == x/(width+3)) && (dy == y/23))) {
							UNLIGHT(edit_w,width,dx,dy);
							dx = x/(width+3);
							dy = y/23;
							if ((dx == 3) && (dy >= 2)) {
								selected = FALSE;
								dx= -1; dy= -1;
							}
							else {
								selected = TRUE;
								HIGHLIGHT(edit_w,width,dx,dy);
							}
						}
					}
				}
				else if (selected) {
					selected = FALSE;
					UNLIGHT(edit_w,width,dx,dy);
					dx= -1; dy= -1;
				}
				break;
			case ButtonPress:
				if (((XButtonEvent *)next_p)->button == Button1) {
					if (selected) {
						done = TRUE;
						if (dx==3) {
							if (dy==0) k = ALL_HULLS;
							if (dy==1) k = USABLE_HULLS;
						}
						else { 
							k=135+(dy*15)+(dx*15*3);
						}
					}
				}
				break;
			default:
				break;
		}
	} while (!done);

	XDestroyWindow(mydisplay, edit_w);
	XFlush(mydisplay);
	return(k);
}


static int examine_current_ships(hull_size,eff_size,team_no,show_size,
	num_ships, list, goodlist)
int *hull_size,eff_size,team_no,*show_size;
int *num_ships;
SHIPFILENAMESTRUCT *list[],*goodlist[];
{
	int whichship,list_size,returnval,action,i;
	char *filename;
	Boolean more_ships;

	do {
		more_ships = FALSE;
		list_size = rebuild_lists(*num_ships, *hull_size, *show_size,
			team_no, goodlist, list);

		if ((*num_ships > 0) || (buildship)) {
			whichship = select_ship(goodlist,list_size,
				*hull_size,eff_size,*show_size,&action);
			switch (action) {
				case USE_OLD_SHIP: 
					if (retrieve_shipfile(goodlist[whichship]->filename,
								team_no,goodlist[whichship]->hull_size,
								(printit ? NEED_DVAL_MASK : 0))) {
						hull = goodlist[whichship]->hull_size;
						complete_ship();
						returnval = action;
					}
					else {
						returnval = REDO_OLD_SHIP;
					}
					break;
				case COPY_OLD_SHIP: 
					if (retrieve_shipfile(goodlist[whichship]->filename,
								team_no,goodlist[whichship]->hull_size,
								0)) {
						if ((Fuel += (*hull_size -
								goodlist[whichship]->hull_size)) < 0) {
							break;
						}
						else {
							hull = *hull_size;
							complete_ship();
							/* save the new ship */
							filename = save_builtship(team_no,
								*hull_size,CONSTRUCT_NEW_SHIP);
							returnval = action;
						}
					}
					else {
						returnval = REDO_OLD_SHIP;
					}
					break;
				case EDIT_OLD_SHIP:
				case VIEW_OLD_SHIP:
					if (retrieve_shipfile(goodlist[whichship]->filename,
							team_no,goodlist[whichship]->hull_size,
							NEED_DVAL_MASK)) {
						hull = goodlist[whichship]->hull_size;
						returnval = action;
					}
					else {
						returnval = REDO_OLD_SHIP;
					}
					break;
				case DELETE_OLD_SHIP:
					delete_ship(goodlist[whichship]->filename);
					goodlist[whichship]->hull_size = -1;
					more_ships = TRUE;
					break;
				case SET_HULL_SIZE:
					*show_size = select_hulls();
					*hull_size = (*show_size);
					if (*show_size == ALL_HULLS) *hull_size = 500;
					else if (*show_size == USABLE_HULLS)
						*hull_size = eff_size;
					more_ships = TRUE;
					break;
				case CONSTRUCT_NEW_SHIP:
					returnval = CONSTRUCT_NEW_SHIP;
					break;
				case GET_DEFAULT: 
					if (retrieve_default(*hull_size,team_no)
							== USE_OLD_SHIP) {
						returnval = GET_DEFAULT;
					}
					else {
						more_ships = TRUE;
					}
					break;
			}
        }
 
		if ((returnval == COPY_OLD_SHIP) 
			 	&& ((*num_ships+1) < MAXSHIPS)) {
			/* copy the old ship stats to the new one */
			list[*num_ships] = (SHIPFILENAMESTRUCT *)
				malloc(sizeof(SHIPFILENAMESTRUCT));
			strcpy(list[*num_ships]->filename,filename);
			strncpy(list[*num_ships]->short_shipname,filename,9);
			if ((i = strlen(filename)-5) < 10) {
				list[*num_ships]->short_shipname[i] = '\0';
			}
			else {
				list[*num_ships]->short_shipname[9] = '\0';
			}
			list[*num_ships]->team_no   = team_no;
			list[*num_ships]->hull_size = *hull_size;
			++(*num_ships);
			more_ships = TRUE;
		} 
	} while (more_ships);

	return(returnval);
}


static int retrieve_default(hull_size,team_no)
int hull_size,team_no;
{
	int num_ships;
	SHIPFILENAMESTRUCT *list[MAXSHIPS],*goodlist[MAXSHIPS];
	char holdarchive[MAXPATHLEN];
	int result,i;

	strcpy(holdarchive, shiparchive);
	strcpy(shiparchive,Shiparchives);
	strcat(shiparchive,"/");
	strcat(shiparchive,"shipdefaults");
	buildship=FALSE;
	maxsize=TRUE;
	read_names_from_ar(&num_ships, list);
	i = USABLE_HULLS;
	result = examine_current_ships(&hull_size,hull_size,team_no,
		&i,&num_ships, list, goodlist);
	for(i=0;i<num_ships;i++) {
		free(list[i]);
	}
	strcpy(shiparchive, holdarchive);
	buildship=TRUE; 
	return(result);
}


static void cat_builtship()
{
	/* dump it to stdout */
	if (write(1,(char *) &ship,sizeof(SHIP)) < sizeof(SHIP))
		fprintf(stderr,"write to stdout not successful\n");
	fflush(stdout);
}


#define OPTSTR "h:t:p:rs:dgD:m"

static void badusage()
{
	
	fprintf(stderr,"\nusage: shipmenu [options]\n");
	fprintf(stderr," valid options:\n");
	fprintf(stderr,"    -h <hull_size>\n");
	fprintf(stderr,"    -m = biggest ship available\n");
	fprintf(stderr,"    -t <team #>\n");
	fprintf(stderr,"    -p <player name>\n");
	fprintf(stderr,"    -r = random ship\n");
	fprintf(stderr,"    -s <shipname>\n");
	fprintf(stderr,"    -d = dump in human-readable form to stdout\n");
	fprintf(stderr,"    -g = dump in game form\n");
	fprintf(stderr,"    -D <display>\n");
	fprintf(stderr,"\n");

	exit(1);
}


static Boolean act_on_button(col,dy,dx)
int col, dy, dx;
{
	MENU_ITEM *sysptr;
	int delta,x,y;
	Boolean retval;
	int oldsize, newsize;

	sysptr = &(sys[(display_order[col][dy])]);
	switch (dx) {
		case 0:
			delta=5;
			break;
		case 1:
			delta=1; 
			break;
		case 2:
			delta= -1;
			break;
		case 3:
			delta= -5;
			break;
		default:
			delta=0;
			break;
	}

	oldsize = (int) (sysptr->size * value[sysptr->index]);
	value[sysptr->index] += delta;
	newsize = (int) (sysptr->size * value[sysptr->index]);
		/* this may change during update */

	/* greater than maximum? */
	if (value[sysptr->index] > sysptr->max) {
		value[sysptr->index] -= delta;
		return(TRUE);
	}
	/* is there fuel? */
	if (Fuel < (newsize - oldsize)) {
		value[sysptr->index] -= delta;
		return(TRUE);
	}
	/* else */

	retval = TRUE;
	if (update_value(sysptr)) {
		x = (COLUMN2*col)-(COLUMN1*(col-1));
		y = STARTROW+(dy*YSPACING);
		if (delta > 0) {
			if (sysptr->plus5_on)
				if (value[sysptr->index] > sysptr->max-5) {
					if(dx==0) retval = FALSE;
					XClearArea(mydisplay, base,
						(x + (0*XBTNSPACING)), y,
						HEIGHT+1, HEIGHT+1, FALSE);
					XFlush(mydisplay);
				}
			/* unmap "+5" button */
			if (sysptr->plus_on)
				if (value[sysptr->index] == sysptr->max) {
					if(dx==1) retval = FALSE;
					XClearArea(mydisplay, base,
						(x+(1*XBTNSPACING)), y,
						HEIGHT+1, HEIGHT+1, FALSE);
					XFlush(mydisplay);
				}
		}

		if (delta < 0) {
			if (sysptr->minus5_on)
				if (value[sysptr->index] < sysptr->min+5) {
					if (dx==3) retval = FALSE;
					XClearArea(mydisplay, base, (x+(3*XBTNSPACING)), y,
						HEIGHT+1, HEIGHT+1, FALSE);
					XFlush(mydisplay);
				}
			/* unmap "-5" button */
			if (sysptr->minus_on)
				if (value[sysptr->index] == sysptr->min) {
					if (dx==2) retval = FALSE;
					XClearArea(mydisplay,base,(x+(2*XBTNSPACING)),y,
						HEIGHT+1,HEIGHT+1,FALSE);
					XFlush(mydisplay);
				}
		}

		/* update buttons */
		draw_buttons(sysptr, x, y);

		/* update fuel */
		newsize = (int) (sysptr->size * value[sysptr->index]);
		Fuel -= (newsize - oldsize);
		update_value(Fuelptr);
	}
	else { 
		/* bad value. Reset */
		value[sysptr->index] -= delta;
	}

	return(retval);
}


static Boolean is_btn_on(item, num)
MENU_ITEM *item;
int num;
{
	switch (num) {
		case 0:
			return(item->plus5_on);
			break;
		case 1:
			return(item->plus_on);
			break;
		case 2:
			return(item->minus_on);
			break;
		case 3:
			return(item->minus5_on);
			break;
	}
	return(FALSE);
}


static void print_ship(playername,race)
char *playername;
int race;
{
	register int i;
	register MENU_ITEM *s;

	static char *race_name[] = {
		"Independent -- Orbiter",
		"Federation",
		"Romulan",
		"Klingon",
		"Orion",
		"Independent",
		"Monitor",
		"Planetkiller"
	};

	printf("     Ship: %s\n",ship.s_shipname);
	printf("     Hull size: %d ktons\n",ship.s_hull);
	printf("     Ship architect:  %s\n",playername);
	printf("     Race:  %s\n",race_name[race]);
	printf("\n");

	for (i=0, s = sys; i<NUM_ADJ_SYS; ++i, ++s) {
		s = &sys[print_order[i]];
		if (!(s->show_sys)) continue;
		if (s->index == TITLE_BAR) continue;
		printf("%*s:  %5d%*s\n",
			LABELSIZE, s->label,
			s->dval,
			UNITSIZE, s->units);
	}
}


main(argc,argv)
int argc;
char *argv[];
{
	int i,num_ships;
	SHIPFILENAMESTRUCT *list[MAXSHIPS],*goodlist[MAXSHIPS];
	Boolean selected,exit_selected,valid_ship;
	XEvent next_e, *next_p;
	XFontStruct *fontinfo;
	XSetWindowAttributes att;
	XGCValues gcv;
	char *geom;
	int x,y,dx,dy,col,eff_size;
	int hull_size,team_no,newship,show_size,hold_hull,exit_verify;
	int mytorp,mybeam;
	char playername[80],c,*cptr,*req_display = NULL,*myargv[3];
	struct passwd *pwnam;
	int dumpit;
	unsigned long border, backg;

	/* defaults */
	hull_size   = 500;
	team_no     = -1;
	strcpy(playername,"NOT SPECIFIED");
	buildship   = TRUE;
	reqshipname = NULL;
	printit     = FALSE;
	dumpit      = FALSE;

	initialize();

	OPEN_LOGFILE(Logfilename);

	while ((c = (char) getopt(argc,argv,OPTSTR)) != (char) (EOF)) {
		switch (c) {
			case 'h':
				hull_size = atoi(optarg);
				break;
			case 't':
				team_no = atoi(optarg);
				break;
			case 'p':
				cptr = optarg;
				while (*cptr == ' ') ++cptr;
				strcpy(playername,cptr);
				break;
			case 'r':
				buildship = FALSE;
				reqshipname = NULL;
				break;
			case 's':
				buildship = FALSE;
				cptr = optarg;
				while (*cptr == ' ') ++cptr;
				reqshipname = cptr;
				break;
			case 'd':
				printit = TRUE;
				break;
			case 'g':
				dumpit = TRUE;
				break;
			case 'D':
				req_display = optarg;
				break;
			case 'm':
				maxsize = TRUE;
				break;
			case '?':
				badusage();
				break;
		}
	}

	if ((hull_size < 5) || (hull_size > 999)) {
		fprintf(stderr,"bad hull: %d.\n",hull_size);
		exit(1);
	}
	if ((team_no < 0) || (team_no > PLANETKILLER)) {
		fprintf(stderr,"bad team: %d\n",team_no);
		exit(1);
	}

	strcpy(shiparchive,Shiparchives);
	strcat(shiparchive,"/");
	if((strcmp(playername,"robot") != 0) &&
	   (strcmp(playername,"shipdefaults") != 0)) {
		/* make sure player is in user list */
		robotship = FALSE;
		if (strcmp(playername,"NOT SPECIFIED") == 0) {
			pwnam = getpwuid(getuid());
			if (pwnam == NULL) {
				fprintf(stderr,"cannot get playername from uid\n");
				endpwent();
				exit(1);
			}
			strcat(shiparchive,pwnam->pw_name);
			strcpy(playername, pwnam->pw_name);
		}
		else {
			pwnam = getpwnam(playername);
			if (pwnam == NULL) {
				fprintf(stderr,"bad player: >%s<\n",playername);
				endpwent();
				exit(1);
			}
			strcat(shiparchive,playername);
		}
		endpwent();
	}
	else {
		if (strcmp(playername,"robot") != 0) {
			robotship = FALSE;
			strcat(shiparchive,"shipdefaults");
		}
		else {
			robotship = TRUE;
			strcat(shiparchive,"robot");
		}
	}


	if (buildship) {
		if ((team_no == INDEPENDENT)
				|| (team_no == JUGGERNAUT)
				|| (team_no == MONITOR)
				|| (team_no == PLANETKILLER)) {
			fprintf(stderr,"Choose torp (0-4). \n");
			scanf("%d",&mytorp);
			fprintf(stderr,"Choose beam (5-8). \n");
			scanf("%d",&mybeam);
		}
		else {
			mytorp = mybeam = -1;
		}		
		myargv[0] = argv[0];
		if (req_display != NULL) {
			myargv[1] = "-display";
			myargv[2] = req_display;
		}

		mydisplay = XOpenDisplay(req_display);
		if (mydisplay == NULL)
			mydisplay = XOpenDisplay(getenv("DISPLAY"));
		if (mydisplay == NULL) {
			fprintf(stderr,"Could not open display\n");
			exit(1);
		}

		myscreen = XDefaultScreen(mydisplay);
		parent = XRootWindow(mydisplay, myscreen);
		backg = XWhitePixel(mydisplay, myscreen);
		border = XBlackPixel(mydisplay, myscreen);
		base = XCreateSimpleWindow(mydisplay, parent, 10, 10,
			COLUMN1+COLUMN2+WIDTH+(3*XBTNSPACING)+XSPACING,
			STARTROW+((NUM_ADJ_SYS/2)+2)*YSPACING, 2, border, backg);
		if (base == NULL) {
			fprintf(stderr,"Could not create window\n");
			exit(1);
		}

		size_hints.flags = (PPosition);
		req_x = 100;
		req_y = 100;
		geom = XGetDefault(mydisplay, "dtrek_shipmenu", "geometry");
		if (geom != NULL) {
			int winx, winy, winw, winh;

			req_flags = XParseGeometry(geom, &winx, &winy, &winw, &winh);
			if (req_flags & (XValue | YValue))
				size_hints.flags |= USPosition;
				if (req_flags & XValue) req_x = winx;
				if (req_flags & YValue) req_y = winy;
		}

		size_hints.x = req_x;
		size_hints.y = req_y;
		gc = XDefaultGC(mydisplay, myscreen);
		XSetBackground(mydisplay, gc, backg); 
		XSetForeground(mydisplay, gc, border); 
		gcerase = XCreateGC(mydisplay, base, 0, &gcv);
		XCopyGC(mydisplay, gc, ((1L << 23) - 1), gcerase);
		XSetBackground(mydisplay, gcerase, border); 
		XSetForeground(mydisplay, gcerase, backg); 
		fontinfo = XQueryFont(mydisplay,(XID)gc->gid);
	}

	show_size = USABLE_HULLS;
	eff_size = hull_size;
	read_names_from_ar(&num_ships, list);
	valid_ship=FALSE;
	do {
		exit_verify = FALSE;
		newship = examine_current_ships(&hull_size,eff_size,team_no,
			&show_size, &num_ships, list, goodlist);
		if (newship == GET_DEFAULT) {
			hold_hull = hull_size;
			hull_size = ship.s_hull;
		}

		if (       (newship == EDIT_OLD_SHIP)
				|| (newship == CONSTRUCT_NEW_SHIP)
				|| (newship == GET_DEFAULT)
				|| (newship == REDO_OLD_SHIP)
				|| (newship == VIEW_OLD_SHIP)) {
			if ((newship == CONSTRUCT_NEW_SHIP) 
					|| (newship == REDO_OLD_SHIP)) {
				setup_ship(hull_size,team_no,
					(NEW_VALUES_MASK|NEED_DVAL_MASK),mytorp,mybeam);
			}
			else {
				setup_ship(hull_size,team_no,
					NEED_DVAL_MASK,mytorp,mybeam);
			}

			if ((newship == CONSTRUCT_NEW_SHIP)
					|| (newship == REDO_OLD_SHIP)) {
				/* clear out value array */
				for (i=0; i<NUM_ADJ_SYS; ++i) 
					if (i != FUEL) value[i] = 0;     
			}

			if ((req_flags & XValue) && (req_flags & XNegative)) { 
				size_hints.x =
					WidthOfScreen(DefaultScreenOfDisplay(mydisplay))
					- (COLUMN1+COLUMN2+WIDTH+(3*XBTNSPACING)+XSPACING);
			}
			
			if ((req_flags & YValue) && (req_flags & YNegative)) {
				size_hints.y =
					WidthOfScreen(DefaultScreenOfDisplay(mydisplay))
					- (STARTROW+((NUM_ADJ_SYS/2)+2)*YSPACING);
			}
			
			XSetStandardProperties(mydisplay, base, "Shipmenu",
				"Shipmenu", None, NULL, 0, &size_hints);
			set_class(base, "SHipMenu", "Shipmenu");
			XMapRaised(mydisplay, base);
			draw_buildmenu(fontinfo,exit_verify);
          	att.event_mask = ExposureMask | ButtonPressMask
				| PointerMotionMask;
        	XChangeWindowAttributes(mydisplay, base, CWEventMask, &att);
			selected = FALSE;
			dx= -1;  dy= -1;
			next_p = &next_e;

			do {
				XNextEvent(mydisplay, &next_e);
				switch ((int)next_p->type) {
				case Expose:
					if (((XExposeEvent *)next_p)->count == 0)
						draw_buildmenu(fontinfo,exit_verify);
					break;
				case MotionNotify:
					exit_selected = FALSE;
					y = ((XMotionEvent *)next_p)->y;
					x = ((XMotionEvent *)next_p)->x;
					if ((y > 10) 
						 	&& (((y-5)/25) == ((y-10)/25))) {
						/* not in beteen buttons or in title bar */
						if (((y-5)/25) != ((NUM_ADJ_SYS/2)+1)) {
							/* not EXIT */
							if (newship == VIEW_OLD_SHIP) break;
							/* else */
							if (x > (COLUMN2-COLUMN1)) {
								x -= (COLUMN2-COLUMN1);
								col = 1;
							}
							else {
								col = 0;
							}
							if ((x > 10)
									&& (((x-5)/25) == ((x-10)/25)) 
									&& (((x-5)/25) < 4)
									&& ((display_order[col][((y-5)/25)])!=BLANK_ENTRY)
									&& (sys[(display_order[col][((y-5)/25)])].show_sys)
									&& (sys[(display_order[col][((y-5)/25)])].modifiable)
									&& (is_btn_on(&(sys[(display_order[col][((y-5)/25)])]),((x-5)/25)))) {
								if (!((dx==(x-5)/25)&&(dy==(y-5)/23))) {
									BTNUNLIGHT(base,col,dx,dy);
									dx = (x-5)/25;
									dy = (y-5)/25;
									selected = TRUE;
									BTNLIGHT(base,col,dx,dy);
								}
							}
							else if (selected) {
								selected = FALSE;
								BTNUNLIGHT(base,col,dx,dy);
								dx= -1; dy= -1;
							}
						}
						else {
							/* EXIT button */
							if ((x>COLUMN1) &&
									(x < (COLUMN2+WIDTH+(3*XBTNSPACING)+XSPACING-COLUMN1))) {
								exit_selected=TRUE;
								XDrawRectangle(mydisplay, base,
									gc, COLUMN1 + 2,
									((NUM_ADJ_SYS/2)+1)*YSPACING+STARTROW+2,
									(COLUMN2+WIDTH+(3*XBTNSPACING)+XSPACING-COLUMN1-4),
									HEIGHT-4);
							}
							else if (selected) {
								selected = FALSE;
								BTNUNLIGHT(base,col,dx,dy);
								dx= -1; dy= -1;
							}
						}
					}
					else if (selected) {
						selected=FALSE;
						BTNUNLIGHT(base,col,dx,dy);
						dx= -1; dy= -1;
					}

					if (!exit_selected) {
						XDrawRectangle(mydisplay, base, gcerase,
							COLUMN1 + 2,
							((NUM_ADJ_SYS/2)+1)*YSPACING+STARTROW+2,
							(COLUMN2+WIDTH+(3*XBTNSPACING)+XSPACING-COLUMN1-4),
							HEIGHT-4);
					}
					break;
				case ButtonPress:
					if (((XButtonEvent *)next_p)->button == Button1) {
						if (exit_selected) {
							if (exit_verify) {
								complete_ship();
								exit_verify = 2;
							}
							else {
								exit_verify=TRUE;
								draw_buildmenu(fontinfo,exit_verify);
							}
						}
						else {
							if (selected) {
								selected = act_on_button(col,dy,dx);
								if (!selected) {
									dx= -1;  dy= -1;
								}
							}
							if (exit_verify) {
								exit_verify=FALSE;
								draw_buildmenu(fontinfo,exit_verify);
							}
						}
					}
					break;
				default:
					break;
				}
			} while (exit_verify != 2);
	
			if (buildship) {
				XUnmapWindow(mydisplay, base);
				XFlush(mydisplay);
			}
			
			if (newship != VIEW_OLD_SHIP) {
				char *filename;

				filename = save_builtship(team_no,hull_size,newship);
				if (((newship == CONSTRUCT_NEW_SHIP)
								|| (newship == GET_DEFAULT))
							&& (num_ships < MAXSHIPS)) {
					list[num_ships] = (SHIPFILENAMESTRUCT *)
						malloc(sizeof(SHIPFILENAMESTRUCT));
					strncpy(list[num_ships]->short_shipname,
						ship.s_shipname,9);
					list[num_ships]->short_shipname[9] = '\0';
					strcpy(list[num_ships]->filename,filename);
					list[num_ships]->team_no   = team_no;
					list[num_ships]->hull_size = hull_size;
					++num_ships;
				}
			}
		}
	
		if (ship.s_hull > eff_size) valid_ship = FALSE;
		else if (newship == USE_OLD_SHIP) valid_ship = TRUE;
		else if (newship == GET_DEFAULT) hull_size = hold_hull;
	} while (!valid_ship);

	for (dy=0;dy<num_ships;dy++) free(list[dy]);
                
	if (dumpit) cat_builtship();
	if (printit) print_ship(playername,team_no);

	if (buildship) {
		XDestroyWindow(mydisplay, base);
		XCloseDisplay(mydisplay);
	}		
}
