/* search.c 15.1 06/08/90 10:07:02 */

/* Original code by Daryl Poe.
 *
 * Copyright (c) 1989 Daryl Poe
 *
 * Ported to X11 by Jim Andreas.
 * Tractors, treaties, and other features by Norm Gee.
 * Damage window and shield bitmaps by Tom LaStrange.
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting
 * documentation.  No representations are made about the
 * suitability of this software for any purpose.  It is
 * provided "as is" without express or implied warranty.
 */

#include <stdio.h>
#include "defs.h"
#include "weapon.h"
#include "system.h"
#include "ship.h"
#include "stats.h"
#include "player.h"
#include "torp.h"
#include "status.h"
#include "planet.h"
#include "phaser.h"
#include "message.h"
#include "shmem.h"
#include "data.h"

/* none of the options using "me" should be exercised by the daemon */
extern PLAYER *me;

/*
 *  targtype            data
 *  ========            ====
 *  TARG_PLAYER         search players
 *  TARG_PLANET         search planets
 *  TARG_CLOAK          include cloaked ships
 *  TARG_FRIENDLY       return only friendly players/planets
 *  TARG_UNFRIENDLY     return only unfriendly players/planets
 *  TARG_TRYDATA_FIRST  try <data> first to speed search
 *  TARG_VISIBLE        visible players only
 *  TARG_IMMUNE_OK      immune players too
 *  TARG_MEOWNER        my team owns the planet
 *  TARG_PLFLAGS_AND    planet->pl_flags & data TRUE
 *  TARG_NOTASTEROID	ignore asteroid fields
 */

int closest_planet_dist(x,y,mindist,targtype,data)
register int x,y;
int *mindist,targtype,data;
{
	register PLANET *p;
	register int dist,closest,dx,dy;
	int friendly_only   = ((targtype & TARG_FRIENDLY) != 0);
	int unfriendly_only = ((targtype & TARG_UNFRIENDLY) != 0);
	int meowner_only    = ((targtype & TARG_MEOWNER) != 0);
	int plflag_and_only = ((targtype & TARG_PLFLAGS_AND) != 0);
	int no_asteroids    = ((targtype & TARG_NOTASTEROID) != 0);

	logmsg5("search: looking for closest planet to %d%d, flags %d, data %d\n",
		x,y,targtype,data);

	if (targtype & TARG_TRYDATA_FIRST) {
		friendly_only = unfriendly_only = FALSE;
		*mindist = Ihypot(planets[data].pl_x-x, planets[data].pl_y-y);
		closest = data;
		logmsg2("search: trydata dist %d\n",*mindist);
	}
	else {
		*mindist = MAX_INT;
		closest  = -1;
	}

	for (p=planets; p<planets+MAXPLANETS; ++p) {
		if (friendly_only && !(team[p->pl_owner_no].t_treaty & me->p_team_mask))
			continue;
		if (unfriendly_only && (team[p->pl_owner_no].t_treaty & me->p_team_mask))
			continue;
		if (meowner_only && (p->pl_owner_no != me->p_team_no))
			continue;
		if (plflag_and_only && !(p->pl_flags & data)) continue;
		if (no_asteroids && (p->pl_flags & PLMASK_AST_FIELD)) continue;
		if ((dx = ABS(p->pl_x - x)) >= *mindist) continue;
		if ((dy = ABS(p->pl_y - y)) >= *mindist) continue;
		if ((dist = Ihypot(dx,dy)) <  *mindist) {
			logmsg3("search: found closer planet %d dist %d\n",
				p->pl_no,dist);
			*mindist = dist;
			closest  = p->pl_no;
		}
	}

	return(closest);
}


int closest_player_dist(x,y,mindist,targtype,data)
register int x,y;
int *mindist,targtype,data;
{
	register PLAYER *p;
	register int dist,closest,dx,dy;
	int no_cloaked      = ((targtype & TARG_CLOAK) == 0);
	int friendly_only   = ((targtype & TARG_FRIENDLY) != 0);
	int unfriendly_only = ((targtype & TARG_UNFRIENDLY) != 0);
	int visible_only    = ((targtype & TARG_VISIBLE) != 0);
	int immune_too      = ((targtype & TARG_IMMUNE_OK) != 0);
	unsigned int pmask;

	logmsg5("search: looking for closest player to %d%d, flags %d, data %d\n",
		x,y,targtype,data);

	if (targtype & TARG_TRYDATA_FIRST) {
		friendly_only = unfriendly_only = FALSE;
		*mindist = Ihypot(players[data].p_x-x, players[data].p_y-y);
		closest = data;
		logmsg2("search: trydata dist %d\n",*mindist);
	}
	else {
		*mindist = MAX_INT;
		closest  = -1;
	}

	for (p = players, pmask = 1;
			p < players+MAXPLAYER;
			++p, pmask <<= 1) {
		if (p->p_status != PALIVE) {
			if (!immune_too || (p->p_status != PIMMUNE)) continue;
		}
		if (p == me) continue;
		if (friendly_only && (p->p_team_mask & me->p_hostile_mask))
			continue;
		if (unfriendly_only && !(p->p_team_mask & me->p_hostile_mask))
			continue;
		if (no_cloaked
				&& ((p->p_flags & PFCLOAK)
					|| (p->p_loanflags & PFLCLOAK))) continue;
		if (visible_only && (!(me->p_visible_ships & pmask))) continue;
		if ((dx = ABS(p->p_x - x)) >= *mindist) continue;
		if ((dy = ABS(p->p_y - y)) >= *mindist) continue;
		if ((dist = Ihypot(dx,dy)) <  *mindist) {
			logmsg3("search: found closer player %d dist %d\n",
				p->p_no,dist);
			*mindist = dist;
			closest  = p->p_no;
		}
	}

	return(closest);
}


/* This routine tells whether the closest point on a vector from
 * refx,refy in the direction of "dir" and length "len" to the
 * to the target is within "delta" of the target.
 */
int is_vector_close_to_point(refx,refy,dir,len,targx,targy,delta)
int refx,refy;
COURSE dir;
int len,targx,targy,delta;
{
	register float ax,ay,cx,cy,d;
	register int bx,by;
	float toofar;

	logmsg5("is_vec_close.. refx: %d, refy%d, dir: %d, len: %d\n",
		refx,refy,(int)dir,len);
	logmsg4("  targx: %d, targy: %d, delta: %d\n",
		targx,targy,delta);
	/* can't possibly hit if this far away */
	toofar = (float) (len + delta);

	ax = (float) (refx - targx);
	if (ABS(ax) > toofar) return(FALSE);
	ay = (float) (refy - targy);
	if (ABS(ay) > toofar) return(FALSE);

	cx = Cos[dir];
	cy = Sin[dir];
	if ((d = -ax*cx - ay*cy) > toofar) {
		logmsg3(" d (%f) > toofar(%f).\n",d,toofar);
		return(FALSE);
	}
	/* d is the dist between refx,refy and the point */
	if (d <= 0.0) {
		/* go from the reference point, since dir is the wrong way */
		if ((bx = (int) ABS(ax)) > delta) return(FALSE);
		if ((by = (int) ABS(ay)) > delta) return(FALSE);
	}
	else {
		if ((bx = (int) ABS(ax + d*cx)) > delta) return(FALSE);
		if ((by = (int) ABS(ay + d*cy)) > delta) return(FALSE);
	}
	logmsg3(" bx: %d, by: %d\n",bx,by);
	if (Ihypot(bx,by) <= delta) {
		logmsg2(" hypot < delta (%d)\n",delta);
		return(TRUE);
	}
	else {
		logmsg2(" hypot > delta (%d)\n",delta);
		return(FALSE);
	}
}


/* This routine tells whether a planet is in the way of a torp or
 * beam and returns whether the intersection occurs.
 * If there is no intersection within a range of maxdist, 
 * FALSE is returned.
 */
int is_planet_in_way(x,y,dir,maxdist)
int x,y;
COURSE dir;
int maxdist;
{
	register PLANET *planet;

	logmsg5("is_planet_in_way(%d,%d,%d,%d)\n",x,y,dir,maxdist);

	for (planet=planets; planet<planets+MAXPLANETS; ++planet) {
		if (planet->pl_flags & PLMASK_NEBULA) continue;
		if (is_vector_close_to_point(
				x,y,dir,maxdist,
				planet->pl_x,planet->pl_y,(ORBDIST*3/4))) {
			logmsg1("Planet too close!\n");
			return(TRUE);
		}
	}

	logmsg1("No planets in way.\n");
	return(FALSE);
}


int shield_hit(wpncrs,pcrs)
COURSE wpncrs,pcrs;
{
	COURSE delta;

	delta = wpncrs - pcrs + 128;

	if (delta >= 224) return(FRONT_SHIELD);
	/* else */
	if (delta <= 32)  return(FRONT_SHIELD);
	/* else */
	if (delta <= 96)  return(RIGHT_SHIELD);
	/* else */
	if (delta <= 160) return(REAR_SHIELD);
	/* else */
	return(LEFT_SHIELD);
}
