/* rmove.c 15.1 06/08/90 10:06:45 */

/*

	Copyright (c) 1986 	Chris Guthrie

Permission to use, copy, modify, and distribute this
software and its documentation for any purpose and without
fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright
notice and this permission notice appear in supporting
documentation.  No representations are made about the
suitability of this software for any purpose.  It is
provided "as is" without express or implied warranty.

*/

/* Major overhaul by Daryl Poe.
 *
 * Copyright (c) 1989 Daryl Poe
 *
 * Ported to X11 by Jim Andreas.
 * Tractors, treaties, and other features by Norm Gee.
 * Damage window and shield bitmaps by Tom LaStrange.
 *
 * The above comments apply to this version as well, and this
 * notice must also appear in any copies or follow-ons.
 */

#include <stdio.h>
#include <signal.h>
#include <math.h>
#include "defs.h"
#include "daemon.h"
#include "weapon.h"
#include "system.h"
#include "ship.h"
#include "stats.h"
#include "player.h"
#include "torp.h"
#include "status.h"
#include "planet.h"
#include "phaser.h"
#include "message.h"
#include "shmem.h"
#include "data.h"

#define SIZEOF(s)		(sizeof (s) / sizeof (*(s)))
#define BOUNDRY_DIST	(WARP1*50)
#define STARBASE_RUNAWAY_ZONE	(GWIDTH*GWIDTH/64)
#define STARBASE_AVOID_ZONE		(GWIDTH*GWIDTH/16)
#define CIRCLE_FUSE		25
#define BADPLANET (PLMASK_BLACKHOLE|PLMASK_PULSAR|PLMASK_AST_FIELD)
#define FIRE_ALL_BEAMS	-1

/* directions */
#define UP				0
#define UP_RIGHT		32
#define RIGHT			64
#define DOWN_RIGHT		96
#define DOWN			128
#define DOWN_LEFT		160
#define LEFT			192
#define UP_LEFT			224

#define E_INTRUDER	0x01
#define E_TSHOT		0x02
#define E_PSHOT 	0x04
#define E_BASE		0x08

static int had_cloak,had_sphere,cloak_fuse=0;
static int want_cloak,want_ecm,want_shields;
static int too_close_range,too_far_range;

#define DROP_CLOAK_TO_FIRE 											\
{																	\
	had_cloak = me->p_flags & PFCLOAK;								\
	had_sphere = me->p_flags & PFSPHERE;							\
	me->p_flags &= ~(PFCLOAK|PFSPHERE);								\
}

#define RESTORE_CLOAK												\
{																	\
	if (had_cloak) CLOAK_ON;										\
	if (had_sphere) SPHERE_ON;										\
}

#define CLOAK_ON 													\
{																	\
	want_cloak = TRUE;												\
}
#define SPHERE_ON CLOAK_ON

#define CLOAK_OFF 													\
{ 																	\
	me->p_flags &= (~PFCLOAK); 										\
}		


#define ECM_ON 														\
{ 																	\
	want_ecm = TRUE;												\
}

#define ECM_OFF 													\
{																	\
	me->p_flags &= ~PFECM;											\
}


#define SHIELD_ON 													\
{ 																	\
	want_shields = TRUE;											\
}
#define SHIELDS_ON SHIELD_ON

#define SHIELD_OFF 													\
{ 																	\
	want_shields = FALSE;											\
}
#define SHIELDS_OFF	SHIELD_OFF

#define REPAIR_ON 													\
{ 																	\
	if (!(me->p_flags & PFREPAIR)) { 								\
		me->p_desspeed = 0; 										\
		me->p_flags |= PFREPAIR; 									\
		me->p_flags &= ~(PFSHIELD|PFBOMB|PFBEAMUP|PFBEAMDOWN); 		\
		logmsg2("%d: repair on\n",me->p_no); 						\
	} 																\
}
#define REPAIR_OFF 													\
{ 																	\
	if (me->p_flags & PFREPAIR) { 									\
		me->p_flags &= ~PFREPAIR; 									\
		logmsg2("%d: repair off\n",me->p_no); 						\
	} 																\
}		

#define EXIT_ORBIT													\
{ 																	\
	if (!orbiter && (me->p_flags & PFORBIT)) { 						\
		logmsg2("%d: Exiting orbit!\n",me->p_no);					\
		me->p_flags   &= ~(PFORBIT|PFBOMB); 						\
		me->p_desspeed = attack_speed; 								\
	} 																\
}

typedef struct {
	int e_pno;			/* his player # */
	int e_dist;			/* his distance */
	COURSE e_course;	/* course to enemy */
	COURSE e_tcourse;	/* torpedo intercept course to enemy */
	int e_ttime;        /* time before torp hits */
	int e_tx,e_ty;		/* mine placement point to hit enemy */
	unsigned int e_flags;
} ENEMY;


/* evasion types */
#define EVADE_NONE		0
#define EVADE_AWAY		1
#define EVADE_CLOAK		2
#define EVADE_SIMPLE 	3
#define EVADE_SHIELD	4
#define EVADE_OBJECT	5

/* patrol types */
#define PATROL_NONE			0
#define PATROL_ENEMY		1
#define PATROL_RANDOM		2
#define PATROL_REPAIRING	3
#define PATROL_SEEK_REPAIR	4
#define PATROL_BUDDY		5
#define PATROL_PLANETKILLER	6
#define PATROL_KILLING		7

extern int hostile;
extern int practice;
extern int attack_speed;
extern int offense;
extern int torp_impatience,beam_impatience;
extern int too_close_range,too_far_range;
extern int orbiter; /* TRUE if I am an orbiter */
extern int starbase; /* TRUE if I am an starbase */
extern int patrol_planet;
extern int want_buddy;

static ENEMY *oenemy,*denemy;   /* offense, defense */
extern PLAYER *last_target;
static PLAYER *ohim,*dhim;   /* offense, defense */
static SHIP   *ohisship,*dhisship;
static int	clock = 0;
static int course_deviation = 0;
static int patrol_type = PATROL_NONE, patrol_fuse = 0;
static int evasion_type = EVADE_NONE, evasion_fuse = 0;
static int closest_starbase_active;
static int myquadrant;
static int need_fuel,need_fuel_bad;
static int tractor_str; /* strength I'm being tractored */
static int friendly_tractor; /* is a teammate towing me? */
static int can_fire_beam; /* boolean */

#define NUM_TAUNTS	55
static int taunt_fuse = 0;
static char *taunt[] = {
	"You are nothing but a coward.",
	"Come over here and taste my torps.",
	"You cannot escape me.",
	"I have you now.",
	"Surrender or die.",

	"I do not take prisoners.",
	"Say your prayers.",
	"No one can defeat me!! HA HA HA!",
	"You cannot hurt me, you yellow dog.",
	"I will destroy you.",

	"Your pilots cannot match robot reflexes.",
	"Run to your Starbase, or face defeat.",
	"I have faster reactions and better weapons than you.",
	"I warn you -- stay away from me.",
	"In space, no one can hear you scream.",

	"You will be my next kill.",
	"How is it that they would let a loser like you captain a ship?",
	"Enjoy yourself while you can.",
	"Sterilize...  Sterilize...",
	"Do you mind if I call you \"space dust\"?",

	"We robots will just keep coming until you are all dead.",
	"If your race has death rites, perform them now.",
	"You look lonely.",
	"I would like to probe your ship.  Could you lower your shields?",
	"You\'ve been stirring up trouble.  I am here to put an end to it.",

	"How many times do we have to kill you before you learn your lesson?",
	"REVENGE!!!!",
	"You @&%#^@!",
	"Hit \'Q\' now and save me the trouble of killing you.",
	"I am the dtrek garbage collector.  You are garbage.",

	"Die! Die! Everybody Die!",
	"You can run, but you can\'t hide.",
	"You are infested by parasitic carbon based units. Must Destroy.",
	"Breath vaccum, you scum sucking pig.",
	"You are in a maze of twisty little passages, all different.",

	"No Mercy.",
	"Your marshmellow soft butt is mine.",
	"We came; We saw; We will conquer.",
	"I am He Who Walks on the Backs of Peasants with Sharp Claws.",
	"Where did you learn how to drive?",

	"Where did you learn how to shoot?",
	"You couldn\'t hurt me even with 20 locking photon torpedos, hosehead.",
	"I am invulnerable to your wimpy weapons.",
	"Don\'t mind me -- I\'m not really here.  This is an ECM image.",
	"You are a loser.  Prepare to lose.",

	"Hello. My name is Inigo Montoya. You killed my father. Prepare to die",
	"Eat my torps and die, filthy bag of mostly water.",
	"You are a disease, we are the vaccine.",
	"You fight like a dead kitten buried in concrete.",
	"Want to dance?",

	"Stop clogging up the space lanes, grandpa!",
	"I will toy with you no longer.",
	"Your death will be in vain.",
	"You should not have bothered our planets, fool.",
	"I would like to begin \'Robot Swarm Mode\'",
};


/* compute targeting info for the oplayer */
static void target_enemy(ep,player)
ENEMY *ep;
PLAYER *player;
{
	static int last_tooclose_pno = -1;

	ep->e_flags &= ~(E_TSHOT|E_PSHOT);

	if ((player->p_obscured != OBSCURED_NEBULA)
			|| (ep->e_dist < (SCALE*WINSIDE/2*NEBULA_VISIBILITY))) {
		/* not obscured by nebula */
		/* Get torpedo course to enemy */
		if ((ep->e_dist > (GWIDTH/10)) || !(myship->s_torps)) {
		}
		else if (me->p_torps_unloaded == myship->s_torps) {
			logmsg1("No torps loaded.\n");
		}
		else {
			if (me->p_fuel > myship->s_torpcost) {
				if (offense >= (75-((me->p_fuel
						-myship->s_torpcost)*100/myship->s_maxfuel)))
				{
					if (myship->s_torpclass == WPN_CHAFF)
					{
						if (compute_chaff_course(player,
								&ep->e_tcourse,&ep->e_ttime,20)) {
							if (!is_planet_in_way(me->p_x,me->p_y,
									me->p_dir,
									me->p_speed * WARP1 * ep->e_ttime))
							{
								ep->e_flags |= E_TSHOT;
							}
						}
					}
					else if (myship->s_torpclass == WPN_MINE)
					{
						if (compute_mine_placement(player,&ep->e_tx,
								&ep->e_ty))
						{
							ep->e_flags |= E_TSHOT;
						}
					}
					else
					{
						/* photon, plasma, or missile */
						if (compute_torp_course(player,
								&ep->e_tcourse,&ep->e_ttime,
								(myship->s_torpduration + 20)
								* torp_impatience / 100))
						{
							if (!is_planet_in_way(me->p_x,me->p_y,
									ep->e_tcourse, ep->e_ttime * WARP1
									* myship->s_torpspeed))
							{
								ep->e_flags |= E_TSHOT;
							}
						}
					}
				}
				else {
					logmsg1("Low on fuel.\n");
				}
			}
			else {
				logmsg1("Not enough fuel for one torp.\n");
			}
		}
							
		/* Get phaser shot status */
		if ((myship->s_beams != 0)
				&& (me->p_beams_unloaded < myship->s_beams)
				&& (me->p_fuel > myship->s_beamcost)) {
			can_fire_beam = TRUE;

			/* Is he within a good range? */
			if (ep->e_dist < myship->s_beamrange*beam_impatience/100)
				ep->e_flags |= E_PSHOT;
			/* does he have a down shield toward me? */
			else if ((player->p_shield[shield_hit(ep->e_course,player->p_dir)]
						< 10)
					&& (ep->e_dist < myship->s_beamrange))
				ep->e_flags |= E_PSHOT;
		}
		else {
			can_fire_beam = FALSE;
		}
	}

	if (ep->e_pno != last_tooclose_pno) {
		SHIP *s;
		/* compute what range is "too close" to the enemy */
		last_tooclose_pno = ep->e_pno;
		s = &players[ep->e_pno].p_ship;
		if (me->p_flags & PFPLKILLER) too_close_range = (ORBDIST*2);
		else if (s->s_torps > 0) {
			switch (s->s_torpclass) {
				case WPN_PHOTON:
					too_close_range = s->s_torpspeed * s->s_torpduration
						* (WARP1/2);
					break;
				case WPN_PLASMA:
					too_close_range = s->s_torpspeed
						* (s->s_torpdamage/s->s_torpdecay)
						* (WARP1/2);
					break;
				case WPN_MISSILE:
					too_close_range = s->s_torpspeed * s->s_torpduration
						* WARP1;
					break;
				case WPN_CHAFF:
					too_close_range = (GWIDTH/30);
					break;
				case WPN_MINE:
					too_close_range = WPN_MINE_RANGE;
					break;
			}
		}
		else too_close_range = 0;
	
		if (s->s_beams > 0)
			if (s->s_beamrange > too_close_range)
				too_close_range = s->s_beamrange;

		if (too_close_range > too_far_range)
			too_close_range = DETDIST;

		logmsg3("%d: new too_close_range: %d\n",
			me->p_no,too_close_range);
	}
}


static ENEMY *get_nearest(humans,ofns)
int *humans;
int ofns;
{
	int pcount = 0;  /* number of ships */
	register int i;
	int tdist,last_target_dist,homenum;
	register PLAYER *player;
	static ENEMY oebuf,debuf;
	ENEMY *ep;
	PLAYER **himptr;

	if (ofns) {
		/* offensive target */
		ep = &oebuf;
		himptr = &ohim;
	}
	else {
		/* defensive target */
		ep = &debuf;
		himptr = &dhim;
	}

	ep->e_pno = me->p_no;
	last_target_dist = ep->e_dist = 0x7fffffff;

	tractor_str = friendly_tractor = pcount = *humans = 0;  
	closest_starbase_active = FALSE;

	homenum  = 10*(myquadrant-1);

	/* is there a hostile starbase nearby? */
	for (player = players; player < players+MAXPLAYER; ++player) {
		/* make sure he's alive and not me */
		if ((player->p_status == PALIVE)
				&& ((player->p_flags & (PFORBIT|PFROBOT))
					== (PFORBIT|PFROBOT))
				&& (strcmp(player->p_ship.s_shipname,"Starbase") == 0)
				&& (player->p_hostile_mask & me->p_team_mask)) {
			closest_starbase_active = TRUE;
			break;
		}
	}

	for (i = 0, player = players; i < MAXPLAYER; i++, player++) {
		/* make sure he's alive and not me */
		if ((player->p_status != PALIVE)
				&& (player->p_status != PIMMUNE)) 
			continue;
		if (player == me) continue;
		if (player->p_tractors_on
				&& (player->p_who_tractor == me->p_no)) {
			tractor_str += player->p_tractors_on;
			if (player->p_team_no == me->p_team_no)
				friendly_tractor = TRUE;
		}
		if (ofns
				&& ((player->p_flags & (PFORBIT|PFROBOT))
					== (PFORBIT|PFROBOT))
				&& !(strcmp(player->p_ship.s_shipname,"Starbase"))) {
			/* don't attack starbases */
			continue;
		}
		pcount++;	/* Other players in the game */
		if (!(player->p_flags & PFROBOT)) ++(*humans);
		if (player->p_status == PIMMUNE) continue;
		if ((player->p_flags & PFPLKILLER)
				&& (me->p_team_no == INDEPENDENT)
				&& (offense < 85))
			continue;
		if ((me->p_hostile_mask & player->p_team_mask)
				|| (player->p_flags & PFPLKILLER)) {
			/* We have an enemy */

			/* Get his range */
			tdist = Ihypot(player->p_app_x - me->p_x,
				           player->p_app_y - me->p_y);

			/* make sure he's not too close to the nearest starbase
			 * (unless I am too). or in a nebula.
			 */
			if ((player->p_obscured == OBSCURED_NEBULA)
					&& (tdist > (SCALE*WINSIDE/2*NEBULA_VISIBILITY))) {
				tdist += (GWIDTH * 2);
			}
			else if (ofns
					&& closest_starbase_active
					&& (player->p_closest_planet == homenum)
					&& (me->p_closest_planet != homenum)) {
				logmsg3("%d: semi-ignoring player %d, too close to SB.\n",
					me->p_no,player->p_no);
				tdist += (GWIDTH * 2);
			}

			if (player == last_target) last_target_dist = tdist;

			if (tdist < ep->e_dist) {
				ep->e_pno    = i;
				ep->e_dist   = tdist;
			}
		}
	}

	/* if no targets, return -1 */
	if ((pcount == 0) || (ep->e_pno == me->p_no)) return ((ENEMY *) -1);

	if (ep->e_dist > (GWIDTH*2)) ep->e_dist -= (GWIDTH*2);

	*himptr = players + ep->e_pno;

	/* don't maliciously chase other robots or goners */
	if (last_target
			&& ((last_target->p_flags & PFROBOT)
				|| (last_target->p_status != PALIVE)
				|| !(me->p_hostile_mask & player->p_team_mask))) {
		last_target = *himptr;
	}
	else if (*himptr != last_target) {
		/* maybe I want to keep chasing the same target */
		if ((last_target_dist*torp_impatience/150) < ep->e_dist) {
			/* keep chasing the last target */
			ep->e_pno    = last_target->p_no;
			*himptr      = last_target;
			ep->e_dist   = last_target_dist =
				Ihypot(last_target->p_app_x - me->p_x,
					   last_target->p_app_y - me->p_y);
		}
		else {
			/* no, switch targets */
			last_target = *himptr;
		}
	}

	if (ofns
			&& (me->p_updates > taunt_fuse)
			&& (INTRAND(120) == 0)) {
		if (ep->e_dist < (GWIDTH/10)) {
			char str[80];
	
			taunt_fuse = me->p_updates + (5*60*10); /* 5 minutes */
			/* taunt him */
			sprintf(str,"%3s->%-3s",
				me->p_mapchars,(*himptr)->p_mapchars);
			pmessage(taunt[INTRAND(NUM_TAUNTS)],
				(*himptr)->p_no,MINDIV,str);
		}
		else if ((sharedMemory->mode & ROBOT_INVASION_MASK) 
				&& (me->p_team_no == INDEPENDENT)) {
			char str[80],addr[80];

			taunt_fuse = me->p_updates + (5*60*10); /* 5 minutes */
			switch (INTRAND(4)) {
				case 0:
					sprintf(str,"I\'m over here by %s.  Drop by some time.",
						planets[me->p_closest_planet].pl_name);
					break;
				case 1:
					sprintf(str,"I\'m having a party by %s and you\'re invited.",
						planets[me->p_closest_planet].pl_name);
					break;
				case 2:
					sprintf(str,"Meet me at %s at noon.  I\'m calling you out, %s.",
						planets[me->p_closest_planet].pl_name,
						(*himptr)->p_name);;
				case 3:
					sprintf(str,"The inhabitants of %s cower as I near their puny planet.",
						planets[me->p_closest_planet].pl_name);
					break;
			}
			/* taunt everyone */
			sprintf(addr,"%s->ALL", me->p_mapchars);
			pmessage(str,0,MALL,addr);
		}
	}

	/* Check to see if ship is in our space. */
	if (me->p_team_no == INDEPENDENT)
		ep->e_flags |= E_INTRUDER;
	else if (planets[(*himptr)->p_closest_planet].pl_owner_mask
			== me->p_team_mask)
		ep->e_flags |= E_INTRUDER;
	else if	((offense > 95) && !(INTRAND(8)))
		ep->e_flags |= E_INTRUDER;
	else ep->e_flags &= ~(E_INTRUDER);

	/* get course info */
	ep->e_course = Atan2(me->p_y - (*himptr)->p_app_y,
		(*himptr)->p_app_x - me->p_x);
	
	if ((!ofns)
			&& (strcmp((*himptr)->p_ship.s_shipname,"Starbase") == 0)
			&& (((*himptr)->p_flags & (PFORBIT|PFROBOT))
				== (PFORBIT|PFROBOT))) {
		/* mark as Starbase */
		ep->e_flags |= E_BASE;
	}

	return (ep);
}


/* return value: time that the torp will hit.  -1 if it won't */
/* galactic warp-adjusted units */
static int time_will_hit(px,py,pdx,pdy,tx,ty,tdx,tdy,fuse)
float px,py,pdx,pdy, tx,ty,tdx,tdy;
int fuse;
{
	/* u is time to "target crosses torp vector" */
	/* v is time to "torp crosses target vector" */
	register float u,v;

	logmsg5("time_will_hit-px,py,pdx,pdy: %f,%f,%f,%f\n",px,py,pdx,pdy);
	logmsg5(" -tx,ty,tdx,tdy: %f,%f,%f,%f\n",tx,ty,tdx,tdy);
	logmsg2(" -tfuse %d\n",fuse);
	if ((u = tdx*pdy - tdy*pdx)) {
		/* vectors are not parallel */
		v = ((px*pdy-py*pdx) - (tx*pdy - ty*pdx)) / u;
		if (pdx) u = (tx-px+v*tdx) / pdx;
		else u = (ty-py+v*tdy) / pdy;
	}
	else {
		/* check for zero vectors */
		if ((tdx == 0.0) && (tdy == 0.0)) {
			/* torp not moving -- just check for conincidence */
			if (px != tx) return(-1);
			if (py != ty) return(-1);
			else return(0);
		}
		if ((pdx == 0.0) && (pdy == 0.0)) {
			/* target not moving -- a simple problem */
			if (tdx) {
				if ((u = (px - tx) / tdx) < 0.0) return(-1);
				if (ABS(py-(ty + u*tdy)) > EXPDIST) return(-1);
				if (u > fuse) return(-1);
				else return((int) u);
			}
			else {
				if ((u = (py - ty) / tdy) < 0.0) return(-1);
				if (ABS(px-(tx + u*tdx)) > EXPDIST) return(-1);
				if (u > fuse) return(-1);
				else return((int) u);
			}
		}
		/* else course vectors are parallel */
		/* check for coincidence */
		if (pdy*(tx-px) == pdx*(ty-py)) {
			/* they are coincident */
			/* when will they intersect? */
			if ((tdx == pdx) && (tdy == pdy)) /* never */ return(-1);
			else if (tdy != pdy) u = v = (py-ty)/(tdy-pdy);
			else u = v = (px-ty)/(tdx-pdx);
		}
		else return(-1); /* parallel, non-coincident */
	}
	logmsg3(" u: %f, v: %f\n",u,v);

	if ((u < 0.0) || (v < 0.0)) return(-1);
	if (v > fuse) return(-1);         /* ran out of steam */
	if (hypot(tx-px+u*(tdx-pdx),ty-py+u*(tdy-pdy)) > EXPDIST)
		return(-1); /* missed abow or astern */

	logmsg2(" will hit in %f updates.\n",(u+v)/2.0);
	/* it may well hit */
	return((int)(u+v)/2);
}


static int time_track_will_hit(px,py,pdx,pdy,tx,ty,t_speed,fuse)
float px,py,pdx,pdy, tx,ty;
int t_speed,fuse;
/* galactic warp-adjusted units */
{
	float ex,ey,t,a,b,c;

	logmsg5("time_track_will_hit: p x,y,dx,dy: %f,%f,%f,%f\n",
		px,py,pdx,pdy);
	logmsg5(" torp x,y,spd,fuse %f,%f,%d,%d\n",tx,ty,t_speed,fuse);
	logflush();
	ex = px - tx;
	ey = py - ty;
	logmsg3(" ex,ey: %f,%f\n",ex,ey);

	a = pdx*pdx + pdy*pdy - t_speed*t_speed;
	logmsg2(" a = %f, ",a);

	b = 2.0 * (ex*pdx + ey*pdy);
	logmsg2(" b = %f, ",b);

	c = ex*ex + ey*ey;
	logmsg2(" c = %f, \n",c);
	logflush();

	t = min_quadratic_solution(a,b,c);

	logmsg2(" time %f\n",t);
	if ((int) t > fuse) return(-1);
	else if (t < 0.0) return(-1);
	else {
		logmsg1(" will hit!\n");
		return((int) t);
	}
}


/*           projectDamage
 *
 * Return value: projected damage
 * Output parameters: 
 *   dirP     -- average travel direction of approaching torps
 *   tracking -- are any of them tracking?
 *   closest  -- a pointer to the closest one
 *   lasttime -- how long I have to worry about them?
 */
static int projectDamage(dirP, tracking, closest, lasttime)
COURSE	*dirP;
int *tracking;
TORP **closest;
int *lasttime;
{
	register int i;
	register TORP *t;
	float tx, ty, tdx, tdy;
	float px, py, pdx, pdy;
	int dam, mytime;
	int damage = 0, mintime = MAX_INT;
	float dir_x = 0.0, dir_y = 0.0;

	px  = (float) me->p_x / WARP1;
	py  = (float) me->p_y / WARP1;
	pdx = (float) me->p_speed * Cos[me->p_dir];
	pdy = (float) me->p_speed * Sin[me->p_dir];
	*tracking = FALSE;
	*lasttime = 1;

	for (i = 0, t = &torps[dhim->p_no*MAXTORP]; i < MAXTORP; i++, t++) {
		switch (t->t_status) {
		case TFREE:
			continue;
		case TMOVE:
		case TSTRAIGHT:
			tx  = (float) t->t_x / WARP1;
			ty  = (float) t->t_y / WARP1;
			tdx = (float) t->t_speed * Cos[t->t_dir];
			tdy = (float) t->t_speed * Sin[t->t_dir];
			logmsg2("%d: checking straight torp\n",me->p_no);
			if ((mytime = time_will_hit(px,py,pdx,pdy,
					tx,ty,tdx,tdy,t->t_fuse)) >= 0) {
				logmsg4(" straight torpedo at %d,%d moving %d",
					t->t_x,t->t_y,t->t_dir)
				logmsg5(" will hit me (%d,%d course %d) for %d\n",
					me->p_x,me->p_y,me->p_dir,t->t_damage);

				if (mytime < mintime)   *closest  = t;
				if (mytime > *lasttime) *lasttime = mytime;
				damage += t->t_damage;
				dir_x  += Cos[t->t_dir];
				dir_y  += Sin[t->t_dir];
			}
			break;
		case TTRACK:
			logmsg2("%d: tracking torp detected.\n",me->p_no);
			if (t->t_target != me->p_no) break; /* not at me */
			tx  = (float) t->t_x / WARP1;
			ty  = (float) t->t_y / WARP1;
			if ((mytime = time_track_will_hit(
					px,py,pdx,pdy, tx,ty,t->t_speed,t->t_fuse)) >= 0) {
				logmsg2(" tracking torp will hit in %d turns\n",mytime);

				if (t->t_decay > 0) {
					if ((dam = t->t_damage - mytime * t->t_decay) < 0)
						break;
				}
				else dam = t->t_damage;

				*tracking = TRUE;
				if (mytime < mintime)   *closest  = t;
				if (mytime > *lasttime) *lasttime = mytime;
				/* assume it's moving right at me */
				dir_x += Cos[t->t_dir];
				dir_y += Sin[t->t_dir];
				damage += dam;
				logmsg5("%d: Tracking torp at %d %d projected to hit for %d\n",
					me->p_no,t->t_x,t->t_y,dam);
				logmsg2("   torp dir: %d\n",t->t_dir);
			}
			break;
		}
	}
	*dirP = Atan2((int)(-dir_y*1024),(int)(dir_x*1024));
	return(damage);
}


static void avoid_object(dist,runaway_dist,avoid_dist,x,y)
int dist,runaway_dist,avoid_dist,x,y;
{
	COURSE toward_obj,dir;
	int tmp;
	static int xlast,ylast;
	static COURSE desired_course;
	static unsigned int hadlock;
	float dx,dy;

	if (avoid_dist < (tmp = runaway_dist+WARP1*80)) avoid_dist = tmp;
	toward_obj = Atan2(me->p_y - y, x - me->p_x);
	logmsg5("%d: may be in avoidance zone of object at %d,%d; dist is %d.  ",
		me->p_no,x,y,dist);
	logmsg3("  desdir %d, toward %d\n",me->p_desdir,toward_obj);

	if ((x != xlast) || (y != ylast)) {
		hadlock = me->p_flags & (PFPLOCK|PFPLLOCK);
		desired_course = me->p_desdir;
		xlast = x; ylast = y;
		logmsg3("  holding state: hadlock 0x%x, des course: %d\n",
			hadlock,desired_course);
	}
	/* else use the old desired course */

	if (dist < runaway_dist) {
		/* move directly away from the object */
		logmsg2("%d: too close to object, running away.\n",me->p_no);
		me->p_flags &= ~(PFPLLOCK|PFPLOCK);
		me->p_desdir  = toward_obj + 128;
		me->p_desspeed = myship->s_maxspeed;
		evasion_fuse = 4;
		evasion_type = EVADE_OBJECT;
	}
	else if (dist < avoid_dist) {
		me->p_flags &= ~(PFPLLOCK|PFPLOCK);
		dir = desired_course - toward_obj;
		logmsg2("  In avoidance area, dir=%d\n",dir);
		/* if must cross path, circle around */
		if ((dir <= RIGHT) || (dir >= LEFT)) {
			if (dir <= RIGHT) {
				logmsg2("%d: circling object cntrclockwise.",me->p_no);
				me->p_desdir = toward_obj + 64;
				if (me->p_desspeed < 4) me->p_desspeed = 4;
				logmsg2("  newdirection: %d\n",me->p_desdir);
				evasion_fuse = 8;
				evasion_type = EVADE_OBJECT;
			}
			else {
				/* dir >= LEFT */
				logmsg2("%d: circling object clockwise.\n",me->p_no);
				me->p_desdir = toward_obj - 64;
				if (me->p_desspeed < 4) me->p_desspeed = 4;
				logmsg2("  newdirection: %d\n",me->p_desdir);
				evasion_fuse = 8;
				evasion_type = EVADE_OBJECT;
			}
			dx = Cos[me->p_desdir];
			dy = Sin[me->p_desdir];
			if ((x < avoid_dist) && (dx < 0)) {
				logmsg1("  reversing: obj near left edge\n");
				me->p_desdir += 128;
			}
			else if ((x > (GWIDTH-avoid_dist)) && (dx > 0)) {
				logmsg1("  reversing: obj near right edge\n");
				me->p_desdir += 128;
			}
			if ((y < avoid_dist) && (dy < 0)) {
				logmsg1("  reversing: obj near top edge\n");
				me->p_desdir += 128;
			}
			else if ((y > (GWIDTH-avoid_dist)) && (dy > 0)) {
				logmsg1("  reversing: obj near bottom edge\n");
				me->p_desdir += 128;
			}
		}
		else {
			/* no problem -- continue on course */
			if (me->p_desspeed < 4) me->p_desspeed = 4;
			evasion_type = EVADE_OBJECT;
			evasion_fuse = 8;
			logmsg2("%d: continue moving away from object.\n",me->p_no);
		}
	}
	else {
		/* no problem -- continue on old course */
		logmsg2(" no avoid problem, resuming course %d\n",
			desired_course);
		me->p_desdir = desired_course;
		me->p_flags |= hadlock;
		if (me->p_desspeed < 4) me->p_desspeed = 4;
		evasion_type = EVADE_NONE;
	}
}


static void check_starbase_dist()
{
	register int dx,dy;
	PLANET *home;

	/* Am I getting too close to the starbase? */
	logmsg4("Myquadrant: %d, sbactive: %d, evasion %d\n",myquadrant,
		closest_starbase_active,evasion_fuse);

	if (closest_starbase_active && evasion_fuse <= 0) {
		home = &planets[10*(myquadrant-1)];
		dx = me->p_x - home->pl_x;
		if ((dx = dx * dx) < STARBASE_AVOID_ZONE) {
			dy = me->p_y - home->pl_y;
			if ((dy = dy * dy) < STARBASE_AVOID_ZONE) {
				if ((dx += dy) < STARBASE_AVOID_ZONE) {
					logmsg5("%d: avoiding starbase at %s (%d,%d).\n",
						me->p_no,home->pl_name,home->pl_x,home->pl_y);
					avoid_object(dx,STARBASE_RUNAWAY_ZONE,
						STARBASE_AVOID_ZONE,home->pl_x,home->pl_y);
				}
			}
		}
	}
}


static void check_near_edge()
{
	/* check to see if we're near the boundries */
	if (me->p_x < BOUNDRY_DIST) {
		/* near left edge */
		if (me->p_y < BOUNDRY_DIST) {
			if ((me->p_desdir < RIGHT)
					|| (me->p_desdir > DOWN)) {
				/* he's got me in the upper left corner */
				logmsg1("  cornered, UL\n");
				if ((me->p_desdir >= DOWN_RIGHT)
						&& (me->p_desdir < UP_LEFT))
					me->p_desdir = DOWN;
				else me->p_desdir = RIGHT;
			}
		}
		else if (me->p_y > (GWIDTH-BOUNDRY_DIST)) {
			if (me->p_desdir > RIGHT) {
				/* he's got me in the lower left corner */
				logmsg1("  cornered, LL\n");
				if ((me->p_desdir >= UP_RIGHT)
						&& (me->p_desdir < DOWN_LEFT))
					me->p_desdir = RIGHT;
				else me->p_desdir = UP;
			}
		}
		else {
			if (me->p_desdir > DOWN) {
				/* just near left edge */
				logmsg1("  near left edge\n");
				if ((me->p_desdir >= RIGHT)
						&& (me->p_desdir < LEFT))
					me->p_desdir = DOWN;
				else me->p_desdir = UP;
			}
		}
	}
	else if (me->p_x > (GWIDTH-BOUNDRY_DIST)) {
		/* near right edge */
		if (me->p_y < BOUNDRY_DIST) {
			if ((me->p_desdir < DOWN)
					|| (me->p_desdir > LEFT)) {
				/* upper right corner */
				logmsg1("  cornered, UR\n");
				if ((me->p_desdir >= UP_RIGHT)
						&& (me->p_desdir < DOWN_LEFT))
					me->p_desdir = DOWN;
				else me->p_desdir = LEFT;
			}
		}
		else if (me->p_y > (GWIDTH-BOUNDRY_DIST)) {
			if ((me->p_desdir < LEFT)
					&& (me->p_desdir != UP)) {
				/* lower right corner */
				logmsg1("  cornered, LR\n");
				if ((me->p_desdir >= DOWN_RIGHT)
						&& (me->p_desdir < UP_LEFT)) 
					me->p_desdir = LEFT;
				else me->p_desdir = UP;
			}
		}
		else {
			if ((me->p_desdir < DOWN)
					&& (me->p_desdir != UP)) {
				/* just near right edge */
				logmsg1("  near right edge\n");
				if ((me->p_desdir >= RIGHT)
						&& (me->p_desdir < LEFT))
					me->p_desdir = DOWN;
				else me->p_desdir = UP;
			}
		}
	}
	else if (me->p_y < BOUNDRY_DIST) {
		/* Near top.
		 * Note that corner cases have already been handled.
		 */
		if ((me->p_desdir < RIGHT)
				|| (me->p_desdir > LEFT)) {
			logmsg1("  near top edge\n");
			if (me->p_desdir < DOWN) me->p_desdir = RIGHT;
			else me->p_desdir = LEFT;
		}
	}
	else if (me->p_y > (GWIDTH-BOUNDRY_DIST)) {
		/* Near bottom.
		 * Note that corner cases have already been handled.
		 */
		 if ((me->p_desdir > RIGHT)
				&& (me->p_desdir < LEFT)) {
			logmsg1("  near bottom edge\n");
			if (me->p_desdir < DOWN) me->p_desdir = RIGHT;
			else me->p_desdir = LEFT;
		}
	}
}


/* This function means that the robot has nothing better to do. */
static int patrol(ebuf)
ENEMY *ebuf;
{
	int x,y,dist,mindist,plno,buddy;
	PLAYER *player;
	PLANET *planet;

	evasion_fuse = 0;
	evasion_type = EVADE_NONE;

	CLOAK_OFF
	ECM_OFF

	logmsg1(" patrol\n");
	patrol_type = PATROL_NONE;

	if (orbiter) {
		patrol_fuse = 120;
		patrol_type = PATROL_RANDOM;
	}
	else if (want_buddy 
			&& !(planets[me->p_closest_planet].pl_flags & BADPLANET)) {
		mindist = (GWIDTH*2);
		/* look for a buddy to fly around with */
		logmsg1(" looking for buddy\n");
		for (player=players; player < players+MAXPLAYER; ++player) {
			if (player->p_status != PALIVE) continue;
			if (player->p_team_no != me->p_team_no) continue;
			if (player == me) continue;
			if (player->p_speed == 0) continue;
			if (player->p_engine_health < 200) continue;
			if (planets[player->p_closest_planet].pl_flags & BADPLANET)
				continue;
			if (mindist < (GWIDTH*2)) {
				if ((x = ABS(me->p_x - player->p_x)) > mindist)
					continue;
				if ((y = ABS(me->p_y - player->p_y)) > mindist)
					continue;
				if ((dist = x+y) > mindist) continue;
			}

			/* make sure I don't form a cycle */
			sem_lock();
			buddy = player->p_no;
			while ((buddy = players[buddy].p_buddy) != -1) {
				if (buddy == me->p_no) break;
			}
			if (buddy != me->p_no) {
				/* it's okay! */
				mindist = dist;
				me->p_buddy = player->p_no;
				patrol_type = PATROL_BUDDY;
			}
			sem_unlock();
		}
	}
	else if (me->p_flags & PFPLKILLER) {
		/* first try the last planet we visited */
		planet = planets + me->p_planet;
		if (planet->pl_flags & (PLMASK_HOME|PLMASK_BLACKHOLE
					|PLMASK_PULSAR|PLMASK_BARREN|PLMASK_NEBULA
					|PLMASK_AST_FIELD)) {
			planet = NULL;
			/* find a target planet */
			do {
				planet = planets + INTRAND(MAXPLANETS);
				if (planet->pl_flags & (PLMASK_HOME|PLMASK_BLACKHOLE
							|PLMASK_PULSAR|PLMASK_BARREN|PLMASK_NEBULA
							|PLMASK_AST_FIELD)) {
					planet = NULL;
				}
			} while (planet == NULL);
			me->p_planet = planet->pl_no;
		}
		logmsg4(" heading toward target planet %s at %d %d\n",
			planet->pl_name,planet->pl_x,planet->pl_y);
		me->p_flags |= PFPLLOCK;
		me->p_desspeed = 1;
		patrol_fuse = 1200;
		patrol_type = PATROL_PLANETKILLER;
	}

	if (patrol_type == PATROL_BUDDY) {
		logmsg2(" got buddy: %d\n",player->p_no);
		me->p_playerl  = me->p_buddy;
		me->p_flags   |= PFPLOCK;
		patrol_fuse    = 200;
		if (me->p_fuel < myship->s_maxfuel / 3) {
			me->p_desspeed = 3;
		}
		else if (me->p_fuel > myship->s_maxfuel * 2 / 3) {
			me->p_desspeed = INTRAND(3) + 4;
		}
		else {
			me->p_desspeed = INTRAND(3) + 3;
		}
		if (me->p_desspeed > players[me->p_buddy].p_speed - 2) {
			me->p_desspeed = players[me->p_buddy].p_speed + 1;
		}
	}
	else if (patrol_type == PATROL_NONE) {
		if (me->p_fuel < myship->s_maxfuel / 3) me->p_desspeed = 3;
		else me->p_desspeed = 4;

		plno = -1; dist = (GWIDTH*2);
		if (me->p_team_no != INDEPENDENT) {
			/* find an enemy close to your planet */
			for (player=players; player < players+MAXPLAYER; ++player) {
				if (player->p_status != PALIVE) continue;
				if (!(player->p_hostile_mask & me->p_team_mask))
					continue;
				if ((player->p_ship.s_maxspeed == 0) 
						&& (player->p_flags & PFROBOT))
					continue;
				if (planets[player->p_closest_planet].pl_owner_no
						== me->p_team_no) {
					/* he's near my planet for sure */
					if ((x = Ihypot(me->p_x - player->p_x,
							me->p_y - player->p_y)) < dist) {
						/* move toward the planet he's threatening */
						dist = x;
						plno = player->p_closest_planet;
						patrol_type = PATROL_ENEMY;
					}
				}
				else if (plno == -1) {
					/* haven't found a good one yet */
					/* find my closest planet to him */
					y = closest_planet_dist(player->p_x,player->p_y,
						&x,TARG_PLANET|TARG_MEOWNER,0);
					if (x < (INTERPLANETARY_DIST*3)) {
						/* he's near my planet for sure */
						if ((x = Ihypot(me->p_x - player->p_x,
								me->p_y - player->p_y)) < dist) {
							/* move toward the planet he's threatening */
							dist = x;
							plno = y;
							patrol_type = PATROL_ENEMY;
						}
					}
				}
			}

			if (plno == -1) {
				/* if not near my last target, continue. */
				if ((me->p_closest_planet == patrol_planet)
						|| (planets[patrol_planet].pl_owner_no 
							!= me->p_team_no)) {
					/* go toward a random planet that I own */
					int myplanets[MAXPLANETS],npl;
		
					npl = 0;
					for (planet = planets;
							planet < planets+MAXPLANETS;
							++planet) {
						if (planet->pl_owner_no == me->p_team_no)
							myplanets[npl++] = planet->pl_no;
					}

					if (npl) {
						plno = myplanets[INTRAND(npl)];
						logmsg2(" random myplanet patrol toward %s\n",
							planets[plno].pl_name);
						patrol_type = PATROL_RANDOM;
					}
				}
				else plno = patrol_planet;
			}
		}

		if (plno != -1) {
			x = planets[plno].pl_x - (GWIDTH/40) + INTRAND(GWIDTH/20);
			y = planets[plno].pl_y - (GWIDTH/40) + INTRAND(GWIDTH/20);
			patrol_planet = plno;
		}
		else {
			/* move somewhat towards the middle */
			x = (GWIDTH/3) + INTRAND(GWIDTH/3);
			y = (GWIDTH/3) + INTRAND(GWIDTH/3);
			patrol_type = PATROL_RANDOM;
			logmsg1(" completely random patrol\n");
		}
				
		me->p_desdir = Atan2(me->p_y - y, x - me->p_x);
		dist = Ihypot(me->p_x - x, me->p_y - y);
		if ((patrol_fuse = dist/me->p_desspeed/(WARP1*10) + 8) > 30) {
			patrol_fuse = 30;
		}

#ifdef DEBUG
		if (plno > -1) {
			logmsg5("%d: my closest planet to him is %d at %d,%d",
				me->p_no,plno,planets[plno].pl_x,planets[plno].pl_y);
			logmsg4(" dist %d; owned by %d, iam %d.\n",
				dist,planets[plno].pl_owner_no,me->p_team_no);
		}		
		else {
			logmsg2(" %d: random course\n",me->p_no);
		}
#endif

	}
	logmsg4(" %d: direction: %d, fuse: %d\n\n",
		me->p_no,me->p_desdir,patrol_fuse);

	return(TRUE);
}


static int compute_chaff_course(target,torp_course,torp_time,mydelay)
PLAYER *target;
COURSE *torp_course;
int *torp_time;
int mydelay;
{
	/* Tells whether a chaff or mine launched in py present direction
	 * is likely to hit.
	 */

	float	px, py, tx, ty, tdx, tdy, pdx, pdy;
	int mytime;

	px  = (float) target->p_x / WARP1;
	py  = (float) target->p_y / WARP1;
	pdx = (float) target->p_speed * Cos[target->p_dir];
	pdy = (float) target->p_speed * Sin[target->p_dir];

	tx = ((float) me->p_x + Cos[me->p_dir] * (EXPDIST*2.0)) / WARP1;
	ty = ((float) me->p_y + Sin[me->p_dir] * (EXPDIST*2.0)) / WARP1;
	if (myship->s_torpclass == WPN_CHAFF) {
		tdx = (float) me->p_speed * Cos[me->p_dir];
		tdy = (float) me->p_speed * Sin[me->p_dir];
	}
	else {
		tdx = tdy = 0.0;
	}

	if ((mytime = time_will_hit(px,py,pdx,pdy,tx,ty,tdx,tdy,mydelay)) >= 0){
		logmsg2("%d: chaff will hit.\n",me->p_no);
		logmsg5("  target at %f,%f, travelling %f,%f\n",px,py,pdx,pdy);
		logmsg5("  firer  at %f,%f, travelling %f,%f\n",tx,ty,tdx,tdy);
		*torp_course = me->p_dir;
		*torp_time   = mytime;
		return(TRUE);
	}
	else {
		return(FALSE);
	}
}


static int compute_torp_course(target,torp_course,torp_time,mydelay)
PLAYER *target;
COURSE *torp_course;
int *torp_time;
int mydelay;
/* return value:  is a shot advisable? */
{
	/* Solve to determine at what time a torp moving
	 * at myship->s_torpspeed could intersect a target at
	 * relative position relpos_x,relpos_y moving at a vector of
	 * hiscourse_x,hiscourse_y.  If that time is less than 'delay'
	 * go ahead and target and fire the torp.
	 */

	float relpos_x,relpos_y,hiscourse_x,hiscourse_y,a,b,c;
	int newpos_x,newpos_y;

	hiscourse_x = Cos[target->p_dir] * target->p_speed;
	hiscourse_y = Sin[target->p_dir] * target->p_speed;

	/* solve the quadratic equation */
	if (myship->s_torpclass == WPN_CHAFF) {
		relpos_x =  (target->p_app_x -
			(me->p_x + Cos[me->p_dir] * (EXPDIST*2))) / WARP1;
		relpos_y =  (target->p_app_y -
			(me->p_y + Sin[me->p_dir] * (EXPDIST*2))) / WARP1;
		a = target->p_speed*target->p_speed - me->p_speed*me->p_speed;
	}
	else {
		relpos_x =  (target->p_app_x - me->p_x) / WARP1;
		relpos_y =  (target->p_app_y - me->p_y) / WARP1;
		a = target->p_speed * target->p_speed
			- myship->s_torpspeed * myship->s_torpspeed;
	}
	b = 2.0 * (relpos_x * hiscourse_x + relpos_y * hiscourse_y);
	c = relpos_x * relpos_x + relpos_y * relpos_y;
	logmsg4("%d: compute_torp_course.  relpos: %f,%f\n",
		me->p_no,relpos_x,relpos_y);
	logmsg3(" hiscourse %f,%f\n",hiscourse_x,hiscourse_y);
	logmsg4(" a,b,c: %f,%f,%f\n",a,b,c);
	c = min_quadratic_solution(a,b,c);

	logmsg2(" soonest impact in %f seconds\n",(float) c);
	/* c is time that a torp will impact */
	if (c < 0.0) {
		if (mydelay < 0) {
			/* Can't catch up.
			 * Compromise between "toward target" and "parallel course".
			 */
			newpos_x = (relpos_x + hiscourse_x * 10.0) * WARP1;
			newpos_y = (relpos_y + hiscourse_y * 10.0) * WARP1;
			logmsg3(" cannot hit.  Compromise heading vector %d,%d\n",
				newpos_x,newpos_y);
			*torp_course = Atan2(newpos_y,newpos_x);
		}
		return(FALSE);
	}

	/* will it take too long to hit? */
	if (mydelay > 0)
		if (c > (float) mydelay) {
			logmsg1(" time > fuse.  returning FALSE\n");
			return(FALSE);
		}

	/* if here, torp is okay */
	newpos_x = target->p_app_x + c * hiscourse_x * WARP1;
	newpos_y = target->p_app_y + c * hiscourse_y * WARP1;
	*torp_course = Atan2(me->p_y - newpos_y, newpos_x - me->p_x);
	*torp_time   = (int) c;

	logmsg3("%d: found good torp at target %d\n",me->p_no,target->p_no);
	logmsg3("  course: %d, time: %d\n",*torp_course,*torp_time);
	return(TRUE);
}


static int compute_mine_placement(target,mine_x,mine_y)
PLAYER *target;
int *mine_x,*mine_y;
/* return value:  is a shot advisable? */
{

	/* Simple: put it in front of him */
	*mine_x = target->p_x
		+ Cos[target->p_dir]*target->p_speed*(WARP1*(WPN_MINE_ARM+5));
	if (*mine_x < 0) return(FALSE);
	if (*mine_x > GWIDTH) return(FALSE);

	*mine_y = target->p_y
		+ Sin[target->p_dir]*target->p_speed*(WARP1*(WPN_MINE_ARM+5));
	if (*mine_y < 0) return(FALSE);
	if (*mine_y > GWIDTH) return(FALSE);

	if (Ihypot(me->p_x - *mine_x, me->p_y - *mine_y)
			<= WPN_MINE_RANGE) {
		logmsg1("  good range for mine.\n");
		return(TRUE);
	}
	else {
		logmsg1("  too far for mine.\n");
		return(FALSE);
	}
}
	
					
static int rfire_torps()
{

	logmsg4("%d: firing %s at %d.\n",me->p_no,myship->s_torpname,
		ohim->p_no);

	REPAIR_OFF

	/* if seeking torps, try to get lockon */
	if (myship->s_torpclass == WPN_PLASMA) {
		/* if not cloaked */
		if (!(ohim->p_flags & PFCLOAK)) {
			/* lockon */
			me->p_wpnlock = oenemy->e_pno;
			me->p_flags |= PFWLOCK;
			if (fire_torp(oenemy->e_tcourse,oenemy->e_tx,oenemy->e_ty))
				return(1);
		}
	}
	else if (myship->s_torpclass == WPN_MISSILE) {
		/* if not cloaked */
		if (!(ohim->p_flags & PFCLOAK)) {
			int chance; 

			/* lockon */
			me->p_wpnlock = oenemy->e_pno;
			me->p_flags |= PFWLOCK;
			if (fire_torp(oenemy->e_tcourse,
					oenemy->e_tx,oenemy->e_ty)) {
				chance = 80;
				while ((INTRAND(100) < chance) 
						&& fire_torp(oenemy->e_tcourse,
								oenemy->e_tx,oenemy->e_ty)) {
					chance -= 20;
				}
				return(1);
			}
		}
	}
	else if (myship->s_torpclass == WPN_PHOTON) {
		if (fire_torp(oenemy->e_tcourse,oenemy->e_tx,oenemy->e_ty)) {
			if (me->p_team_no == INDEPENDENT) {
				int fired_it;
				/* now do burst fire */
				fired_it = 1;
				while ((INTRAND(100) < offense 
						* (myship->s_torpspeed*(WARP1*15)/(oenemy->e_dist+1)))
						&& fired_it) {
					fired_it = fire_torp(
						(COURSE) (oenemy->e_tcourse+INTRAND(3)-1),
						oenemy->e_tx,oenemy->e_ty);
				}
			}
			return(1);
		}
	}
	else if (fire_torp(oenemy->e_tcourse,oenemy->e_tx,oenemy->e_ty))
		return(1);
	/* else */

	/* torpedo wasn't fired, for some reason */
	return(0);
}


static int rfire_beams()
{
	int fired_it = FALSE;
	logmsg3("%d: phaser firing at %d\n", me->p_no,ohim->p_no);

	REPAIR_OFF
	if (fire_beam(oenemy->e_course,
			players[oenemy->e_pno].p_app_x,players[oenemy->e_pno].p_app_y,
			FIRE_ALL_BEAMS)) {
		fired_it = TRUE;
	}

	return(fired_it);
}


static int rfire_weapons()
{
	/* Fire weapons!!! */
	/*
    ** get_nearest() has already determined if torpedoes and torps
    ** will hit.  It has also determined the courses which torps and
    ** torps should be fired.  If so we will go ahead and shoot here.
    ** We will lose repair and cloaking for the rest of this interrupt.
    ** if we fire here.
    */

	if (ohim->p_flags & PFCLOAK) {
		/* he's cloaked -- I can't fire very often */
		if (INTRAND(10)) return(0);
	}


	DROP_CLOAK_TO_FIRE

	if (oenemy->e_flags & E_TSHOT) {
		if (rfire_torps()) {
			cloak_fuse = 5;
			return(TRUE);
		}
	}

	/* else */
	if (oenemy->e_flags & E_PSHOT) {
		if (rfire_beams()) {
			cloak_fuse = 5;
			return(TRUE);
		}
	}

	/* else didn't actually fire */
	/* let sphere and tractor work simultaneouly */
	RESTORE_CLOAK

	/* else */
	if (myship->s_tractor
			&& !need_fuel_bad
			&& (oenemy->e_dist < TRACTOR_RANGE)
			&& (!(players[oenemy->e_pno].p_flags & PFORBIT))) {
		if (oenemy->e_dist < too_close_range) {
			if (fire_tractors(oenemy->e_course,REPULSOR,myship->s_tractor)) {
				return(TRUE);
			}
		}
		else if (oenemy->e_dist > too_far_range) {
			if (fire_tractors(oenemy->e_course,TRACTOR,myship->s_tractor)) {
				return(TRUE);
			}
		}
	}

	/* else */
	me->p_tractors_on = 0;

	return(FALSE);
}


static COURSE good_shield_dir(torpdir,present_ok)
COURSE torpdir;
int *present_ok;
{
	int best_shield, current_shield, shield_str, i;

	shield_str = -1;

	logmsg1("good_shield_dir\n");
	for (i=0; i<4; ++i) {
		logmsg3(" shield %d: %d\n",i,me->p_shield[i]);
		if (me->p_shield[i] > shield_str) {
			best_shield = i;
			shield_str = me->p_shield[i];
		}
	}

	current_shield = shield_hit(torpdir,me->p_dir);
	/* if max is close to current, don't bother turning */
	if (ABS(me->p_shield[current_shield] - shield_str) < 20) {
		logmsg3(" current (%d) pretty good at %d.\n",current_shield,
			me->p_shield[current_shield]);
		*present_ok = TRUE;
		return(me->p_dir);
	}
	*present_ok = FALSE;

	logmsg2(" turning so it hits shield %d\n",best_shield);
	/* else */
	me->p_desspeed = 2;
	switch (best_shield) {
		case FRONT_SHIELD:
			return(torpdir+128);
		case RIGHT_SHIELD:
			return(torpdir+64);
		case REAR_SHIELD:
			return(torpdir);
		case LEFT_SHIELD:
			return(torpdir-64);
	}
}


static void turn_good_shield(avDir,worrytime)
COURSE avDir;
int worrytime;
{
	int shldok,timeturn;

	me->p_desdir = good_shield_dir(avDir,&shldok);
	if (!shldok && (me->p_desspeed > 0)) {
		if ((myship->s_turns == 0)
				|| (me->p_engine_health == 0)) {
			me->p_desspeed = 0;
		}
		else {
			timeturn = (int) ((float)
				((COURSE) (me->p_dir - me->p_desdir))
				* 1000.0 / myship->s_turns
				* 256.0  / me->p_engine_health);
			do {
				if ((timeturn<<me->p_desspeed) < worrytime)
					break;
			} while (--me->p_desspeed);
		}
	}
	me->p_flags &= ~(PFPLLOCK|PFPLOCK);
}


/* Avoid torps */
static int ravoid_torps()
{

	COURSE avDir;
	static TORP *torp;
	int worrytime,shield_str,damage,tracking,fleespeed,dist;

	if (practice) return(FALSE);

	if (--evasion_fuse > 0) {	/* we may still be avoiding */
		if (me->p_fuel < myship->s_maxfuel/20) me->p_desspeed = 3;
		else if (me->p_desspeed < 4) me->p_desspeed = 4;
		logmsg2("%d: Still evading.\n",me->p_no);
		if (can_fire_beam
				&& (torp != NULL)
				&& (torp->t_damage > myship->s_beamdamage/2)
				&& (torp->t_status != TFREE)
				&& (torp->t_status != TEXPLODE)
				&& (torp->t_class != WPN_CHAFF) 
				&& !(oenemy->e_flags & E_PSHOT)
				&& !(myship->s_sptechmask & STMASK_SPHERE)
				&& ((dist = Ihypot(me->p_x - torp->t_x,
						me->p_y - torp->t_y)) > (EXPDIST*2))
				&& (dist < (EXPDIST*6))) {
			/* blast it */
			REPAIR_OFF
			DROP_CLOAK_TO_FIRE
			if (fire_beam(Atan2(me->p_y - torp->t_y,torp->t_x - me->p_x),
					torp->t_x,torp->t_y,1)) {
				cloak_fuse   = 5;
				evasion_fuse = 0;
			}
			else {
				RESTORE_CLOAK
			}
		}

		return(TRUE);
	}

	/* else */
	if ((damage = projectDamage(&avDir,&tracking,&torp,&worrytime)) > 0) {
		EXIT_ORBIT
		SHIELD_ON
		shield_str = me->p_shield[shield_hit(torp->t_dir,me->p_dir)];
		logmsg5("%d: %d torpdam from %d moving dir = %d\n",
			me->p_no,damage, dhim->p_no, avDir);
		/* can I beam it to death? */
		if (can_fire_beam
				&& (sharedMemory->mode & DEFENSIVE_BEAM_MASK)
				&& !(oenemy->e_flags & E_PSHOT)
				&& (torp->t_class != WPN_CHAFF) 
				&& !(myship->s_sptechmask & STMASK_SPHERE)
				&& ((dist = Ihypot(me->p_x - torp->t_x,
						me->p_y - torp->t_y)) > (EXPDIST*2))
				&& (dist < (EXPDIST*6))) {
			/* blast it */
			REPAIR_OFF
			DROP_CLOAK_TO_FIRE
			if (fire_beam(Atan2(me->p_y - torp->t_y,torp->t_x - me->p_x),
					torp->t_x,torp->t_y,1)) {
				cloak_fuse = 5;
				damage -= myship->s_beamdamage;
			}
			else {
				RESTORE_CLOAK
			}
		}

		if (damage > (offense/2 + shield_str - 100)) {
			logmsg1(" evasive maneuvering.\n");
			/* Actually avoid Torps */
			evasion_fuse = worrytime/2+2;
			if (tracking) {
				ECM_ON
				if (orbiter) {
					CLOAK_ON
				}
				else {
					/* mobile ship */
					if (worrytime < 20) fleespeed = me->p_speed;
					else if (worrytime < 40)
						fleespeed = (me->p_speed
							+ myship->s_maxspeed
								* me->p_engine_health / 256) / 2;
					else fleespeed = myship->s_maxspeed
						* me->p_engine_health / 256;
					if ((fleespeed -= tractor_str) < 0) fleespeed = 0;
					if ((worrytime < 50) &&
							((COURSE) (torp->t_dir - me->p_dir + 128)
								> 64)) {
						/* must turn radically */
						fleespeed >>= 1;
					}

					logmsg2(" fleespeed: %d\n",fleespeed);

					if (myship->s_sptechmask & STMASK_SPHERE) {
						logmsg1(" trying sphere method\n");
						turn_good_shield(avDir,worrytime);
						CLOAK_ON
						evasion_type = EVADE_CLOAK;
						evasion_fuse = 5;
					}
					/* Do I have cloak? */
					else if ((myship->s_sptechmask & STMASK_CLOAK)
							&& (me->p_fuel > 2000)) {
						/* try to confuse them by cloaking */
						logmsg1(" trying cloak method\n");
						CLOAK_ON
						if (angdist(avDir+64,oenemy->e_tcourse) < 64)
							me->p_desdir = avDir+64;
						else me->p_desdir = avDir-64;
						me->p_desspeed = 6;
						evasion_type = EVADE_CLOAK;
					}
					/* can I avoid if I run straight away? */
					else if ((me->p_fuel > 1000) 
							&& time_track_will_hit(
							(float) me->p_x / WARP1,
							(float) me->p_y / WARP1,
							(float) fleespeed * Cos[torp->t_dir],
							(float) fleespeed * Sin[torp->t_dir],
							(float) torp->t_x / WARP1,
							(float) torp->t_y / WARP1,
							torp->t_speed,torp->t_fuse) < 0)
					{
						/* yes -- run straight away */
						logmsg1(" trying run-straight-away method\n");
						me->p_desdir   = avDir;
						me->p_desspeed = myship->s_maxspeed;
						evasion_type = EVADE_AWAY;
					}
					else {
						/* no escape -- take them on a good shield */
						logmsg1(" no escape.  take on good shield\n");
						turn_good_shield(avDir,worrytime);
						evasion_fuse = 4;
						evasion_type = EVADE_SHIELD;
					}
				}
			}
			else {
				/* not tracking */
				if (myship->s_sptechmask & STMASK_SPHERE) CLOAK_ON
				/* just turn to avoid */
				if (!orbiter) {
					if (myship->s_sptechmask & STMASK_SPHERE) {
						CLOAK_ON
						evasion_type = EVADE_CLOAK;
						evasion_fuse = 5;
						turn_good_shield(avDir,worrytime);
					}
					else if (((me->p_speed < 4)
								&& (worrytime < 30))
							|| (worrytime < 10)) {
						logmsg1(" turning shield to torp\n");
						turn_good_shield(avDir,worrytime);
						evasion_fuse = 4;
						evasion_type = EVADE_SHIELD;
					}
					else {
						/* sheer off */
						logmsg1(" changing course rapidly\n");
						if (INTRAND(2)) me->p_desdir += 64;
						else me->p_desdir -= 64;
						evasion_type = EVADE_AWAY;
						evasion_fuse = 5;
						me->p_desspeed = 4;
					}
				}
			}

			logmsg3(" evading to dir %d, speed %d\n",
				me->p_desdir,me->p_desspeed);
			return(TRUE);
		}
		else {
			/* light damage expected */
			/* simple evasion */
			logmsg2("%d: simple evasion\n",me->p_no);
			if (evasion_type == EVADE_NONE) {
				if (INTRAND(2)) avDir = 20;
				else avDir = -20;
				me->p_desdir += avDir;
				course_deviation += avDir;
				if (tracking) {
					ECM_ON
					CLOAK_ON
				}		
				evasion_fuse = 0;
				evasion_type = EVADE_SIMPLE;
				return(TRUE);
			}
			else return(TRUE);
		}
	}

	/* no damage expected */
	evasion_type = EVADE_NONE;
	evasion_fuse = 0;
	return(FALSE);
}


static int rrun_away()
{
	/*
    ** The robot has taken damage.  He will now attempt to run away from
    ** the closest player.  This obviously won't do him any good if there
    ** is another player in the direction he wants to go.
    ** Note that the robot will not run away if he dodged torps, above.
    ** The robot will lower his shields in hopes of repairing some damage.
    */

	int enemy_near,planet_near;

	if (orbiter) return(FALSE);

	if ((denemy->e_dist < (500*WARP1)) && myship->s_repairable)
		enemy_near = TRUE;
	else enemy_near = FALSE;

	if ((me->p_closest_planet_dist < (ORBDIST*2))
			&& ((me->p_hostile_mask)
				& (planets[me->p_closest_planet].pl_owner_mask))
			&& !(planets[me->p_closest_planet].pl_flags
				& (PLMASK_BLACKHOLE|
					PLMASK_BARREN|PLMASK_NEBULA))) { 
		planet_near = TRUE;
	}
	else {
		planet_near = FALSE;
	}

	if (enemy_near || planet_near) {
		if (me->p_fuel < myship->s_maxfuel/5) me->p_desspeed = 4;
		else if (me->p_etemp > 900) me->p_desspeed = 4;
		else me->p_desspeed = 8;

		if (enemy_near) {
			if (denemy->e_dist < (500*WARP1)) {
				CLOAK_ON
			}		
			ECM_ON
		}
		REPAIR_OFF
		SHIELD_ON

		me->p_desdir = denemy->e_course-128;
		logmsg2(" Enemy is near, running away to %d\n",me->p_desdir);
		return(TRUE);
	}

	return(FALSE);
}


/* Attack. */
/*
** The robot will check and see if the nearest
** enemy fits any of its criterion for attack.  If it does, the robot
** will speed in and deliver a punishing blow.  (Well, maybe)
*/
static int rattack()
{
	static COURSE circle_dir = 64;
	int attackhim = 0;
	int junk;

	/* Don't attack if we're low on fuel */
	if (need_fuel_bad) return(FALSE);

	/* Independents don't go after planetkillers, though they
	 * will fire on them if given the opportunity.
	 */
	if ((me->p_team_no == INDEPENDENT)
			&& (ohim->p_flags & PFPLKILLER))
		return(FALSE);

	/* try to find an excuse to attack */
	/* is he an intruder? */
	if (oenemy->e_flags & E_INTRUDER) attackhim = 1;
	/* is he pretty close? */
	else if (oenemy->e_dist < offense*WARP1) attackhim = 1;
	/* is he not so close but pretty damaged (killer instinct)? */
	else if (oenemy->e_dist < offense*(3*WARP1)) {
		if ((me->p_shield[FRONT_SHIELD] - myship->s_system[1].current_size
				+ 100) * offense/150 >
				(ohim->p_shield[FRONT_SHIELD] - 
				ohisship->s_system[1].current_size + 50 + INTRAND(100)))
			attackhim = 1;
			logmsg1("killer instinct forces attack.\n");
	}

	if (attackhim) {
		if (!orbiter) EXIT_ORBIT
		logmsg5("%d: (hull%d,sh%d) attacking %c",
			me->p_no, myship->s_system[1].current_size,
			me->p_shield[FRONT_SHIELD], teamlet[ohim->p_team_no])
		logmsg5("%d %16s (dam%d,sh%d)\n", ohim->p_no,
			ohim->p_login, ohisship->s_system[1].current_size,
			ohim->p_shield[FRONT_SHIELD]);
		logmsg5("  at %d,%d; course: %d; speed %d\n",
			ohim->p_x,ohim->p_y,ohim->p_dir,ohim->p_speed);

		/* CLOAK, ECM, etc. ON? */
		if ((denemy->e_dist < dhisship->s_beamrange*4/3) 
				|| (oenemy->e_dist < myship->s_beamrange*4/3)
				|| (oenemy->e_dist < (GWIDTH/20))) {
			SHIELD_ON
			ECM_ON
			CLOAK_ON
		}
		else {
			CLOAK_OFF
			ECM_OFF
		}

		/* COURSE */
		if (!orbiter) {
			if ((oenemy->e_dist < (500*WARP1))
					&& (myship->s_torpclass == WPN_CHAFF)
					&& (me->p_torps_unloaded != myship->s_torps)) {
				/* head on intercept course */		
				compute_torp_course(ohim,&me->p_desdir,&junk,-1);
				logmsg4("%d: chaff course to %d, intercept course %d\n",
					me->p_no,oenemy->e_course,me->p_desdir);
			}
			else {
				if (oenemy->e_dist > too_far_range) {
					/* head toward him */
					me->p_desdir = oenemy->e_course;
				}
				else if (oenemy->e_dist > too_close_range) {
					/* circle him at optimal range */
					me->p_desdir = oenemy->e_course + circle_dir;
					if (INTRAND(CIRCLE_FUSE) == 0)
						circle_dir = -circle_dir;
				}
				else {
					/* too close -- move away */
					me->p_desdir = oenemy->e_course+128;
				}
	
				if ((oenemy->e_dist > (GWIDTH/10))
						&& (me->p_closest_planet_dist > (ORBDIST*2))) {
					SHIELD_OFF
				}
				else {
					SHIELD_ON
				}
	
				/* randomly perturb course occasionally */
				if (!(INTRAND(10)))
					course_deviation = INTRAND(20) + INTRAND(20) - 20;
				me->p_desdir += course_deviation;
			}
	
			/* SPEED */
			if (angdist(me->p_desdir,me->p_dir) > 64)
				me->p_desspeed = 4;
			else if (me->p_etemp > 900)
				me->p_desspeed = 3;
			else if (me->p_fuel < myship->s_maxfuel/5)
				me->p_desspeed = 4;
			else if (oenemy->e_dist > (WARP1*1000))
				me->p_desspeed = 4;
			else
				me->p_desspeed = attack_speed;

			patrol_type = PATROL_ENEMY;
			patrol_fuse = 6;
		}
	}
	
	return(attackhim);
}


#define MAX_REPAIR_TRIP (GWIDTH/10)

static int rrepair()
{
	/* Repair if necessary (we are safe) */
	int repair_now,need_repair;
	int plflags,mindist,pno;

	if (me->p_flags & PFRED) return(FALSE);

	repair_now = FALSE;

	need_repair = (myship->s_repairable
		|| (myship->s_damaged && (me->p_flags & PFORBIT)
			&& (planets[me->p_planet].pl_flags & PLMASK_REPAIR)));

	if (orbiter) {
		/* repair any time I'm damaged or low on fuel */
		if (need_repair || need_fuel) repair_now = TRUE;
		else return(FALSE);
	}
	else {
		if (!need_repair && !need_fuel_bad) return(FALSE);

		/* check for a nearby friendly repair/fuel planet */
		if (need_repair) {
			plflags = PLMASK_REPAIR;
		}
		else {
			plflags = 0;
		}

		if (need_fuel) {
			plflags |= PLMASK_FUEL;
		}

		if ((me->p_flags & PFORBIT)
				&& (planets[me->p_planet].pl_flags & plflags)) {
			/* Already orbiting good repair/fuel planet */
			logmsg3("%d: repairing at planet %d\n",
				me->p_no,me->p_planet);
			repair_now = TRUE;
		}
		else if (me->p_flags & PFPLKILLER) {
			repair_now = TRUE;
		}
		else {
			/* find closest good planet */
			if ((myship->s_maxspeed*me->p_engine_health/256 >= 2)
					&& (me->p_fuel > 1000)) {
				pno = closest_planet_dist(me->p_x,me->p_y,
					&mindist,TARG_PLANET|TARG_FRIENDLY|TARG_NOTASTEROID|TARG_PLFLAGS_AND,
					plflags);
				if ((pno >= 0)
						&& (mindist < MAX_REPAIR_TRIP) 
						&& !(planets[me->p_closest_planet].pl_flags
							& BADPLANET)) {
					logmsg5("%d: heading toward repair planet %d at %d,%d\n",
						me->p_no,pno,
						planets[pno].pl_x,planets[pno].pl_y);
					me->p_flags |= PFPLLOCK;
					me->p_planet = pno;
					me->p_desspeed = 4;
					patrol_fuse = mindist/(5*WARP1);
					patrol_type = PATROL_SEEK_REPAIR;
					repair_now  = FALSE;
				}
				else {
					/* too far or no good planet */
					/* just repair in place */
					if (pno < 0) {
						logmsg1(" no good planet. repair in place\n");
					}
					else {
						logmsg1(" planet too far. repair in place\n");
					}
					repair_now = TRUE;
				}
			}
			else {
				/* too damaged -- repair in place */
				logmsg2("%d: too damaged; repairing in place\n",
					me->p_no);
				repair_now = TRUE;
			}
		}
	}

	if (repair_now) {
		logmsg1("  repairing now.\n");
		if (patrol_type == PATROL_SEEK_REPAIR) {
			me->p_flags &= ~PFPLLOCK;
			patrol_fuse = 0;
		}
		me->p_desspeed = 0;
		if (me->p_speed == 0) {
			CLOAK_OFF
			want_cloak = want_ecm = FALSE;
			ECM_OFF
			SHIELD_OFF
			REPAIR_ON
			patrol_fuse = 20;
			patrol_type = PATROL_REPAIRING;
			logmsg3("%d: repairing damage in place. hull: %d\n",
				me->p_no,myship->s_system[1].current_size);
		}
		return(TRUE);
	}

	/* else */
	if (me->p_flags & PFREPAIR) {
		REPAIR_OFF
		SHIELD_ON
	}

	if (patrol_type == PATROL_SEEK_REPAIR) return(TRUE);
	/* else */
	return(FALSE);
}


static void check_course()
{
	PLANET *planet;
	PLAYER *player;

	if (!orbiter) {
		check_starbase_dist();
		planet = &planets[me->p_closest_planet];
		/* put up shields if I'm close to a hostile planet */
		if ((me->p_closest_planet_dist <= (ORBDIST*2))
				&& (me->p_hostile_mask & planet->pl_owner_mask)) {
			SHIELD_ON
			logmsg1("close to planet, shield on\n");
		}
		if (planet->pl_flags & (PLMASK_BLACKHOLE|PLMASK_PULSAR
				|PLMASK_AST_FIELD|PLMASK_NEBULA)) {
			if (planet->pl_flags & PLMASK_BLACKHOLE) {
				logmsg5("%d: avoiding blackhole %s (%d,%d).\n",
					me->p_no,planet->pl_name,planet->pl_x,planet->pl_y);
				avoid_object(me->p_closest_planet_dist,
					NEBULA_RADIUS,INTERPLANETARY_DIST,
					planet->pl_x,planet->pl_y);
			}
			else if (planet->pl_flags & PLMASK_PULSAR) {
				if (me->p_closest_planet_dist < (NEBULA_RADIUS*11/10))
					SHIELD_ON
				logmsg5("%d: avoiding pulsar %s (%d,%d).\n",
					me->p_no,planet->pl_name,planet->pl_x,planet->pl_y);
				avoid_object(me->p_closest_planet_dist,
					NEBULA_RADIUS,INTERPLANETARY_DIST,
					planet->pl_x,planet->pl_y);
			}
			else if (planet->pl_flags & PLMASK_AST_FIELD) {
				if (me->p_closest_planet_dist < AST_FIELD_RADIUS) {
					me->p_desspeed = 2;
					SHIELD_ON
				}
				logmsg5("%d: avoiding ast field %s (%d,%d).\n",
					me->p_no,planet->pl_name,planet->pl_x,planet->pl_y);
				avoid_object(me->p_closest_planet_dist,
					AST_FIELD_RADIUS,INTERPLANETARY_DIST,
					planet->pl_x,planet->pl_y);
			}
			else if ((planet->pl_flags & PLMASK_NEBULA)
					&& (offense < 90)
					&& ((myship->s_sptechmask & STMASK_SPHERE)
						|| (myship->s_torps
						&& ((myship->s_torpclass == WPN_PLASMA)
							|| (myship->s_torpclass == WPN_MISSILE))))) {
				/* avoid nebulae */
				logmsg5("%d: avoiding nebula %s (%d,%d).\n",
					me->p_no,planet->pl_name,planet->pl_x,planet->pl_y);
				avoid_object(me->p_closest_planet_dist,
					NEBULA_RADIUS,(NEBULA_RADIUS*3/2),
					planet->pl_x,planet->pl_y);
			}
		}
		check_near_edge();
	}
	else {
		/* ORBITER */
		if (!starbase) {
			if (friendly_tractor) {
				me->p_flags &= ~(PFORBIT|PFBOMB);
			}
			else if ((patrol_fuse)
					&& ((me->p_updates % 10) == 1)) {
				/* haven't updated friendly tractor lately */
				for (player=players; player<players+MAXPLAYER; ++player) {
					if ((player->p_status != PALIVE)
							&& (player->p_status != PIMMUNE)) 
						continue;
					if (player == me) continue;
					if (player->p_tractors_on
							&& (player->p_who_tractor == me->p_no)
							&& (player->p_team_no == me->p_team_no)) {
						friendly_tractor = TRUE;
						me->p_flags &= ~(PFORBIT|PFBOMB);
						break;
					}
				}
			}
			
			if ((!friendly_tractor)
					&& !(me->p_flags & PFORBIT)
					&& (me->p_closest_planet_dist < ORBDIST)
					&& (planets[me->p_closest_planet].pl_owner_no ==
						me->p_team_no)) {
				me->p_flags |= PFORBIT;
				me->p_planet = me->p_closest_planet;
			}
		}
	}
}


static void final_checks()
{
	if (evasion_type != EVADE_OBJECT) check_course();
	else if (--evasion_fuse <= 0) {
		evasion_type = EVADE_NONE;
	}

	if (want_shields && (patrol_type != PATROL_KILLING)) {
#ifdef DEBUG
		if (!(me->p_flags & PFSHIELD)) {
			logmsg1("  shields on\n");
		}
#endif
		me->p_flags |= PFSHIELD;
		me->p_flags &= ~(PFBOMB|PFREPAIR|PFBEAMUP|PFBEAMDOWN);
	}
	else if (!(me->p_flags & PFRED)) {
#ifdef DEBUG
		if ((me->p_flags & PFSHIELD)) {
			logmsg1("  shields off\n");
		}
#endif
		me->p_flags &= ~PFSHIELD;
	}

	if (((cloak_fuse--) <= 0)
			&& want_cloak
			&& (me->p_fuel > (20*STFUEL_SPHERE))
			&& (myship->s_sptechmask & STMASK_SPHERE)) {
#ifdef DEBUG
		if (me->p_flags & PFCLOAK) {
			logmsg1("   CLOAK off\n");
		}
		if (me->p_flags & PFECM) {
			logmsg1("   ECM off\n");
		}
		if (!(me->p_flags & PFSPHERE)) {
			logmsg1("   SPHERE on\n");
		}
#endif
		me->p_flags &= ~(PFECM|PFCLOAK); 
		me->p_flags |= PFSPHERE; 
		cloak_fuse = 0;
	} 
	else if ((cloak_fuse <= 0)
			&& want_cloak
			&& (me->p_fuel > (20*STFUEL_CLOAK))
			&& (myship->s_sptechmask & STMASK_CLOAK)) {
#ifdef DEBUG
		if (me->p_flags & PFSPHERE) {
			logmsg1("   SPHERE off\n");
		}
		if (me->p_flags & PFECM) {
			logmsg1("   ECM off\n");
		}
		if (!(me->p_flags & PFCLOAK)) {
			logmsg1("  CLOAK on\n");
		}
#endif
		me->p_flags &= ~(PFECM|PFSPHERE); 
		me->p_flags |= PFCLOAK; 
		cloak_fuse = 0;
	} 
	else {
#ifdef DEBUG
		if (me->p_flags & PFSPHERE) {
			logmsg1("   SPHERE off\n");
		}
		if (me->p_flags & PFCLOAK) {
			logmsg1("   CLOAK off\n");
		}
#endif
		me->p_flags &= ~(PFCLOAK|PFSPHERE);
		if (want_ecm
				&& (me->p_fuel > (20*STFUEL_ECM))
				&& (myship->s_sptechmask & STMASK_ECM)) {
#ifdef DEBUG
			if (!(me->p_flags & PFECM)) {
				logmsg1("ECM on");
			}
#endif
			me->p_flags |= PFECM;
		}
		else {
#ifdef DEBUG
			if ((me->p_flags & PFECM)) {
				logmsg1("ECM off");
			}
#endif
			me->p_flags &= ~PFECM;
		}
	}
}


/*******************************************************************
 *
 *                          RMOVE
 *
 *******************************************************************/
void rmove()
{
	int moved, humans;
	static int timer = -1,lastupdate,samelastcount;

	logmsg2("\n%d: rmove beginning", me->p_no);
	logmsg5("  at %d,%d; course: %d; speed %d,",
		me->p_x,me->p_y,me->p_dir,me->p_speed);
	logmsg2(" flags 0x%x\n",me->p_flags);

	clock++;
	/* Check that I'm alive */
	if (me->p_status == PEXPLODE) {
		logmsg2("%d: exploding\n",me->p_no);
		mysignal(SIGALRM, SIG_IGN);
		while (me->p_status == PEXPLODE) /* do nothing */ ;
		while (me->p_ntorp > 0) /* do nothing */ ;
		me->p_status = PFREE;
		exit(0);
	}
	else if (me->p_status == PDEAD) {
		logmsg2("%d: dead\n",me->p_no);
		logflush();
		mysignal(SIGALRM, SIG_IGN);
		me->p_status = PFREE;
		exit(0);
	}
	else if (me->p_status == PIMMUNE) {
		me->p_flags &= ~(PFENTER|PFEXIT);
		me->p_status = PALIVE;
	}

	/* keep ghostbuster away */
	me->p_ghostbuster = 0;

	need_fuel_bad = (me->p_fuel < myship->s_maxfuel/6);
	need_fuel     = (me->p_fuel < myship->s_maxfuel/2);
	want_cloak = want_ecm = FALSE;
	if (me->p_fuel > myship->s_maxfuel*9/10) {
		want_shields = TRUE;
	}
	else {
		want_shields = FALSE;
	}

	if (patrol_fuse > 0) {
		if ((me->p_flags & PFRED)
				|| ((me->p_flags & PFYELLOW)
					&& (patrol_type != PATROL_KILLING))) {
			logmsg2("%d: Alert!  stopping patrol.\n",me->p_no);
			patrol_fuse = 0;
		}
		else if (patrol_type == PATROL_REPAIRING) {
			CLOAK_OFF
			ECM_OFF
			SHIELD_OFF
			REPAIR_ON
			logmsg3("%d: repairing damage on patrol. hull: %d\n",
				me->p_no,myship->s_system[1].current_size);
			if (!(myship->s_repairable)
					&& !(need_fuel)
					&& !(myship->s_damaged && (me->p_flags & PFORBIT)
						&& planets[me->p_planet].pl_flags
							& PLMASK_REPAIR)) {
				/* done repairing */
				logmsg2("%d: done repairing!\n",me->p_no);
				patrol_fuse = 0;
			}
		}	
		else if (patrol_type == PATROL_SEEK_REPAIR) {
			if (me->p_flags & PFORBIT) {
				logmsg1("  made it to repair planet.\n");
				CLOAK_OFF
				ECM_OFF
				SHIELD_OFF
				REPAIR_ON
				patrol_fuse = 100;
				patrol_type = PATROL_REPAIRING;
			}
			else if ((evasion_type == EVADE_NONE)
					|| (evasion_fuse <= 0)) {
				me->p_flags |= PFPLLOCK;
				logmsg2("%d: heading to repair planet.\n",me->p_no);
			}
			else {
				logmsg2("  evading, fuse %d.\n",evasion_fuse);
			}
		}
		else if (patrol_type == PATROL_PLANETKILLER) {
			if (me->p_flags & PFYELLOW) SHIELDS_ON
			if (me->p_flags & PFORBIT) {
				logmsg1("  made it to destroyable planet.\n");
				CLOAK_OFF
				ECM_OFF
				me->p_flags |= PFBOMB;
				patrol_fuse = 1000;
				patrol_type = PATROL_KILLING;
			}
			else if ((evasion_type == EVADE_NONE)
					|| (evasion_fuse <= 0)) {
				me->p_flags |= PFPLLOCK;
				logmsg2("%d: heading to target planet.\n",me->p_no);
			}
			else {
				logmsg2("  evading, fuse %d.\n",evasion_fuse);
			}
		}
		else if (patrol_type == PATROL_KILLING) {
			if (me->p_flags & PFYELLOW) SHIELDS_ON
			if (me->p_flags & PFORBIT) {
				logmsg1("  attacking planet.\n");
				me->p_flags |= PFBOMB;
				if (planets[me->p_planet].pl_armies <= 0) {
					me->p_flags &= ~PFBOMB;
					planets[me->p_planet].pl_armies = 0;
					if (INTRAND(5) > 1) {
						logmsg2("  Killer turned %s TMPBARREN\n",
							planets[me->p_planet].pl_name);
						planets[me->p_planet].pl_flags |= 
							(PLMASK_TMPBARREN|PLMASK_BARREN);
						planets[me->p_planet].pl_deadtime =
							(5+INTRAND(6))*60
							*(1000000/UPDATE)/PLFIGHTFUSE;
					}
					else {
						logmsg2("  Killer turned %s to AST_FIELD\n",
							planets[me->p_planet].pl_name);
						planets[me->p_planet].pl_flags |=
							(PLMASK_AST_FIELD
								| PLMASK_TMPBARREN
								| PLMASK_BARREN);
						planets[me->p_planet].pl_deadtime =
							60*(1000000/UPDATE)/PLFIGHTFUSE;
					}
					patrol_fuse = 0;
					EXIT_ORBIT
				}
			}
			else {
				patrol_fuse = 0;
			}
		}
		else if (patrol_type == PATROL_RANDOM) {
			logmsg3("%d: continuing random patrol, fuse %d\n",
				me->p_no,patrol_fuse);
		}
		else if (patrol_type == PATROL_ENEMY) {
			logmsg3("%d: continuing enemy patrol, fuse %d\n",
				me->p_no,patrol_fuse);
		}
		else if (patrol_type == PATROL_BUDDY) {
			register PLAYER *mybuddy;

			mybuddy = &players[me->p_buddy];	
			if ((mybuddy->p_status != PALIVE) 
					|| (mybuddy->p_speed == 0)
					|| (mybuddy->p_engine_health < 200)) {
				/* break buddy patrol */
				patrol_fuse = 0;
			}
			logmsg3("%d: continuing buddy patrol, fuse %d\n",
				me->p_no,patrol_fuse);
		}
		else if (patrol_type == PATROL_NONE) {
			patrol_fuse = 0;
		}

		if (--patrol_fuse <= 0) {
			if (patrol_type == PATROL_REPAIRING) {
				EXIT_ORBIT
				REPAIR_OFF
			}
			else if (patrol_type == PATROL_SEEK_REPAIR) {
				EXIT_ORBIT
			}
			patrol_type = PATROL_NONE;
			me->p_flags &= ~(PFPLOCK|PFPLLOCK);
		}
		else {
			final_checks();
			return; /* keep patrolling */
		}
	}


	/* if under alert, raise shields */
	if (me->p_flags & (PFRED|PFYELLOW)) {
		SHIELD_ON
	}

	/* which quadrant am I in? */
	if (me->p_x < (GWIDTH/2)) {
		if (me->p_y < (GWIDTH/2)) myquadrant = 2;
		else myquadrant = 1;
	}
	else {
		if (me->p_y < (GWIDTH/2)) myquadrant = 3;
		else myquadrant = 4;
	}

	/* Find closest enemy */
	denemy = get_nearest(&humans,FALSE);
	if (denemy != (ENEMY *) -1) {
		dhisship = &dhim->p_ship;
		if (denemy->e_flags & E_BASE) {
			/* don't attack bases -- find someone else to attack */
			oenemy = get_nearest(&humans,TRUE);
		}
		else {
			/* he's not a base -- attack him */
			oenemy = denemy;
			ohim   = dhim;
		}
	}
	else {
		/* no one to run from, certainly no one to attack. */
		oenemy = denemy;
		ohim   = dhim;
	}

	if (!humans) {
		logmsg2("%d: No humans around.\n",me->p_no);
		if (timer == -1) {
			timer = me->p_updates+1800;
			lastupdate = me->p_updates;
			samelastcount = 0;
		}		
		else if (me->p_updates == lastupdate) {
			if (++samelastcount > 20) {
				/* p_updates hasn't changed in ages */
				mysignal(SIGALRM,SIG_IGN);
				me->p_status = PFREE;
				exit(0);
			}
		}
		else {
			samelastcount = 0;
			lastupdate = me->p_updates;
			if (me->p_updates >= timer) {
				mysignal(SIGALRM, SIG_IGN);
				me->p_status = PFREE;
				exit(0);
			}
		}		
	} else timer = -1;

	if (orbiter && !(me->p_flags & PFORBIT)) {
		/* rotate */
		me->p_dir += 6;
		me->p_desdir = me->p_dir;
	}

	if (oenemy == (ENEMY *) -1) {
		/* no one to attack */
		/* is there someone to run from? */
		if (denemy != (ENEMY *) -1) {
			if (ravoid_torps() || rrun_away()) {
				final_checks();
				return;
			}
		}
		if (rrepair()) return;
		patrol((ENEMY *) NULL);
		logmsg2("%d: No hostile players in game.\n", me->p_no);
		final_checks();
		return;
	}

	/* else */
	if (myship->s_torps > 0) {
		switch (myship->s_torpclass) {
			case WPN_PHOTON:
				too_far_range = myship->s_torpspeed 
					* myship->s_torpduration
					* WARP1 * torp_impatience / 300;
				break;
			case WPN_PLASMA:
				too_far_range = myship->s_torpspeed 
					* (myship->s_torpdamage/myship->s_torpdecay)
					* WARP1 * torp_impatience / 150;
				break;
			case WPN_MISSILE:
				too_far_range = myship->s_torpspeed
					* myship->s_torpduration
					* WARP1 * torp_impatience / 100;
				break;
			case WPN_CHAFF:
				too_far_range = (GWIDTH/100);
				break;
			case WPN_MINE:
				too_far_range = WPN_MINE_RANGE;
				break;
		}
	}
	else too_far_range = GWIDTH;

	if (myship->s_beams > 0) {
		int range;

		if ((range = myship->s_beamrange*beam_impatience/100)
				< too_far_range)
			too_far_range = range;
	}

	if ((denemy->e_dist < dhisship->s_beamrange*4/3) 
			|| (oenemy->e_dist < myship->s_beamrange*4/3)
			|| (oenemy->e_dist < (GWIDTH/20))) {
		CLOAK_ON
		ECM_ON
	}
	else {
		CLOAK_OFF
		ECM_OFF
	}

	/* Someone to kill */
	target_enemy(oenemy,ohim);
	ohisship = &ohim->p_ship;
	logmsg5("%d: closest #%d, dist %d.  Course to %d,",
		me->p_no, ohim->p_no, oenemy->e_dist, oenemy->e_course);
	logmsg3(" tcourse %d, eflags %d\n", oenemy->e_tcourse,
		oenemy->e_flags);
	if (orbiter) {
		rfire_weapons();
		if (!ravoid_torps())
			if (!rrepair())
				if (!rattack())
					patrol(oenemy);
	}
	else if (me->p_flags & PFPLKILLER) {
		rfire_weapons();
		if(!ravoid_torps())
			if (!rrun_away())
				if (!rrepair())
					patrol(oenemy);
	}
	else {
		/* mobile ship */
		if (offense > 70 ) {
			rfire_weapons();
			if(!ravoid_torps())
				if (!rrun_away())
					if (!rrepair())
						if (!rattack())
							patrol(oenemy);
		}
		else if (offense > 30 ) {
			moved = ravoid_torps();
			rfire_weapons();
			if (!moved)
				if (!rrun_away())
					if (!rrepair())
						if (!rattack())
							patrol(oenemy);
		}
		else {
			moved = ravoid_torps();
			rfire_weapons();
			if (!moved)
				if (!rrun_away())
					if (!rrepair())
						patrol(oenemy);
		}
	}

	final_checks();
}
