/* planets.c 15.1 06/08/90 10:06:14 */

/* Original by Daryl Poe.
 *
 * Copyright (c) 1989 Daryl Poe
 *
 * Ported to X11 by Jim Andreas.
 * Tractors, treaties, and other features by Norm Gee.
 * Damage window and shield bitmaps by Tom LaStrange.
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting
 * documentation.  No representations are made about the
 * suitability of this software for any purpose.  It is
 * provided "as is" without express or implied warranty.
 */


#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <fcntl.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include "defs.h"
#include "weapon.h"
#include "system.h"
#include "ship.h"
#include "stats.h"
#include "player.h"
#include "torp.h"
#include "status.h"
#include "planet.h"
#include "phaser.h"
#include "message.h"
#include "shmem.h"
#include "daemon.h"
#include "data.h"

#define bcopy(from, to, length)		memcpy((to), (from), (length))

#define HOMESMALL (GWIDTH*2/9)
#define HOMELARGE (GWIDTH-HOMESMALL)

PLANET pdata[MAXPLANETS];

static char *planetnames[] = {
	"Earth",
	"Sasus",
	"Candeleron",
	"Beta III",
	"Janus",
	"Deneb VI",
	"Ceti IV",
	"Altar",
	"Dekar",
	"Daltus",
	"Romulus",
	"Ethen",
	"Amur",
	"Remus",
	"Bal",
	"Tahndar",
	"Dact",
	"Sirius II",
	"Rakhir",
	"Rus",
	"Klingus",
	"Malatrakir",
	"Amakron",
	"Laltir",
	"Khartair",
	"Monpur III",
	"Sectus",
	"Makus",
	"Jakar",
	"Gendus",
	"Orion",
	"Amterion",
	"Lumecis",
	"Bitar V",
	"Prastan",
	"Sorlen",
	"Zetus",
	"Jih",
	"Markus II",
	"Oren",
};


static char *prename[] = {
	"Darkstar",

	"Hermes",

	"Syrinx",
	"Styx",

	"Midnight",
	"Lifebane",

	"Mutara",
	"Paradise",

	"Cerebus",
	"TNG-473",
	"Pagar",
	"Golar",
	"Lethal",
	"Forbidden",
	"Detroit",
	"Walrus",
	"Xanadu",
	"Glorious",
	"Purgatory",
	"Vortigern",
	"Yagara",
	"Galileo\'s",
	"Abandoned",
	"Fenson",
	"Menoran",
	"Gartivia",
	"Dragon",
	"Ascolais",
	"Pavia",
	"Xeres",
	"Ilibrial",
	"Hanion",
	"Craven",
	"Hassan\'s",
	"Inkarien",
	"Justice",
	"Ajax",
	"Mordor",
	"Hades",
	"Kipling\'s",
	"B&D\'s",
	"Schwartz",
	"Calenor",
	"Novgren",
	"Deadly",
	"Quiet",
	"Raed\'s",
	"Basilisk",
	"Untouched",
	"Half",
	"RUXK-23",
	"TGP-371",
	"BXX II",

};

#define SECTOR_CENTER 	0
#define SECTOR_EDGE 	1
#define SECTOR_CORNER	2
#define SECTOR_BAD		3

#define LOCATION_AWAYHOME	-3
#define LOCATION_FIXED		-2
#define LOCATION_RANDOM		-1

static void locate_planet(planet,quadrant)
PLANET *planet;
int quadrant;
{
	register int x,y,dx,dy;
	register PLANET *p;
	int sector,hextant;
	static int hextant_type[] = {
		SECTOR_CORNER,
		SECTOR_EDGE,
		SECTOR_EDGE,
		SECTOR_CORNER,

		SECTOR_EDGE,
		SECTOR_CENTER,
		SECTOR_CENTER,
		SECTOR_EDGE,

		SECTOR_EDGE,
		SECTOR_CENTER,
		SECTOR_CENTER,
		SECTOR_EDGE,

		SECTOR_CORNER,
		SECTOR_EDGE,
		SECTOR_EDGE,
		SECTOR_CORNER
	};

	do {
		x = (GWIDTH/100) + INTRAND(GWIDTH*98/100);
		y = (GWIDTH/100) + INTRAND(GWIDTH*98/100);
		hextant = (x/(GWIDTH/4)) + 4*(y/(GWIDTH/4));
		sector  = hextant_type[hextant];
		/* modify the chances of corner and edge hextants */
		if (sector == SECTOR_CORNER) {
			if (INTRAND(3)) sector = SECTOR_BAD;
		}
		else if (sector == SECTOR_EDGE) {
			if (INTRAND(2)) sector = SECTOR_BAD;
		}

		/* if quadrant specified, make sure we're in it */
		if ((sector != SECTOR_BAD)
				&& (quadrant != LOCATION_RANDOM)
				&& (quadrant != LOCATION_AWAYHOME)) {
			switch (quadrant) {
				case 0:
					if (x > (GWIDTH/2)) x = (GWIDTH-1) - x;
					if (y < (GWIDTH/2)) y = (GWIDTH-1) - y;
					break;
				case 1:
					if (x > (GWIDTH/2)) x = (GWIDTH-1) - x;
					if (y > (GWIDTH/2)) y = (GWIDTH-1) - y;
					break;
				case 2:
					if (x < (GWIDTH/2)) x = (GWIDTH-1) - x;
					if (y > (GWIDTH/2)) y = (GWIDTH-1) - y;
					break;
				case 3:
					if (x < (GWIDTH/2)) x = (GWIDTH-1) - x;
					if (y < (GWIDTH/2)) y = (GWIDTH-1) - y;
					break;
			}
		}

		/* search existing planets for one too close */
		if (sector != SECTOR_BAD) {
			for (p=planets; p<planets+MAXPLANETS; ++p) {
				if (p->pl_no == -1) continue;
				dx = p->pl_x - x;
				if (ABS(dx) > INTERPLANETARY_DIST) continue;
				dy = p->pl_y - y;
				if (ABS(dy) > INTERPLANETARY_DIST) continue;
				dx = Ihypot(dx,dy);
				if (ABS(dx) < INTERPLANETARY_DIST) 
					sector = SECTOR_BAD;
			}
		}

		if ((sector != SECTOR_BAD) && (quadrant == LOCATION_AWAYHOME)) {
			/* make sure I'm not near someone's home planet */
			if (x < (GWIDTH/2)) dx = x;
			else dx = GWIDTH - x;
			if (ABS(dx - HOMESMALL) < (INTERPLANETARY_DIST*2)) {
				sector = SECTOR_BAD;
			}
			else {
				if (y < (GWIDTH/2)) dy = y;
				else dy = GWIDTH - y;
				if (ABS(dy - HOMESMALL) < (INTERPLANETARY_DIST*2)) {
					sector = SECTOR_BAD;
				}
			}
		}
	} while (sector == SECTOR_BAD);

	planet->pl_x = x;
	planet->pl_y = y;
}


static void make_planet(num,flags,nowtime,quadrant,x,y,name)
int num,flags;
long nowtime;
int quadrant,x,y;
char *name;
{
	PLANET *planet;
	int i;
	static int pncount=0;
	int armiesguess;

	planet = &planets[num];

	if (flags & FED_MASK) {
		planet->pl_owner_no = FEDERATION;
	}
	else if (flags & ROM_MASK) {
		planet->pl_owner_no = ROMULAN;
	}
	else if (flags & KLI_MASK) {
		planet->pl_owner_no = KLINGON;
	}
	else if (flags & ORI_MASK) {
		planet->pl_owner_no = ORION;
	}
	else {
		flags |= IND_MASK;
		planet->pl_owner_no = INDEPENDENT;
	}
	planet->pl_owner_mask = flags
		& (IND_MASK|FED_MASK|ROM_MASK|KLI_MASK|ORI_MASK);


	if (quadrant == LOCATION_FIXED) {
		planet->pl_x = x;
		planet->pl_y = y;
	}
	else {
		locate_planet(planet,quadrant);
	}
	planet->pl_dx = 0;
	planet->pl_dy = 0;
	planet->pl_deadtime = 0;
	planet->pl_couptime = 0;

	if (flags & (FED_MASK|ROM_MASK|KLI_MASK|ORI_MASK)) {
		/* pre-owned planet */
		strcpy(planet->pl_name,name);
		planet->pl_armies = 20 + INTRAND(20);
		if (flags & PLMASK_HOME) {
			armiesguess = 40;
		}
		else if (flags & PLMASK_AST_FIELD) {
			armiesguess = 10;
		}
		else {
			armiesguess = 30;
		}
		if (INTRAND(4) == 0) flags |= PLMASK_REPAIR;
		if (INTRAND(2) == 0) flags |= PLMASK_FUEL;
	}
	else {
		if (flags & PLMASK_WANDERER) {
			COURSE dir;
	
			planet->pl_armies = 5 + INTRAND(30);
			sprintf(planet->pl_name,"%s",prename[pncount++]);
			dir = (COURSE) INTRAND(256);
			planet->pl_dx = (int)(Cos[dir]*WARP1);
			planet->pl_dy = (int)(Sin[dir]*WARP1);
			if (INTRAND(4) == 0) flags |= PLMASK_REPAIR;
			if (INTRAND(2) == 0) flags |= PLMASK_FUEL;
			armiesguess = 30;
		}
		else if (flags & PLMASK_BLACKHOLE) {
			planet->pl_armies = 0;
			sprintf(planet->pl_name,"%s BHole",prename[pncount++]);
			armiesguess = 0;
		}
		else if (flags & PLMASK_PULSAR) {
			planet->pl_armies = 0;
			sprintf(planet->pl_name,"%s Pulsar",prename[pncount++]);
			armiesguess = 0;
		}
		else if (flags & PLMASK_NEBULA) {
			planet->pl_armies = 0;
			sprintf(planet->pl_name,"%s Nebula",prename[pncount++]);
			armiesguess = 0;
		}
		else if (flags & PLMASK_BARREN) {
			planet->pl_armies = 0;
			sprintf(planet->pl_name,"%s",prename[pncount++]);
			armiesguess = 0;
		}
		else if (flags & PLMASK_AST_FIELD) {
			planet->pl_armies = 10;
			sprintf(planet->pl_name,"%s",prename[pncount++]);
			armiesguess = 10;
			if (INTRAND(4) == 0) flags |= PLMASK_REPAIR;
		}
		else {
			planet->pl_armies = 5 + INTRAND(30);
			armiesguess = 20;
			if (INTRAND(8) == 0) flags |= PLMASK_REPAIR;
			if (INTRAND(4) == 0) flags |= PLMASK_FUEL;
			sprintf(planet->pl_name,"%s",prename[pncount++]);
		}
	}
	planet->pl_flags = flags;
	planet->pl_namelen = strlen(planet->pl_name);

	for (i=0; i<NUMTEAM; ++i) {
		planet->pl_info[i].pi_flags     = flags 
			& ~(PLMASK_REPAIR|PLMASK_FUEL);
		planet->pl_info[i].pi_owner_no  = planet->pl_owner_no;
		planet->pl_info[i].pi_armies    = armiesguess;
		planet->pl_info[i].pi_timestamp = nowtime;
	}
	planet->pl_no = num;
}


void read_planet_file()
{
	int plfd,donew,plnum;
	PLANET *planet;
	long nowtime;
	struct stat statbuf;

	if (stat(Plfile,&statbuf) == -1) {
		fprintf(stderr, "No planet file.  Restarting galaxy\n");
		donew = TRUE;
	}
	else if (statbuf.st_size != sizeof(pdata)) {
		fprintf(stderr, "Planet file wrong size.  Restarting galaxy\n");
		donew = TRUE;
	}
	else donew = FALSE;

	if ((plfd = open(Plfile,O_RDWR|O_CREAT,0777)) < 0) {
		fprintf(stderr, "Couldn't open planet file. Aborting.\n");
		exit(1);
	}

	if (!donew) {
		/* read the existing planet file */
		if (read(plfd,(char *)planets,sizeof(pdata))
				!= sizeof(pdata)) {
			fprintf(stderr, "Couldn't read planet file. Restarting galaxy\n");
			donew = TRUE;
		}
	}

	if (donew) {
		/* START A NEW PLANETS FILE */
		nowtime = time((long *) 0);

		/* pl_no == -1 signifies uninitialized */
		for (planet=planets; planet < planets+MAXPLANETS; ++planet) {
			planet->pl_no = -1;
		}

		/* initialize home planets */
		make_planet( 0,(FED_MASK|PLMASK_HOME|PLMASK_FUEL|PLMASK_REPAIR),
			nowtime,LOCATION_FIXED,HOMESMALL,HOMELARGE,
			"Earth");
		planets[ 0].pl_armies += 10;
		planets[ 0].pl_flags  |= (PLMASK_REPAIR|PLMASK_FUEL);
		make_planet(10,(ROM_MASK|PLMASK_HOME|PLMASK_FUEL|PLMASK_REPAIR),
			nowtime,LOCATION_FIXED,HOMESMALL,HOMESMALL,
			"Romulus");
		planets[10].pl_armies += 10;
		planets[10].pl_flags  |= (PLMASK_REPAIR|PLMASK_FUEL);
		make_planet(20,(KLI_MASK|PLMASK_HOME|PLMASK_FUEL|PLMASK_REPAIR),
			nowtime,LOCATION_FIXED,HOMELARGE,HOMESMALL,
			"Klingus");
		planets[20].pl_armies += 10;
		planets[20].pl_flags  |= (PLMASK_REPAIR|PLMASK_FUEL);
		make_planet(30,(ORI_MASK|PLMASK_HOME|PLMASK_FUEL|PLMASK_REPAIR),
			nowtime,LOCATION_FIXED,HOMELARGE,HOMELARGE,
			"Orion");
		planets[30].pl_armies += 10;
		planets[30].pl_flags  |= (PLMASK_REPAIR|PLMASK_FUEL);
		/* planet 40 is an independent home planet */
		make_planet(40,
			(IND_MASK|PLMASK_HOME|PLMASK_REPAIR|PLMASK_FUEL),
			nowtime,LOCATION_FIXED,(GWIDTH/2),(GWIDTH/2),NULL);

		/* do fed planets */
		for (plnum= 1; plnum<9; ++plnum) {
			make_planet(plnum,FED_MASK,nowtime,0,0,0,planetnames[plnum]);
		}
		make_planet(plnum++,FED_MASK|PLMASK_AST_FIELD|PLMASK_REPAIR,
			nowtime,0,0,0,planetnames[plnum]);

		/* do rom planets */
		for (plnum=11; plnum<19; ++plnum) {
			make_planet(plnum,ROM_MASK,nowtime,1,0,0,planetnames[plnum]);
		}
		make_planet(plnum++,ROM_MASK|PLMASK_AST_FIELD|PLMASK_REPAIR,
			nowtime,1,0,0,planetnames[plnum]);

		/* do kli planets */
		for (plnum=21; plnum<29; ++plnum) {
			make_planet(plnum,KLI_MASK,nowtime,2,0,0,planetnames[plnum]);
		}
		make_planet(plnum++,KLI_MASK|PLMASK_AST_FIELD|PLMASK_REPAIR,
			nowtime,2,0,0,planetnames[plnum]);

		/* do ori planets */
		for (plnum=31; plnum<39; ++plnum) {
			make_planet(plnum,ORI_MASK,nowtime,3,0,0,planetnames[plnum]);
		}
		make_planet(plnum++,ORI_MASK|PLMASK_AST_FIELD|PLMASK_REPAIR,
			nowtime,3,0,0,planetnames[plnum]);

		/* planet 40 is an independent home planet, already done */
		plnum = 41;

		/* planet 41 is a wanderer, starting in the center */
		make_planet(plnum++,
			(IND_MASK|PLMASK_WANDERER|PLMASK_REPAIR|PLMASK_FUEL),
			nowtime,LOCATION_AWAYHOME,0,0,NULL);

		/* one independent per race's quadrant */
		make_planet(plnum++,IND_MASK,nowtime,0,0,0,NULL);
		make_planet(plnum++,IND_MASK,nowtime,1,0,0,NULL);
		make_planet(plnum++,IND_MASK,nowtime,2,0,0,NULL);
		make_planet(plnum++,IND_MASK,nowtime,3,0,0,NULL);

		/* and one asteroid field */
		make_planet(plnum++,IND_MASK|PLMASK_AST_FIELD,
			nowtime,0,0,0,NULL);
		make_planet(plnum++,IND_MASK|PLMASK_AST_FIELD,
			nowtime,1,0,0,NULL);
		make_planet(plnum++,IND_MASK|PLMASK_AST_FIELD,
			nowtime,2,0,0,NULL);
		make_planet(plnum++,IND_MASK|PLMASK_AST_FIELD,
			nowtime,3,0,0,NULL);

		/* one blackhole and pulsar and nebula per sector */
		make_planet(plnum++,PLMASK_BLACKHOLE,nowtime,0,0,0,NULL);
		make_planet(plnum++,PLMASK_PULSAR,   nowtime,0,0,0,NULL);
		make_planet(plnum++,PLMASK_NEBULA,   nowtime,0,0,0,NULL);
		make_planet(plnum++,PLMASK_BLACKHOLE,nowtime,1,0,0,NULL);
		make_planet(plnum++,PLMASK_PULSAR,   nowtime,1,0,0,NULL);
		make_planet(plnum++,PLMASK_NEBULA,   nowtime,1,0,0,NULL);
		make_planet(plnum++,PLMASK_BLACKHOLE,nowtime,2,0,0,NULL);
		make_planet(plnum++,PLMASK_PULSAR,   nowtime,2,0,0,NULL);
		make_planet(plnum++,PLMASK_NEBULA,   nowtime,2,0,0,NULL);
		make_planet(plnum++,PLMASK_BLACKHOLE,nowtime,3,0,0,NULL);
		make_planet(plnum++,PLMASK_PULSAR,   nowtime,3,0,0,NULL);
		make_planet(plnum++,PLMASK_NEBULA,   nowtime,3,0,0,NULL);

		/* other objects are random */
		for (; plnum<MAXPLANETS; ++plnum) {
			switch (INTRAND(12)) {
				case 0:
					/* black hole */
					make_planet(plnum,PLMASK_BLACKHOLE,
						nowtime,LOCATION_RANDOM,0,0,NULL);
					break;
				case 1:
					/* pulsar */
					make_planet(plnum,PLMASK_PULSAR,
						nowtime,LOCATION_RANDOM,0,0,NULL);
					break;
				case 2:
					/* independent wanderer */
					make_planet(plnum,(PLMASK_WANDERER|IND_MASK),
						nowtime,LOCATION_RANDOM,0,0,NULL);
					break;
				case 3:
					/* nebula */
					make_planet(plnum,PLMASK_NEBULA,
						nowtime,LOCATION_RANDOM,0,0,NULL);
					break;
				case 4:
					/* independent good planet */
					make_planet(plnum,IND_MASK,
						nowtime,LOCATION_RANDOM,0,0,NULL);
					break;
				case 5:
					/* independent asteroid field */
					make_planet(plnum,IND_MASK|PLMASK_AST_FIELD,
						nowtime,LOCATION_RANDOM,0,0,NULL);
					break;
				case 6:
					/* barren asteroid field */
					make_planet(plnum,IND_MASK|PLMASK_AST_FIELD|PLMASK_BARREN,
						nowtime,LOCATION_RANDOM,0,0,NULL);
					break;
				default:
					/* barren planet */
					make_planet(plnum,PLMASK_BARREN,
						nowtime,LOCATION_RANDOM,0,0,NULL);
					break;
			}
		}
	}

	close(plfd);
}

