/* general.c 15.1 06/08/90 10:05:23 */

/* Original code by Daryl Poe.
 *
 * Copyright (c) 1989 Daryl Poe
 *
 * Ported to X11 by Jim Andreas.
 * Tractors, treaties, and other features by Norm Gee.
 * Damage window and shield bitmaps by Tom LaStrange.
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting
 * documentation.  No representations are made about the
 * suitability of this software for any purpose.  It is
 * provided "as is" without express or implied warranty.
 */

/* GENERAL UTILITIES */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <stdio.h>
#include <fcntl.h>
#include <time.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/signal.h>
#include <sys/time.h>
#include <sys/times.h>
#include <sys/resource.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/dir.h>
#include <sys/errno.h>
#include "defs.h"
#include "weapon.h"
#include "system.h"
#include "ship.h"
#include "stats.h"
#include "player.h"
#include "torp.h"
#include "status.h"
#include "planet.h"
#include "phaser.h"
#include "message.h"
#include "shmem.h"
#include "data.h"
#include "filenames.h"

#ifdef HPUX7
extern clock_t times();
#else
extern long times();
#endif

typedef int (*INT_FUNC)();
typedef void (*VOID_FUNC)();

void seedrandom()
{
	long i;
	struct timeval tp;
	struct timezone tzp;

	i = getpid() + getppid();
	gettimeofday(&tp,&tzp);
	i ^= tp.tv_usec;

	srand48(i);
	srand((unsigned)(i&0xffffffff));
}


void getrusage(foo, buf)
int foo;
struct rusage *buf;
{
	struct tms t;

	times(&t);
	buf->ru_utime.tv_sec = t.tms_utime/HZ;
	buf->ru_stime.tv_sec = t.tms_stime/HZ;
}

/* Use this instead of system-supplied version to avoid SV_RESETHAND
 * of signal(2).  Using mysignal insures the signal handling routine
 * is not reset every time the signal is caught.
 */
int (*mysignal(sig, funct))()
int sig;
int (*funct)();
{
 	struct sigvec vec,ovec;
 
 	vec.sv_handler = (VOID_FUNC) funct;
 	vec.sv_mask    = 0;
 	vec.sv_flags   = 0;
 	sigvector(sig, &vec, &ovec);
	return((INT_FUNC) ovec.sv_handler);
}

/* do all the necessary stuff for a proper fork */
void detach()
{
    mysignal(SIGALRM, (INT_FUNC) SIG_IGN);
	setpgrp();
	close(0);
	close(1);
	fclose(logfile);
}

/* do all the necessary stuff for a proper piped fork */
void pipedetach(pipeout)
int pipeout;
{
    mysignal(SIGALRM, (INT_FUNC) SIG_IGN);
	setpgrp();
	close(1);		/* close stdout */
	dup(pipeout); 	/* duplicate stdout to pipeout */
	close(0);		/* close stdin */
	/* close(2); */
	fclose(logfile);
	/* shmdt(sharedMemory); */
}


/* construct filename variables from defaults and overrides */
void construct_filenames()
{
	char *cptr;
	char ldir[MAXPATHLEN];

	if ((cptr = getenv("DTREK_DIR")) != NULL) strcpy(Dtrek_dir,cptr);
	else strcpy(Dtrek_dir,DTREK_DIR);

	if ((cptr = getenv("DTREK_LDIR")) != NULL) strcpy(ldir,cptr);
	else strcpy(ldir,Dtrek_dir);

	if (Dtrek_dir[strlen(Dtrek_dir)-1] != '/') strcat(Dtrek_dir,"/");
	if (ldir[strlen(ldir)-1] != '/') strcat(ldir,"/");

	strcpy(Plfile,Dtrek_dir);
	strcat(Plfile,PLFILE);

	strcpy(Modefile,Dtrek_dir);
	strcat(Modefile,MODEFILE);

	strcpy(Scoredir,Dtrek_dir);
	strcat(Scoredir,SCOREDIR);

	strcpy(Motd,Dtrek_dir);
	strcat(Motd,MOTD);

	strcpy(Daemon,Dtrek_dir);
	strcat(Daemon,DAEMON);

	strcpy(Robot,Dtrek_dir);
	strcat(Robot,ROBOT);

	strcpy(Shipmenu,Dtrek_dir);
	strcat(Shipmenu,SHIPMENU);

	strcpy(Bitmaps,Dtrek_dir);
	strcat(Bitmaps,BITMAPS);

	strcpy(Logfilename,ldir);
	strcat(Logfilename,LOGFILENAME);

	strcpy(Dlogfilename,ldir);
	strcat(Dlogfilename,DLOGFILENAME);

	strcpy(Rlogfilename,ldir);
	strcat(Rlogfilename,RLOGFILENAME);

	strcpy(Shiparchives,Dtrek_dir);
	strcat(Shiparchives,SHIPARCHIVES);

	strcpy(Scores,Dtrek_dir);
	strcat(Scores,SCORES);

	strcpy(Control_file,Dtrek_dir);
	strcat(Control_file,CONTROL_FILE);
}


void sem_lock()
{
	static struct sembuf sema_op = { 0, -1, SEM_UNDO };
	int count,result;

	count = 0;
	while (((result = semop(semid,&sema_op,1)) == -1)
			&& (errno == EINTR)) {
		/* received signal during semaphore op */
		if (++count > 10) break;
	}

	if (result == -1) {
		perror("error on semaphore lock: ");
	}
}


void sem_unlock()
{
	static struct sembuf sema_op = { 0, 1, SEM_UNDO };
	int count,result;

	count = 0;
	while (((result = semop(semid,&sema_op,1)) == -1)
			&& (errno == EINTR)) {
		/* received signal during semaphore op */
		if (++count > 10) break;
	}

	if (result == -1) {
		perror("error on semaphore unlock: ");
	}
}


#define BUFSIZE 256
int ar_table_of_contents_size(arfile)
char *arfile;
{
	int i,count;
	int pipefd[2];
	char buf[BUFSIZE],*cptr;

	if (access(arfile,0) == -1) return(0);
	/* else */

	pipe(pipefd);
	if (fork() == 0) {
		pipedetach(pipefd[1]);
		/* do "ar t arfile" */
		execl("/bin/ar","ar","t",arfile,0);
		/* should never get here anyway */
		exit(0);
	}
	else {
		/* parent only */
		/* count # lines read from ar command */
		wait((int *)0);
		count = 0;
		do {
			i = read(pipefd[0],buf,BUFSIZE);
			for (cptr=buf; cptr<buf+i; ++cptr)
				if (*cptr == '\012') ++count;
		} while (i == BUFSIZE);

		close(pipefd[0]);
		close(pipefd[1]);
	}
	return(count);
}


int ar_check_entry(arfile,entryname)
char *arfile,*entryname;
{
	int pipemask,readit;
	int pipefd[2];
	char buf[BUFSIZE],*cptr;
	struct timeval waittime;

	if (access(arfile,4) == -1) return(FALSE);
	/* else */

	pipe(pipefd);
	if (fork() == 0) {
		pipedetach(pipefd[1]);
		/* do "ar t filename" */
		execl("/bin/ar","ar","t",arfile,0);
		/* should never get here anyway */
		exit(0);
	}
	else {
		/* parent only */
		/* read from ar command */
		wait((int *)0);
		pipemask = 1<<pipefd[0];
		waittime.tv_sec  = 10;
		waittime.tv_usec = 0;
		/* is pipe output ready? */
		if (select(pipefd[0]+1,&pipemask,0,0,&waittime) <= 0)
			return(FALSE);
		waittime.tv_sec  = 1;
		do {
			cptr = buf;
			do {
				if (select(pipefd[0]+1,&pipemask,0,0,&waittime) <= 0) {
					readit = FALSE;
					break;
				}
				/* else */
				readit = read(pipefd[0],cptr++,1);
			} while (*(cptr-1) != '\012');
			if (readit) {
				*(cptr-1) = '\0';
				if (strcmp(entryname,buf) == 0) {
					close(pipefd[0]);
					close(pipefd[1]);
					return(TRUE);
				}
			}
		} while (readit);

		close(pipefd[0]);
		close(pipefd[1]);
	}

	return(FALSE);
}


int ar_read_entry(arfile,entryname,data_addr,data_size)
char *arfile,*entryname,*data_addr;
unsigned data_size;
{
	int pipefd[2];

	if (access(arfile,4) == -1) return(FALSE);
	/* else */

	pipe(pipefd);
	if (fork() == 0) {
		pipedetach(pipefd[1]);
		/* do "ar p arfile entryname" */
		execl("/bin/ar","ar","p",arfile,entryname,0);
		/* should never get here anyway */
		exit(0);
	}
	else {
		/* parent only */
		wait((int *)0);
		/* get the data from the ar command */
		if (read(pipefd[0],data_addr,data_size) < data_size) {
			fprintf(stderr,"Could not read value array from ar\n");
			close(pipefd[0]);
			close(pipefd[1]);
			return(FALSE);
		}
		close(pipefd[0]);
		close(pipefd[1]);
	}

	return(TRUE);
}


int ar_write_entry(arfile,entryname,data_addr,data_size)
char *arfile,*entryname,*data_addr;
unsigned data_size;
{
	char tempfile[MAXPATHLEN];
	char str[MAXPATHLEN*3];
	int  ofile;

	sprintf(tempfile,"/tmp/%s",entryname);
	ofile = open(tempfile,O_WRONLY|O_CREAT,0666);
	if (write(ofile,data_addr,data_size) < data_size) {
		fprintf(stderr,"write to ar entry file not successful\n");
		close(ofile);
		return(FALSE);
	}
	close(ofile);
	chmod(tempfile,0644);
	chown(tempfile,(int) getuid(),(int) getgid());
	sprintf(str,"/bin/ar rc %s %s",arfile,tempfile);
	system(str);
	unlink(tempfile);

	return(TRUE);
}


/*  Fields that are functions of weapons or ship races
 * which are not directly dependent on how the ship was
 * built should be computed here.  This lets them be
 * modified without throwing away the ship archives.
 */
void compute_fields(ship)
SHIP *ship;
{

	if (ship->s_sptechmask & STMASK_SPHERE) {
		ship->s_torpclass = WPN_SPHERE_BACKLASH;
	}
	else if (ship->s_torps > 0) {
		switch (ship->s_torpclass) {
			case WPN_PLASMA:
				ship->s_torpdecay = ship->s_torpdamage / 
					WPN_PLASMA_DURATION * WPN_PLASMA_ATTEN;
				ship->s_torpduration = 10 * WPN_PLASMA_DURATION;
				break;
			default:
				ship->s_torpdecay = 0;
				break;
		}
	}

	if (ship->s_beams > 0) {
		switch (ship->s_beamclass) {
			case WPN_PHASER:
				ship->s_beamatten = ship->s_beamrange 
					/ ship->s_beamdamage * WPN_PHASER_ATTEN;
				break;
			case WPN_DISRUPTER:
				ship->s_beamatten = ship->s_beamrange 
					/ ship->s_beamdamage * WPN_DISRUPTER_ATTEN;
				break;
			case WPN_FUSION:
				ship->s_beamatten = ship->s_beamrange 
					/ ship->s_beamdamage * WPN_FUSION_ATTEN;
				break;
			default:
				ship->s_beamatten = GWIDTH;
				break;
		}
	}
}


void normalize_string(in,out)
register char *in,*out;
{
	register char c;

	while ((c = *in++) != '\0') {
		if ((c >= 'a') && (c <= 'z')) {
			/* capitalize */
			*out++ = c + 'A' - 'a';
		}
		else if ((c == '-') || (c == '/') || (c == ' ')) {
			/* ignore these */
		}
		else {
			*out++ = c;
		}
	}
	*out = '\0';
}


int is_substr(sub,full)
register char *sub,*full;
{
	register char *sptr;

	sptr = sub;
	while (*full != '\0') {
		if (*sptr == '\0') return(1);
		else if (*sptr == *full) {
			++sptr;
			++full;
			if ((*full == '\0') && (*sptr == '\0')) return(1);
		}
		else {
			sptr = sub;
			++full;
		}
	}
	
	return(0);
}


int myfgets(sptr,max,fp)
register char *sptr;
int max;
FILE *fp;
{
	register int count;
	register char c;

	count = 0;
	--max;
	while ((c = fgetc(fp)) != (char) EOF) {
		if (c == '\n') {
			*sptr = '\0';
			return(count);
		}
		else {
			*sptr++ = c;
			if ((++count) >= max) {
				*sptr = '\0';
				return(count);
			}
		}
	}

	/* got EOF */
	if (count > 0) {
		*sptr = '\0';
		return(count);
	}
	else {
		return((int) EOF);
	}
}

