/* gcontrol.c 15.1 06/08/90 10:05:19 */

/* Original code by Daryl Poe.
 *
 * Copyright (c) 1990 Daryl Poe
 *
 * Ported to X11 by Jim Andreas.
 * Tractors, treaties, and other features by Norm Gee.
 * Damage window and shield bitmaps by Tom LaStrange.
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting
 * documentation.  No representations are made about the
 * suitability of this software for any purpose.  It is
 * provided "as is" without express or implied warranty.
 */

#include <stdio.h>
#include <time.h>
#include <ndir.h>
#include <sys/param.h>
#include "defs.h"
#include "weapon.h"
#include "system.h"
#include "ship.h"
#include "stats.h"
#include "player.h"
#include "torp.h"
#include "status.h"
#include "planet.h"
#include "phaser.h"
#include "message.h"
#include "shmem.h"
#include "data.h"
#include "filenames.h"

#define MAX_NUM_EVENTS	256


typedef struct {
	char game[256];
	int  dow;
	int  hour;
	int  min;
} EVENT;
EVENT events[MAX_NUM_EVENTS];
static int num_events;

static char *dowlabel[] = {
	"Sunday", "Monday", "Tuesday", "Wednesday",
	"Thursday", "Friday", "Saturday"
};

/* should be run by "daemon" */
static void maked(path)
char *path;
{
	logmsg2(" Making directory %s\n",path);
	if (mkdir(path,0755) == -1) {
		fprintf(stderr,"Could not mkdir(%s,0755)\n",path);
		logmsg2("Could not mkdir(%s,0755)\n",path);
		perror("");
	}

	if (chmod(path,0755) == -1) {
		fprintf(stderr,"Could not chmod(%s,0755)\n",path);
		logmsg2("Could not chmod(%s,0755)\n",path);
		perror("");
	}
}


static int dir_exists(path)
char *path;
{
	DIR *dir;

	if ((dir = opendir(path)) == NULL) {
		return(FALSE);
	}
	else {
		closedir(dir);
		return(TRUE);
	}
}


static int symlink_exists(path)
char *path;
{
	char buf[MAXPATHLEN];

	if (readlink(path,buf,sizeof(buf)) == -1) return(FALSE);
	else return(TRUE);
}


static void swap_game(gname)
char *gname;
{
	char path[MAXPATHLEN],cmd[MAXPATHLEN+256],gamename[MAXPATHLEN];

	if (gname[0] != '/') {
		strcpy(gamename,"/");
		strcat(gamename,gname);
	}
	else strcpy(gamename,gname);

	/* Ensure new game exists */
	strcpy(path,Dtrek_dir);
	strcat(path,GAMESDIR);
	if (!dir_exists(path)) maked(path);
	strcat(path,gamename);
	if (!dir_exists(path)) maked(path);
	strcat(path,SCOREDIR);
	if (!dir_exists(path)) maked(path);

	/* Ensure DEFAULT game exists */
	strcpy(path,Dtrek_dir);
	strcat(path,DEFAULT_GAMESDIR);
	if (!dir_exists(path)) maked(path);
	strcat(path,SCOREDIR);
	if (!dir_exists(path)) maked(path);

	if (!symlink_exists(Scoredir) 
			&& dir_exists(Scoredir)) {
		/* The scores directory is not symbolically linked */
		logmsg2(" scores not symlinked.  Moving contents to %s\n",path);

		/* path is still Dtrek_dir/GAMESDIR/DEFAULT_GAME/SCOREDIR */

		/* Move it to the DEFAULT game */
		sprintf(cmd,"(cd %s; ls -1 | cpio -pmu %s >/dev/null)",
			Scoredir,path);
		system(cmd);
		sprintf(cmd,"rm -rf %s",Scoredir);
		system(cmd);
		rmdir(Scoredir);
		symlink(path,Scoredir);
	}

	/* move any old-style, non-linked game files to DEFAULT game */
	if (!symlink_exists(Plfile)) {
		strcpy(path,Dtrek_dir);
		strcat(path,DEFAULT_GAMESDIR);
		strcat(path,PLFILE);
		logmsg3(" Old style %s: moving to %s.\n",Plfile,path);
		sprintf(cmd,"mv -f %s %s",Plfile,path); system(cmd);
		symlink(path,Plfile);
	}

	if (!symlink_exists(Modefile)) {
		strcpy(path,Dtrek_dir);
		strcat(path,DEFAULT_GAMESDIR);
		strcat(path,MODEFILE);
		logmsg3(" Old style %s: moving to %s.\n",Modefile,path);
		sprintf(cmd,"mv -f %s %s",Modefile,path); system(cmd);
		symlink(path,Modefile);
	}

	if (!symlink_exists(Motd)) {
		strcpy(path,Dtrek_dir);
		strcat(path,DEFAULT_GAMESDIR);
		strcat(path,MOTD);
		logmsg3(" Old style %s: moving to %s.\n",Motd,path);
		sprintf(cmd,"mv -f %s %s",Motd,path); system(cmd);
		symlink(path,Motd);
	}

	logmsg1(" Now swapping...\n");
	/* Switch to the specified game */
	strcpy(path,Dtrek_dir);
	strcat(path,GAMESDIR);
	strcat(path,gamename);
	strcat(path,SCOREDIR);
	logmsg3("    ln -s %s %s\n",path,Scoredir);
	unlink(Scoredir);
	symlink(path,Scoredir);

	strcpy(path,Dtrek_dir);
	strcat(path,GAMESDIR);
	strcat(path,gamename);
	strcat(path,PLFILE);
	logmsg3("    ln -s %s %s\n",path,Plfile);
	unlink(Plfile);
	symlink(path,Plfile);

	strcpy(path,Dtrek_dir);
	strcat(path,GAMESDIR);
	strcat(path,gamename);
	strcat(path,MOTD);
	logmsg3("    ln -s %s %s\n",path,Motd);
	unlink(Motd);
	symlink(path,Motd);

	strcpy(path,Dtrek_dir);
	strcat(path,GAMESDIR);
	strcat(path,gamename);
	strcat(path,MODEFILE);
	logmsg3("    ln -s %s %s\n",path,Modefile);
	unlink(Modefile);
	symlink(path,Modefile);

	logmsg1(" Game swap done.\n");
}


static int eventcmp(e1,e2)
EVENT *e1,*e2;
{
	if (e1->dow < e2->dow) return(-1);
	else if (e1->dow > e2->dow) return(1);

	/* dow equal */
	if (e1->hour < e2->hour) return(-1);
	else if (e1->hour > e2->hour) return(1);

	/* and hour */
	if (e1->min < e2->min) return(-1);
	else if (e1->min > e2->min) return(1);

	return(0);
}


void game_control(game_end_time,thisgame,nextgamemsg)
long *game_end_time;
char *thisgame;
char *nextgamemsg;
{
	FILE *infile;
	char str[256],dowstr[256],gamename[256],*cptr;
	int min,hour,i;
	EVENT *event,*nextevent;
	char *ampm;
    struct timeval timeval;
	struct timezone tz;
	struct tm *tm;
	unsigned short euid,uid;
	long nowtime;

	logmsg1("\nGame control...\n");
	if ((infile = fopen(Control_file,"r")) == NULL) {
		/* don't interrupt the game */
		logmsg2(" Cannot open %s, no game control.\n",Control_file);
		*game_end_time = 0;
		strncpy(thisgame,"This game",256);
		strncpy(nextgamemsg,"",512);
		return;
	}

	event = events;
	num_events = 0;
	while (myfgets(str,sizeof(str),infile) != EOF) {
		if ((str[0] != '#') 
				&& (sscanf(str,"%d %d %s %s",
					&min,&hour,dowstr,gamename) == 4)) {
			cptr = dowstr;
			while (*cptr) {
				if ((*cptr >= '0') && (*cptr <= '6')) {
					strcpy(event->game,gamename);
					event->dow    = (int) (*cptr - '0');
					event->hour   = hour;
					event->min = min;
					++event;
					if ((++num_events) > MAX_NUM_EVENTS) {
						fprintf(stderr,"Too many events in %s\n",
							Control_file);
						logmsg1(" Too many events in game control file.  No game control\n");
						/* don't interrupt the game */
						*game_end_time = 0;
						strncpy(thisgame,"This game",256);
						strncpy(nextgamemsg,"",512);
						return;
					}
				}
				++cptr;
				if (*cptr == ',') ++cptr;
			}
		}
	}
	fclose(infile);

	if (num_events == 0) {
		logmsg1(" No events in game control file.  No game control\n");
		/* don't interrupt the game */
		*game_end_time = 0;
		strncpy(thisgame,"This game",256);
		strncpy(nextgamemsg,"",512);
		return;
	}

	/* get current time */
	nowtime = time((long *) 0);
	gettimeofday(&timeval, &tz);
	tm = localtime(&timeval.tv_sec);

	/* increment times before now by 7 days */
	for (i=0,event=events; i<num_events; ++i,++event) {
		if (event->dow > tm->tm_wday) continue;
		else if (event->dow == tm->tm_wday) {
			if (event->hour > tm->tm_hour) continue;
			else if (event->hour == tm->tm_hour) {
				if (event->min > tm->tm_min) continue;
			}
		}
		/* increment */
		event->dow += 7;
	}

	/* now sort */
	qsort((char *) events,num_events,sizeof(EVENT),eventcmp);

#ifdef DEBUG
	logmsg1("\n\nGAME CONTROL file contents (sorted):\n");
	for (i=0,event=events; i<num_events; ++i,++event) {
		logmsg5("%3s %2d:%02d %s\n",
			dowlabel[event->dow % 7],
			event->hour,event->min,event->game);
	}
	logmsg2("\nTime is %s.\n",asctime(tm));
#endif

	/* find the next non-disabled game */
	for (i=0,nextevent=events; i<num_events; ++i,++nextevent) {
		if (strncmp(nextevent->game,"DISABLED",8) != 0) break;
	}
	if (i == num_events) {
		nextevent = NULL;
	}
	else {
		if (nextevent->hour < 12) {
			ampm = "AM";
			if (nextevent->hour == 0) hour = 12;
			else hour = nextevent->hour;
		}
		else {
			ampm = "PM";
			if (nextevent->hour == 12) hour = 12;
			else hour = nextevent->hour - 12;
		}
	}
	
	/* the current game is the *last* event */
	event = &events[num_events-1];
	if (strncmp(event->game,"DISABLED",8) == 0) {
		/* The game is currently disabled */
		logmsg1(" Game currently disabled, exiting.\n");
		fprintf(stderr,"Game disabled by dtrek control file.\n");
		if (nextevent == NULL) {
			fprintf(stderr,"No games are enabled on this machine.\n");
		}
		else {
			fprintf(stderr,"Next game %s at %d:%02d %s.\n",
				dowlabel[nextevent->dow % 7],hour,nextevent->min,ampm);
		}
		fprintf(stderr,"Aborting.\n");
		exit(1);
	}

	/* swap to new game */
	normalize_string(event->game,gamename);
	uid = getuid();
	euid = geteuid();
	setresuid(euid,euid,uid);
	swap_game(gamename);
	setresuid(uid,euid,uid);

	if (num_events > 1) {
		/* Now set up time to let the daemon know when this game ends.
		 * The first event is the next one to happen.
		 */
		event = events;
		*game_end_time = 
			nowtime                                  +
			(event->dow  - tm->tm_wday) * (24*60*60) +
			(event->hour - tm->tm_hour) * (60*60)    +
			(event->min  - tm->tm_min)  * 60 - 30;
		strncpy(thisgame,events[num_events-1].game,256);
		if (nextevent == NULL) {
			strncpy(nextgamemsg,"",512);
		}
		else {
			sprintf(nextgamemsg,"(%s begins %s at %d:%02d %s.)",
				nextevent->game,
				dowlabel[nextevent->dow % 7],hour,nextevent->min,ampm);
		}
		logmsg2(" Next game msg: %s\n",nextgamemsg);
		return;
	}
	else {
		/* only one event -- let it go forever */
		*game_end_time = 0;
		strncpy(thisgame,"This game",256);
		strncpy(nextgamemsg,"",512);
		logmsg2(" Next game msg: %s\n",nextgamemsg);
		return;
	}
}

