/* damage.c 15.1 06/08/90 10:04:38 */

/* Original code by Daryl Poe.
 *
 * Copyright (c) 1989 Daryl Poe
 *
 * Ported to X11 by Jim Andreas.
 * Tractors, treaties, and other features by Norm Gee.
 * Damage window and shield bitmaps by Tom LaStrange.
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting
 * documentation.  No representations are made about the
 * suitability of this software for any purpose.  It is
 * provided "as is" without express or implied warranty.
 */

#include <stdio.h>
#include "defs.h"
#include "weapon.h"
#include "system.h"
#include "ship.h"
#include "stats.h"
#include "player.h"
#include "torp.h"
#include "status.h"
#include "planet.h"
#include "phaser.h"
#include "message.h"
#include "shmem.h"
#include "daemon.h"
#include "damage.h"
#include "data.h"

/* 1/DA_FRACTION is the amount of damage absorbed by ABSORBERS */
#define DA_FRACTION 5
/* DA_FUEL_CONVERSION is the amount of fuel given per damage absorbed */
#define DA_FUEL_CONVERSION (5*DA_FRACTION)

static void kill_player(victim,attacker,wpn_number)
PLAYER *victim,*attacker;
int wpn_number;
{
	logmsg2("damage: killing player #%d\n",victim->p_no);

	victim->p_status = PEXPLODE;
	victim->p_explode = 10/PLAYERFUSE;
	victim->p_stats.st_losses++;
		
	if (wpn_number == WPN_PLANET) {
		/* attacker was a planet */
		PLANET *planet = (PLANET *) attacker;
		char str[80];

		victim->p_whydead = KPLANET;
		victim->p_whodead = planet->pl_no;
		if (planet->pl_flags & PLMASK_BLACKHOLE) {
			sprintf(str,"%s (%s) sucked into a black hole",
				victim->p_name, victim->p_mapchars);
		}
		else if (planet->pl_flags & PLMASK_AST_FIELD) {
			sprintf(str,"%s (%s) destroyed by an asteroid",
				victim->p_name, victim->p_mapchars);
		}
		else if (planet->pl_flags & PLMASK_PULSAR) {
			sprintf(str,"%s (%s) killed by pulsar radiation",
				victim->p_name, victim->p_mapchars);
		}
		else {
			sprintf(str,"%s (%s) killed by %s",
				victim->p_name, victim->p_mapchars,
				planet->pl_name);
		}
		pmessage(str, 0, MALL, "Game->ALL");
	}
	else if ((wpn_number == WPN_PROB_VORTEX)
			|| (wpn_number == WPN_PROB_VORTEX_NS)) {
		char str[80];

		victim->p_whydead = KVORTEX;
		victim->p_whodead = 0;
		sprintf(str,"%s (%s) destroyed by Probability Vortex",
			victim->p_name, victim->p_mapchars);
		pmessage(str, 0, MALL, "Game->ALL");
	}
	else {
		/* attacker was a player */
		/* you don't get points for killing a friend! */
	  	if ((attacker->p_hostile_mask & victim->p_team_mask)
				&& (!(victim->p_flags & PFPRACTR))) {
			attacker->p_kills += 
				  (float) victim->p_ship.s_hull / STD_HULL_SIZE
		    	+ (float) victim->p_armies      * 0.1
				+ (float) victim->p_kills       * 0.1;
		}

		switch (wpn_number) {
			case WPN_PHOTON:
			case WPN_MISSILE:
			case WPN_PLASMA:
			case WPN_MINE:
			case WPN_CHAFF:
			case WPN_SPHERE_BACKLASH:
				victim->p_whydead = KTORP;
				break;
			case WPN_PHASER:
			case WPN_TACHYON:
			case WPN_FUSION:
			case WPN_DISRUPTER:
				victim->p_whydead = KPHASER;
				break;
			case WPN_DESTRUCT:
				victim->p_whydead = KSHIP;
				break;
		}
		victim->p_whodead = attacker->p_no;
		killmess(victim,attacker);
	}

	if ((victim->p_flags & PFROBOT)
			&& SAMESTR(victim->p_ship.s_shipname,"Starbase")) {
		team[victim->p_team_no].t_starbase_died = TRUE;
	}
}


static int damage_system(attacker,victim,sysunit_no,fulldamage,
	wpn_number)
PLAYER *attacker,*victim;
int sysunit_no,fulldamage,wpn_number;
{
	int oldsize,damage;
	SYSTEM_UNIT *sysunitptr;

	logmsg4("damage: damaging player %d, sysunit %d, fulldamage %d\n",
		victim->p_no,sysunit_no,fulldamage);

	sysunitptr = &victim->p_ship.s_system[sysunit_no];

	/* hit smaller non-armor systems for less damage (on average) */
	if (sysunitptr->item == ARMOR || sysunitptr->item == HULL) {
		if (fulldamage <= sysunitptr->current_size) 
			damage = fulldamage;
		else damage = sysunitptr->current_size;
	}
	else {
		if (fulldamage < (damage = INTRAND(sysunitptr->current_size)+1))
			damage = fulldamage;
		if (damage > DAMAGE_SPLITUP) damage = INTRAND(DAMAGE_SPLITUP)+1;
		victim->p_ship.s_systemsize -= damage;
	}

	if ((oldsize = sysunitptr->current_size)
			== sysunitptr->undamaged_size)
		sysunitptr->fractional_size = 0;
	sysunitptr->current_size -= damage;

	switch (sysunitptr->item) {
		case HULL:
			/* just check for hull==0 (bad news for victim) */
			logmsg3("damage: hit hull.  oldsize %d, newsize %d\n",
				oldsize,sysunitptr->current_size);
				
			if (sysunitptr->current_size == 0) {
				logmsg1("damage: hull gone. killing player\n");
				kill_player(victim,attacker,wpn_number);
				/* don't bother with more damage allocation */
				return(fulldamage);
			}
			break;
		case ENGINE:
			/* mess up acceleration, etc. (if there ever was any) */
			logmsg3("damage: hit engine.  oldsize %d, newsize %d\n",
				oldsize,sysunitptr->current_size);
				
			if (sysunitptr->undamaged_size)
				victim->p_engine_health =
					sysunitptr->current_size * 
					256 / sysunitptr->undamaged_size;
			break;
		case FUEL:
			/* decrement current fuel tankage & fuel if nec. */
			logmsg3("damage: hit fuel.  oldsize %d, newsize %d\n",
				oldsize,sysunitptr->current_size);
				
			victim->p_ship.s_maxfuel -= 500 * damage;
			if (victim->p_fuel > victim->p_ship.s_maxfuel)
				victim->p_fuel = victim->p_ship.s_maxfuel;
			break;
		case BEAM_NUM:
			logmsg4("damage: hit beam %d.  oldsize %d, newsize %d\n",
				sysunitptr->subitem,oldsize,sysunitptr->current_size);
				
			if (oldsize == sysunitptr->undamaged_size) {
				logmsg1("  damage: newly damaged beam.\n");
				/* this beam is newly damaged */
				--victim->p_ship.s_beams;
			}
			break;
		case CLOAK:
			logmsg3("damage: hit cloak.  oldsize %d, newsize %d\n",
				oldsize,sysunitptr->current_size);
				
			victim->p_ship.s_sptechmask &= ~STMASK_CLOAK;
			victim->p_flags &= ~PFCLOAK;
			break;
		case IMP_SENS:
			logmsg3("damage: hit imp sens.  oldsize %d, newsize %d\n",
				oldsize,sysunitptr->current_size);
				
			victim->p_ship.s_sptechmask &= ~STMASK_IMP_SENS;
			break;
		case ECM:
			logmsg3("damage: hit ecm.  oldsize %d, newsize %d\n",
				oldsize,sysunitptr->current_size);
				
			victim->p_ship.s_sptechmask &= ~STMASK_ECM;
			victim->p_flags &= ~PFECM;
			break;
		case HYPERSPACE:
			logmsg3("damage: hit hyperspace.  oldsize %d, newsize %d\n",
				oldsize,sysunitptr->current_size);
				
			victim->p_ship.s_sptechmask &= ~STMASK_HYPERSPACE;
			victim->p_flags &= ~PFHYPERSPACE;
			break;
		case SPHERE:
			logmsg3("damage: hit sphere.  oldsize %d, newsize %d\n",
				oldsize,sysunitptr->current_size);
				
			victim->p_ship.s_sptechmask &= ~STMASK_SPHERE;
			victim->p_flags &= ~PFSPHERE;
			break;
		case AFC:
			logmsg3("damage: hit AFC.  oldsize %d, newsize %d\n",
				oldsize,sysunitptr->current_size);
				
			victim->p_ship.s_sptechmask &= ~STMASK_AFC;
			break;
		case DAMAGE_ABS:
			logmsg3("damage: hit dam abs.  oldsize %d, newsize %d\n",
				oldsize,sysunitptr->current_size);

			victim->p_ship.s_sptechmask &= ~STMASK_DAMAGE_ABS;
			break;
		case TORP_NUM:
			logmsg4("damage: hit torp %d.  oldsize %d, newsize %d\n",
				sysunitptr->subitem,oldsize,sysunitptr->current_size);
				
		  if(victim->p_ship.s_torpclass != WPN_PLASMA ) {
			if (oldsize == sysunitptr->undamaged_size) {
				logmsg1("  damage: newly damaged torp.\n");
				/* this torp newly damaged */
				--victim->p_ship.s_torps;
				/* lucky day--hit an unloaded tube */
				if (victim->p_torps_unloaded > 0)
					--victim->p_torps_unloaded;
			}
		  }
			break;
		case TRACTOR_BEAM:
			logmsg4("damage: hit tractor %d.  oldsize %d, newsize %d\n",
				sysunitptr->subitem,oldsize,sysunitptr->current_size);
				
			if (oldsize == sysunitptr->undamaged_size) {
				logmsg1("  damage: newly damaged tractor.\n");
				/* this tractor newly damaged */
				--victim->p_ship.s_tractor;
			}
			break;
		case DILITHIUM:
			logmsg3("damage: hit dilithium.  oldsize %d, newsize %d\n",
				oldsize,sysunitptr->current_size);
			
			victim->p_ship.s_recharge -= damage;
			break;
		case CARGO:
			logmsg3("damage: hit cargo.  oldsize %d, newsize %d\n",
				oldsize,sysunitptr->current_size);
				
			victim->p_ship.s_cargo -= damage;
			if (victim->p_armies > victim->p_ship.s_cargo)
				victim->p_armies = victim->p_ship.s_cargo;
			break;
		case ARMOR:
			logmsg3("damage: hit armor.  oldsize %d, newsize %d\n",
				oldsize,sysunitptr->current_size);
				
			victim->p_ship.s_armor -= damage;
			break;
		default:
			logmsg2("damage: ERROR damage_system not complete for system %d.\n",
				sysunitptr->item);
			break;
	}

	return(damage);
}





/* This module does damage to a ship.
 * If wpn_number==WPN_PLANET, the damage was done by planet
 * (PLANET *) attacker.
 */
void do_damage(attacker,victim,damage,wpn_number,shield_number)
PLAYER *attacker,*victim;
int damage,wpn_number,shield_number;
{
	SHIP *vship;
	int hit,i,j;

	if ((damage <= 0) || (victim->p_status != PALIVE)) return;

	logmsg4("damage: hit player %d for %d in shield %d\n",
		victim->p_no,damage,shield_number);

	vship = &victim->p_ship;

	/* Use black sphere if available */
	if (victim->p_flags & PFSPHERE) {
		/* sphere backlash */
		if ((wpn_number != WPN_DESTRUCT)
				&& (wpn_number != WPN_PROB_VORTEX)
				&& (wpn_number != WPN_PROB_VORTEX_NS)
				&& (wpn_number != WPN_SPHERE_BACKLASH)
				&& (wpn_number != WPN_PLANET)) {
			sphere_backlash(victim,attacker,damage);
		}
		/* and through */
		if ((damage /= 2) == 0) return;
	}

	/* Use damage absorbers if available */
	if (vship->s_sptechmask & STMASK_DAMAGE_ABS) {
		if ((i = damage / DA_FRACTION) == 0) i = 1;
		if (i >= damage) {
			/* fully absorbed by DA's */
			if ((victim->p_fuel += DA_FUEL_CONVERSION*damage)
					> vship->s_maxfuel)
				victim->p_fuel = vship->s_maxfuel;
			return;
		}
		/* else partial absorption */
		damage -= i;
		if ((victim->p_fuel += DA_FUEL_CONVERSION*i) > vship->s_maxfuel)
			victim->p_fuel = vship->s_maxfuel;
	}

	/* Make the shields absorb some of the damage */
	/* If it's tachyon, it ignores shields */
	if ((wpn_number != WPN_TACHYON)
			&& (wpn_number != WPN_PROB_VORTEX_NS)) {
		/* Are the shields up? */
		if (victim->p_flags & PFSHIELD) {
			if (victim->p_shield[shield_number] >= damage) {
				/* all damage absorbed by shield */
				logmsg3("damage: %d points fully absorbed by shield %d\n",
					damage,shield_number);
				victim->p_shield[shield_number] -= damage;
				return;
			}
			/* else */
			logmsg3("damage: %d points absorbed by shield %d\n",
				victim->p_shield[shield_number],shield_number);
			damage -= victim->p_shield[shield_number];
			victim->p_shield[shield_number] = 0;
			victim->p_subshield = 0;
		}
	}

	if (damage <= 0) return;

	vship->s_damaged = TRUE;

	/* Made it through shields.  Blow away armor now. */
	if (vship->s_armor > 0) {
		/* Armor absorbs damage first */
		damage -= damage_system(attacker,victim,0,damage,wpn_number);
		if (damage <= 0) return;
	}

	vship->s_repairable = TRUE;

	/* Hit "HULL" with the remainder */
	damage_system(attacker,victim,1,damage,wpn_number);
	if (victim->p_status != PALIVE) {
		/* the ship's been destroyed */
		return;
	}

	/* Cannot remove more damage than there are systems */
	if (damage > vship->s_systemsize) damage = vship->s_systemsize;

	/* Now blow away systems at random */
	while (damage > 0) {
		logmsg2("damage: systemsize %d  ",vship->s_systemsize);
		hit = INTRAND(vship->s_systemsize)+1;
		logmsg2("hit %d  ",hit);
		/* skip armor and hull */
		i = 2;
		while ((j = vship->s_system[i].current_size) < hit) {
			hit -= j;
			++i;
		}
		if (i < vship->s_numsys) {
			/* damage the system */
			logmsg2("damaging system %d\n",i);
			damage -= damage_system(attacker,victim,i,damage,wpn_number);
			logmsg1("damage: this system done.\n");
			if (vship->s_system[i].repair_rate)
				vship->s_repairable = TRUE;
		}
		else {
			logmsg3("tried to damage system %d when numsys is %d.\n",
				i,vship->s_numsys);
		}
	}
	logmsg1("done with damage.\n");
}



