/* daemon.c 15.1 06/08/90 10:04:25 */

/*

	Copyright (c) 1986 	Chris Guthrie

Permission to use, copy, modify, and distribute this
software and its documentation for any purpose and without
fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright
notice and this permission notice appear in supporting
documentation.  No representations are made about the
suitability of this software for any purpose.  It is
provided "as is" without express or implied warranty.

*/

/* Major overhaul by Daryl Poe.
 *
 * Copyright (c) 1989 Daryl Poe
 *
 * Ported to X11 by Jim Andreas.
 * Tractors, treaties, and other features by Norm Gee.
 * Damage window and shield bitmaps by Tom LaStrange.
 *
 * The above comments apply to this version as well, and this
 * notice must also appear in any copies or follow-ons.
 */

#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/sem.h>
#include <time.h>
#include <fcntl.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <signal.h>
#include <sys/signal.h>
#include <setjmp.h>

#include "defs.h"
#include "weapon.h"
#include "system.h"
#include "ship.h"
#include "stats.h"
#include "player.h"
#include "torp.h"
#include "status.h"
#include "planet.h"
#include "phaser.h"
#include "message.h"
#include "shmem.h"
#include "daemon.h"
#include "data.h"

#define bcopy(from, to, length)		memcpy((to), (from), (length))
#define fuse(X) ((ticks % (X)) == 0)
#define RRANGE ((GWIDTH)/20)
#define LOANRANGE	TRACTOR_RANGE

#define JUGGERCHANCE 	16
#define MAXJUGGERS		3
#define MAXIND			5

/* in 1/10 second units */
#define JUGGERFUSE			300		/* 30 seconds */
#define BASEFUSE			6000	/* 10 minutes */
#define GAMEENDFUSE			300		/* thirty seconds */
#define PLANETKILLERFUSE	6000	/* 10 minutes */
#define PLANETKILLERCHANCE	6		/* 1in6 chance */

#ifdef VORTEX_TESTING
# define PVORTEXFUSE	200		/* 20 sec */
#else
# define PVORTEXFUSE	600		/* one minute */
#endif

#define INDCHANCE		10
#define PLREDRAWDIST	7500
#define WANDERERFUSE	(12000/PLFIGHTFUSE)
	/* how long (on average) between changes in wanderer direction */

/* GLOBALS */

static char *_version = "@(#)15.1 06/08/90 daemon for dtrek";
static struct itimerval udt;
static int shmid;

static int ticks = 0;

jmp_buf env;

static int defender_count[NUMTEAM],human_count[NUMTEAM],
	player_count[NUMTEAM];
static int nplayers = 0;
static int nactiveplayers = 0;
static int dietime = -1;

/* To keep planets shut up for awhile */
static int pl_warning_fuse[MAXPLANETS];

/* To limit the number of robots */
static int robot_fuse[NUMTEAM];

/* To allow a coup */
static int coup_fuse[NUMTEAM];

/* update a team's info on a foreign planet */
static void update_planet_info(whichteam,planet)
int whichteam;
PLANET *planet;
{
	PLANET_INFO *pi;

	/* independents don't care */
	if ((whichteam < FEDERATION) || (whichteam > ORION)) return;

	pi = &(planet->pl_info[whichteam]);

	pi->pi_flags     = planet->pl_flags;
	pi->pi_owner_no  = planet->pl_owner_no;
	pi->pi_armies    = planet->pl_armies;
	pi->pi_timestamp = time((long *) 0);
}


/* send a ghostbuster when ghostbuster strikes */
static void ghostmess(victim)
PLAYER *victim;
{
	char 		buf[80];
	static float	ghostkills = 0.0;

	logmsg1("daemon: sending ghostmessage\n");

	ghostkills += 1.0 + victim->p_armies * 0.1 + victim->p_kills * 0.1;
	sprintf(buf, "%s (%s) was kill %0.2f for the GhostBusters",
	    victim->p_name, victim->p_mapchars, ghostkills);
	pmessage(buf, 0, MALL, "Game->ALL");
}

/* send out a message when someone is killed */
void killmess(victim, killer)
PLAYER *victim, *killer;
{
	char buf[80];

	logmsg1("daemon:  sending killmess");

	sprintf(buf, "%s (%s) was kill %0.2f for %s (%s)",
	    victim->p_name,
		victim->p_mapchars,
	    killer->p_kills,
	    killer->p_name,
		killer->p_mapchars);
	pmessage(buf, 0, MALL, "Game->ALL");
}


/* destroy a ship, do damage to surrounding ships */
static void blowup(dying)
PLAYER *dying;
{
	int i;
	PLAYER *player;
	int dx, dy, dist, damage,basedam;

	logmsg2("daemon: blowing up player %d\n",dying->p_no);

	basedam = dying->p_ship.s_hull * 2/3;
	for (i = 0, player = players; i < MAXPLAYER; i++, player++) {
		if (player->p_status != PALIVE) continue;
		if (player == dying) continue;
		if ((dx = ABS(dying->p_x - player->p_x)) > DETDIST) continue;
		if ((dy = ABS(dying->p_y - player->p_y)) > DETDIST) continue;
		if ((dist = dx * dx + dy * dy) > (DETDIST*DETDIST)) continue;
		if (dist > (EXPDIST * EXPDIST)) {
			damage = basedam * (DETDIST - Isqrt(dist)) /
			    (DETDIST - EXPDIST);
		}
		else {
			damage = basedam;
		}
		logmsg3("daemon blowup: %d damage to player %d.\n",damage,player->p_no);
		if (damage > 0) 
			do_damage(dying,player,damage,WPN_DESTRUCT,
				shield_hit(Atan2(dying->p_y - player->p_y,
					player->p_x - dying->p_x), player->p_dir));
	}
}


/* Set players' alert status and player visibility */
static void udalert_visibility()
{
	PLAYER *player1,*player2;
	int i,j,np;
	int dist,flags[MAXPLAYER];
	unsigned int teamvis[NUMTEAM],mask1,mask2;

	for (i=0; i<NUMTEAM; ++i) teamvis[i] = 0;
	for (i=0; i<MAXPLAYER; ++i) flags[i] = PFGREEN;

	np = 0;
	for (player1 = players, mask1 = 1;
			player1 < players+MAXPLAYER;
			player1++, mask1 <<= 1) {
		if ((player1->p_status == PALIVE)
				|| (player1->p_status == PEXPLODE)
				|| (player1->p_status == PIMMUNE)) {
			teamvis[player1->p_team_no] |= mask1;

			player1->p_loanflags         = 0;
			player1->p_loanoutflags      = 0;
			if (player1->p_flags & PFMONITOR) {
				/* loan stuff to friends */
				if (player1->p_flags & PFECM)
					player1->p_loanoutflags |= PFLECM;
				if (player1->p_ship.s_sptechmask & STMASK_IMP_SENS)
					player1->p_loanoutflags |= PFLIMP_SENS;
				if (player1->p_flags & PFCLOAK)
					player1->p_loanoutflags |= PFLCLOAK;
			}

			/* what's my sensor range? */
			if (player1->p_obscured == OBSCURED_NEBULA) {
				player1->p_sensor_range =
					(SCALE*WINSIDE/2*NEBULA_VISIBILITY);
			}
			else if ((player1->p_ship.s_sptechmask & STMASK_IMP_SENS)
					|| (player1->p_loanflags & PFLIMP_SENS)) {
				player1->p_sensor_range = (SCALE*WINSIDE*3);
			}
			else {
				player1->p_sensor_range = (SCALE*WINSIDE*3/2);
			}
			++np;
		}
	}

	if (np > 1) {
	logmsg1("daemon:  updating alert statuses\n");
	for (player1 = players, mask1 = 1;
			player1 < players + (MAXPLAYER-1);
			++player1, mask1 <<= 1) {
		if ((player1->p_status != PALIVE)
				&& (player1->p_status != PIMMUNE)
				&& (player1->p_status != PEXPLODE)) {
			/* ignore this player */
			continue;
		}

		for (player2 = player1+1, mask2 = mask1<<1;
				player2 < players + MAXPLAYER;
				++player2, mask2 <<= 1) {
			if ((player2->p_status != PALIVE)
					&& (player2->p_status != PIMMUNE)
					&& (player2->p_status != PEXPLODE)) {
				/* ignore this player */
				continue;
			}

			/* figure the distance between them */
			dist = Ihypot(player1->p_x - player2->p_x, 
				player1->p_y - player2->p_y);

			if (player1->p_team_no == player2->p_team_no) {
				if (dist < LOANRANGE) {
					player2->p_loanflags |= player1->p_loanoutflags;
					player1->p_loanflags |= player2->p_loanoutflags;
				}
				/* keep going -- you won't change anything */
				continue;
			}
			else if ((dist < LOANRANGE)
					&& (team[player1->p_team_no].t_ally
						& (1 << player2->p_team_no))) {
				player2->p_loanflags |= player1->p_loanoutflags;
				player1->p_loanflags |= player2->p_loanoutflags;
			}

			/* change alert status -- are they enemies? */
			if ((player1->p_status == PALIVE)
					&& (player2->p_status == PALIVE)
					&& ((player2->p_team_mask & player1->p_hostile_mask)
						|| (player1->p_flags & PFPLKILLER)
						|| (player2->p_flags & PFPLKILLER))) {
				if (dist < RRANGE) {
					flags[player1->p_no] = 
						flags[player2->p_no] =
						PFRED;
				}
				else {
					if ((flags[player1->p_no] != PFRED)
							&& (dist < player2->p_yrange)) {
						flags[player1->p_no] = PFYELLOW;
					}
					if ((flags[player2->p_no] != PFRED)
							&& (dist < player1->p_yrange)) {
						flags[player2->p_no] = PFYELLOW;
					}
				}
			}

			/* now change player1's team's visibility flags */
			if (!(teamvis[player1->p_team_no] & mask2)
					&& ((player2->p_status != PIMMUNE)
						|| !(player2->p_flags & PFSTART))) {
				if (player2->p_obscured == OBSCURED_NEBULA) {
					/* I can see him only if he can see me */
					if (dist < player2->p_sensor_range) {
						teamvis[player1->p_team_no] |= mask2;
					}
				}
				else if (dist < player1->p_sensor_range) {
					/* I can see him if he's within sensor range */
					teamvis[player1->p_team_no] |= mask2;
				}
				else if (planets[player2->p_closest_planet].pl_owner_no
						== player1->p_team_no) {
					/* He's close to one of my planets */
					teamvis[player1->p_team_no] |= mask2;
				}
			}

			/* now change player2's team's visibility flags */
			if (!(teamvis[player2->p_team_no] & mask1)
					&& ((player1->p_status != PIMMUNE)
						|| !(player1->p_flags & PFSTART))) {
				if (player1->p_obscured == OBSCURED_NEBULA) {
					/* I can see him only if he can see me */
					if (dist < player1->p_sensor_range) {
						teamvis[player2->p_team_no] |= mask1;
					}
				}
				else if (dist < player2->p_sensor_range) {
					/* I can see him if he's within sensor range */
					teamvis[player2->p_team_no] |= mask1;
				}
				else if (planets[player1->p_closest_planet].pl_owner_no
						== player2->p_team_no) {
					/* He's close to one of my planets */
					teamvis[player2->p_team_no] |= mask1;
				}
			}
		}
	}

	/* OR together the masks of allies */
	for (i = 0, mask1 = 1; i < (NUMTEAM-1); ++i, mask1 <<= 1) {
		for (j = i+1, mask2 = mask1<<1; j < NUMTEAM; ++j, mask2 <<= 1) {
			if (team[i].t_ally & mask2) {
				teamvis[i] |= teamvis[j];
				teamvis[j] |= teamvis[i];
			}
		}
	}
	}

	/* Now, for everyone on each team, set visibility equal for all
	 * players on that team.
	 */
	for (player1 = players; player1 < players + MAXPLAYER; ++player1) {
		if ((player1->p_status == PALIVE)
				|| (player1->p_status == PEXPLODE)
				|| (player1->p_status == PIMMUNE)) {
			player1->p_visible_ships = teamvis[player1->p_team_no];
			player1->p_flags &= ~(PFGREEN|PFYELLOW|PFRED);
			player1->p_flags |= (flags[player1->p_no]
				& (PFGREEN|PFYELLOW|PFRED));
		}
	}
}


static void disrupt_lockons(pno,chance)
int pno,chance;
{
    register PLAYER *p;

	logmsg2("daemon: disrupting lockons to player %d\n",pno);

    /* break lockons for those locked on cloaked ship */
    for (p=players; p<players+MAXPLAYER; ++p)
        if ((p->p_status == PALIVE) || (p->p_status == PIMMUNE)) {
            if (p->p_playerl == pno)
                if (p->p_flags & PFPLOCK)
                    if (INTRAND(chance) == 0)
                        p->p_flags &= ~PFPLOCK;
			if (p->p_wpnlock == pno)
				if (p->p_flags & PFWLOCK)
					if (INTRAND(chance) == 0)
						p->p_flags &= ~PFWLOCK;
		}
}


static void changedir(player)
PLAYER *player;
{
	COURSE delta,turns;

	logmsg2("daemon: changing dir for player %d\n",player->p_no);

	if (player->p_speed == 0) {
		player->p_dir    = player->p_desdir;
		player->p_subdir = 0;
	}
	else {
		player->p_subdir += player->p_ship.s_turns
			* player->p_engine_health / 256 / (1 << player->p_speed);
		if ((turns = (COURSE) (player->p_subdir / 1000))) {
			if (turns >
					(delta = (COURSE) (player->p_dir - player->p_desdir))) {
				player->p_dir    = player->p_desdir;
				player->p_subdir = 0;
			}
			else if (delta > 127) {
				player->p_dir    += turns;
				player->p_subdir %= 1000;
			}
			else {
				player->p_dir    -= turns;
				player->p_subdir %= 1000;
			}
		}
	}
}


static void udplayers()
{
	register int i;
	register PLAYER *player;
	PLANET *planet;
	int maxspeed;

	nactiveplayers = nplayers = 0;
	for (i=0; i<NUMTEAM; ++i) {
		defender_count[i] = 0;
		human_count[i] = 0;
		player_count[i] = 0;
	}
	for (i=0, status->active_mask = 0, player = players;
			i < MAXPLAYER; i++, player++) {
		switch (player->p_status) {
		case POUTFIT:
			if (++(player->p_ghostbuster) > OUTFITTIME)
				player->p_status = PFREE;
			break;
		case PFREE:
			nplayers++;
			/* stop from hosing new players */
			player->p_ghostbuster = 0;
			break;
		case PDEAD:
			if (--player->p_explode <= 0) { 	/* Ghost Buster */
				player->p_status = PFREE;
			}
			break;
		case PEXPLODE:
			player->p_updates++;
			player->p_flags &= ~(PFCLOAK|PFSPHERE);
			player->p_loanflags &= ~PFLCLOAK;
			if (player->p_explode == (10/PLAYERFUSE)) {
				blowup(player);		/* damage everyone else around */
				disrupt_lockons(player->p_no,1);
					/* break all lockons */
			}
			if (--player->p_explode <= 0) {
				player->p_status = PDEAD;
				/* set ghost buster */
				player->p_explode = (600/PLAYERFUSE);
			}
			break;
		case PALIVE:
		case PIMMUNE:
			if (status->active_mask == 0)
				logmsg1("daemon: updating players\n");
			if (++(player->p_ghostbuster) > GHOSTTIME) {
				player->p_status = PEXPLODE;
				player->p_explode = (10/PLAYERFUSE);
				ghostmess(player);
				player->p_whydead = KGHOST;
				player->p_whodead = i;
			}

			status->active_mask |= (1<<i);
			++player->p_updates;

			if (!(player->p_flags & PFROBOT)) ++nactiveplayers;

			/**** PIMMUNE CUTOFF POINT ****/
			if (player->p_status == PIMMUNE) break;

			++player_count[player->p_team_no];
			if (!(player->p_flags & PFROBOT))
				++human_count[player->p_team_no];
			if ((player->p_team_no == INDEPENDENT)
					|| !(player->p_flags & PFROBOT)
					|| (player->p_ship.s_maxspeed > 0)) {
				/* don't count bases */
				++defender_count[player->p_team_no];
			}

			/* reload torps */
			if (player->p_torps_unloaded)
				if (--player->p_subreload <= 0)
					if (--player->p_torps_unloaded)
						player->p_subreload =
							player->p_ship.s_torpreload;
					else player->p_subreload = 0;

			/* reload beams */
			if (player->p_beams_unloaded)
				if (--player->p_beams_reload <= 0)
					if (--player->p_beams_unloaded)
						player->p_beams_reload =
							player->p_ship.s_beamreload;
					else player->p_beams_reload = 0;

			/* Charge for cloaking */
			if (player->p_flags & PFCLOAK) {
				if (player->p_fuel < STFUEL_CLOAK) {
					player->p_flags &= ~PFCLOAK;
				}
				else {
					player->p_fuel -= STFUEL_CLOAK;
					if (!(player->p_updates % 10))
						disrupt_lockons(player->p_no,4);
				}
			}
			else if (player->p_loanflags & PFLCLOAK) {
				if (!(player->p_updates % 10))
					disrupt_lockons(player->p_no,4);
			}

			/* Charge for sphere */
			if (player->p_flags & PFSPHERE) {
				if (player->p_fuel < STFUEL_SPHERE) {
					player->p_flags &= ~PFSPHERE;
				}
				else {
					player->p_fuel -= STFUEL_SPHERE;
				}
			}

			/* charge for Tractor Beam */
			if (player->p_tractors_on) {
				if ((player->p_fuel -=
							TRACTOR_FUEL * player->p_tractors_on)
						<= (TRACTOR_FUEL*4)) {
					player->p_fuel = 
						player->p_tractors_on =
						0;
				}
			}

			/* Charge for shields */
			if (player->p_flags & PFSHIELD) {
				player->p_fuel -= 1;
			}

			/* Check for hyperspace jump */
			if (player->p_flags & PFHYPERSPACE) {
				if (player->p_etemp > 900) {
					if (player->p_fuel >= STFUEL_HYPERSPACE) {
						PLAYER *him;
						int doit;

						/* make sure no one is tractoring me */
						doit = TRUE;
						for (him=players; him < players+MAXPLAYER; ++him) {
							if (him->p_status != PALIVE) continue;
							if (him == player) continue;
							if (!him->p_tractors_on) continue;
							if (him->p_who_tractor != player->p_no)
								continue;
							doit = FALSE;
							break;
						}

						/* Do it... jump! */
						if (doit) {
							player->p_x = player->p_hyp_x;
							player->p_y = player->p_hyp_y;
							player->p_fuel -= STFUEL_HYPERSPACE;
							/* unload all torps and beams */
							player->p_torps_unloaded =
								player->p_ship.s_torps;
							player->p_subreload = 
								player->p_ship.s_torpreload;
							player->p_beams_unloaded = 
								player->p_ship.s_beams;
							player->p_beams_reload = 
								player->p_ship.s_beamreload;
						}
						player->p_flags &= ~(PFBOMB|PFORBIT|PFBEAMUP|
							PFBEAMDOWN|PFPLOCK);
					}
					player->p_flags &= ~PFHYPERSPACE;
				}
			}		
		
			/* Add fuel */
			if (player->p_fuel < player->p_ship.s_maxfuel) {
				if ((player->p_flags & PFORBIT)
			    		&& (planets[player->p_planet].pl_flags & PLMASK_FUEL)
			    		&& (!(planets[player->p_planet].pl_owner_mask
			    			& (player->p_hostile_mask)))) {
					player->p_fuel += (player->p_ship.s_recharge << 1) + 6;
				}
				else {
					player->p_fuel += (player->p_ship.s_recharge << 1);
				}

				if (player->p_fuel > player->p_ship.s_maxfuel) {
					player->p_fuel = player->p_ship.s_maxfuel;
				}
				else if (player->p_fuel < 0) {
					player->p_fuel = player->p_desspeed = 0;
					player->p_flags &= ~(PFCLOAK|PFSPHERE|PFECM|PFHYPERSPACE);
				}
			}

			/* Heat engines */
			{
				int d_temp;

				if (player->p_flags & PFHYPERSPACE) {
					d_temp = 10;
				}
				else {
					if ((d_temp = player->p_speed - player->p_ecool)
							< -4) d_temp = -4;
				}

				if ((player->p_etemp += d_temp) < 0)
					player->p_etemp = 0;
			}

			/* Check for overheat */
			if ((player->p_flags & PFENG) 
					&& (player->p_etemp < 100)) {
				player->p_flags &= ~PFENG;
			}
			else if ((player->p_etemp > 1000)
					&& (!(player->p_flags & PFHYPERSPACE))) {
				if (!(INTRAND(40))) {
					player->p_flags |= PFENG;
					player->p_flags &= ~PFHYPERSPACE;
					player->p_desspeed = 0;
				}
			}

			logmsg1("moving\n");
			/* MOVE PLAYER */
			if (player->p_flags & PFORBIT) {
				/* Move Player in orbit */
				COURSE curdir;

				curdir = Atan2(planets[player->p_planet].pl_y - player->p_y,
					player->p_x - planets[player->p_planet].pl_x);
				logmsg4("ORBIT: x,y,dir %d,%d,%d\n",
					player->p_x - planets[player->p_planet].pl_x,
					player->p_y - planets[player->p_planet].pl_y,
					curdir);
				curdir += 2;
				player->p_x = planets[player->p_planet].pl_x
				    + ORBDIST * Cos[curdir];
				player->p_y = planets[player->p_planet].pl_y
				    + ORBDIST * Sin[curdir];
				logmsg4("ORBIT: new x,y,dir %d,%d,%d\n",
					player->p_x - planets[player->p_planet].pl_x,
					player->p_y - planets[player->p_planet].pl_y,
					curdir);

				if (player->p_ship.s_maxspeed == 0) {
					/* rotate in orbit */
					player->p_dir -= 3;
				}
				else {
					/* face orbit direction */
					player->p_dir = curdir + 64;
				}
				player->p_desdir = player->p_dir;
			}
			else {
				/* Move player through space */

				if (player->p_flags & PFPLLOCK) {
					/* set course to planet x */
					int dist;
					int dist_val;
					PLANET *p;

					p = &planets[player->p_planet];
					dist = Ihypot(player->p_x - p->pl_x,
						player->p_y - p->pl_y);

					dist_val=10 + 50 * (player->p_speed * (player->p_speed+1));
					/* If ship has good deceleration, change cut off distance */

					if ( player->p_ship.s_decint > 160 ) { 
						dist_val = dist_val / ( player->p_ship.s_decint / 160);
					}
					if (dist < dist_val )
						player->p_desspeed = 2;

					if ((dist < ENTORBDIST)
							&& (player->p_speed <= 2))	{
						player->p_flags &= ~PFPLLOCK;
						orbit(player);
					}
					else {
						player->p_desdir =
							Atan2(player->p_y - p->pl_y,
								p->pl_x - player->p_x);
					}
				}
				else if (player->p_flags & PFPLOCK) {
					/* set course to player x */
					PLAYER *pl;

					pl = &players[player->p_playerl];
					if (pl->p_status != PALIVE)
						player->p_flags &= ~PFPLOCK;
					player->p_desdir = Atan2(player->p_y - pl->p_app_y,
						pl->p_app_x - player->p_x);
				}

				/* Change directions */
				if (player->p_dir != player->p_desdir)
					changedir(player);

				/* Charge for speed */
				if ((player->p_fuel -= (player->p_ship.s_warpcost * player->p_speed))
						<= 0) {
					player->p_desspeed = 0;
				}

				/* Alter speed */
				if ((maxspeed = 36 * player->p_ship.s_engines
							* player->p_engine_health
							/ player->p_ship.s_hull / 256)
						> player->p_ship.s_maxspeed)
					maxspeed = player->p_ship.s_maxspeed;
				if (player->p_flags & PFENG)
					player->p_desspeed = 0;
				else if (player->p_desspeed > maxspeed)
					player->p_desspeed = maxspeed;

				if (player->p_desspeed > player->p_speed) {
					player->p_subspeed += player->p_ship.s_accint * 
						player->p_engine_health / 256;
				}
				else if (player->p_desspeed < player->p_speed) {
					player->p_subspeed -= player->p_ship.s_decint *
						player->p_engine_health / 256;
				}

				if (player->p_subspeed / 1000) {
					player->p_speed += player->p_subspeed / 1000;
					player->p_subspeed = player->p_subspeed % 1000;
					if (player->p_speed < 0) player->p_speed = 0;
					if (player->p_speed > maxspeed)
						player->p_speed = maxspeed;
				}

				player->p_x +=
					(double) player->p_speed * Cos[player->p_dir]*WARP1;
				player->p_y +=
					(double) player->p_speed * Sin[player->p_dir]*WARP1;

				/* black holes */
				planet = &planets[player->p_closest_planet];
				if (planet->pl_flags & PLMASK_BLACKHOLE) {
					int strength,dx,dy,dist;

					/* displace the victim */
					dx = planet->pl_x - player->p_x;
					dy = planet->pl_y - player->p_y;
					if (dist = Ihypot(dx,dy)) {
						strength = (BLACKHOLE_STR -
							(dist * BLACKHOLE_STR / NEBULA_RADIUS))
							* WARP1;
						if (strength > 0) {
							/* dx,dy is displacement vector */
							dx = dx * strength / dist;
							dy = dy * strength / dist;
							/* move player closer to black hole */
							player->p_x += dx;
							player->p_y += dy;
							if (dist < (ORBDIST*3/2)) {
								/* touching it */
								if ((sharedMemory->mode & WORMHOLE_MASK)
										|| (player->p_flags & PFPLKILLER)) {
									int pulsar,pdist;

									/* four hits of 25 strength */
									do_damage((PLAYER *) planet, player,
										25, WPN_PLANET, INTRAND(4));
									do_damage((PLAYER *) planet, player,
										25, WPN_PLANET, INTRAND(4));
									do_damage((PLAYER *) planet, player,
										25, WPN_PLANET, INTRAND(4));
									do_damage((PLAYER *) planet, player,
										25, WPN_PLANET, INTRAND(4));

									if (player->p_status == PALIVE) {
										if (player->p_speed == 0) {
											dx = INTRAND(200) - 100;
											dy = INTRAND(200) - 100;
										}
										else {
											dx = player->p_speed * 10 
												* Cos[player->p_dir];
											dy = player->p_speed * 10 
												* Sin[player->p_dir];
										}
	
										/* find the nearest pulsar */
										pulsar = closest_planet_dist(
											planet->pl_x + dx * (GWIDTH/100),
											planet->pl_y + dy * (GWIDTH/100),
											&pdist,
											TARG_PLANET|TARG_PLFLAGS_AND,
											PLMASK_PULSAR);
										if ((pdist = Ihypot(dx,dy)) == 0)
											dist = 1;
										player->p_x = planets[pulsar].pl_x
											+ dx*(ORBDIST*3)/pdist;
										player->p_y = planets[pulsar].pl_y
											+ dy*(ORBDIST*3)/pdist;
										player->p_closest_planet = pulsar;
									}
								}
								else {
									/* one hit of 500 strength */
									do_damage((PLAYER *) planet, player,
										500, WPN_PLANET, INTRAND(4));
								}
							}
						}
					}
				}
				else if (planet->pl_flags & PLMASK_AST_FIELD) {
					int k;

					if ((ABS(planet->pl_x - player->p_x)
								< AST_FIELD_RADIUS)
							&& (ABS(planet->pl_y - player->p_y)
								< AST_FIELD_RADIUS)) {
						if (player->p_speed) {
							/* damage the victim */
							if ((k = INTRAND(4)) == REAR_SHIELD)
								k = FRONT_SHIELD;
							do_damage((PLAYER *) planet, player,
								player->p_speed*2/3, WPN_PLANET, k);
							/* navagation is difficult */
							k = INTRAND(11) - 5;
							player->p_dir += k;
							player->p_desdir += k;
						}
					}
				}

				/* Bounce off the sides of the galaxy */
				if (player->p_x < 0) {
					player->p_x = -player->p_x;
					player->p_dir    = 256 - player->p_dir;
					player->p_desdir = 256 - player->p_desdir;
				}
				else if (player->p_x > GWIDTH) {
					player->p_x = (GWIDTH + GWIDTH) - player->p_x;
					player->p_dir    = 256 - player->p_dir;
					player->p_desdir = 256 - player->p_desdir;
				}
				if (player->p_y < 0) {
					player->p_y = -player->p_y;
					player->p_dir    = 128 - player->p_dir;
					player->p_desdir = 128 - player->p_desdir;
				}
				else if (player->p_y > GWIDTH) {
					player->p_y = (GWIDTH + GWIDTH) - player->p_y;
					player->p_dir    = 128 - player->p_dir;
					player->p_desdir = 128 - player->p_desdir;
				}
			}

			/* Charge for ECM and perturb ship image */
			if (player->p_flags & PFECM) {
				if (player->p_fuel < STFUEL_ECM) {
					player->p_flags &= ~PFECM;
					player->p_app_x = player->p_x;
					player->p_app_y = player->p_y;
				}
				else {
					player->p_fuel -= STFUEL_ECM;
					if (INTRAND(10) == 0) {
						/* perturb the ship's image */
						player->p_ecm_dx = INTRAND(2200) - 1100;
						player->p_ecm_dy = INTRAND(2200) - 1100;
					}
					player->p_app_x = player->p_x + player->p_ecm_dx;
					player->p_app_y = player->p_y + player->p_ecm_dy;
				}
			}
			else if (player->p_loanflags & PFLECM) {
				/* getting ECM from someone else */
				if (INTRAND(10) == 0) {
					/* perturb the ship's image */
					player->p_ecm_dx = INTRAND(2200) - 1100;
					player->p_ecm_dy = INTRAND(2200) - 1100;
				}
				player->p_app_x = player->p_x + player->p_ecm_dx;
				player->p_app_y = player->p_y + player->p_ecm_dy;
			}
			else {
				/* ship appears just where it is */
				player->p_app_x = player->p_x;
				player->p_app_y = player->p_y;
			}


			break;
		} /* end switch */
	}
	if (nplayers == MAXPLAYER) {
		if (dietime == -1)
			dietime = ticks + (600 / PLAYERFUSE);
	}
	else {
		dietime = -1;
	}
}


/* See if there is someone close enough to explode for */
static PLAYER *near(torp)
TORP *torp;
{
	register int dx,dy;
	register PLAYER *player;

	for (player = players; player < players+MAXPLAYER; ++player) {
		if (player->p_status != PALIVE) continue;
		if (torp->t_owner_no == player->p_no) continue;
		/* If not at war when fired, continue. */
		if ((sharedMemory->mode & SMART_WEAP_MASK)
		  		&& !(torp->t_war_mask & player->p_team_mask)
				&& !(player->p_flags & PFPLKILLER))
			continue;
		if ((dx = ABS(torp->t_x - player->p_x)) > EXPDIST) continue;
		if ((dy = ABS(torp->t_y - player->p_y)) > EXPDIST) continue;
		dx *= dx;   dy *= dy;  dx += dy;
		logmsg2("checking tdist with eucl dist %d\n",dx);
		if (dx < (EXPDIST * EXPDIST)) return(player);
	}

	/* no one near enough */
	return(NULL);
}


/* See if there is someone close enough to explode for */
static PLAYER *near_vortex(torp)
TORP *torp;
{
	register int dx,dy;
	register PLAYER *player;

	for (player = players; player < players+MAXPLAYER; ++player) {
		if (player->p_status != PALIVE) continue;
		if ((dx = ABS(torp->t_x - player->p_x)) > EXPDIST) continue;
		if ((dy = ABS(torp->t_y - player->p_y)) > EXPDIST) continue;
		dx *= dx;   dy *= dy;  dx += dy;
		logmsg2("checking tdist with eucl dist %d\n",dx);
		if (dx < (EXPDIST * EXPDIST)) return(player);
	}

	/* no one near enough */
	return(NULL);
}


/* Do damage to all surrounding players */
static int explode(torp)
TORP *torp;
{
	register int i;
	int damage,shield,dx,dy,dist;
	register PLAYER *player,*attacker;

	logmsg1("daemon: exploding torp\n");

	attacker = &players[torp->t_owner_no];

	for (i = 0, player = players; i < MAXPLAYER; i++, player++) {
		if (player->p_status != PALIVE) continue;
		if ((dx = ABS(torp->t_x - player->p_x)) > DETDIST) continue;
		if ((dy = ABS(torp->t_y - player->p_y)) > DETDIST) continue;
		if ((dist = dx*dx + dy*dy) > (DETDIST * DETDIST)) continue;
		if (dist > (EXPDIST * EXPDIST))
			damage = torp->t_damage * 
				(DETDIST - Isqrt(dist)) / (DETDIST-EXPDIST);
		else damage = torp->t_damage;

		shield = shield_hit(Atan2(torp->t_y - player->p_y,
			player->p_x - torp->t_x),player->p_dir);
		logmsg4("daemon: torp hits player %d in shield %d for %d\n",
			player->p_no,shield,damage);

		do_damage(attacker,player,damage,torp->t_class,shield);
	}
	torp->t_status = TEXPLODE;
	torp->t_fuse = (10/TORPFUSE);
}


static void udtorps()
{
	register int i,dx,dy;
	register TORP *torp;
	register PLAYER *target;
	PLANET *planet;
	COURSE olddir;

	logmsg1("daemon: updating torps\n");

	for (i = 0, torp = torps;
			i < (MAXPLAYER+1) * MAXTORP;
			i++, torp++) {
		if (torp->t_status == TFREE) continue;

		if (torp->t_status == TEXPLODE) {
			if (torp->t_fuse-- <= 0) {
				torp->t_status = TFREE;
				if (--players[torp->t_owner_no].p_ntorp < 0)
					players[torp->t_owner_no].p_ntorp = 0;
			}
			continue;
		}

		if (torp->t_status == TOFF) {
			torp->t_status = TFREE;
			if (--players[torp->t_owner_no].p_ntorp < 0)
				players[torp->t_owner_no].p_ntorp = 0;
			continue;
		}

       	if (torp->t_status == TTRACK) {
			target = &players[torp->t_target];
            /* if he's alive */
			if (target->p_status != PALIVE) {
				torp->t_status = TSTRAIGHT;
			}
			else if (target->p_obscured == OBSCURED_NEBULA) {
				torp->t_status = TSTRAIGHT;
			}
			else if (((target->p_flags & PFCLOAK)
						|| (target->p_loanflags & PFLCLOAK)
						|| (target->p_obscured == OBSCURED_PULSAR))
					&& INTRAND(15)) {
				/* don't correct course */
			}
			else {
				/* correct course */
				olddir = torp->t_dir;
                torp->t_dir = Atan2(torp->t_y - target->p_app_y, 
					target->p_app_x - torp->t_x);
				if (olddir != torp->t_dir) {
					torp->t_dx = Cos[torp->t_dir] *
						torp->t_speed * WARP1;
					torp->t_dy = Sin[torp->t_dir] *
						torp->t_speed * WARP1;
				}
			}
		}

		if ((torp->t_status == TMOVE) && (INTRAND(5)==0)) {
			olddir = INTRAND(5) - 2;
			if (olddir) {
				torp->t_dir += olddir;
				torp->t_dx = Cos[torp->t_dir] *
					torp->t_speed * WARP1;
				torp->t_dy = Sin[torp->t_dir] *
					torp->t_speed * WARP1;
			}
		}

		if ((torp->t_status == TRANDOM) && (INTRAND(20) == 0)) {
			/* offset the torp some */
			torp->t_x += ORBDIST - INTRAND(ORBDIST*2);
			torp->t_y += ORBDIST - INTRAND(ORBDIST*2);
			if (INTRAND(16) > torp->t_speed) {
				++torp->t_speed;
			}
			else {
				--torp->t_speed;
			}
			torp->t_dx = torp->t_speed * WARP1 * Cos[torp->t_dir];
			torp->t_dy = torp->t_speed * WARP1 * Sin[torp->t_dir];
		}

		if (torp->t_fuse-- <= 0) {
			torp->t_status = TOFF;
			break;
		}

		if ((torp->t_x += torp->t_dx) < 0) {
			torp->t_x = 0;
			torp->t_status = TOFF;
			break;
		}
		else if (torp->t_x > GWIDTH) {
			torp->t_x = GWIDTH;
			torp->t_status = TOFF;
			break;
		}

		if ((torp->t_y += torp->t_dy) < 0) {
			torp->t_y = 0;
			torp->t_status = TOFF;
			break;
		}
		else if (torp->t_y > GWIDTH) {
			torp->t_y = GWIDTH;
			torp->t_status = TOFF;
			break;
		}

		if (torp->t_class != WPN_PROB_VORTEX) {
			/* check planetary impact */
			planet = planets + torp->t_closest_planet;
			dx = torp->t_x - planet->pl_x;
			dy = torp->t_y - planet->pl_y;
			dx *= dx; dy *= dy; dx += dy;

			if ((dx < (ORBDIST*ORBDIST/2))
					&& !(planet->pl_flags & PLMASK_NEBULA)) {
				torp->t_status = TOFF;
				break;
			}
			else if ((planet->pl_flags & PLMASK_AST_FIELD)
					&& (dx < (AST_FIELD_RADIUS*AST_FIELD_RADIUS))) {
				/* hit asteroid field */
				explode(torp);
				break;
			}
			else if (dx > (INTERPLANETARY_DIST*INTERPLANETARY_DIST/2)) {
				torp->t_closest_planet = 
					closest_planet_dist(torp->t_x,torp->t_y,
						&torp->t_closest_planet_dist,
						TARG_PLANET|TARG_TRYDATA_FIRST,
						torp->t_closest_planet);
				planet = planets + torp->t_closest_planet;
			}

			/* check blackhole */
			if (planet->pl_flags & PLMASK_BLACKHOLE) {
				int strength,bdx,bdy,dist;
	
				/* displace the torp */
				bdx = planet->pl_x - torp->t_x;
				bdy = planet->pl_y - torp->t_y;
				dist = Ihypot(bdx,bdy);
				if (dist) {
					strength = (BLACKHOLE_STR -
						(dist * BLACKHOLE_STR / NEBULA_RADIUS))
						* WARP1;
					if (strength > 0) {
						torp->t_x += strength * bdx / dist;
						torp->t_y += strength * bdy / dist;
					}
				}
			}
		}
	
		if ((ticks % (10/TORPFUSE)) == 0)
			torp->t_damage -= torp->t_decay;

		if (torp->t_arm-- <= 0) {
			if (torp->t_class == WPN_PROB_VORTEX) {
				if ((target = near_vortex(torp)) != NULL) {
					explode_vortex(target,torp);
				}
			}
			else {
				if (near(torp) != NULL) {
					explode(torp);
				}
				if (torp->t_class == WPN_CHAFF) {
					/* Chaff keeps going after doing damage */
					torp->t_status = TSTRAIGHT;
				}
			}
		}
	}
}


/* save the planetfile */
static void save_planets()
{
	int plfd;

	logmsg1("daemon: saving planets\n");

	if ((plfd = open(Plfile,O_WRONLY|O_CREAT|O_TRUNC,0644)) >= 0) {
		write(plfd, (char *) planets, MAXPLANETS*sizeof(PLANET));
		close(plfd);
	}
	else {
		fprintf(stderr,"Couldn't open planet file for saving.\n");
	}
}


/* daemon signal handler and game-ending routine */
static void freemem(sig)
int sig;
{
	register PLAYER *player;
	int whydead;

	mysignal(SIGALRM,SIG_IGN);

	if (sig == -1) {
		whydead = KWINNER;
	}
	else if (sig == SIGUSR2) {
		whydead = KOVER;
	}
	else {
		whydead = KDAEMON;
	}
		
	/* Blow players out of the game */
	for (player = players; player < players+MAXPLAYER; player++) {
		if (player->p_status != PFREE) {
			player->p_status  = PDEAD;
			player->p_whydead = whydead;
			player->p_ntorp   = 0;
		}
	}

	sleep(4);
	for (player = players; player < players+MAXPLAYER; player++) {
		if ((player->p_status != PFREE) 
				&& (player->p_pid > 20)
				&& (kill(player->p_pid,0) == 0)) {
			kill(player->p_pid,SIGUSR1);
		}
	}

	if (sig == 0) logmsg1("\ndaemon: freemem because no players.\n");
	if (sig > 0) {
		FILE *diedlog;

		logmsg2("\ndaemon: RECEIVED SIGNAL %d\n",sig);
		diedlog = fopen("/tmp/dtrek.whydied","w");
		fprintf(diedlog,"dtrek daemon died after receiving signal %d\n",
			sig);
		fclose(diedlog);
	}
	logmsg1("daemon: freeing up shared memory\n");

	if (sig == -1) {
		/* delete the planetfile */
		unlink(Plfile);
	}
	else {
		save_planets();
	}
	sleep(30);
	shmctl(shmid, IPC_RMID, (struct shmid_ds *) 0);
	semctl(semid, 0, IPC_RMID, NULL);
	exit(0);
}


/* This function is called when a planet has been taken over.
 * It checks all the planets to see if the victory conditions
 * are right.  If so, it blows everyone out of the game and
 * resets the galaxy
 */
static void checkwin(winner)
PLAYER *winner;
{
	register int i;
	register PLANET *planet;
	register PLAYER *player;
	int teamcount[NUMTEAM];

	logmsg1("daemon: checking for winner\n");

	for (i = FEDERATION; i <= ORION; ++i) teamcount[i] = 0;

	for (planet = planets; planet < planets+MAXPLANETS; ++planet) {
		if ((planet->pl_owner_no >= FEDERATION)
				&& (planet->pl_owner_no <= ORION)) {
			++teamcount[planet->pl_owner_no];
		}
	}

	for (i = FEDERATION; i <= ORION; ++i) {
		if (teamcount[i] >= VICTORY) {
			mysignal(SIGALRM,SIG_IGN);
			/* We have a winning team */
			for (player = players; player < players+MAXPLAYER; player++) {
				player->p_status = PDEAD;
				player->p_whydead = KWINNER;
				player->p_whodead = winner->p_no;
				player->p_ntorp = 0;
			}
			++winner->p_stats.st_conqs;
			sleep(10);
			freemem(-1);
		}
	}
}


static void change_planet_owner(planet,taker)
PLANET *planet;
PLAYER *taker;
{
	int old_owner_no,old_owner_mask;
	char message[256],address[80];

	logmsg1("change_planet_owner\n");

	old_owner_no   = planet->pl_owner_no;
	old_owner_mask = planet->pl_owner_mask;

	if (taker != (PLAYER *) NULL) {
		if (planet->pl_armies > 0) {
			/* It's been taken over */
			logmsg1(" taken over\n");
			sprintf(message, "%s taken over by %s (%s)",
		   		planet->pl_name,
		    	taker->p_name,
				taker->p_mapchars);
			planet->pl_owner_mask = taker->p_team_mask;
			planet->pl_owner_no   = taker->p_team_no;
			checkwin(taker);
		}
		else {
			/* It's been destroyed */
			logmsg1(" destroyed\n");
			planet->pl_owner_mask = IND_MASK;
			planet->pl_owner_no   = INDEPENDENT;
			if (INTRAND(8) == 0) {
				/* barrenize it */
				planet->pl_flags |= (PLMASK_BARREN|PLMASK_TMPBARREN);
				planet->pl_deadtime = (5+INTRAND(6))*60
					*(1000000/UPDATE)/PLFIGHTFUSE;
				sprintf(message, "%s turned radioactive by %s (%s)",
		   			planet->pl_name,
		    		taker->p_name,
					taker->p_mapchars);
			}
			else {
				sprintf(message, "%s destroyed by %s (%s)",
		   			planet->pl_name,
		    		taker->p_name,
					taker->p_mapchars);
			}
		}
		/* tell bomber team & update their info */
		sprintf(address,"%-3s->%-3s",
	   		planet->pl_name,teamshort[taker->p_team_no]);
		pmessage(message, taker->p_team_mask, MTEAM, address);
		update_planet_info(taker->p_team_no,planet);
	}
	else {
		logmsg1(" independentized\n");
		/* It's gone independent */
       	sprintf(message,
			"%s has overthrown your stinking puppet dictator!",
			planet->pl_name);
		planet->pl_owner_mask = IND_MASK;
		planet->pl_owner_no   = INDEPENDENT;
	}

	if (old_owner_no != INDEPENDENT) {
		/* The losing team knows it's been taken */
		update_planet_info(old_owner_no,planet);

		/* tell old team */
		sprintf(address,"%-3s->%-3s",
	    	planet->pl_name,teamshort[old_owner_no]);
		pmessage(message, old_owner_mask, MTEAM, address);
	}
}



static void udplanets()
{
    register int i,defend_planet;
    register PLANET *planet;
    register PLAYER *player;
	int chance;

    for (i = 0, planet = planets; i < MAXPLANETS; i++, planet++) {
        if (planet->pl_couptime) planet->pl_couptime--;
        if (((planet->pl_armies < 5) || (planet->pl_no == IND_HOME))
				&& !(planet->pl_flags &
					(planet->pl_owner_mask
						| (PLMASK_BLACKHOLE
							| PLMASK_PULSAR
							| PLMASK_BARREN
							| PLMASK_NEBULA)))) {
            chance = INDCHANCE << planet->pl_armies;
			if (planet->pl_no == IND_HOME) {
				if ((chance >>= 2) < 2) chance = 2;
			}
            if (!(INTRAND(chance))) {
                if (planet->pl_armies > 0) {
                    /* Give planet to nobody */
                    planet->pl_armies = 0;
					change_planet_owner(planet,(PLAYER *) NULL);
                }
                /* check to make sure noone's orbiting the planet */
				if (planet->pl_flags & PLMASK_HOME) {
					defend_planet = FALSE;
				}
				else {
					defend_planet = TRUE;
                	for (player=players; player<=&players[MAXPLAYER]; ++player)
                    if (player->p_status == PALIVE)
                        if (player->p_flags & PFORBIT)
                            if (player->p_planet == planet->pl_no) {
                                defend_planet = FALSE;
                                break;
                            }
				}

                if (defend_planet)
                    if (INTRAND(MAXIND) <
							(MAXIND-player_count[INDEPENDENT])) {
						if (fork() == 0) {
							detach();
							setresuid(geteuid(),-1,-1);
							if (planet->pl_no == IND_HOME) {
								start_robot(INDEPENDENT,5.0,-1,
									IND_HOME,FALSE,"Starbase");
							}
							else {
                       			start_robot(INDEPENDENT,0.0,-1,
									planet->pl_no,FALSE,NULL);
							}
							_exit(1);
						}
                       	robot_fuse[INDEPENDENT] =
						 	(1800+(INTRAND(1800)))/TEAMFUSE;
                    }
                continue;
            }
		}
        if (planet->pl_armies > 0) {
	        if (INTRAND(3000) < planet->pl_armies)
	            planet->pl_armies -= INTRAND(planet->pl_armies/2);
	        else if ((planet->pl_armies < 4) && !(INTRAND(20)))
	            planet->pl_armies++;
	        else if (!INTRAND(10))
	            planet->pl_armies += INTRAND(3) + 1;
			if ((planet->pl_flags & PLMASK_AST_FIELD)
					&& (planet->pl_armies > 10))
				planet->pl_armies = 10;
		}
    }
}


static void udphaser()
{
	register int i;
	register PHASER *phas;
	register PLAYER *victim,*attacker;
	TORP *tptr;
	int shield = -1,k;

	logmsg1("daemon: updating phasers\n");
	phas = phasers;

	for (attacker=players; attacker < players+MAXPLAYER; ++attacker) {
		for (i=0; i<MAXBEAM; ++i, ++phas) {
			if (phas->ph_status == PHFREE) continue;
			if ((phas->ph_status == PHMISS)
					|| (phas->ph_status == PHHITLOCATION)) {
				if (phas->ph_fuse-- == 1) phas->ph_status = PHFREE;
			}
			else if (phas->ph_status == PHHITPLAYER) {
				victim = &players[phas->ph_target];
				if ((phas->ph_fuse)-- == attacker->p_ship.s_beamreload) {
					/* do the damage */
					shield = shield_hit(Atan2(attacker->p_y - victim->p_y,
						victim->p_x - attacker->p_x), victim->p_dir);
					if (phas->ph_class == WPN_FUSION) {
						for (k=0; k<4; ++k) {
							if (k == shield) {
								do_damage(attacker,victim,
									phas->ph_damage/2,phas->ph_class,k);
							}
							else {
								do_damage(attacker,victim,
									phas->ph_damage/6,phas->ph_class,k);
							}
						}
					}
					else {
						do_damage(attacker,victim,
							phas->ph_damage,phas->ph_class,shield);
					}
				}
				else if (victim->p_status != PALIVE) {
					/* make the beam stationary */
					phas->ph_x = victim->p_x; phas->ph_y = victim->p_y;
					phas->ph_status = PHHITLOCATION;
				}
				if (phas->ph_fuse == 0) phas->ph_status = PHFREE;
			}
			else {
				/* phas->ph_status == PHHITTORP */
				tptr = &torps[phas->ph_target];
				if ((phas->ph_fuse)-- == attacker->p_ship.s_beamreload) {
					/* do the damage */
					if (tptr->t_class == WPN_PROB_VORTEX) {
						explode_vortex((PLAYER *) NULL,tptr);
					}
					else {
						if (phas->ph_damage > tptr->t_damage)
							explode(tptr);
						else tptr->t_damage -= phas->ph_damage;
					}
				}
				else if (tptr->t_status == TFREE) {
					/* make the beam stationary */
					phas->ph_x = tptr->t_x; phas->ph_y = tptr->t_y;
					phas->ph_status = PHHITLOCATION;
				}
				if (phas->ph_fuse == 0) phas->ph_status = PHFREE;
			}
		}
	}
}



static void udtractor()
{
	register int i;
	register PLAYER *j,*victim;
	int dx,dy,len;
	int ddx,ddy;

    logmsg1("daemon: updating tractor\n");
	for (i=0,j=players; i<MAXPLAYER; ++i,++j) {
		if (j->p_status == PALIVE && j->p_tractors_on) {
		   /* we have got someone in our tractor beam */
		   logmsg1("tractor: Now trying to change things\n");
		   victim = &players[j->p_who_tractor];
		   if ( victim->p_status == PALIVE ) {
		   logmsg1("tractor: Now trying to change victim's x,y\n");
		   logmsg2("address of victim: %x\n",victim);
		   logmsg3("%d %d \n",victim->p_x,victim->p_y);
		   dx = victim->p_x - j->p_x;
		   dy = victim->p_y - j->p_y;
		   len = Ihypot(dx,dy);
		   if ( len <= TRACTOR_RANGE) {
		     ddx = (float)-dx/(float)(len+1) * WARP1 * j->p_tractors_on * 
			   ((float)STD_HULL_SIZE / (float)victim->p_ship.s_hull) ;
		     
		     ddy = (float)-dy/(float)(len+1) * WARP1 *  j->p_tractors_on *
			   ((float)STD_HULL_SIZE / (float)victim->p_ship.s_hull) ;

			if (victim->p_flags & PFMONITOR) {
				/* double tractor effectiveness */
				ddx <<= 1;
				ddy <<= 1;
			}

             if (j->p_tractor_type == TRACTOR ) {
		       victim->p_y += ddy;
		       victim->p_x += ddx;
			 }
			 else if ( j->p_tractor_type == REPULSOR ) {
		       victim->p_y -= ddy;
		       victim->p_x -= ddx;
			 }
			 else if ( j->p_tractor_type == WINCH ) {
		       j->p_y -= ddy;
		       j->p_x -= ddx;
			 }
			 else if ( j->p_tractor_type == PUSHER ) {
		       j->p_y += ddy;
		       j->p_x += ddx;
			 }
		   }
		   else {
			 j->p_tractors_on = 0;
		   }

           }
		   else {
			 j->p_tractors_on = 0;
           }
		}
	}
}


static void teamtimers()
{
	register int i;

	for (i = 0; i < NUMTEAM; i++) {
		if (robot_fuse[i] > 0) robot_fuse[i]--;
		if (coup_fuse[i] > 0)   coup_fuse[i]--;
	}
}

static void plfight()
{
	register int h,i;
	register PLAYER *player;
	register PLANET *planet,*pln;
	int rnd;
	char buf[80];
	char buf1[80];

	for (h = 0, planet = planets; h < MAXPLANETS; h++, planet++) {
		if (planet->pl_flags & PLMASK_COUP) {
			int oldowner = planet->pl_owner_no;

			planet->pl_flags &= ~PLMASK_COUP;
			planet->pl_owner_mask = (planet->pl_flags & ALL_MASK);
			planet->pl_owner_no = log_two[planet->pl_owner_mask];
			planet->pl_armies = 5;
			update_planet_info(oldowner, planet);
		}
		planet->pl_flags &= ~PLMASK_REDRAW;
		if (pl_warning_fuse[h] > 0) pl_warning_fuse[h]--;
		if (planet->pl_flags & PLMASK_WANDERER) {
			planet->pl_x += planet->pl_dx;
			planet->pl_y += planet->pl_dy;
			if (!INTRAND(WANDERERFUSE)) {
				COURSE dir;

				/* new course! */
				dir = (COURSE) INTRAND(256);
				planet->pl_dx = (int)(Cos[dir]*WARP1);
				planet->pl_dy = (int)(Sin[dir]*WARP1);
			}
			if (planet->pl_x < ORBDIST) {
				planet->pl_x = ORBDIST;
				planet->pl_dx = ABS(planet->pl_dx);
			}
			else if (planet->pl_x > (GWIDTH-ORBDIST)) {
				planet->pl_x = (GWIDTH-ORBDIST);
				planet->pl_dx = -ABS(planet->pl_dx);
			}
			if (planet->pl_y < ORBDIST) {
				planet->pl_y = ORBDIST;
				planet->pl_dy = ABS(planet->pl_dy);
			}
			else if (planet->pl_y > (GWIDTH-ORBDIST)) {
				planet->pl_y = (GWIDTH-ORBDIST);
				planet->pl_dy = -ABS(planet->pl_dy);
			}
			/* redraw any close-by planets */
			for (pln=planets; pln<planets+MAXPLANETS; ++pln) {
				if (ABS(planet->pl_x - pln->pl_x) < PLREDRAWDIST)
					pln->pl_flags |= PLMASK_REDRAW;
				else if (ABS(planet->pl_y - pln->pl_y) < PLREDRAWDIST)
					pln->pl_flags |= PLMASK_REDRAW;
			}
		}
		if (planet->pl_flags & PLMASK_TMPBARREN) {
			if (--(planet->pl_deadtime) <= 0) {
				planet->pl_flags   &= ~(PLMASK_BARREN|PLMASK_TMPBARREN);
				planet->pl_deadtime = 0;
				planet->pl_armies   = 1;
			}
		}
	}

	for (i = 0, player = players; i < MAXPLAYER; i++, player++) {
        if (player->p_status == PALIVE) {
            if (!(player->p_flags & PFORBIT)) {
				/* not orbiting */
                planet = &planets[player->p_closest_planet];
				player->p_closest_planet_dist = 
					Ihypot(player->p_x - planet->pl_x,
				           player->p_y - planet->pl_y);
				if (player->p_closest_planet_dist <= (ORBDIST+10)) {
					/* close enough to orbit */
                   	planet->pl_flags |= PLMASK_REDRAW;
					/* if pulsar, kill the player */
					if (planet->pl_flags & PLMASK_PULSAR) {
						/* one hit of 500 strength */
						do_damage((PLAYER *) planet, player,
							500, WPN_PLANET, INTRAND(4));
					}
					/* blackhole damage handled elsewhere */
					else if (!(planet->pl_flags
							& (PLMASK_BLACKHOLE|PLMASK_NEBULA))) {
						/* update planet info if has sp. sens. */
						if ((player->p_ship.s_sptechmask & STMASK_IMP_SENS)
								|| (player->p_loanflags & PFLIMP_SENS))
							update_planet_info(player->p_team_no, planet);
                		if ((player->p_hostile_mask)
						 		& planet->pl_owner_mask)
							/* At war with planet */		
                    		if (planet->pl_armies > 0)
								do_damage((PLAYER *) planet, player,
									planet->pl_armies/10+2, WPN_PLANET,
									INTRAND(4));
					}
				}
				else {
					if (player->p_closest_planet_dist
							> (INTERPLANETARY_DIST/2)) {
                		player->p_closest_planet =
                    		closest_planet_dist(player->p_x,player->p_y,
								&player->p_closest_planet_dist,
								(TARG_PLANET|TARG_TRYDATA_FIRST),
								player->p_closest_planet);
						planet = &planets[player->p_closest_planet];
					}
                	if (player->p_closest_planet_dist < PLREDRAWDIST) {
                    	planet->pl_flags |= PLMASK_REDRAW;
					}
				}
            }
            else {
				logmsg1("daemon: planets fighting\n");
				/* Orbiting */
                player->p_closest_planet = player->p_planet;
                player->p_closest_planet_dist = 1;
                planet = &planets[player->p_planet];
				update_planet_info(player->p_team_no,planet);
                planet->pl_flags |= PLMASK_REDRAW;
                if (((player->p_hostile_mask) & planet->pl_owner_mask)
						|| (player->p_flags & PFPLKILLER)) {
					/* At war with planet */		
                    if (planet->pl_armies > 0)
						do_damage((PLAYER *) planet, player,
							planet->pl_armies/10+2, WPN_PLANET,
							INTRAND(4));

					/* do bombing */
					if ((player->p_flags & PFBOMB)
							&& ((player->p_flags & PFPLKILLER)
								|| ((player->p_team_mask != planet->pl_owner_mask) 
									&& (planet->pl_armies >= 5)))) {
						/* Warn owning team */
						if (pl_warning_fuse[planet->pl_no] <= 0) {
							pl_warning_fuse[planet->pl_no] = 50/PLFIGHTFUSE;
							sprintf(buf, "%s%s %s.",
								"We are being attacked by ",
							    player->p_name, player->p_mapchars);
							sprintf(buf1, "%-3s->%-3s",
							    planet->pl_name,
								teamshort[planet->pl_owner_no]);
							pmessage(buf, planet->pl_owner_mask,
								MTEAM, buf1);
						}
			
						rnd = INTRAND(100);
						if (rnd > 60) {
							if (rnd < 85) {
								planet->pl_armies -= 1;
								/* player->p_kills += 0.02; */
							}
							else if (rnd < 96) {
								planet->pl_armies -= 2;
								/* player->p_kills += 0.04; */
							}
							else {
								planet->pl_armies -= 3;
								/* player->p_kills += 0.06; */
							}
			
							if ((planet->pl_owner_no != INDEPENDENT)
									&& (defender_count[planet->pl_owner_no]
										< defender_count[player->p_team_no])
									&& (robot_fuse[planet->pl_owner_no] <= 0)) {
								/* start defender robot */
								if (fork() == 0) {
									detach();
									setresuid(geteuid(),-1,-1);
									start_robot(planet->pl_owner_no,
										player->p_kills, player->p_no,-1,
										FALSE,NULL);
									_exit(1);
								}
								robot_fuse[planet->pl_owner_no] = 
									(900 + INTRAND(900)) / TEAMFUSE;
							}
						}
					} /* done bombing */
				} /* done fighting */			
			} /* done orbiting */
			if (player->p_closest_planet_dist
					< NEBULA_RADIUS) {
				if (planet->pl_flags & PLMASK_NEBULA) {
					player->p_obscured = OBSCURED_NEBULA;
					disrupt_lockons(player->p_no,10);
				}
				else if (planet->pl_flags & PLMASK_PULSAR) {
					player->p_obscured = OBSCURED_PULSAR;
				}
				else {
					player->p_obscured = OBSCURED_NONE;
				}
			}
			else {
				player->p_obscured = OBSCURED_NONE;
			}
			if (player->p_obscured == OBSCURED_PULSAR) {
				do_damage((PLAYER *) planet, player,
					3, WPN_PLANET, 
					shield_hit(Atan2(planet->pl_y - player->p_y,
					player->p_x - planet->pl_x),player->p_dir));
			}
		}
	}
}

/* This function checks to see if a team has been genocided --
 * their last planet has been beamed down to zero.  It will set
 * a timer on their home planet that will prevent them from
 * couping for a random number of minutes.
 */
static void checkgen(loser_mask)
int loser_mask;
{
	register int i;
	register PLANET *planet,*homep;

	logmsg1("daemon: checking for genocide\n");

	for (i = 0, planet = planets; i < MAXPLANETS; i++, planet++) {
		if (planet->pl_owner_mask == loser_mask)
			/* nope, he's still got a planet */
			return;
		if (((planet->pl_flags & ALL_MASK) == loser_mask)
				&& (planet->pl_flags & PLMASK_HOME))
			homep = planet;
	}

	/* well, I guess they are loser_masks.  Hose them now */
	homep->pl_couptime = (18000 + INTRAND(18000)) / PLANETFUSE;
}


static void beam()
{
	register int i;
	register PLAYER *player;
	register PLANET *planet;

	for (i = 0, player = players; i < MAXPLAYER; i++, player++) {
		if ((player->p_status != PALIVE)
				|| (!(player->p_flags & PFORBIT)))
			continue;
		planet = &planets[player->p_planet];
		if (player->p_flags & PFBEAMUP) {
			if (planet->pl_flags & (PLMASK_BLACKHOLE|PLMASK_PULSAR|
					PLMASK_BARREN|PLMASK_NEBULA)) {
				player->p_flags &= ~PFBEAMUP;
				continue;
			}
			logmsg1("daemon: beaming up\n");
			if (planet->pl_armies < 5) continue;
			if (player->p_armies == player->p_ship.s_cargo) continue;
			if (player->p_armies == (int) (player->p_kills * 2.0))
				continue;
			if (player->p_team_mask != planet->pl_owner_mask) continue;
			if (( player->p_hostile_mask)
					& planet->pl_owner_mask)
				continue;
			player->p_armies++;
			planet->pl_armies--;
		}
		else if (player->p_flags & PFBEAMDOWN) {
			if (planet->pl_flags & (PLMASK_BLACKHOLE|PLMASK_PULSAR|
					PLMASK_BARREN|PLMASK_NEBULA)) {
				player->p_flags &= ~PFBEAMDOWN;
				continue;
			}
			else if ((planet->pl_flags & PLMASK_AST_FIELD)
					&& (planet->pl_armies >= 10)) {
				player->p_flags &= ~PFBEAMDOWN;
				planet->pl_armies = 10;
				continue;
			}

			logmsg1("daemon: beaming down\n");
			if (player->p_armies == 0) continue;
			if ((!(( player->p_hostile_mask)
						& planet->pl_owner_mask))
			   	 	&& (player->p_team_mask != planet->pl_owner_mask)
					&& (planet->pl_owner_mask != IND_MASK))
				continue;

			if (player->p_team_mask != planet->pl_owner_mask) {
				player->p_armies--;
				if ((defender_count[planet->pl_owner_no] == 0)
						&& (planet->pl_owner_no != INDEPENDENT)
			   	 		&& robot_fuse[planet->pl_owner_no] <= 0) {
					/* send in a robotic defender */
					if (fork() == 0) {
						detach();
						setresuid(geteuid(),-1,-1);
						start_robot(planet->pl_owner_no,
							player->p_kills, player->p_no,-1,
							FALSE,NULL);
						_exit(1);
					}
					robot_fuse[planet->pl_owner_no] = 
						(1800 + INTRAND(1800)) / TEAMFUSE;
				}
				if (planet->pl_armies) {
					int oldowner_mask = planet->pl_owner_mask;

					planet->pl_armies--;
					player->p_kills += 0.02;
					if (planet->pl_armies == 0) {
						logmsg1("giving planet to noone\n");
						/* Give planet to nobody */
						change_planet_owner(planet,player);
						checkgen(oldowner_mask);
					}
				}
				else { 	/* planet taken over */
					logmsg1("planet taken over\n");
					planet->pl_armies++;
					player->p_stats.st_planets++;
					player->p_kills += 0.25;
					change_planet_owner(planet,player);
				}
			}
			else {
				player->p_armies--;
				planet->pl_armies++;
			}
		}
	}
}


static void repairplayers()
{
	PLAYER *player;

	for (player=players; player<players+MAXPLAYER; ++player) {
		if (player->p_status != PALIVE) continue;
		repair_player(player);
	}
}

/* send in an independent robot (perhaps) */
static void jugger()
{
	int numplayers;

	if (nactiveplayers == 0) return;

	if (sharedMemory->mode & ROBOT_SWARM_MASK) {
		if (nactiveplayers > 4) numplayers = 4;
		else numplayers = nactiveplayers;
		if (INTRAND(numplayers)+INTRAND(numplayers)+8
				> defender_count[INDEPENDENT]) {
			logmsg1("daemon: invading jugger\n");
			/* send in a juggernaut */
			if (fork() == 0) {
				detach();
				setresuid(geteuid(),-1,-1);
				start_robot(JUGGERNAUT,2.0,-1,-1,FALSE,NULL);
				_exit(1);
			}
		}
	}
	else if (sharedMemory->mode & ROBOT_INVASION_MASK) {
		if (nactiveplayers > 4) numplayers = 4;
		else numplayers = nactiveplayers;
		if (INTRAND(numplayers)+INTRAND(numplayers)+3
				> defender_count[INDEPENDENT]) {
			logmsg1("daemon: invading jugger\n");
			/* send in a juggernaut */
			if (fork() == 0) {
				detach();
				setresuid(geteuid(),-1,-1);
				start_robot(JUGGERNAUT,0.0,-1,-1,FALSE,NULL);
				_exit(1);
			}
		}
	}
	else {
		if (INTRAND(JUGGERCHANCE) == 0)
			if (defender_count[INDEPENDENT] < MAXJUGGERS)
				if (INTRAND(3)*INTRAND(3) > defender_count[INDEPENDENT]) {
					logmsg1("daemon: jugger\n");
					if (fork() == 0) {
						detach();
						setresuid(geteuid(),-1,-1);
						/* send in a juggernaut */
						start_robot(JUGGERNAUT,0.0,-1,-1,FALSE,NULL);
						_exit(1);
					}
				}
	}
}


/* send in defense bases for each team */
static void playerbases()
{
	if (fork() == 0) {
		detach();
		setresuid(geteuid(),-1,-1);
		start_defense_bases();
		_exit(1);
	}
}


/* send in monitors for each team */
static void playermonitors()
{
	if (fork() == 0) {
		detach();
		setresuid(geteuid(),-1,-1);
		start_monitors();
		_exit(1);
	}
}


/* send in a planetkiller */
static void send_in_planetkiller()
{
	if (INTRAND(PLANETKILLERCHANCE)) return;

	if (fork() == 0) {
		detach();
		setresuid(geteuid(),-1,-1);
		start_planetkiller();
		_exit(1);
	}
}


void launch_pvortex()
{
	TORP *torp;
	int count;

	/* count number presently active */
	for (torp = &torps[MAXPLAYER*MAXTORP], count=MAXPLAYER*MAXTORP;
			torp < &torps[MAXPLAYER*MAXTORP + MAXPVORTICES];
			++torp,++count) {
		if (torp->t_status == TFREE) break;
	}

	if (torp == &torps[MAXPLAYER*MAXTORP + MAXPVORTICES]) {
		/* no room for more */
		return;
	}

	torp->t_no        = count;
	torp->t_status    = TRANDOM;
	torp->t_owner_no  = 0;

	torp->t_damage    = 0;
	torp->t_speed     = 8;
	torp->t_fuse      = 100000;
	torp->t_arm       = 0;
	torp->t_war_mask  = ALL_MASK;
	torp->t_team_mask = IND_MASK;
	torp->t_decay     = 0;
	torp->t_class     = WPN_PROB_VORTEX;
	torp->t_closest_planet = 0;
	torp->t_closest_planet = 1000;

	switch (INTRAND(4)) {
		case 0:
			torp->t_x   = GWIDTH/10 + INTRAND(GWIDTH*8/10);
			torp->t_y   = 10;
			torp->t_dir = 72 + INTRAND(112);
			break;
		case 1:
			torp->t_x   = GWIDTH/10 + INTRAND(GWIDTH*8/10);
			torp->t_y   = GWIDTH-10;
			torp->t_dir = 200 + INTRAND(112);
			break;
		case 2:
			torp->t_x   = 10;
			torp->t_y   = GWIDTH/10 + INTRAND(GWIDTH*8/10);
			torp->t_dir = 8 + INTRAND(112);
			break;
		case 3:
			torp->t_x   = GWIDTH-10;
			torp->t_y   = GWIDTH/10 + INTRAND(GWIDTH*8/10);
			torp->t_dir = 136 + INTRAND(112);
			break;
	}

#ifdef VORTEX_TESTING
	torp->t_x = GWIDTH/2;
	torp->t_y = 10;
	torp->t_dir = 128;
#endif

	torp->t_dx        = Cos[torp->t_dir] * torp->t_speed * WARP1;
	torp->t_dy        = Sin[torp->t_dir] * torp->t_speed * WARP1;
}


static void move()
{
	if (++ticks == dietime)	/* no player for 1 minute. kill self */
		freemem(0);
	if (fuse(PLAYERFUSE)) {
		udplayers();
	}
	if (fuse(ALERTFUSE)) {
		udalert_visibility();
	}
	if (fuse(REPAIRFUSE)) {
		repairplayers();
	}
	if (fuse(TORPFUSE)) {
		udtorps();
	}
	if (fuse(PHASERFUSE)) {
		udphaser();
		udtractor();
	}
	if (fuse(TEAMFUSE)) {
		teamtimers();
	}
	if (fuse(PLFIGHTFUSE)) {
		plfight();
	}
	if (fuse(BEAMFUSE)) {
		beam();
	}
	if (fuse(SYNCFUSE)) {
		save_planets();
	}
	if (fuse(PLANETFUSE)) {
		udplanets();
	}
	if (fuse(JUGGERFUSE)) {
		jugger();
	}
	if (sharedMemory->mode & DEFENSE_BASE_MASK)  {
		if (fuse(BASEFUSE)) {
			playerbases();
		}
	}
	if (sharedMemory->mode & MONITOR_MASK)  {
		if (fuse(BASEFUSE)) {
			playermonitors();
		}
	}
	if (sharedMemory->mode & PLANETKILLER_MASK)  {
		if (fuse(PLANETKILLERFUSE)) {
			send_in_planetkiller();
		}
	}
	if (sharedMemory->mode & PROB_VORTEX_MASK)  {
		if (fuse(PVORTEXFUSE)) {
			launch_pvortex();
		}
	}
	if ((sharedMemory->game_end_time > 0)
			&& fuse(GAMEENDFUSE)
			&& (time((long *) 0) > sharedMemory->game_end_time)) {
		/* kill myself in an orderly manner */
		freemem(SIGUSR2);
	}
}


void warning(str)
char *str;
{
	/* a stub */
}


main(argc,argv)
int argc;
char *argv[];
{
	register int i;
	key_t shmemKey = PKEY;
	struct shmid_ds smbuf;

	construct_filenames();

	OPEN_LOGFILE(Dlogfilename);

	seedrandom();

	for (i = 0; i < NSIG; i++) {
		mysignal(i, freemem);
	}
	mysignal(SIGUSR1,SIG_IGN);
	mysignal(SIGCLD,SIG_IGN);
	setpgrp();

	/* Kill any existing segments */

	if ((shmid = shmget(shmemKey, 0, 0)) >= 0) {
		fprintf(stderr, "Killing existing segment\n");
		shmctl(shmid, IPC_RMID, (struct shmid_ds *) 0);
	}

	shmid = shmget(shmemKey, sizeof(MEMORY), IPC_CREAT | 0700);
	if (shmid < 0) {
		perror("can't open shared memory");
		exit (1);
	}
	/* Hose the robots */
	shmctl(shmid, IPC_STAT, &smbuf);
	smbuf.shm_perm.uid = geteuid();
	smbuf.shm_perm.mode = 0700;
	shmctl(shmid, IPC_SET, &smbuf);

	sharedMemory = (MEMORY *) shmat(shmid, 0, 0);
	if (sharedMemory == (MEMORY *) -1) {
		perror("shm attach");
		exit (1);
	}

	players  = sharedMemory->players;
	torps    = sharedMemory->torps;
	status   = sharedMemory->status;
	planets  = sharedMemory->planets;
	phasers  = sharedMemory->phasers;
	mctl     = sharedMemory->mctl;
	messages = sharedMemory->messages;
	team	 = sharedMemory->team;
	watchers = sharedMemory->watchers;

	for (i = 0; i < MAXPLAYER; i++) players[i].p_status = PFREE;
	for (i = 0; i < MAXWATCHERS; ++i) watchers[i].w_status = PFREE;

	/*  assign initial team Treaty mask  */
	team[FEDERATION].t_treaty = FED_MASK;
	team[ORION].t_treaty = ORI_MASK;
	team[KLINGON].t_treaty = KLI_MASK;
	team[ROMULAN].t_treaty = ROM_MASK;
	team[INDEPENDENT].t_treaty = IND_MASK;

	team[FEDERATION].t_ally = FED_MASK;
	team[ORION].t_ally = ORI_MASK;
	team[KLINGON].t_ally = KLI_MASK;
	team[ROMULAN].t_ally = ROM_MASK;
	team[INDEPENDENT].t_ally = IND_MASK;

	read_planet_file();
	status->active_mask = 0;

	/* mysignal(SIGCLD, reaper); */
	mysignal(SIGCLD,SIG_IGN);
	mysignal(SIGALRM, move);

	logmsg1("daemon: setting up itimer\n");
	{
		PLAYER *player;
		int breakit = FALSE,count=0;

		while (++count < 300) {
			for (player=players;player<players+MAXPLAYER;++player)
				if ((player->p_status == PALIVE)
						|| (player->p_status == PIMMUNE)) {
					breakit = TRUE;
				}
			if (breakit) break;
			sleep(1);
		}
	}
	udt.it_interval.tv_sec = 0;
	udt.it_interval.tv_usec = UPDATE;
	udt.it_value.tv_sec = 0;
	udt.it_value.tv_usec = UPDATE;
	(void) setitimer(ITIMER_REAL, &udt, (struct itimerval *) 0);

	(void) setjmp(env);

	while (TRUE) {
		pause();
	}
}
