      subroutine bzone_screen_init (bitmap_desc)
c
c    *******************************************************************
c    *****                                                         *****
c    *****                BATTLE ZONE VERSION 1.0                  *****
c    *****                                                         *****
c    *****                      written by                         *****
c    *****                                                         *****
c    *****                 Justin S. Revenaugh                     *****
c    *****                                                         *****
c    *****                        5/86                             *****
c    *****                                                         *****
c    *****        Massachussetts Institute of Technology           *****
c    *****  Department of Earth, Atmospheric and Planetary Science *****
c    *****                                                         *****
c    *******************************************************************
c
c     BZONE_SCREEN_INIT initializes the GPR package, and draws the
c     static display panel.
c
% include '/sys/ins/base.ins.ftn'
% include '/sys/ins/gpr.ins.ftn'
c
      integer*2 config, size(2), font_1, font_2, window(2, 2), xpt, ypt
      integer*2 pt(2), radius, box_x(4, 4), box_y(4, 4), font_3
      integer*2 line_x(2, 8), line_y(2, 8), scan_x(8), scan_y(8)
      integer*2 score_x(4), score_y(4), flash_x(4, 2), flash_y(4, 2)
      integer*2 scrn_x(9), scrn_y(9), key_set(16), hdm_size(2)
      integer*2 tank_x(13), tank_y(13), sw_trap(6, 2)
      integer*4 status, bitmap_desc, value(16), icol(0:15)
      logical in_color
      character text*30, answer*1
      common /color$/ icol, in_color
c
c    data for window sizes
c
      data size / 2*1024/
      data value / 0, 16777215, 65280, 57600, 51200, 44800, 38400,
     &             32000, 16777000, 14803200, 13158400, 11513600,
     &             9868800, 8224000, 9868950, 16711700/
      data box_x /2*50, 2*950, 2*53, 2*947, 2*50, 2*950, 2*53, 2*947/
      data box_y /450, 2*50, 450, 447, 2*53, 447, 660, 2*460, 660, 657,
     &            2*463, 657/
      data line_x /58, 66, 56, 68, 938, 938, 942, 934, 56, 68, 934,
     &             942, 58, 66, 932, 944/
      data line_y /66, 58, 438, 438, 56, 68, 442, 434, 472, 472, 468,
     &             476, 644, 652, 648, 648/
      data scan_x /500, 500, 500, 500, 435, 440, 565, 560/
      data scan_y /495, 500, 625, 620, 560, 560, 560, 560/
      data score_x /590, 590, 710, 710/
      data score_y /550, 520, 520, 550/
      data flash_x /590, 590, 645, 645, 655, 655, 710, 710/
      data flash_y /600, 570, 570, 600, 600, 570, 570, 600/
      data scrn_x / 75, 925, 936, 936, 925, 75, 64, 64, 75/
      data scrn_y / 64, 64, 75, 425, 436, 436, 425, 75, 64/
      data tank_x / 974, 976, 960, 960, 978, 980, 988, 990, 994,
     &              990, 965, 960, 994/
      data tank_y / 47, 43, 43, 42, 42, 40, 40, 45, 44, 55, 55, 49, 44/
      data sw_trap / 969, 971, 60, 969, 971, 70, 969, 971, 100, 969,
     &               971, 110/
      data key_set / 16 * 16#ffff/
      do 10 i = 1, 15
        icol(i) = 1
 10   continue
      icol(0) = 0
      in_color = .false.
c
c    if this is a color node, then prompt for color option
c
      call gpr_$inq_config (config, status)
      if (config .ge. 2 .and. config .lt. 9) then
        in_color = .true.
        write (*, 100) 'Do you want fading colors? (y/n): '
100     format ( a, $)
        read (*,'(a)') answer
        if (answer .eq. 'n' .or. answer .eq. 'N') then
          do 20 i = 4, 8
            value(i) = value(3)
 20       continue
          do 30 i = 10, 14
            value(i) = value(9)
 30       continue
        end if
      end if
c
c    init screen in borrow mode
c
      call gpr_$init (gpr_$borrow, int2(1), size, int2(3), bitmap_desc,
     &                status)
c
c    load color map if this is a color node
c
      if (in_color) then
        call gpr_$set_color_map (0, int2(16), value, status)
        do 40 i = 0, 15
          icol(i) = i
 40     continue
      end if
c
c    load all necessary fonts
c
      call gpr_$load_font_file ('/sys/dm/fonts/f5x9', int2(18), font_1,
     &                          status)
      call gpr_$load_font_file ('/sys/dm/fonts/i.12', int2(18), font_2,
     &                          status)
      call gpr_$load_font_file ('/sys/dm/fonts/f7x13',int2(19), font_3,
     &                          status)
c
c    set up all the draw and text values
c
      call gpr_$set_text_path (gpr_$right, status)
      call gpr_$set_text_value (icol(1), status)
      call gpr_$set_text_background_value (icol(0), status)
      call gpr_$set_fill_value (icol(1), status)
      call gpr_$set_draw_value (icol(1), status)
c
c    draw high and low sensitivity swithces
c
      radius = 4
      pt(1) = 970
      pt(2) = 70
      call gpr_$circle (pt, radius, status)
      radius = 5
      call gpr_$circle (pt, radius, status)
      radius = 4
      pt(1) = 970
      pt(2) = 100
      call gpr_$circle (pt, radius, status)
      radius = 5
      call gpr_$circle (pt, radius, status)
      call gpr_$trapezoid (sw_trap(1, 1), status)
      call gpr_$trapezoid (sw_trap(1, 2), status)
c
c    draw the joystick ball
c
      pt(1) = 980
      pt(2) = 20
      radius = 10
      call gpr_$circle_filled (pt, radius, status)
      call gpr_$set_fill_value (0, status)
c
c    draw the players tank for tanks left display
c
      call gpr_$move (tank_x(1), tank_y(1), status)
      call gpr_$polyline (tank_x, tank_y, int2(13), status)
c
c    begin to draw the info panel, start with the outline boxes
c
      call gpr_$move (box_x(4, 1), box_y(4, 1), status)
      call gpr_$polyline (box_x(1, 1), box_y(1, 1), int2(4), status)
      call gpr_$move (box_x(4, 2), box_y(4, 2), status)
      call gpr_$polyline (box_x(1, 2), box_y(1, 2), int2(4), status)
      call gpr_$move (box_x(4, 3), box_y(4, 3), status)
      call gpr_$polyline (box_x(1, 3), box_y(1, 3), int2(4), status)
      call gpr_$move (box_x(4, 4), box_y(4, 4), status)
      call gpr_$polyline (box_x(1, 4), box_y(1, 4), int2(4), status)
      call gpr_$move (scrn_x(9), scrn_y(9), status)
      call gpr_$polyline (scrn_x, scrn_y, int2(9), status)
      pt(1) = 62
      pt(2) = 62
      radius = 6
      call gpr_$circle (pt, radius, status)
      pt(1) = 938
      call gpr_$circle (pt, radius, status)
      pt(2) = 438
      call gpr_$circle (pt, radius, status)
      pt(1) = 62
      call gpr_$circle (pt, radius, status)
      pt(1) = 62
      pt(2) = 472
      call gpr_$circle (pt, radius, status)
      pt(1) = 938
      call gpr_$circle (pt, radius, status)
      pt(2) = 648
      call gpr_$circle (pt, radius, status)
      pt(1) = 62
      call gpr_$circle (pt, radius, status)
      call gpr_$multiline (line_x(1, 1), line_y(1, 1), int2(2), status)
      call gpr_$multiline (line_x(1, 2), line_y(1, 2), int2(2), status)
      call gpr_$multiline (line_x(1, 3), line_y(1, 3), int2(2), status)
      call gpr_$multiline (line_x(1, 4), line_y(1, 4), int2(2), status)
      call gpr_$multiline (line_x(1, 5), line_y(1, 5), int2(2), status)
      call gpr_$multiline (line_x(1, 6), line_y(1, 6), int2(2), status)
      call gpr_$multiline (line_x(1, 7), line_y(1, 7), int2(2), status)
      call gpr_$multiline (line_x(1, 8), line_y(1, 8), int2(2), status)
c
c    draw scanner circle and joystick circle
c
      pt(1) = 500
      pt(2) = 560
      radius = 65
      call gpr_$circle (pt, radius, status)
      radius = 64
      call gpr_$circle (pt, radius, status)
      pt(1) = 855
      pt(2) = 560
      radius = 65
      call gpr_$circle (pt, radius, status)
      radius = 64
      call gpr_$circle (pt, radius, status)
      call gpr_$multiline (scan_x, scan_y, int2(8), status)
      call gpr_$move (flash_x(4, 1), flash_y(4, 1), status)
      call gpr_$polyline (flash_x(1, 1), flash_y(1, 1), int2(4), status)
      call gpr_$move (flash_x(4, 2), flash_y(4, 2), status)
      call gpr_$polyline (flash_x(1, 2), flash_y(1, 2), int2(4), status)
      call gpr_$move (score_x(4), score_y(4), status)
      call gpr_$polyline (score_x(1), score_y(1), int2(4), status)
c
c    insert text
c
      call gpr_$set_text_font (font_2, status)
      call gpr_$move (int2(50), int2(45), status)
      text = 'BATTLE ZONE V.I'
      call gpr_$text (text, int2(30), status)
      call gpr_$move (int2(850), int2(503), status)
      text = 'F'
      call gpr_$set_text_value (icol(15), status)
      call gpr_$text (text, int2(30), status)
      call gpr_$move (int2(850), int2(630), status)
      text = 'B'
      call gpr_$text (text, int2(30), status)
      call gpr_$move (int2(913), int2(570), status)
      text = 'R'
      call gpr_$text (text, int2(30), status)
      call gpr_$move (int2(785), int2(570), status)
      text = 'L'
      call gpr_$text (text, int2(30), status)
      call gpr_$set_text_font (font_1, status)
      call gpr_$move (int2(595), int2(590), status)
      call gpr_$set_text_value (icol(1), status)
      text = 'Missile'
      call gpr_$text (text, int2(7), status)
      call gpr_$move (int2(664), int2(590), status)
      text = 'Lander'
      call gpr_$text (text, int2(6), status)
      call gpr_$move (int2(920), int2(608), status)
      text = 'H'
      call gpr_$text (text, int2(1), status)
      call gpr_$move (int2(920), int2(642), status)
      text = 'L'
      call gpr_$text (text, int2(1), status)
c
c    enable mouse and keyboard events
c
      call gpr_$enable_input (gpr_$buttons, key_set, status)
      call gpr_$enable_input (gpr_$keystroke, key_set, status)
c
c    set clipping window and raster op
c
      window(1,1) = 70
      window(2,1) = 75
      window(1,2) = 860
      window(2,2) = 350
      call gpr_$set_clip_window (window, status)
      call gpr_$set_clipping_active (.true., status)
      call gpr_$set_text_font (font_3, status)
c
c    set the cursor to the zero speed and rotate point
c
      pt(1) = 500
      pt(2) = 400
      call gpr_$set_cursor_position (pt ,status)
      return
      end




      subroutine bzone_update_display (missile, lander, score, num_left,
     &                                 sens, bitmap_desc)
c
c     BZONE_UPDATE_DISPLAY is called whenever the static display
c     needs to be updated. This occurs when the player scores, dies or
c     when a enemy missile or lander is generated or lost.  Program
c     keeps current values in static storage so as to only update
c     changes.
c
% include '/sys/ins/base.ins.ftn'
% include '/sys/ins/gpr.ins.ftn'
c
      integer*2 wind_1(2, 2), dest_1(2), wind_2(2, 2), dest_2(2)
      integer*2 wind_3(2, 2), origin(2, 4), wind_4(2, 2), wind_5(2, 2)
      integer*2 switch(2)
      integer*4 score, current_score, bitmap_desc, status
      integer*4 num_left, current_num_left, icol(0:15)
      logical missile, lander, flasher(2), in_color, sens
      logical current_sens
      character text*10
      common /color$/ icol, in_color
      save flasher, current_num_left, current_score
      save dest_1, dest_2, origin, current_sens, switch
      save wind_1, wind_2, wind_3, wind_4, wind_5
      data wind_1 / 590, 570, 55, 30/
      data wind_2 / 655, 570, 55, 30/
      data wind_3 / 960, 40, 35, 16/
      data wind_4 / 964, 59, 12, 22/
      data wind_5 / 964, 89, 12, 22/
      data dest_1 / 590, 570/
      data dest_2 / 655, 570/
      data switch / 916, 610/
      data flasher / 2*.false./
      data origin / 733, 493, 733, 533, 733, 573, 733, 613/
      data current_score, current_num_left / -1, 0/
      data current_sens / .true./
c
c    check for new info
c
      if ((flasher(1) .and. .not. (missile)) .or.
     &    (.not. (flasher(1)) .and. missile)) then
        flasher(1) = missile
        call gpr_$set_raster_op (int2(0), int2(10), status)
        call gpr_$bit_blt (bitmap_desc, wind_1, int2(0), dest_1,
     &                     int2(0), status)
        call gpr_$set_raster_op (int2(0), int2(3), status)
      end if
      if ((flasher(2) .and. .not. (lander)) .or.
     &    (.not. (flasher(2)) .and. lander)) then
        flasher(2) = lander
        call gpr_$set_raster_op (int2(0), int2(10), status)
        call gpr_$bit_blt (bitmap_desc, wind_2, int2(0), dest_2,
     &                     int2(0), status)
        call gpr_$set_raster_op (int2(0), int2(3), status)
      end if
      if (score .ne. current_score) then
        current_score = score
        call gpr_$move (int2(608), int2(542), status)
        write (text, '(i10)') score
        call gpr_$text (text, int2(10), status)
      end if
c
c    do number of tanks left
c
      if (num_left .ne. current_num_left) then
        if ((num_left .lt. current_num_left) .and.
     &      (num_left .ge. 0)) then
c
c    erase the next tank on the list
c
          call gpr_$set_raster_op (int2(0), int2(6), status)
          call gpr_$bit_blt (bitmap_desc, wind_3, int2(0),
     &                       origin(1, current_num_left), int2(0),
     &                       status)
          call gpr_$set_raster_op (int2(0), int2(3), status)
          current_num_left = num_left
        else if (current_num_left .lt. 4) then
c
c    add a new tank
c
          call gpr_$bit_blt (bitmap_desc, wind_3, int2(0),
     &                       origin(1, num_left), int2(0), status)
          current_num_left = num_left
        end if
      end if
c
c    change the sensitivity setting
c
      if (sens .and. (.not. current_sens)) then
        call gpr_$bit_blt (bitmap_desc, wind_4, int2(0),
     &                     switch, int2(0), status)
        current_sens = sens
      else if ((.not. sens) .and. current_sens) then
        call gpr_$bit_blt (bitmap_desc, wind_5, int2(0),
     &                     switch, int2(0), status)
        current_sens = sens
      end if
      return
      end






      subroutine bzone_joystick (bitmap_desc, position, sens, speed,
     &                           rotate)
c
c     BZONE_JOYSTICK draws the joystick after calculating the current
c     speed and rotation rate settings.
c
% include '/sys/ins/base.ins.ftn'
% include '/sys/ins/gpr.ins.ftn'
c
      integer*2 position(2), dest_origin(2), window(4)
      integer*2 stick_x(4), stick_y(4)
      integer*4 bitmap_desc status, icol(0:15)
      real speed, rotate, sscale(2), rscale(2)
      logical active, first, sens, draw, in_color
      character*1 key
      common /color$/ icol, in_color
      save window, sscale, rscale, dest_origin, stick_x, stick_y
      save first, draw
      data window /970, 10, 2*21/
      data dest_origin / 845, 550/
      data sscale, rscale / 0.05, 0.025, 0.00015, 0.000075/
      data stick_x, stick_y / 8*0/
      data first, draw / 2 * .true./
c
c    locate the cursor and calculate speed and rotation rate
c
      index = 2
      if (sens) index = 1
      speed = -(position(2) - 400.0) * sscale(index)
      rotate = (position(1) - 500.0) * rscale(index)
      if (abs(speed) .gt. 10.0) speed = sign (10.0, speed)
      if (abs(rotate) .gt. 0.030) rotate = sign (0.030, rotate)
c
c    scale the two quantities to reflect tank track restrictions
c
      scrot = rotate * 10.0 / 0.030
      sctot = sqrt(speed**2 + scrot**2)
      if (sctot .gt. 10.0) then
        scale = 10.0 / sctot
        speed = speed * scale
        rotate = rotate * scale
      end if
c
c    delete old joystick
c
      if (draw) then
        call gpr_$set_raster_op (int2(0), int2(6), status)
        call gpr_$set_clipping_active (.false., status)
        if (.not. first) then
          call gpr_$bit_blt (bitmap_desc, window, int2(0), dest_origin,
     &                       int2(0), status)
          call gpr_$set_draw_value (icol(0), status)
          call gpr_$set_raster_op (int2(0), int2(3), status)
          call gpr_$multiline (stick_x, stick_y, int2(4), status)
        else
          first = .false.
          call gpr_$set_raster_op (int2(0), int2(3), status)
        end if
        call gpr_$set_draw_value (icol(1), status)
c
c    draw the new stick
c
        dest_origin(1) =  35.0 * (rotate / 0.03) + 845.0
        dest_origin(2) = -35.0 * (speed  / 10.0)  + 550.0
        call gpr_$bit_blt (bitmap_desc, window, int2(0), dest_origin,
     &                     int2(0), status)
        xl = rotate / 0.030
        yl = speed / 10.0
        tl = sqrt(xl**2 + yl**2)
        if (tl .gt. 0.05) then
          yoff = xl / tl * 9.0
          xoff = yl / tl * 9.0
        else
          yoff = 0.0
          xoff = 0.0
        end if
        stick_x(1) = 855
        stick_x(2) = dest_origin(1) + 10 + xoff / 2
        stick_x(3) = 855
        stick_x(4) = dest_origin(1) + 10 - xoff / 2
        stick_y(1) = 560
        stick_y(2) = dest_origin(2) + 10 + yoff / 2
        stick_y(3) = 560
        stick_y(4) = dest_origin(2) + 10 - yoff / 2
        call gpr_$multiline (stick_x, stick_y, int2(4), status)
        call gpr_$set_clipping_active (.true., status)
c
c    flip the sign of rotate so that angles are measured ccw
c
      end if
      draw = .not. draw
      rotate = - rotate
      return
      end




      subroutine bzone_scanner (pro_x, pro_y, orange)
c
c     BZONE_SCANNER draws the radar view of the surrounds
c
% include '/sys/ins/base.ins.ftn'
% include '/sys/ins/gpr.ins.ftn'
c
      integer*2 xpt(2), ypt(2), dot_x(4), dot_y(4)
      integer*4 icol(0:15), count, color
      real pro_x(12), pro_y(12), orange(12), scale, sda, cda, sa, ca
      logical missile, dir, in_color, draw
      common /color$/ icol, in_color
      save scale, xpt, ypt, dot_x, dot_y, sa, ca, sda, cda, draw
      save count
      data sda, cda /0.087156, 0.996195/
      data sa, ca/ 0.0, 1.0/
      data xpt, ypt / 500, 500, 560, 560/
      data scale /0.03/
      data dot_x, dot_y / 4 * 75, 4 * 120/
      data draw / .true./
      data count / 0/
c
c    erase and redraw the scanner
c
      call gpr_$set_draw_value (icol(0), status)
      call gpr_$multiline (xpt, ypt, int2(2), status)
      call gpr_$multiline (dot_x, dot_y, int2(4), status)
      temp = ca
      ca = ca * cda - sa * sda
      sa = sa * cda + sda * temp
      xpt(2) = 60.0 * ca + 500
      ypt(2) = 60.0 * sa + 560
      call gpr_$set_draw_value (icol(2), status)
      call gpr_$multiline (xpt, ypt, int2(2), status)
c
c    note that distances are not squared
c
      if (orange(1) .gt. 2000.0 .or. orange(1) .eq. 0.0) then
        dot_x(1) = 500
        dot_y(1) = 560
        dot_x(2) = dot_x(1)
        dot_y(2) = dot_y(1)
        dot_x(3) = dot_x(1)
        dot_y(3) = dot_y(1)
        dot_x(4) = dot_x(1)
        dot_y(4) = dot_y(1)
      else
        dot_x(1) = 500 + pro_x(1) * scale - 1
        dot_y(1) = 560 - pro_y(1) * scale + 1
        dot_x(2) = dot_x(1) + 2
        dot_y(2) = dot_y(1) - 2
        dot_x(3) = dot_x(1)
        dot_y(3) = dot_y(1) - 2
        dot_x(4) = dot_x(1) + 2
        dot_y(4) = dot_y(1)
      end if
      call gpr_$multiline (dot_x, dot_y, int2(4), status)
      return
      end





      subroutine bzone_x_hairs (aligned)
c
c    BZONE_X_HAIRS draws the cross-hairs at the screen center.
c    Since these don't move and should superimpose upon other
c    objects they are redrawn instead of bit blt'ed.
c
% include '/sys/ins/base.ins.ftn'
% include '/sys/ins/gpr.ins.ftn'
c
      integer*2 x_pt1(16), y_pt1(16), x_pt2(12), y_pt2(12), num_pos(2)
      integer*4 status, icol(0:15)
      logical aligned, last, in_color, draw
      common /color$/ icol, in_color
      save last, draw
      save x_pt1, y_pt1, x_pt2, y_pt2, num_pos
      data last / .false./
      data x_pt1 / 500, 500, 500, 500, 475, 475, 475, 525, 525, 525,
     &             475, 475, 475, 525, 525, 525/
      data y_pt1 / 215, 235, 285, 305, 240, 235, 235, 235, 235, 240,
     &             280, 285, 285, 285, 285, 280/
      data x_pt2 / 500, 500, 500, 500, 475, 500, 500, 525, 475, 500,
     &             500, 525/
      data y_pt2 / 215, 235, 285, 305, 245, 235, 235, 245, 275, 285,
     &             285, 275/
      data num_pos / 16, 12/
      data draw / .true./
      if (draw) then
        if (aligned) then
          if (last) then
            call gpr_$set_draw_value (icol(1), status)
            call gpr_$multiline (x_pt2, y_pt2, num_pos(2), status)
            last = .true.
          else
            call gpr_$set_draw_value (icol(0), status)
            call gpr_$multiline (x_pt1, y_pt1, num_pos(1), status)
            call gpr_$set_draw_value (icol(1), status)
            call gpr_$multiline (x_pt2, y_pt2, num_pos(2), status)
            last = .true.
          end if
        else
          if (last) then
            call gpr_$set_draw_value (icol(0), status)
            call gpr_$multiline (x_pt2, y_pt2, num_pos(2), status)
            call gpr_$set_draw_value (icol(14), status)
            call gpr_$multiline (x_pt1, y_pt1, num_pos(1), status)
            last = .false.
          else
            call gpr_$set_draw_value (icol(14), status)
            call gpr_$multiline (x_pt1, y_pt1, num_pos(1), status)
          end if
        end if
      end if
      draw = .not. draw
      return
      end






      subroutine bzone_draw_horizon (azm)
c
c     BZONE_DRAW_HORIZON erases and redraws the mountains on the
c     skyline and the moon.
c
% include '/sys/ins/base.ins.ftn'
% include '/sys/ins/gpr.ins.ftn'
c
      integer*2 number, x_pt(48), y_pt(48), clust_x(12, 6)
      integer*2 clust_y(12, 6), num_clust(6), index_min, index_max
      integer*2 pt(2), radius, xpt(4), ypt(4)
      integer*4 status, vcount, icol(0:15)
      real azm, rdc, scale, deg_min, deg_max
      logical flag_1, flag_2, moon, in_color
      common /color$/ icol, in_color
      save x_pt, y_pt, number, rdc, scale, num_clust, clust_x, clust_y
      save pt, radius, xpt, ypt, moon
      data x_pt, y_pt / 48*500, 48*240/
      data number / 0/
      data moon / .false./
      data rdc, scale / 0.01745329251, 15.0/
      data num_clust / 8, 12, 10, 12, 10, 10/
      data clust_x / 150, 140, 150, 60, 85, -100, -50, -150, 4*0,
     &               150, 50, 75, 0, 0, 7, 0, -40, 0, -110, -25, -150,
     &               150, 115, 115, 105, 115, -10, 0, -125, -85, -150,
     &               2*0, 150, 130, 150, 10, 25, -35, -35, 0, -35, -140,
     &               -110, -150, 150, -10, 25, -70, -50, -125, -125,
     &               -100, -125, -150, 2*0, 150, 100, 140, 0, 0, -30,
     &               0, -120, -60, -150, 2*0/
      data clust_y / 160, 220, 160, 250, 225, 245, 250, 240, 4*0, 240,
     &               227, 245, 190, 190, 215, 190, 225, 190, 213, 240,
     &               200, 200, 189, 189, 225, 189, 244, 250, 165, 213,
     &               135, 2*0, 135, 200, 135, 210, 225, 165, 165, 230,
     &               165, 228, 210, 215, 215, 217, 230, 195, 210, 150,
     &               150, 210, 150, 173, 2*0, 173, 209, 225, 170, 170,
     &               225, 170, 230, 200, 160, 2*0/
c
c    erase the old skyline
c
      call gpr_$set_draw_value (icol(0), status)
      call gpr_$multiline (x_pt, y_pt, number, status)
      if (moon) then
        call gpr_$circle (pt, radius, status)
      end if
      call gpr_$set_draw_value (icol(14), status)
c
c    draw the new skyline
c
      deg = azm / rdc
      if (deg .ge. 360.0) deg = deg - 360.0
      deg_min = deg - 30.0
      deg_max = deg + 30.0
      flag_1 = .false.
      flag_2 = .false.
      if (deg_min .lt. 0.0) then
        deg_min = 360.0 + deg_min
        flag_1 = .true.
      end if
      if (deg_max .ge. 360.0) then
        deg_max = deg_max - 360.0
        flag_2 = .true.
      end if
      index_min = int(deg_min / 20.0) + 1
      index_max = int(deg_max / 20.0) + 1
c
c    do uninterrupted terrain
c
      if (.not. flag_1 .and. .not. flag_2) then
        number = 0
        do 10 i = index_min, index_max
          center = 10.0 + (i - 1) * 20.0
          xoff = (deg - center) * scale + 500.0
          i1 = mod(i - 1, 6) + 1
          do 20 j = 1, num_clust(i1)
            number = number + 1
            x_pt(number) = clust_x(j, i1) + xoff
            y_pt(number) = clust_y(j, i1)
 20       continue
 10     continue
      else if (flag_1) then
c
c    do terrain for index_min < 0
c
        number = 0
        do 30 i = index_min, 18
          center = 10.0 + (i - 1) * 20.0
          xoff = (360.0 + deg - center) * scale + 500.0
          i1 = mod(i - 1, 6) + 1
          do 40 j = 1, num_clust(i1)
            number = number + 1
            x_pt(number) = clust_x(j, i1) + xoff
            y_pt(number) = clust_y(j, i1)
 40       continue
 30     continue
        do 50 i = 1, index_max
          center = 10.0 + (i - 1) * 20.0
          xoff = (deg - center) * scale + 500.0
          i1 = mod(i - 1, 6) + 1
          do 60 j = 1, num_clust(i1)
            number = number + 1
            x_pt(number) = clust_x(j, i1) + xoff
            y_pt(number) = clust_y(j, i1)
 60       continue
 50     continue
      else if (flag_2) then
c
c    do terrain for index_max > 18
c
        number = 0
        do 70 i = index_min, 18
          center = 10.0 + (i - 1) * 20.0
          xoff = (deg - center) * scale + 500.0
          i1 = mod(i - 1, 6) + 1
          do 80 j = 1, num_clust(i1)
            number = number + 1
            x_pt(number) = clust_x(j, i1) + xoff
            y_pt(number) = clust_y(j, i1)
 80       continue
 70     continue
        do 90 i = 1, index_max
          center = 10.0 + (i - 1) * 20.0
          xoff = (deg - 360.0 - center) * scale + 500.0
          i1 = mod(i - 1, 6) + 1
          do 100 j = 1, num_clust(i1)
            number = number + 1
            x_pt(number) = clust_x(j, i1) + xoff
            y_pt(number) = clust_y(j, i1)
100       continue
 90     continue
      end if
      call gpr_$multiline (x_pt, y_pt, number, status)
c
c    draw the moon if on the screen
c
      if ((deg .le. 360.0) .and. (deg .ge. 270.0)) then
        moon = .true.
        pt(1) = 500 + (deg - 315.0) * scale
        pt(2) = 105
        radius = 20
        call gpr_$circle (pt, radius, status)
      else
        moon = .false.
      end if
      return
      end





      subroutine bzone_draw_cracks
c
c     BZONE_DRAW_CRACKS draws cracks on the screen.
c
% include '/sys/ins/base.ins.ftn'
% include '/sys/ins/gpr.ins.ftn'
c
      integer*2 xpt(38), ypt(38)
      integer*4 status, icol(0:15)
      logical in_color
      common /color$/ icol, in_color
      save xpt, ypt
      data xpt / 190, 270, 270, 230, 240, 330, 310, 390, 370, 380, 200,
     &           290, 270, 300, 630, 560, 580, 626, 570, 510, 520, 490,
     &           500, 440, 810, 720, 720, 695, 700, 590, 630, 620, 710,
     &           780, 770, 730, 740, 760/
      data ypt / 75, 175, 175, 265, 243, 290, 280, 235, 246, 195, 425,
     &           365, 378, 325, 425, 335, 361, 295, 348, 355, 354, 275,
     &           301, 290, 75, 125, 125, 205, 200, 210, 206, 256, 157,
     &           245, 232, 305, 287, 315/
c
c    draw the cracks
c
      call gpr_$set_draw_value (icol(1), status)
      call gpr_$multiline (xpt, ypt, int2(38), status)
      return
      end





      subroutine bzone_clear_screen (bitmap_desc)
c
c     BZONE_CLEAR_SCREEN clears the screen completely.
c
% include '/sys/ins/base.ins.ftn'
% include '/sys/ins/gpr.ins.ftn'
c
      integer*2 window(2, 2), dest_origin(2)
      integer*4 status, bitmap_desc, icol(0:15)
      logical in_color
      common /color$/ icol, in_color
      save window, dest_origin
      data dest_origin / 70, 75/
      data window / 70, 75, 860, 350/
c
c    erase the whole screen
c
      call gpr_$set_raster_op (int2(0), int2(6), status)
      call gpr_$bit_blt (bitmap_desc, window, int2(0), dest_origin,
     &                   int2(0), status)
      call gpr_$set_raster_op (int2(0), int2(3), status)
      if (in_color) then
        call gpr_$set_raster_op (int2(1), int2(6), status)
        call gpr_$bit_blt (bitmap_desc, window, int2(1), dest_origin,
     &                     int2(1), status)
        call gpr_$set_raster_op (int2(1), int2(3), status)
        call gpr_$set_raster_op (int2(2), int2(6), status)
        call gpr_$bit_blt (bitmap_desc, window, int2(2), dest_origin,
     &                     int2(2), status)
        call gpr_$set_raster_op (int2(2), int2(3), status)
        call gpr_$set_raster_op (int2(3), int2(6), status)
        call gpr_$bit_blt (bitmap_desc, window, int2(3), dest_origin,
     &                     int2(3), status)
        call gpr_$set_raster_op (int2(3), int2(3), status)
      end if
      return
      end




      subroutine bzone_draw_cube (xcrd, ycrd, ca, sa, obx, oby,
     &                            orange, seen)
c
c     BZONE_DRAW_CUBE draws the cube located at obx(index) oby(index)
c     with an azimuth of oazm(index) as seen from the players tank.
c     Keeps track of the last draw so as to erase the object.  Object
c     is only redrawn if seen(index) = .true..  Routine knows if the
c     cube was seen last pass.
c
% include '/sys/ins/base.ins.ftn'
% include '/sys/ins/gpr.ins.ftn'
c
      integer*2 poly_x(10, 12), poly_y(10, 12), xp(10), yp(10), zp(10)
      integer*2 mult_x(6, 12), mult_y(6, 12), xm(6), ym(6), zm(6)
      integer*4 index, status, color, icol(0:15)
      real xcrd, ycrd, ca, sa, obx(12), oby(12), orange(12)
      real mrx, mry, prx, pry, xpoff, ypoff, xmoff, ymoff
      logical seen(12), last(12), in_color
      common /color$/ icol, in_color
      save last, poly_x, poly_y, mult_x, mult_y, xm, ym, zm, xp, yp, zp
      data last / 12 * .false./
      data xp / -40, -40, 40, 40, -40, -40, 40, 40, -40, -40/
      data yp / 40, 40, 40, 40, 40, -40, -40, -40, -40, -40/
      data zp / -40, 40, 40, -40, -40, -40, -40, 40, 40, -40/
      data xm / -40, -40, 40, 40, 40, 40/
      data ym / -40, 40, -40, 40, -40, 40/
      data zm / 40, 40, 40, 40, -40, -40/
c
c    erase the cube if drawn last pass
c
      do 10 index = 5, 8
        if (last(index)) then
          call gpr_$set_draw_value (icol(0), status)
          call gpr_$move (poly_x(1, index), poly_y(1, index), status)
          call gpr_$polyline (poly_x(1, index), poly_y(1, index),
     &                        int2(10), status)
          call gpr_$multiline (mult_x(1, index), mult_y(1, index),
     &                       int2(6), status)
        end if
c
c    draw the new cube if seen
c
        if (seen(index)) then
c
c    form offsets from player tank and rotate them about it
c
          color = orange(index) / 300 + 8
          if (color .gt. 13) color = 13
          call gpr_$set_draw_value (icol(color), status)
          do 20 i = 1, 10
            xpoff = xp(i) + obx(index) - xcrd
            ypoff = yp(i) + oby(index) - ycrd
            prx =  xpoff * ca + ypoff * sa
            pry = -xpoff * sa + ypoff * ca
            if (pry .lt. 5.0) pry = 5.0
            poly_x(i, index) = 500 + prx / pry * 450
            poly_y(i, index) = 260 - zp(i) / pry * 450
 20       continue
          do 30 i = 1, 6
            xmoff = xm(i) + obx(index) - xcrd
            ymoff = ym(i) + oby(index) - ycrd
            mrx =  xmoff * ca + ymoff * sa
            mry = -xmoff * sa + ymoff * ca
            if (mry .lt. 5.0) mry = 5.0
            mult_x(i, index) = 500 + mrx / mry * 450
            mult_y(i, index) = 260 - zm(i) / mry * 450
 30       continue
          call gpr_$move (poly_x(1, index), poly_y(1, index), status)
          call gpr_$polyline (poly_x(1, index), poly_y(1, index),
     &                        int2(10), status)
          call gpr_$multiline (mult_x(1, index), mult_y(1, index),
     &                         int2(6), status)
          last(index) = .true.
        else
          last(index) = .false.
        end if
 10   continue
      return
      end




      subroutine bzone_draw_pyramid (xcrd, ycrd, ca, sa, obx,
     &                               oby, orange, seen)
c
c     BZONE_DRAW_PYRAMID draws the pyramid located at obx, oby
c     as seen from the players tank. Keeps track of the last draw
c     so as to erase the object.  Object is only redrawn if
c     seen(index) = .true..  Routine knows if the cube was seen
c     last pass.
c
% include '/sys/ins/base.ins.ftn'
% include '/sys/ins/gpr.ins.ftn'
c
      integer*2 poly_x(10, 12), poly_y(10, 12), xp(10), yp(10), zp(10)
      integer*4 index, status, color, icol(0:15)
      real xcrd, ycrd, ca, sa, obx(12), oby(12), orange(12)
      real prx, pry, xpoff, ypoff
      logical seen(12), last(12), in_color
      common /color$/ icol, in_color
      save last, poly_x, poly_y, xp, yp, zp
      data last / 12 * .false./
      data xp / -40, 40, 40, 40, 0, -40, -40, 0, 40, -40/
      data yp / 40, 40, -40, 40, 0, 40, -40, 0, -40, -40/
      data zp / -40, -40, -40, -40, 40, -40, -40, 40, -40, -40/
c
c    erase the pyramid if drawn last pass
c
      do 10 index = 9, 12
        if (last(index)) then
          call gpr_$set_draw_value (0, status)
          call gpr_$move (poly_x(1, index), poly_y(1, index), status)
          call gpr_$polyline (poly_x(1, index), poly_y(1, index),
     &                        int2(10), status)
          call gpr_$set_draw_value (1, status)
        end if
c
c    draw the new pyramid if seen
c
        if (seen(index)) then
c
c    form offsets from player tank and rotate them about it
c
          color = orange(index) / 300 + 8
          if (color .gt. 13) color = 13
          call gpr_$set_draw_value (icol(color), status)
          do 20 i = 1, 10
            xpoff = xp(i) + obx(index) - xcrd
            ypoff = yp(i) + oby(index) - ycrd
            prx =  xpoff * ca + ypoff * sa
            pry = -xpoff * sa + ypoff * ca
            if (pry .lt. 7.5) pry = 7.5
            poly_x(i, index) = 500 + prx / pry * 450
            poly_y(i, index) = 260 - zp(i) / pry * 450
 20       continue
          call gpr_$move (poly_x(1, index), poly_y(1, index), status)
          call gpr_$polyline (poly_x(1, index), poly_y(1, index),
     &                        int2(10), status)
          last(index) = .true.
        else
          last(index) = .false.
        end if
 10   continue
      return
      end




      subroutine bzone_draw_salvo (xcrd, ycrd, ca, sa, obx, oby, oazm,
     &                             orange, seen)
c
c     BZONE_DRAW_SALVO draws the player and enemy tank fire
c
% include '/sys/ins/base.ins.ftn'
% include '/sys/ins/gpr.ins.ftn'
c
      integer*2 poly_x(10, 4), poly_y(10, 4), xp(10), yp(10), zp(10)
      integer*4 index, status, color, icol(0:15)
      real xcrd, ycrd, ca, sa, obx(12), oby(12), orange(12), oazm(12)
      real prx, pry, csa, ssa, rx, ry
      logical seen(12), last(12), in_color
      common /color$/ icol, in_color
      save last, poly_x, poly_y, xp, yp, zp
      data last / 12 * .false./
      data xp / 0, 8, 0, 8, 0, 0, -8, 0, 0, -8/
      data yp / -10, -10, -10, -10, 10, -10, -10, 10, -10, -10/
      data zp / -8, 0, 8, 0, 0, 8, 0, 0, -8, 0/
c
c    erase the salvo if drawn last pass
c
      do 10 index = 3, 4
        if (last(index)) then
          call gpr_$set_draw_value (icol(0), status)
          call gpr_$move (poly_x(1, index), poly_y(1, index), status)
          call gpr_$polyline (poly_x(1, index), poly_y(1, index),
     &                        int2(10), status)
        end if
c
c    draw the new salvo if seen
c
        if (seen(index)) then
          color = orange(index) / 300 + 8
          if (color .gt. 13) color = 13
          call gpr_$set_draw_value (icol(color), status)
          csa = cos(oazm(index))
          ssa = sin(oazm(index))
          dx = xcrd - obx(index)
          dy = ycrd - oby(index)
          xn =  dx * csa + dy * ssa
          yn = -dx * ssa + dy * csa
          cn = ca * csa + sa * ssa
          sn = sa * csa - ssa * ca
          do 20 i = 1, 10
            rx =  xp(i) - xn
            ry =  yp(i) - yn
            prx =  rx * cn + ry * sn
            pry = -rx * sn + ry * cn
            if (pry .lt. 7.5) pry = 7.5
            poly_x(i, index) = 500 + prx / pry * 450
            poly_y(i, index) = 260 + zp(i) / pry * 450
 20       continue
          call gpr_$move (poly_x(1, index), poly_y(1, index), status)
          call gpr_$polyline (poly_x(1, index), poly_y(1, index),
     &                        int2(10), status)
          last(index) = .true.
        else
          last(index) = .false.
        end if
 10   continue
      return
      end




      subroutine bzone_draw_lander (xcrd, ycrd, ca, sa, obx, oby,
     &                              orange, seen, cla, sla)
c
c     BZONE_DRAW_LANDER erases the old lander and draws a new as seen
c     from the player tank.
c
% include '/sys/ins/base.ins.ftn'
% include '/sys/ins/gpr.ins.ftn'
c
      integer*2 poly_x(21), poly_y(21), xp(21), yp(21), zp(21)
      integer*4 color, icol(0:15)
      real xcrd, ycrd, ca, sa, obx(12), oby(12), orange(12)
      real cla, sla, cda, sda, temp, rx, ry, prx, pry
      logical seen(12), last, in_color
      common /color$/ icol, in_color
      save last, poly_x, poly_y, xp, yp, zp, cda, sda
      data last /.false./
      data xp /0, 80, 40, -40, -80, 0, 0, 0, 0, 0, 0, -57, -28, 28, 57,
     &         0, 57, 28, -28, -57, 0/
      data yp /0, 0, 0, 0, 0, 0, 80, 40, -40, -80, 0, -57, -28, 28, 57,
     &         0, -57, -28, 28, 57, 0/
      data zp /20, -20, -40, -40, -20, 20, -20, -40, -40, -20, 20, -20,
     &         -40, -40, -20, 20, -20, -40, -40, -20, 20/
      data sda, cda /0.087156, 0.996195/
c
c    erase the old lander if one was drawn
c
      if (last) then
        call gpr_$set_draw_value (icol(0), status)
        call gpr_$move (poly_x(1), poly_y(1), status)
        call gpr_$polyline (poly_x, poly_y, int2(21), status)
      end if
c
c    draw the new lander if seen
c
      if (seen(2)) then
        temp = cla
        cla = cla * cda - sla * sda
        sla = sla * cda + sda * temp
        dx = xcrd - obx(2)
        dy = ycrd - oby(2)
        xn =  dx * cla + dy * sla
        yn = -dx * sla + dy * cla
        cn = ca * cla + sa * sla
        sn = sa * cla - ca * sla
        do 10 i = 1, 21
          rx = xp(i) - xn
          ry = yp(i) - yn
          prx =  rx * cn + ry * sn
          pry = -rx * sn + ry * cn
          if (pry .lt. 10.0) pry = 10.0
          poly_x(i) = 500 + prx / pry * 450
          poly_y(i) = 260 - zp(i) / pry * 450
 10     continue
        color = orange(2) / 300 + 2
        if (color .gt. 7) color = 7
        call gpr_$set_draw_value (icol(color), status)
        call gpr_$move (poly_x(1), poly_y(1), status)
        call gpr_$polyline (poly_x, poly_y, int2(21), status)
        last = .true.
      else
        last = .false.
      end if
      return
      end




      subroutine bzone_draw_missile (xcrd, ycrd, ca, sa, obx, oby,
     &                               oazm, orange, seen, height)
c
c     BZONE_DRAW_MISSILE  erases and redraws the missile
c
% include '/sys/ins/base.ins.ftn'
% include '/sys/ins/gpr.ins.ftn'
c
      integer*2 capx(11), capy(11), capz(11), conx(9), cony(9), conz(9)
      integer*2 finx(4), finy(4), finz(4), cap_x(11), cap_y(11)
      integer*2 con_x(9), con_y(9), fin1_x(4), fin1_y(4), fin2_x(4)
      integer*2 fin2_y(4)
      integer*4 status, color, icol(0:15)
      real obx(12), oby(12), oazm(12), orange(12), xcrd, ycrd
      real sa, ca, height, rx, ry, prx, pry, prz
      logical seen(12), last, in_color
      common /color$/ icol, in_color
      save capx, capy, capz, conx, cony, conz, finx, finy, finz
      save cap_x, cap_y, con_x, con_y, fin1_x, fin1_y, fin2_x
      save fin2_y, last
      data capx / 15, 25, 0, 15, -15, 0, 15, -15, 0, -25, -15/
      data capy / -30, -30, -45, -30, -30, -45, -30, -30, -45,
     &            -30, -30/
      data capz / -25, 0, 0, -25, -25, 0, 25, 25, 0, 0, 25/
      data conx / 15, 0, 25, 15, 0, -15, -25, 0, -15/
      data cony / -30, 50, -30, -30, 50, -30, -30, 50, -30/
      data conz / -25, 0, 0, 25, 0, -25, 0, 0, 25/
      data finx / 15, 23, 23, 13/
      data finy / -30, -38, 0, -17/
      data finz / -25, -40, -40, -21/
      data last / .false./
c
c    erase the image if drawn last turn
c
      if (last) then
        call gpr_$set_draw_value (0, status)
        call gpr_$move (cap_x(1), cap_y(1), status)
        call gpr_$polyline (cap_x, cap_y, int2(11), status)
        call gpr_$move (con_x(1), con_y(1), status)
        call gpr_$polyline (con_x, con_y, int2(9), status)
        call gpr_$move (fin1_x(4), fin1_y(4), status)
        call gpr_$polyline (fin1_x, fin1_y, int2(4), status)
        call gpr_$move (fin2_x(4), fin2_y(4), status)
        call gpr_$polyline (fin2_x, fin2_y, int2(4), status)
      end if
c
c    draw the new image if seen
c
      if (seen(1)) then
        color = orange(1) / 300 + 2
        if (color .gt. 7) color = 7
        call gpr_$set_draw_value (icol(color), status)
        csa = cos(oazm(1))
        ssa = sin(oazm(1))
        dx = xcrd - obx(1)
        dy = ycrd - oby(1)
        xn =  dx * csa + dy * ssa
        yn = -dx * ssa + dy * csa
        cn = ca * csa + sa * ssa
        sn = sa * csa - ssa * ca
        do 10 i = 1, 11
          rx =  capx(i) - xn
          ry =  capy(i) - yn
          prx =  rx * cn + ry * sn
          pry = -rx * sn + ry * cn
          prz =  capz(i) + height
          if (pry .lt. 10.0) pry = 10.0
          cap_x(i) = 500 + prx / pry * 450
          cap_y(i) = 260 - prz / pry * 450
 10     continue
        call gpr_$move (cap_x(1), cap_y(1), status)
        call gpr_$polyline (cap_x, cap_y, int2(11), status)
        do 20 i = 1, 9
          rx =  conx(i) - xn
          ry =  cony(i) - yn
          prx =  rx * cn + ry * sn
          pry = -rx * sn + ry * cn
          prz =  conz(i) + height
          if (pry .lt. 10.0) pry = 10.0
          con_x(i) = 500 + prx / pry * 450
          con_y(i) = 260 - prz / pry * 450
 20     continue
        call gpr_$move (con_x(1), con_y(1), status)
        call gpr_$polyline (con_x, con_y, int2(9), status)
        do 30 i = 1, 4
          rx1 =  finx(i) - xn
          ry1 =  finy(i) - yn
          rx2 = -finx(i) - xn
          ry2 =  finy(i) - yn
          prx1 =  rx1 * cn + ry1 * sn
          pry1 = -rx1 * sn + ry1 * cn
          prx2 =  rx2 * cn + ry2 * sn
          pry2 = -rx2 * sn + ry2 * cn
          prz =  finz(i) + height
          if (pry1 .lt. 10.0) pry1 = 10.0
          if (pry2 .lt. 10.0) pry2 = 10.0
          fin1_x(i) = 500 + prx1 / pry1 * 450
          fin1_y(i) = 260 - prz / pry1 * 450
          fin2_x(i) = 500 + prx2 / pry2 * 450
          fin2_y(i) = 260 - prz / pry2 * 450
 30     continue
        call gpr_$move (fin1_x(4), fin1_y(4), status)
        call gpr_$polyline (fin1_x, fin1_y, int2(4), status)
        call gpr_$move (fin2_x(4), fin2_y(4), status)
        call gpr_$polyline (fin2_x, fin2_y, int2(4), status)
        last = .true.
      else
        last = .false.
      end if
      return
      end






      subroutine bzone_draw_copter (xcrd, ycrd, ca, sa, obx, oby, oazm,
     &                              orange, seen, height)
c
c     BZONE_DRAW_COPTER draws an enemy helicopter
c
% include '/sys/ins/base.ins.ftn'
% include '/sys/ins/gpr.ins.ftn'
c
      integer*2 proxp(5), proyp(5), prozp(5), prop_xp(5), prop_yp(5)
      integer*2 extxm(10), extym(10), extzm(10), extr_xm(10)
      integer*2 extr_ym(10), talxp(11), talyp(11), talzp(11)
      integer*2 tail_xp(11), tail_yp(11), bodxp(19), bodyp(19)
      integer*2 bodzp(19), body_xp(19), body_yp(19)
      integer*2 bodxm(10), bodym(10), bodzm(10), body_xm(10)
      integer*2 body_ym(10)
      integer*4 status, icol(0:15), color
      real xcrd, ycrd, obx(12), oby(12), ca, sa, orange(12), oazm(12)
      real zoff, rx, ry, cn, sn, xn, yn, prx, pry, prz, height
      logical last, seen(12), in_color
      common /color$/ icol, in_color
      save last
      save cta, sta, cda, sda, proxp, proyp, prozp, prop_xp, prop_yp
      save extxm, extym, extzm, extr_xm, extr_ym
      save talxp, talyp, talzp, tail_xp, tail_yp
      save bodxp, bodyp, bodzp, body_xp, body_yp
      save bodxm, bodym, bodzm, body_xm, body_ym
      data cta, sta, cda, sda /1.0, 0.0, 0.819152, 0.573576/
      data proxp / 6, -6, 6, -6, 6/
      data proyp / 100, -100, -100, 100, 100/
      data prozp / 5 * 30/
      data extxm / 0, 0, -14, -14, -14, -14, 14, 14, 14, 14/
      data extym / 0, 0, -30, 40, 40, 44, -30, 40, 40, 44/
      data extzm / 16, 36, -40, -40, -40, -36, -40, -40, -40, -36/
      data talxp / 0, 7, 10, 0, 0, 0, 0, 0, -7, -10, 0/
      data talyp / -114, -30, -34, -116, -134, -144, -130, -114, -30,
     &             -34, -116/
      data talzp / -10, -34, -14, 0, 28, 28, -10, -10, -34, -14, 0/
      data bodxp / -14, 14, 0, -14, -7, 7, 14, 26, 0, -26, -10, 10,
     &             26, 4, 0, -4, -4, 4, 4/
      data bodyp / 34, 34, 60, 34, -30, -30, 34, 34, 60, 34, -34,
     &             -34, 34, 20, 22, 20, -22, -22, 20/
      data bodzp / -34, -34, -14, -34, -34, -34, -34, -14, -14, -14,
     &             -14, -14, -14, 16, 16, 16, 16, 16, 16/
      data bodxm / -10, -4, 10, 4, 0, 0, -14, -26, -26, -4/
      data bodym / -34, -22, -34, -22, 22, 60, 34, 34, 34, 20/
      data bodzm / -14, 16, -14, 16, 16, -14, -34, -14, -14, 16/
      data last / .false./
c
c    erase figure if drawn last time
c
      if (last) then
        call gpr_$set_draw_value (icol(0), status)
        call gpr_$move (prop_xp(5), prop_yp(5), status)
        call gpr_$polyline (prop_xp, prop_yp, int2(5), status)
        call gpr_$multiline (extr_xm, extr_ym, int2(10), status)
        call gpr_$move (tail_xp(1), tail_yp(1), status)
        call gpr_$polyline (tail_xp, tail_yp, int2(11), status)
        call gpr_$move (body_xp(1), body_yp(1), status)
        call gpr_$polyline (body_xp, body_yp, int2(19), status)
        call gpr_$multiline (body_xm, body_ym, int2(10), status)
      end if
c
c    build the vectors for the next draw
c
      if (seen(1)) then
        color = orange(1) / 300 + 2
        if (color .gt. 7) color = 7
        call gpr_$set_draw_value (icol(color), status)
        temp = cta
        cta = cta * cda - sta * sda
        sta = sta * cda + sda * temp
        csa = cos(oazm(1))
        ssa = sin(oazm(1))
        dx = xcrd - obx(1)
        dy = ycrd - oby(1)
        xn =  dx * csa + dy * ssa
        yn = -dx * ssa + dy * csa
        x1 =  dx * cta + dy * sta
        y1 = -dx * sta + dy * cta
        cn = ca * csa + sa * ssa
        sn = sa * csa - ssa * ca
        c1 = ca * cta + sa * sta
        s1 = sa * cta - sta * ca
c
c    prop first
c
        do 10 i = 1, 5
          rx = proxp(i) - x1
          ry = proyp(i) - y1
          prx =  rx * c1 + ry * s1
          pry = -rx * s1 + ry * c1
          prz = prozp(i) + height
          if (pry .lt. 10.0) pry = 10.0
          prop_xp(i) = 500 + prx / pry * 450
          prop_yp(i) = 260 - prz / pry * 450
 10     continue
c
c    extras
c
        do 20 i = 1, 10
          rx = extxm(i) - xn
          ry = extym(i) - yn
          prx =  rx * cn + ry * sn
          pry = -rx * sn + ry * cn
          prz = extzm(i) + height
          if (pry .lt. 10.0) pry = 10.0
          extr_xm(i) = 500 + prx / pry * 450
          extr_ym(i) = 260 - prz / pry * 450
 20     continue
c
c    tail
c
        do 30 i = 1, 11
          rx = talxp(i) - xn
          ry = talyp(i) - yn
          prx =  rx * cn + ry * sn
          pry = -rx * sn + ry * cn
          prz = talzp(i) + height
          if (pry .lt. 10.0) pry = 10.0
          tail_xp(i) = 500 + prx / pry * 450
          tail_yp(i) = 260 - prz / pry * 450
 30     continue
c
c    polyline part of the cabin
c
        do 40 i = 1, 19
          rx = bodxp(i) - xn
          ry = bodyp(i) - yn
          prx =  rx * cn + ry * sn
          pry = -rx * sn + ry * cn
          prz = bodzp(i) + height
          if (pry .lt. 10.0) pry = 10.0
          body_xp(i) = 500 + prx / pry * 450
          body_yp(i) = 260 - prz / pry * 450
 40     continue
c
c    multiline part of the cabin
c
        do 50 i = 1, 10
          rx = bodxm(i) - xn
          ry = bodym(i) - yn
          prx =  rx * cn + ry * sn
          pry = -rx * sn + ry * cn
          prz = bodzm(i) + height
          if (pry .lt. 10.0) pry = 10.0
          body_xm(i) = 500 + prx / pry * 450
          body_ym(i) = 260 - prz / pry * 450
 50     continue
        call gpr_$move (prop_xp(5), prop_yp(5), status)
        call gpr_$polyline (prop_xp, prop_yp, int2(5), status)
        call gpr_$multiline (extr_xm, extr_ym, int2(10), status)
        call gpr_$move (tail_xp(1), tail_yp(1), status)
        call gpr_$polyline (tail_xp, tail_yp, int2(11), status)
        call gpr_$move (body_xp(1), body_yp(1), status)
        call gpr_$polyline (body_xp, body_yp, int2(19), status)
        call gpr_$multiline (body_xm, body_ym, int2(10), status)
        last = .true.
      else
        last = .false.
      end if
      return
      end








      subroutine bzone_draw_super (xcrd, ycrd, ca, sa, obx, oby, oazm,
     &                             orange, seen)
c
c     BZONE_DRAW_SUPER draws the enemy tank (super).
c
% include '/sys/ins/base.ins.ftn'
% include '/sys/ins/gpr.ins.ftn'
c
      integer*2 base_xp(8), base_yp(8), turt_xp(9), turt_yp(9)
      integer*2 musl_xp(4), musl_yp(4), musl_xm(8), musl_ym(8)
      integer*2 radr_xm(2), radr_ym(2), radxm(2), radym(2)
      integer*2 radzm(2), musxm(8), musym(8), muszm(8), musxp(4)
      integer*2 musyp(4), muszp(4), turxp(9), turyp(9), turzp(9)
      integer*2 basxm(4), basym(4), baszm(4), basxp(8), basyp(8)
      integer*2 baszp(8), base_xm(4), base_ym(4), turxm(4), turym(4)
      integer*2 turzm(4), turt_xm(4), turt_ym(4)
      integer*4 status, color, icol(0:15)
      real xcrd, ycrd, ca, sa, obx(12), oby(12), oazm(12), orange(12)
      real rx, ry, prx, pry
      logical seen(12), last, in_color
      common /color$/ icol, in_color
      save last, base_xp, base_yp, turt_xp, turt_yp
      save musl_xp, musl_yp, musl_xm, musl_ym
      save radr_xm, radr_ym, radxm, radym, radzm
      save musxm, musym, muszm, musxp, musyp, muszp, turxp, turyp
      save turzp, basxm, basym, baszm, basxp, basyp, baszp
      save base_xm, base_ym, turxm, turym, turzm, turt_xm, turt_ym
      data last / .false./
      data radxm / 22, 22/
      data radym / -52, -52/
      data radzm / -8, 28/
      data musxm / 3, 3, -3, -3, 3, 3, -3, -3/
      data musym / -17, 55, -17, 55, -8, 55, -8, 55/
      data muszm / 0, 0, 0, 0, -6, -6, -6, -6/
      data musxp / 3, -3, -3, 3/
      data musyp / 4 * 55/
      data muszp / 0, 0, -6, -6/
      data turxp / 0, 13, 11, 11, 0, -13, -11, -11, 0/
      data turyp / 35, -60, -60, -25, 35, -60, -60, -25, 35/
      data turzp / -33, -6, 4, 4, -33, -6, 4, 4, -33/
      data turxm / 11, -11, 11, -11/
      data turym / -60, -60, -25, -25/
      data turzm / 4, 4, 4, 4/
      data basxp / 15, 30, 30, 15, -15, -30, -30, -15/
      data basyp / 60, -60, -60, 60, 60, -60, -60, 60/
      data baszp / -40, -40, -6, -40, -40, -40, -6, -40/
      data basxm / -30, 30, -30, 30/
      data basym / -60, -60, -60, -60/
      data baszm / -6, -6, -40, -40/
c
c    erase the old image if drawn last turn through
c
      if (last) then
        call gpr_$set_draw_value (icol(0), status)
        call gpr_$move (base_xp(1), base_yp(1), status)
        call gpr_$polyline (base_xp, base_yp, int2(8), status)
        call gpr_$multiline (base_xm, base_ym, int2(4), status)
        call gpr_$move (turt_xp(1), turt_yp(1), status)
        call gpr_$polyline (turt_xp, turt_yp, int2(9), status)
        call gpr_$multiline (turt_xm, turt_ym, int2(4), status)
        call gpr_$multiline (musl_xm, musl_ym, int2(8), status)
        call gpr_$move (musl_xp(4), musl_yp(4), status)
        call gpr_$polyline (musl_xp, musl_yp, int2(4), status)
        call gpr_$multiline (radr_xm, radr_ym, int2(2), status)
      end if
c
c    build the vectors for the new tank, first the radar dish
c
      if (seen(1)) then
        color = orange(1) / 300 + 2
        if (color .gt. 7) color = 7
        call gpr_$set_draw_value (icol(color), status)
        csa = cos(oazm(1))
        ssa = sin(oazm(1))
        dx = xcrd - obx(1)
        dy = ycrd - oby(1)
        xn =  dx * csa + dy * ssa
        yn = -dx * ssa + dy * csa
        cn = ca * csa + sa * ssa
        sn = sa * csa - ssa * ca
        do 10 i = 1, 2
          rx =  radxm(i) - xn
          ry =  radym(i) - yn
          prx =  rx * cn + ry * sn
          pry = -rx * sn + ry * cn
          if (pry .lt. 10.0) pry = 10.0
          radr_xm(i) = 500 + prx / pry * 450
          radr_ym(i) = 260 - radzm(i) / pry * 450
 10     continue
c
c    do the gun muzzle
c
        do 20 i = 1, 8
          rx =  musxm(i) - xn
          ry =  musym(i) - yn
          prx =  rx * cn + ry * sn
          pry = -rx * sn + ry * cn
          if (pry .lt. 10.0) pry = 10.0
          musl_xm(i) = 500 + prx / pry * 450
          musl_ym(i) = 260 - muszm(i) / pry * 450
 20     continue
        do 30 i = 1, 4
          rx =  musxp(i) - xn
          ry =  musyp(i) - yn
          prx =  rx * cn + ry * sn
          pry = -rx * sn + ry * cn
          if (pry .lt. 10.0) pry = 10.0
          musl_xp(i) = 500 + prx / pry * 450
          musl_yp(i) = 260 - muszp(i) / pry * 450
 30     continue
c
c    do the turret
c
        do 40 i = 1, 4
          rx =  turxm(i) - xn
          ry =  turym(i) - yn
          prx =  rx * cn + ry * sn
          pry = -rx * sn + ry * cn
          if (pry .lt. 10.0) pry = 10.0
          turt_xm(i) = 500 + prx / pry * 450
          turt_ym(i) = 260 - turzm(i) / pry * 450
 40     continue
        do 50 i = 1, 9
          rx =  turxp(i) - xn
          ry =  turyp(i) - yn
          prx =  rx * cn + ry * sn
          pry = -rx * sn + ry * cn
          if (pry .lt. 10.0) pry = 10.0
          turt_xp(i) = 500 + prx / pry * 450
          turt_yp(i) = 260 - turzp(i) / pry * 450
 50     continue
c
c    do the base
c
        do 60 i = 1, 4
          rx =  basxm(i) - xn
          ry =  basym(i) - yn
          prx =  rx * cn + ry * sn
          pry = -rx * sn + ry * cn
          if (pry .lt. 10.0) pry = 10.0
          base_xm(i) = 500 + prx / pry * 450
          base_ym(i) = 260 - baszm(i) / pry * 450
 60     continue
        do 70 i = 1, 8
          rx =  basxp(i) - xn
          ry =  basyp(i) - yn
          prx =  rx * cn + ry * sn
          pry = -rx * sn + ry * cn
          if (pry .lt. 10.0) pry = 10.0
          base_xp(i) = 500 + prx / pry * 450
          base_yp(i) = 260 - baszp(i) / pry * 450
 70     continue
        call gpr_$move (base_xp(1), base_yp(1), status)
        call gpr_$polyline (base_xp, base_yp, int2(8), status)
        call gpr_$multiline (base_xm, base_ym, int2(4), status)
        call gpr_$move (turt_xp(1), turt_yp(1), status)
        call gpr_$polyline (turt_xp, turt_yp, int2(9), status)
        call gpr_$multiline (turt_xm, turt_ym, int2(4), status)
        call gpr_$multiline (musl_xm, musl_ym, int2(8), status)
        call gpr_$move (musl_xp(4), musl_yp(4), status)
        call gpr_$polyline (musl_xp, musl_yp, int2(4), status)
        call gpr_$multiline (radr_xm, radr_ym, int2(2), status)
        last = .true.
      else
        last = .false.
      end if
      return
      end




      subroutine bzone_draw_tank (xcrd, ycrd, ca, sa, obx, oby, oazm,
     &                            orange, seen)
c
c     BZONE_DRAW_TANK draws the enemy tank (regular).
c
% include '/sys/ins/base.ins.ftn'
% include '/sys/ins/gpr.ins.ftn'
c
      integer*2 base_xp(10), base_yp(10), turt_xp(12), turt_yp(12)
      integer*2 musl_xp(4), musl_yp(4), musl_xm(8), musl_ym(8)
      integer*2 radr_xp(8), radr_yp(8), radr_xm(2), radr_ym(2)
      integer*2 radxp(8), radyp(8), radzp(8), radxm(2), radym(2)
      integer*2 radzm(2), musxm(8), musym(8), muszm(8), musxp(4)
      integer*2 musyp(4), muszp(4), turxp(12), turyp(12), turzp(12)
      integer*2 basxm(6), basym(6), baszm(6), basxp(10), basyp(10)
      integer*2 baszp(10), base_xm(6), base_ym(6)
      integer*4 status, color, icol(0:15)
      real xcrd, ycrd, ca, sa, obx(12), oby(12), oazm(12), orange(12)
      real cta, sta, cda, sda, rx, ry, rz, prx, pry, prz
      logical seen(12), last, in_color
      common /color$/ icol, in_color
      save last, cta, sta, cda, sda, base_xp, base_yp, turt_xp, turt_yp
      save musl_xp, musl_yp, musl_xm, musl_ym, radr_xp, radr_yp
      save radr_xm, radr_ym, radxp, radyp, radzp, radxm, radym, radzm
      save musxm, musym, muszm, musxp, musyp, muszp, turxp, turyp
      save turzp, basxm, basym, baszm, basxp, basyp, baszp
      save base_xm, base_ym
      data last / .false./
      data sda, cda /0.087156, 0.996195/
      data sta, cta / 0.0, 1.0/
      data radxp / 5, 5, 3, -3, -5, -5, -3, 3/
      data radyp / 3, 3, 0, 0, 3, 3, 0, 0/
      data radzp / 15, 19, 21, 21, 19, 15, 13, 13/
      data radxm / 5, 5/
      data radym / -40, -40/
      data radzm / 10, 13/
      data musxm / 3, 3, -3, -3, -3, -3, 3, 3/
      data musym / 4, 60, 4, 60, 19, 60, 19, 60/
      data muszm / 3, 3, 3, 3, -3, -3, -3, -3/
      data musxp / 3, 3, -3, -3/
      data musyp / 4 * 60/
      data muszp / -3, 3, 3, -3/
      data turxp / -10, -10, 10, 10, 25, 25, 10, -10, -25, -25, -10, 10/
      data turyp / -45, -13, -13, -45, -50, 60, -13, -13, 60, -50, -45,
     &             -45/
      data turzp / 10, 10, 10, 10, -11, -20, 10, 10, -20, -11, 10, 10/
      data basxp / 30, 35, 35, 30, -30, -35, -35, -30, 30, 30/
      data basyp / -53, -60, 60, 37, 37, 60, -60, -53, -53, 37/
      data baszp / -40, -10, -20, -40, -40, -20, -10, -40, -40, -40/
      data basxm / 35, -35, 35, -35, -30, -30/
      data basym / -60, -60, 60, 60, -53, 37/
      data baszm / -10, -10, -20, -20, -40, -40/
c
c    erase the old image if drawn last turn through
c
      if (last) then
        call gpr_$set_draw_value (icol(0), status)
        call gpr_$move (base_xp(1), base_yp(1), status)
        call gpr_$polyline (base_xp, base_yp, int2(10), status)
        call gpr_$multiline (base_xm, base_ym, int2(6), status)
        call gpr_$move (turt_xp(1), turt_yp(1), status)
        call gpr_$polyline (turt_xp, turt_yp, int2(12), status)
        call gpr_$multiline (musl_xm, musl_ym, int2(8), status)
        call gpr_$move (musl_xp(4), musl_yp(4), status)
        call gpr_$polyline (musl_xp, musl_yp, int2(4), status)
        call gpr_$move (radr_xp(8), radr_yp(8), status)
        call gpr_$polyline (radr_xp, radr_yp, int2(8), status)
        call gpr_$multiline (radr_xm, radr_ym, int2(2), status)
      end if
c
c    build the vectors for the new tank
c
      if (seen(1)) then
        color = orange(1) / 300 + 2
        if (color .gt. 7) color = 7
        call gpr_$set_draw_value (icol(color), status)
        temp = cta
        cta = cta * cda - sta * sda
        sta = sta * cda + sda * temp
        csa = cos(oazm(1))
        ssa = sin(oazm(1))
        dx = xcrd - obx(1)
        dy = ycrd - oby(1)
        xn =  dx * csa + dy * ssa
        yn = -dx * ssa + dy * csa
        cn = ca * csa + sa * ssa
        sn = sa * csa - ssa * ca
c
c    do the radar dish
c
        pole_x = radxm(1) - xn
        pole_y = radym(1) - yn
        do 10 i = 1, 2
          rx = radxm(i) - xn
          ry = radym(i) - yn
          prx =  rx * cn + ry * sn
          pry = -rx * sn + ry * cn
          if (pry .lt. 10.0) pry = 10.0
          radr_xm(i) = 500 + prx / pry * 450
          radr_ym(i) = 260 - radzm(i) / pry * 450
 10     continue
        do 20 i = 1, 8
          rx = radxp(i) * cta - radyp(i) * sta + pole_x
          ry = radxp(i) * sta + radyp(i) * cta + pole_y
          prx =  rx * cn + ry * sn
          pry = -rx * sn + ry * cn
          if (pry .lt. 10.0) pry = 10.0
          radr_xp(i) = 500 + prx / pry * 450
          radr_yp(i) = 260 - radzp(i) / pry * 450
 20     continue
c
c    do the gun muzzle
c
        do 30 i = 1, 8
          rx =  musxm(i) - xn
          ry =  musym(i) - yn
          prx =  rx * cn + ry * sn
          pry = -rx * sn + ry * cn
          if (pry .lt. 10.0) pry = 10.0
          musl_xm(i) = 500 + prx / pry * 450
          musl_ym(i) = 260 - muszm(i) / pry * 450
 30     continue
        do 40 i = 1, 4
          rx =  musxp(i) - xn
          ry =  musyp(i) - yn
          prx =  rx * cn + ry * sn
          pry = -rx * sn + ry * cn
          if (pry .lt. 10.0) pry = 10.0
          musl_xp(i) = 500 + prx / pry * 450
          musl_yp(i) = 260 - muszp(i) / pry * 450
 40     continue
c
c    do the turret
c
        do 50 i = 1, 12
          rx =  turxp(i) - xn
          ry =  turyp(i) - yn
          prx =  rx * cn + ry * sn
          pry = -rx * sn + ry * cn
          if (pry .lt. 10.0) pry = 10.0
          turt_xp(i) = 500 + prx / pry * 450
          turt_yp(i) = 260 - turzp(i) / pry * 450
 50     continue
c
c    do the base
c
        do 60 i = 1, 6
          rx =  basxm(i) - xn
          ry =  basym(i) - yn
          prx =  rx * cn + ry * sn
          pry = -rx * sn + ry * cn
          if (pry .lt. 10.0) pry = 10.0
          base_xm(i) = 500 + prx / pry * 450
          base_ym(i) = 260 - baszm(i) / pry * 450
 60     continue
        do 70 i = 1, 10
          rx =  basxp(i) - xn
          ry =  basyp(i) - yn
          prx =  rx * cn + ry * sn
          pry = -rx * sn + ry * cn
          if (pry .lt. 10.0) pry = 10.0
          base_xp(i) = 500 + prx / pry * 450
          base_yp(i) = 260 - baszp(i) / pry * 450
 70     continue
        call gpr_$move (base_xp(1), base_yp(1), status)
        call gpr_$polyline (base_xp, base_yp, int2(10), status)
        call gpr_$multiline (base_xm, base_ym, int2(6), status)
        call gpr_$move (turt_xp(1), turt_yp(1), status)
        call gpr_$polyline (turt_xp, turt_yp, int2(12), status)
        call gpr_$multiline (musl_xm, musl_ym, int2(8), status)
        call gpr_$move (musl_xp(4), musl_yp(4), status)
        call gpr_$polyline (musl_xp, musl_yp, int2(4), status)
        call gpr_$move (radr_xp(8), radr_yp(8), status)
        call gpr_$polyline (radr_xp, radr_yp, int2(8), status)
        call gpr_$multiline (radr_xm, radr_ym, int2(2), status)
        last = .true.
      else
        last = .false.
      end if
      return
      end





      subroutine bzone_explode_salvo (xcrd, ycrd, ca, sa, obx, oby,
     &                                oazm, orange, seen, seed, new)
c
c     BZONE_EXPLODE_SALVO draws the player and enemy tank fire as
c     it hits an object
c
% include '/sys/ins/base.ins.ftn'
% include '/sys/ins/gpr.ins.ftn'
c
      integer*2 mult_x(10, 4), mult_y(10, 4)
      integer*4 index, status, color, count, icol(0:15)
      real xcrd, ycrd, ca, sa, obx(4), oby(4), orange(4), oazm(4)
      real seed, gravity, xoff(5, 4), yoff (5, 4), zoff(5, 4)
      real xvel(5, 4), yvel (5, 4), zvel(5, 4), mrx(10), mry(10)
      real mrz(10), xmoff(10), ymoff(10), zmoff(10)
      logical seen(4), last(4), new(12), in_color
      common /color$/ icol, in_color
      save last, mult_x, mult_y, gravity, xoff, zoff, yoff
      save xvel, yvel, zvel, count
      data last / 4 * .false./
      data gravity / 1.0/
c
c    if this is a new blast then generate the projectile motion vectors
c
      do 10 i = 3, 4
        if (new(i)) then
          count = 0
          new(i) = .false.
          do 20 j = 1, 5
            call rand (seed)
            xvel(j, i) = seed * 10.0
            call rand (seed)
            if (seed .ge. 0.5) xvel(j, i) = - xvel(j, i)
            call rand (seed)
            yvel(j, i) = seed * 10.0
            call rand (seed)
            if (seed .ge. 0.5) yvel(j, i) = - yvel(j, i)
            call rand (seed)
            zvel(j, i) = -7.5 - seed * 5.0
            xoff(j, i) = 0.0
            yoff(j, i) = 0.0
            zoff(j, i) = 0.0
 20       continue
        end if
 10   continue
c
c    erase the salvo if drawn last pass
c
      do 30 index = 3, 4
        if (last(index)) then
          call gpr_$set_draw_value (icol(0), status)
          call gpr_$multiline (mult_x(1, index), mult_y(1, index),
     &                         int2(10), status)
        end if
c
c    draw the new salvo if seen
c
        if (seen(index)) then
          color = count / 4 + 8
          if (color .gt. 13) color = 13
          call gpr_$set_draw_value (icol(color), status)
          do 40 i = 1, 5
            xoff(i, index) = xoff(i, index) + xvel(i, index)
            yoff(i, index) = yoff(i, index) + yvel(i, index)
            zvel(i, index) = zvel(i, index) + gravity
            zoff(i, index) = zoff(i, index) - zvel(i, index)
 40       continue
          do 50 i = 1, 5
            k = 2 * i - 1
            l = k + 1
            xmoff(k) = xoff(i, index) + obx(index) - xcrd
            ymoff(k) = yoff(i, index) + oby(index) - ycrd
            zmoff(k) = zoff(i, index)
            mrx(k) =  xmoff(k) * ca + ymoff(k) * sa
            mry(k) = -xmoff(k) * sa + ymoff(k) * ca
            mrz(k) =  zmoff(k)
            if (mry(k) .lt. 7.5) mry(k) = 7.5
            mult_x(k, index) = 500 + mrx(k) / mry(k) * 450
            mult_y(k, index) = 260 - mrz(k) / mry(k) * 450
            mult_x(l, index) = mult_x(k, index) + 2
            mult_y(l, index) = mult_y(k, index) + 2
 50       continue
          call gpr_$multiline (mult_x(1, index), mult_y(1, index),
     &                         int2(10), status)
          last(index) = .true.
        else
          last(index) = .false.
        end if
 30   continue
      count = count + 1
      return
      end





      subroutine bzone_explode_lander (xcrd, ycrd, ca, sa, obx, oby,
     &                                 orange, seen, cla, sla)
c
c     BZONE_EXPLODE_LANDER erases the old lander and draws a new as seen
c     from the player tank.
c
% include '/sys/ins/base.ins.ftn'
% include '/sys/ins/gpr.ins.ftn'
c
      integer*2 poly_x(21), poly_y(21), xp(21), yp(21), zp(21)
      integer*4 count, color, icol(0:15)
      real xcrd, ycrd, ca, sa, obx(4), oby(4), orange(4)
      real cla, sla, cda, sda, temp, rx, ry, prx, pry
      logical seen(4), last, in_color
      common /color$/ icol, in_color
      save last, poly_x, poly_y, xp, yp, zp, cda, sda, count
      data last / .false./
      data xp /0, 80, 40, -40, -80, 0, 0, 0, 0, 0, 0, -57, -28, 28, 57,
     &         0, 57, 28, -28, -57, 0/
      data yp /0, 0, 0, 0, 0, 0, 80, 40, -40, -80, 0, -57, -28, 28, 57,
     &         0, -57, -28, 28, 57, 0/
      data zp /20, -20, -40, -40, -20, 20, -20, -40, -40, -20, 20, -20,
     &         -40, -40, -20, 20, -20, -40, -40, -20, 20/
      data sda, cda /0.087156, 0.996195/
      data count / 3/
c
c    erase the old lander if one was drawn
c
      count = count + 1
      if (last) then
        call gpr_$set_draw_value (icol(0), status)
        call gpr_$move (poly_x(1), poly_y(1), status)
        call gpr_$polyline (poly_x, poly_y, int2(21), status)
      end if
c
c    draw the new lander if seen
c
      if (seen(2) .and. count .ge. 4) then
        count = 1
        temp = cla
        cla = cla * cda - sla * sda
        sla = sla * cda + sda * temp
        dx = xcrd - obx(2)
        dy = ycrd - oby(2)
        xn =  dx * cla + dy * sla
        yn = -dx * sla + dy * cla
        cn = ca * cla + sa * sla
        sn = sa * cla - ca * sla
        do 10 i = 1, 21
          rx = xp(i) - xn
          ry = yp(i) - yn
          prx =  rx * cn + ry * sn
          pry = -rx * sn + ry * cn
          if (pry .lt. 10.0) pry = 10.0
          poly_x(i) = 500 + prx / pry * 450
          poly_y(i) = 260 - zp(i) / pry * 450
 10     continue
        color = orange(2) / 300 + 2
        if (color .gt. 7) color = 7
        call gpr_$set_draw_value (icol(color), status)
        call gpr_$move (poly_x(1), poly_y(1), status)
        call gpr_$polyline (poly_x, poly_y, int2(21), status)
        last = .true.
      else
        last = .false.
      end if
      return
      end





      subroutine bzone_explode_missile (xcrd, ycrd, ca, sa, obx, oby,
     &                                  oazm, orange, seen, height,
     &                                  new, count, seed)
c
c     BZONE_EXPLODE_MISSILE draws the enemy missile exploding.
c     The routine handles all the projections internally and thus
c     should be called even if seen is false.
c
% include '/sys/ins/base.ins.ftn'
% include '/sys/ins/gpr.ins.ftn'
c
      integer*2 capx(11), capy(11), capz(11), conx(9), cony(9), conz(9)
      integer*2 finx(4), finy(4), finz(4), cap_x(11), cap_y(11)
      integer*2 con_x(9), con_y(9), fin1_x(4), fin1_y(4), fin2_x(4)
      integer*2 fin2_y(4)
      integer*4 status, color, count, icol(0:15)
      real xcrd, ycrd, ca, sa, obx(4), oby(4), oazm(4), orange(4)
      real seed, cta(4), sta(4), ctp(4), stp(4), xvel(4), yvel(4)
      real zvel(4), xpos(4), ypos(4), zpos(4), gravity, pro_x(4)
      real pro_y(4), check(4), range(4), threshold, cda(4), sda(4)
      real cdp(4), sdp(4), rx, ry, rz, prx, pry, prz, offset(4)
      logical seen(4), new, last(4), draw(4), in_color
      common /color$/ icol, in_color
      save capx, capy, capz, conx, cony, conz, finx, finy, finz
      save cap_x, cap_y, con_x, con_y, fin1_x, fin1_y, fin2_x
      save fin2_y, last, gravity, threshold, xvel, yvel, zvel
      save xpos, ypos, zpos, cda, sda, cdp, sdp, cta, sta, ctp, stp
      save offset
      data capx / 15, 25, 0, 15, -15, 0, 15, -15, 0, -25, -15/
      data capy / -30, -30, -45, -30, -30, -45, -30, -30, -45,
     &            -30, -30/
      data capz / -25, 0, 0, -25, -25, 0, 25, 25, 0, 0, 25/
      data conx / 15, 0, 25, 15, 0, -15, -25, 0, -15/
      data cony / -30, 50, -30, -30, 50, -30, -30, 50, -30/
      data conz / -25, 0, 0, 25, 0, -25, 0, 0, 25/
      data finx / 15, 23, 23, 13/
      data finy / -30, -38, 0, -17/
      data finz / 5, -10, -10, 11/
      data offset / 0.0, 0.0, -30.0, -30.0/
      data last / 4 * .false./
      data sda / 2 * 0.087156, 2 * -0.087156/
      data cda / 2 * 0.996195, 2 *  0.996195/
      data sdp / 2 * 0.087156, 2 * -0.087156/
      data cdp / 2 * 0.996195, 2 *  0.996195/
      data gravity, threshold / 1.0, 0.8/
c
c    erase the old bits and pieces if seen last time through
c
      call gpr_$set_draw_value (icol(0), status)
      if (last(1)) then
        call gpr_$move (cap_x(1), cap_y(1), status)
        call gpr_$polyline (cap_x, cap_y, int2(11), status)
      end if
      if (last(2)) then
        call gpr_$move (con_x(1), con_y(1), status)
        call gpr_$polyline (con_x, con_y, int2(9), status)
      end if
      if (last(3)) then
        call gpr_$move (fin1_x(4), fin1_y(4), status)
        call gpr_$polyline (fin1_x, fin1_y, int2(4), status)
      end if
      if (last(4)) then
        call gpr_$move (fin2_x(4), fin2_y(4), status)
        call gpr_$polyline (fin2_x, fin2_y, int2(4), status)
      end if
      if (count .eq. 40) return
c
c    if new then get the random velocity vectors and starting sines
c    and cosines. Note that subscripts are as follows
c
c    1. cap
c    2. cone
c    3. fin
c    4. fin
c
      if (new) then
        new = .false.
        cta(1) = cos(oazm(1))
        sta(1) = sin(oazm(1))
        ctp(1) = 1.0
        stp(1) = 0.0
        do 10 i = 2, 4
          cta(i) = cta(1)
          sta(i) = sta(1)
          ctp(i) = 1.0
          stp(i) = 0.0
 10     continue
        do 20 i = 1, 4
          xpos(i) = 0.0
          ypos(i) = 0.0
          zpos(i) = height + offset(i)
 20     continue
        do 30 i = 1, 4
          call rand (seed)
          xvel(i) = seed * 10.0
          call rand (seed)
          if (seed .ge. 0.5) xvel(i) = - xvel(i)
          call rand (seed)
          yvel(i) = seed * 10.0
          call rand (seed)
          if (seed .ge. 0.5) yvel(i) = - yvel(i)
          call rand (seed)
          zvel(i) = 12.5 + seed * 7.5
 30     continue
      end if
c
c    project all the pieces into player-centric space
c
      do 40 i = 1, 4
        dx = obx(1) + xpos(i) - xcrd
        dy = oby(1) + ypos(i) - ycrd
        range(i) = sqrt(dx**2 + dy**2)
        if (range(i) .lt. 2000.0) then
          pro_y(i) = -dx * sa + dy * ca
          pro_x(i) =  dx * ca + dy * sa
          check(i) =  pro_y(i) / (abs(pro_x(i)) + 1.0)
          if (check(i) .gt. threshold) then
            draw(i) = .true.
          else
            draw(i) = .false.
          end if
        else
          draw(i) = .false.
        end if
 40   continue
c
c    draw the pieces that can been seen, cap first
c
      if (draw(1)) then
        dx = xcrd - obx(1) - xpos(1)
        dy = ycrd - oby(1) - ypos(1)
        dz = - zpos(1)
        xn =  dx * cta(1)          + dy * sta(1)
        yn = -dx * sta(1) * ctp(1) + dy * cta(1) * ctp(1) + dz * stp(1)
        zn =  dx * sta(1) * stp(1) - dy * cta(1) * stp(1) + dz * ctp(1)
        cn = ca * cta(1) + sa * sta(1)
        sn = sa * cta(1) - ca * sta(1)
        cp = ctp(1)
        sp =-stp(1)
        do 50 i = 1, 11
          rx =  capx(i) - xn
          ry =  capy(i) - yn
          rz =  capz(i) - zn
          prx =  rx * cn + ry * sn * cp + rz * sn * sp
          pry = -rx * sn + ry * cn * cp + rz * cn * sp
          prz =          - ry * sp      + rz * cp
          if (pry .lt. 10.0) pry = 10.0
          cap_x(i) = 500 + prx / pry * 450
          cap_y(i) = 260 - prz / pry * 450
 50     continue
        last(1) = .true.
        color = range(1) / 300 + 2
        if (color .gt. 7) color = 7
        call gpr_$set_draw_value (icol(color), status)
        call gpr_$move (cap_x(1), cap_y(1), status)
        call gpr_$polyline (cap_x, cap_y, int2(11), status)
      else
        last(1) = .false.
      end if
c
c    cone next
c
      if (draw(2)) then
        dx = xcrd - obx(1) - xpos(2)
        dy = ycrd - oby(1) - ypos(2)
        dz = - zpos(2)
        xn =  dx * cta(2)          + dy * sta(2)
        yn = -dx * sta(2) * ctp(2) + dy * cta(2) * ctp(2) + dz * stp(2)
        zn =  dx * sta(2) * stp(2) - dy * cta(2) * stp(2) + dz * ctp(2)
        cn = ca * cta(2) + sa * sta(2)
        sn = sa * cta(2) - ca * sta(2)
        cp = ctp(2)
        sp =-stp(2)
        do 60 i = 1, 9
          rx =  conx(i) - xn
          ry =  cony(i) - yn
          rz =  conz(i) - zn
          prx =  rx * cn + ry * sn * cp + rz * sn * sp
          pry = -rx * sn + ry * cn * cp + rz * cn * sp
          prz =          - ry * sp      + rz * cp
          if (pry .lt. 10.0) pry = 10.0
          con_x(i) = 500 + prx / pry * 450
          con_y(i) = 260 - prz / pry * 450
 60     continue
        last(2) = .true.
        color = range(2) / 300 + 2
        if (color .gt. 7) color = 7
        call gpr_$set_draw_value (icol(color), status)
        call gpr_$move (con_x(1), con_y(1), status)
        call gpr_$polyline (con_x, con_y, int2(9), status)
      else
        last(2) = .false.
      end if
c
c    fins
c
      if (draw(3)) then
        dx = xcrd - obx(1) - xpos(3)
        dy = ycrd - oby(1) - ypos(3)
        dz = - zpos(3)
        xn =  dx * cta(3)          + dy * sta(3)
        yn = -dx * sta(3) * ctp(3) + dy * cta(3) * ctp(3) + dz * stp(3)
        zn =  dx * sta(3) * stp(3) - dy * cta(3) * stp(3) + dz * ctp(3)
        cn = ca * cta(3) + sa * sta(3)
        sn = sa * cta(3) - ca * sta(3)
        cp = ctp(3)
        sp =-stp(3)
        do 80 i = 1, 4
          rx =  finx(i) - xn
          ry =  finy(i) - yn
          rz =  finz(i) - zn
          prx =  rx * cn + ry * sn * cp + rz * sn * sp
          pry = -rx * sn + ry * cn * cp + rz * cn * sp
          prz =          - ry * sp      + rz * cp
          if (pry .lt. 10.0) pry = 10.0
          fin1_x(i) = 500 + prx / pry * 450
          fin1_y(i) = 260 - prz / pry * 450
 80     continue
        last(3) = .true.
        color = range(3) / 300 + 2
        if (color .gt. 7) color = 7
        call gpr_$set_draw_value (icol(color), status)
        call gpr_$move (fin1_x(4), fin1_y(4), status)
        call gpr_$polyline (fin1_x, fin1_y, int2(4), status)
      else
        last(3) = .false.
      end if
      if (draw(4)) then
        dx = xcrd - obx(1) - xpos(4)
        dy = ycrd - oby(1) - ypos(4)
        dz = - zpos(4)
        xn =  dx * cta(4)          + dy * sta(4)
        yn = -dx * sta(4) * ctp(4) + dy * cta(4) * ctp(4) + dz * stp(4)
        zn =  dx * sta(4) * stp(4) - dy * cta(4) * stp(4) + dz * ctp(4)
        cn = ca * cta(4) + sa * sta(4)
        sn = sa * cta(4) - ca * sta(4)
        cp = ctp(4)
        sp =-stp(4)
        do 100 i = 1, 4
          rx = -finx(i) - xn
          ry =  finy(i) - yn
          rz =  finz(i) - zn
          prx =  rx * cn + ry * sn * cp + rz * sn * sp
          pry = -rx * sn + ry * cn * cp + rz * cn * sp
          prz =          - ry * sp      + rz * cp
          if (pry .lt. 10.0) pry = 10.0
          fin2_x(i) = 500 + prx / pry * 450
          fin2_y(i) = 260 - prz / pry * 450
100     continue
        last(4) = .true.
        color = range(4) / 300 + 2
        if (color .gt. 7) color = 7
        call gpr_$set_draw_value (icol(color), status)
        call gpr_$move (fin2_x(4), fin2_y(4), status)
        call gpr_$polyline (fin2_x, fin2_y, int2(4), status)
      else
        last(4) = .false.
      end if
c
c    update the vector offsets and velocities and rotation angles
c
      do 130 i = 1, 4
        xpos(i) = xpos(i) + xvel(i)
        ypos(i) = ypos(i) + yvel(i)
        zvel(i) = zvel(i) - gravity
        zpos(i) = zpos(i) + zvel(i)
        if (zpos(i) .lt. -40.0) then
          zpos(i) = -40.0
          zvel(i) = -0.2 * zvel(i)
        end if
        temp = cta(i)
        cta(i) = cta(i) * cda(i) - sta(i) * sda(i)
        sta(i) = sta(i) * cda(i) + sda(i) * temp
        temp = ctp(i)
        ctp(i) = ctp(i) * cdp(i) - stp(i) * sdp(i)
        stp(i) = stp(i) * cdp(i) + sdp(i) * temp
130   continue
      return
      end






      subroutine bzone_explode_copter (xcrd, ycrd, ca, sa, obx, oby,
     &                                 oazm, orange, seen, height, new,
     &                                 count, seed)
c
c     BZONE_EXPLODE_COPTER explodes an enemy helicopter
c
% include '/sys/ins/base.ins.ftn'
% include '/sys/ins/gpr.ins.ftn'
c
      integer*2 proxp(5), proyp(5), prozp(5), prop_xp(5), prop_yp(5)
      integer*2 extxm(10), extym(10), extzm(10), extr_xm(10)
      integer*2 extr_ym(10), talxp(11), talyp(11), talzp(11)
      integer*2 tail_xp(11), tail_yp(11), bodxp(19), bodyp(19)
      integer*2 bodzp(19), body_xp(19), body_yp(19)
      integer*2 bodxm(10), bodym(10), bodzm(10), body_xm(10)
      integer*2 body_ym(10)
      integer*4 status, icol(0:15), color, count
      real xcrd, ycrd, ca, sa, obx(4), oby(4), oazm(4), orange(4)
      real seed, cta(3), sta(3), ctp(3), stp(3), xvel(3), yvel(3)
      real zvel(3), xpos(3), ypos(3), zpos(3), gravity, pro_x(3)
      real pro_y(3), check(3), range(3), threshold
      real cda(3), sda(3), cdp(3), sdp(3)
      real rx, ry, rz, prx, pry, prz, yoffset(3),  zoffset(3)
      logical seen(4), new, last(3), draw(3), in_color
      common /color$/ icol, in_color
      save last, dcp, dsp, cpa, spa
      save cta, sta, cda, sda, proxp, proyp, prozp, prop_xp, prop_yp
      save extxm, extym, extzm, extr_xm, extr_ym
      save talxp, talyp, talzp, tail_xp, tail_yp
      save bodxp, bodyp, bodzp, body_xp, body_yp
      save bodxm, bodym, bodzm, body_xm, body_ym
      save last, xvel, yvel, zvel, xpos, ypos, zpos
      save ctp, stp, gravity, threshold, sda, cda, sdp, cdp
      save yoffset, zoffset
      data last / 3*.false./
      data cpa, spa, dcp, dsp / 1.0, 0.0, 0.819152, 0.573576/
      data sda / 0.573576, 0.087156, -0.087156/
      data cda / 0.819152, 0.996195,  0.996195/
      data sdp / 2 * 0.087156, -0.087156/
      data cdp / 2 * 0.996195,  0.996195/
      data gravity, threshold / 1.0, 0.8/
      data proxp / 6, -6, 6, -6, 6/
      data proyp / 100, -100, -100, 100, 100/
      data prozp / 5 * 0/
      data extxm / 0, 0, -14, -14, -14, -14, 14, 14, 14, 14/
      data extym / 0, 0, -30, 40, 40, 44, -30, 40, 40, 44/
      data extzm / 16, 36, -40, -40, -40, -36, -40, -40, -40, -36/
      data talxp / 0, 7, 10, 0, 0, 0, 0, 0, -7, -10, 0/
      data talyp / -32, 52, 48, -34, -52, -62, -48, -32, 52, 48, -34/
      data talzp / -10, -34, -14, 0, 28, 28, -10, -10, -34, -14, 0/
      data bodxp / -14, 14, 0, -14, -7, 7, 14, 26, 0, -26, -10, 10,
     &             26, 4, 0, -4, -4, 4, 4/
      data bodyp / 34, 34, 60, 34, -30, -30, 34, 34, 60, 34, -34,
     &             -34, 34, 20, 22, 20, -22, -22, 20/
      data bodzp / -34, -34, -14, -34, -34, -34, -34, -14, -14, -14,
     &             -14, -14, -14, 16, 16, 16, 16, 16, 16/
      data bodxm / -10, -4, 10, 4, 0, 0, -14, -26, -26, -4/
      data bodym / -34, -22, -34, -22, 22, 60, 34, 34, 34, 20/
      data bodzm / -14, 16, -14, 16, 16, -14, -34, -14, -14, 16/
      data yoffset / 0.0, -82.0, 0.0/
      data zoffset / 30.0, 0.0, 0.0/
c
c    erase figure if drawn last time
c
      call gpr_$set_draw_value (icol(0), status)
      if (last(1)) then
        call gpr_$move (prop_xp(5), prop_yp(5), status)
        call gpr_$polyline (prop_xp, prop_yp, int2(5), status)
      end if
      if (last(2)) then
        call gpr_$move (tail_xp(1), tail_yp(1), status)
        call gpr_$polyline (tail_xp, tail_yp, int2(11), status)
      end if
      if (last(3)) then
        call gpr_$move (body_xp(1), body_yp(1), status)
        call gpr_$polyline (body_xp, body_yp, int2(19), status)
        call gpr_$multiline (body_xm, body_ym, int2(10), status)
      end if
      if (count .eq. 40) return
c
c    build the vectors for the next draw
c
      if (new) then
        new = .false.
        cta(1) = cos(oazm(1))
        sta(1) = sin(oazm(1))
        ctp(1) = 1.0
        stp(1) = 0.0
        do 10 i = 2, 3
          cta(i) = cta(1)
          sta(i) = sta(1)
          ctp(i) = 1.0
          stp(i) = 0.0
 10     continue
        do 20 i = 1, 3
          xpos(i) =-yoffset(i) * sta(1)
          ypos(i) = yoffset(i) * cta(1)
          zpos(i) = height + zoffset(i)
 20     continue
        do 30 i = 1, 3
          call rand (seed)
          xvel(i) = seed * 10.0
          call rand (seed)
          if (seed .ge. 0.5) xvel(i) = - xvel(i)
          call rand (seed)
          yvel(i) = seed * 10.0
          call rand (seed)
          if (seed .ge. 0.5) yvel(i) = - yvel(i)
          call rand (seed)
          zvel(i) = 12.5 + seed * 7.5
 30     continue
        zvel(1) = zvel(1) + 6.0
      end if
c
c    project all the pieces into player-centric space
c
      do 40 i = 1, 3
        dx = obx(1) + xpos(i) - xcrd
        dy = oby(1) + ypos(i) - ycrd
        range(i) = sqrt(dx**2 + dy**2)
        if (range(i) .lt. 2000.0) then
          pro_y(i) = -dx * sa + dy * ca
          pro_x(i) =  dx * ca + dy * sa
          check(i) =  pro_y(i) / (abs(pro_x(i)) + 1.0)
          if (check(i) .gt. threshold) then
            draw(i) = .true.
          else
            draw(i) = .false.
          end if
        else
          draw(i) = .false.
        end if
 40   continue
c
c    prop first
c
      if (draw(1)) then
        dx = xcrd - obx(1) - xpos(1)
        dy = ycrd - oby(1) - ypos(1)
        dz = - zpos(1)
        xn =  dx * cta(1) + dy * sta(1) * ctp(1) + dz * sta(1) * stp(1)
        yn = -dx * sta(1) + dy * cta(1) * ctp(1) + dz * cta(1) * stp(1)
        zn =              - dy * stp(1)          + dz * ctp(1)
        c1 = ca * cta(1)
        s1 = sa * sta(1)
        c2 = ca * sta(1)
        s2 = sa * cta(1)
        cp = ctp(1)
        sp = stp(1)
        t1 = c1 + s1 * cp
        t2 = -c2 + s2 * cp
        t3 = -sa * sp
        t4 = -s2 + c2 * cp
        t5 = s1 + c1 * cp
        t6 = -ca * sp
        t7 = sta(1) * sp
        t8 = cta(1) * sp
        t9 = cp
        do 50 i = 1, 5
          rx = proxp(i) - xn
          ry = proyp(i) - yn
          rz = prozp(i) - zn
          prx = rx * t1 + ry * t2 + rz * t3
          pry = rx * t4 + ry * t5 + rz * t6
          prz = rx * t7 + ry * t8 + rz * t9
          if (pry .lt. 10.0) pry = 10.0
          prop_xp(i) = 500 + prx / pry * 450
          prop_yp(i) = 260 - prz / pry * 450
 50     continue
        last(1) = .true.
        color = range(1) / 300 + 2
        if (color .gt. 7) color = 7
        call gpr_$set_draw_value (icol(color), status)
        call gpr_$move (prop_xp(5), prop_yp(5), status)
        call gpr_$polyline (prop_xp, prop_yp, int2(5), status)
      else
        last(1) = .false.
      end if
c
c    tail
c
      if (draw(2)) then
        dx = xcrd - obx(1) - xpos(2)
        dy = ycrd - oby(1) - ypos(2)
        dz = - zpos(2)
        xn =  dx * cta(2)          + dy * sta(2)
        yn = -dx * sta(2) * ctp(2) + dy * cta(2) * ctp(2) + dz * stp(2)
        zn =  dx * sta(2) * stp(2) - dy * cta(2) * stp(2) + dz * ctp(2)
        cn = ca * cta(2) + sa * sta(2)
        sn = sa * cta(2) - ca * sta(2)
        cp = ctp(2)
        sp =-stp(2)
        do 60 i = 1, 11
          rx = talxp(i) - xn
          ry = talyp(i) - yn
          rz = talzp(i) - zn
          prx =  rx * cn + ry * sn * cp + rz * sn * sp
          pry = -rx * sn + ry * cn * cp + rz * cn * sp
          prz =          - ry * sp      + rz * cp
          if (pry .lt. 10.0) pry = 10.0
          tail_xp(i) = 500 + prx / pry * 450
          tail_yp(i) = 260 - prz / pry * 450
 60     continue
        last(2) = .true.
        color = range(2) / 300 + 2
        if (color .gt. 7) color = 7
        call gpr_$set_draw_value (icol(color), status)
        call gpr_$move (tail_xp(1), tail_yp(1), status)
        call gpr_$polyline (tail_xp, tail_yp, int2(11), status)
      else
        last(2) = .false.
      end if
c
c    cabin
c
      if (draw(3)) then
        dx = xcrd - obx(1) - xpos(3)
        dy = ycrd - oby(1) - ypos(3)
        dz = - zpos(3)
        xn =  dx * cta(3)          + dy * sta(3)
        yn = -dx * sta(3) * ctp(3) + dy * cta(3) * ctp(3) + dz * stp(3)
        zn =  dx * sta(3) * stp(3) - dy * cta(3) * stp(3) + dz * ctp(3)
        cn = ca * cta(3) + sa * sta(3)
        sn = sa * cta(3) - ca * sta(3)
        cp = ctp(3)
        sp =-stp(3)
        do 70 i = 1, 19
          rx = bodxp(i) - xn
          ry = bodyp(i) - yn
          rz = bodzp(i) - zn
          prx =  rx * cn + ry * sn * cp + rz * sn * sp
          pry = -rx * sn + ry * cn * cp + rz * cn * sp
          prz =          - ry * sp      + rz * cp
          if (pry .lt. 10.0) pry = 10.0
          body_xp(i) = 500 + prx / pry * 450
          body_yp(i) = 260 - prz / pry * 450
 70     continue
        do 80 i = 1, 10
          rx = bodxm(i) - xn
          ry = bodym(i) - yn
          rz = bodzm(i) - zn
          prx =  rx * cn + ry * sn * cp + rz * sn * sp
          pry = -rx * sn + ry * cn * cp + rz * cn * sp
          prz =          - ry * sp      + rz * cp
          if (pry .lt. 10.0) pry = 10.0
          body_xm(i) = 500 + prx / pry * 450
          body_ym(i) = 260 - prz / pry * 450
 80     continue
        last(3) = .true.
        color = range(3) / 300 + 2
        if (color .gt. 7) color = 7
        call gpr_$set_draw_value (icol(color), status)
        call gpr_$move (body_xp(1), body_yp(1), status)
        call gpr_$polyline (body_xp, body_yp, int2(19), status)
        call gpr_$multiline (body_xm, body_ym, int2(10), status)
      else
        last(3) = .false.
      end if
c
c    update the vector offsets and velocities and rotation angles
c
      do 90 i = 1, 3
        xpos(i) = xpos(i) + xvel(i)
        ypos(i) = ypos(i) + yvel(i)
        zvel(i) = zvel(i) - gravity
        zpos(i) = zpos(i) + zvel(i)
        if (zpos(i) .lt. -40.0) then
          zpos(i) = -40.0
          zvel(i) = -0.2 * zvel(i)
        end if
        temp = cta(i)
        cta(i) = cta(i) * cda(i) - sta(i) * sda(i)
        sta(i) = sta(i) * cda(i) + sda(i) * temp
        temp = ctp(i)
        ctp(i) = ctp(i) * cdp(i) - stp(i) * sdp(i)
        stp(i) = stp(i) * cdp(i) + sdp(i) * temp
 90   continue
      return
      end







      subroutine bzone_explode_super (xcrd, ycrd, ca, sa, obx, oby,
     &                                oazm, orange, seen, new, count,
     &                                seed)
c
c     BZONE_EXPLODE_SUPER draws the enemy tank exploding (super).
c     The routine handles all the projections internally and thus
c     should be called even if seen is false.
c
% include '/sys/ins/base.ins.ftn'
% include '/sys/ins/gpr.ins.ftn'
c
      integer*2 base_xp(8), base_yp(8), turt_xp(9), turt_yp(9)
      integer*2 musl_xp(4), musl_yp(4), musl_xm(8), musl_ym(8)
      integer*2 musxm(8), musym(8), muszm(8), musxp(4)
      integer*2 musyp(4), muszp(4), turxp(9), turyp(9), turzp(9)
      integer*2 basxm(4), basym(4), baszm(4), basxp(8), basyp(8)
      integer*2 baszp(8), base_xm(4), base_ym(4), turxm(4), turym(4)
      integer*2 turzm(4), turt_xm(4), turt_ym(4)
      integer*4 status, count, color, icol(0:15)
      real xcrd, ycrd, ca, sa, obx(4), oby(4), oazm(4), orange(4)
      real seed, cta(3), sta(3), ctp(3), stp(3), xvel(3), yvel(3)
      real zvel(3), xpos(3), ypos(3), zpos(3), gravity, pro_x(3)
      real pro_y(3), check(3), range(3), threshold
      real cda(3), sda(3), cdp(3), sdp(3)
      real rx, ry, rz, prx, pry, prz, yoffset(3), zoffset(3)
      logical seen(4), new, last(3), draw(3), in_color
      common /color$/ icol, in_color
      save last, xvel, yvel, zvel, xpos, ypos, zpos, cta, sta
      save ctp, stp, gravity, threshold, sda, cda, sdp, cdp
      save last, base_xp, base_yp, turt_xp, turt_yp, turt_xm, turt_ym
      save musl_xp, musl_yp, musl_xm, musl_ym, turxm, turym, turzm
      save musxm, musym, muszm, musxp, musyp, muszp, turxp, turyp
      save turzp, basxm, basym, baszm, basxp, basyp, baszp
      save base_xm, base_ym, yoffset, zoffset
      data last / 3*.false./
      data sda / 2 * 0.087156, -0.087156/
      data cda / 2 * 0.996195,  0.996195/
      data sdp / 2 * 0.087156, -0.087156/
      data cdp / 2 * 0.996195,  0.996195/
      data gravity, threshold / 1.0, 0.8/
      data musxm / 3, 3, -3, -3, 3, 3, -3, -3/
      data musym / -36, 36, -36, 36, -27, 36, -27, 36/
      data muszm / 0, 0, 0, 0, -6, -6, -6, -6/
      data musxp / 3, -3, -3, 3/
      data musyp / 4 * 36/
      data muszp / 0, 0, -6, -6/
      data turxp / 0, 13, 11, 11, 0, -13, -11, -11, 0/
      data turyp / 47, -48, -48, -13, 47, -48, -48, -13, 47/
      data turzp / -18, 9, 19, 19, -18, 9, 19, 19, -18/
      data turxm / 11, -11, 11, -11/
      data turym / -48, -48, -13, -13/
      data turzm / 19, 19, 19, 19/
      data basxp / 15, 30, 30, 15, -15, -30, -30, -15/
      data basyp / 60, -60, -60, 60, 60, -60, -60, 60/
      data baszp / -15, -15, 19, -15, -15, -15, 19, -15/
      data basxm / -30, 30, -30, 30/
      data basym / -60, -60, -60, -60/
      data baszm / 19, 19, -15, -15/
      data yoffset / 0.0, -12.0, 19.0/
      data zoffset / -25.0, -15.0, 0.0/
c
c    erase the old bits and pieces if seen last time through
c
      call gpr_$set_draw_value (icol(0), status)
      if (last(1)) then
        call gpr_$move (base_xp(1), base_yp(1), status)
        call gpr_$polyline (base_xp, base_yp, int2(8), status)
        call gpr_$multiline (base_xm, base_ym, int2(4), status)
      end if
      if (last(2)) then
        call gpr_$move (turt_xp(1), turt_yp(1), status)
        call gpr_$polyline (turt_xp, turt_yp, int2(9), status)
        call gpr_$multiline (turt_xm, turt_ym, int2(4), status)
      end if
      if (last(3)) then
        call gpr_$multiline (musl_xm, musl_ym, int2(8), status)
        call gpr_$move (musl_xp(4), musl_yp(4), status)
        call gpr_$polyline (musl_xp, musl_yp, int2(4), status)
      end if
      if (count .eq. 40) return
c
c    if new then get the random velocity vectors and starting sines
c    and cosines. Note that subscripts are as follows
c
c    1. base
c    2. turret
c    3. muzzle
c
      if (new) then
        new = .false.
        cta(1) = cos(oazm(1))
        sta(1) = sin(oazm(1))
        ctp(1) = 1.0
        stp(1) = 0.0
        do 10 i = 2, 3
          cta(i) = cta(1)
          sta(i) = sta(1)
          ctp(i) = 1.0
          stp(i) = 0.0
 10     continue
        do 20 i = 1, 3
          xpos(i) = -yoffset(i) * sta(1)
          ypos(i) =  yoffset(i) * cta(1)
          zpos(i) =  zoffset(i)
 20     continue
        do 30 i = 1, 3
          call rand (seed)
          xvel(i) = seed * 10.0
          call rand (seed)
          if (seed .ge. 0.5) xvel(i) = - xvel(i)
          call rand (seed)
          yvel(i) = seed * 10.0
          call rand (seed)
          if (seed .ge. 0.5) yvel(i) = - yvel(i)
          call rand (seed)
          zvel(i) = 12.5 + seed * 7.5
 30     continue
      end if
c
c    project all the pieces into player-centric space
c
      do 40 i = 1, 3
        dx = obx(1) + xpos(i) - xcrd
        dy = oby(1) + ypos(i) - ycrd
        range(i) = sqrt(dx**2 + dy**2)
        if (range(i) .lt. 2000.0) then
          pro_y(i) = -dx * sa + dy * ca
          pro_x(i) =  dx * ca + dy * sa
          check(i) =  pro_y(i) / (abs(pro_x(i)) + 1.0)
          if (check(i) .gt. threshold) then
            draw(i) = .true.
          else
            draw(i) = .false.
          end if
        else
          draw(i) = .false.
        end if
 40   continue
c
c    draw the pieces that can been seen, base first
c
      if (draw(1)) then
        dx = xcrd - obx(1) - xpos(1)
        dy = ycrd - oby(1) - ypos(1)
        dz = - zpos(1)
        xn =  dx * cta(1)          + dy * sta(1)
        yn = -dx * sta(1) * ctp(1) + dy * cta(1) * ctp(1) + dz * stp(1)
        zn =  dx * sta(1) * stp(1) - dy * cta(1) * stp(1) + dz * ctp(1)
        cn = ca * cta(1) + sa * sta(1)
        sn = sa * cta(1) - ca * sta(1)
        cp = ctp(1)
        sp =-stp(1)
        do 50 i = 1, 4
          rx =  basxm(i) - xn
          ry =  basym(i) - yn
          rz =  baszm(i) - zn
          prx =  rx * cn + ry * sn * cp + rz * sn * sp
          pry = -rx * sn + ry * cn * cp + rz * cn * sp
          prz =          - ry * sp      + rz * cp
          if (pry .lt. 10.0) pry = 10.0
          base_xm(i) = 500 + prx / pry * 450
          base_ym(i) = 260 - prz / pry * 450
 50     continue
        do 60 i = 1, 8
          rx =  basxp(i) - xn
          ry =  basyp(i) - yn
          rz =  baszp(i) - zn
          prx =  rx * cn + ry * sn * cp + rz * sn * sp
          pry = -rx * sn + ry * cn * cp + rz * cn * sp
          prz =          - ry * sp      + rz * cp
          if (pry .lt. 10.0) pry = 10.0
          base_xp(i) = 500 + prx / pry * 450
          base_yp(i) = 260 - prz / pry * 450
 60     continue
        last(1) = .true.
        color = range(1) / 300 + 2
        if (color .gt. 7) color = 7
        call gpr_$set_draw_value (icol(color), status)
        call gpr_$move (base_xp(1), base_yp(1), status)
        call gpr_$polyline (base_xp, base_yp, int2(8), status)
        call gpr_$multiline (base_xm, base_ym, int2(4), status)
      else
        last(1) = .false.
      end if
c
c    do the turret
c
      if (draw(2)) then
        dx = xcrd - obx(1) - xpos(2)
        dy = ycrd - oby(1) - ypos(2)
        dz = - zpos(2)
        xn =  dx * cta(2)          + dy * sta(2)
        yn = -dx * sta(2) * ctp(2) + dy * cta(2) * ctp(2) + dz * stp(2)
        zn =  dx * sta(2) * stp(2) - dy * cta(2) * stp(2) + dz * ctp(2)
        cn = ca * cta(2) + sa * sta(2)
        sn = sa * cta(2) - ca * sta(2)
        cp = ctp(2)
        sp =-stp(2)
        do 70 i = 1, 9
          rx =  turxp(i) - xn
          ry =  turyp(i) - yn
          rz =  turzp(i) - zn
          prx =  rx * cn + ry * sn * cp + rz * sn * sp
          pry = -rx * sn + ry * cn * cp + rz * cn * sp
          prz =          - ry * sp      + rz * cp
          if (pry .lt. 10.0) pry = 10.0
          turt_xp(i) = 500 + prx / pry * 450
          turt_yp(i) = 260 - prz / pry * 450
 70     continue
        do 80 i = 1, 4
          rx =  turxm(i) - xn
          ry =  turym(i) - yn
          rz =  turzm(i) - zn
          prx =  rx * cn + ry * sn * cp + rz * sn * sp
          pry = -rx * sn + ry * cn * cp + rz * cn * sp
          prz =          - ry * sp      + rz * cp
          if (pry .lt. 10.0) pry = 10.0
          turt_xm(i) = 500 + prx / pry * 450
          turt_ym(i) = 260 - prz / pry * 450
 80     continue
        last(2) = .true.
        color = range(2) / 300 + 2
        if (color .gt. 7) color = 7
        call gpr_$set_draw_value (icol(color), status)
        call gpr_$move (turt_xp(1), turt_yp(1), status)
        call gpr_$polyline (turt_xp, turt_yp, int2(9), status)
        call gpr_$multiline (turt_xm, turt_ym, int2(4), status)
      else
        last(2) = .false.
      end if
c
c    do the muzzle
c
      if (draw(3)) then
        dx = xcrd - obx(1) - xpos(3)
        dy = ycrd - oby(1) - ypos(3)
        dz = - zpos(3)
        xn =  dx * cta(3)          + dy * sta(3)
        yn = -dx * sta(3) * ctp(3) + dy * cta(3) * ctp(3) + dz * stp(3)
        zn =  dx * sta(3) * stp(3) - dy * cta(3) * stp(3) + dz * ctp(3)
        cn = ca * cta(3) + sa * sta(3)
        sn = sa * cta(3) - ca * sta(3)
        cp = ctp(3)
        sp =-stp(3)
        do 90 i = 1, 8
          rx =  musxm(i) - xn
          ry =  musym(i) - yn
          rz =  muszm(i) - zn
          prx =  rx * cn + ry * sn * cp + rz * sn * sp
          pry = -rx * sn + ry * cn * cp + rz * cn * sp
          prz =          - ry * sp      + rz * cp
          if (pry .lt. 10.0) pry = 10.0
          musl_xm(i) = 500 + prx / pry * 450
          musl_ym(i) = 260 - prz / pry * 450
 90     continue
        do 100 i = 1, 4
          rx =  musxp(i) - xn
          ry =  musyp(i) - yn
          rz =  muszp(i) - zn
          prx =  rx * cn + ry * sn * cp + rz * sn * sp
          pry = -rx * sn + ry * cn * cp + rz * cn * sp
          prz =          - ry * sp      + rz * cp
          if (pry .lt. 10.0) pry = 10.0
          musl_xp(i) = 500 + prx / pry * 450
          musl_yp(i) = 260 - prz / pry * 450
100     continue
        last(3) = .true.
        color = range(3) / 300 + 2
        if (color .gt. 7) color = 7
        call gpr_$set_draw_value (icol(color), status)
        call gpr_$multiline (musl_xm, musl_ym, int2(8), status)
        call gpr_$move (musl_xp(4), musl_yp(4), status)
        call gpr_$polyline (musl_xp, musl_yp, int2(4), status)
      else
        last(3) = .false.
      end if
c
c    update the vector offsets and velocities and rotation angles
c
      do 170 i = 1, 3
        xpos(i) = xpos(i) + xvel(i)
        ypos(i) = ypos(i) + yvel(i)
        zvel(i) = zvel(i) - gravity
        zpos(i) = zpos(i) + zvel(i)
        if (zpos(i) .lt. -40.0) then
          zpos(i) = -40.0
          zvel(i) = -0.2 * zvel(i)
        end if
        temp = cta(i)
        cta(i) = cta(i) * cda(i) - sta(i) * sda(i)
        sta(i) = sta(i) * cda(i) + sda(i) * temp
        temp = ctp(i)
        ctp(i) = ctp(i) * cdp(i) - stp(i) * sdp(i)
        stp(i) = stp(i) * cdp(i) + sdp(i) * temp
170   continue
      return
      end







      subroutine bzone_explode_tank (xcrd, ycrd, ca, sa, obx, oby, oazm,
     &                               orange, seen, new, count, seed)
c
c     BZONE_EXPLODE_TANK draws the enemy tank exploding (regular).
c     The routine handles all the projections internally and thus
c     should be called even if seen is false.
c
% include '/sys/ins/base.ins.ftn'
% include '/sys/ins/gpr.ins.ftn'
c
      integer*2 base_xp(10), base_yp(10), turt_xp(12), turt_yp(12)
      integer*2 musl_xp(4), musl_yp(4), musl_xm(8), musl_ym(8)
      integer*2 musxm(8), musym(8), muszm(8), musxp(4)
      integer*2 musyp(4), muszp(4), turxp(12), turyp(12), turzp(12)
      integer*2 basxm(6), basym(6), baszm(6), basxp(10), basyp(10)
      integer*2 baszp(10), base_xm(6), base_ym(6)
      integer*4 status, count, color, icol(0:15)
      real xcrd, ycrd, ca, sa, obx(4), oby(4), oazm(4), orange(4)
      real seed, cta(3), sta(3), ctp(3), stp(3), xvel(3), yvel(3)
      real zvel(3), xpos(3), ypos(3), zpos(3), gravity, pro_x(3)
      real pro_y(3), check(3), range(3), threshold
      real cda(3), sda(3), cdp(3), sdp(3)
      real rx, ry, rz, prx, pry, prz, yoffset(3),  zoffset(3)
      logical seen(4), new, last(3), draw(3), in_color
      common /color$/ icol, in_color
      save last, xvel, yvel, zvel, xpos, ypos, zpos, cta, sta
      save ctp, stp, gravity, threshold, sda, cda, sdp, cdp
      save last, base_xp, base_yp, turt_xp, turt_yp
      save musl_xp, musl_yp, musl_xm, musl_ym
      save musxm, musym, muszm, musxp, musyp, muszp, turxp, turyp
      save turzp, basxm, basym, baszm, basxp, basyp, baszp
      save base_xm, base_ym, yoffset, zoffset
      data last / 3*.false./
      data sda / 2 * 0.087156, -0.087156/
      data cda / 2 * 0.996195,  0.996195/
      data sdp / 2 * 0.087156, -0.087156/
      data cdp / 2 * 0.996195,  0.996195/
      data gravity, threshold / 1.0, 0.8/
      data musxm / 3, 3, -3, -3, -3, -3, 3, 3/
      data musym / -28, 28, -28, 28, -13, 28, -13, 28/
      data muszm / 3, 3, 3, 3, -3, -3, -3, -3/
      data musxp / 3, 3, -3, -3/
      data musyp / 4 * 28/
      data muszp / -3, 3, 3, -3/
      data turxp / -10, -10, 10, 10, 25, 25, 10, -10, -25, -25, -10, 10/
      data turyp / -50, -18, -18, -50, -55, 55, -18, -18, 55, -55, -50,
     &             -50/
      data turzp / 10, 10, 10, 10, -11, -20, 10, 10, -20, -11, 10, 10/
      data basxp / 30, 35, 35, 30, -30, -35, -35, -30, 30, 30/
      data basyp / -53, -60, 60, 37, 37, 60, -60, -53, -53, 37/
      data baszp / -15, 15, -5, -15, -15, -5, 15, -15, -15, -15/
      data basxm / 35, -35, 35, -35, -30, -30/
      data basym / -60, -60, 60, 60, -53, 37/
      data baszm / 15, 15, -5, -5, -15, -15/
      data yoffset / 0.0, 5.0, 32.0/
      data zoffset / -25.0, 0.0, 0.0/
c
c    erase the old bits and pieces if seen last time through
c
      call gpr_$set_draw_value (icol(0), status)
      if (last(1)) then
        call gpr_$move (base_xp(1), base_yp(1), status)
        call gpr_$polyline (base_xp, base_yp, int2(10), status)
        call gpr_$multiline (base_xm, base_ym, int2(6), status)
      end if
      if (last(2)) then
        call gpr_$move (turt_xp(1), turt_yp(1), status)
        call gpr_$polyline (turt_xp, turt_yp, int2(12), status)
      end if
      if (last(3)) then
        call gpr_$multiline (musl_xm, musl_ym, int2(8), status)
        call gpr_$move (musl_xp(4), musl_yp(4), status)
        call gpr_$polyline (musl_xp, musl_yp, int2(4), status)
      end if
      if (count .eq. 40) return
c
c    if new then get the random velocity vectors and starting sines
c    and cosines. Note that subscripts are as follows
c
c    1. base
c    2. turret
c    3. muzzle
c
      if (new) then
        new = .false.
        cta(1) = cos(oazm(1))
        sta(1) = sin(oazm(1))
        ctp(1) = 1.0
        stp(1) = 0.0
        do 10 i = 2, 3
          cta(i) = cta(1)
          sta(i) = sta(1)
          ctp(i) = 1.0
          stp(i) = 0.0
 10     continue
        do 20 i = 1, 3
          xpos(i) = -yoffset(i) * sta(1)
          ypos(i) =  yoffset(i) * cta(1)
          zpos(i) =  zoffset(i)
 20     continue
        do 30 i = 1, 3
          call rand (seed)
          xvel(i) = seed * 10.0
          call rand (seed)
          if (seed .ge. 0.5) xvel(i) = - xvel(i)
          call rand (seed)
          yvel(i) = seed * 10.0
          call rand (seed)
          if (seed .ge. 0.5) yvel(i) = - yvel(i)
          call rand (seed)
          zvel(i) = 12.5 + seed * 7.5
 30     continue
      end if
c
c    project all the pieces into player-centric space
c
      do 40 i = 1, 3
        dx = obx(1) + xpos(i) - xcrd
        dy = oby(1) + ypos(i) - ycrd
        range(i) = sqrt(dx**2 + dy**2)
        if (range(i) .lt. 2000.0) then
          pro_y(i) = -dx * sa + dy * ca
          pro_x(i) =  dx * ca + dy * sa
          check(i) =  pro_y(i) / (abs(pro_x(i)) + 1.0)
          if (check(i) .gt. threshold) then
            draw(i) = .true.
          else
            draw(i) = .false.
          end if
        else
          draw(i) = .false.
        end if
 40   continue
c
c    draw the pieces that can been seen, base first
c
      if (draw(1)) then
        dx = xcrd - obx(1) - xpos(1)
        dy = ycrd - oby(1) - ypos(1)
        dz = - zpos(1)
        xn =  dx * cta(1)          + dy * sta(1)
        yn = -dx * sta(1) * ctp(1) + dy * cta(1) * ctp(1) + dz * stp(1)
        zn =  dx * sta(1) * stp(1) - dy * cta(1) * stp(1) + dz * ctp(1)
        cn = ca * cta(1) + sa * sta(1)
        sn = sa * cta(1) - ca * sta(1)
        cp = ctp(1)
        sp =-stp(1)
        do 50 i = 1, 6
          rx =  basxm(i) - xn
          ry =  basym(i) - yn
          rz =  baszm(i) - zn
          prx =  rx * cn + ry * sn * cp + rz * sn * sp
          pry = -rx * sn + ry * cn * cp + rz * cn * sp
          prz =          - ry * sp      + rz * cp
          if (pry .lt. 10.0) pry = 10.0
          base_xm(i) = 500 + prx / pry * 450
          base_ym(i) = 260 - prz / pry * 450
 50     continue
        do 60 i = 1, 10
          rx =  basxp(i) - xn
          ry =  basyp(i) - yn
          rz =  baszp(i) - zn
          prx =  rx * cn + ry * sn * cp + rz * sn * sp
          pry = -rx * sn + ry * cn * cp + rz * cn * sp
          prz =          - ry * sp      + rz * cp
          if (pry .lt. 10.0) pry = 10.0
          base_xp(i) = 500 + prx / pry * 450
          base_yp(i) = 260 - prz / pry * 450
 60     continue
        last(1) = .true.
        color = range(1) / 300 + 2
        if (color .gt. 7) color = 7
        call gpr_$set_draw_value (icol(color), status)
        call gpr_$move (base_xp(1), base_yp(1), status)
        call gpr_$polyline (base_xp, base_yp, int2(10), status)
        call gpr_$multiline (base_xm, base_ym, int2(6), status)
      else
        last(1) = .false.
      end if
c
c    do the turret
c
      if (draw(2)) then
        dx = xcrd - obx(1) - xpos(2)
        dy = ycrd - oby(1) - ypos(2)
        dz = - zpos(2)
        xn =  dx * cta(2)          + dy * sta(2)
        yn = -dx * sta(2) * ctp(2) + dy * cta(2) * ctp(2) + dz * stp(2)
        zn =  dx * sta(2) * stp(2) - dy * cta(2) * stp(2) + dz * ctp(2)
        cn = ca * cta(2) + sa * sta(2)
        sn = sa * cta(2) - ca * sta(2)
        cp = ctp(2)
        sp =-stp(2)
        do 70 i = 1, 12
          rx =  turxp(i) - xn
          ry =  turyp(i) - yn
          rz =  turzp(i) - zn
          prx =  rx * cn + ry * sn * cp + rz * sn * sp
          pry = -rx * sn + ry * cn * cp + rz * cn * sp
          prz =          - ry * sp      + rz * cp
          if (pry .lt. 10.0) pry = 10.0
          turt_xp(i) = 500 + prx / pry * 450
          turt_yp(i) = 260 - prz / pry * 450
 70     continue
        last(2) = .true.
        color = range(2) / 300 + 2
        if (color .gt. 7) color = 7
        call gpr_$set_draw_value (icol(color), status)
        call gpr_$move (turt_xp(1), turt_yp(1), status)
        call gpr_$polyline (turt_xp, turt_yp, int2(12), status)
      else
        last(2) = .false.
      end if
c
c    do the muzzle
c
      if (draw(3)) then
        dx = xcrd - obx(1) - xpos(3)
        dy = ycrd - oby(1) - ypos(3)
        dz = - zpos(3)
        xn =  dx * cta(3)          + dy * sta(3)
        yn = -dx * sta(3) * ctp(3) + dy * cta(3) * ctp(3) + dz * stp(3)
        zn =  dx * sta(3) * stp(3) - dy * cta(3) * stp(3) + dz * ctp(3)
        cn = ca * cta(3) + sa * sta(3)
        sn = sa * cta(3) - ca * sta(3)
        cp = ctp(3)
        sp =-stp(3)
        do 80 i = 1, 8
          rx =  musxm(i) - xn
          ry =  musym(i) - yn
          rz =  muszm(i) - zn
          prx =  rx * cn + ry * sn * cp + rz * sn * sp
          pry = -rx * sn + ry * cn * cp + rz * cn * sp
          prz =          - ry * sp      + rz * cp
          if (pry .lt. 10.0) pry = 10.0
          musl_xm(i) = 500 + prx / pry * 450
          musl_ym(i) = 260 - prz / pry * 450
 80     continue
        do 90 i = 1, 4
          rx =  musxp(i) - xn
          ry =  musyp(i) - yn
          rz =  muszp(i) - zn
          prx =  rx * cn + ry * sn * cp + rz * sn * sp
          pry = -rx * sn + ry * cn * cp + rz * cn * sp
          prz =          - ry * sp      + rz * cp
          if (pry .lt. 10.0) pry = 10.0
          musl_xp(i) = 500 + prx / pry * 450
          musl_yp(i) = 260 - prz / pry * 450
 90     continue
        last(3) = .true.
        color = range(3) / 300 + 2
        if (color .gt. 7) color = 7
        call gpr_$set_draw_value (icol(color), status)
        call gpr_$multiline (musl_xm, musl_ym, int2(8), status)
        call gpr_$move (musl_xp(4), musl_yp(4), status)
        call gpr_$polyline (musl_xp, musl_yp, int2(4), status)
      else
        last(3) = .false.
      end if
c
c    update the vector offsets and velocities and rotation angles
c
      do 100 i = 1, 3
        xpos(i) = xpos(i) + xvel(i)
        ypos(i) = ypos(i) + yvel(i)
        zvel(i) = zvel(i) - gravity
        zpos(i) = zpos(i) + zvel(i)
        if (zpos(i) .lt. -40.0) then
          zpos(i) = -40.0
          zvel(i) = -0.2 * zvel(i)
        end if
        temp = cta(i)
        cta(i) = cta(i) * cda(i) - sta(i) * sda(i)
        sta(i) = sta(i) * cda(i) + sda(i) * temp
        temp = ctp(i)
        ctp(i) = ctp(i) * cdp(i) - stp(i) * sdp(i)
        stp(i) = stp(i) * cdp(i) + sdp(i) * temp
100   continue
      return
      end





      subroutine bzone_message (number, bell)
c
c     BZONE_MESSAGE prints a textual message on the bottom of the
c     screen.  Used to alert the user to new enemy ships, path blocked
c     etc.  Note this routine also can beep the system bell.
c
% include '/sys/ins/base.ins.ftn'
% include '/sys/ins/gpr.ins.ftn'
c
      integer*2 xpt, ypt(4), clock(3)
      integer*4 status
      logical bell
      character*40 messages(4), text
      save xpt, ypt, clock, messages
      data xpt, ypt /70, 507, 567, 627, 627/
      data clock /0, 0, 7500/
      data messages /'ENEMY TANK IN FIRING ARC',
     &               'MOVEMENT BLOCKED BY OBJECT',
     &               'SALVO FIRED BY ENEMY TANK',
     &               'SALVO FIRED BY ENEMY COPTER'/
      index = number
c
c    write or erase the specified message
c
      if (index .lt. 0) then
        text = ' '
      else
        text = messages(index)
      end if
      index = iabs(index)
      call gpr_$set_clipping_active (.false., status)
      call gpr_$move (xpt, ypt(index), status)
      call gpr_$text (text, int2(40), status)
      call gpr_$set_clipping_active (.true., status)
      if (bell) then
        call tone_$time (clock)
      end if
      return
      end



      subroutine bzone_pause
c
c     BZONE_PAUSE causes the game to pause until a key is struck.
c
% include '/sys/ins/gpr.ins.ftn'
% include '/sys/ins/kbd.ins.ftn'
c
      integer*2 rect_1(2, 2), rect_2(2, 2), rect_3(2, 2), size(2)
      integer*2 xpt, ypt, cpos(2), event_type
      integer*4 status, icol(0:15)
      character key*1, text(3)*80
      logical in_color, event
      save rect_1, rect_2, rect_3, text
      common /color$/ icol, in_color
      data rect_1 / 426, 216, 148, 69/
      data rect_2 / 428, 218, 144, 65/
      data rect_3 / 430, 220, 140, 61/
      data text / 'Game Suspended', 'Hit Any Key', 'To Resume Play'/
c
c    write the instruction box
c
      call gpr_$set_fill_value (icol(0), status)
      call gpr_$rectangle (rect_1, status)
      call gpr_$set_fill_value (icol(1), status)
      call gpr_$rectangle (rect_2, status)
      call gpr_$set_fill_value (icol(0), status)
      call gpr_$rectangle (rect_3, status)
      call gpr_$set_text_value (icol(2), status)
      call gpr_$inq_text_extent (text(1), int2(14), size, status)
      xpt = 500 - size(1) / 2
      ypt = 225 + 15
      call gpr_$move (xpt, ypt, status)
      call gpr_$text (text(1), int2(14), status)
      call gpr_$inq_text_extent (text(2), int2(11), size, status)
      xpt = 500 - size(1) / 2
      ypt = 225 + 31
      call gpr_$move (xpt, ypt, status)
      call gpr_$text (text(2), int2(11), status)
      call gpr_$inq_text_extent (text(3), int2(14), size, status)
      xpt = 500 - size(1) / 2
      ypt = 225 + 47
      call gpr_$move (xpt, ypt, status)
      call gpr_$text (text(3), int2(14), status)
c
c    wait for a key
c
 10   continue
      event = gpr_$event_wait (event_type, key, cpos, status)
      if (event_type .ne. gpr_$keystroke) goto 10
      if (key .eq. kbd_$f1u .or. key .eq. kbd_$r6u) goto 10
c
c    erase the message box and resume
c
      call gpr_$set_fill_value (icol(0), status)
      call gpr_$rectangle (rect_1, status)
      call gpr_$set_text_value (icol(1), status)
      return
      end
