#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* OpenBSD pledge(2) */
#ifdef __OpenBSD__
#include <unistd.h>
#else
#define pledge(p1,p2) 0
#endif

/* This is the blocksize of my disk, use atleast an equal or higher value and
  a multiple of 2 for better performance ((struct stat).st_blksize). */
#define READ_BUF_SIZ 16384
#define LEN(s)     (sizeof(s)/sizeof(*s))

struct variable {
	char *key, *value;
	struct variable *next;
};

#include "arg.h"
char *argv0;

#include "config.h"

static struct variable *global;

/* Escape characters below as HTML 2.0 / XML 1.0. */
void
xmlencode(const char *s, FILE *fp)
{
	for (; *s; s++) {
		switch(*s) {
		case '<':  fputs("&lt;",   fp); break;
		case '>':  fputs("&gt;",   fp); break;
		case '\'': fputs("&#39;",  fp); break;
		case '&':  fputs("&amp;",  fp); break;
		case '"':  fputs("&quot;", fp); break;
		default:   fputc(*s, fp);
		}
	}
}

char *
estrdup(const char *s)
{
	char *p;

	if (!(p = strdup(s))) {
		fprintf(stderr, "strdup: %s\n", strerror(errno));
		exit(1);
	}
	return p;
}

void *
ecalloc(size_t nmemb, size_t size)
{
	void *p;

	if (!(p = calloc(nmemb, size))) {
		fprintf(stderr, "calloc: %s\n", strerror(errno));
		exit(1);
	}
	return p;
}

FILE *
efopen(const char *path, const char *mode)
{
	FILE *fp;

	if (!(fp = fopen(path, mode))) {
		fprintf(stderr, "fopen: %s, mode: %s: %s\n",
			path, mode, strerror(errno));
		exit(1);
	}
	return fp;
}

void
catfile(FILE *fpin, const char *ifile, FILE *fpout, const char *ofile)
{
	char buf[READ_BUF_SIZ];
	size_t r, w;

	while (!feof(fpin)) {
		if (!(r = fread(buf, 1, sizeof(buf), fpin)))
			break;
		if ((w = fwrite(buf, 1, r, fpout)) != r)
			break;
		if (r != sizeof(buf))
			break;
	}
	if (ferror(fpin)) {
		fprintf(stderr, "%s -> %s: error reading data from stream: %s\n",
		        ifile, ofile, strerror(errno));
		exit(1);
	}
	if (ferror(fpout)) {
		fprintf(stderr, "%s -> %s: error writing data to stream: %s\n",
		        ifile, ofile, strerror(errno));
		exit(1);
	}
}

char *
readfile(const char *file)
{
	FILE *fp;
	char *buf;
	size_t n, len = 0, size = 0;

	fp = efopen(file, "rb");
	buf = ecalloc(1, size + 1); /* always allocate an empty buffer */
	while (!feof(fp)) {
		if (len + READ_BUF_SIZ + 1 > size) {
			/* allocate size: common case is small textfiles */
			size += READ_BUF_SIZ;
			if (!(buf = realloc(buf, size + 1))) {
				fprintf(stderr, "realloc: %s\n", strerror(errno));
				exit(1);
			}
		}
		if (!(n = fread(&buf[len], 1, READ_BUF_SIZ, fp)))
			break;
		len += n;
		buf[len] = '\0';
		if (n != READ_BUF_SIZ)
			break;
	}
	if (ferror(fp)) {
		fprintf(stderr, "fread: file: %s: %s\n", file, strerror(errno));
		exit(1);
	}
	fclose(fp);

	return buf;
}

struct variable *
newvar(const char *key, const char *value)
{
	struct variable *v;

	v = ecalloc(1, sizeof(*v));
	v->key = estrdup(key);
	v->value = estrdup(value);

	return v;
}

/* uses var->key as key */
void
setvar(struct variable **vars, struct variable *var, int override)
{
	struct variable *p, *v;

	/* new */
	if (!*vars) {
		*vars = var;
		return;
	}

	/* search: set or append */
	for (p = NULL, v = *vars; v; v = v->next) {
		if (!strcmp(var->key, v->key)) {
			if (!override)
				return;
			/* NOTE: keep v->next */
			var->next = v->next;
			if (p)
				p->next = var;
			else
				*vars = var;
			free(v->key);
			free(v->value);
			free(v);
			return;
		}
		/* append */
		if (!v->next) {
			var->next = NULL;
			v->next = var;
			return;
		}
		p = v;
	}
}

struct variable *
getvar(struct variable *vars, char *key)
{
	struct variable *v;

	for (v = vars; v; v = v->next)
		if (!strcmp(key, v->key))
			return v;
	return NULL;
}

void
freevars(struct variable *vars)
{
	struct variable *v, *tmp;

	for (v = vars; v; ) {
		tmp = v->next;
		free(v->key);
		free(v->value);
		free(v);
		v = tmp;
	}
}

struct variable *
parsevars(const char *file, const char *s)
{
	struct variable *vars = NULL, *v;
	const char *keystart, *keyend, *valuestart, *valueend;
	size_t line = 0;

	for (; *s; ) {
		if (*s == '\r' || *s == '\n') {
			line += (*s == '\n');
			s++;
			continue;
		}

		/* comment start with #, skip to newline */
		if (*s == '#') {
			s++;
			s = &s[strcspn(s, "\n")];
			continue;
		}

		/* trim whitespace before key */
		s = &s[strspn(s, " \t")];

		keystart = s;
		s = &s[strcspn(s, "=\r\n")];
		if (*s != '=') {
			fprintf(stderr, "%s:%zu: error: no variable\n",
			        file, line + 1);
			exit(1);
		}

		/* trim whitespace at end of key: but whitespace in names are
		   allowed */
		for (keyend = s++; keyend > keystart &&
		                 (keyend[-1] == ' ' || keyend[-1] == '\t');
		     keyend--)
			;
		/* no variable name: skip */
		if (keystart == keyend) {
			fprintf(stderr, "%s:%zu: error: invalid variable\n",
			        file, line + 1);
			exit(1);
		}

		valuestart = s;
		s = &s[strcspn(s, "\r\n")];
		valueend = s;

		v = ecalloc(1, sizeof(*v));
		v->key = ecalloc(1, keyend - keystart + 1);
		memcpy(v->key, keystart, keyend - keystart);
		v->value = ecalloc(1, valueend - valuestart + 1);
		memcpy(v->value, valuestart, valueend - valuestart);

		setvar(&vars, v, 1);
	}
	return vars;
}

struct variable *
readconfig(const char *file)
{
	struct variable *c;
	char *data;

	data = readfile(file);
	c = parsevars(file, data);
	free(data);

	return c;
}

void
writepage(FILE *fp, const char *file, struct variable *c, char *s)
{
	FILE *fpin;
	struct variable *v;
	char *key;
	size_t keylen, line = 0;
	int op, tmpc;

	for (; *s; s++) {
		op = *s;
		switch (*s) {
		case '#': /* insert value non-escaped */
		case '$': /* insert value escaped */
		case '%': /* insert contents of filename set in variable */
			if (*(s + 1) == '{') {
				s += 2;
				break;
			}
			fputc(*s, fp);
			continue;
		case '\n':
			line++; /* FALLTHROUGH */
		default:
			fputc(*s, fp);
			continue;
		}

		/* variable case */
		for (; *s && isspace((unsigned char)*s); s++)
			;
		key = s;
		for (keylen = 0; *s && *s != '}'; s++)
			keylen++;
		/* trim right whitespace */
		for (; keylen && isspace((unsigned char)key[keylen - 1]); )
			keylen--;

		/* temporary NUL terminate */
		tmpc = key[keylen];
		key[keylen] = '\0';

		if (!c || !(v = getvar(c, key)))
			v = getvar(global, key);
		key[keylen] = tmpc; /* restore NUL terminator to original */

		if (!v) {
			fprintf(stderr, "%s:%zu: error: undefined variable: '%.*s'\n",
			        file, line + 1, (int)keylen, key);
			exit(1);
		}

		switch (op) {
		case '#':
			fputs(v->value, fp);
			break;
		case '$':
			xmlencode(v->value, fp);
			break;
		case '%':
			if (!v->value[0])
				break;
			fpin = efopen(v->value, "rb");
			catfile(fpin, v->value, fp, file);
			fclose(fpin);
			break;
		}
	}
}

void
usage(void)
{
	fprintf(stderr, "%s [-c config] [-o outputdir] [-t templatedir] "
	                "pages...\n", argv0);
	exit(1);
}

int
main(int argc, char *argv[])
{
	struct block *b;
	struct variable *c, *v;
	char file[PATH_MAX + 1], htmlfile[PATH_MAX + 1];
	char outputfile[PATH_MAX + 1], *p;
	size_t i, j, k;
	int r;

	if (pledge("stdio cpath rpath wpath", NULL) == -1) {
		fprintf(stderr, "pledge: %s\n", strerror(errno));
		return 1;
	}

	ARGBEGIN {
	case 'c':
		configfile = EARGF(usage());
		break;
	case 'o':
		outputdir = EARGF(usage());
		break;
	case 't':
		templatedir = EARGF(usage());
		break;
	default:
		usage();
	} ARGEND;

	/* global config */
	global = readconfig(configfile);

	/* load templates: all templates must be loaded correctly first. */
	for (i = 0; i < LEN(templates); i++) {
		for (j = 0; j < LEN(templates[i].blocks); j++) {
			b = &templates[i].blocks[j];
			r = snprintf(file, sizeof(file), "%s/%s", templatedir,
			             b->name);
			if (r < 0 || (size_t)r >= sizeof(file)) {
				fprintf(stderr, "path truncated: '%s/%s'\n",
				        templatedir, b->name);
				exit(1);
			}
			b->data = readfile(file);
		}
	}

	/* header */
	for (i = 0; i < LEN(templates); i++) {
		if (!strcmp(templates[i].name, "page"))
			continue;
		r = snprintf(file, sizeof(file), "%s/%s", outputdir,
		             templates[i].name);
		if (r < 0 || (size_t)r >= sizeof(file)) {
			fprintf(stderr, "path truncated: '%s/%s'\n", outputdir,
			        templates[i].name);
			exit(1);
		}
		templates[i].fp = efopen(file, "wb");
		b = &templates[i].blocks[BlockHeader];
		writepage(templates[i].fp, b->name, NULL, b->data);
	}

	/* pages */
	for (i = 0; i < (size_t)argc; i++) {
		c = readconfig(argv[i]);

		if ((p = strrchr(argv[i], '.')))
			r = snprintf(htmlfile, sizeof(htmlfile), "%.*s.html",
			             (int)(p - argv[i]), argv[i]);
		else
			r = snprintf(htmlfile, sizeof(htmlfile), "%s.html", argv[i]);
		if (r < 0 || (size_t)r >= sizeof(htmlfile)) {
			fprintf(stderr, "path truncated for file: '%s'\n", argv[i]);
			exit(1);
		}
		/* set htmlfile, but allow to override it */
		setvar(&c, newvar("htmlfile", htmlfile), 0);

		/* set HTML output filename (with part removed), but allow to
		   override it */
		if ((p = strrchr(htmlfile, '/')))
			setvar(&c, newvar("filename", &htmlfile[p - htmlfile + 1]), 0);
		else
			setvar(&c, newvar("filename", htmlfile), 0);

		/* get output filename */
		v = getvar(c, "filename");

		/* item block */
		for (j = 0; j < LEN(templates); j++) {
			/* TODO: page is a special case for now */
			if (!strcmp(templates[j].name, "page")) {
				r = snprintf(outputfile, sizeof(outputfile), "%s/%s",
				             outputdir, v->value);
				if (r < 0 || (size_t)r >= sizeof(outputfile)) {
					fprintf(stderr, "path truncated: '%s/%s'\n",
					        outputdir, v->value);
					exit(1);
				}

				templates[j].fp = efopen(outputfile, "wb");
				for (k = 0; k < LEN(templates[j].blocks); k++) {
					b = &templates[j].blocks[k];
					writepage(templates[j].fp, b->name, c, b->data);
				}
				fclose(templates[j].fp);
			} else {
				v = getvar(c, "index");
				if (v && v->value[0] == '0')
					continue;
				b = &templates[j].blocks[BlockItem];
				writepage(templates[j].fp, b->name, c, b->data);
			}
		}
		freevars(c);
	}

	for (i = 0; i < LEN(templates); i++) {
		if (!strcmp(templates[i].name, "page"))
			continue;
		b = &templates[i].blocks[BlockFooter];
		writepage(templates[i].fp, b->name, NULL, b->data);
	}

	return 0;
}
