#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef __OpenBSD__
#include <unistd.h>
#else
#define pledge(a,b) 0
#endif

#include "json.h"

static int nflag = 0; /* -n flag: show indices count for arrays */
static int rflag = 0; /* -r flag: show all control-characters */
static int fs = '\t', rs = '\n';

static void (*printvalue)(const char *);

void
tsv_printvalue(const char *s)
{
	for (; *s; s++) {
		/* escape some chars */
		switch (*s) {
		case '\n': putchar('\\'); putchar('n'); break;
		case '\\': putchar('\\'); putchar('\\'); break;
		case '\t': putchar('\\'); putchar('t'); break;
		default:
			/* ignore other control chars */
			if (!rflag && iscntrl((unsigned char)*s))
				continue;
			putchar(*s);
		}
	}
}

void
rs_printvalue(const char *s)
{
	for (; *s; s++) {
		if (*s == fs || *s == rs)
			continue;

		switch (*s) {
		case '\n':
		case '\t':
			putchar(*s);
			break;
		default:
			/* ignore other control chars */
			if (!rflag && iscntrl((unsigned char)*s))
				continue;
			putchar(*s);
		}
	}
}

/* optimized printing an unsigned number (compared to printf("%zu")) */
void
printnum(uintmax_t x)
{
	char buf[64], *s, *e;
	unsigned long y;

	if (!x) {
		putchar('0');
		return;
	}

	s = e = buf + sizeof(buf) - 1;

	for (; x > ULONG_MAX; x /= 10)
		*--s = '0' + x % 10;
	for (y = x; y; y /= 10)
		*--s = '0' + y % 10;

	for (; s < e; s++)
		putchar(*s);
}

void
processnode(struct json_node *nodes, size_t depth, const char *value)
{
	size_t i;

	for (i = 0; i < depth; i++) {
		printvalue(nodes[i].name);

		if (i + 1 == depth &&
		    (nodes[i].type == JSON_TYPE_OBJECT ||
		     nodes[i].type == JSON_TYPE_ARRAY))
			continue;

		if (nodes[i].type == JSON_TYPE_OBJECT) {
			putchar('.');
		} else if (nodes[i].type == JSON_TYPE_ARRAY) {
			putchar('[');
			if (nflag)
				printnum(nodes[i].index);
			putchar(']');
		}
	}

	putchar(fs);
	putchar(nodes[depth - 1].type);
	putchar(fs);
	printvalue(value);
	putchar(rs);
}

int
readnum(const char *s, int base)
{
	long l;
	char *end;

	errno = 0;
	l = strtol(s, &end, base);
	if (errno || s == end || *end != '\0' || l < 0 || l > 255) {
		fprintf(stderr, "invalid number\n");
		exit(3);
	}

	return (int)l;
}

int
readchar(const char *s)
{
	if (!*s) {
		fprintf(stderr, "invalid character\n");
		exit(3);
	} else if (strlen(s) == 1) {
		return *s;
	} else if (*s == '\\') {
		s++;
		switch (*s) {
		case '\\': return '\\';
		case 't': return '\t';
		case 'n': return '\n';
		case 'r': return '\r';
		case 'x': return readnum(++s, 16); /* hexadecimal */
		default:
			fprintf(stderr, "unsupported escape character\n");
			exit(3);
		}
	}
	/* base 0 (decimal, octal, hex) using strtol() format */
	return readnum(s, 0);
}

void
usage(const char *argv0)
{
	fprintf(stderr, "usage: %s [-n] [-r] [-F fs] [-R rs]\n", argv0);
	exit(3);
}

int
main(int argc, char *argv[])
{
	int i, j;

	if (pledge("stdio", NULL) == -1) {
		fprintf(stderr, "pledge stdio: %s\n", strerror(errno));
		return 1;
	}

	printvalue = tsv_printvalue;
	for (i = 1; i < argc; i++) {
		for (j = 1; i < argc && argv[i][j]; j++) {
			switch (argv[i][j]) {
			case 'n':
				nflag = 1;
				break;
			case 'r':
				rflag = 1;
				break;
			case 'F':
				if (i + 1 >= argc)
					usage(argv[0]);
				fs = readchar(argv[++i]);
				printvalue = rs_printvalue;
				goto nextarg;
			case 'R':
				if (i + 1 >= argc)
					usage(argv[0]);
				rs = readchar(argv[++i]);
				printvalue = rs_printvalue;
				goto nextarg;
			default:
				usage(argv[0]);
				break;
			}
		}
nextarg:;
	}

	switch (parsejson(processnode)) {
	case JSON_ERROR_MEM:
		fputs("error: cannot allocate enough memory\n", stderr);
		return 2;
	case JSON_ERROR_INVALID:
		fputs("error: invalid JSON\n", stderr);
		return 1;
	}

	return 0;
}
