#include <sys/socket.h>
#include <sys/time.h>

#include <ctype.h>
#include <err.h>
#include <errno.h>
#include <netdb.h>
#include <locale.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include <tls.h>

#include "arg.h"

#define READ_BUF_SIZ	16384

#ifndef __OpenBSD__
#define pledge(p1,p2) 0
#define unveil(p1,p2) 0
#endif

#ifndef TLS_CA_CERT_FILE
#define TLS_CA_CERT_FILE "/etc/ssl/cert.pem"
#endif

/* uri */
struct uri {
	char proto[48];
	char host[256];
	char path[2048];
	char port[6];     /* numeric port */
};

char *argv0;

/* raw header(s) to add */
static const char *config_headers = "";
/* max response size in bytes, 0 is unlimited */
static size_t config_maxresponsesiz = 0;
/* time-out in seconds */
static time_t config_timeout = 0;
/* legacy ciphers? */
static int config_legacy = 0;
/* parsed uri */
static struct uri u;
/* raw command-line argument */
static char *url;
/* TLS config */
static struct tls_config *tls_config;

void
sighandler(int signo)
{
	if (signo == SIGALRM)
		 _exit(2);
}

int
parseuri(const char *s, struct uri *u)
{
	const char *p = s, *b;
	char *endptr = NULL;
	size_t i;
	unsigned long l;

	u->proto[0] = u->host[0] = u->path[0] = u->port[0] = '\0';
	if (!*p)
		return 0;

	/* protocol part */
	for (p = s; *p && (isalpha((unsigned char)*p) || isdigit((unsigned char)*p) ||
		       *p == '+' || *p == '-' || *p == '.'); p++)
		;
	if (!strncmp(p, "://", 3)) {
		if ((size_t)(p - s) >= sizeof(u->proto))
			return -1; /* protocol too long */
		memcpy(u->proto, s, p - s);
		u->proto[p - s] = '\0';
		p += 3; /* skip "://" */
	} else {
		return -1; /* no protocol specified */
	}

	/* IPv6 address */
	if (*p == '[') {
		/* bracket not found or host too long */
		if (!(b = strchr(p, ']')) || (size_t)(b - p) >= (ssize_t)sizeof(u->host))
			return -1;
		memcpy(u->host, p + 1, b - p - 1);
		u->host[b - p - 1] = '\0';
		p = b + 1;
	} else {
		/* domain / host part, skip until port, path or end. */
		if ((i = strcspn(p, ":/")) >= sizeof(u->host))
			return -1; /* host too long */
		memcpy(u->host, p, i);
		u->host[i] = '\0';
		p = &p[i];
	}
	/* port */
	if (*p == ':') {
		if ((i = strcspn(++p, "/")) >= sizeof(u->port))
			return -1; /* port too long */
		memcpy(u->port, p, i);
		u->port[i] = '\0';
		/* check for valid port: range 1 - 65535 */
		errno = 0;
		l = strtoul(u->port, &endptr, 10);
		if (errno || u->port[0] == '\0' || *endptr ||
		    !l || l > 65535)
			return -1;
		p = &p[i];
	}
	if (u->host[0]) {
		p = &p[strspn(p, "/")];
		memcpy(u->path, "/", 2);
	} else {
		return -1;
	}
	/* treat truncation as an error */
	if (strlcat(u->path, p, sizeof(u->path)) >= sizeof(u->path))
		return -1;
	return 0;
}

int
edial(const char *host, const char *port)
{
	struct addrinfo hints, *res, *res0;
	int error, save_errno, s;
	const char *cause = NULL;
	struct timeval timeout;

	memset(&hints, 0, sizeof(hints));
	hints.ai_family = AF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_flags = AI_NUMERICSERV; /* numeric port only */
	if ((error = getaddrinfo(host, port, &hints, &res0)))
		errx(1, "%s: %s: %s:%s", __func__, gai_strerror(error), host, port);
	s = -1;
	for (res = res0; res; res = res->ai_next) {
		s = socket(res->ai_family, res->ai_socktype,
		           res->ai_protocol);
		if (s == -1) {
			cause = "socket";
			continue;
		}

		if (connect(s, res->ai_addr, res->ai_addrlen) == -1) {
			cause = "connect";
			save_errno = errno;
			close(s);
			errno = save_errno;
			s = -1;
			continue;
		}
		break;
	}
	if (s == -1)
		errx(1, "%s: %s: %s:%s", __func__, cause, host, port);
	freeaddrinfo(res0);

	return s;
}

int
https_request(void)
{
	struct tls *t;
	char buf[READ_BUF_SIZ], *p;
	const char *errstr;
	size_t n, len;
	ssize_t r;
	int fd = -1, httpok = 0, ret = 1, stdport;

	if (pledge("stdio dns inet rpath unveil", NULL) == -1)
		err(1, "pledge");

	if (unveil(TLS_CA_CERT_FILE, "r") == -1)
		err(1, "unveil: %s", TLS_CA_CERT_FILE);
	if (unveil(NULL, NULL) == -1)
		err(1, "unveil");

	if (!(t = tls_client())) {
		fprintf(stderr, "tls_client: %s\n", tls_error(t));
		goto err;
	}
	if (tls_configure(t, tls_config)) {
		fprintf(stderr, "tls_configure: %s\n", tls_error(t));
		goto err;
	}

	fd = edial(u.host, u.port);
	if (tls_connect_socket(t, fd, u.host) == -1)
		errx(1, "tls_connect: %s", tls_error(t));

	if (pledge("stdio", NULL) == -1)
		err(1, "pledge");

	stdport = u.port[0] == '\0' || strcmp(u.port, "443") == 0;

	/* create and send HTTP header */
	r = snprintf(buf, sizeof(buf),
		"GET %s HTTP/1.0\r\n"
		"Host: %s%s%s\r\n"
		"Connection: close\r\n"
		"%s%s"
		"\r\n", u.path, u.host,
		stdport ? "" : ":",
		stdport ? "" : u.port,
		config_headers, config_headers[0] ? "\r\n" : "");
	if (r < 0 || (size_t)r >= sizeof(buf)) {
		fprintf(stderr, "not writing header because it is truncated");
		goto err;
	}

	for (len = r, p = buf; len > 0; ) {
		r = tls_write(t, p, len);
		if (r == TLS_WANT_POLLIN || r == TLS_WANT_POLLOUT) {
			continue;
		} else if (r == -1) {
			fprintf(stderr, "tls_write: %s\n", tls_error(t));
			goto err;
		}
		p += r;
		len -= r;
	}

	/* NOTE: HTTP header must fit in the buffer */
	for (len = 0; len < sizeof(buf);) {
		/* NOTE: buffer size is -1 to NUL terminate the buffer for a
		         string comparison. */
		r = tls_read(t, &buf[len], sizeof(buf) - len - 1);
		if (r == TLS_WANT_POLLIN || r == TLS_WANT_POLLOUT) {
			continue;
		} else if (r == 0) {
			break;
		} else if (r == -1) {
			errstr = tls_error(t);
			fprintf(stderr, "tls_read: %s\n", errstr ? errstr : "");
			goto err;
		}
		len += r;
	}
	buf[len] = '\0';

	if (!strncmp(buf, "HTTP/1.0 200 ", sizeof("HTTP/1.0 200 ") - 1) ||
	    !strncmp(buf, "HTTP/1.1 200 ", sizeof("HTTP/1.1 200 ") - 1))
		httpok = 1;

	if (!(p = strstr(buf, "\r\n\r\n"))) {
		fprintf(stderr, "no HTTP header found or header too big\n");
		goto err;
	}
	*p = '\0'; /* NUL terminate header part */
	p += strlen("\r\n\r\n");

	if (httpok) {
		n = len - (p - buf);
		r = fwrite(p, 1, n, stdout);
		if (ferror(stdout)) {
			fprintf(stderr, "fwrite: stdout: %s\n", strerror(errno));
			goto err;
		}
	} else {
		/* if not 200 OK print header */
		fputs(buf, stderr);
		fputs("\r\n\r\n", stderr);
		/* NOTE: we are nice and keep reading (not closing) until the server is done. */
	}

	while (1) {
		r = tls_read(t, &buf, sizeof(buf));
		if (r == TLS_WANT_POLLIN || r == TLS_WANT_POLLOUT) {
			continue;
		} else if (r == 0) {
			break;
		} else if (r == -1) {
			errstr = tls_error(t);
			fprintf(stderr, "tls_read: %s\n", errstr ? errstr : "");
			goto err;
		}
		len += r;

		if (httpok) {
			r = fwrite(buf, 1, r, stdout);
			if (ferror(stdout)) {
				fprintf(stderr, "fwrite: stdout: %s\n", strerror(errno));
				goto err;
			}
		}

		if (config_maxresponsesiz && len >= config_maxresponsesiz)
			break;
	}
	if (config_maxresponsesiz && len >= config_maxresponsesiz) {
		fprintf(stderr, "tls_read: response too big: %zu >= %zu\n",
		        len, config_maxresponsesiz);
		goto err;
	}
	ret = 0;

err:
	if (t) {
		tls_close(t);
		tls_free(t);
	}

	return httpok ? ret : 2;
}

int
http_request(void)
{
	char buf[READ_BUF_SIZ], *p;
	size_t n, len;
	ssize_t r;
	int fd = -1, httpok = 0, ret = 1, stdport;

	if (pledge("stdio dns inet", NULL) == -1)
		err(1, "pledge");

	fd = edial(u.host, u.port);

	if (pledge("stdio", NULL) == -1)
		err(1, "pledge");

	stdport = u.port[0] == '\0' || strcmp(u.port, "80") == 0;

	/* create and send HTTP header */
	r = snprintf(buf, sizeof(buf),
		"GET %s HTTP/1.0\r\n"
		"Host: %s%s%s\r\n"
		"Connection: close\r\n"
		"%s%s"
		"\r\n", u.path, u.host,
		stdport ? "" : ":",
		stdport ? "" : u.port,
		config_headers, config_headers[0] ? "\r\n" : "");
	if (r < 0 || (size_t)r >= sizeof(buf)) {
		fprintf(stderr, "not writing header because it is truncated");
		goto err;
	}
	if ((r = write(fd, buf, r)) == -1) {
		fprintf(stderr, "write: %s\n", strerror(errno));
		goto err;
	}

	/* NOTE: HTTP header must fit in the buffer */
	for (len = 0; len < sizeof(buf); len += r) {
		/* NOTE: buffer size is -1 to NUL terminate the buffer for a
		         string comparison. */
		if ((r = read(fd, &buf[len], sizeof(buf) - len - 1)) == 0)
			break;
		if (r == -1) {
			fprintf(stderr, "read: %s\n", strerror(errno));
			goto err;
		}
	}
	buf[len] = '\0';

	if (!strncmp(buf, "HTTP/1.0 200 ", sizeof("HTTP/1.0 200 ") - 1) ||
	    !strncmp(buf, "HTTP/1.1 200 ", sizeof("HTTP/1.1 200 ") - 1))
		httpok = 1;

	if (!(p = strstr(buf, "\r\n\r\n"))) {
		fprintf(stderr, "no HTTP header found or header too big\n");
		goto err;
	}
	*p = '\0'; /* NUL terminate header part */
	p += strlen("\r\n\r\n");

	if (httpok) {
		n = len - (p - buf);
		r = fwrite(p, 1, n, stdout);
		if (ferror(stdout)) {
			fprintf(stderr, "fwrite: stdout: %s\n", strerror(errno));
			goto err;
		}
	} else {
		/* if not 200 OK print header */
		fputs(buf, stderr);
		fputs("\r\n\r\n", stderr);
		/* NOTE: we are nice and keep reading (not closing) until the server is done. */
	}

	while (1) {
		r = read(fd, &buf, sizeof(buf));
		if (r == 0)
			break;
		if (r == -1) {
			fprintf(stderr, "read: %s\n", strerror(errno));
			goto err;
		}
		len += r;

		if (httpok) {
			r = fwrite(buf, 1, r, stdout);
			if (ferror(stdout)) {
				fprintf(stderr, "fwrite: stdout: %s\n", strerror(errno));
				goto err;
			}
		}

		if (config_maxresponsesiz && len >= config_maxresponsesiz)
			break;
	}
	if (config_maxresponsesiz && len >= config_maxresponsesiz) {
		fprintf(stderr, "read: response too big: %zu >= %zu\n",
		        len, config_maxresponsesiz);
		goto err;
	}
	ret = 0;

err:
	if (fd != -1)
		close(fd);
	return httpok ? ret : 2;
}

int
gopher_request(void)
{
	char buf[READ_BUF_SIZ], *p;
	const char *errstr;
	size_t len = 0;
	ssize_t r;
	int fd = -1, ret = 1;

	if (pledge("stdio dns inet", NULL) == -1)
		err(1, "pledge");

	fd = edial(u.host, u.port);

	if (pledge("stdio", NULL) == -1)
		err(1, "pledge");

	/* create and send path, skip type part */
	r = snprintf(buf, sizeof(buf), "%s\r\n", u.path + 2);
	if (r < 0 || (size_t)r >= sizeof(buf)) {
		fprintf(stderr, "not writing header because it is truncated");
		goto err;
	}

	if ((r = write(fd, buf, strlen(buf))) == -1) {
		fprintf(stderr, "write: %s\n", strerror(errno));
		goto err;
	}

	while (1) {
		r = read(fd, &buf, sizeof(buf));
		if (r == 0) {
			break;
		} else if (r == -1) {
			fprintf(stderr, "read: %s\n", strerror(errno));
			goto err;
		}
		len += r;

		r = fwrite(buf, 1, r, stdout);
		if (ferror(stdout)) {
			fprintf(stderr, "fwrite: stdout: %s\n", strerror(errno));
			goto err;
		}

		if (config_maxresponsesiz && len >= config_maxresponsesiz)
			break;
	}
	if (config_maxresponsesiz && len >= config_maxresponsesiz) {
		fprintf(stderr, "read: response too big: %zu >= %zu\n",
		        len, config_maxresponsesiz);
		goto err;
	}
	ret = 0;

err:
	if (fd != -1)
		close(fd);
	return ret;
}

int
gophers_request(void)
{
	struct tls *t;
	char buf[READ_BUF_SIZ], *p;
	const char *errstr;
	size_t len = 0;
	ssize_t r;
	int fd = -1, ret = 1;

	if (pledge("stdio dns inet rpath unveil", NULL) == -1)
		err(1, "pledge");

	if (unveil(TLS_CA_CERT_FILE, "r") == -1)
		err(1, "unveil: %s", TLS_CA_CERT_FILE);
	if (unveil(NULL, NULL) == -1)
		err(1, "unveil");

	if (!(t = tls_client())) {
		errstr = tls_error(t);
		fprintf(stderr, "tls_client: %s\n", errstr ? errstr : "");
		goto err;
	}
	if (tls_configure(t, tls_config)) {
		errstr = tls_error(t);
		fprintf(stderr, "tls_configure: %s\n", errstr ? errstr : "");
		goto err;
	}

	fd = edial(u.host, u.port);
	if (tls_connect_socket(t, fd, u.host) == -1)
		errx(1, "tls_connect: %s", tls_error(t));

	if (pledge("stdio", NULL) == -1)
		err(1, "pledge");

	/* create and send path, skip type part */
	r = snprintf(buf, sizeof(buf), "%s\r\n", u.path + 2);
	if (r < 0 || (size_t)r >= sizeof(buf)) {
		fprintf(stderr, "not writing header because it is truncated");
		goto err;
	}

	for (len = r, p = buf; len > 0; ) {
		r = tls_write(t, p, len);
		if (r == TLS_WANT_POLLIN || r == TLS_WANT_POLLOUT) {
			continue;
		} else if (r == -1) {
			fprintf(stderr, "tls_write: %s\n", tls_error(t));
			goto err;
		}
		p += r;
		len -= r;
	}

	while (1) {
		r = tls_read(t, &buf, sizeof(buf));
		if (r == TLS_WANT_POLLIN || r == TLS_WANT_POLLOUT) {
			continue;
		} else if (r == 0) {
			break;
		} else if (r == -1) {
			errstr = tls_error(t);
			fprintf(stderr, "tls_read: %s\n", errstr ? errstr : "");
			goto err;
		}
		len += r;

		r = fwrite(buf, 1, r, stdout);
		if (ferror(stdout)) {
			fprintf(stderr, "fwrite: stdout: %s\n", strerror(errno));
			goto err;
		}

		if (config_maxresponsesiz && len >= config_maxresponsesiz)
			break;
	}
	if (config_maxresponsesiz && len >= config_maxresponsesiz) {
		fprintf(stderr, "read: response too big: %zu >= %zu\n",
		        len, config_maxresponsesiz);
		goto err;
	}
	ret = 0;

err:
	if (t) {
		tls_close(t);
		tls_free(t);
	}

	if (fd != -1)
		close(fd);
	return ret;
}

void
usage(void)
{
	fprintf(stderr, "usage: %s [-H headers] [-l] [-m maxresponse] "
	        "[-t timeout] url\n", argv0);
	exit(1);
}

int
main(int argc, char **argv)
{
	char *end;
	int statuscode;
	long long l;

	ARGBEGIN {
	case 'H': /* header(s) */
		config_headers = EARGF(usage());
		break;
	case 'l': /* legacy ciphers */
		config_legacy = 1;
		break;
	case 'm': /* max filesize */
		errno = 0;
		l = strtoll(EARGF(usage()), &end, 10);
		if (errno || *end != '\0' || l < 0)
			usage();
		config_maxresponsesiz = l;
		break;
	case 't': /* timeout */
		errno = 0;
		l = strtoll(EARGF(usage()), &end, 10);
		if (errno || *end != '\0' || l < 0)
			usage();
		config_timeout = l;
		break;
	default:
		usage();
	} ARGEND

	if (argc != 1)
		usage();

	url = argv[0];
	if (parseuri(url, &u) == -1)
		errx(1, "invalid url: %s", url);

	if (config_timeout > 0) {
		signal(SIGALRM, sighandler);
		if (alarm(config_timeout) == -1)
			err(1, "alarm");
	}

	if (!strcmp(u.proto, "https")) {
		if (tls_init())
			errx(1, "tls_init failed");
		if (!(tls_config = tls_config_new()))
			errx(1, "tls config failed");
		if (config_legacy) {
			/* enable legacy cipher and negotiation. */
			if (tls_config_set_ciphers(tls_config, "legacy"))
				errx(1, "tls set ciphers failed: %s",
				     tls_config_error(tls_config));
		}
		if (!u.port[0] && !strcmp(u.proto, "https"))
			memcpy(u.port, "443", 4);
		statuscode = https_request();
	} else if (!strcmp(u.proto, "http")) {
		if (!u.port[0])
			memcpy(u.port, "80", 3);
		statuscode = http_request();
	} else if (!strcmp(u.proto, "gopher")) {
		if (!u.port[0])
			memcpy(u.port, "70", 3);

		if (u.path[0] != '/' || u.path[1] == '\0')
			errx(1, "must specify type");

		statuscode = gopher_request();
	} else if (!strcmp(u.proto, "gophers")) {
		if (tls_init())
			errx(1, "tls_init failed");
		if (!(tls_config = tls_config_new()))
			errx(1, "tls config failed");
		if (config_legacy) {
			/* enable legacy cipher and negotiation. */
			if (tls_config_set_ciphers(tls_config, "legacy"))
				errx(1, "tls set ciphers failed: %s",
				     tls_config_error(tls_config));
		}
		if (!u.port[0])
			memcpy(u.port, "70", 3);

		if (u.path[0] != '/' || u.path[1] == '\0')
			errx(1, "must specify type");

		statuscode = gophers_request();
	} else {
		if (u.proto[0])
			errx(1, "unsupported protocol specified: %s", u.proto);
		else
			errx(1, "no protocol specified");
	}

	return statuscode;
}
