#include <sys/socket.h>
#include <sys/types.h>

#include <ctype.h>
#include <errno.h>
#include <netdb.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "https.h"
#include "util.h"
#include "youtube.h"

#define OUT(s) (fputs((s), stdout))

extern char **environ;

/* page title */
char title[1024];

/* CGI parameters */
static char rawsearch[4096], search[4096], order[16], page[64];
static char videoid[256];
static char channelid[256], userid[256];

/* Escape characters below as HTML 2.0 / XML 1.0.
   Translate multi-line to <br/> */
void
xmlencode_multiline(const char *s)
{
	for (; *s; s++) {
		switch(*s) {
		case '<':  fputs("&lt;", stdout);   break;
		case '>':  fputs("&gt;", stdout);   break;
		case '\'': fputs("&#39;", stdout);  break;
		case '&':  fputs("&amp;", stdout);  break;
		case '"':  fputs("&quot;", stdout); break;
		case '\n': fputs("<br/>", stdout);  break;
		default:   putchar(*s);
		}
	}
}

void
parsecgi(void)
{
	char *query, *p;
	size_t len;

	if (!(query = getenv("QUERY_STRING")))
		query = "";

	/* order */
	if ((p = getparam(query, "o"))) {
		if (decodeparam(order, sizeof(order), p) == -1 ||
			(strcmp(order, "date") &&
			strcmp(order, "relevance") &&
			strcmp(order, "views") &&
			strcmp(order, "rating")))
			order[0] = '\0';
	}
	if (!order[0])
		snprintf(order, sizeof(order), "relevance");

	/* search */
	if ((p = getparam(query, "q"))) {
		if ((len = strcspn(p, "&")) && len + 1 < sizeof(rawsearch)) {
			memcpy(rawsearch, p, len);
			rawsearch[len] = '\0';
		}

		if (decodeparam(search, sizeof(search), p) == -1) {
			OUT("Status: 401 Bad Request\r\n\r\n");
			exit(1);
		}
	}

	/* channel ID */
	if ((p = getparam(query, "chan"))) {
		if (decodeparam(channelid, sizeof(channelid), p) == -1)
			channelid[0] = '\0';
	}

	/* user ID */
	if ((p = getparam(query, "user"))) {
		if (decodeparam(userid, sizeof(userid), p) == -1)
			userid[0] = '\0';
	}

	/* video ID */
	if ((p = getparam(query, "v"))) {
		if (decodeparam(videoid, sizeof(videoid), p) == -1)
			videoid[0] = '\0';
	}
}

void
header(void)
{
	OUT(
		"Content-Type: text/html; charset=utf-8\r\n\r\n"
		"<!DOCTYPE html>\n<html>\n<head>\n"
		"<meta name=\"referrer\" content=\"no-referrer\" />\n"
		"<meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\" />\n");

	if (title[0]) {
		OUT("<title>");
		xmlencode(title);
		OUT("</title>");
	}

	OUT(
		"<link rel=\"stylesheet\" href=\"css/style.css\" type=\"text/css\" media=\"screen\" />\n"
		"<link rel=\"icon\" type=\"image/png\" href=\"/favicon.png\" />\n"
		"<meta content=\"width=device-width\" name=\"viewport\" />\n"
		"</head>\n"
	        "<body class=\"search\">\n"
		"<form method=\"get\" action=\"\">\n");

	OUT(
		"<table class=\"search\" width=\"100%\" border=\"0\" cellpadding=\"0\" cellspacing=\"0\">\n"
		"<tr>\n"
		"	<td width=\"100%\" class=\"input\">\n"
		"		<input type=\"search\" name=\"q\" value=\"");
	xmlencode(search);
	OUT(
		"\" placeholder=\"Search...\" size=\"72\" autofocus=\"autofocus\" class=\"search\" accesskey=\"f\" />\n"
		"	</td>\n"
		"	<td nowrap class=\"nowrap\">\n"
		"		<input type=\"submit\" value=\"Search\" class=\"button\"/>\n"
		"		<select name=\"o\" title=\"Order by\" accesskey=\"o\">\n");
	printf("			<option value=\"date\"%s>Creation date</option>\n", !strcmp(order, "date") ? " selected=\"selected\"" : "");
	printf("			<option value=\"relevance\"%s>Relevance</option>\n", !strcmp(order, "relevance") ? " selected=\"selected\"" : "");
	printf("			<option value=\"views\"%s>Views</option>\n", !strcmp(order, "views") ? " selected=\"selected\"" : "");
	printf("			<option value=\"rating\"%s>Rating</option>\n", !strcmp(order, "rating") ? " selected=\"selected\"" : "");
	OUT(
		"		</select>\n"
		"	</td>\n"
		"</tr>\n"
		"</table>\n"
		"</form>\n");
}

void
footer(void)
{
	OUT("</body>\n</html>\n");
}

int
render_search(struct search_response *r)
{
	struct item *v;
	int n;
	size_t i, len;

	if (pledge("stdio", NULL) == -1) {
		OUT("Status: 500 Internal Server Error\r\n\r\n");
		exit(1);
	}

	n = -1;
	if (search[0])
		n = snprintf(title, sizeof(title), "Search: \"%s\" sorted by %s", search, order);
	else if (channelid[0])
		n = snprintf(title, sizeof(title), "Channel videos: %s", channelid);
	else if (userid[0])
		n = snprintf(title, sizeof(title), "User videos: %s", userid);
	if (n < 0 || n >= sizeof(title))
		title[0] = '\0';

	header();

	if (r && r->nitems) {
		OUT(
			"<hr/>\n"
			"<table class=\"videos\" width=\"100%\" border=\"0\" cellpadding=\"0\" cellspacing=\"0\">\n"
			"<tbody>\n");

		for (i = 0; i < r->nitems; i++) {
			v = &(r->items[i]);

			OUT("<tr class=\"v\">\n"
			"	<td class=\"thumb\" width=\"120\" align=\"center\">\n");

			if (v->id[0]) {
				OUT("		<a href=\"https://www.youtube.com/embed/");
				xmlencode(v->id);
				OUT("\"><img src=\"https://i.ytimg.com/vi/");
				xmlencode(v->id);
				OUT("/default.jpg\" alt=\"\" height=\"90\" border=\"0\" /></a>\n");
			} else {
				/* placeholder image */
				OUT("		<img src=\"https://i.ytimg.com/vi/\" alt=\"\" height=\"90\" border=\"0\" />\n");
			}
			OUT("	</td>\n"
				"	<td>\n"
				"		<span class=\"title\">");

			if (v->id[0]) {
				OUT("<a href=\"https://www.youtube.com/embed/");
				xmlencode(v->id);
				printf("\" accesskey=\"%zu\"", i);
/*				if (v->shortdescription[0]) {
					OUT(" title=\"");
					xmlencode(v->shortdescription);
					OUT("\"");
				}*/
				OUT(">");
			}

			if (v->membersonly)
				xmlencode(MEMBERS_ONLY);

			switch (v->linktype) {
			case Channel:
				OUT("[Channel] ");
				xmlencode(v->channeltitle);
				break;
			case Movie:
				OUT("[Movie] ");
				xmlencode(v->title);
				break;
			case Playlist:
				OUT("[Playlist] ");
				xmlencode(v->title);
				break;
			default:
				xmlencode(v->title);
				break;
			}

			if (v->id[0])
				OUT("</a>");

			/* link to video information */
			if (v->id[0]) {
				OUT(" | <a href=\"?v=");
				xmlencode(v->id);
				OUT("\" title=\"More video details\">Details</a>");
			}

			OUT(
				"</span><br/>\n"
				"\t\t<span class=\"channel\">");

			if (v->channelid[0]) {
				OUT("<a href=\"?chan=");
				xmlencode(v->channelid);
				OUT("\">");
				xmlencode(v->channeltitle);
				OUT("</a>");
			} else if (v->userid[0]) {
				OUT("<a href=\"?user=");
				xmlencode(v->channelid);
				OUT("\">");
				xmlencode(v->channeltitle);
				OUT("</a>");
			} else {
				xmlencode(v->channeltitle);
			}

			if (v->channelid[0] || v->userid[0]) {
				OUT(" | <a title=\"");
				xmlencode(v->channeltitle);
				OUT(" Atom feed\" href=\"https://www.youtube.com/feeds/videos.xml?");
				if (v->channelid[0]) {
					OUT("channel_id=");
					xmlencode(v->channelid);
				} else if (v->userid[0]) {
					OUT("user=");
					xmlencode(v->userid);
				}
				OUT("\">Atom feed</a>");
			}

			OUT("</span><br/>\n");
			if (v->publishedat[0]) {
				OUT("		<span class=\"publishedat\">Published: ");
				OUT(v->publishedat);
				OUT("</span><br/>\n");
			}
			OUT("		<span class=\"stats\">");
			if (v->viewcount[0]) {
				if (!printnumsep(v->viewcount))
					OUT("0");
				OUT(" views");
			}
			OUT(
				"</span><br/>\n"
				"	</td>\n"
				"	<td align=\"right\" class=\"a-r\">\n"
				"		<span class=\"duration\">");
			OUT(v->duration);
			OUT(
				"</span>\n"
				"	</td>\n"
				"</tr>\n"
				"<tr class=\"hr\">\n"
				"	<td colspan=\"3\"><hr/></td>\n"
				"</tr>\n");
		}
		OUT("</tbody>\n</table>\n");
	}

	footer();

	return 0;
}

int
render_video(struct video_response *r)
{
	char buf[256];
	int n;

	if (pledge("stdio", NULL) == -1) {
		OUT("Status: 500 Internal Server Error\r\n\r\n");
		exit(1);
	}

	n = snprintf(title, sizeof(title), "%s - %s", r->title, r->author);
	if (n < 0 || n >= sizeof(title))
		title[0] = '\0';

	header();

	OUT("<hr/>\n");

	OUT("<table class=\"video\" border=\"0\" cellpadding=\"0\" cellspacing=\"0\">\n");
	OUT("<tbody>\n");

	OUT("<tr><td colspan=\"2\"><center>\n");
	OUT("<a href=\"https://www.youtube.com/embed/");
        xmlencode(r->id);
	OUT("\"><img src=\"https://i.ytimg.com/vi/");
	xmlencode(r->id);
	OUT("/hqdefault.jpg\" alt=\"\" border=\"0\" /></a>\n");
	OUT("</center><br/></td></tr>\n");

	OUT("<tr><td><b>Title:</b></td><td>");
	OUT("<a href=\"https://www.youtube.com/embed/");
        xmlencode(r->id);
	OUT("\">");
	xmlencode(r->title);
	OUT("</a></td></tr>\n");

	if (r->lengthseconds > 0) {
		OUT("<tr><td><b>Length:</b></td><td>");
		if (durationstr(r->lengthseconds, buf, sizeof(buf)) < sizeof(buf))
			xmlencode(buf);
		OUT("</td></tr>\n");
	}

	/* show playability state and reason: for example when it is a members-only video */
	if (r->playabilityreason[0]) {
		OUT("<tr><td><b>Status:</b></td><td><b>");
		if (r->playabilitystatus[0]) {
			OUT(r->playabilitystatus);
			OUT(": ");
		}
		OUT(r->playabilityreason);
		OUT("</b></td></tr>\n");
	}

	if (r->author[0]) {
		OUT("<tr><td><b>Channel:</b></td><td>");
		if (r->channelid[0]) {
			OUT("<a href=\"?chan=");
			xmlencode(r->channelid);
			OUT("\">");
			xmlencode(r->author);
			OUT("</a>");
			OUT(": <a href=\"https://www.youtube.com/feeds/videos.xml?channel_id=");
			xmlencode(r->channelid);
			OUT("\">Atom feed</a>");
		} else {
			xmlencode(r->author);
		}
		OUT("</td></tr>\n");
	}

	OUT("<tr><td><b>Views:</b></td><td>");
	snprintf(buf, sizeof(buf), "%ld", r->viewcount);
	if (!printnumsep(buf))
		OUT("0");
	OUT("</td></tr>\n");

	if (r->publishdate[0]) {
		OUT("<tr><td><b>Published:</b></td><td>");
		xmlencode(r->publishdate);
		OUT("</td></tr>\n");
	}

	if (r->uploaddate[0]) {
		OUT("<tr><td><b>Uploaded:</b></td><td>");
		xmlencode(r->uploaddate);
		OUT("</td></tr>\n");
	}

	if (r->shortdescription[0]) {
		OUT("<tr><td valign=\"top\"><b>Description:&nbsp;</b></td><td><code>");
		xmlencode_multiline(r->shortdescription);
		OUT("</code></td></tr>\n");
	}

	OUT("</tbody>\n");
	OUT("</table>\n");

	footer();

	return 0;
}

int
main(void)
{
	struct search_response *r = NULL;
	struct video_response *vr = NULL;

	if (pledge("stdio dns inet rpath unveil", NULL) == -1 ||
	    unveil(TLS_CA_CERT_FILE, "r") == -1 ||
	    unveil(NULL, NULL) == -1) {
		OUT("Status: 500 Internal Server Error\r\n\r\n");
		exit(1);
	}

	parsecgi();

	if (rawsearch[0]) {
		r = youtube_search(rawsearch, page, order);
	} else if (channelid[0]) {
		r = youtube_channel_videos(channelid);
	} else if (userid[0]) {
		r = youtube_user_videos(userid);
	} else if (videoid[0]) {
		vr = youtube_video(videoid);
		if (!vr || vr->isfound == 0) {
			OUT("Status: 500 Internal Server Error\r\n\r\n");
			exit(1);
		}
		render_video(vr);
		return 0;
	} else {
		goto show;
	}
	if (!r || r->nitems == 0) {
		OUT("Status: 500 Internal Server Error\r\n\r\n");
		exit(1);
	}

show:
	render_search(r);

	return 0;
}
