#include <sys/socket.h>
#include <sys/types.h>

#include <ctype.h>
#include <errno.h>
#include <netdb.h>
#include <stdarg.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "https.h"
#include "json.h"
#include "reddit.h"
#include "util.h"

static char *
reddit_request(const char *path)
{
	return request("old.reddit.com", path, "");
}

/* unmarshal JSON response, skip HTTP headers */
int
json_unmarshal(const char *data,
	void (*cb)(struct json_node *, size_t, const char *, size_t, void *),
	void *pp)
{
	const char *s;

	/* strip/skip header part */
	if (!(s = strstr(data, "\r\n\r\n"))) {
		fprintf(stderr, "error parsing HTTP response header\n");
		return -1; /* invalid response */
	}
	s += strlen("\r\n\r\n");

	/* parse */
	if (parsejson(s, strlen(s), cb, pp) < 0) {
		fprintf(stderr, "error parsing JSON\n");
		return -1;
	}

	return 0;
}

char *
reddit_list_data(const char *subreddit, int limit,
	const char *before, const char *after)
{
	char path[4096];
	int r;

	if (limit <= 0)
		limit = 25;

	if (subreddit[0])
		r = snprintf(path, sizeof(path), "/r/%s/.json?raw_json=1&limit=%d",
		             subreddit, limit);
	else
		r = snprintf(path, sizeof(path), "/.json?raw_json=1&limit=%d",
		             limit);

	if (before[0]) {
		strlcat(path, "&before=", sizeof(path));
		strlcat(path, before, sizeof(path));
	} else if (after[0]) {
		strlcat(path, "&after=", sizeof(path));
		strlcat(path, after, sizeof(path));
	}

	if (r < 0 || (size_t)r >= sizeof(path))
		return NULL;

	return reddit_request(path);
}

void
reddit_list_processnode(struct json_node *nodes, size_t depth, const char *value, size_t valuelen,
	void *pp)
{
	struct list_response *r = (struct list_response *)pp;
	struct item *item;
	struct json_node *node;
	struct tm *tm;

	if (depth == 3 &&
	    nodes[0].type == JSON_TYPE_OBJECT &&
	    nodes[1].type == JSON_TYPE_OBJECT &&
	    nodes[2].type == JSON_TYPE_STRING &&
	    !strcmp(nodes[0].name, "") &&
	    !strcmp(nodes[1].name, "data")) {
		if (!strcmp(nodes[2].name, "before")) {
			strlcpy(r->before, value, sizeof(r->before));
		} else if (!strcmp(nodes[2].name, "after")) {
			strlcpy(r->after, value, sizeof(r->after));
		}
	}

	if (r->nitems > MAX_ITEMS)
		return;

	/* new item */
	if (depth == 5 &&
	    nodes[0].type == JSON_TYPE_OBJECT &&
	    nodes[1].type == JSON_TYPE_OBJECT &&
	    nodes[2].type == JSON_TYPE_ARRAY &&
	    nodes[3].type == JSON_TYPE_OBJECT &&
	    nodes[4].type == JSON_TYPE_OBJECT &&
	    !strcmp(nodes[0].name, "") &&
	    !strcmp(nodes[1].name, "data") &&
	    !strcmp(nodes[2].name, "children") &&
	    !strcmp(nodes[3].name, "") &&
	    !strcmp(nodes[4].name, "data")) {
		r->nitems++;
		return;
	}

	if (r->nitems == 0)
		return;
	item = &(r->items[r->nitems - 1]);

	if (depth >= 5 &&
	    nodes[0].type == JSON_TYPE_OBJECT &&
	    nodes[1].type == JSON_TYPE_OBJECT &&
	    nodes[2].type == JSON_TYPE_ARRAY &&
	    nodes[3].type == JSON_TYPE_OBJECT &&
	    nodes[4].type == JSON_TYPE_OBJECT &&
	    !strcmp(nodes[0].name, "") &&
	    !strcmp(nodes[1].name, "data") &&
	    !strcmp(nodes[2].name, "children") &&
	    !strcmp(nodes[3].name, "") &&
	    !strcmp(nodes[4].name, "data")) {
		if (depth == 6) {
			node = &nodes[5];
			switch (node->type) {
			case JSON_TYPE_BOOL:
				if (!strcmp(node->name, "is_video"))
					item->is_video = value[0] == 't';
				break;
			case JSON_TYPE_NUMBER:
				if (!strcmp(node->name, "ups"))
					item->ups = strtol(value, NULL, 10);
				else if (!strcmp(node->name, "downs"))
					item->downs = strtol(value, NULL, 10);
				else if (!strcmp(node->name, "num_comments"))
					item->num_comments = strtol(value, NULL, 10);
				else if (!strcmp(node->name, "created_utc")) {
					item->created_utc = strtoll(value, NULL, 10);
					/* convert to struct tm */
					tm = gmtime(&(item->created_utc));
					memcpy(&(item->created_tm), tm, sizeof(*tm));
				}
				break;
			case JSON_TYPE_STRING:
				if (!strcmp(node->name, "name"))
					strlcpy(item->name, value, sizeof(item->name));
				else if (!strcmp(node->name, "title"))
					strlcpy(item->title, value, sizeof(item->title));
				else if (!strcmp(node->name, "url"))
					strlcpy(item->url, value, sizeof(item->url));
				else if (!strcmp(node->name, "permalink"))
					strlcpy(item->permalink, value, sizeof(item->permalink));
				else if (!strcmp(node->name, "subreddit"))
					strlcpy(item->subreddit, value, sizeof(item->subreddit));
				else if (!strcmp(node->name, "author"))
					strlcpy(item->author, value, sizeof(item->author));
				else if (!strcmp(node->name, "thumbnail"))
					strlcpy(item->thumbnail, value, sizeof(item->thumbnail));
				else if (!strcmp(node->name, "link_flair_text"))
					strlcpy(item->link_flair_text, value, sizeof(item->link_flair_text));
				else if (!strcmp(node->name, "link_flair_background_color") && value[0] == '#')
					strlcpy(item->link_flair_background_color, value, sizeof(item->link_flair_background_color));
				break;
			default:
				break;
			}
		} else if (depth == 8 &&
		    nodes[5].type == JSON_TYPE_OBJECT &&
		    nodes[6].type == JSON_TYPE_OBJECT &&
		    (!strcmp(nodes[5].name, "media") || !strcmp(nodes[5].name, "secure_media")) &&
		    !strcmp(nodes[6].name, "reddit_video")) {
			node = &nodes[7];

			switch (node->type) {
			case JSON_TYPE_NUMBER:
				/* prefer "insecure" */
				if (nodes[5].name[0] == 's' && item->duration)
					break;
				if (!strcmp(node->name, "duration"))
					item->duration = strtol(value, NULL, 10);
				break;
			case JSON_TYPE_STRING:
				/* prefer "insecure" */
				if (nodes[5].name[0] == 's' && item->dash_url[0])
					break;
				if (!strcmp(node->name, "dash_url"))
					strlcpy(item->dash_url, value, sizeof(item->dash_url));
				break;
			default:
				break;
			}
		}
	}
}

struct list_response *
reddit_list(const char *subreddit, int limit,
	const char *before, const char *after)
{
	struct list_response *r;
	char *data;

	if (!(data = reddit_list_data(subreddit, limit, before, after))) {
		fprintf(stderr, "%s\n", __func__);
		return NULL;
	}

	if (!(r = calloc(1, sizeof(*r)))) {
		fprintf(stderr, "calloc\n");
		return NULL;
	}

	if (json_unmarshal(data, reddit_list_processnode, r) == -1) {
		free(r);
		r = NULL;
	}
	free(data);

	return r;
}

int
reddit_isvalidlink(const char *s)
{
	char *end = NULL;
	unsigned long long l;

	/* type prefix: reddit link is "t3_" */
	if (strncmp(s, "t3_", 3))
		return 0;
	s += 3;

	/* link is base36 */
	errno = 0;
	l = strtoull(s, &end, 36);
	return (!errno && s != end && !*end && l > 0);
}
