#include <sys/socket.h>
#include <sys/types.h>

#include <ctype.h>
#include <errno.h>
#include <netdb.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "https.h"
#include "json.h"
#include "util.h"
#include "youtube.h"

static long long
getnum(const char *s)
{
	long long l;

	l = strtoll(s, 0, 10);
	if (l < 0)
		l = 0;
	return l;
}

static char *
youtube_request(const char *path)
{
	return request("www.youtube.com", path, "");
}

static char *
request_video(const char *videoid)
{
	char path[2048];
	int r;

	r = snprintf(path, sizeof(path), "/watch?v=%s", videoid);
	/* check if request is too long (truncation) */
	if (r < 0 || (size_t)r >= sizeof(path))
		return NULL;

//	return readfile("/tmp/data"); // DEBUG

	return youtube_request(path);
}

static char *
request_channel_videos(const char *channelid)
{
	char path[2048];
	int r;

	r = snprintf(path, sizeof(path), "/channel/%s/videos", channelid);
	/* check if request is too long (truncation) */
	if (r < 0 || (size_t)r >= sizeof(path))
		return NULL;

	return youtube_request(path);
}

static char *
request_user_videos(const char *user)
{
	char path[2048];
	int r;

	r = snprintf(path, sizeof(path), "/user/%s/videos", user);
	/* check if request is too long (truncation) */
	if (r < 0 || (size_t)r >= sizeof(path))
		return NULL;

	return youtube_request(path);
}

static char *
request_search(const char *s, const char *page, const char *order)
{
	char path[4096];

	snprintf(path, sizeof(path), "/results?search_query=%s", s);

	/* NOTE: pagination doesn't work at the moment:
	   this parameter is not supported anymore by Youtube */
	if (page[0]) {
		strlcat(path, "&page=", sizeof(path));
		strlcat(path, page, sizeof(path));
	}

	if (order[0] && strcmp(order, "relevance")) {
		strlcat(path, "&sp=", sizeof(path));
		if (!strcmp(order, "date"))
			strlcat(path, "CAI%3D", sizeof(path));
		else if (!strcmp(order, "views"))
			strlcat(path, "CAM%3D", sizeof(path));
		else if (!strcmp(order, "rating"))
			strlcat(path, "CAE%3D", sizeof(path));
	}

	/* check if request is too long (truncation) */
	if (strlen(path) >= sizeof(path) - 1)
		return NULL;

	return youtube_request(path);
}

static int
extractjson_search(const char *s, const char **start, const char **end)
{
	*start = strstr(s, "window[\"ytInitialData\"] = ");
	if (*start) {
		(*start) += sizeof("window[\"ytInitialData\"] = ") - 1;
	} else {
		*start = strstr(s, "var ytInitialData = ");
		if (*start)
			(*start) += sizeof("var ytInitialData = ") - 1;
	}
	if (!*start)
		return -1;
	*end = strstr(*start, "};\n");
	if (!*end)
		*end = strstr(*start, "}; \n");
	if (!*end)
		*end = strstr(*start, "};<");
	if (!*end)
		return -1;
	(*end)++;

	return 0;
}

static int
extractjson_video(const char *s, const char **start, const char **end)
{
	*start = strstr(s, "var ytInitialPlayerResponse = ");
	if (!*start)
		return -1;
	(*start) += sizeof("var ytInitialPlayerResponse = ") - 1;
	*end = strstr(*start, "};<");
	if (!*end)
		return -1;
	(*end)++;

	return 0;
}

static void
processnode_search(struct json_node *nodes, size_t depth, const char *value, size_t valuelen,
	void *pp)
{
	struct search_response *r = (struct search_response *)pp;
	static struct item *item;

	if (r->nitems > MAX_VIDEOS)
		return;

	/* new item, structures can be very deep, just check the end for:
	   (items|contents)[].videoRenderer objects */
	if (depth >= 3 &&
	    nodes[depth - 1].type == JSON_TYPE_OBJECT &&
	    !strcmp(nodes[depth - 1].name, "videoRenderer")) {
		r->nitems++;
		return;
	}

	if (r->nitems == 0)
		return;
	item = &(r->items[r->nitems - 1]);

	if (depth >= 4 &&
	    nodes[depth - 1].type == JSON_TYPE_STRING &&
	    !strcmp(nodes[depth - 2].name, "videoRenderer") &&
	    !strcmp(nodes[depth - 1].name, "videoId")) {
		strlcpy(item->id, value, sizeof(item->id));
	}

	if (depth >= 7 &&
	    nodes[depth - 5].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 4].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 3].type == JSON_TYPE_ARRAY &&
	    nodes[depth - 2].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 1].type == JSON_TYPE_STRING &&
	    !strcmp(nodes[depth - 5].name, "videoRenderer") &&
	    !strcmp(nodes[depth - 4].name, "title") &&
	    !strcmp(nodes[depth - 3].name, "runs") &&
	    !strcmp(nodes[depth - 1].name, "text") &&
		!item->title[0]) {
		strlcpy(item->title, value, sizeof(item->title));
	}

	/* in search listing there is a short description, string items are appended */
	if (depth >= 8 &&
	    nodes[depth - 7].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 6].type == JSON_TYPE_ARRAY &&
	    nodes[depth - 5].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 4].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 3].type == JSON_TYPE_ARRAY &&
	    nodes[depth - 2].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 1].type == JSON_TYPE_STRING &&
	    !strcmp(nodes[depth - 7].name, "videoRenderer") &&
	    !strcmp(nodes[depth - 6].name, "detailedMetadataSnippets") &&
	    !strcmp(nodes[depth - 4].name, "snippetText") &&
	    !strcmp(nodes[depth - 3].name, "runs") &&
	    !strcmp(nodes[depth - 1].name, "text")) {
		strlcat(item->shortdescription, value, sizeof(item->shortdescription));
	}

	/* in channel/user videos listing there is a short description, string items are appended */
	if (depth >= 7 &&
	    nodes[depth - 5].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 4].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 3].type == JSON_TYPE_ARRAY &&
	    nodes[depth - 2].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 1].type == JSON_TYPE_STRING &&
	    !strcmp(nodes[depth - 5].name, "videoRenderer") &&
	    !strcmp(nodes[depth - 4].name, "descriptionSnippet") &&
	    !strcmp(nodes[depth - 3].name, "runs") &&
	    !strcmp(nodes[depth - 1].name, "text")) {
		strlcat(item->shortdescription, value, sizeof(item->shortdescription));
	}

	if (depth >= 5 &&
	    nodes[depth - 4].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 3].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 2].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 1].type == JSON_TYPE_STRING &&
	    !strcmp(nodes[depth - 3].name, "videoRenderer") &&
	    !strcmp(nodes[depth - 1].name, "simpleText")) {
		if (!strcmp(nodes[depth - 2].name, "viewCountText") &&
		    !item->viewcount[0]) {
			strlcpy(item->viewcount, value, sizeof(item->viewcount));
		} else if (!strcmp(nodes[depth - 2].name, "lengthText") &&
		    !item->duration[0]) {
			strlcpy(item->duration, value, sizeof(item->duration));
		} else if (!strcmp(nodes[depth - 2].name, "publishedTimeText") &&
		    !item->publishedat[0]) {
			strlcpy(item->publishedat, value, sizeof(item->publishedat));
		}
	}

	if (depth >= 9 &&
	    nodes[depth - 8].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 7].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 6].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 5].type == JSON_TYPE_ARRAY &&
	    nodes[depth - 4].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 3].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 2].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 1].type == JSON_TYPE_STRING &&
	    !strcmp(nodes[depth - 7].name, "videoRenderer") &&
	    !strcmp(nodes[depth - 6].name, "longBylineText") &&
	    !strcmp(nodes[depth - 5].name, "runs") &&
	    !strcmp(nodes[depth - 3].name, "navigationEndpoint") &&
	    !strcmp(nodes[depth - 2].name, "browseEndpoint")) {
		if (!strcmp(nodes[depth - 1].name, "browseId")) {
			strlcpy(item->channelid, value, sizeof(item->channelid));
		}
	}

	if (depth >= 7 &&
	    nodes[depth - 6].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 5].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 4].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 3].type == JSON_TYPE_ARRAY &&
	    nodes[depth - 2].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 1].type == JSON_TYPE_STRING &&
	    !strcmp(nodes[depth - 5].name, "videoRenderer") &&
	    !strcmp(nodes[depth - 4].name, "longBylineText") &&
	    !strcmp(nodes[depth - 3].name, "runs")) {
		if (!strcmp(nodes[depth - 1].name, "text") &&
		    !item->channeltitle[0]) {
			strlcpy(item->channeltitle, value, sizeof(item->channeltitle));
		}
	}
}

static struct search_response *
parse_search_response(const char *data)
{
	struct search_response *r;
	const char *s, *start, *end;
	int ret;

	if (!(s = strstr(data, "\r\n\r\n")))
		return NULL; /* invalid response */
	/* skip header */
	s += strlen("\r\n\r\n");

//	s = data; // DEBUG

	if (!(r = calloc(1, sizeof(*r))))
		return NULL;

	if (extractjson_search(s, &start, &end) == -1) {
		free(r);
		return NULL;
	}

	ret = parsejson(start, end - start, processnode_search, r);
	if (ret < 0) {
		free(r);
		return NULL;
	}
	return r;
}

static void
processnode_video(struct json_node *nodes, size_t depth, const char *value, size_t valuelen,
	void *pp)
{
	struct video_response *r = (struct video_response *)pp;
	struct video_format *f;

	if (depth > 1) {
		if (nodes[0].type == JSON_TYPE_OBJECT &&
		    !strcmp(nodes[1].name, "streamingData")) {
			if (depth == 2 &&
			    nodes[2].type == JSON_TYPE_STRING &&
			    !strcmp(nodes[2].name, "expiresInSeconds")) {
				r->expiresinseconds = getnum(value);
			}

			if (depth >= 3 &&
			    nodes[2].type == JSON_TYPE_ARRAY &&
			    (!strcmp(nodes[2].name, "formats") ||
			    !strcmp(nodes[2].name, "adaptiveFormats"))) {
				if (r->nformats > MAX_FORMATS)
					return; /* ignore: don't add too many formats */

				if (depth == 4 && nodes[3].type == JSON_TYPE_OBJECT)
					r->nformats++;

				if (r->nformats == 0)
					return;
				f = &(r->formats[r->nformats - 1]); /* current video format item */

				if (depth == 5 &&
				    nodes[2].type == JSON_TYPE_ARRAY &&
				    nodes[3].type == JSON_TYPE_OBJECT &&
				    (nodes[4].type == JSON_TYPE_STRING ||
				    nodes[4].type == JSON_TYPE_NUMBER ||
				    nodes[4].type == JSON_TYPE_BOOL)) {
					if (!strcmp(nodes[4].name, "width")) {
						f->width = getnum(value);
					} else if (!strcmp(nodes[4].name, "height")) {
						f->height = getnum(value);
					} else if (!strcmp(nodes[4].name, "url")) {
						strlcpy(f->url, value, sizeof(f->url));
					} else if (!strcmp(nodes[4].name, "signatureCipher")) {
						strlcpy(f->signaturecipher, value, sizeof(f->signaturecipher));
					} else if (!strcmp(nodes[4].name, "qualityLabel")) {
						strlcpy(f->qualitylabel, value, sizeof(f->qualitylabel));
					} else if (!strcmp(nodes[4].name, "quality")) {
						strlcpy(f->quality, value, sizeof(f->quality));
					} else if (!strcmp(nodes[4].name, "fps")) {
						f->fps = getnum(value);
					} else if (!strcmp(nodes[4].name, "bitrate")) {
						f->bitrate = getnum(value);
					} else if (!strcmp(nodes[4].name, "averageBitrate")) {
						f->averagebitrate = getnum(value);
					} else if (!strcmp(nodes[4].name, "mimeType")) {
						strlcpy(f->mimetype, value, sizeof(f->mimetype));
					} else if (!strcmp(nodes[4].name, "itag")) {
						f->itag = getnum(value);
					} else if (!strcmp(nodes[4].name, "contentLength")) {
						f->contentlength = getnum(value);
					} else if (!strcmp(nodes[4].name, "lastModified")) {
						f->lastmodified = getnum(value);
					} else if (!strcmp(nodes[4].name, "audioChannels")) {
						f->audiochannels = getnum(value);
					} else if (!strcmp(nodes[4].name, "audioSampleRate")) {
						f->audiosamplerate = getnum(value);
					}
				}
			}
		}
	}

	if (depth == 4 &&
	    nodes[0].type == JSON_TYPE_OBJECT &&
	    nodes[1].type == JSON_TYPE_OBJECT &&
	    nodes[2].type == JSON_TYPE_OBJECT &&
	    nodes[3].type == JSON_TYPE_STRING &&
	    !strcmp(nodes[1].name, "microformat") &&
	    !strcmp(nodes[2].name, "playerMicroformatRenderer")) {
		r->isfound = 1;

		if (!strcmp(nodes[3].name, "publishDate")) {
			strlcpy(r->publishdate, value, sizeof(r->publishdate));
		} else if (!strcmp(nodes[3].name, "uploadDate")) {
			strlcpy(r->uploaddate, value, sizeof(r->uploaddate));
		} else if (!strcmp(nodes[3].name, "category")) {
			strlcpy(r->category, value, sizeof(r->category));
		} else if (!strcmp(nodes[3].name, "isFamilySafe")) {
			r->isfamilysafe = !strcmp(value, "true");
		} else if (!strcmp(nodes[3].name, "isUnlisted")) {
			r->isunlisted = !strcmp(value, "true");
		}
	}

	if (depth == 3) {
		if (nodes[0].type == JSON_TYPE_OBJECT &&
		    nodes[2].type == JSON_TYPE_STRING &&
		    !strcmp(nodes[1].name, "videoDetails")) {
			r->isfound = 1;

			if (!strcmp(nodes[2].name, "title")) {
				strlcpy(r->title, value, sizeof(r->title));
			} else if (!strcmp(nodes[2].name, "videoId")) {
				strlcpy(r->id, value, sizeof(r->id));
			} else if (!strcmp(nodes[2].name, "lengthSeconds")) {
				r->lengthseconds = getnum(value);
			} else if (!strcmp(nodes[2].name, "author")) {
				strlcpy(r->author, value, sizeof(r->author));
			} else if (!strcmp(nodes[2].name, "viewCount")) {
				r->viewcount = getnum(value);
			} else if (!strcmp(nodes[2].name, "channelId")) {
				strlcpy(r->channelid, value, sizeof(r->channelid));
			} else if (!strcmp(nodes[2].name, "shortDescription")) {
				strlcpy(r->shortdescription, value, sizeof(r->shortdescription));
			}
		}
	}
}

static struct video_response *
parse_video_response(const char *data)
{
	struct video_response *r;
	const char *s, *start, *end;
	int ret;

	if (!(s = strstr(data, "\r\n\r\n")))
		return NULL; /* invalid response */
	/* skip header */
	s += strlen("\r\n\r\n");

//	s = data; // DEBUG

	if (!(r = calloc(1, sizeof(*r))))
		return NULL;

	if (extractjson_video(s, &start, &end) == -1) {
		free(r);
		return NULL;
	}

	ret = parsejson(start, end - start, processnode_video, r);
	if (ret < 0) {
		free(r);
		return NULL;
	}
	return r;
}

struct search_response *
youtube_search(const char *rawsearch, const char *page, const char *order)
{
	const char *data;

	if (!(data = request_search(rawsearch, page, order)))
		return NULL;

	return parse_search_response(data);
}

struct search_response *
youtube_channel_videos(const char *channelid)
{
	const char *data;

	if (!(data = request_channel_videos(channelid)))
		return NULL;

	return parse_search_response(data);
}

struct search_response *
youtube_user_videos(const char *user)
{
	const char *data;

	if (!(data = request_user_videos(user)))
		return NULL;

	return parse_search_response(data);
}

struct video_response *
youtube_video(const char *videoid)
{
	const char *data;

	if (!(data = request_video(videoid)))
		return NULL;

	return parse_video_response(data);
}
